from numpy import empty, zeros
from ..pakbase import Package

class ModflowMnw1(Package):
    """
    MODFLOW Multi-Node Well 1 Package Class.

    Parameters
    ----------
    model : model object
        The model object (of type :class:`flopy.modflow.mf.Modflow`) to which
        this package will be added.
    mxmnw : integer
        maximum number of multi-node wells to be simulated
    ipakcb : integer
        A flag that is used to determine if cell-by-cell budget data should be
        saved. If ipakcb is non-zero cell-by-cell budget data will be saved.
        (default is 0).
    iwelpt : integer
        verbosity flag
    nomoiter : integer
        the number of iterations for which flow in MNW wells is calculated
    kspref : string
        which set of water levels are to be used as reference values for
        calculating drawdown
    losstype : string
        head loss type for each well
    wel1_bynode_qsum : list of lists
        nested list containing file names, unit numbers, and ALLTIME flag for
        auxilary output, e.g. [['test.ByNode',92,'ALLTIME']]
    itmp : array
        number of wells to be simulated for each stress period (shape : (NPER))
    lay_row_col_qdes_mn_multi : list of arrays
        lay, row, col, qdes, and MN or MULTI flag for all well nodes
        (length : NPER)
    mnwname : string
        prefix name of file for outputting time series data from MNW1
    extension : string
        Filename extension (default is 'mnw1')
    unitnumber : int
        File unit number (default is 33).
    filenames : string or list of strings
        File name of the package (with extension) or a list with the filename
        of the package and the cell-by-cell budget file for ipakcb. Default
        is None.

    Attributes
    ----------

    Methods
    -------

    See Also
    --------

    Notes
    -----
    Parameters are not supported in FloPy.

    The functionality of the ADD flag in data set 4 is not supported. Also
    not supported are all water-quality parameters (Qwval Iqwgrp), water-level
    limitations (Hlim, Href, DD), non-linear well losses, and pumping
    limitations (QCUT, Q-%CUT, Qfrcmn, Qfrcmx, DEFAULT).

    Examples
    --------

    >>> import flopy
    >>> ml = flopy.modflow.Modflow()
    >>> mnw1 = flopy.modflow.ModflowMnw1(ml, ...)

    """

    def __init__(self, model, mxmnw=0, ipakcb=None, iwelpt=0, nomoiter=0,
                 kspref=1, wel1_bynode_qsum=None,
                 itmp=0,
                 lay_row_col_qdes_mn_multi=None,
                 mnwname=None,
                 extension='mnw1', unitnumber=None, filenames=None):
        # set default unit number of one is not specified
        if unitnumber is None:
            unitnumber = ModflowMnw1.defaultunit()

        # set filenames
        if filenames is None:
            filenames = [None, None]
        elif isinstance(filenames, str):
            filenames = [filenames, None]
        elif isinstance(filenames, list):
            if len(filenames) < 2:
                filenames.append(None)

        # update external file information with cbc output, if necessary
        if ipakcb is not None:
            fname = filenames[1]
            model.add_output_file(ipakcb, fname=fname,
                                  package=ModflowMnw1.ftype())
        else:
            ipakcb = 0

        # set package name
        fname = [filenames[0]]

        # Call ancestor's init to set self.parent, extension, name, and unit number
        Package.__init__(self, model, extension, ModflowMnw1.ftype(),
                         unitnumber, filenames=fname)

        self.url = 'mnw1.htm'
        self.nper = self.parent.nrow_ncol_nlay_nper[-1]
        self.heading = '# {} package for '.format(self.name[0]) + \
                       ' {}, '.format(model.version_types[model.version]) + \
                       'generated by Flopy.'
        self.mxmnw = mxmnw              #-maximum number of multi-node wells to be simulated
        self.ipakcb = ipakcb
        self.iwelpt = iwelpt            #-verbosity flag
        self.nomoiter = nomoiter        #-integer indicating the number of iterations for which flow in MNW wells is calculated
        self.kspref = kspref            #-alphanumeric key indicating which set of water levels are to be used as reference values for calculating drawdown
        self.losstype = 'SKIN'          #-string indicating head loss type for each well
        self.wel1_bynode_qsum = wel1_bynode_qsum #-nested list containing file names, unit numbers, and ALLTIME flag for auxilary output, e.g. [['test.ByNode',92,'ALLTIME']]
        self.itmp = itmp                #-array containing # of wells to be simulated for each stress period (shape = (NPER))  
        self.lay_row_col_qdes_mn_multi = lay_row_col_qdes_mn_multi  #-list of arrays containing lay, row, col, qdes, and MN or MULTI flag for all well nodes (length = NPER)
        self.mnwname = mnwname          #-string prefix name of file for outputting time series data from MNW1

        #-create empty arrays of the correct size
        self.itmp = zeros( self.nper,dtype='int32' )

        #-assign values to arrays
        self.assignarray_old( self.itmp, itmp )

        #-input format checks:
        lossTypes = ['SKIN','LINEAR']
        assert self.losstype in lossTypes, 'LOSSTYPE (%s) must be one of the following: "%s" or "%s"' % ( self.losstype, lossTypes[0], lossTypes[1] )
        auxFileExtensions = ['wl1','ByNode','Qsum']
        for each in self.wel1_bynode_qsum:
            assert each[0].split('.')[1] in auxFileExtensions, 'File extensions in "wel1_bynode_qsum" must be one of the following: ".wl1", ".ByNode", or ".Qsum".'
        assert self.itmp.max() <= self.mxmnw, 'ITMP cannot exceed maximum number of wells to be simulated.'
        
        self.parent.add_package(self)

    def write_file(self):
        """
        Write the package file.

        Returns
        -------
        None

        """

        #-open file for writing
        # f_mnw1 = open( self.file_name[0], 'w' )
        f_mnw1 = open( self.fn_path, 'w' )

        #-write header
        f_mnw1.write( '%s\n' % self.heading )

        #-Section 1 - MXMNW ipakcb IWELPT NOMOITER REF:kspref
        f_mnw1.write( '%10i%10i%10i%10i REF = %s\n' % ( self.mxmnw, self.ipakcb, self.iwelpt, self.nomoiter, self.kspref) )
        
        #-Section 2 - LOSSTYPE {PLossMNW}
        f_mnw1.write( '%s\n' % ( self.losstype ) )                        

        #-Section 3a - {FILE:filename WEL1:iunw1}
        for each in self.wel1_bynode_qsum:
            if each[0].split('.')[1] == 'wl1':
                f_mnw1.write( 'FILE:%s WEL1:%10i\n' % ( each[0],each[1] ) )

        #-Section 3b - {FILE:filename BYNODE:iunby} {ALLTIME}
        for each in self.wel1_bynode_qsum:
            if each[0].split('.')[1] == 'ByNode':
                if len(each) == 2:
                    f_mnw1.write( 'FILE:%s BYNODE:%10i\n' % ( each[0],each[1] ) )
                elif len(each) == 3:
                    f_mnw1.write( 'FILE:%s BYNODE:%10i %s\n' % ( each[0],each[1],each[2] ) )

        #-Section 3C - {FILE:filename QSUM:iunqs} {ALLTIME}
        for each in self.wel1_bynode_qsum:
            if each[0].split('.')[1] == 'Qsum':
                if len(each) == 2:
                    f_mnw1.write( 'FILE:%s QSUM:%10i\n' % ( each[0],each[1] ) )
                elif len(each) == 3:
                    f_mnw1.write( 'FILE:%s QSUM:%10i %s\n' % ( each[0],each[1],each[2] ) )
        

        #-Repeat NPER times:
        for p in range(self.nper):
            #-Section 4 - ITMP ({ADD} flag is not supported)
            f_mnw1.write( '%10i\n' % ( self.itmp[p] ) )

            #-Section 5 - Lay Row Col Qdes {(MN MULTI) QWval Rw Skin Hlim Href (DD) Cp:C (QCUT Q-%CUT: Qfrcmn Qfrcmx) DEFAULT SITE: MNWsite}
            for node in self.lay_row_col_qdes_mn_multi:
                f_mnw1.write( '%10i%10i%10i %10f %s \n' % ( node[0],node[1],node[2],node[3],node[4] ) )

        #-Un-numbered section PREFIX:MNWNAME
        if self.mnwname:
            f_mnw1.write('PREFIX:%s\n' % ( self.mnwname ) )


        f_mnw1.close()


    @staticmethod
    def ftype():
        return 'MNW1'


    @staticmethod
    def defaultunit():
        return 33

