from __future__ import absolute_import
from datetime import datetime

import pytest
from freezegun import freeze_time

from charlatan import testing
from charlatan import depgraph
from charlatan import FixturesManager


class TestFixturesManager(testing.TestCase):

    def test_load_two_files(self):
        """Verify we can load two files."""
        fixtures_manager = FixturesManager()
        fixtures_manager.load(
            './charlatan/tests/data/relationships_without_models.yaml')
        fixtures_manager.load(
            './charlatan/tests/data/simple.yaml')
        assert 'simple_dict' in fixtures_manager.keys()

    def test_install_fixture(self):
        """install_fixture should return the fixture."""
        fixtures_manager = FixturesManager()
        fixtures_manager.load(
            './charlatan/tests/data/relationships_without_models.yaml')

        fixture = fixtures_manager.install_fixture('simple_dict')

        self.assertEqual(fixture, {
            'field1': 'lolin',
            'field2': 2,
        })

    def test_get_all_fixtures(self):
        fixtures_manager = FixturesManager()
        fixtures_manager.load('./charlatan/tests/data/simple.yaml')
        assert len(fixtures_manager.get_all_fixtures()) == 1

    def test_uninstall_all_fixtures(self):
        fixtures_manager = FixturesManager()
        fixtures_manager.load('./charlatan/tests/data/simple.yaml')
        assert len(fixtures_manager.install_all_fixtures()) == 1
        fixtures_manager.uninstall_all_fixtures()
        assert len(fixtures_manager.installed_keys) == 0

    @freeze_time("2014-12-31 11:00:00")
    def test_install_fixture_with_now(self):
        """Verify that we can install a fixture with !now tag."""
        fixtures_manager = FixturesManager()
        fixtures_manager.load('./charlatan/tests/data/simple.yaml')
        fixture = fixtures_manager.install_fixture('fixture')
        self.assertEqual(fixture,
                         {'now': datetime(2014, 12, 30, 11, 0)})

    def test_install_fixture_override(self):
        """Verify that we can override a fixture field."""
        fixtures_manager = FixturesManager()
        fixtures_manager.load('./charlatan/tests/data/simple.yaml')
        fixture = fixtures_manager.install_fixture('fixture',
                                                   overrides={'now': None})
        self.assertEqual(fixture, {'now': None})

    def test_uninstall_fixture(self):
        fixtures_manager = FixturesManager()
        fixtures_manager.load(
            './charlatan/tests/data/relationships_without_models.yaml')

        fixtures_manager.install_fixture('simple_dict')
        fixtures_manager.uninstall_fixture('simple_dict')

        # verify we are forgiving with list inputs
        fixtures_manager.install_fixtures('simple_dict')
        fixtures_manager.uninstall_fixtures('simple_dict')

    def test_uninstall_non_installed_fixture(self):
        fixtures_manager = FixturesManager()
        fixtures_manager.load(
            './charlatan/tests/data/relationships_without_models.yaml')
        fixtures_manager.uninstall_fixture('simple_dict')

    def test_dependency_parsing(self):
        fm = FixturesManager()
        fm.load(
            './charlatan/tests/data/dependencies.yaml'
        )
        assert fm.depgraph.has_edge_between('fixture1', 'fixture2')
        assert fm.depgraph.has_edge_between('fixture1', 'fixture3')
        assert fm.depgraph.has_edge_between('fixture4', 'fixture3')
        assert fm.depgraph.has_edge_between('fixture2', 'fixture4')

    def test_notices_cyclic_dependencies(self):
        fm = FixturesManager()
        self.assertRaises(
            depgraph.HasACycle,
            fm.load,
            './charlatan/tests/data/cyclic_dependencies.yaml'
        )

    def test_constructs_ancestors(self):
        fm = FixturesManager()
        fm.load(
            './charlatan/tests/data/dependencies.yaml'
        )
        assert not fm.cache
        # loading fixture3 should load fixture1 and fixture2 also
        fm.get_fixture('fixture3')
        self.assertIn('fixture1', fm.cache)
        self.assertIn('fixture4', fm.cache)

    def test_invalid_hook(self):
        """Verify that can't set an invalid hook."""

        manager = FixturesManager()
        with pytest.raises(KeyError):
            manager.set_hook("invalid", lambda p: p)

    def test_set_hook(self):
        """Verify that we can set a hook."""
        manager = FixturesManager()
        manager.set_hook("before_save", lambda p: p)
