# -*- coding: utf-8 -*-
#*****************************************************************************
#       Copyright (C) 2006  Michael Graz. <mgraz@plan10.com>
#
#  Distributed under the terms of the BSD License.  The full license is in
#  the file COPYING, distributed as part of this software.
#*****************************************************************************

import sys, unittest,pdb
sys.path.append (u'../..')
from pyreadline.modes.vi import *
from pyreadline import keysyms
from pyreadline.lineeditor import lineobj
from pyreadline.logger import log
import pyreadline.logger as logger
from common import *

from common import *
#----------------------------------------------------------------------

class ViModeTest (ViMode):
    tested_commands={}
    def __init__ (self):
        ViMode.__init__ (self, MockReadline())
        self.mock_console = MockConsole ()
        self.init_editing_mode (None)
        self.vi_set_insert_mode (True)
        self.lst_completions = []
        self.completer = self.mock_completer
        self.completer_delims = u' u'
        self.tabstop = 4

    def get_mock_console (self):
        return self.mock_console
    console = property (get_mock_console)

    def _set_line (self, text):
        self.l_buffer.set_line (text)

    def get_line (self):
        return self.l_buffer.get_line_text ()
    line = property (get_line)

    def get_line_cursor (self):
        return self.l_buffer.point
    line_cursor = property (get_line_cursor)

    def input (self, keytext):
        if keytext[0] == u'"' and keytext[-1] == u'"':
            lst_key = [u'"%s"' % c for c in keytext[1:-1]]
        else:
            lst_key = [keytext]
        for key in lst_key:
            keyinfo, event = keytext_to_keyinfo_and_event (key)
            dispatch_func = self.key_dispatch.get( keyinfo.tuple(),self.vi_key)
            self.tested_commands[dispatch_func.__name__]=dispatch_func
            dispatch_func (event)

    def vi_accept_line (self, e):
        if ViMode.vi_accept_line (self, e):
            # simulate return
            # self.add_history (self.line)
            self.l_buffer.reset_line ()

    def mock_completer (self, text, state):
        return self.lst_completions [state]

class ViExternalEditorTest (ViExternalEditor):
    def __init__ (self, line):
        import StringIO
        self.sio_write = StringIO.StringIO ()
        self.sio_read = StringIO.StringIO (u'qwerty after')
        ViExternalEditor.__init__ (self, line)

    def get_tempfile (self):
        return u'temp.py'

    def get_editor (self):
        return u'vim.exe'

    def file_open (self, filename, mode):
        if mode == u'w':
            return self.sio_write
        else:
            return self.sio_read

    def file_remove (self, filename):
        self.remove = filename

    def run_command (self, command):
        self.command = command

#----------------------------------------------------------------------

class Tests (unittest.TestCase):

    def test_keyinfo (self):
        keyinfo, event = keytext_to_keyinfo_and_event (u'"d"')
        self.assertEqual (u'd', event.char)
        keyinfo, event = keytext_to_keyinfo_and_event (u'"D"')
        self.assertEqual (u'D', event.char)
        keyinfo, event = keytext_to_keyinfo_and_event (u'"$"')
        self.assertEqual (u'$', event.char)
        keyinfo, event = keytext_to_keyinfo_and_event (u'Escape')
        self.assertEqual (u'\x1b', event.char)

    def test_simple (self):
        r = ViModeTest ()
        r._set_line (u'abc')
        self.assert_ (r.vi_is_insert_mode)
        r.input (u'"d"')
        self.assert_ (r.vi_is_insert_mode)
        self.assertEqual (u'abcd', r.line)
        r.input (u'Escape')
        self.assert_ (not r.vi_is_insert_mode)
        self.assertEqual (u'abcd', r.line)
        r.input (u'"i"')
        self.assert_ (r.vi_is_insert_mode)
        self.assertEqual (u'abcd', r.line)
        r.input (u'"x"')
        self.assertEqual (u'abcxd', r.line)
        r.input (u'"y"')
        self.assertEqual (u'abcxyd', r.line)

    def test_begin_and_end (self):
        r = ViModeTest ()
        r._set_line (u'abc')
        self.assertEqual (3, r.line_cursor)
        r.input (u'Escape')
        self.assertEqual (2, r.line_cursor)
        r.input (u'"0"')
        self.assertEqual (0, r.line_cursor)
        r.input (u'"$"')
        self.assertEqual (2, r.line_cursor)
        r.input (u'"^"')
        self.assertEqual (0, r.line_cursor)

    def test_history_alpha (self):
        r = ViModeTest ()
        r.add_history (u'abc')
        r.add_history (u'def')
        r._set_line (u'ghi')
        r.input (u'Escape')
        r.input (u'"k"')
        self.assertEqual (0, r.line_cursor)
        self.assertEqual (u'def', r.line)
        r.input (u'"k"')
        self.assertEqual (u'abc', r.line)
        self.assertEqual (0, r.console.bell_count)
        r.input (u'"k"')
        self.assertEqual (u'abc', r.line)
        # TODO up history with no more -> error ?
        # self.assertEqual (1, r.console.bell_count)
        r.input (u'"j"')
        self.assertEqual (u'def', r.line)
        self.assertEqual (0, r.line_cursor)
        r.input (u'"j"')
        self.assertEqual (u'ghi', r.line)
        # TODO returning to original line loses cursor pos?
        # self.assertEqual (2, r.line_cursor)
        r.input (u'"j"')
        self.assertEqual (u'ghi', r.line)
        self.assertEqual (2, r.line_cursor)

    def test_history_arrow (self):
        r = ViModeTest ()
        r.add_history (u'abc')
        r.add_history (u'def')
        r.input (u'"ghi"')
        self.assertEqual (3, r.line_cursor)
        r.input (u'Up')
        self.assertEqual (3, r.line_cursor)
        self.assertEqual (u'def', r.line)
        r.input (u'Up')
        self.assertEqual (3, r.line_cursor)
        self.assertEqual (u'abc', r.line)
        self.assertEqual (0, r.console.bell_count)
        r.input (u'Up')
        self.assertEqual (u'abc', r.line)
        # TODO up history with no more -> error ?
        # self.assertEqual (1, r.console.bell_count)
        r.input (u'Down')
        self.assertEqual (u'def', r.line)
        self.assertEqual (3, r.line_cursor)
        r.input (u'Down')
        self.assertEqual (u'ghi', r.line)
        self.assertEqual (3, r.line_cursor)
        r.input (u'Down')
        self.assertEqual (u'ghi', r.line)
        self.assertEqual (3, r.line_cursor)

    def test_history_arrow_after_undo (self):
        r = ViModeTest ()
        r.input (u'"aaa"')
        r.input (u'Return')
        r.input (u'"bbb ccc"')
        r.input (u'Escape')
        r.input (u'"F D"')
        self.assertEqual (r.line, u'bbb')
        r.input (u'"u"')
        self.assertEqual (r.line, u'bbb ccc')
        r.input (u'Return')
        self.assertEqual (r.console.bell_count, 0)
        r.input (u'"ddd"')
        r.input (u'Down')
        self.assertEqual (r.line, u'ddd')
        # TODO down history with no more -> error ?
        # self.assertEqual (r.console.bell_count, 1)

    def test_vi_is_word (self):
        self.assert_ (vi_is_word (u'a'))
        self.assert_ (not vi_is_word (u' u'))

    def test_pos_word_short (self):
        self.assertEqual (0, vi_pos_word_short (u''))
        self.assertEqual (4, vi_pos_word_short (u'abc def u'))
        self.assertEqual (4, vi_pos_word_short (u'abc def u', 2))
        self.assertEqual (4, vi_pos_word_short (u'abc def u', 3))
        self.assertEqual (8, vi_pos_word_short (u'abc def u', 4))
        self.assertEqual (3, vi_pos_word_short (u'abc.def u'))
        self.assertEqual (5, vi_pos_word_short (u'abc..def u', 3))
        self.assertEqual (8, vi_pos_word_short (u'ab  cd  ef  gh', count=2))
        self.assertEqual (8, vi_pos_word_short (u'ab  cd  ef  gh', 1, 2))
        self.assertEqual (8, vi_pos_word_short (u'ab  cd  ef  gh', 3, 2))
        self.assertEqual (14, vi_pos_word_short (u'ab  cd  ef  gh', count=10))

    def test_pos_word_long (self):
        self.assertEqual (0, vi_pos_word_long (u''))
        self.assertEqual (4, vi_pos_word_long (u'abc def u'))
        self.assertEqual (4, vi_pos_word_long (u'abc def u', 2))
        self.assertEqual (4, vi_pos_word_long (u'abc def u', 3))
        self.assertEqual (8, vi_pos_word_long (u'abc def u', 4))
        self.assertEqual (8, vi_pos_word_long (u'abc.def u'))
        self.assertEqual (9, vi_pos_word_long (u'abc..def u', 3))
        self.assertEqual (10, vi_pos_word_long (u'a.b  c.d  e.f  g.h', count=2))
        self.assertEqual (10, vi_pos_word_long (u'a.b  c.d  e.f  g.h', 2, 2))
        self.assertEqual (10, vi_pos_word_long (u'a.b  c.d  e.f  g.h', 4, 2))

    def test_pos_end_short (self):
        self.assertEqual (0, vi_pos_end_short (u''))
        self.assertEqual (2, vi_pos_end_short (u'abc def u'))
        self.assertEqual (6, vi_pos_end_short (u'abc def u', 2))
        self.assertEqual (6, vi_pos_end_short (u'abc def u', 3))
        self.assertEqual (6, vi_pos_end_short (u'abc def u', 4))
        self.assertEqual (2, vi_pos_end_short (u'abc.def u'))
        self.assertEqual (7, vi_pos_end_short (u'abc  ...  def u', 3))
        self.assertEqual (7, vi_pos_end_short (u'abc  ...  def u', 5))
        self.assertEqual (12, vi_pos_end_short (u'abc  ...  def u', 8))
        self.assertEqual (5, vi_pos_end_short (u'ab  cd  ef  gh', count=2))
        self.assertEqual (9, vi_pos_end_short (u'ab  cd  ef  gh', 1, 2))
        self.assertEqual (9, vi_pos_end_short (u'ab  cd  ef  gh', 3, 2))

    def test_pos_end_long (self):
        self.assertEqual (0, vi_pos_end_long (u''))
        self.assertEqual (2, vi_pos_end_long (u'abc def u'))
        self.assertEqual (6, vi_pos_end_long (u'abc def u', 2))
        self.assertEqual (6, vi_pos_end_long (u'abc def u', 3))
        self.assertEqual (6, vi_pos_end_long (u'abc def u', 4))
        self.assertEqual (6, vi_pos_end_long (u'abc.def u'))
        self.assertEqual (10, vi_pos_end_long (u'  abc...def u'))
        self.assertEqual (10, vi_pos_end_long (u'abc  ...def u', 5))
        self.assertEqual (7, vi_pos_end_long (u'a.b  c.d  e.f  g.h', count=2))
        self.assertEqual (12, vi_pos_end_long (u'a.b  c.d  e.f  g.h', 2, 2))
        self.assertEqual (12, vi_pos_end_long (u'a.b  c.d  e.f  g.h', 4, 2))

    def test_pos_back_short (self):
        self.assertEqual (0, vi_pos_back_short (u''))
        self.assertEqual (4, vi_pos_back_short (u'abc def', 6))
        self.assertEqual (4, vi_pos_back_short (u'abc def', 5))
        self.assertEqual (0, vi_pos_back_short (u'abc def', 4))
        self.assertEqual (0, vi_pos_back_short (u'abc def', 3))
        self.assertEqual (8, vi_pos_back_short (u'abc  ...def u', 11))
        self.assertEqual (5, vi_pos_back_short (u'abc  ...def u', 8))
        self.assertEqual (0, vi_pos_back_short (u'abc  ...def u', 5))
        self.assertEqual (0, vi_pos_back_short (u'abc  ...def u'))
        self.assertEqual (8, vi_pos_back_short (u'abc  def... u', 11))
        self.assertEqual (5, vi_pos_back_short (u'abc  def... u', 8))
        self.assertEqual (0, vi_pos_back_short (u'abc  def... u', 5))
        self.assertEqual (11, vi_pos_back_short (u'abc def... ghi...', 16, count=2))
        self.assertEqual (0, vi_pos_back_short (u'abc def... ghi...', 11, count=3))

    def test_pos_back_long (self):
        self.assertEqual (0, vi_pos_back_long (u''))
        self.assertEqual (4, vi_pos_back_long (u'abc def', 6))
        self.assertEqual (4, vi_pos_back_long (u'abc def', 5))
        self.assertEqual (0, vi_pos_back_long (u'abc def', 4))
        self.assertEqual (0, vi_pos_back_long (u'abc def', 3))
        self.assertEqual (5, vi_pos_back_long (u'abc  ...def u', 11))
        self.assertEqual (0, vi_pos_back_long (u'abc  ...def u', 5))
        self.assertEqual (0, vi_pos_back_long (u'abc  ...def u'))
        self.assertEqual (5, vi_pos_back_long (u'abc  def... u', 11))
        self.assertEqual (0, vi_pos_back_long (u'abc  def... u', 5))
        self.assertEqual (4, vi_pos_back_long (u'abc def... ghi...', 16, count=2))

    def test_pos_find_char_forward (self):
        self.assertEqual (-1, vi_pos_find_char_forward (u'', u'x'))
        self.assertEqual (-1, vi_pos_find_char_forward (u'abc def', u'x'))
        self.assertEqual (4, vi_pos_find_char_forward (u'abc def', u'd'))
        self.assertEqual (4, vi_pos_find_char_forward (u'abc def', u'd', 3))
        self.assertEqual (-1, vi_pos_find_char_forward (u'abc def', u'd', 4))
        self.assertEqual (-1, vi_pos_find_char_forward (u'abc def', u'd', count=2))
        self.assertEqual (12, vi_pos_find_char_forward (u'abc def abc def', u'd', count=2))

    def test_pos_find_char_backward (self):
        self.assertEqual (-1, vi_pos_find_char_backward (u'', u'x'))
        self.assertEqual (-1, vi_pos_find_char_backward (u'abc def', u'x', 6))
        self.assertEqual (4, vi_pos_find_char_backward (u'abc def', u'd', 6))
        self.assertEqual (4, vi_pos_find_char_backward (u'abc def', u'd', 5))
        self.assertEqual (-1, vi_pos_find_char_backward (u'abc def', u'd', 4))
        self.assertEqual (-1, vi_pos_find_char_backward (u'abc def', u'd', 6, count=2))
        self.assertEqual (4, vi_pos_find_char_backward (u'abc def abc def', u'd', 14, count=2))

    def test_pos_to_char_forward (self):
        self.assertEqual (-1, vi_pos_to_char_forward (u'', u'x'))
        self.assertEqual (-1, vi_pos_to_char_forward (u'abc def', u'x'))
        self.assertEqual (3,  vi_pos_to_char_forward (u'abc def', u'd'))
        self.assertEqual (3,  vi_pos_to_char_forward (u'abc def', u'd', 2))
        self.assertEqual (-1, vi_pos_to_char_forward (u'abc def', u'd', 4))
        self.assertEqual (-1, vi_pos_to_char_forward (u'abc def', u'd', count=2))
        self.assertEqual (11, vi_pos_to_char_forward (u'abc def abc def', u'd', count=2))

    def test_pos_to_char_backward (self):
        self.assertEqual (-1, vi_pos_to_char_backward (u'', u'x'))
        self.assertEqual (-1, vi_pos_to_char_backward (u'abc def', u'x', 6))
        self.assertEqual (5,  vi_pos_to_char_backward (u'abc def', u'd', 6))
        self.assertEqual (5,  vi_pos_to_char_backward (u'abc def', u'd', 5))
        self.assertEqual (-1, vi_pos_to_char_backward (u'abc def', u'd', 4))
        self.assertEqual (-1, vi_pos_to_char_backward (u'abc def', u'd', 6, count=2))
        self.assertEqual (5,  vi_pos_to_char_backward (u'abc def abc def', u'd', 14, count=2))

    def test_motion_word (self):
        u'''motions: lowercase mode is alpha, digit and _, uppercase is delim by spaces
        w/W: forward short/long word'''
        r = ViModeTest ()
        r._set_line (u'abc_123  def--456.789  x')
        r.input (u'Escape')
        r.input (u'"0"')
        r.input (u'"w"')
        self.assertEqual (9, r.line_cursor)
        r.input (u'"w"')
        self.assertEqual (12, r.line_cursor)
        r.input (u'"w"')
        self.assertEqual (14, r.line_cursor)
        r.input (u'"W"')
        self.assertEqual (23, r.line_cursor)

    def test_motion_word_multiplier (self):
        r = ViModeTest ()
        r._set_line (u'a b c d e f g h i j k l m n o p q r s t u v w x y z')
        r.input (u'Escape')
        r.input (u'"0"')
        r.input (u'"2"')
        self.assertEqual (0, r.line_cursor)
        r.input (u'"w"')
        self.assertEqual (4, r.line_cursor)
        r.input (u'"2"')
        r.input (u'"0"')
        r.input (u'"w"')
        self.assertEqual (44, r.line_cursor)
        r.input (u'"2"')
        r.input (u'"W"')
        self.assertEqual (48, r.line_cursor)

    def test_motion_end (self):
        u'''motions: lowercase mode is alpha, digit and _, uppercase is delim by spaces
        e/E: to end of short/long word'''
        r = ViModeTest ()
        r._set_line (u'  abc_123  --def--456.789  x')
        r.input (u'Escape')
        r.input (u'"0"')
        r.input (u'"e"')
        self.assertEqual (8, r.line_cursor)
        r.input (u'"e"')
        self.assertEqual (12, r.line_cursor)
        r.input (u'"e"')
        self.assertEqual (15, r.line_cursor)
        r.input (u'"E"')
        self.assertEqual (24, r.line_cursor)

    def test_motion_end_multiplier (self):
        r = ViModeTest ()
        r._set_line (u'ab cd ef gh ij kl mn op qr st uv wx yz')
        r.input (u'Escape')
        r.input (u'"0"')
        r.input (u'"3"')
        r.input (u'"e"')
        self.assertEqual (7, r.line_cursor)
        r.input (u'"4"')
        r.input (u'"E"')
        self.assertEqual (19, r.line_cursor)

    def test_motion_backward (self):
        u'''motions: lowercase mode is alpha, digit and _, uppercase is delim by spaces
        b/B: backward short/long word'''
        r = ViModeTest ()
        r._set_line (u'abc_123  def--456.789  x')
        r.input (u'Escape')
        r.input (u'"$"')
        self.assertEqual (23, r.line_cursor)
        r.input (u'"b"')
        self.assertEqual (18, r.line_cursor)
        r.input (u'"b"')
        self.assertEqual (17, r.line_cursor)
        r.input (u'"B"')
        self.assertEqual (9, r.line_cursor)
        r.input (u'"B"')
        self.assertEqual (0, r.line_cursor)

    def test_motion_backward_multiplier (self):
        r = ViModeTest ()
        r._set_line (u'ab cd ef gh ij kl mn op qr st uv wx yz')
        r.input (u'Escape')
        r.input (u'"$"')
        self.assertEqual (37, r.line_cursor)
        r.input (u'"3"')
        r.input (u'"b"')
        self.assertEqual (30, r.line_cursor)
        r.input (u'"5"')
        r.input (u'"b"')
        self.assertEqual (15, r.line_cursor)

    def test_motion_find_char_forward (self):
        r = ViModeTest ()
        r._set_line (u'abc_123  def--456.789  x')
        r.input (u'Escape')
        r.input (u'"0"')
        r.input (u'"f"')
        self.assertEqual (0, r.line_cursor)
        r.input (u'"c"')
        self.assertEqual (2, r.line_cursor)

    def test_motion_find_char_backward (self):
        r = ViModeTest ()
        r._set_line (u'abc_123  def--456.789  x')
        r.input (u'Escape')
        r.input (u'"$"')
        self.assertEqual (23, r.line_cursor)
        r.input (u'"F"')
        self.assertEqual (23, r.line_cursor)
        r.input (u'"c"')
        self.assertEqual (2, r.line_cursor)

    def test_motion_find_char_forward_multiplier (self):
        r = ViModeTest ()
        r._set_line (u'ab cd ef 01 23 45 ab cd ef 01 23 45 ab cd ef 01 23 45')
        r.input (u'Escape')
        r.input (u'"0"')
        r.input (u'"2"')
        r.input (u'"f"')
        r.input (u'"0"')
        self.assertEqual (27, r.line_cursor)

    def test_motion_find_char_backward_multiplier (self):
        r = ViModeTest ()
        r._set_line (u'ab cd ef 01 23 45 ab cd ef 01 23 45 ab cd ef 01 23 45')
        r.input (u'Escape')
        r.input (u'"$"')
        self.assertEqual (52, r.line_cursor)
        r.input (u'"2"')
        r.input (u'"F"')
        r.input (u'"0"')
        self.assertEqual (27, r.line_cursor)

    def test_motion_find_char_again (self):
        r = ViModeTest ()
        r._set_line (u'1234512345123451234512345')
        r.input (u'Escape')
        r.input (u'"0"')
        r.input (u'"2"')
        r.input (u'"f"')
        r.input (u'"3"')
        self.assertEqual (7, r.line_cursor)
        r.input (u'";"')
        self.assertEqual (12, r.line_cursor)
        r.input (u'"2"')
        r.input (u'";"')
        self.assertEqual (22, r.line_cursor)
        r.input (u'","')
        self.assertEqual (17, r.line_cursor)
        r.input (u'"2"')
        r.input (u'","')
        self.assertEqual (7, r.line_cursor)

    def test_motion_find_char_opposite (self):
        r = ViModeTest ()
        r._set_line (u'1234512345123451234512345')
        r.input (u'Escape')
        r.input (u'"$"')
        r.input (u'"2"')
        r.input (u'"F"')
        r.input (u'"3"')
        self.assertEqual (17, r.line_cursor)
        r.input (u'";"')
        self.assertEqual (12, r.line_cursor)
        r.input (u'"2"')
        r.input (u'";"')
        self.assertEqual (2, r.line_cursor)
        r.input (u'","')
        self.assertEqual (7, r.line_cursor)
        r.input (u'"2"')
        r.input (u'","')
        self.assertEqual (17, r.line_cursor)
        r.input (u'"2"')
        r.input (u'","')
        self.assertEqual (17, r.line_cursor)

    def test_motion_to_char_forward (self):
        r = ViModeTest ()
        r._set_line (u'abc_123  def--456.789  x')
        r.input (u'Escape')
        r.input (u'"0"')
        r.input (u'"t"')
        self.assertEqual (0, r.line_cursor)
        r.input (u'"c"')
        self.assertEqual (1, r.line_cursor)

    def test_motion_to_char_backward (self):
        r = ViModeTest ()
        r._set_line (u'abc_123  def--456.789  x')
        r.input (u'Escape')
        r.input (u'"$"')
        self.assertEqual (23, r.line_cursor)
        r.input (u'"T"')
        self.assertEqual (23, r.line_cursor)
        r.input (u'"c"')
        self.assertEqual (3, r.line_cursor)

    def test_motion_to_char_forward_multiplier (self):
        r = ViModeTest ()
        r._set_line (u'ab cd ef 01 23 45 ab cd ef 01 23 45 ab cd ef 01 23 45')
        r.input (u'Escape')
        r.input (u'"0"')
        r.input (u'"2"')
        r.input (u'"t"')
        r.input (u'"0"')
        self.assertEqual (26, r.line_cursor)

    def test_motion_to_char_backward_multiplier (self):
        r = ViModeTest ()
        r._set_line (u'ab cd ef 01 23 45 ab cd ef 01 23 45 ab cd ef 01 23 45')
        r.input (u'Escape')
        r.input (u'"$"')
        self.assertEqual (52, r.line_cursor)
        r.input (u'"2"')
        r.input (u'"T"')
        r.input (u'"0"')
        self.assertEqual (28, r.line_cursor)

    def test_delete_word (self):
        r = ViModeTest ()
        r._set_line (u'abc de fghi jkl mnopq rst')
        r.input (u'Escape')
        r.input (u'"0"')
        r.input (u'"d"')
        r.input (u'"w"')
        self.assertEqual (0, r.line_cursor)
        self.assertEqual (r.line, u'de fghi jkl mnopq rst')
        r.input (u'"d"')
        r.input (u'"2"')
        r.input (u'"w"')
        self.assertEqual (0, r.line_cursor)
        self.assertEqual (r.line, u'jkl mnopq rst')
        r.input (u'"2"')
        r.input (u'"d"')
        r.input (u'"w"')
        self.assertEqual (0, r.line_cursor)
        self.assertEqual (r.line, u'rst')

    def test_delete_word_two_multipliers (self):
        r = ViModeTest ()
        r._set_line (u'abc de fghi jkl mnopq rst uv wx yz')
        r.input (u'Escape')
        r.input (u'"0w"')
        r.input (u'"2d3w"')
        self.assertEqual (4, r.line_cursor)
        self.assertEqual (r.line, u'abc wx yz')

    def test_delete_find_char_forward_two_multipliers (self):
        r = ViModeTest ()
        r._set_line (u'0123456789012345678901234567890123456789012345678901234567890123456789')
        r.input (u'Escape')
        r.input (u'"0"')
        r.input (u'"2d3f4"')
        self.assertEqual (r.line, u'567890123456789')

    def test_delete_end_of_line (self):
        r = ViModeTest ()
        r._set_line (u'abc de fghi jkl mnopq rst uv wx yz')
        r.input (u'Escape')
        r.input (u'"0w"')
        r.input (u'"D"')
        self.assertEqual (r.line, u'abc ')

    def test_two_lines (self):
        r = ViModeTest ()
        r.input (u'"abcdef"')
        self.assertEqual (r.line, u'abcdef')
        r.input (u'Escape')
        r.input (u'"0iqq"')
        self.assertEqual (r.line, u'qqabcdef')
        r.input (u'Return')
        self.assertEqual (r.line, u'')
        r.input (u'"xyz"')
        self.assertEqual (r.line, u'xyz')

    def test_delete_word_short_to_end_of_line (self):
        r = ViModeTest ()
        r._set_line (u'abc def ghi')
        r.input (u'Escape')
        r.input (u'"03dw"')
        self.assertEqual (u'', r.line)
        r._set_line (u'abc def ghi  ')
        r.input (u'Escape')
        r.input (u'"03dw"')
        self.assertEqual (u'', r.line)

    def test_delete_word_long_to_end_of_line (self):
        r = ViModeTest ()
        r._set_line (u'a.c d.f g.i')
        r.input (u'Escape')
        r.input (u'"03dW"')
        self.assertEqual (u'', r.line)
        r._set_line (u'a.c d.f g.i  ')
        r.input (u'Escape')
        r.input (u'"03dW"')
        self.assertEqual (u'', r.line)

    def test_delete_end_short_to_end_of_line (self):
        r = ViModeTest ()
        r._set_line (u'abc def ghi')
        r.input (u'Escape')
        r.input (u'"03de"')
        self.assertEqual (u'', r.line)
        r._set_line (u'abc def ghi  ')
        r.input (u'Escape')
        r.input (u'"03de"')
        self.assertEqual (u'  ', r.line)

    def test_delete_end_long_to_end_of_line (self):
        r = ViModeTest ()
        r._set_line (u'a.c d.f g.i')
        r.input (u'Escape')
        r.input (u'"03dE"')
        self.assertEqual (u'', r.line)
        r._set_line (u'a.c d.f g.i  ')
        r.input (u'Escape')
        r.input (u'"03dE"')
        self.assertEqual (u'  ', r.line)

    def test_delete_back_short_to_begining_of_line (self):
        r = ViModeTest ()
        r._set_line (u'abc def ghi')
        r.input (u'Escape')
        r.input (u'"$3db"')
        self.assertEqual (u'i', r.line)
        r._set_line (u'abc def ghi  ')
        r.input (u'Escape')
        r.input (u'"$3db"')
        self.assertEqual (u' ', r.line)

    def test_delete_back_long_to_begining_of_line (self):
        r = ViModeTest ()
        r._set_line (u'a.c d.f g.i')
        r.input (u'Escape')
        r.input (u'"$3dB"')
        self.assertEqual (u'i', r.line)
        r._set_line (u'a.c d.f g.i  ')
        r.input (u'Escape')
        r.input (u'"$3dB"')
        self.assertEqual (u' ', r.line)

    def test_delete_dollar (self):
        r = ViModeTest ()
        r._set_line (u'abc def')
        r.input (u'Escape')
        r.input (u'"0ld$"')
        self.assertEqual (r.line, u'a')
        self.assertEqual (r.line_cursor, 0)

    def test_motion_left (self):
        r = ViModeTest ()
        r._set_line (u'abc def ghi')
        r.input (u'Escape')
        r.input (u'"$"')
        self.assertEqual (10, r.line_cursor)
        r.input (u'"h"')
        self.assertEqual (9, r.line_cursor)
        r.input (u'"2h"')
        self.assertEqual (7, r.line_cursor)
        r.input (u'"2d3h"')
        self.assertEqual (1, r.line_cursor)
        self.assertEqual (u'a ghi', r.line)
        r.input (u'"4dh"')
        self.assertEqual (0, r.line_cursor)
        self.assertEqual (u' ghi', r.line)

    def test_motion_right (self):
        r = ViModeTest ()
        r.input (u'Escape')
        self.assertEqual (r.line, u'')
        self.assertEqual (r.line_cursor, 0)
        r.input (u'"a"')
        self.assertEqual (r.line, u'')
        self.assertEqual (r.line_cursor, 0)
        r.input (u'"abc"')
        self.assertEqual (r.line, u'abc')
        self.assertEqual (r.line_cursor, 3)
        r.input (u'Escape')
        self.assertEqual (r.line, u'abc')
        self.assertEqual (r.line_cursor, 2)
        r.input (u'"l"')
        self.assertEqual (r.line, u'abc')
        self.assertEqual (r.line_cursor, 2)
        r.input (u'Left')
        self.assertEqual (r.line, u'abc')
        self.assertEqual (r.line_cursor, 1)
        r.input (u'"l"')
        self.assertEqual (r.line, u'abc')
        self.assertEqual (r.line_cursor, 2)
        r.input (u'"l"')
        self.assertEqual (r.line, u'abc')
        self.assertEqual (r.line_cursor, 2)

    def test_motion_right_delete (self):
        r = ViModeTest ()
        r._set_line (u'abc def ghi')
        r.input (u'Escape')
        r.input (u'"0"')
        self.assertEqual (0, r.line_cursor)
        r.input (u'"l"')
        self.assertEqual (1, r.line_cursor)
        r.input (u'"2l"')
        self.assertEqual (3, r.line_cursor)
        r.input (u'"2d3l"')
        self.assertEqual (3, r.line_cursor)
        self.assertEqual (u'abchi', r.line)
        r.input (u'"4dl"')
        self.assertEqual (2, r.line_cursor)
        self.assertEqual (u'abc', r.line)

    def test_backspace_motion (self):
        r = ViModeTest ()
        r._set_line (u'abc def ghi')
        r.input (u'Escape')
        r.input (u'"$"')
        self.assertEqual (10, r.line_cursor)
        r.input (u'BackSpace')
        self.assertEqual (9, r.line_cursor)
        r.input (u'"2"')
        r.input (u'BackSpace')
        self.assertEqual (7, r.line_cursor)
        r.input (u'"2d3"')
        r.input (u'BackSpace')
        self.assertEqual (1, r.line_cursor)
        self.assertEqual (u'a ghi', r.line)
        r.input (u'"4d"')
        r.input (u'BackSpace')
        self.assertEqual (0, r.line_cursor)
        self.assertEqual (u' ghi', r.line)

    def test_backspace_insert (self):
        r = ViModeTest ()
        r._set_line (u'abc def ghi')
        r.input (u'Escape')
        r.input (u'"$"')
        self.assertEqual (10, r.line_cursor)
        r.input (u'"i"')
        self.assertEqual (10, r.line_cursor)
        r.input (u'BackSpace')
        self.assertEqual (9, r.line_cursor)
        self.assertEqual (u'abc def gi', r.line)

    def test_insert_lower_i (self):
        r = ViModeTest ()
        r._set_line (u'abc def ghi')
        r.input (u'Escape')
        r.input (u'"0w"')
        r.input (u'"i"')
        r.input (u'"zz"')
        self.assert_ (r.vi_is_insert_mode)
        self.assertEqual (r.line, u'abc zzdef ghi')
        r.input (u'Escape')
        r.input (u'"w"')
        r.input (u'"2iyy"')
        r.input (u'Escape')
        self.assertEqual (r.line, u'abc zzdef yyyyghi')

    def test_insert_upper_i (self):
        r = ViModeTest ()
        r._set_line (u'abc def ghi')
        r.input (u'Escape')
        r.input (u'"0w"')
        r.input (u'"I"')
        r.input (u'"zz"')
        self.assert_ (r.vi_is_insert_mode)
        self.assertEqual (r.line, u'zzabc def ghi')
        r.input (u'Escape')
        r.input (u'"w"')
        r.input (u'"2Iyy"')
        r.input (u'Escape')
        self.assertEqual (r.line, u'yyyyzzabc def ghi')

    def test_append_lower_a (self):
        r = ViModeTest ()
        r._set_line (u'abc def ghi')
        r.input (u'Escape')
        r.input (u'"0"')
        r.input (u'"a"')
        r.input (u'"zz"')
        self.assertEqual (r.line, u'azzbc def ghi')
        self.assert_ (r.vi_is_insert_mode)
        r.input (u'Escape')
        r.input (u'"w"')
        r.input (u'"2ayy"')
        r.input (u'Escape')
        self.assertEqual (r.line, u'azzbc dyyyyef ghi')

    def test_append_upper_a_simple (self):
        r = ViModeTest ()
        r._set_line (u'')
        r.input (u'Escape')
        r.input (u'"2A"')
        r.input (u'"jj"')
        r.input (u'Escape')
        self.assertEqual (r.line, u'jjjj')
        self.assert_ (not r.vi_is_insert_mode)

    def test_append_upper_a (self):
        r = ViModeTest ()
        r._set_line (u'abc def ghi')
        r.input (u'Escape')
        r.input (u'"0"')
        r.input (u'"A"')
        r.input (u'"zz"')
        self.assertEqual (r.line, u'abc def ghizz')
        self.assert_ (r.vi_is_insert_mode)
        r.input (u'Escape')
        r.input (u'"0w"')
        r.input (u'"2Ayy"')
        r.input (u'Escape')
        self.assertEqual (r.line, u'abc def ghizzyyyy')

    def test_delete_lower_x (self):
        r = ViModeTest ()
        r._set_line (u'abc def')
        r.input (u'Escape')
        r.input (u'"0w"')
        r.input (u'"x"')
        self.assertEqual (r.line, u'abc ef')
        r.input (u'"4x"')
        self.assertEqual (r.line, u'abc ')
        self.assertEqual (r.line_cursor, 3)
        r.input (u'"x"')
        self.assertEqual (r.line, u'abc')
        self.assertEqual (r.line_cursor, 2)
        r.input (u'"x"')
        self.assertEqual (r.line, u'ab')
        self.assertEqual (r.line_cursor, 1)
        r.input (u'"x"')
        self.assertEqual (r.line, u'a')
        self.assertEqual (r.line_cursor, 0)
        r.input (u'"x"')
        self.assertEqual (r.line, u'')
        self.assertEqual (r.line_cursor, 0)
        r.input (u'"x"')
        self.assertEqual (r.line, u'')
        self.assertEqual (r.line_cursor, 0)

    def test_delete_upper_x (self):
        r = ViModeTest ()
        r._set_line (u'abc def')
        self.assertEqual (r.line_cursor, 7)
        r.input (u'Escape')
        self.assertEqual (r.line_cursor, 6)
        r.input (u'"$"')
        self.assertEqual (r.line_cursor, 6)
        r.input (u'"X"')
        self.assertEqual (r.line, u'abc df')
        self.assertEqual (r.line_cursor, 5)
        r.input (u'"4X"')
        self.assertEqual (r.line, u'af')
        self.assertEqual (r.line_cursor, 1)
        r.input (u'"2X"')
        self.assertEqual (r.line, u'f')
        self.assertEqual (r.line_cursor, 0)
        r.input (u'"X"')
        self.assertEqual (r.line, u'f')
        self.assertEqual (r.line_cursor, 0)

    def test_substitute_lower_s (self):
        r = ViModeTest ()
        r._set_line (u'abc def')
        r.input (u'Escape')
        r.input (u'"0"')
        r.input (u'"s"')
        r.input (u'"qq"')
        r.input (u'Escape')
        self.assertEqual (r.line, u'qqbc def')
        self.assertEqual (r.line_cursor, 1)
        r.input (u'"3s"')
        r.input (u'"yyy"')
        r.input (u'Escape')
        self.assertEqual (r.line, u'qyyy def')
        self.assertEqual (r.line_cursor, 3)
        r.input (u'"w"')
        r.input (u'"5"')
        r.input (u'"s"')
        r.input (u'"zz"')
        r.input (u'Escape')
        self.assertEqual (r.line, u'qyyy zz')
        self.assertEqual (r.line_cursor, 6)

    def test_change_to_end_of_line (self):
        r = ViModeTest ()
        r._set_line (u'abc def ghi')
        r.input (u'Escape')
        r.input (u'"0w"')
        r.input (u'"C"')
        self.assert_ (r.vi_is_insert_mode)
        r.input (u'"123"')
        self.assertEqual (r.line, u'abc 123')
        r.input (u'Escape')
        self.assert_ (not r.vi_is_insert_mode)

    def test_change_whole_line (self):
        r = ViModeTest ()
        r._set_line (u'abc def ghi')
        r.input (u'Escape')
        r.input (u'"0w"')
        r.input (u'"S"')
        self.assert_ (r.vi_is_insert_mode)
        self.assertEqual (r.line, u'')
        r.input (u'"123"')
        self.assertEqual (r.line, u'123')
        r.input (u'Escape')
        self.assert_ (not r.vi_is_insert_mode)

    def test_change_word_short (self):
        r = ViModeTest ()
        r._set_line (u'abc def ghi')
        r.input (u'Escape')
        r.input (u'"0cwzzz"')
        self.assert_ (r.vi_is_insert_mode)
        self.assertEqual (r.line, u'zzz def ghi')
        self.assertEqual (r.line_cursor, 3)
        r.input (u'Escape')
        self.assert_ (not r.vi_is_insert_mode)
        self.assertEqual (r.line_cursor, 2)
        r.input (u'"w"')
        self.assertEqual (r.line_cursor, 4)
        r.input (u'"2cwyyy"')
        self.assertEqual (r.line, u'zzz yyy')
        r.input (u'Escape')
        self.assertEqual (r.line, u'zzz yyy')

    def test_change_word_long (self):
        r = ViModeTest ()
        r._set_line (u'abc.def ghi.jkl mno.pqr')
        r.input (u'Escape')
        r.input (u'"0cWss"')
        self.assertEqual (r.line, u'ss ghi.jkl mno.pqr')
        self.assert_ (r.vi_is_insert_mode)
        self.assertEqual (r.line_cursor, 2)
        r.input (u'Escape')
        self.assert_ (not r.vi_is_insert_mode)
        self.assertEqual (r.line_cursor, 1)
        r.input (u'"w2."')
        self.assertEqual (r.line, u'ss ss')
        self.assert_ (not r.vi_is_insert_mode)
        self.assertEqual (r.line_cursor, 4)

    def test_change_end_short (self):
        r = ViModeTest ()
        r._set_line (u'abc def ghi')
        r.input (u'Escape')
        r.input (u'"0cezzz"')
        self.assert_ (r.vi_is_insert_mode)
        self.assertEqual (r.line, u'zzz def ghi')
        self.assertEqual (r.line_cursor, 3)
        r.input (u'Escape')
        self.assert_ (not r.vi_is_insert_mode)
        self.assertEqual (r.line_cursor, 2)
        r.input (u'"w2."')
        self.assertEqual (r.line, u'zzz zzz')

    def test_change_end_long (self):
        r = ViModeTest ()
        r._set_line (u'abc.def ghi jkl.mno pqr stu.vwx')
        r.input (u'Escape')
        r.input (u'"02cEzz"')
        self.assert_ (r.vi_is_insert_mode)
        self.assertEqual (r.line, u'zz jkl.mno pqr stu.vwx')
        self.assertEqual (r.line_cursor, 2)
        r.input (u'Escape')
        self.assert_ (not r.vi_is_insert_mode)
        self.assertEqual (r.line_cursor, 1)
        r.input (u'"w2."')
        self.assertEqual (r.line, u'zz zz stu.vwx')
        self.assertEqual (r.line_cursor, 5)

    def test_change_back_short (self):
        r = ViModeTest ()
        r._set_line (u'abc def ghi')
        r.input (u'Escape')
        r.input (u'"$cbzz"')
        self.assert_ (r.vi_is_insert_mode)
        self.assertEqual (r.line, u'abc def zzi')
        self.assertEqual (r.line_cursor, 10)
        r.input (u'Escape')
        self.assert_ (not r.vi_is_insert_mode)
        self.assertEqual (r.line_cursor, 9)
        r.input (u'"b2."')
        self.assertEqual (r.line, u'zzzzi')
        self.assertEqual (r.line_cursor, 2)

    def test_change_back_long (self):
        r = ViModeTest ()
        r._set_line (u'abc.def ghi jkl.mno pqr stu.vwx')
        r.input (u'Escape')
        r.input (u'"$2cBzz"')
        self.assert_ (r.vi_is_insert_mode)
        self.assertEqual (r.line, u'abc.def ghi jkl.mno zzx')
        self.assertEqual (r.line_cursor, 22)
        r.input (u'Escape')
        self.assert_ (not r.vi_is_insert_mode)
        self.assertEqual (r.line_cursor, 21)
        r.input (u'"5."')
        self.assertEqual (r.line, u'zzzx')
        self.assertEqual (r.line_cursor, 2)
        self.assert_ (not r.vi_is_insert_mode)

    def test_change_find_lower (self):
        r = ViModeTest ()
        r._set_line (u'aa bb cc dd ee aa bb cc dd ee')
        r.input (u'Escape')
        r.input (u'"0cfbzz"')
        self.assert_ (r.vi_is_insert_mode)
        self.assertEqual (r.line, u'zzb cc dd ee aa bb cc dd ee')
        self.assertEqual (r.line_cursor, 2)
        r.input (u'Escape')
        self.assert_ (not r.vi_is_insert_mode)
        self.assertEqual (r.line_cursor, 1)
        r.input (u'"c2fcyy"')
        self.assertEqual (r.line, u'zyy dd ee aa bb cc dd ee')
        r.input (u'Escape')
        r.input (u'"."')
        self.assertEqual (r.line, u'zyyy dd ee')

    def test_change_find_upper (self):
        r = ViModeTest ()
        r._set_line (u'aa bb cc aa bb cc')
        r.input (u'Escape')
        r.input (u'"$2c2Fazz"')
        self.assert_ (r.vi_is_insert_mode)
        self.assertEqual (r.line, u'zzc')
        self.assertEqual (r.line_cursor, 2)
        r.input (u'Escape')
        self.assert_ (not r.vi_is_insert_mode)
        self.assertEqual (r.line_cursor, 1)
        r.input (u'"."')
        self.assertEqual (r.line, u'zzc')

    def test_change_to_lower (self):
        r = ViModeTest ()
        r._set_line (u'aa bb cc aa bb cc aa bb cc')
        r.input (u'Escape')
        r.input (u'"02c2ta"')
        self.assert_ (r.vi_is_insert_mode)
        self.assertEqual (r.line, u'aa bb cc')
        self.assertEqual (r.line_cursor, 0)
        r.input (u'"zz "')
        self.assertEqual (r.line, u'zz aa bb cc')
        self.assertEqual (r.line_cursor, 3)
        r.input (u'Escape')
        self.assert_ (not r.vi_is_insert_mode)
        self.assertEqual (r.line_cursor, 2)

    def test_change_to_upper (self):
        r = ViModeTest ()
        r._set_line (u'aa bb cc aa bb cc aa bb cc')
        r.input (u'Escape')
        r.input (u'"$2c2Ta"')
        self.assert_ (r.vi_is_insert_mode)
        self.assertEqual (r.line, u'aa bb cc ac')
        self.assertEqual (r.line_cursor, 10)
        r.input (u'"zz"')
        self.assertEqual (r.line, u'aa bb cc azzc')
        self.assertEqual (r.line_cursor, 12)
        r.input (u'Escape')
        self.assert_ (not r.vi_is_insert_mode)
        self.assertEqual (r.line_cursor, 11)
        r.input (u'"3."')
        self.assertEqual (r.line, u'azzzc')
        # The following fails but it does not seem that important
        # self.assertEqual (r.line_cursor, 2)
        self.assert_ (not r.vi_is_insert_mode)

    def test_pos_matching (self):
        self.assertEqual (6, vi_pos_matching (u'aa (bb)'))
        self.assertEqual (6, vi_pos_matching (u'aa (bb)', 3))
        self.assertEqual (3, vi_pos_matching (u'aa (bb)', 6))
        self.assertEqual (11, vi_pos_matching (u'aa (bb (cc))'))
        self.assertEqual (3, vi_pos_matching (u'aa (bb (cc))', 11))
        self.assertEqual (10, vi_pos_matching (u'aa (bb (cc))', 4))
        self.assertEqual (7, vi_pos_matching (u'aa (bb (cc))', 10))
        self.assertEqual (7, vi_pos_matching (u'aa (bb (cc))', 8))
        self.assertEqual (3, vi_pos_matching (u'aa (bb (cc) dd)', 12))
        self.assertEqual (3, vi_pos_matching (u'aa (bb (cc) dd)', 14))
        self.assertEqual (-1, vi_pos_matching (u'aa ((bb (cc) dd)', 3))
        self.assertEqual (-1, vi_pos_matching (u'aa (bb (cc) dd) ee)', 16))
        self.assertEqual (-1, vi_pos_matching (u'aa (bb (cc) dd) ee)', 18))
        self.assertEqual (6, vi_pos_matching (u'aa <bb>'))
        self.assertEqual (11, vi_pos_matching (u'aa <bb <cc>>'))
        self.assertEqual (10, vi_pos_matching (u'aa <bb <cc>>', 4))
        self.assertEqual (6, vi_pos_matching (u'aa {bb}'))
        self.assertEqual (11, vi_pos_matching (u'aa {bb {cc}}'))
        self.assertEqual (10, vi_pos_matching (u'aa {bb {cc}}', 4))
        self.assertEqual (6, vi_pos_matching (u'aa [bb]'))
        self.assertEqual (11, vi_pos_matching (u'aa [bb [cc]]'))
        self.assertEqual (10, vi_pos_matching (u'aa [bb [cc]]', 4))

    def test_matching_paren_forward (self):
        r = ViModeTest ()
        r._set_line (u'abc (def (ghi)) jkl')
        r.input (u'Escape')
        r.input (u'"0w"')
        r.input (u'"d"')
        r.input (u'"%"')
        self.assertEqual (r.line, u'abc  jkl')
        self.assertEqual (r.line_cursor, 4)

    def test_matching_paren_backward (self):
        r = ViModeTest ()
        r._set_line (u'abc (def (ghi)) jkl')
        r.input (u'Escape')
        r.input (u'"0w"')
        r.input (u'"%"')
        self.assertEqual (r.line_cursor, 14)
        r.input (u'"d"')
        r.input (u'"%"')
        self.assertEqual (r.line, u'abc  jkl')
        self.assertEqual (r.line_cursor, 4)

    def test_yank_and_put (self):
        r = ViModeTest ()
        r._set_line (u'abc def')
        r.input (u'Escape')
        r.input (u'"0"')
        self.assert_ (not r.vi_is_insert_mode)
        r.input (u'"yw"')
        self.assert_ (not r.vi_is_insert_mode)
        self.assertEqual (r.line, u'abc def')
        self.assertEqual (r.line_cursor, 0)
        r.input (u'"P"')
        self.assertEqual (r.line, u'abc abc def')
        self.assertEqual (r.line_cursor, 3)
        r.input (u'"p"')
        self.assertEqual (r.line, u'abc abc abc def')
        self.assertEqual (r.line_cursor, 7)

    def test_put_multiple (self):
        r = ViModeTest ()
        r._set_line (u'001122')
        r.input (u'Escape')
        r.input (u'"0"')
        r.input (u'"y3l"')
        self.assert_ (not r.vi_is_insert_mode)
        self.assertEqual (r.line_cursor, 0)
        r.input (u'"2P"')
        self.assertEqual (r.line, u'001001001122')
        self.assertEqual (r.line_cursor, 5)
        r.input (u'"f2"')
        r.input (u'"3p"')
        self.assertEqual (r.line, u'001001001120010010012')
        self.assertEqual (r.line_cursor, 19)

    def test_put_undo (self):
        r = ViModeTest ()
        r._set_line (u'aaa b ccc')
        r.input (u'Escape')
        r.input (u'"0ywwp"')
        self.assertEqual (r.line, u'aaa baaa  ccc')
        r.input (u'"u"')
        self.assertEqual (r.line, u'aaa b ccc')
        r.input (u'"P"')
        self.assertEqual (r.line, u'aaa aaa b ccc')
        r.input (u'"u"')
        self.assertEqual (r.line, u'aaa b ccc')

    def test_x_and_p (self):
        r = ViModeTest ()
        r._set_line (u'abc')
        r.input (u'Escape')
        r.input (u'"0xp"')
        self.assertEqual (r.line, u'bac')

    def test_delete_and_put (self):
        r = ViModeTest ()
        r._set_line (u'abc def')
        r.input (u'Escape')
        r.input (u'"0dwep"')
        self.assertEqual (r.line, u'defabc ')
        r.input (u'"0xp"')
        self.assertEqual (r.line, u'edfabc ')
        r.input (u'"p"')
        self.assertEqual (r.line, u'eddfabc ')

    def test_dot_simple (self):
        r = ViModeTest ()
        r._set_line (u'abc def')
        r.input (u'Escape')
        r.input (u'"0x"')
        self.assertEqual (r.line, u'bc def')
        r.input (u'"."')
        self.assertEqual (r.line, u'c def')
        r.input (u'"3."')
        self.assertEqual (r.line, u'ef')

    def test_dot_movement_not_repeated_one (self):
        r = ViModeTest ()
        r._set_line (u'abc def')
        r.input (u'Escape')
        r.input (u'"0x"')
        self.assertEqual (r.line, u'bc def')
        r.input (u'"$."')
        self.assertEqual (r.line, u'bc de')
        r.input (u'"0."')
        self.assertEqual (r.line, u'c de')
        r.input (u'"$."')
        self.assertEqual (r.line, u'c d')
        r.input (u'"^."')
        self.assertEqual (r.line, u' d')

    def test_dot_movement_not_repeated_two (self):
        r = ViModeTest ()
        r._set_line (u'abc def ghi jkl mno pqr')
        r.input (u'Escape')
        r.input (u'"0x"')
        self.assertEqual (r.line, u'bc def ghi jkl mno pqr')
        r.input (u'"w."')
        self.assertEqual (r.line, u'bc ef ghi jkl mno pqr')
        r.input (u'"fg."')
        self.assertEqual (r.line, u'bc ef hi jkl mno pqr')
        r.input (u'"2b."')
        self.assertEqual (r.line, u'c ef hi jkl mno pqr')
        r.input (u'"3e."')
        self.assertEqual (r.line, u'c ef hi jk mno pqr')
        r.input (u'"Fh."')
        self.assertEqual (r.line, u'c ef i jk mno pqr')
        r.input (u'"tn."')
        self.assertEqual (r.line, u'c ef i jk no pqr')
        r.input (u'"3h."')
        self.assertEqual (r.line, u'c ef i k no pqr')
        r.input (u'"5l."')
        self.assertEqual (r.line, u'c ef i k no qr')

    def test_dot_insert (self):
        r = ViModeTest ()
        r._set_line (u'abc def')
        r.input (u'Escape')
        r.input (u'"0"')
        r.input (u'"2izz "')
        r.input (u'Escape')
        self.assertEqual (r.line, u'zz zz abc def')
        r.input (u'"2w."')
        self.assertEqual (r.line, u'zz zz abc zz zz def')

    def test_dot_delete_word (self):
        r = ViModeTest ()
        r._set_line (u'0 1 2 3 4 5 6 7 8 9')
        r.input (u'Escape')
        r.input (u'"02dw"')
        self.assertEqual (r.line, u'2 3 4 5 6 7 8 9')
        r.input (u'"."')
        self.assertEqual (r.line, u'4 5 6 7 8 9')
        r.input (u'"1."')
        self.assertEqual (r.line, u'5 6 7 8 9')

    def test_dot_override_multiplier (self):
        r = ViModeTest ()
        r._set_line (u'ab ab ab ab  ab ab ab ab  ab ab ab ab  ab ab ab ab')
        r.input (u'Escape')
        r.input (u'"02d2fb"')
        self.assertEqual (r.line, u'  ab ab ab ab  ab ab ab ab  ab ab ab ab')
        r.input (u'"."')
        self.assertEqual (r.line, u'  ab ab ab ab  ab ab ab ab')
        r.input (u'"3."')
        self.assertEqual (r.line, u' ab  ab ab ab ab')
        r.input (u'"."')
        self.assertEqual (r.line, u' ab ab')

    def test_dot_yank_and_put (self):
        r = ViModeTest ()
        r._set_line (u'abc def')
        r.input (u'Escape')
        r.input (u'"0ywP"')
        self.assertEqual (r.line, u'abc abc def')
        r.input (u'"."')
        self.assertEqual (r.line, u'abcabc  abc def')
        r.input (u'"p"')
        self.assertEqual (r.line, u'abcabc abc  abc def')
        r.input (u'"2."')
        self.assertEqual (r.line, u'abcabc abc abc abc  abc def')

    def test_dot_insert_begin (self):
        r = ViModeTest ()
        r._set_line (u'abc def')
        r.input (u'Escape')
        r.input (u'"Izz "')
        r.input (u'Escape')
        self.assertEqual (r.line, u'zz abc def')
        r.input (u'"."')
        self.assertEqual (r.line, u'zz zz abc def')
        r.input (u'"2."')
        self.assertEqual (r.line, u'zz zz zz zz abc def')

    def test_dot_append_end (self):
        r = ViModeTest ()
        r._set_line (u'abc def')
        r.input (u'Escape')
        r.input (u'"A yy"')
        r.input (u'Escape')
        self.assertEqual (r.line, u'abc def yy')
        r.input (u'"."')
        self.assertEqual (r.line, u'abc def yy yy')
        r.input (u'"2."')
        self.assertEqual (r.line, u'abc def yy yy yy yy')

    def test_dot_insert_lower (self):
        r = ViModeTest ()
        r._set_line (u'abc def')
        r.input (u'Escape')
        r.input (u'"b2izz "')
        r.input (u'Escape')
        self.assertEqual (r.line, u'abc zz zz def')
        r.input (u'"3."')
        self.assertEqual (r.line, u'abc zz zzzz zz zz  def')

    def test_dot_append_lower (self):
        r = ViModeTest ()
        r._set_line (u'abc def')
        r.input (u'Escape')
        r.input (u'"0e2a zz"')
        r.input (u'Escape')
        self.assertEqual (r.line, u'abc zz zz def')
        r.input (u'"1."')
        self.assertEqual (r.line, u'abc zz zz zz def')

    def test_dot_substitute_lower (self):
        r = ViModeTest ()
        r._set_line (u'abc def')
        r.input (u'Escape')
        r.input (u'"03sqq"')
        r.input (u'Escape')
        self.assertEqual (r.line, u'qq def')
        r.input (u'"2."')
        self.assertEqual (r.line, u'qqqdef')

    def test_undo (self):
        r = ViModeTest ()
        r._set_line (u'abc def ghi')
        r.input (u'Escape')
        r.input (u'"0wdw"')
        self.assertEqual (r.line, u'abc ghi')
        self.assertEqual (r.line_cursor, 4)
        r.input (u'"u"')
        self.assertEqual (r.line, u'abc def ghi')
        self.assertEqual (r.line_cursor, 4)

    def test_undo_line (self):
        r = ViModeTest ()
        r._set_line (u'')
        r.input (u'"abc def ghi"')
        r.input (u'Escape')
        r.input (u'"0dwdw"')
        self.assertEqual (r.line, u'ghi')
        r.input (u'"U"')
        self.assertEqual (r.line, u'')

    def test_undo_line_with_history (self):
        r = ViModeTest ()
        r.add_history (u'abc 123')
        r._set_line (u'')
        r.input (u'"abc def ghi"')
        r.input (u'Escape')
        r.input (u'"0dwdw"')
        self.assertEqual (r.line, u'ghi')
        r.input (u'"U"')
        self.assertEqual (r.line, u'')

    def test_history_no_match (self):
        r = ViModeTest ()
        r.add_history (u'abc 123')
        r.add_history (u'def 456')
        r.add_history (u'ghi 789')
        r._set_line (u'abc def ghi')
        r.input (u'Escape')
        r.input (u'"/"')
        self.assertEqual (r.line, u'/')
        self.assertEqual (r.line_cursor, 1)
        r.input (u'"zz"')
        self.assertEqual (r.line, u'/zz')
        self.assertEqual (r.line_cursor, 3)
        self.assertEqual (r.console.bell_count, 0)
        r.input (u'Return')
        # TODO should bell be rung here?
        # self.assertEqual (r.console.bell_count, 1)
        self.assertEqual (r.line, u'abc def ghi')
        self.assertEqual (r.line_cursor, 10)

    def test_history_found_match (self):
        r = ViModeTest ()
        r.add_history (u'abc 123')
        r.add_history (u'def 456')
        r.add_history (u'ghi 789')
        r._set_line (u'abc def ghi')
        r.input (u'Escape')
        r.input (u'"/"')
        self.assertEqual (r.line, u'/')
        self.assertEqual (r.line_cursor, 1)
        r.input (u'"de"')
        self.assertEqual (r.line, u'/de')
        self.assertEqual (r.line_cursor, 3)
        r.input (u'Return')
        self.assertEqual (r.line, u'def 456')
        self.assertEqual (r.line_cursor, 0)

    def test_history_multi_match (self):
        r = ViModeTest ()
        r.add_history (u'xyz 123')
        r.add_history (u'xyz 456')
        r.add_history (u'xyz 789')
        r._set_line (u'abc def')
        r.input (u'Escape')
        r.input (u'"/xyz"')
        r.input (u'Return')
        self.assertEqual (r.line, u'xyz 789')
        r.input (u'"n"')
        self.assertEqual (r.line, u'xyz 456')
        r.input (u'"n"')
        self.assertEqual (r.line, u'xyz 123')
        self.assertEqual (r.console.bell_count, 0)
        r.input (u'"n"')
        # TODO check bell ringing
        # self.assertEqual (r.console.bell_count, 1)
        self.assertEqual (r.line, u'xyz 123')
        r.input (u'"N"')
        self.assertEqual (r.line, u'xyz 456')
        r.input (u'"N"')
        self.assertEqual (r.line, u'xyz 789')
        # TODO check bell ringing
        # self.assertEqual (r.console.bell_count, 1)
        r.input (u'"N"')
        # TODO check bell ringing
        # self.assertEqual (r.console.bell_count, 2)
        self.assertEqual (r.line, u'xyz 789')

    def test_history_search_empty_string (self):
        r = ViModeTest ()
        r.add_history (u'xyz 123')
        r.add_history (u'xyz 456')
        r.add_history (u'xyz 789')
        r.input (u'Escape')
        r.input (u'"/"')
        r.input (u'Return')
        self.assertEqual (r.line, u'')
        # TODO check bell ringing
        # self.assertEqual (r.console.bell_count, 1)
        r.input (u'"/"')
        r.input (u'Return')
        self.assertEqual (r.line, u'')
        # TODO check bell ringing
        # self.assertEqual (r.console.bell_count, 2)
        r.input (u'"/x"')
        r.input (u'Return')
        self.assertEqual (r.line, u'xyz 789')
        r.input (u'"/"')
        r.input (u'Return')
        self.assertEqual (r.line, u'xyz 456')

    def test_history_search_again_after_return (self):
        r = ViModeTest ()
        r.add_history (u'xyz 123')
        r.add_history (u'xyz 456')
        r.add_history (u'xyz 789')
        r._set_line (u'abc def')
        r.input (u'Escape')
        r.input (u'"/xyz"')
        r.input (u'Return')
        self.assertEqual (r.line, u'xyz 789')
        r.input (u'"n"')
        self.assertEqual (r.line, u'xyz 456')
        r.input (u'Return')
        self.assertEqual (r.line, u'')
        r.input (u'Escape')
        r.input (u'"n"')
        self.assertEqual (r.line, u'xyz 123')
        r.input (u'Return')
        r.input (u'Escape')
        r.input (u'"N"')
        self.assertEqual (r.line, u'xyz 456')

    def test_history_search_again_after_search_failed (self):
        r = ViModeTest ()
        r.add_history (u'xyz 123')
        r.add_history (u'xyz 456')
        r.add_history (u'xyz 789')
        r._set_line (u'abc def')
        r.input (u'Escape')
        r.input (u'"/xyz"')
        r.input (u'Return')
        self.assertEqual (r.line, u'xyz 789')
        r.input (u'"C"')
        self.assertEqual (r.line, u'')
        r.input (u'Escape')
        self.assertEqual (r.console.bell_count, 0)
        r.input (u'"/abc"')
        r.input (u'Return')
        # TODO check bell ringing
        # self.assertEqual (r.console.bell_count, 1)
        self.assertEqual (r.line, u'')
        r.input (u'Escape')
        r.input (u'"n"')
        self.assertEqual (r.line, u'')

    def test_history_search_and_backspace (self):
        r = ViModeTest ()
        r.add_history (u'aaa')
        r.add_history (u'bbb')
        r._set_line (u'')
        r.input (u'Escape')
        r.input (u'"/aaz"')
        self.assertEqual (r.line, u'/aaz')
        r.input (u'BackSpace')
        self.assertEqual (r.line, u'/aa')
        r.input (u'Return')
        self.assertEqual (r.line, u'aaa')
        r.input (u'Escape')
        r.input (u'"/z"')
        r.input (u'BackSpace')
        r.input (u'BackSpace')
        self.assertEqual (r.line, u'')
        r.input (u'"j"')
        self.assertEqual (r.line, u'bbb')
        r.input (u'"k"')
        self.assertEqual (r.line, u'aaa')

    def test_history_insert_mode (self):
        r = ViModeTest ()
        r.add_history (u'aaa')
        r.add_history (u'bbb')
        r.add_history (u'ccc')
        r.input (u'Up')
        self.assertEqual (r.line, u'ccc')
        self.assert_ (r.vi_is_insert_mode)
        r.input (u'"z"')
        self.assertEqual (r.line, u'cccz')
        r.input (u'Escape')
        self.assert_ (not r.vi_is_insert_mode)
        r.input (u'Up')
        self.assertEqual (r.line, u'bbb')
        self.assert_ (r.vi_is_insert_mode)
        r.input (u'"z"')
        self.assertEqual (r.line, u'bbbz')
        r.input (u'Escape')
        r.input (u'"k"')
        self.assertEqual (r.line, u'aaa')
        self.assert_ (not r.vi_is_insert_mode)
        r.input (u'"iz"')
        self.assertEqual (r.line, u'zaaa')
        r.input (u'Down')
        self.assertEqual (r.line, u'bbb')
        self.assert_ (r.vi_is_insert_mode)
        r.input (u'"y"')
        self.assertEqual (r.line, u'bbby')
        r.input (u'Escape')
        r.input (u'"j"')
        self.assertEqual (r.line, u'ccc')
        self.assert_ (not r.vi_is_insert_mode)
        r.input (u'"iy"')
        self.assertEqual (r.line, u'yccc')
        self.assert_ (r.vi_is_insert_mode)

    def test_history_cursor_arrow (self):
        r = ViModeTest ()
        self.assertEqual (r._history.history_cursor, 0)
        r.add_history (u'aaa')
        self.assertEqual (r._history.history_cursor, 1)
        r.add_history (u'bbb')
        self.assertEqual (r._history.history_cursor, 2)
        self.assertEqual (r.line, u'')
        r.input (u'Up')
        self.assertEqual (r._history.history_cursor, 1)
        self.assertEqual (r.line, u'bbb')
        r.input (u'Up')
        self.assertEqual (r._history.history_cursor, 0)
        self.assertEqual (r.line, u'aaa')
        r.input (u'Down')
        self.assertEqual (r._history.history_cursor, 1)
        self.assertEqual (r.line, u'bbb')
        r.input (u'Down')
        self.assertEqual (r._history.history_cursor, 2)
        self.assertEqual (r.line, u'')
        r.input (u'Up')
        self.assertEqual (r._history.history_cursor, 1)
        self.assertEqual (r.line, u'bbb')

    def test_history_control_n_and_p (self):
        r = ViModeTest ()
        r.add_history (u'aa')
        r.add_history (u'bbb')
        self.assertEqual (r.line, u'')
        r.input (u'Control-p')
        self.assertEqual (r.line, u'bbb')
        self.assertEqual (r.line_cursor, 3)
        self.assert_ (r.vi_is_insert_mode)
        r.input (u'Control-p')
        self.assertEqual (r.line, u'aa')
        self.assertEqual (r.line_cursor, 2)
        self.assert_ (r.vi_is_insert_mode)
        r.input (u'Control-n')
        self.assertEqual (r.line, u'bbb')
        self.assertEqual (r.line_cursor, 3)
        self.assert_ (r.vi_is_insert_mode)
        r.input (u'Control-n')
        self.assertEqual (r.line, u'')
        self.assertEqual (r.line_cursor, 0)
        self.assert_ (r.vi_is_insert_mode)
        r.input (u'Control-p')
        self.assertEqual (r.line, u'bbb')
        self.assertEqual (r.line_cursor, 3)
        self.assert_ (r.vi_is_insert_mode)
        r.input (u'Escape')
        self.assertEqual (r.line, u'bbb')
        self.assertEqual (r.line_cursor, 2)
        self.assert_ (not r.vi_is_insert_mode)
        r.input (u'Control-p')
        self.assertEqual (r.line, u'aa')
        self.assertEqual (r.line_cursor, 0)
        self.assert_ (not r.vi_is_insert_mode)
        r.input (u'Control-n')
        self.assertEqual (r.line, u'bbb')
        self.assertEqual (r.line_cursor, 0)
        self.assert_ (not r.vi_is_insert_mode)
        r.input (u'Control-n')
        self.assertEqual (r.line, u'')
        self.assertEqual (r.line_cursor, 0)
        self.assert_ (not r.vi_is_insert_mode)
        r.input (u'"/a"')
        r.input (u'Return')
        self.assertEqual (r.line, u'aa')
        self.assertEqual (r.line_cursor, 0)
        self.assert_ (not r.vi_is_insert_mode)
        r.input (u'Control-n')
        self.assertEqual (r.line, u'bbb')
        self.assertEqual (r.line_cursor, 0)
        self.assert_ (not r.vi_is_insert_mode)

    def test_history_cursor_j_and_k (self):
        r = ViModeTest ()
        r.add_history (u'aaa')
        r.input (u'Escape')
        r.input (u'"kiz"')
        self.assertEqual (r.line, u'zaaa')
        r.input (u'Escape')
        r.input (u'"j"')
        self.assertEqual (r.line, u'')

    def test_history_input_j_and_k (self):
        r = ViModeTest ()
        r.add_history (u'aaa')
        r.input (u'Escape')
        r.input (u'"kAjk"')
        self.assertEqual (r.line, u'aaajk')

    def test_history_cursor_search (self):
        r = ViModeTest ()
        r.add_history (u'aaa')
        r.input (u'Escape')
        r.input (u'"/a"')
        r.input (u'Return')
        self.assertEqual (r.line, u'aaa')
        r.input (u'"iz"')
        self.assertEqual (r.line, u'zaaa')
        self.assertEqual (r.console.bell_count, 0)
        r.input (u'Escape')
        r.input (u'"j"')
        self.assertEqual (r.line, u'zaaa')
        # TODO check bell ringing
        # self.assertEqual (r.console.bell_count, 1)

    def test_history_undo (self):
        r = ViModeTest ()
        r.add_history (u'aaa')
        r.input (u'Escape')
        r.input (u'"k"')
        r.input (u'"A b"')
        r.input (u'Escape')
        r.input (u'"A c"')
        r.input (u'Escape')
        self.assertEqual (r.line, u'aaa b c')
        r.input (u'"U"')
        self.assertEqual (r.line, u'aaa')

    def test_history_arrow_undo (self):
        r = ViModeTest ()
        r.add_history (u'aaa')
        r.input (u'Up')
        r.input (u'" zz"')
        self.assertEqual (r.line, u'aaa zz')
        r.input (u'Escape')
        r.input (u'"u"')
        self.assertEqual (r.line, u'aaa')

# TODO: mode support?
#     def test_mode (self):
#         r = ViModeTest ()
#         self.assertEqual (r.editing_mode, Readline.mode_vi)
#         self.assertEqual (r.mode (verbose=False), Readline.mode_vi)
#         self.assertEqual (r.count_vi_editing_mode, 1)
#         self.assertEqual (r.count_emacs_editing_mode, 0)
#         r.vi ()
#         self.assertEqual (r.editing_mode, Readline.mode_vi)
#         self.assertEqual (r.mode (verbose=False), Readline.mode_vi)
#         self.assertEqual (r.count_vi_editing_mode, 1)
#         self.assertEqual (r.count_emacs_editing_mode, 0)
#         r.emacs ()
#         self.assertEqual (r.editing_mode, Readline.mode_emacs)
#         self.assertEqual (r.mode (verbose=False), Readline.mode_emacs)
#         self.assertEqual (r.count_vi_editing_mode, 1)
#         self.assertEqual (r.count_emacs_editing_mode, 1)
#         r.emacs ()
#         self.assertEqual (r.editing_mode, Readline.mode_emacs)
#         self.assertEqual (r.mode (verbose=False), Readline.mode_emacs)
#         self.assertEqual (r.count_vi_editing_mode, 1)
#         self.assertEqual (r.count_emacs_editing_mode, 1)
#         r.vi ()
#         self.assertEqual (r.editing_mode, Readline.mode_vi)
#         self.assertEqual (r.mode (verbose=False), Readline.mode_vi)
#         self.assertEqual (r.count_vi_editing_mode, 2)
#         self.assertEqual (r.count_emacs_editing_mode, 1)
#
#     def test_switch_mode (self):
#         r = ViModeTest ()
#         r._set_line (u'')
#         r.input (u'Escape')
#         self.assertEqual (r.editing_mode, Readline.mode_vi)
#         self.assertEqual (r.count_vi_editing_mode, 1)
#         self.assertEqual (r.count_emacs_editing_mode, 0)
#         r.input (u'"abc"')
#         r.input (u'Control-e')
#         self.assertEqual (r.editing_mode, Readline.mode_emacs)
#         self.assertEqual (r.count_vi_editing_mode, 1)
#         self.assertEqual (r.count_emacs_editing_mode, 1)
#         r.input (u'Meta-Control-j')
#         self.assertEqual (r.editing_mode, Readline.mode_vi)
#         self.assertEqual (r.count_vi_editing_mode, 2)
#         self.assertEqual (r.count_emacs_editing_mode, 1)

# TODO: show history support?
#     def test_history_output (self):
#         import StringIO
#         sio = StringIO.StringIO ()
#         r = ViModeTest ()
#         r.add_history (u'abc')
#         r.add_history (u'def')
#         r.add_history (u'ghi')
#         r.show_history (sio)
#         sio.seek (0)
#         self.assertEqual (sio.read(), u'  1 abc\n  2 def\n  3 ghi\n')

    def test_editor (self):
        vee = ViExternalEditorTest (u'qwerty before')
        self.assert_ (vee.sio_write.closed)
        self.assertEqual (vee.command, u'vim.exe temp.py')
        self.assert_ (vee.sio_read.closed)
        self.assertEqual (vee.remove, u'temp.py')
        self.assertEqual (vee.result, u'qwerty after')

    def test_completer (self):
        r = ViModeTest ()
        r.lst_completions = ['aab', u'aac', u'aad', ]
        r.input (u'"aa"')
        r.input (u'Tab')
        self.assertEqual (r.line, u'aa')
        self.assertEqual (r.console.text, u'\naab \naac \naad \n')

    def test_completer_star (self):
        r = ViModeTest ()
        r.lst_completions = ['bbc', u'bbd', u'bbe', ]
        r.input (u'"aa bb"')
        r.input (u'Escape')
        self.assert_ (not r.vi_is_insert_mode)
        r.input (u'"*"')
        self.assertEqual (r.line, u'aa bbc bbd bbe ')
        self.assertEqual (r.line_cursor, 15)
        self.assert_ (r.vi_is_insert_mode)
        self.assertEqual (r.console.text, u'')
        r.input (u'" "')
        r.input (u'Escape')
        self.assertEqual (r.line, u'aa bbc bbd bbe  ')
        self.assertEqual (r.line_cursor, 15)
        r.input (u'"."')
        self.assertEqual (r.line, u'aa bbc bbd bbe bbc bbd bbe ')
        self.assertEqual (r.line_cursor, 27)

    def test_completer_beginning_of_line (self):
        r = ViModeTest ()
        r.input (u'Tab')
        self.assertEqual (r.line, u'    ')
        self.assertEqual (r.line_cursor, 4)
        r.input (u'Space')
        self.assertEqual (r.line, u'     ')
        self.assertEqual (r.line_cursor, 5)
        r.input (u'Tab')
        self.assertEqual (r.line, u'        ')
        self.assertEqual (r.line_cursor, 8)
        r.input (u'Space')
        r.input (u'Space')
        r.input (u'Space')
        self.assertEqual (r.line, u'           ')
        self.assertEqual (r.line_cursor, 11)
        r.input (u'Tab')
        self.assertEqual (r.line, u'            ')
        self.assertEqual (r.line_cursor, 12)
        r.input (u'Tab')
        self.assertEqual (r.line, u'                ')
        self.assertEqual (r.line_cursor, 16)

    def test_replace_lower (self):
        r = ViModeTest ()
        r._set_line (u'aaa bbb ccc')
        r.input (u'Escape')
        r.input (u'"0ry"')
        self.assertEqual (r.line, u'yaa bbb ccc')
        self.assertEqual (r.line_cursor, 0)
        self.assert_ (not r.vi_is_insert_mode)
        r.input (u'"i"')
        self.assertEqual (r.line, u'yaa bbb ccc')
        self.assertEqual (r.line_cursor, 0)
        self.assert_ (r.vi_is_insert_mode)
        r.input (u'"x"')
        self.assertEqual (r.line, u'xyaa bbb ccc')
        self.assertEqual (r.line_cursor, 1)
        self.assert_ (r.vi_is_insert_mode)
        r.input (u'Escape')
        self.assertEqual (r.line, u'xyaa bbb ccc')
        self.assertEqual (r.line_cursor, 0)
        self.assert_ (not r.vi_is_insert_mode)
        r.input (u'"ll"')
        r.input (u'"2rz"')
        self.assertEqual (r.line, u'xyzz bbb ccc')
        self.assertEqual (r.line_cursor, 3)
        self.assert_ (not r.vi_is_insert_mode)
        r.input (u'"w7."')
        self.assertEqual (r.line, u'xyzz zzzzzzz')
        self.assertEqual (r.line_cursor, 11)
        self.assert_ (not r.vi_is_insert_mode)

    def test_replace_lower_undo (self):
        r = ViModeTest ()
        r._set_line (u'aaa')
        r.input (u'Escape')
        # print u'xx', sys._getframe().f_lineno, r._vi_undo_cursor, r._vi_undo_stack
        r.input (u'"0rz"')
        self.assertEqual (r.line, u'zaa')
        r.input (u'"u"')
        self.assertEqual (r.line, u'aaa')
        r.input (u'"2."')
        self.assertEqual (r.line, u'zza')
        r.input (u'"u"')
        self.assertEqual (r.line, u'aaa')

    def test_replace_lower_escape (self):
        r = ViModeTest ()
        r._set_line (u'aaa')
        r.input (u'Escape')
        r.input (u'"0r"')
        self.assertEqual (r.line, u'aaa')
        r.input (u'Escape')
        self.assertEqual (r.line, u'aaa')
        r.input (u'"r"')
        self.assertEqual (r.line, u'aaa')
        r.input (u'"z"')
        self.assertEqual (r.line, u'zaa')

    def test_replace_lower_escape_undo (self):
        r = ViModeTest ()
        r._set_line (u'aa bb cc')
        r.input (u'Escape')
        r.input (u'"0cwdd"')
        r.input (u'Escape')
        self.assertEqual (r.line, u'dd bb cc')
        r.input (u'"wr"')
        r.input (u'Escape')
        self.assertEqual (r.line, u'dd bb cc')
        r.input (u'"."')
        self.assertEqual (r.line, u'dd dd cc')
        r.input (u'"u"')
        self.assertEqual (r.line, u'dd bb cc')
        r.input (u'"u"')
        self.assertEqual (r.line, u'aa bb cc')

    def test_replace_dot (self):
        r = ViModeTest ()
        r._set_line (u'ab')
        r.input (u'Escape')
        r.input (u'"0rzl"')
        self.assertEqual (r.line, u'zb')
        self.assertEqual (r.line_cursor, 1)
        self.assert_ (not r.vi_is_insert_mode)
        r.input (u'"r."')
        self.assertEqual (r.line, u'z.')
        self.assertEqual (r.line_cursor, 1)
        self.assert_ (not r.vi_is_insert_mode)

    def test_replace_upper (self):
        r = ViModeTest ()
        r._set_line (u'aaa bbb')
        r.input (u'Escape')
        r.input (u'"0wR"')
        self.assertEqual (r.line, u'aaa bbb')
        self.assertEqual (r.line_cursor, 4)
        self.assert_ (r.vi_is_insert_mode)
        r.input (u'"z"')
        self.assertEqual (r.line, u'aaa zbb')
        self.assertEqual (r.line_cursor, 5)
        self.assert_ (r.vi_is_insert_mode)
        r.input (u'"zzz"')
        self.assertEqual (r.line, u'aaa zzzz')
        self.assertEqual (r.line_cursor, 8)
        self.assert_ (r.vi_is_insert_mode)
        r.input (u'Escape')
        self.assertEqual (r.line_cursor, 7)
        self.assert_ (not r.vi_is_insert_mode)

    def test_replace_upper_dot (self):
        r = ViModeTest ()
        r._set_line (u'aaa bbb ccc ddd')
        r.input (u'Escape')
        r.input (u'"02Rz"')
        r.input (u'Escape')
        self.assertEqual (r.line, u'zza bbb ccc ddd')
        self.assertEqual (r.line_cursor, 1)
        self.assert_ (not r.vi_is_insert_mode)
        r.input (u'"w."')
        self.assertEqual (r.line, u'zza zzb ccc ddd')
        self.assertEqual (r.line_cursor, 5)
        self.assert_ (not r.vi_is_insert_mode)
        r.input (u'"w6."')
        self.assertEqual (r.line, u'zza zzb zzzzzzd')
        self.assertEqual (r.line_cursor, 13)
        self.assert_ (not r.vi_is_insert_mode)

    def test_replace_upper_undo (self):
        r = ViModeTest ()
        r._set_line (u'aaa bbb ccc')
        r.input (u'Escape')
        r.input (u'"0Rzz"')
        r.input (u'Escape')
        self.assertEqual (r.line, u'zza bbb ccc')
        r.input (u'"w3."')
        self.assertEqual (r.line, u'zza zzzzzzc')
        r.input (u'"u"')
        self.assertEqual (r.line, u'zza bbb ccc')
        r.input (u'"u"')
        self.assertEqual (r.line, u'aaa bbb ccc')

    def test_replace_backspace_and_dot (self):
        r = ViModeTest ()
        r._set_line (u'aa bb')
        r.input (u'Escape')
        r.input (u'"0wRc"')
        self.assertEqual (r.line, u'aa cb')
        self.assertEqual (r.line_cursor, 4)
        self.assert_ (r.vi_is_insert_mode)
        r.input (u'"c"')
        self.assertEqual (r.line, u'aa cc')
        self.assertEqual (r.line_cursor, 5)
        r.input (u'"c"')
        self.assertEqual (r.line, u'aa ccc')
        self.assertEqual (r.line_cursor, 6)
        r.input (u'BackSpace')
        self.assertEqual (r.line, u'aa cc')
        self.assertEqual (r.line_cursor, 5)
        self.assert_ (r.vi_is_insert_mode)
        r.input (u'BackSpace')
        self.assertEqual (r.line, u'aa cb')
        self.assertEqual (r.line_cursor, 4)
        self.assert_ (r.vi_is_insert_mode)
        r.input (u'BackSpace')
        self.assertEqual (r.line, u'aa bb')
        self.assertEqual (r.line_cursor, 3)
        self.assert_ (r.vi_is_insert_mode)
        r.input (u'BackSpace')
        self.assertEqual (r.line, u'aa bb')
        self.assertEqual (r.line_cursor, 2)
        self.assert_ (r.vi_is_insert_mode)
        r.input (u'"d"')
        self.assertEqual (r.line, u'aadbb')
        self.assertEqual (r.line_cursor, 3)
        self.assert_ (r.vi_is_insert_mode)
        r.input (u'Escape')
        self.assertEqual (r.line, u'aadbb')
        self.assertEqual (r.line_cursor, 2)
        self.assert_ (not r.vi_is_insert_mode)
        r.input (u'"u"')
        self.assertEqual (r.line, u'aa bb')
        self.assertEqual (r.line_cursor, 3)
        self.assert_ (not r.vi_is_insert_mode)
        r.input (u'"hh"')
        r.input (u'"."')
        self.assertEqual (r.line, u'da bb')
        self.assert_ (not r.vi_is_insert_mode)

    def test_yank_line (self):
        r = ViModeTest ()
        r._set_line (u'aa bb')
        r.input (u'Escape')
        r.input (u'"0wY"')
        self.assertEqual (r.line, u'aa bb')
        self.assert_ (not r.vi_is_insert_mode)
        r.input (u'"P"')
        self.assertEqual (r.line, u'aa aa bbbb')
        self.assert_ (not r.vi_is_insert_mode)
        r.input (u'"u"')
        self.assertEqual (r.line, u'aa bb')
        self.assert_ (not r.vi_is_insert_mode)

    def test_column (self):
        r = ViModeTest ()
        r._set_line (u'aaa bbb')
        r.input (u'Escape')
        r.input (u'"099|"')
        self.assertEqual (r.line, u'aaa bbb')
        self.assertEqual (r.line_cursor, 6)
        r.input (u'"4|"')
        self.assertEqual (r.line, u'aaa bbb')
        self.assertEqual (r.line_cursor, 3)
        r.input (u'"d1|"')
        self.assertEqual (r.line, u' bbb')
        self.assertEqual (r.line_cursor, 0)
        r.input (u'"u"')
        self.assertEqual (r.line, u'aaa bbb')
        self.assertEqual (r.line_cursor, 3)
        r.input (u'"d7|"')
        self.assertEqual (r.line, u'aaab')
        self.assertEqual (r.line_cursor, 3)

    def test_change_case (self):
        r = ViModeTest ()
        r._set_line (u'aaa B7B ccc')
        r.input (u'Escape')
        r.input (u'"0~"')
        self.assertEqual (r.line, u'Aaa B7B ccc')
        self.assertEqual (r.line_cursor, 1)
        self.assert_ (not r.vi_is_insert_mode)
        r.input (u'"h."')
        self.assertEqual (r.line, u'aaa B7B ccc')
        self.assertEqual (r.line_cursor, 1)
        self.assert_ (not r.vi_is_insert_mode)
        r.input (u'"9~"')
        self.assertEqual (r.line, u'aAA b7b CCc')
        self.assertEqual (r.line_cursor, 10)
        self.assert_ (not r.vi_is_insert_mode)
        r.input (u'"u"')
        self.assertEqual (r.line, u'aaa B7B ccc')
        self.assertEqual (r.line_cursor, 1)
        self.assert_ (not r.vi_is_insert_mode)

    def test_redo (self):
        r = ViModeTest ()
        r._set_line (u'')
        r.input (u'Escape')
        r.input (u'"Saaa"')
        self.assertEqual (r.line, u'aaa')
        r.input (u'Escape')
        r.input (u'"Sbbb"')
        self.assertEqual (r.line, u'bbb')
        r.input (u'Escape')
        r.input (u'"Sccc"')
        r.input (u'Escape')
        self.assertEqual (r.line, u'ccc')
        r.input (u'"u"')
        self.assertEqual (r.line, u'bbb')
        r.input (u'Control-r')
        self.assertEqual (r.line, u'ccc')
        r.input (u'"u"')
        self.assertEqual (r.line, u'bbb')
        r.input (u'"u"')
        self.assertEqual (r.line, u'aaa')
        r.input (u'"u"')
        self.assertEqual (r.line, u'')
        r.input (u'"u"')
        self.assertEqual (r.line, u'')
        r.input (u'Control-r')
        self.assertEqual (r.line, u'aaa')
        r.input (u'Control-r')
        self.assertEqual (r.line, u'bbb')
        r.input (u'Control-r')
        self.assertEqual (r.line, u'ccc')
        r.input (u'Control-r')
        self.assertEqual (r.line, u'ccc')
        r.input (u'"u"')
        self.assertEqual (r.line, u'bbb')
        r.input (u'"Szzz"')
        r.input (u'Escape')
        self.assertEqual (r.line, u'zzz')
        r.input (u'"u"')
        self.assertEqual (r.line, u'bbb')
        r.input (u'Control-r')
        self.assertEqual (r.line, u'zzz')
        r.input (u'"U"')
        self.assertEqual (r.line, u'')
        r.input (u'Control-r')
        self.assertEqual (r.line, u'aaa')
        r.input (u'Control-r')
        self.assertEqual (r.line, u'bbb')
        r.input (u'Control-r')
        self.assertEqual (r.line, u'zzz')
        r.input (u'Control-r')
        self.assertEqual (r.line, u'zzz')

#----------------------------------------------------------------------
# utility functions


#----------------------------------------------------------------------

if __name__ == u'__main__':
    Tester()

    tested=ViModeTest.tested_commands.keys()    
    tested.sort()
    print " Tested functions ".center(60,"-")
    print "\n".join(tested)
    print
    
    all_funcs=dict([(x.__name__,x) for x in ViModeTest().key_dispatch.values()])
    all_funcs=all_funcs.keys()
    not_tested=[x for x in all_funcs if x not in tested]
    not_tested.sort()
    print " Not tested functions ".center(60,"-")
    print "\n".join(not_tested)
    
    
