import logging
import uuid

from pypipes.context.manager import PipeContextManager
from pypipes.message import FrozenMessage
from pypipes.processor import pipe_processor
from pypipes.processor.event import Event

logger = logging.getLogger(__name__)

EVENT_JOB_DONE = 'job_done_{}'

WAIT_ALL_MESSAGE = '_wait_all_message'
WAIT_ALL_COUNTER = '_wait_all_counter'


class Sync(object):
    def __init__(self, pipeline):
        self.pipeline = pipeline

    @property
    def wait_all(self):
        """
        Wait till all synchronised jobs are completed.
        This sync helper requires counter.wait_all injection of CounterClient type

        Usage:
        start_processor >> Sync(sync_first >> ...  >> sync_last).wait_all >> next_processor

        next_processor will receives only one message per each one message received by sync_first
        all messages generated by sync_last processor will be ignored.
        Thus it's better to design sync_last processor to not emit anything
        to avoid additional message traffic
        """
        job_done_event = EVENT_JOB_DONE.format(uuid.uuid4().hex[-8:])
        context = count_job_context(job_done_event)
        # each processor should increments sync lock counter when emit a new message
        # so we add a contextmanager that will do it for us
        locked_pipeline = context(self.pipeline)
        # append a processor that will count completed tasks
        return locked_pipeline >> (Event.on(job_done_event) >> wait_all)


def count_job_context(job_done_event):
    def _context(message, response, counter):
        # get counter name from input message
        counter_name = message.pop(WAIT_ALL_COUNTER, None)
        sync_message = dict(message.pop(WAIT_ALL_MESSAGE, message))

        if not counter_name:
            # start a new job counter is it's a first processor in synchronised pipeline
            counter_name = uuid.uuid4().hex
            counter.wait_all.increment(counter_name, 1)

        def _filter(msg):
            counter.wait_all.increment(counter_name, 1)
            msg[WAIT_ALL_COUNTER] = counter_name
            msg[WAIT_ALL_MESSAGE] = sync_message
            return msg

        def _count_jobs_on_flush(original_flush):
            # flush all new messages
            original_flush()
            # count rest of job
            if counter.wait_all.increment(counter_name, -1) <= 0:
                # notify wait_all processor that last job is done
                response.send_event(job_done_event,
                                    apply_filters=True)

        response.add_message_filter(_filter)
        response.extend_flush(_count_jobs_on_flush)

        yield {'wait_all_counter': counter_name,
               'wait_all_message': FrozenMessage(sync_message)}

    return PipeContextManager(_context)


@pipe_processor
def wait_all(message, response, counter, event):
    counter_name = message[WAIT_ALL_COUNTER]
    if event:
        # counter was already decremented in processor
        # just double check that there is no new task
        value = counter.wait_all.increment(counter_name, -1)
    else:
        value = counter.wait_all.increment(counter_name, -1)

    if value <= 0:
        logger.debug('All jobs are complete, go ahead.')
        # all synchronised processors have finished there jobs
        # emit a message to activate next processor
        response.emit_message(message[WAIT_ALL_MESSAGE])
        # delete sync lock counter
        counter.wait_all.delete(counter_name)
    else:
        logger.debug('One more job is complete, %s jobs left', value)
