#!/usr/bin/env ipython
"""Graphing of benchmark results.

Requires IPython (Notebook or matplotlib).
"""
from __future__ import print_function
import collections
import json
import operator
import random

import matplotlib.pyplot as plt
import numpy as np


plt.rcdefaults()


class RandomRGBPallette(dict):

    def __missing__(self, key):
        rgb = []
        for _ in range(3):
            v = random.randrange(0, 10) * 0.1
            rgb.append(v)
        color = tuple(rgb)
        if color in self.values():
            color = rgb()
        self[key] = color
        return color


def _fixup_durations(durations):
    return tuple(max(duration, duration + 0.5)
                 for duration in durations)


def load_benchmark_data(benchmark_results_path):
    results = collections.OrderedDict()
    with open(benchmark_results_path) as fp:
        results.update(json.load(fp))
    return results


def group_durations_by_action(results, classifiers):
    grouped_durations = collections.OrderedDict()
    for classifier in classifiers:
        result = results[classifier][b'action-duration']
        for action, duration in sorted(result.items()):
            grouped_durations.setdefault(action, []).append(duration)
    return grouped_durations


def auto_label(ax, rects):
    """Attach some text labels to a rectangle.

    :param rects: Rectangles.
    """
    for rect in rects:
        rh = rect.get_height()
        height = round(rh, 3)
        ax.annotate(b'%0.2fs' % (height,),
                    (rect.get_x() + 0.02, height),
                    fontsize=8,
                    xytext=None)


# Determine if we need to fixup durations (attempt)
# When durations are sub-miliseconds,
# it's hard to tell if it's an error with the code
# or just the graph.
# all_durations = np.asarray(all_durations)
# dur_range = np.arange(all_durations.min(), all_durations.max())
# do_duration_fixup = dur_range.std() > 10


def plot_benchmark_results(benchmark_results_path):
    """Plot benchmark results and render a graph with matplotlib.

    :param benchmark_results_path: The path to the JSON file
                                   generated by benchmarks.
    """
    figure_title = u'Time taken to re-index Object Security'
    benchmark_runs = load_benchmark_data(benchmark_results_path)
    nrows = len(benchmark_runs)
    fig, axes = plt.subplots(nrows=nrows, sharex='row')
    fig.suptitle(figure_title)
    colors = RandomRGBPallette()
    bmruns = sorted(benchmark_runs.values(),
                    key=operator.itemgetter(b'timestamp', 'n_objects'),
                    reverse=True)
    for (ax_idx, benchmark_run) in enumerate(bmruns):
        ax = axes[ax_idx]
        n_objects = benchmark_run[b'n_objects']
        settings = benchmark_run[b'settings']
        labels_by_action = settings[b'action-labels']
        results = benchmark_run[b'results']
        classifiers = sorted(results, reverse=True)
        all_durations = []
        dba = group_durations_by_action(results, classifiers)
        action_labels = tuple(labels_by_action[a] for a in dba)
        n_groups = len(action_labels)
        index = np.arange(n_groups)
        width = 0.2
        bars = []
        ax.set_title(b'Processed {:d} objects'.format(n_objects),
                     fontsize=14)
        ax.set_xticklabels(classifiers)
        for action, durations in dba.items():
            all_durations.extend(durations)
            index = index + width
            rects = ax.bar(index,
                           durations,
                           width,
                           color=colors[action],
                           label=labels_by_action[action])
            auto_label(ax, rects)
            bars.append(rects)
        ax.set_xticks(index - width)
        ax.set_ylabel(u'Time (s)')
    fig.legend(bars, action_labels, loc=b'upper right', fancybox=True)
    plt.subplots_adjust(left=0.05,
                        bottom=0.05,
                        right=0.77,
                        top=0.91,
                        wspace=0.21,
                        hspace=0.61)
    plt.show()


if __name__ == '__main__':
    plot_benchmark_results(b'benchmark-results.json.sample')
