import exceptions

import time
import datetime
 
import core
from query import Query

class PodColumnError(exceptions.BaseException):
    pass

class Expression(object):
    
    def __init__(self, left = None, right = None, op = None, parens = True):            
        self.left    = left
        self.right   = right
        self.op      = op
        self.parens  = parens
    
    def get_sql(self, full_name = False, type = None):
        
        self.expression = ""
        self.args       = []
        self.class_pods = set()
        self.full_name  = full_name
        #type includes select, raw_select, where, group_by, order_by
        self.type       = type
        
        self.get_my_sql(head = self)

    def get_my_sql(self, head):
        
        """ left """
        if head.type == 'where' and self.parens:
            head.expression += "("
        if isinstance(self.left, Expression):
            self.left.get_my_sql(head = head)
        elif self.left is not None:
            self.process_arg(head = head, value = self.left)
            
        if self.op:
            head.expression += self.op
        else:
            raise PodColumnError, "Expression must be either terminal or have an operation . . . "
        
        """ right """
        if isinstance(self.right, Expression):
            self.right.get_my_sql(head = head)
        elif self.right is not None:
            self.process_arg(head = head, value = self.right)

        if head.type == 'where' and self.parens: 
            head.expression += ")"
                 
    def process_arg(self, head, value):
        if isinstance(value, Expression):
            value.get_my_sql(head = head)
        elif isinstance(value, core.Object):
            head.expression += "?"
            head.args.append(value.get_full_id())
        elif isinstance(value, bool):
            head.expression += "?"
            head.args.append(int(bool(value)))
        elif isinstance(value, (list,set)):
            head.expression += "("
            for item in value:
                self.process_arg(head = head, value = item)
                head.expression += ","
            head.expression = head.expression[:-1] + ")"    
        else:
            head.expression += "?"
            head.args.append(value)
        
    """ EXPRESSION CHAINING """
    def __and__(self, other):
        raise PodColumnError, "The INTERSECT '&' operator not supported on column expressions . . . "
    
    def __or__(self, other):
        return Expression(left = self, right = other, op = ',')
    
    """ THE AS NAME NAMING OPERATOR """
    def __rshift__(self, as_name):
        #if self.as_name and as_name:
        #    raise PodColumnError, "You tried to set column expression to '" + as_name + "' but it's already set to '" + str(self.as_name) + "' . . . "
        return AsName(left = self, right = None, op = ' AS ' + as_name + ' ')
    
    """ MATH OPERATORS """
    def __add__(self, other):
        return Math(left = self, right = other, op = '||' if isinstance(other, str) else ' + ' )
    def __radd__(self, other):
        return Math(left = other, right = self, op = '||' if isinstance(other, str) else ' + ' )
    def __sub__(self, other):
        return Math(left = self, right = other, op = "-")
    def __rsub__(self, other):
        return Math(left = other, right = self, op = "-")
    def __mul__(self, other):
        return Math(left = self, right = other, op = "*")
    def __rmul__(self, other):
        return Math(left = other, right = self, op = "*")
    def __div__(self, other):
        return Math(left = self, right = other, op = "/")
    def __rdiv__(self, other):
        return Math(left = other, right = self, op = "/")
    
    """ COMPARISON OPERATORS """
    def __eq__(self, value):
        if value is None:
            return Condition(left = self, op = ' IS NULL', right = None)
        else:
            return Condition(left = self, op = '=', right = value)

    def __ne__(self, value):
        if value is None:
            return Condition(left = self, op = 'IS NOT NULL', right = None)
        else:
            return Condition(left = self, op = '<>', right = value)
    
    def contains(self, value):
        return Condition(left = self, op = ' LIKE ',   right = '%' + str(value) + '%')
    
    def startswith(self, value):
        return Condition(left = self, op = ' LIKE ',   right = str(value) + '%')
    
    def endswith(self, value):
        return Condition(left = self, op = ' LIKE ', right = '%' + str(value))
    
    def is_in(self, value):
        return Condition(left = self, op = ' IN ', right = value) 
        
    def __gt__(self, value):
        return Condition(left = self, op = ' > ',  right = value)
    
    def __ge__(self, value):
        return Condition(left = self, op = ' >= ', right = value)
    
    def __lt__(self, value):
        return Condition(left = self, op = ' < ',  right = value)
      
    def __le__(self, value):
        return Condition(left = self, op = '<= ', right = value)

    def between(self, low, high):
        return Condition(left = self, op = ' BETWEEN ', right = Condition(left = low, right = high, op = ' AND ', parens = False))
    
    """ order by """
    def asc(self):
        return OrderAsc(col = self)
    
    def desc(self):
        return OrderDesc(col = self)
        
class AsName(Expression):
    
    
    def get_my_sql(self, head):
        if head.type != 'select':
            raise PodColumnError, "You cannot use AS operator '>>' in anything but the select statement . . ."
        return Expression.get_my_sql(self, head)
    
class Math(Expression):
    
    def __init__(self, left = None, right = None, op = None, parens = True):            
        self.left    = left
        self.right   = right
        self.op      = op
        self.parens  = parens

class Condition(Expression):
    
    def __init__(self, left = None, right = None, op = None, parens = True):            
        self.left    = left
        self.right   = right
        self.op      = op
        self.parens  = parens
        
    def __and__(self, other):
        return Condition(left = self, right = other, op = ' AND ')

    def __or__(self, other):
        return Condition(left = self, right = other, op = ' OR ')
    
    def __iter__(self):
        return Query(where = self).__iter__()

    def delete(self):
        return Query(where = self).delete()
    
    def get_one(self, error_on_multiple = True):
        return Query(where = self).get_one(error_on_multiple = error_on_multiple)

""" ############################################################################################
###
###    THE COLUMNS
###
""" ############################################################################################

class Column(Expression):

    def __init__(self, index = False, cls_pod = None, name = None):        
        Expression.__init__(self)
        self.index = index
        if cls_pod and name:
            self.cls_pod = cls_pod
            self.name    = name

    def __getstate__(self):
        return {'index': self.index}
         
    def on_activate(self):
        pass
            
    def drop(self):
        self.cls_pod.column_drop_and_delete_forever(self)
            
    def get_deep_copy(self, cls_pod, name):
        return self.__class__(index = self.index, cls_pod = cls_pod, name = name)
        
    def update(self, cls_pod, name):
        self.cls_pod = cls_pod
        self.name    = name
               
    def get_alter_table(self):
        return self.name + " " + self.db_type
    
    def dump(self, value):
        return None if value is None else self.py_type(value)
    
    def load(self, value):
        return None if value is None else self.py_type(value)
    
    def is_same_column_type(self, other):
        return other.index == self.index and other.__class__ is self.__class__
    
    def get_my_sql(self, head):
        head.expression += self.get_sql_column_name(head = head)
        head.class_pods.add(self.cls_pod)
                   
    def get_sql_column_name(self, head):
        return self.cls_pod.table + '.' + self.name if head.full_name else self.name
                   
""" Boolean """
class Boolean(Column):
    
    db_type = 'INTEGER'

    def dump(self, value):
        return None if value is None else int(bool(value))
    
    def load(self, value):
        return None if value is None else bool(value)

""" String """    
class String(Column):
    
    db_type = 'TEXT'
    py_type = str
        
""" Number """
class Number(Column):
    pass

class Int(Number):
    db_type = 'INTEGER'
    py_type = int

class Float(Number):
    db_type = 'REAL'
    py_type = float

# can be True, False, None

""" Objects -- This is a biggie! """
class Object(Column):
    
    db_type = 'TEXT'
                           
    def dump(self, value):
        if value is None:
            return None
        elif isinstance(value, core.Object):
            return value.get_full_id()
        elif isinstance(value, core.Undefined):
            return value.get_full_id()
        elif isinstance(value, core.Deleted):
            return None
        else:
            raise PodColumnError, "Value '" + str(value) + "' is not of type pod.Object . . . "
    
    def load(self, value):
        if value:
            value = value.split(":")
            return self.cls_pod.db.cache.get_inst(cls_id = int(value[0]), inst_id = int(value[1]))
        else:
            return None

""" Collection """    
class Collection(Column):    
    
    db_type = 'TEXT'
    
    def get_str(self, value):
        output = ">"
        for item in list(value):
            output += self.get_element_value(item)
        return output + "<"
    
    def get_element_value(self, value):
        if value is None:
            output = "<n>"
        elif isinstance(value, bool):
            output = "<b" + str(value) + ">"
        elif isinstance(value, str):
            output = "<s" + value.replace("><", "\>\<") + ">"
        elif isinstance(value, int):
            output = "<i" + str(value) + ">"
        elif isinstance(value, float):
            output = "<f" + str(value) + ">"
        elif isinstance(value, core.Object):
            output = "<o" + value.get_full_id() + ">"
        elif isinstance(value, core.Undefined):
            output = "<o" + value.get_full_id() + ">"
        elif isinstance(value, core.Deleted):
            output = "<n>"
        else:
            raise PodColumnError, "Type " + str(type(value)) + " on value " + str(value) + " not supported . . . only NoneType,bool,str,int,float,pod.Object supported at this time"
        return output 
    
    def load(self, value):
        if value is not None:
            values = str(value).split("><")[1:-1]
            new_list = []
            for item in values:
                if item[0] == "n":
                    new_list.append(None)
                elif item[0] == "b":
                    new_list.append(bool(str(item[1:])))
                elif item[0] == "s":
                    new_list.append(str(item[1:]).replace("\>\<", "><"))
                elif item[0] == "i":
                    new_list.append(int(str(item[1:])))
                elif item[0] == "f":
                    new_list.append(float(str(item[1:])))
                elif item[0] == "o":
                    o = str(item[1:]).split(":")
                    new_list.append(self.cls_pod.db.cache.get_inst(cls_id = int(o[0]), inst_id = int(o[1])))
            return new_list
        else:
            return None

    """ COMPARISON OPERATORS """
    def __eq__(self, value):
        if value is None:
            return Condition(left = self, op = ' IS NULL', right = None)
        else:
            return Condition(left = self, op = '=', right = self.get_str(value))

    def __ne__(self, value):
        if value is None:
            return Condition(left = self, op = 'IS NOT NULL', right = None)
        else:
            return Condition(left = self, op = '<>', right = self.get_str(value))

    def contains(self, value):
        return Condition(left = self, op = ' LIKE ',   right = '%' + self.get_element_value(value) + '%')

class List(Collection):
    
    def dump(self, value):
        if value is None:
            return None
        else:
            if isinstance(value, list):
                return self.get_str(value)
            else:
                raise PodColumnError, "Value must be a list for column.List type.  You gave " + str(value) + " . . ."

class Set(Collection):

    def dump(self, value):
    
        if(value is None):
            return None
        else:
            if isinstance(value, set):
                return self.get_str(value)
            else:
                raise PodColumnError, "Value must be a set for column.Set type.  You gave " + str(value) + " . . ."

    def load(self, value):
        return None if value is None else set(Collection.load(self, value))
                   
""" Special columns that do automagic things """

class Time(Int):
    
    @staticmethod
    def utc_timestamp(): 
        #t = datetime.datetime.utcnow()
        #return time.mktime(t.timetuple()) +1e-6*t.microsecond
        return int(time.mktime(datetime.datetime.utcnow().timetuple()))
    
    def __init__(self, index = False, date_string = False, local_time = True, convert_time_stamp = True, cls_pod = None, name = None):
        Int.__init__(self, index = index, cls_pod = cls_pod, name = name)
        self.date_string        = date_string
        self.local_time         = local_time
        self.convert_time_stamp = convert_time_stamp
        self.utc_offset         = Time.utc_timestamp()-int(time.time())

    def load(self, value):
        if self.date_string and self.convert_time_stamp:
            return self.convert_utc_stamp_to_time(Int.load(self, value))
        else:
            return Int.load(self, value)
    
    def dump(self, value):
        if self.date_string and self.convert_time_stamp:
            return Int.dump(self, self.convert_time_to_utc_stamp(value))
        else:
            return Int.dump(self, value)
                
    def older_than(self, years = 0, weeks = 0, days = 0, hours = 0, minutes = 0, seconds = 0):
        ctime = years*365*24*3600 + weeks*7*24*3600 + days*24*3600 + hours*3600 + minutes*60 + seconds
        return self <= int(Time.utc_timestamp()-ctime)
    
    def younger_than(self, years = 0, weeks = 0, days = 0, hours = 0, minutes = 0, seconds = 0):
        ctime = years*365*24*3600 + weeks*7*24*3600 + days*24*3600 + hours*3600 + minutes*60 + seconds
        return self >= int(Time.utc_timestamp()-ctime)
    
    def pre_dates(self, year = None, month = None, day = None, hour = None, minute = None, second = None, time_stamp = None, local_time = None):
        return self <= self.args_to_epoch(year,month,day,hour,minute,second,time_stamp,local_time if local_time is not None else self.local_time)
        
    def post_dates(self, year = None, month = None, day = None, hour = None, minute = None, second = None, time_stamp = None, local_time = None):
        return self >= self.args_to_epoch(year,month,day,hour,minute,second,time_stamp,local_time if local_time is not None else self.local_time)
    
    def is_date(self, year = None, month = None, day = None, local_time = None):
        local_time = local_time if local_time is not None else self.local_time
        if day:
            year = year if year is not None else datetime.datetime.now().year
            month = month if month is not None else datetime.datetime.now().month
            return self.post_dates(year,month,day,None,None,None,None,local_time) & self.pre_dates(year,month,day+1,None,None,None,None,local_time)
        elif month:
            year = year if year is not None else datetime.datetime.now().year
            return self.post_dates(year,month,day,None,None,None,None,local_time) & self.pre_dates(year,month+1,day,None,None,None,None,local_time)
        elif year:
            return self.post_dates(year,month,day,None,None,None,None,local_time) & self.pre_dates(year+1,month,day,None,None,None,None,local_time)
        else:
            raise PodQueryError, "You did not provide a year, month, or day for is_date . . . "

    def args_to_epoch(self,y,m,d,h,n,s,time_stamp,local_time):        
        if time_stamp and self.date_string is None:
            raise PodQueryError, "You provided a time_stamp '" + str(time_stamp) + "' but this column does not have a date_string convertor set . . . "
        elif time_stamp and self.date_string:
            ctime = time.mktime(time.strptime(time_stamp, self.date_string))
        else:
            y = y if y is not None else datetime.datetime.now().year
            m = m if m is not None else 1
            d = d if d is not None else 1
            h = h if h is not None else 0
            n = n if n is not None else 0
            s = s if s is not None else 0        
            ctime = time.mktime(datetime.datetime(y,m,d,h,n,0).timetuple())+s        
        return int(ctime + int(local_time)*self.utc_offset)
        
    def convert_time_to_utc_stamp(self, value, date_string = None, local_time = None):
        date_string = date_string if date_string is not None else self.date_string
        value = datetime.datetime.fromtimestamp(value) if isinstance(value, (int,float,long)) else value
        value = time.strptime(value, date_string) if date_string else value.timetuple()                  
        return int(time.mktime(value) + self.utc_offset*int(local_time if local_time is not None else self.local_time))
    
    def convert_utc_stamp_to_time(self, value, date_string = None, local_time = None):
        date_string = date_string if date_string is not None else self.date_string
        value = int(value-self.utc_offset*int(local_time if local_time is not None else self.local_time))
        return datetime.datetime.fromtimestamp(value).strftime(date_string) if date_string else value
    
class TimeCreate(Time):
    
    def update(self, cls_pod, name):
        Time.update(self, cls_pod = cls_pod, name = name)
        cls_pod.column_callbacks_create.add(self)
        
    def on_inst_create(self, inst, kwargs):
        kwargs = {} if kwargs is None else kwargs
        kwargs[self.name] = self.load(Time.utc_timestamp())
        return inst,kwargs
     
class TimeLoad(Time):
    
    def update(self,cls_pod,name):
        Time.update(self, cls_pod = cls_pod, name = name)
        cls_pod.column_callbacks_load.add(self)        

    def on_inst_create(self, inst, kwargs):
        kwargs = {} if kwargs is None else kwargs
        kwargs[self.name] = self.load(Time.utc_timestamp())
        return inst,kwargs
    
    def on_inst_load(self, inst):
        object.__getattribute__(inst, '__setattr__')(self.name, self.load(Time.utc_timestamp()))
        return inst
        

""" Pickled """
class Pickle(Column):
    
    db_type = 'BINARY'
                     
    def on_activate(self):
        self.pickler = self.cls_pod.db.pickler
    
    def dump(self, value):
        return self.pickler.dump(value)

    def load(self, value):
        return self.pickler.load(value)

class Order(Expression):
    
    def __init__(self, col):
        Expression.__init__(self)
        self.col = col
        
    
    def get_my_sql(self, head):
        head.expression += self.col.get_sql_column_name(head) + " " + self.TYPE

class OrderAsc(Order):
    TYPE = 'ASC'

class OrderDesc(Order):
    TYPE = 'DESC'
        
        


