use std::sync::atomic::{AtomicU32, Ordering};

use criterion::{BatchSize, BenchmarkId, Criterion};
use relay_base_schema::{metrics::MetricNamespace, project::ProjectId};
use relay_cardinality::{
    limiter::{Entry, EntryId, Limiter, Scoping},
    CardinalityLimit, CardinalityScope, RedisSetLimiter, SlidingWindow,
};
use relay_redis::{redis, RedisConfigOptions, RedisPool};

fn build_redis() -> RedisPool {
    let url =
        std::env::var("RELAY_REDIS_URL").unwrap_or_else(|_| "redis://127.0.0.1:6379".to_owned());

    RedisPool::single(&url, RedisConfigOptions::default()).unwrap()
}

fn build_limiter(redis: RedisPool, reset_redis: bool) -> RedisSetLimiter {
    let mut client = redis.client().unwrap();
    let mut connection = client.connection().unwrap();

    if reset_redis {
        redis::cmd("FLUSHALL").execute(&mut connection);
    }

    RedisSetLimiter::new(redis)
}

fn build_scoping() -> Scoping {
    Scoping {
        organization_id: 1,
        project_id: ProjectId::new(100),
    }
}

#[derive(Debug)]
struct Params {
    limits: Vec<CardinalityLimit>,

    rounds: usize,
    num_hashes: usize,
}

impl Params {
    fn new(limit: u64, rounds: usize, num_hashes: usize) -> Self {
        Self {
            limits: vec![CardinalityLimit {
                id: "limit".to_owned(),
                window: SlidingWindow {
                    window_seconds: 3600,
                    granularity_seconds: 360,
                },
                limit,
                scope: CardinalityScope::Organization,
                namespace: None,
            }],
            rounds,
            num_hashes,
        }
    }

    /// Every round contains the same hashes.
    fn rounds(&self) -> Vec<Vec<Entry>> {
        let entries = (0..self.num_hashes)
            .map(|i| Entry::new(EntryId(i), MetricNamespace::Custom, u32::MAX - (i as u32)))
            .collect::<Vec<_>>();

        (0..self.rounds)
            .map(move |_| entries.clone())
            .collect::<Vec<_>>()
    }

    /// High cardinality, every round contains unique hashes.
    fn rounds_unique(&self) -> Vec<Vec<Entry>> {
        let hash = AtomicU32::new(u32::MAX);

        (0..self.rounds)
            .map(move |_| {
                (0..self.num_hashes)
                    .map(|i| {
                        Entry::new(
                            EntryId(i),
                            MetricNamespace::Custom,
                            hash.fetch_sub(1, Ordering::SeqCst),
                        )
                    })
                    .collect::<Vec<_>>()
            })
            .collect::<Vec<_>>()
    }

    /// Entry which is never generated by either [`Self::rounds`] or [`Self::rounds_unique`].
    fn never_entry() -> Entry {
        Entry::new(EntryId(usize::MAX), MetricNamespace::Custom, 0)
    }

    /// A vector of entries which is never generated by either [`Self::rounds`] or [`Self::rounds_unique`].
    fn never_entries(&self) -> Vec<Entry> {
        (0..self.limits[0].limit)
            .map(|i| {
                Entry::new(
                    EntryId(usize::MAX - i as usize),
                    MetricNamespace::Custom,
                    i as u32,
                )
            })
            .collect::<Vec<_>>()
    }
}

impl std::fmt::Display for Params {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        write!(f, "{:?}", self)
    }
}

pub fn bench_simple(c: &mut Criterion) {
    let params = [
        Params::new(10_000, 1000, 50),
        Params::new(10_000, 1000, 500),
        Params::new(10_000, 50, 10_000),
    ];

    let redis = build_redis();
    let scoping = build_scoping();

    let mut g = c.benchmark_group("Simple");

    for params in params {
        g.throughput(criterion::Throughput::Elements(params.rounds as u64));
        g.bench_with_input(BenchmarkId::new("simple", &params), &params, |b, params| {
            b.iter_batched(
                || {
                    let limiter = build_limiter(redis.clone(), true);
                    (limiter, params.rounds())
                },
                |(limiter, rounds)| {
                    for entries in rounds {
                        limiter
                            .check_cardinality_limits(scoping, &params.limits, entries)
                            .unwrap()
                            .for_each(|_| {});
                    }
                },
                BatchSize::SmallInput,
            )
        });
    }
}

pub fn bench_big_set_small_queries(c: &mut Criterion) {
    let redis = build_redis();
    let scoping = build_scoping();

    let params = Params::new(10_000, 1000, 50);

    let mut g = c.benchmark_group("Big set small queries");
    g.throughput(criterion::Throughput::Elements(params.rounds as u64));
    g.bench_function("big_set_small_queries", |b| {
        b.iter_batched(
            || {
                let limiter = build_limiter(redis.clone(), true);
                let rounds = params.rounds();

                // Seed with the round data
                limiter
                    .check_cardinality_limits(scoping, &params.limits, rounds[0].clone())
                    .unwrap()
                    .for_each(|_| {});

                (limiter, rounds)
            },
            |(limiter, rounds)| {
                for entries in rounds {
                    limiter
                        .check_cardinality_limits(scoping, &params.limits, entries)
                        .unwrap()
                        .for_each(|_| {});
                }
            },
            BatchSize::SmallInput,
        )
    });
}

pub fn bench_high_cardinality(c: &mut Criterion) {
    let redis = build_redis();
    let scoping = build_scoping();

    let params = Params::new(10_000, 1000, 50);

    let mut g = c.benchmark_group("High Cardinality");
    g.throughput(criterion::Throughput::Elements(params.rounds as u64));
    g.bench_function("high_cardinality", |b| {
        b.iter_batched(
            || {
                let limiter = build_limiter(redis.clone(), true);
                (limiter, params.rounds_unique())
            },
            |(limiter, rounds)| {
                for entries in rounds {
                    limiter
                        .check_cardinality_limits(scoping, &params.limits, entries)
                        .unwrap()
                        .for_each(|_| {});
                }
            },
            BatchSize::SmallInput,
        )
    });
}

pub fn bench_cache_never_full(c: &mut Criterion) {
    let redis = build_redis();
    let scoping = build_scoping();

    let params = Params::new(10_000, 1_000, 50);

    let mut g = c.benchmark_group("Cache Never Full");
    g.throughput(criterion::Throughput::Elements(params.rounds as u64));
    g.bench_function("cache_never_full", |b| {
        b.iter_batched(
            || {
                let limiter = build_limiter(redis.clone(), true);
                limiter
                    .check_cardinality_limits(scoping, &params.limits, vec![Params::never_entry()])
                    .unwrap()
                    .for_each(|_| {});

                // New limiter to reset cache.
                let limiter = build_limiter(redis.clone(), false);
                (limiter, params.rounds_unique())
            },
            |(limiter, rounds)| {
                for entries in rounds {
                    limiter
                        .check_cardinality_limits(scoping, &params.limits, entries)
                        .unwrap()
                        .for_each(|_| {});
                }
            },
            BatchSize::SmallInput,
        )
    });
}

pub fn bench_cache_worst_case(c: &mut Criterion) {
    let redis = build_redis();
    let scoping = build_scoping();

    let params = Params::new(10_000, 1000, 50);

    let mut g = c.benchmark_group("Cache Worst Case");
    g.throughput(criterion::Throughput::Elements(params.rounds as u64));
    g.bench_function("cache_worst_case", |b| {
        b.iter_batched(
            || {
                let limiter = build_limiter(redis.clone(), true);
                limiter
                    .check_cardinality_limits(scoping, &params.limits, params.never_entries())
                    .unwrap()
                    .for_each(|_| {});

                // New limiter to reset cache.
                let limiter = build_limiter(redis.clone(), false);
                (limiter, params.rounds_unique())
            },
            |(limiter, rounds)| {
                for entries in rounds {
                    limiter
                        .check_cardinality_limits(scoping, &params.limits, entries)
                        .unwrap()
                        .for_each(|_| {});
                }
            },
            BatchSize::SmallInput,
        )
    });
}
