# Copyright (c) 2015, The MITRE Corporation. All rights reserved.
# See LICENSE.txt for complete terms.

import stix
import stix.bindings.exploit_target as exploit_target_binding
from stix.common import StructuredTextList
from mixbox import entities, fields

class Weakness(stix.Entity):
    """Implementation of STIX ``Weakness``.

    Args:
        cwe_id(optional): Common Weakness Enumeration value as a string
        description (optional): A string description.

    """
    _binding = exploit_target_binding
    _binding_class = _binding.WeaknessType
    _namespace = "http://docs.oasis-open.org/cti/ns/stix/exploit-target-1"

    descriptions = fields.TypedField("Description", type_="stix.common.StructuredTextList")
    cwe_id = fields.TypedField("CWE_ID")

    def __init__(self, description=None, cwe_id=None):
        super(Weakness, self).__init__()
        self.description = StructuredTextList(description)
        self.cwe_id = cwe_id

    @property
    def description(self):
        """A single description about the contents or purpose of this object.

        Default Value: ``None``

        Note:
            If this object has more than one description set, this will return
            the description with the lowest ordinality value.

        Returns:
            An instance of :class:`.StructuredText`
        """
        return next(iter(self.descriptions or []), None)

    @description.setter
    def description(self, value):
        self.descriptions = value

    def add_description(self, description):
        """Adds a description to the ``descriptions`` collection.

        This is the same as calling "foo.descriptions.add(bar)".
        """
        self.descriptions.add(description)

class _Weaknesses(stix.TypedList):
    _contained_type = Weakness
