# Copyright (c) 2015, The MITRE Corporation. All rights reserved.
# See LICENSE.txt for complete terms.

import stix
from stix.common import StructuredTextList
import stix.bindings.exploit_target as exploit_target_binding
from mixbox import fields, entities

class Configuration(stix.Entity):
    """Implementation of STIX ``Configuration``.

    Args:
        cce_id(optional): Common Configuration Enumeration value as a string
        description (optional): A string description.
        short_description (optional): A string short description.

    """
    _binding = exploit_target_binding
    _binding_class = _binding.ConfigurationType
    _namespace = "http://docs.oasis-open.org/cti/ns/stix/exploit-target-1"

    descriptions = fields.TypedField("Description", type_="stix.common.StructuredTextList")
    short_descriptions = fields.TypedField("Short_Description", type_="stix.common.StructuredTextList")
    cce_id = fields.TypedField("CCE_ID")

    def __init__(self, description=None, short_description=None, cce_id=None):
        super(Configuration, self).__init__()
        self.description = StructuredTextList(description)
        self.short_description = StructuredTextList(short_description)
        self.cce_id = cce_id

    @property
    def description(self):
        """A single description about the contents or purpose of this object.

        Default Value: ``None``

        Note:
            If this object has more than one description set, this will return
            the description with the lowest ordinality value.

        Returns:
            An instance of :class:`.StructuredText`
        """
        if self.descriptions is None:
            return None
        return next(iter(self.descriptions), None)

    @description.setter
    def description(self, value):
        self.descriptions = value

    def add_description(self, description):
        """Adds a description to the ``descriptions`` collection.

        This is the same as calling "foo.descriptions.add(bar)".
        """
        if self.descriptions is None:
            self.descriptions = StructuredTextList()
        self.descriptions.add(description)

    @property
    def short_description(self):
        """A single short description about the contents or purpose of this
        object.

        Default Value: ``None``

        Note:
            If this object has more than one short description set, this will
            return the description with the lowest ordinality value.

        Returns:
            An instance of :class:`.StructuredText`
        """
        if self.short_descriptions is None:
            self.short_descriptions = StructuredTextList()
        return next(iter(self.short_descriptions), None)

    @short_description.setter
    def short_description(self, value):
        self.short_descriptions = value

    def add_short_description(self, description):
        """Adds a description to the ``short_descriptions`` collection.

        This is the same as calling "foo.short_descriptions.add(bar)".
        """
        self.short_descriptions.add(description)
