"""Importing this module enables command line editing using GNU readline."""

import pygnurl

_READLINE_VERSION = pygnurl.readline.version
_READLINE_RUNTIME_VERSION = pygnurl.readline.version

_history_length = -1  # pylint: disable=invalid-name
_completer = None  # pylint: disable=invalid-name
_begidx = _endidx = 0  # pylint: disable=invalid-name

pygnurl.readline.completion.word_break_characters = \
    " \t\n`~!@#$%^&*()-=+[{]}\\|;:'\",<>/?"


def parse_and_bind(string):
    """Parse and execute single line of a readline init file."""
    pygnurl.readline.parse_and_bind(string)


def get_line_buffer():
    """Return the current contents of the line buffer."""
    return pygnurl.readline.line_buffer


def insert_text(text):
    """Insert text into the command line."""
    pygnurl.readline.insert_text(text)


def read_init_file(filename=None):
    """Parse a readline initialization file.
    The default filename is the last filename used.
    """
    pygnurl.readline.read_init_file(filename)


def read_history_file(filename=None):
    """Load a readline history file.
    The default filename is ~/.history.
    """
    pygnurl.readline.history.read_file(filename)


def write_history_file(filename=None):
    """Save a readline history file.
    The default filename is ~/.history.
    """
    pygnurl.readline.history.write_file(filename)
    if _history_length >= 0:
        pygnurl.readline.history.truncate_file(_history_length, filename)


def clear_history():
    """Clear the current readline history."""
    pygnurl.readline.history.clear()


def get_history_length():
    """Return the maximum number of items that will be written to
    the history file.
    """
    return _history_length


def set_history_length(length):
    """Set the maximal number of items which will be written to
    the history file. A negative length is used to inhibit
    history truncation.
    """
    global _history_length  # pylint: disable=global-statement,invalid-name
    _history_length = length


def get_current_history_length():
    """Return the current (not the maximum) length of history."""
    return len(pygnurl.readline.history)


def get_history_item(index):
    """Return the current contents of history item at index."""
    # Old implementation is relative to history_base.
    index -= pygnurl.readline.history.base
    try:
        return pygnurl.readline.history[index]
    except IndexError:
        return None


def remove_history_item(pos):
    """Remove history item given by its position."""
    if pos < 0:
        raise ValueError('History index cannot be negative')
    try:
        del pygnurl.readline.history[pos]
    except IndexError:
        raise ValueError('No history item at position {}'.format(pos))


def replace_history_item(pos, line):
    """Replaces history item given by its position with contents of line."""
    if pos < 0:
        raise ValueError('History index cannot be negative')
    try:
        pygnurl.readline.history[pos] = line
    except:
        raise ValueError('No history item at position {}'.format(pos))


def redisplay():
    """Change what's displayed on the screen to reflect the current
    contents of the line buffer.
    """
    pygnurl.readline.redisplay()


def set_startup_hook(function=None):
    """Set or remove the startup_hook function.
    The function is called with no arguments just
    before readline prints the first prompt.
    """
    pygnurl.readline.startup_hook = function


def set_pre_input_hook(function=None):
    """Set or remove the pre_input_hook function.
    The function is called with no arguments after the first prompt
    has been printed and just before readline starts reading input
    characters.
    """
    pygnurl.readline.pre_input_hook = function


def set_completer(function=None):
    """Set or remove the completer function.
    The function is called as function(text, state),
    for state in 0, 1, 2, ..., until it returns a non-string.
    It should return the next possible completion starting with 'text'.
    """
    global _completer  # pylint: disable=global-statement,invalid-name
    _completer = function

    if function is not None:
        pygnurl.readline.completion.completer = _get_completions
    else:
        pygnurl.readline.completion.completer = None


def _get_completions(text, start, end):
    """Returns the list of completions generated by _completer.

    Also stores start and end globally for later retrieval by
    get_begidx() and get_endidx().
    """
    global _begidx, _endidx  # pylint: disable=global-statement,invalid-name
    _begidx, _endidx = start, end

    pygnurl.readline.completion.append_character = '\0'
    pygnurl.readline.completion.suppress_append = False

    completions = []
    while True:
        completion = _completer(text, len(completions))
        if completion is None:
            break
        completions.append(completion)

    return completions


def get_completer():
    """Returns current completer function."""
    return _completer


def get_completion_type():
    """Get the type of completion being attempted."""
    return ord(pygnurl.readline.completion.type)


def get_begidx():
    """Get the beginning index of the readline tab-completion scope."""
    return _begidx


def get_endidx():
    """Get the ending index of the readline tab-completion scope."""
    return _endidx


def set_completer_delims(string):
    """Set the readline word delimiters for tab-completion."""
    pygnurl.readline.completion.word_break_characters = string


def get_completer_delims():
    """Get the readline word delimiters for tab-completion."""
    return pygnurl.readline.completion.word_break_characters


def set_completion_display_matches_hook(function=None):
    """Set or remove the completion display function.
    The function is called as
      function(substitution, [matches], longest_match_length)
    once each time matches need to be displayed.
    """
    # pylint: disable=invalid-name
    pygnurl.readline.completion_display_matches_hook = function


def add_history(line):
    """Add a line to the history buffer."""
    pygnurl.readline.history.append(line)


def GetOutputFile():  # pylint: disable=invalid-name
    """Return the console object used by readline
    so that it can be used for printing in color.
    """
    import sys

    if sys.platform != 'win32':
        # This function is only necessary to imitate pyreadline.
        raise NotImplementedError

    # For IPython, fake pyreadline's GetOutputFile function.
    import colorama.initialise

    colorama.initialise.init()
    colorama.initialise.deinit()

    output_file = colorama.initialise.wrapped_stdout

    try:
        # pylint: disable=protected-access,no-member
        wrapped = output_file._StreamWrapper__wrapped
    except AttributeError:
        pass
    else:
        # colorama is proxying method calls using __getattr__ but
        # IPython.utils.io.IOStream is looking through the dir, so we
        # have to promote any proxied attributes to real ones.
        for attr in dir(wrapped):
            if attr not in dir(output_file):
                proxied = getattr(output_file, attr)
                setattr(output_file, attr, proxied)

    return output_file
