#!/usr/bin/env python
"""Library for processing of artifacts.

This file contains non-GRR specific pieces of artifact processing and is
intended to end up as an independent library.
"""

import re
from typing import Iterable, Optional

from grr_response_core.lib import interpolation
from grr_response_proto import knowledge_base_pb2


class Error(Exception):
  """Base exception."""


class ConditionError(Error):
  """An invalid artifact condition was specified."""


class ArtifactProcessingError(Error):
  """Unable to process artifact."""


class KbInterpolationMissingAttributesError(Error):
  """An exception class for missing knowledgebase attributes."""

  def __init__(self, attrs: Iterable[str]) -> None:
    message = "Some attributes could not be located in the knowledgebase: {}"
    message = message.format(", ".join(attrs))
    super().__init__(message)

    self.attrs = list(attrs)


class KbInterpolationUnknownAttributesError(Error):
  """An exception class for non-existing knowledgebase attributes."""

  def __init__(self, attrs: Iterable[str]) -> None:
    message = "Some attributes are not part of the knowledgebase: {}"
    message = message.format(", ".join(attrs))
    super().__init__(message)

    self.attrs = list(attrs)


class KnowledgeBaseUninitializedError(Error):
  """Attempt to process artifact without a valid Knowledge Base."""


class KnowledgeBaseAttributesMissingError(Error):
  """Knowledge Base is missing key attributes."""


INTERPOLATED_REGEX = re.compile(r"%%([^%]+?)%%")


def InterpolateKbAttributes(
    pattern: str,
    knowledge_base: Optional[knowledge_base_pb2.KnowledgeBase],
) -> Iterable[str]:
  """Interpolate all knowledgebase attributes in pattern.

  Args:
    pattern: A string with potential interpolation markers. For example:
      "/home/%%users.username%%/Downloads/"
    knowledge_base: The knowledge_base to interpolate parameters from, if
      knowledge_base is None, then the pattern must not have placeholders.

  Raises:
    KbInterpolationMissingAttributesError: If any of the required pattern
      parameters is not present in the knowledgebase.
    KbInterpolationUnknownAttributesError: If any of the specified pattern
      parameters is not a valid knowledgebase attribute.
    KnowledgeBaseUninitializedError: If the pattern requires knowledgebase
      attributes, but the knowledgebase is not initialized.

  Returns:
    An iterator over all unique strings generated by expanding the pattern.
  """

  # TODO(hanuszczak): Control flow feels a bit awkward here because of error
  # handling that tries not to break any functionality. With the new utilities
  # it should be possible to improve the code, changing the behaviour to a more
  # sane one.
  interpolator = interpolation.Interpolator(pattern)

  if not knowledge_base:
    if interpolator.Vars() or interpolator.Scopes():
      raise KnowledgeBaseUninitializedError(
          "Knowledge base is not initialized, but the pattern requires it."
      )
    return interpolator.Interpolate()

  missing_attr_names = set()
  unknown_attr_names = set()

  for var_id in interpolator.Vars():
    var_name = interpolation.GetVarName(var_id)

    if var_name not in knowledge_base.DESCRIPTOR.fields_by_name:
      unknown_attr_names.add(var_name)
      continue

    value = getattr(knowledge_base, var_name)
    if not value:
      missing_attr_names.add(var_name)
      continue

    interpolator.BindVar(var_name, value)  # pytype: disable=wrong-arg-types

  for scope_id in interpolator.Scopes():
    scope_name = interpolation.GetScopeName(scope_id)

    # We are currently only having one scope which is `users`. Restricting the
    # implementation to this, to not having to differentiate between nested
    # lists, protos, deeply nested scopes etc.
    if scope_name != "users":
      unknown_attr_names.add(scope_name)
      continue

    users = knowledge_base.users
    if not users:
      missing_attr_names.add(scope_name)
      continue

    scope_var_ids = interpolator.ScopeVars(scope_id)
    scope_bound = False
    scope_missing_var_names = set()
    for user in users:
      bindings = {}
      for scope_var_id in scope_var_ids:
        scope_var_name = interpolation.GetVarName(scope_var_id)

        if scope_var_name not in user.DESCRIPTOR.fields_by_name:
          unknown_attr_names.add(f"{scope_name}.{scope_var_name}")
          continue

        value = getattr(user, scope_var_name)
        if not value:
          scope_missing_var_names.add(f"{scope_name}.{scope_var_name}")
          continue

        bindings[scope_var_id] = value

      if set(bindings) == set(scope_var_ids):
        interpolator.BindScope(scope_id, bindings)
        scope_bound = True

    if not scope_bound:
      missing_attr_names.update(scope_missing_var_names)

  if unknown_attr_names:
    raise KbInterpolationUnknownAttributesError(unknown_attr_names)

  if missing_attr_names:
    raise KbInterpolationMissingAttributesError(missing_attr_names)

  return interpolator.Interpolate()


def GetWindowsEnvironmentVariablesMap(knowledge_base):
  """Return a dictionary of environment variables and their values.

  Implementation maps variables mentioned in
  https://en.wikipedia.org/wiki/Environment_variable#Windows to known
  KB definitions.

  Args:
    knowledge_base: A knowledgebase object.

  Returns:
    A dictionary built from a given knowledgebase object where keys are
    variables names and values are their values.
  """

  environ_vars = {}

  if knowledge_base.environ_path:
    environ_vars["path"] = knowledge_base.environ_path

  if knowledge_base.environ_temp:
    environ_vars["temp"] = knowledge_base.environ_temp

  if knowledge_base.environ_systemroot:
    environ_vars["systemroot"] = knowledge_base.environ_systemroot

  if knowledge_base.environ_windir:
    environ_vars["windir"] = knowledge_base.environ_windir

  if knowledge_base.environ_programfiles:
    environ_vars["programfiles"] = knowledge_base.environ_programfiles
    environ_vars["programw6432"] = knowledge_base.environ_programfiles

  if knowledge_base.environ_programfilesx86:
    environ_vars["programfiles(x86)"] = knowledge_base.environ_programfilesx86

  if knowledge_base.environ_systemdrive:
    environ_vars["systemdrive"] = knowledge_base.environ_systemdrive

  if knowledge_base.environ_allusersprofile:
    environ_vars["allusersprofile"] = knowledge_base.environ_allusersprofile
    environ_vars["programdata"] = knowledge_base.environ_allusersprofile

  if knowledge_base.environ_allusersappdata:
    environ_vars["allusersappdata"] = knowledge_base.environ_allusersappdata

  for user in knowledge_base.users:
    if user.appdata:
      environ_vars.setdefault("appdata", []).append(user.appdata)

    if user.localappdata:
      environ_vars.setdefault("localappdata", []).append(user.localappdata)

    if user.userdomain:
      environ_vars.setdefault("userdomain", []).append(user.userdomain)

    if user.userprofile:
      environ_vars.setdefault("userprofile", []).append(user.userprofile)

  return environ_vars


def ExpandWindowsEnvironmentVariables(data_string, knowledge_base):
  r"""Take a string and expand any windows environment variables.

  Args:
    data_string: A string, e.g. "%SystemRoot%\\LogFiles"
    knowledge_base: A knowledgebase object.

  Returns:
    A string with available environment variables expanded. If we can't expand
    we just return the string with the original variables.
  """
  win_environ_regex = re.compile(r"%([^%]+?)%")
  components = []
  offset = 0
  for match in win_environ_regex.finditer(data_string):
    components.append(data_string[offset : match.start()])

    # KB environment variables are prefixed with environ_.
    kb_value = getattr(
        knowledge_base, "environ_%s" % match.group(1).lower(), None
    )
    if isinstance(kb_value, str) and kb_value:
      components.append(kb_value)
    else:
      # Failed to expand, leave the variable as it was.
      components.append("%%%s%%" % match.group(1))
    offset = match.end()
  components.append(data_string[offset:])  # Append the final chunk.
  return "".join(components)
