import re
from unittest.mock import MagicMock, Mock

from cloudshell.snmp.autoload.services.system_info_table import SnmpSystemInfo

# test_system_info_table.py - Generated by CodiumAI


"""
Code Analysis:
- The class 'SnmpSystemInfo' is responsible for retrieving system information from SNMPv2 MIB data.
- It takes in a SNMP handler, logger, and vendor as input parameters.
- It has methods to set the file path for the device model map, device model pattern, and OS version pattern.
- It has a method to validate the device OS using SNMP and a list of supported OS patterns.
- It has methods to retrieve the device model, vendor, and device OS version from the SNMPv2 MIB data.
- It has a method to fill the attributes of a resource model with the retrieved system information.
- The class uses regular expressions to match patterns in the SNMPv2 MIB data.
- It has a dependency on the 'SnmpV2MibData' class, which is responsible for retrieving SNMPv2 MIB data.
- The class is designed to be used as part of an autoload process for network devices.
"""

"""
Test Plan:
- test_is_valid_device_os_with_supported_os_as_string(): tests that the method returns True when the supported OS is provided as a string. Tags: [happy path]
- test_is_valid_device_os_with_supported_os_as_list(): tests that the method returns True when the supported OS is provided as a list. Tags: [happy path]
- test_is_valid_device_os_with_empty_system_description(): tests that the method returns False when the system description is empty. Tags: [edge case]
- test_is_valid_device_os_with_unsupported_os(): tests that the method returns False when the device OS is not supported. Tags: [edge case]
- test_is_valid_device_os_with_supported_os_as_pattern(): tests that the method returns True when the supported OS is provided as a regular expression pattern. Tags: [happy path]
- test_get_device_model(): tests that the method returns the correct device model. Tags: [happy path]
- test_get_device_model_with_no_match(): tests that the method returns an empty string when there is no match for the device model pattern. Tags: [edge case]
- test_get_vendor(): tests that the method returns the correct vendor. Tags: [happy path]
- test_get_vendor_with_no_match(): tests that the method returns the default vendor when there is no match for the vendor OID pattern. Tags: [edge case]
- test_get_device_os_version(): tests that the method returns the correct device OS version. Tags: [happy path]
- test_fill_attributes(): tests that the method fills the resource attributes correctly. Tags: [happy path]
"""


class TestSnmpSystemInfo:
    def test_is_valid_device_os_with_supported_os_as_string(self):
        snmp_handler = Mock()
        logger = Mock()
        snmp_system_info = SnmpSystemInfo(snmp_handler, logger)
        snmp_system_info._snmp_v2_obj.get_system_description = Mock(
            return_value="Cisco IOS Software, "
            "C2960X Software (C2960X-UNIVERSALK9-M), "
            "Version 15.2(2)E7, RELEASE SOFTWARE (fc3)"
        )
        assert snmp_system_info.is_valid_device_os("IOS") is True

    def test_is_valid_device_os_with_supported_os_as_list(self):
        snmp_handler = Mock()
        logger = Mock()
        snmp_system_info = SnmpSystemInfo(snmp_handler, logger)
        snmp_system_info._snmp_v2_obj.get_system_description = Mock(
            return_value="Cisco IOS Software, C2960X Software "
            "(C2960X-UNIVERSALK9-M), Version 15.2(2)E7, "
            "RELEASE SOFTWARE (fc3)"
        )
        assert snmp_system_info.is_valid_device_os(["IOS", "Software"]) is True

    def test_is_valid_device_os_with_empty_system_description(self):
        snmp_handler = Mock()
        logger = Mock()
        snmp_system_info = SnmpSystemInfo(snmp_handler, logger)
        snmp_system_info._snmp_v2_obj.get_system_description = Mock(return_value="")
        assert snmp_system_info.is_valid_device_os("IOS") is False

    def test_is_valid_device_os_with_unsupported_os(self):
        snmp_handler = Mock()
        logger = Mock()
        snmp_system_info = SnmpSystemInfo(snmp_handler, logger)
        snmp_system_info._snmp_v2_obj.get_system_description = Mock(
            return_value="Cisco IOS Software, C2960X Software (C2960X-UNIVERSALK9-M), "
            "Version 15.2(2)E7, RELEASE SOFTWARE (fc3)"
        )
        assert snmp_system_info.is_valid_device_os("JunOS") is False

    def test_is_valid_device_os_with_supported_os_as_pattern(self):
        snmp_handler = Mock()
        logger = Mock()
        snmp_system_info = SnmpSystemInfo(snmp_handler, logger)
        snmp_system_info._snmp_v2_obj.get_system_description = Mock(
            return_value="Cisco IOS Software, C2960X Software (C2960X-UNIVERSALK9-M), "
            "Version 15.2(2)E7, RELEASE SOFTWARE (fc3)"
        )
        assert snmp_system_info.is_valid_device_os(re.compile(r"IOS")) is True

    def test_get_device_model(self):
        snmp_handler = Mock()
        logger = Mock()
        create_autospec_mock = MagicMock()
        create_autospec_mock.__str__.return_value = "1.3.6.1.4.1.9.1.1234"
        snmp_system_info = SnmpSystemInfo(snmp_handler, logger)
        snmp_system_info._device_model_pattern = re.compile(
            r"1\.3\.6\.1\.4\.1\.\d+\.(" r"?P<model>\S+$)"
        )
        snmp_system_info._snmp_v2_obj = Mock()
        snmp_system_info._snmp_v2_obj.get_system_object_id = Mock(
            return_value=create_autospec_mock
        )
        assert snmp_system_info._get_device_model() == "1.1234"
