__author__ = 'Brian Bradley'
__version__ = '1.2.6'

from rpisoc import *
from math import log

class digitalPin(object):
    """
    :Class:

        Provides functionality for use of the GPIO on the RPiSoC as a digital input or output, with a number of different drive mode configurations.

    :Example:

        Define a digitalPin object in the following way::

            My_input = digitalPin(Port,Pin,'IN')
            My_output = digitalPin(Port,Pin,'OUT')
            '''or more generally'''
            My_pin = digitalPin(Port,Pin,DRIVE_MODE)

    :Note:

            Modification of which pins are available should be easy. Just add new pins to you RPiSoC schematic, give them an input and an output, and then name them as GPIO_PORT_PIN.
            After you do this, confirm that it was done correctly by using the DEBUG feature of the RPiSoC class.
    """

    def __init__(self,PORT, PIN, CONFIG = None):
        """
        +-------------------+--------------------------------------------------------------------------------------------------------------------------------------------------+
        | Method            | __init__                                                                                                                                         |
        +===================+==================================================================================================================================================+
        | **Description**   |Constructs and initializes a digitalPin object                                                                                                    |
        +-------------------+--------------------------------------------------------------------------------------------------------------------------------------------------+
        | **Parameters**    |* **int**      *PORT*                                                                                                                             |
        |                   |* **int**      *PIN*                                                                                                                              |
        |                   |* **string**   *CONFIG*                                                                                                                           |
        +-------------------+--------------------------------------------------------------------------------------------------------------------------------------------------+
        | *PORT*            |Port on the RPiSoC                                                                                                                                |
        |                   |   * GPIO Ports are chosen by default as Ports *2, 5* and *12* for V1.2, but they are easily changed                                              |
        |                   |   * Port *4* does not contain Pin *7* (it is reserved)                                                                                           |
        |                   |   * Port 5 uses 3.3V logic instead of 5V, as do pins 12[6] and 12[7].                                                                            |
        +-------------------+--------------------------------------------------------------------------------------------------------------------------------------------------+
        | *PIN*             |Pin relative to the chosen Port                                                                                                                   |
        |                   |   * Valid arguments are between *0* and *7*, unless noted to have a Port specific exception, or if that pin is removed from your RPiSoC schematic|
        +-------------------+--------------------------------------------------------------------------------------------------------------------------------------------------+
        | *CONFIG*          |Drive mode of the pin. Valid choices are:                                                                                                         |
        |                   |   * 'IN' - sets pin as input                                                                                                                     |
        |                   |   * 'OUT' - sets pin as output                                                                                                                   |
        |                   |   * 'PULL_UP' - sets pin as resisitive pullup                                                                                                    |
        |                   |   * 'PULL_DWN' - sets pin as resistive pulldown                                                                                                  |
        |                   |   * 'OPEN_DRAIN_LO' - sets pin as open-drain (drives low)                                                                                        |
        |                   |   * 'OPEN_DRAIN_HI' - sets pin as open-drain (drives high)                                                                                       |
        |                   |   * 'PULL_UP_DWN' - sets pin as resistive pull-up/down                                                                                           |
        |                   |   * 'HIGH_Z_DIG' - sets pin as high impedence (digital)                                                                                          |
        |                   |   * 'STRONG_DRIVE' - sets pin as strong drive                                                                                                    |
        +-------------------+--------------------------------------------------------------------------------------------------------------------------------------------------+
        | **Returns**       |No return                                                                                                                                         |
        +-------------------+--------------------------------------------------------------------------------------------------------------------------------------------------+
        """

        self.address = RPiSoC.GPIO_REGISTER


        if int(PORT) not in RPiSoC.GPIO.keys():
            raise ValueError('Invalid PORT: Port numbers found on RPiSoC are ', RPiSoC.GPIO.keys())
        else:
            self.port = int(PORT)

        if int(PIN) not in RPiSoC.GPIO[self.port]:
            raise ValueError('Invalid PIN: the second argument should be the pin number relative to the desired port. Valid entries on this port are ', RPiSoC.GPIO[self.port])
        else:
            self.pin = int(PIN)

        self.Configure(CONFIG)
        self.Read()

    def Configure(self, CONFIG):
        """
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | Method            | Configure                                                                                                                                                                                                     |
        +===================+===============================================================================================================================================================================================================+
        | **Description**   |Sets the drive mode of the pin as input (high impedence digital), resistive pullup, resistive pull down, open drain (drives low), open drain (drives high), output (strong drive), or resistive pull-up/down   |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | **Parameters**    |* **string**   *CONFIG*                                                                                                                                                                                        |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | *CONFIG*          |Drive mode of the pin. Valid choices are:                                                                                                                                                                      |
        |                   |   * 'IN' - sets pin as input                                                                                                                                                                                  |
        |                   |   * 'OUT' - sets pin as output                                                                                                                                                                                |
        |                   |   * 'PULL_UP' - sets pin as resisitive pullup                                                                                                                                                                 |
        |                   |   * 'PULL_DWN' - sets pin as resistive pulldown                                                                                                                                                               |
        |                   |   * 'OPEN_DRAIN_LO' - sets pin as open-drain (drives low)                                                                                                                                                     |
        |                   |   * 'OPEN_DRAIN_HI' - sets pin as open-drain (drives high)                                                                                                                                                    |
        |                   |   * 'PULL_UP_DWN' - sets pin as resistive pull-up/down                                                                                                                                                        |
        |                   |   * 'HIGH_Z_DIG' - sets pin as high impedence (digital)                                                                                                                                                       |
        |                   |   * 'STRONG_DRIVE' - sets pin as strong drive                                                                                                                                                                 |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | **Returns**       |No return                                                                                                                                                                                                      |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        """

        cmd = 0x03
        if CONFIG == 'IN' or CONFIG == 'HIGH_Z_DIG' or CONFIG == 'input':
            self.config = 0x02
        elif CONFIG == 'PULL_UP' or CONFIG == 'pull up':
            self.config = 0x03
        elif CONFIG == 'PULL_DWN':
            self.config = 0x04
        elif CONFIG == 'OPEN_DRAIN_LO':
            self.config = 0x05
        elif CONFIG == 'OPEN_DRAIN_HI':
            self.config = 0x06
        elif CONFIG == 'OUT' or CONFIG == 'STRONG_DRIVE' or CONFIG == 'output':
            self.config = 0x07
        elif CONFIG == 'PULL_UP_DWN':
            self.config = 0x08
        elif CONFIG is None:
            self.config = None
        else:
            raise ValueError('Invalid pin configuration: Choose one of the following\nIN\nOUT\nHIGH_Z_DIG\nPULL_UP\nPULL_DWN\nOPEN_DRAIN_LO\nOPEN_DRAIN_HI\n_STRONG_DRIVE\nPULL_IP_DWN')
        if not self.config is None:
            dat = (self.config<<8)|(self.port<<4) | (self.pin<<1)
            RPiSoC.commChannel.sendData((self.address,cmd,dat))
            self.Read()

    def Write(self, val):
        """
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | Method            | Write                                                                                                                                                                                                         |
        +===================+===============================================================================================================================================================================================================+
        | **Description**   |* Writes a new value to the output port, without affecting the other pins attached to that port                                                                                                                |
        |                   |* Before committing the write operation, this function will confirm that the new value is different than the previous value, and only operate on the RPiSoC's I/O if it is                                     |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | **Parameters**    |* **int**     *val*                                                                                                                                                                                            |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        |  *val*            |* Value to be written to the initialized digitalPin object                                                                                                                                                     |
        |                   |* Accepts only *1* or *0* for writing the output HIGH or LOW, respectively                                                                                                                                     |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | **Returns**       |No return                                                                                                                                                                                                      |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        """

        cmd = 0x01
        dat = (self.port<<4) | (self.pin<<1) | int(val)
        if val != self.state:
            RPiSoC.commChannel.sendData((self.address,cmd,dat))
            self.state = val

    def Toggle(self):
        """
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | Method            | Toggle                                                                                                                                                                                                        |
        +===================+===============================================================================================================================================================================================================+
        | **Description**   |Toggles the state of the specified output and commits that value to the RPiSoC                                                                                                                                 |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | **Parameters**    |None                                                                                                                                                                                                           |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | **Returns**       |No return                                                                                                                                                                                                      |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+

        """
        val = int(not (self.state==1))
        self.Write(val)

    def Read(self):
        """
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | Method            | Read                                                                                                                                                                                                          |
        +===================+===============================================================================================================================================================================================================+
        | **Description**   |Determines the state of the digital pin on the RPiSoC                                                                                                                                                          |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | **Parameters**    |None                                                                                                                                                                                                           |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | **Returns**       |**bool** *state*                                                                                                                                                                                               |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | *state*           |A True or False value when the output is determined to be HIGH or LOW, respectively                                                                                                                            |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        """
        cmd = 0x00
        dat = (self.port<<4) | (self.pin<<1)
        self.state = bool(RPiSoC.commChannel.receiveData((self.address, cmd, dat)))
        return self.state

class PWM(object):
    """
    :Class:

        This class provides functionality for use of the PWM components available on the RPiSoC.

    :Example:

        Define PWM objects in the following way::

            My_PWM       = PWM(0)
            My_other_PWM = PWM(1)
            '''etc etc...    '''
            My_last_PWM  = PWM(7)

	"""
    def __init__(self, portNumber):
        """
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | Method            | __init__                                                                                                                                                                                                      |
        +===================+===============================================================================================================================================================================================================+
        | **Description**   |Constructs and initializes PWM object                                                                                                                                                                          |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | **Parameters**    |**int**   *portNumber*                                                                                                                                                                                         |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        |   *portNumber*    |Corresponds to one of the physical PWM pins on your RPiSoC                                                                                                                                                     |
        |                   |   * By default, there are 8 PWM channels available, so valid inputs are 0-7                                                                                                                                   |
        |                   |   * Valid inputs are 0-(N-1), assuming that there are N PWM channels in your RPiSoC schematic                                                                                                                 |
        |                   |   * For the default version of V1.2.4, this number corresponds to the respective pin on Port 6 for channels 1-7, and P0[2] for PWM(0)                                                                         |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | **Returns**       |No return                                                                                                                                                                                                      |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+

        """

        if portNumber not in range(RPiSoC.PWM_NUM ):
            raise ValueError('Invalid PWM Channel specified, valid entires are 0 through %d' %RPiSoC.PWM_NUM)
        for key in RPiSoC.PWM_clks:
            for i in RPiSoC.PWM_clks[key][2]:
                if int(portNumber) ==  i[0]:
                    addr_str = "RPiSoC.PWM_REGISTER"+str(portNumber)
                    setattr(self, 'address', eval(addr_str))
                    self.clk_number = int(key)
                    self.resolution_in_bits = int(i[1])

        self.max_num = pow(2,self.resolution_in_bits) - 1
        self.max_clk = RPiSoC.PWM_clks[self.clk_number][0]
        self.min_clk = int(RPiSoC.PWM_clks[self.clk_number][0]/65535) + 1
        self.Start()
        self.period = self.ReadPeriod()
        self.cmp = self.ReadCompare()
        self.Stop()
        self.__running = False
        self.__sleeping = True

        if RPiSoC.DEBUG:
            if self.address in RPiSoC.REGISTERS_IN_USE:
                print('WARNING: Attempting to initialize object at register %d which is already in use.' %self.address)
        RPiSoC.REGISTERS_IN_USE.append(self.address)

    def Start(self):
        """
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | Method            | Start                                                                                                                                                                                                         |
        +===================+===============================================================================================================================================================================================================+
        | **Description**   |Starts component operation. Sets the initVar variable, calls the PWM_Init function, and then calls the PWM_Enable function                                                                                     |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | **Parameters**    |None                                                                                                                                                                                                           |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | **Returns**       |No return                                                                                                                                                                                                      |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        """
        cmd = 0x00
        RPiSoC.commChannel.sendData((self.address, cmd))
        self.__running = True

    def isRunning(self):
        """
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | Method            | isRunning                                                                                                                                                                                                     |
        +===================+===============================================================================================================================================================================================================+
        | **Description**   |Checks to see if the PWM component is currently operational                                                                                                                                                    |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | **Parameters**    |None                                                                                                                                                                                                           |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | **Returns**       |**bool** *__running*                                                                                                                                                                                           |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | *__running*       |A private boolean variable which evaluates to *True* if the PWM component is operational, or *False* if it is not                                                                                              |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+

        """
        return self.__running

    def Stop(self):
        """
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | Method            | Stop                                                                                                                                                                                                          |
        +===================+===============================================================================================================================================================================================================+
        | **Description**   |Disables the PWM operation                                                                                                                                                                                     |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | **Parameters**    |None                                                                                                                                                                                                           |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | **Returns**       |No return                                                                                                                                                                                                      |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+

        """
        cmd = 0x01
        RPiSoC.commChannel.sendData((self.address, cmd))
        self.__running = False

    def WritePeriod(self, period, safety = False):
        """

        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | Method            | WritePeriod                                                                                                                                                                                                   |
        +===================+===============================================================================================================================================================================================================+
        | **Description**   |Writes the period value used by the PWM hardware                                                                                                                                                               |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | **Parameters**    |**int**   *period*                                                                                                                                                                                             |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        |   *period*        |The length, in *counts*, which defines how long a PWM Cycle will be                                                                                                                                            |
        |                   |   * The value must not be greater than **65535** (for 16-bit mode, which is the default)                                                                                                                      |
        |                   |   * for 8-bit mode it must not be greater than **255**                                                                                                                                                        |
        |                   |   * The period must be greater than the comparison value                                                                                                                                                      |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | **Returns**       |No return                                                                                                                                                                                                      |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+

        """
        cmd = 0x0C
        self.period = int(period)
        if self.period<0 or self.period>self.max_num:
            if not safety:
                raise ValueError('Invalid range for WritePeriod operation')
            if self.period<0:
                self.period = 0
            else:
                self.period = self.max_num

        if self.period<self.cmp:
            if RPiSoC.DEBUG:
                print('WARNING: Attempting to write a PWM period value less than its comparison value. \nDecreasing comparison value to be equal to period before continuing to prevent run time errors.')
            self.cmp = self.period
            self.WriteCompare(self.cmp)

        RPiSoC.commChannel.sendData((self.address, cmd, self.period))

    def ReadPeriod(self):
        """
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | Method            | ReadPeriod                                                                                                                                                                                                    |
        +===================+===============================================================================================================================================================================================================+
        | **Description**   |Determines the period value, in counts, currently used by the PWM component                                                                                                                                    |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | **Parameters**    |None                                                                                                                                                                                                           |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | **Returns**       |**int** *period*                                                                                                                                                                                               |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | *period*          |The period value, in counts, currently used by the PWM component                                                                                                                                               |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+

        """
        cmd = 0x0D
        self.period = RPiSoC.commChannel.receiveData((self.address, cmd))
        return self.period

    def WriteCompare(self, cmp, safety = False):
        """
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | Method            | WriteCompare                                                                                                                                                                                                  |
        +===================+===============================================================================================================================================================================================================+
        | **Description**   |Writes the period value used by the PWM hardware                                                                                                                                                               |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | **Parameters**    |**int**   *cmp*                                                                                                                                                                                                |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        |   *cmp*           |An integer value, in *counts*, which determines how long during a PWM cycle that the output is HIGH                                                                                                            |
        |                   |   * The value must not be greater than **65535** (for 16-bit mode, which is the default)                                                                                                                      |
        |                   |   * for 8-bit mode it must not be greater than **255**                                                                                                                                                        |
        |                   |   * The comparison value must also be less than or equal to the period                                                                                                                                        |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | **Returns**       |No return                                                                                                                                                                                                      |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+

        """
        cmd = 0x0E
        self.cmp = int(cmp)

        if self.cmp<0 or self.cmp>self.max_num:
            if not safety:
                raise ValueError('Invalid range for WriteCompare() method')
            if self.cmp<0:
                self.cmp = 0
            else:
                self.cmp = self.max_num

        if self.period<self.cmp:
            if RPiSoC.DEBUG:
                print('WARNING: Attempting to write comparison value larger than period. \nIncreasing period to be equal to comparison value before continuing to prevent run time errors.')
            self.period = self.cmp
            self.WritePeriod(self.period)

        RPiSoC.commChannel.sendData((self.address, cmd, self.cmp))

    def ReadCompare(self):
        """
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | Method            | ReadCompare                                                                                                                                                                                                   |
        +===================+===============================================================================================================================================================================================================+
        | **Description**   |Determines the compare value, in counts, currently used by the PWM component                                                                                                                                   |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | **Parameters**    |None                                                                                                                                                                                                           |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | **Returns**       |**int** *cmp*                                                                                                                                                                                                  |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | *cmp*             |The compare value, in counts, currently used by the PWM component                                                                                                                                              |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+

        """
        cmd = 0x0F
        self.cmp = RPiSoC.commChannel.receiveData((self.address, cmd))
        return self.cmp

    def ClearFIFO(self):
        """
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | Method            | ClearFIFO                                                                                                                                                                                                     |
        +===================+===============================================================================================================================================================================================================+
        | **Description**   |Clears the capture FIFO of any previously captured data. Here PWM_ReadCapture() is called until the FIFO is empty                                                                                              |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | **Parameters**    |None                                                                                                                                                                                                           |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | **Returns**       |No return                                                                                                                                                                                                      |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+

        """
        cmd = 0x18
        RPiSoC.commChannel.sendData((self.address, cmd))

    def Sleep(self):
        """
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | Method            | Sleep                                                                                                                                                                                                         |
        +===================+===============================================================================================================================================================================================================+
        | **Description**   |Stops the PWM operation, puts the PWM component into it's lowest power state, and saves the current configuration                                                                                              |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | **Parameters**    |None                                                                                                                                                                                                           |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | **Returns**       |No return                                                                                                                                                                                                      |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+

        """
        cmd = 0x19
        RPiSoC.commChannel.sendData((self.address, cmd))
        self.__sleeping = True

    def Wakeup(self):
        """
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | Method            | Wakeup                                                                                                                                                                                                        |
        +===================+===============================================================================================================================================================================================================+
        | **Description**   |Restores and enables the most recently saved configuration                                                                                                                                                     |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | **Parameters**    |None                                                                                                                                                                                                           |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | **Returns**       |No return                                                                                                                                                                                                      |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+

        """
        cmd = 0x1A
        RPiSoC.commChannel.sendData((self.address, cmd))
        self.__sleeping = False

    def SetClocks(self, frequency, safety = False):
        """
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | Method            | SetClocks                                                                                                                                                                                                     |
        +===================+===============================================================================================================================================================================================================+
        | **Description**   |* Attempts to set the PWM Clock rate to a desired frequency using an appropriate clock divider                                                                                                                 |
        |                   |* It will wait for a response from the RPiSoC which represents the actual clock divider which was set                                                                                                          |
        |                   |* This value is then saved in the RPiSoC class so that it can be referenced to calculate the actual clock frequency at request                                                                                 |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | **Parameters**    |**float**   *frequency*                                                                                                                                                                                        |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        |   *frequency*     | A frequency in Hz which represents the desired clock rate                                                                                                                                                     |
        |                   |   * This is NOT the frequency of the PWM, it is the frequency of the clock which drives it                                                                                                                    |
        |                   |   * Changing the clock frequency for any single PWM that shares it's clock source will affect all PWMs that share this clock                                                                                  |
        |                   |   * For the suggested clock source of 24 MHz, this value cannot be less than *367* Hz and cannot be more than 24 MHz (*24000000* Hz)                                                                          |
        |                   |   * Generally, the frequency cannot be less than your clock source divided by 65535, and cannot be greater than your clock source itself                                                                      |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | **Returns**       |No return                                                                                                                                                                                                      |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        |   **Warning**     | Because frequencies are achieved with dividers, high frequencies are difficult to accurately achieve, and they might have more error.                                                                         |
        |                   |   * For a 24 MHz source, frequencies higher than 2.526 MHz (*2526318* Hz) accuracy cannot be guaranteed to be within a tolerance of 5%                                                                        |
        |                   |   * For a 24 MHz source, frequencies higher than 5.333 MHz (*5333345* Hz) accuracy cannot be guaranteed to be within a tolerance of 10%                                                                       |
        |                   |   * The frequency might still be accurate at high frequencies; tolerances are worst case scenarios                                                                                                            |
        |                   |   * Use :meth:`~GetClocks` to get the actually achieved frequency                                                                                                                                             |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+

        """
        if frequency>self.max_clk or frequency<self.min_clk:
            if not safety:
                raise ValueError('Invalid range specified for PWM clock frequency. Must be less than %dMHz and greater than %dHz'%(int(float(self.max_clk)/1000000),self.min_clk))
            else:
                if frequency<0:
                    frequency = self.min_clk
                else:
                    frequency = self.max_clk
        if RPiSoC.DEBUG:
            if frequency>5333345: #figure this out for arbitrary clock. Only valid for 24MHz
                print("WARNING: Attempted to set PWM clock frequency greater than 5.333 MHz; this frequency cannot be gauranteed within a tolerance of 10%. Get the actual frequency with the GetClocks() method")
            elif frequency>2526318:
                print("WARNING: Attempted to set PWM clock frequency greater than 2.526 MHz; this frequency cannot be gauranteed within a tolerance of 5%. Get the actual frequency with the GetClocks() method")

        cmd = 0xFF
        attempt_divider = int((RPiSoC.PWM_clks[self.clk_number][0]/float(frequency)) + 0.5)
        RPiSoC.PWM_clks[self.clk_number][1] = (RPiSoC.commChannel.receiveData((self.address,cmd, attempt_divider))) + 1

    def GetClocks(self):
        """
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | Method            | GetClocks                                                                                                                                                                                                     |
        +===================+===============================================================================================================================================================================================================+
        | **Description**   |Calculates the actual clock rate of the PWM's based on the most recently confirmed clock divider value                                                                                                         |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | **Parameters**    |None                                                                                                                                                                                                           |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | **Returns**       |**float** *clk*                                                                                                                                                                                                |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | *clk*             |The actual frequency of the PWM clock                                                                                                                                                                          |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+

        """
        return ((RPiSoC.PWM_clks[self.clk_number][0])/(RPiSoC.PWM_clks[self.clk_number][1]))

    def GetClockDivider(self):
        """
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | Method            | GetClockDivider                                                                                                                                                                                               |
        +===================+===============================================================================================================================================================================================================+
        | **Description**   |Gets the most recently confirmed clock divider value, which is used to determine the clocking frequency of the PWM                                                                                             |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | **Parameters**    |None                                                                                                                                                                                                           |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | **Returns**       |**int** *div*                                                                                                                                                                                                  |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | *div*             |16-bit clock divider value                                                                                                                                                                                     |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+

        """

        return RPiSoC.PWM_clks[self.clk_number][1]

    def SetClockDivider(self, divider):
        """
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | Method            | SetClockDivider                                                                                                                                                                                               |
        +===================+===============================================================================================================================================================================================================+
        | **Description**   |Sets a divider to the desired PWM's clock source                                                                                                                                                               |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | **Parameters**    |**int** divider                                                                                                                                                                                                |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | *div*             |An integer between *0* and *65535* which the PWM clock will be divided by                                                                                                                                      |
        |                   |   * Changing the clock's divider, and consequently its frequency, will affect any other PWMs which share that clock                                                                                           |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | **Returns**       |No return                                                                                                                                                                                                      |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+


        """

        cmd = 0xFF
        divider = int(divider + .5)
        if divider<0 or divider>65535:
            raise ValueError('Invalide range for SetClockDivider() method')
        RPiSoC.PWM_clks[self.clk_number][1] = (RPiSoC.commChannel.receiveData((self.address,cmd, divider))) + 1

    #eventually move this algorithm to psoc side for greater portability...
    def SetFrequency(self,freq, MAX_ERROR = 5, MIN_PERIOD = 10, safety = False):
        """
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | Method            | SetFrequency                                                                                                                                                                                                  |
        +===================+===============================================================================================================================================================================================================+
        | **Description**   |* Attempts to set the PWM wave frequency to a desired rate by calculating an appropriate period value and/or clock rate                                                                                        |
        |                   |* It will try to maintain the clock rate, unless it is impossible without compromising the duty cycle too severely                                                                                             |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | **Parameters**    |* **float** freq                                                                                                                                                                                               |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        |   *Optional*      |* **float** MAX_ERROR                                                                                                                                                                                          |
        |   *Parameters*    |* **int** MIN_PERIOD                                                                                                                                                                                           |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        |   *freq*          |A frequency in Hz which represents the desired wave rate                                                                                                                                                       |
        |                   |   * This is NOT the frequency of the clock which drive the PWM, it is the actual output frequency                                                                                                             |
        |                   |   * If the frequency cannot be reached without changing the clock rate, any PWM's sharing that clock will be affected                                                                                         |
        |                   |   * For the suggested source clock rate of 24MHz, *freq* cannot be less than .006 Hz and cannot be more than 2.4 MHz (*2400000* Hz)                                                                           |
        |                   |   * Generally, freq cannot be less than :math:`\\frac{clock freq}{65535*(2^n - 1)}`, where n is the PWM resolution in bits, and it cannot be more than :math:`\\frac{clock freq}{MIN PERIOD}`                   |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        |   *MAX_ERROR*     |The largest percentage of error that can be tolerated between the desired frequency and the achieved frequency                                                                                                 |
        |                   |   * This defaults to 5                                                                                                                                                                                        |
        |                   |   * Verify the actual frequency with :meth:`~GetFrequency`                                                                                                                                                    |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        |   *MIN_PERIOD*    |The lowest possible period that can be tolerated when the wave parameters are being calculated                                                                                                                 |
        |                   |   * Valid between 0 and 65535 for a 16-bit PWM                                                                                                                                                                |
        |                   |   * Valid between 0 and 255 for an 8-bit PWM                                                                                                                                                                  |
        |                   |   * Default behavior is a *MIN_PERIOD* of 10 on 16-bit PWM                                                                                                                                                    |
        |                   |   * Lower values allow for easier wave parameter calculation, without having to modify the clock source                                                                                                       |
        |                   |   * Higher values allow finer Duty Cycle maintenence                                                                                                                                                          |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | **Returns**       |No return                                                                                                                                                                                                      |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+


        """
        if freq<=(float(self.min_clk)/65534):
            if RPiSoC.DEBUG:
                print('WARNING: cannot generate frequencies less than %f Hz. Frequency will be set to %f Hz'%(float(self.min_clk)/65534,float(self.min_clk)/65534))
            freq = self.min_clk/65534.0
        if freq>(self.max_clk/float(MIN_PERIOD)):
            if RPiSoC.DEBUG:
                print('WARNING: cannot generate a frequency greater than %dHz without compromosing Duty Cycle beyond specification. Frequency will be set to %dHz to prevent this. '%(self.max_clk/float(MIN_PERIOD),self.max_clk/float(MIN_PERIOD)))
            freq = self.max_clk/float(MIN_PERIOD)

        DutyCycle_cur = self.GetDutyCycle()/100.0
        period_new = int((self.GetClocks()/float(freq)) + 0.5)

        if period_new<float(MIN_PERIOD):
            period_new = MIN_PERIOD
        elif period_new>self.max_num:
            period_new = self.max_num

        compare_new = int(period_new*DutyCycle_cur + 0.5)
        try_freq = self.GetClocks()/float(period_new)
        error = abs(((100.0*(try_freq - freq))/freq))

        err_chk_strt = True
        div_cur = self.GetClockDivider()
        while abs(error)>MAX_ERROR:
            if err_chk_strt:
                if RPiSoC.DEBUG:
                    print('WARNING: Could not acheive desired frequency within 5% tolerance without editing the clock rate. This change will affect any PWM channels sharing this clock.')
                clock_rate = int(freq*self.max_num + (self.min_clk - freq*self.max_num)*(freq*self.max_num<self.min_clk) - (freq*self.max_num>self.max_clk)*(freq*self.max_num - self.max_clk))
                self.SetClocks(clock_rate)
                clk_new = self.GetClocks()
                div_cur = self.GetClockDivider()
                err_chk_strt = False
            else:
                div_cur = div_cur + shift
                clk_new = RPiSoC.PWM_clks[self.clk_number][0]/div_cur
                if div_cur<1:
                    div_cur = 1
                    print('Could not achieve desired frequency within tolerance')
                    break
                elif div_cur>self.max_num:
                    div_cur = self.max_num
                    print('Could not achieve desired frequency within tolerance')
                    break

            period_new = int((clk_new/float(freq)) + 0.5)

            if period_new<float(MIN_PERIOD):
                period_new = MIN_PERIOD
            elif period_new>self.max_num:
                period_new = self.max_num

            compare_new = int(period_new*DutyCycle_cur + 0.5)

            try_freq = clk_new/float(period_new)
            error = ((100.0*(try_freq - freq))/freq)
            shift = int(error>0) - 2*(error<0)
            #print(div_cur, period_new, error, shift)

        else:
            if not err_chk_strt:
                self.SetClockDivider(div_cur)
            if period_new != self.period:
                if self.cmp>period_new:
                    self.WriteCompare(compare_new)
                    self.WritePeriod(period_new)
                else:
                    self.WritePeriod(period_new)
                    self.WriteCompare(compare_new)


    def SetMIDI(self, MIDI, MAX_ERROR = 5, MIN_PERIOD = 10):
        """
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | Method            | SetMIDI                                                                                                                                                                                                       |
        +===================+===============================================================================================================================================================================================================+
        | **Description**   |Generates a PWM wave frequency the corresponds to the specified MIDI note                                                                                                                                      |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | **Parameters**    |* **int** MIDI                                                                                                                                                                                                 |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        |   *Optional*      |* **float** MAX_ERROR                                                                                                                                                                                          |
        |   *Parameters*    |* **int** MIN_PERIOD                                                                                                                                                                                           |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        |   *MIDI*          |A MIDI note, as defined by the MIDI standard to correspond with a specific musical note                                                                                                                        |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        |   *MAX_ERROR*     |The largest percentage of error that can be tolerated between the desired frequency and the achieved frequency                                                                                                 |
        |                   |   * This defaults to 5                                                                                                                                                                                        |
        |                   |   * This error rate is in reference to the wave frquency, in Hz, not the MIDI note number which is scaled differently than the frequency                                                                      |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        |   *MIN_PERIOD*    |The lowest possible period that can be tolerated when the wave parameters are being calculated                                                                                                                 |
        |                   |   * Valid between 0 and 65535 for a 16-bit PWM                                                                                                                                                                |
        |                   |   * Valid between 0 and 255 for an 8-bit PWM                                                                                                                                                                  |
        |                   |   * Default behavior is a *MIN_PERIOD* of 10 on 16-bit PWM                                                                                                                                                    |
        |                   |   * Lower values allow for easier wave parameter calculation, without having to modify the clock source                                                                                                       |
        |                   |   * Higher values allow finer Duty Cycle maintenence                                                                                                                                                          |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | **Returns**       |No return                                                                                                                                                                                                      |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+

        """

        self.SetFrequency(pow(2, (MIDI-69)/12.0)*440, MAX_ERROR, MIN_PERIOD)

    def GetMIDI(self):
        """
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | Method            | GetMIDI                                                                                                                                                                                                       |
        +===================+===============================================================================================================================================================================================================+
        | **Description**   |Calculates what MIDI note corresponds to the current PWM frequency                                                                                                                                             |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | **Parameters**    |None                                                                                                                                                                                                           |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | **Returns**       |**int** *MIDI*                                                                                                                                                                                                 |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | *MIDI*            |The MIDI note, as defined by the MIDI standard to correspond with a specific musical note, which most closely resembles the PWM frequency                                                                      |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+

        """
        return int(69.5 + 12.0*(log((self.GetFrequency()/440.0), 2)))

    def GetDutyCycle(self, PRECISION = 2):
        """
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | Method            | GetDutyCycle                                                                                                                                                                                                  |
        +===================+===============================================================================================================================================================================================================+
        | **Description**   |Calculates the current duty cycle based on the current period and comparison values                                                                                                                            |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        |   *Optional*      |**int** PRECISION                                                                                                                                                                                              |
        |   *Parameters*    |                                                                                                                                                                                                               |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        |   *PRECISION*     |The number of decimal places which the result should be calculated to                                                                                                                                          |
        |                   |   * Defaults to 2                                                                                                                                                                                             |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | **Returns**       |**float** *duty_cycle*                                                                                                                                                                                         |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | *duty_cycle*      |A value which is representative of the percentage of time, between 0 and 100%, that the PWM signal is on, relative to the total length of its period                                                           |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+

        """
        return round((100.0*(float(self.cmp)/float(self.period))),PRECISION)

    def GetFrequency(self, PRECISION = 2):
        """
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | Method            | GetFrequency                                                                                                                                                                                                  |
        +===================+===============================================================================================================================================================================================================+
        | **Description**   |Calculates the current PWM wave frequency based on the current clock rate and period value                                                                                                                     |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        |   *Optional*      |**int** PRECISION                                                                                                                                                                                              |
        |   *Parameters*    |                                                                                                                                                                                                               |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        |   *PRECISION*     |The number of decimal places which the result should be calculated to                                                                                                                                          |
        |                   |   * Defaults to 2                                                                                                                                                                                             |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | **Returns**       |**float** *freq*                                                                                                                                                                                               |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | *freq*            |A frequency in Hz, which is representative of the current PWM signal frequency                                                                                                                                 |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+

        """
        return round((self.GetClocks()/float(self.period)),PRECISION)

    def SetDutyCycle(self,duty_cycle, safety = False):
        """
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | Method            | SetDutyCycle                                                                                                                                                                                                  |
        +===================+===============================================================================================================================================================================================================+
        | **Description**   |Simplifies the process of setting a meaningful comparison value by calculating the correct cmp value based on a desired duty cycle                                                                             |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | **Parameters**    |**float** *duty_cycle*                                                                                                                                                                                         |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        |   *duty_cycle*    | A a number between 0 and 100 which indicates the percentage of time that the PWM should be HIGH during one period                                                                                             |
        |                   |   * A *duty_cycle* of 100 indicates that the PWM is always on                                                                                                                                                 |
        |                   |   * A *duty_cycle* of 50 indicates that the PWM is on half of the time                                                                                                                                        |
        |                   |   * A *duty_cycle* of 0 indicates that the PWM is always off                                                                                                                                                  |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | **Returns**       |No return                                                                                                                                                                                                      |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        """
        if duty_cycle<0 or duty_cycle>100:
            if not safety:
                raise ValueError('Invalid range for SetDutyCycle: Valid range is 0 to 100')
            if duty_cycle<0:
                duty_cycle = 0.0
            else:
                duty_cycle = 100.0
        self.cmp = int(self.period * ((duty_cycle)/100.0) + 0.5)
        self.WriteCompare(self.cmp)

class Servo:
    """
        :Class:
            This class provides allows for easy manipulation of standard Servo Motors using PWM.

        :Example:

            Define Servo objects in any the following ways::

                My_simple_servo       = Servo(0)
                My_calibrated_servo   = Servo(1, 1.0, 2.5)
                My_descriptive_servo  = Servo(7, 1.0, 2.5, 0, 180)
    """
    def __init__(self, servo_id, min_pulse = 1.0, max_pulse = 2.0, min_angle = 0, max_angle = 180):
        """
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | Method            | __init__                                                                                                                                                                                                      |
        +===================+===============================================================================================================================================================================================================+
        | **Description**   |Creates a servo object with the given parameter set                                                                                                                                                            |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | **Parameters**    |* **int** servo_id                                                                                                                                                                                             |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        |   *Optional*      |* **float** min_pulse                                                                                                                                                                                          |
        |   *Parameters*    |* **float** max_pulse                                                                                                                                                                                          |
        |                   |* **float** min_angle                                                                                                                                                                                          |
        |                   |* **float** max_angle                                                                                                                                                                                          |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        |   *servo_id*      |Corresponds to the pin which the signal line of the servo will be connected to                                                                                                                                 |
        |                   |   * These pins will be defined exactly the same way that PWM pins are defined                                                                                                                                 |
        |                   |   * By default, servo 0 is on P0[2], and servos 1-7 are on Port 6 Pins 1-7                                                                                                                                    |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        |   *min_pulse*     |The pulse width necessary to obtain angular position min_angle.                                                                                                                                                |
        |                   |   * Values must be greater than zero                                                                                                                                                                          |
        |                   |   * Normal values are between 0.8 and 1.2                                                                                                                                                                     |
        |                   |   * Find an appropriate value through calibration; it will default to 1.0 if no parameter is given                                                                                                            |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        |   *max_pulse*     |The pulse width necessary to obtain angular position max_angle.                                                                                                                                                |
        |                   |   * Values must be greater than zero                                                                                                                                                                          |
        |                   |   * Normal values are between 1.8 and 2.3                                                                                                                                                                     |
        |                   |   * Find an appropriate value through calibration; it will default to 2.0 if no parameter is given                                                                                                            |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        |   *min_angle*     |the angle which the servo will return to if applied with a pulse width of *min_pulse*                                                                                                                          |
        |                   |   * This defaults to 0                                                                                                                                                                                        |
        |                   |   * Negative angular positions are valid                                                                                                                                                                      |
        |                   |   * Angles can be any angular unit: degrees, radians, or other arbitrary (linear) scale                                                                                                                       |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        |   *max_angle*     |the angle which the servo will return to if applied with a pulse width of *max_pulse*                                                                                                                          |
        |                   |   * This defaults to 180                                                                                                                                                                                      |
        |                   |   * Negative angular positions are valid                                                                                                                                                                      |
        |                   |   * Angles can be any angular unit: degrees, radians, or other arbitrary (linear) scale                                                                                                                       |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | **Returns**       |No return                                                                                                                                                                                                      |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        |   **Note**        |A servo with servo_id, n, will make PWM(n) unnavailable, since the servo controller is implemented using that PWM object. For fine control over                                                                |
        |                   |that PWM, you can expose the object using My_servo.servo_PWM, and then you can use any of the PWM methods using My_servo.servo_PWM.method(). This is                                                           |
        |                   |advised against though, because servos are very particular about the construction of their data signals. If you change the wrong parameter of the PWM signal, you                                              |
        |                   |might damage the servo                                                                                                                                                                                         |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+

		"""

        self.min_pulse = float(min_pulse)
        self.max_pulse = float(max_pulse)
        self.min_angle = float(min_angle)
        self.max_angle = float(max_angle)
        self.servo_id = servo_id

        self.pulse_range = float(max_pulse-min_pulse)
        self.angle_range = float(max_angle-min_angle)


        self.servo_PWM = PWM(self.servo_id)
        if abs(self.servo_PWM.GetFrequency() - 50) > 1:
            self.servo_PWM.SetFrequency(50, MAX_ERROR = 1)

    def changeAngles(self, MIN, MAX):
        """
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | Method            | changeAngles                                                                                                                                                                                                  |
        +===================+===============================================================================================================================================================================================================+
        | **Description**   |Changes the angle range that defines the minimum and maximum positions of the motor                                                                                                                            |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | **Parameters**    |* **float** *MIN*                                                                                                                                                                                              |
        |                   |* **float** *MAX*                                                                                                                                                                                              |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        |   *MIN*           |the new angle which the servo will return to if applied with the minimum pulse width                                                                                                                           |
        |                   |   * Negative angular positions are valid                                                                                                                                                                      |
        |                   |   * Angles can be any angular unit: degrees, radians, or other arbitrary (linear) scale                                                                                                                       |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        |   *MAX*           |the new angle which the servo will return to if applied with the maximum pulse width                                                                                                                           |
        |                   |   * Negative angular positions are valid                                                                                                                                                                      |
        |                   |   * Angles can be any angular unit: degrees, radians, or other arbitrary (linear) scale                                                                                                                       |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | **Returns**       |No return                                                                                                                                                                                                      |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        """

        self.min_angle = float(MIN)
        self.max_angle = float(MAX)
        self.angle_range = float(MAX-MIN)

    def SetPulse(self, pulse_ms):
        """
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | Method            | SetPulse                                                                                                                                                                                                      |
        +===================+===============================================================================================================================================================================================================+
        |   **Description** |Sets a servo position based on a pulse width in ms                                                                                                                                                             |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        |   **Parameters**  |**float** *pulse_ms*                                                                                                                                                                                           |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        |   *pulse_ms*      |A pulse width in milliseconds, which will be applied to the signal wire of the Servo at 50 Hz                                                                                                                  |
        |                   |   * Normal values are between 0.8 and 2.3, but they must be between the defined *min_pulse* and *max_pulse* values                                                                                            |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        |   **Returns**     |No return                                                                                                                                                                                                      |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+


		"""
        if pulse_ms   < self.min_pulse:
            pulse_ms  = self.min_pulse
            if RPiSoC.DEBUG:
                print('Tried to set a pulse less than defined range, setting pulse to minimum accepted value')
        elif pulse_ms > self.max_pulse:
            pulse_ms  = self.max_pulse
            if RPiSoC.DEBUG:
                print('Tried to set a pulse greater than defined range, setting pulse to maximum accepted value')
        #cmp = int((self.servo_PWM.GetClocks()*((pulse_ms)/1000.0)) + 0.5)
        #cmp = int(  (float(pulse_ms)/(1.0/(self.servo_PWM.GetFrequency()))) * self.servo_PWM.ReadPeriod() )
        self.servo_PWM.SetDutyCycle((float(pulse_ms)/(1000.0/(self.servo_PWM.GetFrequency())))*100)
        #self.servo_PWM.WriteCompare(cmp)
    def ReadPulse(self):
        """
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | Method            | ReadPulse                                                                                                                                                                                                     |
        +===================+===============================================================================================================================================================================================================+
        | **Description**   |Calculates the current pulse width of the PWM which is driving the servomotor                                                                                                                                  |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | **Parameters**    |None                                                                                                                                                                                                           |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | **Returns**       |**float** *pulse*                                                                                                                                                                                              |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | *pulse*           |A pulse duration in milliseconds, representative of the PWM signal pulse width being applied ot the servomotor                                                                                                 |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+

        """
        return  float(self.servo_PWM.GetDutyCycle()*(10.0/self.servo_PWM.GetFrequency()))

    def ReadAngle(self):
        """
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | Method            | ReadAngle                                                                                                                                                                                                     |
        +===================+===============================================================================================================================================================================================================+
        | **Description**   |Calculates the current angle of the servomotor, linearized relative to the provided maximum and minimum angles                                                                                                 |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | **Parameters**    |None                                                                                                                                                                                                           |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | **Returns**       |**float** *angle_cur*                                                                                                                                                                                          |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | *angle_cur*       |A value representative of the angle that the servo motor is held, according to the provided scale                                                                                                              |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+

        """

        pulse_cur = self.servo_PWM.GetDutyCycle()*(10.0/self.servo_PWM.GetFrequency())
        angle_perc = (pulse_cur - self.min_pulse)/self.pulse_range
        angle_cur = angle_perc*self.angle_range + self.min_angle

        return angle_cur

    def SetAngle(self, angle):
        """
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | Method            | SetAngle                                                                                                                                                                                                      |
        +===================+===============================================================================================================================================================================================================+
        | **Description**   |Calculates a pulse width based on the given angular position and the current min/max configuration parameters, then calls :meth:`~SetPulse` to set the position of the servo                                   |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | **Parameters**    |**float** *angle*                                                                                                                                                                                              |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        |   *angle*         |The angle to which the servo should be set, linearized relative to the defined minimum and maximum angles                                                                                                      |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | **Returns**       |No return                                                                                                                                                                                                      |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        """
        angle_perc = float(angle-self.min_angle)/self.angle_range
        pulse = self.min_pulse + angle_perc*self.pulse_range

        self.SetPulse(pulse)

    def Stop(self):
        """
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | Method            | Stop                                                                                                                                                                                                          |
        +===================+===============================================================================================================================================================================================================+
        |   **Description** |Stops the servo object by terminating the PWM channel that drives it                                                                                                                                           |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        |   **Parameters**  |None                                                                                                                                                                                                           |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        |   **Returns**     |No return                                                                                                                                                                                                      |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        |   **Side Effects**|This may cause the servo to move slightly out of position                                                                                                                                                      |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        """
        self.servo_PWM.Stop()

    def isRunning(self):
        """
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | Method            | isRunning                                                                                                                                                                                                     |
        +===================+===============================================================================================================================================================================================================+
        | **Description**   |Checks to see if the Servo component is currently operational                                                                                                                                                  |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | **Parameters**    |None                                                                                                                                                                                                           |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | **Returns**       |**bool** *__running*                                                                                                                                                                                           |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | *__running*       |A private boolean variable which evaluates to *True* if the PWM component is operational, or *False* if it is not                                                                                              |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+

        """
        return self.servo_PWM._PWM__running

    def Start(self):
        """
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | Method            | Start                                                                                                                                                                                                         |
        +===================+===============================================================================================================================================================================================================+
        | **Description**   |Starts component operation.                                                                                                                                                                                    |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | **Parameters**    |None                                                                                                                                                                                                           |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | **Returns**       |No return                                                                                                                                                                                                      |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+

        """
        self.servo_PWM.Start()

class rangeFinder(object):
    """
    :Class:

        This class provides functionality for use of ultrasonic range finder devices that use standard GPIO pulse width measurement protocols

    :Example:

        Define a rangeFinder object in the following way::

            my_ranger = rangerFinder([SIG_PORT,SIG_PIN], [TRIG_PORT, TRIG_PIN])

    """

    def __init__(self, SIG, TRIGGER = None, DELAYus = 10, TIMEOUTus = 30000):
        """
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | Method            | __init__                                                                                                                                                                                                      |
        +===================+===============================================================================================================================================================================================================+
        | **Description**   |Initializes an object by describing which pin the echo will be measured on. Optionally, provide a trigger pin if the ranger is a 4 or 5 pin form factor,                                                       |
        |                   |also optionally provide a trigger pulse width and timeout value in milliseconds                                                                                                                                |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | **Parameters**    |* **list**   *SIG*                                                                                                                                                                                             |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | *Optional*        |* **list**   *TRIGGER*                                                                                                                                                                                         |
        | *Parameters*      |* **int**    *DELAYus*                                                                                                                                                                                         |
        |                   |* **int**    *TIMEOUTus*                                                                                                                                                                                       |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        |   *SIG*           |A list constructed as *[PORT, PIN]* which defines which exact pin will be used to communicate with the rangers signal/echo pin                                                                                 |
        |                   |   - PORT is the port on the RPiSoC connected to the rangers signal pin                                                                                                                                        |
        |                   |   - PIN is the pin relative to that port                                                                                                                                                                      |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        |   *TRIGGER*       |A list constructed as *[PORT, PIN]* which defines which exact pin will be used to communicate with the rangers trigger pin                                                                                     |
        |                   |   - By default, if no argument is provided, it will assume the trigger pin is the same as the echo pin; this is true for 3-pin devices and so no argument is needed                                           |
        |                   |   - PORT is the port on the RPiSoC connected to the rangers trigger pin                                                                                                                                       |
        |                   |   - PIN is the pin relative to that port                                                                                                                                                                      |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        |   *DELAYus*       |The width of the trigger pulse, in microseconds, that will be sent from the TRIGGER pin, to signal the ranger to send a ping                                                                                   |
        |                   |   - This defaults as 10 microseconds, but most rangers are okay with much less                                                                                                                                |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        |   *TIMEOUTus*     |The maximum length of time, in microseconds, that the RPiSoC will wait for a confirmed echo                                                                                                                    |
        |                   |   - If this time is exceeded, the ranger will immediately terminate its timing process and return the timeout as a response                                                                                   |
        |                   |   - This defaults as 30000 microseconds, which is equivalent to 30 ms, and is much longer than generally needed and so won't usually need to be altered                                                       |
        |                   |   - Refer to your devices documentation for a more specific timeout choice                                                                                                                                    |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | **Returns**       |No return                                                                                                                                                                                                      |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+

        """

        if TRIGGER == None:
            TRIGGER = SIG[:]

        if int(SIG[0]) not in RPiSoC.GPIO.keys():
            raise ValueError('Invalid PORT for signal: Port numbers found on RPiSoC are ', RPiSoC.GPIO.keys())
        else:
             self.sigport = SIG[0]

        if int(SIG[1]) not in RPiSoC.GPIO[self.sigport]:
            raise ValueError('Invalid PIN for signal: the second argument should be the pin number relative to the desired port. Valid entries on this port are ', RPiSoC.GPIO[self.sigport])
        else:
            self.sigpin = SIG[1]

        if int(TRIGGER[0]) not in RPiSoC.GPIO.keys():
            raise ValueError('Invalid PORT for trigger: Port numbers found on RPiSoC are ', RPiSoC.GPIO.keys())
        else:
             self.trigport = TRIGGER[0]

        if int(TRIGGER[1]) not in RPiSoC.GPIO[self.trigport]:
            raise ValueError('Invalid PIN for trigger: the second argument should be the pin number relative to the desired port. Valid entries on this port are ', RPiSoC.GPIO[self.trigport])
        else:
            self.trigpin = TRIGGER[1]

        self.address = RPiSoC.RANGE_FINDER
        self.delayus = DELAYus
        self.packed_dat = (self.trigport<<10)|(self.trigpin<<7)|(self.sigport<<3)|(self.sigpin)
        self.raw = 0
        self.meters = 0
        self.inches = 0
        self.centimeters = 0
        self.reading = False
        self.setDelay(DELAYus)
        self.setTimeout(TIMEOUTus)

    def setTimeout(self, timeout_us):
        """
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | Method            | setTimeout                                                                                                                                                                                                    |
        +===================+===============================================================================================================================================================================================================+
        | **Description**   |Sets the timeout length in microseconds. If the RPiSoC is still waiting for a completed response after this amount of time, it                                                                                 |
        |                   |will stop counting and immediately return the result                                                                                                                                                           |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | **Parameters**    |* **int**   *timeout_us*                                                                                                                                                                                       |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        |   *timeout_us*    |Amount of time, in microseconds, that the RPiSoC will wait for a completed response                                                                                                                            |
        |                   | * This is a 16 bit value, and so it must be between *0* and *65535*                                                                                                                                           |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | **Returns**       |No return                                                                                                                                                                                                      |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        |   **Notes**       |This is handled in :meth:`~__init__`, so it should only be called under unique circumstances                                                                                                                   |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+

        """

        cmd = 0x01
        if ((timeout_us>65535) or (timeout_us<=0)):
            raise ValueError('Timeout must be between 1 and 65535 microseconds: provided %d' %timeout_us)
        RPiSoC.commChannel.sendData((self.address, cmd, timeout_us))
        self.timeout = timeout_us

    def setDelay(self, delay_us):
        """
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | Method            | setDelay                                                                                                                                                                                                      |
        +===================+===============================================================================================================================================================================================================+
        | **Description**   |Sets the length of the trigger pulse, in microseconds, which will be used to tell the device to send out a ping                                                                                                |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | **Parameters**    |* **int**   *delay_us*                                                                                                                                                                                         |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        |   *delay_us*      |Amount of time, in microseconds, that the RPiSoC will hold its trigger pulse high                                                                                                                              |
        |                   |   * This is a 6 bit value, and so it must be between *0* and *63*                                                                                                                                             |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | **Returns**       |No return                                                                                                                                                                                                      |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        |   **Notes**       |This is handled in :meth:`~__init__`, so it should only be called under unique circumstances                                                                                                                   |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+

        """
        cmd = 0x02
        if (delay_us>63 or delay_us<=0):
            raise ValueError('Delay must be between 1 and 63 microseconds: provided %d' %delay_us)
        RPiSoC.commChannel.sendData((self.address, cmd, delay_us))

    def readRaw(self):
        """
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | Method            | readRaw                                                                                                                                                                                                       |
        +===================+===============================================================================================================================================================================================================+
        | **Description**   |* Gets a raw value from the RPiSoC, which is representative of how many microseconds the rangers echo pin was held high                                                                                        |
        |                   |* It will trigger the ranger by sending a short pulse, the length of which was defined upon construction of the object                                                                                         |
        |                   |* The result will be measured on the desired signal pin, and stop measuring when a timeout is reached                                                                                                          |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | **Parameters**    |None                                                                                                                                                                                                           |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | **Returns**       |**int** *reading*                                                                                                                                                                                              |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | *reading*         |* The length of time, in microseconds, that it took for the ping to echo back to the ultrasonic ranger                                                                                                         |
        |                   |* This value can be used to calculate distance to an object using the speed of sound                                                                                                                           |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+

        """
        cmd = 0x00
        reading = 0
        self.reading = True
        while reading <=0 or reading>=self.timeout:
            reading = RPiSoC.commChannel.receiveData((self.address, cmd, self.packed_dat), delay = 0.04)
            if RPiSoC.DEBUG:
                print (reading)
                if reading <= 0:
                    print ('Timeout occured waiting for signal pin to be asserted; verify connection')
                elif reading>=self.timeout:
                    print('Timeout occured while reading signal pin; verify hardware is functional')
        self.reading = False
        self.raw = reading
        return reading

    def readMeters(self, sound = 343.0, PRECISION = 2):
        """
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | Method            | readMeters                                                                                                                                                                                                    |
        +===================+===============================================================================================================================================================================================================+
        | **Description**   | Uses readRaw to get a raw time value in microseconds, and then calculates the distance between the ranger and the pinged object in meters                                                                     |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | *Optional*        |* **float**   *sound*                                                                                                                                                                                          |
        | *Parameters*      |* **int**    *PRECISION*                                                                                                                                                                                       |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        |   *sound*         |The speed of sound which is used to calcuate the distance of the object detected by the ranger                                                                                                                 |
        |                   |   - Defaults to 343 m/s (approximate value based on room temperature of air)                                                                                                                                  |
        |                   |   - Must be m/s                                                                                                                                                                                               |
        |                   |   - Modify this according to your environmental needs (different temperature/medium)                                                                                                                          |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        |   *PRECISION*     |The number of decimal points to be included in the returned result                                                                                                                                             |
        |                   |   - Defaults to 2                                                                                                                                                                                             |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | **Returns**       |**float** meters                                                                                                                                                                                               |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | *meters*          |The distance, in meters, between the ultrasonic ranger and the pinged object                                                                                                                                   |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+


        """
        period_counts = 1850
        period_time = .00185

        #time_high = (float(self.readRaw())/period_counts)*period_time
        time_high = float(self.readRaw())/1000000.0;
        # x = vt; speed of sound = 340.29, x is distance from ranger, to object, back to ranger. So twice the desired distance.
        self.meters = round((sound*time_high)/2.0, PRECISION)
        return self.meters

    def readCentimeters(self, sound = 343.0, PRECISION = 2):
        """
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | Method            | readCentimeters                                                                                                                                                                                               |
        +===================+===============================================================================================================================================================================================================+
        | **Description**   | Uses readRaw to get a raw time value in microseconds, and then calculates the distance between the ranger and the pinged object in centimeters                                                                |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | *Optional*        |* **float**   *sound*                                                                                                                                                                                          |
        | *Parameters*      |* **int**    *PRECISION*                                                                                                                                                                                       |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        |   *sound*         |The speed of sound which is used to calcuate the distance of the object detected by the ranger                                                                                                                 |
        |                   |   - Defaults to 343 m/s (approximate value based on room temperature of air)                                                                                                                                  |
        |                   |   - Must be m/s                                                                                                                                                                                               |
        |                   |   - Modify this according to your environmental needs (different temperature/medium)                                                                                                                          |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        |   *PRECISION*     |The number of decimal points to be included in the returned result                                                                                                                                             |
        |                   |   - Defaults to 2                                                                                                                                                                                             |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | **Returns**       |**float** meters                                                                                                                                                                                               |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | *meters*          |The distance, in meters, between the ultrasonic ranger and the pinged object                                                                                                                                   |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+

        """
        self.centimeters = round((self.readMeters(sound, PRECISION = 9))*100.0, PRECISION)
        return self.centimeters

    def readInches(self, sound = 343.0, PRECISION = 2):
        """
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | Method            | readInches                                                                                                                                                                                                    |
        +===================+===============================================================================================================================================================================================================+
        | **Description**   | Uses readRaw to get a raw time value in microseconds, and then calculates the distance between the ranger and the pinged object in inches                                                                     |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | *Optional*        |* **float**   *sound*                                                                                                                                                                                          |
        | *Parameters*      |* **int**    *PRECISION*                                                                                                                                                                                       |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        |   *sound*         |The speed of sound which is used to calcuate the distance of the object detected by the ranger                                                                                                                 |
        |                   |   - Defaults to 343 m/s (approximate value based on room temperature of air)                                                                                                                                  |
        |                   |   - Must be m/s                                                                                                                                                                                               |
        |                   |   - Modify this according to your environmental needs (different temperature/medium)                                                                                                                          |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        |   *PRECISION*     |The number of decimal points to be included in the returned result                                                                                                                                             |
        |                   |   - Defaults to 2                                                                                                                                                                                             |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | **Returns**       |**float** meters                                                                                                                                                                                               |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | *meters*          |The distance, in meters, between the ultrasonic ranger and the pinged object                                                                                                                                   |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+

        """
        self.inches = round(self.readCentimeters(sound, PRECISION = 9)/2.54, PRECISION)
        return self.inches
class NeoPixelShield(object):
    """
    :Class:

        This class provides functionality for use of an Arduino NeoPixels shield on an RPiSoC through Python.

    :Example:

        Create a NeoPixelShield object in the following way::

            shield = NeoPixelShield()

	"""
    def __init__(self):
        """
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | Method            | __init__                                                                                                                                                                                                      |
        +===================+===============================================================================================================================================================================================================+
        | **Description**   |Defines the register address for the striplight controller used by the NeoPixels, and it defines 140 colors as class attributes, named according to their standardized HTML and CSS names                      |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | **Parameters**    |None                                                                                                                                                                                                           |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | **Returns**       |No return                                                                                                                                                                                                      |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+

        """
        self.address = RPiSoC.STRIPLIGHT_REGISTER
        self.AliceBlue = 0xfff0f8
        self.AntiqueWhite = 0xd7faeb
        self.Aqua = 0xff00ff
        self.Aquamarine = 0xd47fff
        self.Azure = 0xfff0ff
        self.Beige = 0xdcf5f5
        self.Bisque = 0xc4ffe4
        self.Black = 0x0
        self.BlanchedAlmond = 0xcdffeb
        self.Blue = 0xff0000
        self.BlueViolet = 0xe28a2b
        self.Brown = 0x2aa52a
        self.BurlyWood = 0x87deb8
        self.CadetBlue = 0xa05f9e
        self.Chartreuse = 0x7fff
        self.Chocolate = 0x1ed269
        self.Coral = 0x50ff7f
        self.CornflowerBlue = 0xed6495
        self.Cornsilk = 0xdcfff8
        self.Crimson = 0x3cdc14
        self.Cyan = 0xff00ff
        self.DarkBlue = 0x8b0000
        self.DarkCyan = 0x8b008b
        self.DarkGoldenRod = 0xbb886
        self.DarkGray = 0xa9a9a9
        self.DarkGreen = 0x64
        self.DarkKhaki = 0x6bbdb7
        self.DarkMagenta = 0x8b8b00
        self.DarkOliveGreen = 0x2f556b
        self.DarkOrange = 0xff8c
        self.DarkOrchid = 0xcc9932
        self.DarkRed = 0x8b00
        self.DarkSalmon = 0x7ae996
        self.DarkSeaGreen = 0x8f8fbc
        self.DarkSlateBlue = 0x8b483d
        self.DarkSlateGray = 0x4f2f4f
        self.DarkTurquoise = 0xd100ce
        self.DarkViolet = 0xd39400
        self.DeepPink = 0x93ff14
        self.DeepSkyBlue = 0xff00bf
        self.DimGray = 0x696969
        self.DodgerBlue = 0xff1e90
        self.FireBrick = 0x22b222
        self.FloralWhite = 0xf0fffa
        self.ForestGreen = 0x22228b
        self.Fuchsia = 0xffff00
        self.Gainsboro = 0xdcdcdc
        self.GhostWhite = 0xfff8f8
        self.Gold = 0xffd7
        self.GoldenRod = 0x20daa5
        self.Gray = 0x808080
        self.Green = 0x80
        self.GreenYellow = 0x2fadff
        self.HoneyDew = 0xf0f0ff
        self.HotPink = 0xb4ff69
        self.IndianRed = 0x5ccd5c
        self.Indigo = 0x824b00
        self.Ivory = 0xf0ffff
        self.Khaki = 0x8cf0e6
        self.Lavender = 0xfae6e6
        self.LavenderBlush = 0xf5fff0
        self.LawnGreen = 0x7cfc
        self.LemonChiffon = 0xcdfffa
        self.LightBlue = 0xe6add8
        self.LightCoral = 0x80f080
        self.LightCyan = 0xffe0ff
        self.LightGoldenRodYellow = 0xd2fafa
        self.LightGray = 0xd3d3d3
        self.LightGreen = 0x9090ee
        self.LightPink = 0xc1ffb6
        self.LightSalmon = 0x7affa0
        self.LightSeaGreen = 0xaa20b2
        self.LightSkyBlue = 0xfa87ce
        self.LightSlateGray = 0x997788
        self.LightSteelBlue = 0xdeb0c4
        self.LightYellow = 0xe0ffff
        self.Lime = 0xff
        self.LimeGreen = 0x3232cd
        self.Linen = 0xe6faf0
        self.Magenta = 0xffff00
        self.Maroon = 0x8000
        self.MediumAquaMarine = 0xaa66cd
        self.MediumBlue = 0xcd0000
        self.MediumOrchid = 0xd3ba55
        self.MediumPurple = 0xdb9370
        self.MediumSeaGreen = 0x713cb3
        self.MediumSlateBlue = 0xee7b68
        self.MediumSpringGreen = 0x9a00fa
        self.MediumTurquoise = 0xcc48d1
        self.MediumVioletRed = 0x85c715
        self.MidnightBlue = 0x701919
        self.MintCream = 0xfaf5ff
        self.MistyRose = 0xe1ffe4
        self.Moccasin = 0xb5ffe4
        self.NavajoWhite = 0xadffde
        self.Navy = 0x800000
        self.OldLace = 0xe6fdf5
        self.Olive = 0x8080
        self.OliveDrab = 0x236b8e
        self.Orange = 0xffa5
        self.OrangeRed = 0xff45
        self.Orchid = 0xd6da70
        self.PaleGoldenRod = 0xaaeee8
        self.PaleGreen = 0x9898fb
        self.PaleTurquoise = 0xeeafee
        self.PaleVioletRed = 0x93db70
        self.PapayaWhip = 0xd5ffef
        self.PeachPuff = 0xb9ffda
        self.Peru = 0x3fcd85
        self.Pink = 0xcbffc0
        self.Plum = 0xdddda0
        self.PowderBlue = 0xe6b0e0
        self.Purple = 0x808000
        self.Red = 0xff00
        self.RosyBrown = 0x8fbc8f
        self.RoyalBlue = 0xe14169
        self.SaddleBrown = 0x138b45
        self.Salmon = 0x72fa80
        self.SandyBrown = 0x60f4a4
        self.SeaGreen = 0x572e8b
        self.SeaShell = 0xeefff5
        self.Sienna = 0x2da052
        self.Silver = 0xc0c0c0
        self.SkyBlue = 0xeb87ce
        self.SlateBlue = 0xcd6a5a
        self.SlateGray = 0x907080
        self.Snow = 0xfafffa
        self.SpringGreen = 0x7f00ff
        self.SteelBlue = 0xb44682
        self.Tan = 0x8cd2b4
        self.Teal = 0x800080
        self.Thistle = 0xd8d8bf
        self.Tomato = 0x47ff63
        self.Turquoise = 0xd040e0
        self.Violet = 0xeeee82
        self.Wheat = 0xb3f5de
        self.White = 0xffffff
        self.WhiteSmoke = 0xf5f5f5
        self.Yellow = 0xffff
        self.YellowGreen = 0x329acd
        self.__running = False
        #self.__recentpixel = [0,0,self.Black]


    def Start(self):
        """
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | Method            | Start                                                                                                                                                                                                         |
        +===================+===============================================================================================================================================================================================================+
        | **Description**   |Powers up and enables the needed hardware for the NeoPixels component                                                                                                                                          |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | **Parameters**    |None                                                                                                                                                                                                           |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | **Returns**       |No return                                                                                                                                                                                                      |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        """
        cmd = 0x00
        RPiSoC.commChannel.receiveData((self.address, cmd))
        self.__clearMem()
        self.__running = True

    def __refresh(self, recentpixel):
        """
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | Method            | __refresh                                                                                                                                                                                                     |
        +===================+===============================================================================================================================================================================================================+
        | **Description**   |Private class used to redraw pixels after changing the brightness level of the screen                                                                                                                          |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | **Parameters**    |**list** or **tuple** recentpixel                                                                                                                                                                              |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | *recentpixel*     |A list or tuple with three elements, which describes the most recently drawn pixel to the display                                                                                                              |
        |                   |   * first element is the row                                                                                                                                                                                  |
        |                   |   * second element is the column                                                                                                                                                                              |
        |                   |   * third element is the color                                                                                                                                                                                |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | **Returns**       |No return                                                                                                                                                                                                      |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+

        """
        self.SetPixel(recentpixel[0], recentpixel[1], recentpixel[2])

    def isRunning(self):
        """
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | Method            | isRunning                                                                                                                                                                                                     |
        +===================+===============================================================================================================================================================================================================+
        | **Description**   |Checks to see if the NeoPixels display is currently powered up                                                                                                                                                 |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | **Parameters**    |None                                                                                                                                                                                                           |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | **Returns**       |**bool** *__running*                                                                                                                                                                                           |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | *__running*       |A private boolean variable which evaluates to *True* if the NeoPixels display is running, or *False* if it is not                                                                                              |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+

        """
        return self.__running

    def Stop(self):
        """
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | Method            | Stop                                                                                                                                                                                                          |
        +===================+===============================================================================================================================================================================================================+
        | **Description**   |Powers down and disables the needed hardware for the NeoPixels component                                                                                                                                       |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | **Parameters**    |None                                                                                                                                                                                                           |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | **Returns**       |No return                                                                                                                                                                                                      |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+

        """
        cmd = 0x01
        RPiSoC.commChannel.receiveData((self.address, cmd))
        self.__running = False

    def RGB_toHex(self, RGB):
        """
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | Method            | RGB_toHex                                                                                                                                                                                                     |
        +===================+===============================================================================================================================================================================================================+
        | **Description**   |Converts RGB content to the needed BRG Hex value taken by the NeoPixelShield device                                                                                                                            |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | **Parameters**    |**list** or **tuple** *RGB*                                                                                                                                                                                    |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        |   *RGB*           |A list or tuple containing three elements, each of which is limited to one byte (0-255)                                                                                                                        |
        |                   |   * The first element is the Red byte                                                                                                                                                                         |
        |                   |   * The second element is the Green byte                                                                                                                                                                      |
        |                   |   * The thirf element is the Blue byte                                                                                                                                                                        |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | **Returns**       |**int** *hex_val*                                                                                                                                                                                              |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | *hex_val*         |A hex value which can be given to a NeoPixelShield object as a valid color                                                                                                                                     |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        """
        RGB = list(RGB)
        for i in range(len(RGB)):
            if RGB[i] >255:
                if RPiSoC.DEBUG:
                    print('Each color is limited to one byte only (0-255): provided %d as an element. Setting to 255' %i)
                    RGB[i] = 255
            elif RGB[i] <0:
                if RPiSoC.DEBUG:
                    print('Each color is limited to one byte only (0-255): provided %d as an element. Setting to 0' %i)
                    RGB[i] = 0

        RED = RGB[0]
        GREEN = RGB[1]
        BLUE = RGB[2]

        return ((BLUE<<16)|(RED<<8)|GREEN)

    def SetPixel(self, row, column, color):
        """
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | Method            | SetPixel                                                                                                                                                                                                      |
        +===================+===============================================================================================================================================================================================================+
        | **Description**   |Sets the given pixel at location (x,y) = (row, column), to a color defined by a 24 bit Blue-Red-Green (BRG) value (8-bits each)                                                                                |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | **Parameters**    |* **int**   *row*                                                                                                                                                                                              |
        |                   |* **int**   *column*                                                                                                                                                                                           |
        |                   |* **int**   *color*                                                                                                                                                                                            |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        |   *row*           |Row which contains the desired pixel to be set                                                                                                                                                                 |
        |                   |   * valid between 0 and 4                                                                                                                                                                                     |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        |   *column*        |Column which contains the desired pixel to be set                                                                                                                                                              |
        |                   |   * valid between 0 and 7                                                                                                                                                                                     |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        |   *color*         |A 24-bit number representative of a BRG value                                                                                                                                                                  |
        |                   |   * BRG components are given equal weight, so 8-bits each                                                                                                                                                     |
        |                   |   * There are predefined colors inside of :meth:`~__init__`, which can be called as shield.[*color name*]; example: :code:`shield.Blue`                                                                       |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | **Returns**       |No return                                                                                                                                                                                                      |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+

        """
        cmd = 0x02
        if color>0xFFFFFF:
            raise ValueError('Color value too large. Color is 24 bit only.')
        if row not in range(5):
            raise ValueError('NeoPixel shield only has 5 rows available, choose 0-4')
        if column not in range(8):
            raise ValueError('NeoPixel shield only has 8 rows available, choose 0-7')

        #first send high 16 bits bits of color
        RPiSoC.commChannel.sendData((self.address, cmd, color>>16))
        #now send the remaining 16 bits in the data bytes, set the row as the addr, and set the column as the cmd
        RPiSoC.commChannel.receiveData((row, column, color&0xFFFF))
        #self.__recentpixel = [row, column, color]

    def Stripe(self, pixelnum, color, safety = False):
        """
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | Method            | Stripe                                                                                                                                                                                                        |
        +===================+===============================================================================================================================================================================================================+
        | **Description**   |Draws a line of length *pixelnum* of the stated *color*, starting from the first pixel, and extending as far as the 40th (last) pixel. It will wrap around rows                                                |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | **Parameters**    |* **int**   *pixelnum*                                                                                                                                                                                         |
        |                   |* **int**   *color*                                                                                                                                                                                            |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        |   *pixelnum*      |Indicates how many pixels, starting with pixel (0,0), will be filled                                                                                                                                           |
        |                   |   * valid between 1 and 40                                                                                                                                                                                    |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        |   *color*         |A 24-bit number representative of a BRG value                                                                                                                                                                  |
        |                   |   * BRG components are given equal weight, so 8-bits each                                                                                                                                                     |
        |                   |   * There are predefined colors inside of :meth:`~__init__`, which can be called as shield.[*color name*]; example: :code:`shield.Blue`                                                                       |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | **Returns**       |No return                                                                                                                                                                                                      |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+

        """
        if pixelnum not in range(1, 41):
            if not safety:
                raise ValueError('Valid stripe length is between 1 and 40: stripe length of %d was requested' %pixelnum)
            try:
                pixelnum = int(pixelnum)
            except:
                return False
            if pixelnum>40:
                pixelnum = 40
            elif pixelnum <1:
                pixelnum = 1
        pixelnum-=1
        cmd = 0x03

        if color>0xFFFFFF:
            if not safety:
                raise ValueError('Color value too large. Color is 24 bit only.')
            color = 0xFFFFFF
        #first send high 8 bits bits of color
        RPiSoC.commChannel.sendData((self.address, cmd, color>>16))
        #now send the remaining 16 bits in the data bytes, and the pixel num set as the address, cmd is arbitrary
        RPiSoC.commChannel.receiveData((pixelnum, cmd, color&0xFFFF))
        #self.__recentpixel = [0,0, color]
        self.__Trigger()

    def DrawRow(self, row, color):
        """
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | Method            | DrawRow                                                                                                                                                                                                       |
        +===================+===============================================================================================================================================================================================================+
        | **Description**   |Fully draws the chosen *row* the desired *color*                                                                                                                                                               |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | **Parameters**    |* **int**   *row*                                                                                                                                                                                              |
        |                   |* **int**   *color*                                                                                                                                                                                            |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        |   *row*           |Row which will be fully drawn                                                                                                                                                                                  |
        |                   |   * valid between 0 and 4                                                                                                                                                                                     |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        |   *color*         |A 24-bit number representative of a BRG value                                                                                                                                                                  |
        |                   |   * BRG components are given equal weight, so 8-bits each                                                                                                                                                     |
        |                   |   * There are predefined colors inside of :meth:`~__init__`, which can be called as shield.[*color name*]; example: :code:`shield.Blue`                                                                       |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | **Returns**       |No return                                                                                                                                                                                                      |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+

        """
        cmd = 0x05
        if color>0xFFFFFF:
            raise ValueError('Color value too large. Color is 24 bit only.')
        #first send high 8 bits bits of color
        RPiSoC.commChannel.sendData((self.address, cmd, color>>16))
        #now send the remaining 16 bits in the data bytes, and the pixel num set as the address, cmd is arbitrary
        RPiSoC.commChannel.receiveData((row, cmd, color&0xFFFF))
        #self.__recentpixel = [row, 0, color]
        self.__Trigger()

    def DrawColumn(self, column, color):
        """
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | Method            | DrawColumn                                                                                                                                                                                                    |
        +===================+===============================================================================================================================================================================================================+
        | **Description**   |Fully draws the chosen *column* the desired *color*                                                                                                                                                            |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | **Parameters**    |* **int**   *column*                                                                                                                                                                                           |
        |                   |* **int**   *color*                                                                                                                                                                                            |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        |   *column*        |Column which will be fully drawn                                                                                                                                                                               |
        |                   |   * valid between 0 and 7                                                                                                                                                                                     |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        |   *color*         |A 24-bit number representative of a BRG value                                                                                                                                                                  |
        |                   |   * BRG components are given equal weight, so 8-bits each                                                                                                                                                     |
        |                   |   * There are predefined colors inside of :meth:`~__init__`, which can be called as shield.[*color name*]; example: :code:`shield.Blue`                                                                       |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | **Returns**       |No return                                                                                                                                                                                                      |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+


        """

        cmd = 0x06
        if color>0xFFFFFF:
            raise ValueError('Color value too large. Color is 24 bit only.')
        #first send high 8 bits bits of color
        RPiSoC.commChannel.sendData((self.address, cmd, color>>16))
        #now send the remaining 16 bits in the data bytes, and the pixel num set as the address, cmd is arbitrary
        RPiSoC.commChannel.receiveData((column, cmd, color&0xFFFF))
        #self.__recentpixel = [0, column, color]
        self.__Trigger()

    def Dim(self, DIM_LEVEL):
        """
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | Method            | Dim                                                                                                                                                                                                           |
        +===================+===============================================================================================================================================================================================================+
        | **Description**   |Preserves the color to be drawn to the shield, but dims the brightness. Effects take place immediately.                                                                                                        |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | **Parameters**    |* **int**   *DIM_LEVEL*                                                                                                                                                                                        |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        |   *DIM_LEVEL*     |An integer between 0 and 4                                                                                                                                                                                     |
        |                   |   * 0 indicating full brightness, or minimum dim level                                                                                                                                                        |
        |                   |   * 4 indicating minimum brightness, or full dim level                                                                                                                                                        |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+
        | **Returns**       |No return                                                                                                                                                                                                      |
        +-------------------+---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------+

        """
        cmd = 0x04
        if DIM_LEVEL not in range(5):
            raise ValueError('Dim level must be between 0 and 4')
        RPiSoC.commChannel.receiveData((self.address, cmd, DIM_LEVEL))

    def Fill(self, color):
        cmd = 0x07
        RPiSoC.commChannel.sendData((self.address, cmd))#address and command
        RPiSoC.commChannel.receiveData((color>>16, (color>>8)&0xFF, color&0xFF))#color split into GRB bytes
        self.__Trigger()

    def __Trigger(self):
        cmd = 0x08
        RPiSoC.commChannel.receiveData((self.address, cmd))

    def __clearMem(self):
        cmd = 0x09
        RPiSoC.commChannel.receiveData((self.address, cmd))





