"""Functions intended for library internal use."""
import os
from pathlib import Path
import re
import time


def get_latest_timestamped_file_path(file_path: str) -> str:
    """Get the latest version of a file, based on the unix timestamp in its name.

    Args:
        file_path: The file path of which to search for a timestamped version of the file

    Raises:
        FileNotFoundError: Raised if no timestamped version of the file can be found.

    Returns:
        The path of an existing timestamped version of the provided path.
    """
    file_no_extension, extension = os.path.splitext(file_path)

    # Regular expression for matching timestamped files in the format:
    # <filename>_<timestamp><extension> e.g. '/foo/bar_12345.baz'
    file_regex = rf'{re.escape(file_no_extension)}_(?P<unix_timestamp>\d+){re.escape(extension)}'

    # Get all files in directory of given file
    files_in_dir = [str(f) for f in Path(file_path).parent.glob('*') if f.is_file()]

    # Find version of requested file with latest timestamp
    latest_file_path = None
    latest_unix_timestamp = -1
    for file_name in files_in_dir:
        match = re.match(file_regex, file_name)

        # No match -> file is not a timestamped version of file_path
        if match is None:
            continue

        unix_timestamp = int(match.group('unix_timestamp'))

        # Unix timestamp is higher -> file is newer
        if unix_timestamp > latest_unix_timestamp:
            latest_file_path = file_name
            latest_unix_timestamp = unix_timestamp

    # Raise error if no timestamped file exists
    if latest_file_path is None:
        raise FileNotFoundError(f"No timestamped file for {file_path}")

    return latest_file_path


def generate_timestamped_file_path(file_path: str, exists_ok: bool = True) -> str:
    """Add a unix timestamp at the end of the filename but before the file extension.
    This function only generates the path name, it does not actually create or modify any files.

    Args:
        file_path: The path to which to add a unix timestamp
        exists_ok: Whether to raise an error if a file already exists at the generated path. Defaults to True.

    Raises:
        FileExistsError: Raised if `exists_ok` is `False` and a file at the path generated by this
            function already exists.

    Returns:
        A path with an added unix timestamp.
    """
    file_no_extension, extension = os.path.splitext(file_path)

    # Get unix timestamp, rounded down to seconds
    unix_timestamp = int(time.time())

    # Add timestamp to file path
    timestamped_file_path = f'{file_no_extension}_{unix_timestamp}{extension}'

    # Raise an exception if the file already exists
    if not exists_ok and os.path.exists(timestamped_file_path):
        raise FileExistsError(f'Timestamped file {timestamped_file_path} already exists')

    return timestamped_file_path

