#!/usr/bin/python
# -*- coding: latin-1 -*- vim: ts=8 sts=4 et si sw=4 tw=79 
u"""\
%(prog)s.py: Python-Modul fr Fehlerausgaben (fr Konsolenprogramme)
"""

__author__ = "Tobias Herp <tobias.herp@gmx.net>"

# TODO:
# * introspection:
#   - skip foreign functions and objects (from imported modules)
#   - for pairs of identical functions (e.g. err and error),
#     print once and refer to it
#   - filtering by name
#   - put the whole introspection functionality in a special module

from sys import stdout, stderr, exit, argv, exc_info
from os.path import basename

_beispiele = u"""\
Einfaches Anwendungsbeispiel (Python-Skript):

Das folgende Python-Skript erwartet ganze Zahlen als Argumente.
Es berprft zunchst alle bergebenen Argumente und gibt fr *jedes* falsche
Argument eine Fehlermeldung aus; wenn Fehler aufgetreten sind, beendet es das
Programm, und gibt ansonsten das Produkt aller bergebenen Zahlen aus:

<snip>
from %(prog)s import err, check_errors
from sys import argv
numbers = []
for a in argv[1:]:
    try:
        numbers.append(int(a))
    except ValueError:
        err('%%r ist keine ganze Zahl' %% a)

check_errors()
print reduce(lambda a, b: a*b, numbers)
</snip>

(In einer interaktiven Python-Sitzung knnen Sie Zeile 2 durch so etwas wie
"argv = 'ignored 1 a 2 b 3 4'.split()" ersetzen.  Wenn argv[1:] Nicht-Zahlen
enthlt, wird check_errors() die Python-Shell beenden.)

Hilfe, auch zu Funktionen und Daten: %(prog)s.py -h
"""
VERSION = '.'.join(map(str,
                   (0,
                    3, # modinfo
                    1, # progname: strip -script.py as well
                    'rev-%s' % '$Rev: 959 $'[6:-2],
                    )))

RC_ERROR = 1   # bei Fehler ans System zurckzugeben, wenn nicht ein Zhler o..
RC_OK = 0      # trivial, aber hiermit dokumentiert
RC_HELP = 3    # nach Hilfeausgabe zurckzugeben
RC_CANCEL = 98 # bei Abbruch durch Benutzer, z. B. nach Prompt
RC_ABORT = 99  # bei Abbruch durch Benutzer durch Interrupt

def set_progname(name=None):
    """
    setze den von warning, err, fatal und info auszugebenden Namen;
    in der Regel nicht ntiger Aufruf:
    set_progname(progname())
    """
    global _PROGNAME, _LOGNAME
    if name:
        _PROGNAME = name+':'
        _LOGNAME = '['+name+']'
    else:
        _PROGNAME = ''
        _LOGNAME = ''

set_progname('errors')

def info(text, to=None):
    """
    gib die bergebene Info aus, per Default zur Standardausgabe
    """
    if to is None:
        to = stdout
    print >> to, _PROGNAME+'i', text

WARNINGS = 0
def warn(text):
    """
    gib die bergebene Warnung aus und erhhe den Warnungszhler
    """
    global WARNINGS
    WARNINGS += 1
    print >> stderr, _PROGNAME+'W', text
warning = warn

ERRORS = 0
def err(text, code=None):
    """
    gib die bergebene Fehlermeldung aus und beende das Programm,
    sofern ein Code ungleich None bergeben wird; andernfalls
    wird der Fehlerzhler erhht
    """
    global ERRORS
    print >> stderr, _PROGNAME+'E', text
    if code is not None:
        exit(code or 1)
    ERRORS += 1
error = err

def fatal(text=None,
          code=None,
          tell=True,
          count=None,
          help=None):
    """
    Gib einen Text aus und beende jedenfalls das Programm.
    Argumente:

    text -- der auszugebende Text

    Optional:

    code -- der Returncode; Default: die Anzahl ERRORS der bisher
            aufgetretenen Fehler, oder 1
    tell -- 1 bzw. true: Info ber Anzahl bisheriger Fehler ausgeben
            2: auch Info ber Anzahl bisheriger Warnungen ausgeben
            Wenn None bergeben wird, schlgt eine Automatik zu
    count -- boolean: den aktuellen Fehler mitzhlen?

    help -- wenn bergeben, ein Hinweis auf die Hilfe
    """
    global ERRORS
    if count is None:
        count = False # bool(text)
    if count:
        ERRORS += 1
    if tell is None: # Automatik
        tell = not text or not count

    if text is not None:
        print >> stderr, _PROGNAME+'!', text
    RC = code or ERRORS or RC_ERROR or 1
    if tell:
        if ERRORS:
            info('%d Fehler' % ERRORS, stderr)
        if WARNINGS and tell > 1:
            info('%d Warnung[en]' % WARNINGS, stderr)
    if help:
        info(help)
    if tell:
        info('Beende mit RC %d' % RC, stderr)
    exit(RC)

def errline(text=''):
    """
    gib eine (per Default leere) Zeile zur Standardfehlerausgabe aus

    text -- der auszugebende Text
    """
    print >> stderr, text

def progname(stripchar=None, stripext=True, stripscript=True):
    """
    gib den Namen des Programms zurck, ohne Verzeichnisangaben und
    Extension
    """
    tmp = basename(argv[0])
    tails = []
    if stripext:
        tails.extend(['.py', '.exe',
                      ])
    if stripscript:
        tails.append('-script')
    for tail in tails:
        if tmp.endswith(tail):
            tmp = tmp[:len(tail)]
    if stripchar:
        tmp = tmp.rstrip(stripchar)
    return tmp

def check_errors(text=None,
                 code=None):
    """
    ggf. zu fatal() durchgereichte Argumente:

    text -- als help-Argument durchgereicht, ein Hinweis
            auf die Hilfe (unterdrcken durch '')
    """
    if text is None:
        text = u"Aufruf mit -h oder --help fr Hilfe"
    if ERRORS:
        fatal(code=code, tell=True, count=False, help=text)

def prompt(txt):
    """
    vorlufige Version; noch hartcodierte Antwortmglichkeiten
    (yes/ja vs. no/nein), und noch kein yes/no/all/none/abort
    """
    idx = 0
    txt += ' (y/n) '
    txt2 = 'Bitte mit yes/ja oder no/nein antworten! '
    try:
        while True:
            try:
                answer = raw_input(idx and txt2 or txt)
            except AssertionError, e: # der isses nicht...
                print >> stderr, idx and txt2 or txt,
                answer = raw_input(': ')
            except EOFError:
                warn('stdin is redirected')
                return 0
            answer = answer.strip().lower()
            if not answer:
                idx = 1
                continue
            if 'yes'.startswith(answer):
                return 1
            elif 'ja'.startswith(answer):
                return 1
            elif 'no'.startswith(answer):
                return 0
            elif 'nein'.startswith(answer):
                return 0
            idx = (idx + 1) % 5
    finally:
        print

set_progname(progname())

try:
    from gettext import gettext as _
except ImportError:
    def _(message):
        return message

# direkter Aufruf auf Kommandozeile (nicht ausgefhrt bei Import):
if __name__ == '__main__':
    import thebops.modinfo
    thebops.modinfo.main(version=VERSION)
