# meta.py - retrieve feature system from config file section

from ._compat import string_types, copyreg

import fileconfig

__all__ = ['Config', 'FeatureSystemMeta', 'FeatureSetMeta']

DEFAULT = 'default'


class Config(fileconfig.Stacked):
    """Define possible feature combinations and their minimal specification."""

    filename = 'config.ini'

    _encoding = 'utf_8_sig'

    def __init__(self, key, context, format='table', aliases=None, inherits=None,
                 str_maximal=False, description=None):
        self.key = key
        self.context = context.strip()
        self.format = format
        self.aliases = [] if aliases is None else aliases
        self.inherits = inherits
        self.str_maximal = (False if not str_maximal
            else True if str_maximal is True
            else str_maximal.lower() in ('1', 'yes', 'true', 'on'))
        self.description = '' if description is None else description.strip()


class FeatureSystemMeta(type):
    """Idempotently cache and return feature system instances by config."""

    __map = {}

    def __call__(self, config=DEFAULT, string=None):
        if isinstance(config, self):
            return config

        if isinstance(config, string_types):
            config = Config(config)

        if config.key is not None and config.key in self.__map:
            inst = self.__map[config.key]
        else:
            inst = super(FeatureSystemMeta, self).__call__(config)
            self.__map.update(dict.fromkeys(inst._config.names, inst))

        if string is not None:
            if string == -1:  # unpickle set class
                return inst.FeatureSet
            return inst(string)
        return inst


class FeatureSetMeta(type):

    system = None

    def __call__(self, string=''):
        return self.system(string)

    def __repr__(self):
        if self.system is None:
            return type.__repr__(self)
        return '<class %r of %r>' % (self.__name__, self.system)

    def __reduce__(self):
        if self.system is None:
            return self.__name__
        elif self.system.key is None:
            return self.system.__class__, (self.system._config, -1)
        return self.system.__class__, (self.system.key, -1)


copyreg.pickle(FeatureSetMeta, FeatureSetMeta.__reduce__)
