# Copyright 2010 Boris Figovsky <borfig@gmail.com>
#
# This file is part of pybfc.

# pybfc is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.

# pybfc is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.

# You should have received a copy of the GNU General Public License
# along with pybfc.  If not, see <http://www.gnu.org/licenses/>.
"""
Process-related utilities:

Running code in another process:

>>> import os, sys
>>> func = os.getpid
>>> func() != run_synchronically_in_another_process(func)
True

Note: everything must be picklable! Use bfc.picklablemodule if import files dynammically!

"""

from multiprocessing import Process, Queue
from threading import Thread

__all__ = ['run_synchronically_in_another_process', 'remote_file']

def run_synchronically_in_another_process(func, *args, **kws):
    p = Process(target = func, args = args, kwargs = kws)
    p.start()
    p.join()

class RemoteFile(object):
    __slots__ = ['_queue',
                 ]
    def __init__(self, queue):
        self._queue = queue

    def write(self, data):
        self._queue.put(str(data))

    def flush(self):
        pass

    def close(self):
        self._queue.put(None)

class LocalFile(Thread):
    __slots__ = ['_queue',
                 '_file',
                 ]
    def __init__(self, queue, fileobj):
        Thread.__init__(self)
        self._queue = queue
        self._file = fileobj

    def run(self): # pragma: no cover
        # this is run in another thread so it is not covered by coverage/nosetests :(
        while True:
            data = self._queue.get()
            if data is None:
                return
            self._file.write(data)

    def end(self):
        self._queue.put(None)
        self.join()

def remote_file(fileobj):
    queue = Queue()
    local_file = LocalFile(queue, fileobj)
    remote_file = RemoteFile(queue)
    local_file.start()
    return local_file, remote_file
    
