use std::sync::atomic::{AtomicU32, Ordering};

use criterion::{BatchSize, BenchmarkId, Criterion};
use relay_cardinality::{
    limiter::{Entry, EntryId, Limiter},
    RedisSetLimiter, SlidingWindow,
};
use relay_redis::{redis, RedisConfigOptions, RedisPool};

fn build_redis() -> RedisPool {
    let url =
        std::env::var("RELAY_REDIS_URL").unwrap_or_else(|_| "redis://127.0.0.1:6379".to_owned());

    RedisPool::single(&url, RedisConfigOptions::default()).unwrap()
}

fn build_limiter(redis: RedisPool, reset_redis: bool) -> RedisSetLimiter {
    let mut client = redis.client().unwrap();
    let mut connection = client.connection().unwrap();

    if reset_redis {
        redis::cmd("FLUSHALL").execute(&mut connection);
    }

    let window = SlidingWindow {
        window_seconds: 3600,
        granularity_seconds: 360,
    };
    RedisSetLimiter::new(redis, window)
}

#[derive(Debug)]
struct Params {
    limit: usize,

    rounds: usize,
    num_hashes: usize,
}

impl Params {
    /// Every round contains the same hashes.
    fn rounds(&self) -> Vec<Vec<Entry<&'static str>>> {
        let entries = (0..self.num_hashes)
            .map(|i| Entry::new(EntryId(i), "custom", u32::MAX - (i as u32)))
            .collect::<Vec<_>>();

        (0..self.rounds)
            .map(move |_| entries.clone())
            .collect::<Vec<_>>()
    }

    /// High cardinality, every round contains unique hashes.
    fn rounds_unique(&self) -> Vec<Vec<Entry<&'static str>>> {
        let hash = AtomicU32::new(u32::MAX);

        (0..self.rounds)
            .map(move |_| {
                (0..self.num_hashes)
                    .map(|i| Entry::new(EntryId(i), "custom", hash.fetch_sub(1, Ordering::SeqCst)))
                    .collect::<Vec<_>>()
            })
            .collect::<Vec<_>>()
    }

    /// Entry which is never generated by either [`Self::rounds`] or [`Self::rounds_unique`].
    fn never_entry() -> Entry<&'static str> {
        Entry::new(EntryId(usize::MAX), "custom", 0)
    }

    /// A vector of entries which is never generated by either [`Self::rounds`] or [`Self::rounds_unique`].
    fn never_entries(&self) -> Vec<Entry<&'static str>> {
        (0..self.limit)
            .map(|i| Entry::new(EntryId(usize::MAX - i), "custom", i as u32))
            .collect::<Vec<_>>()
    }
}

impl std::fmt::Display for Params {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        write!(f, "{:?}", self)
    }
}

pub fn bench_simple(c: &mut Criterion) {
    let params = [
        Params {
            limit: 10000,
            rounds: 1000,
            num_hashes: 50,
        },
        Params {
            limit: 10000,
            rounds: 1000,
            num_hashes: 500,
        },
        Params {
            limit: 10000,
            rounds: 50,
            num_hashes: 10000,
        },
    ];

    let redis = build_redis();

    let mut g = c.benchmark_group("Simple");

    for params in params {
        g.throughput(criterion::Throughput::Elements(params.rounds as u64));
        g.bench_with_input(BenchmarkId::new("simple", &params), &params, |b, params| {
            b.iter_batched(
                || {
                    let limiter = build_limiter(redis.clone(), true);
                    (limiter, params.rounds())
                },
                |(limiter, rounds)| {
                    for entries in rounds {
                        limiter
                            .check_cardinality_limits(1, entries, params.limit)
                            .unwrap()
                            .for_each(|_| {});
                    }
                },
                BatchSize::SmallInput,
            )
        });
    }
}

pub fn bench_big_set_small_queries(c: &mut Criterion) {
    let redis = build_redis();

    let params = Params {
        num_hashes: 50,
        rounds: 1000,
        limit: 10_000,
    };

    let mut g = c.benchmark_group("Big set small queries");
    g.throughput(criterion::Throughput::Elements(params.rounds as u64));
    g.bench_function("big_set_small_queries", |b| {
        b.iter_batched(
            || {
                let limiter = build_limiter(redis.clone(), true);
                let rounds = params.rounds();

                // Seed with the round data
                limiter
                    .check_cardinality_limits(1, rounds[0].clone(), params.limit)
                    .unwrap()
                    .for_each(|_| {});

                (limiter, rounds)
            },
            |(limiter, rounds)| {
                for entries in rounds {
                    limiter
                        .check_cardinality_limits(1, entries, params.limit)
                        .unwrap()
                        .for_each(|_| {});
                }
            },
            BatchSize::SmallInput,
        )
    });
}

pub fn bench_high_cardinality(c: &mut Criterion) {
    let redis = build_redis();

    let params = Params {
        num_hashes: 50,
        rounds: 1000,
        limit: 10_000,
    };

    let mut g = c.benchmark_group("High Cardinality");
    g.throughput(criterion::Throughput::Elements(params.rounds as u64));
    g.bench_function("high_cardinality", |b| {
        b.iter_batched(
            || {
                let limiter = build_limiter(redis.clone(), true);
                (limiter, params.rounds_unique())
            },
            |(limiter, rounds)| {
                for entries in rounds {
                    limiter
                        .check_cardinality_limits(1, entries, params.limit)
                        .unwrap()
                        .for_each(|_| {});
                }
            },
            BatchSize::SmallInput,
        )
    });
}

pub fn bench_cache_never_full(c: &mut Criterion) {
    let redis = build_redis();

    let params = Params {
        num_hashes: 50,
        rounds: 1000,
        limit: 10_000,
    };

    let mut g = c.benchmark_group("Cache Never Full");
    g.throughput(criterion::Throughput::Elements(params.rounds as u64));
    g.bench_function("cache_never_full", |b| {
        b.iter_batched(
            || {
                let limiter = build_limiter(redis.clone(), true);
                limiter
                    .check_cardinality_limits(1, vec![Params::never_entry()], 1)
                    .unwrap()
                    .for_each(|_| {});

                // New limiter to reset cache.
                let limiter = build_limiter(redis.clone(), false);
                (limiter, params.rounds_unique())
            },
            |(limiter, rounds)| {
                for entries in rounds {
                    limiter
                        .check_cardinality_limits(1, entries, params.limit)
                        .unwrap()
                        .for_each(|_| {});
                }
            },
            BatchSize::SmallInput,
        )
    });
}

pub fn bench_cache_worst_case(c: &mut Criterion) {
    let redis = build_redis();

    let params = Params {
        num_hashes: 50,
        rounds: 1000,
        limit: 10_000,
    };

    let mut g = c.benchmark_group("Cache Worst Case");
    g.throughput(criterion::Throughput::Elements(params.rounds as u64));
    g.bench_function("cache_worst_case", |b| {
        b.iter_batched(
            || {
                let limiter = build_limiter(redis.clone(), true);
                limiter
                    .check_cardinality_limits(1, params.never_entries(), params.limit)
                    .unwrap()
                    .for_each(|_| {});

                // New limiter to reset cache.
                let limiter = build_limiter(redis.clone(), false);
                (limiter, params.rounds_unique())
            },
            |(limiter, rounds)| {
                for entries in rounds {
                    limiter
                        .check_cardinality_limits(1, entries, params.limit)
                        .unwrap()
                        .for_each(|_| {});
                }
            },
            BatchSize::SmallInput,
        )
    });
}
