"""
Read an ASCII Multi Channel
Analyzer (**MCA**) spectrum file (SPE extension) created with the
Ortec program MAESTRO.

Example::

    >>> import LT.MCA as M
    >>> sp = M.MCA('spectrum.spe')


--------------------------------------------------------


"""
# read an mca file

import numpy as np

# use calibration to calculate the energies
def get_energy(cal_par, ch):
    if len(cal_par) == 2:
        return cal_par[1]*ch + cal_par[0]
    else:
        return (cal_par[2]*ch + cal_par[1])*ch + cal_par[0]
#

class MCA:
    """
    MCA class for ascii spectrum files generated by MAESTRO software

    Usage::
        >>> sp = LT.MCA.MCA('spectrum.spe')

    Assuming that LT.MCA was imported

    """
    def __init__(self, file = None):
        self.spectrum = {
            '$SPEC_ID': '', \
                '$SPEC_REM': [], \
                '$DATE_MEA': '', \
                '$MEAS_TIM': '', \
                '$DATA': [], \
                '$DATA_RANGE': [], \
                '$ROI': [], \
                '$PRESETS': [], \
                '$ENER_FIT': '', \
                '$MCA_CAL': [], \
                '$SHAPE_CAL': [], \
                }
        if file != None:
            self.read_ascii_file(file)
            self.data_setup()

    def read_ascii_file(self, file):
        self.sp_data = []
        data = open(file).readlines()
        exist = False
        for l in data:
            if l.strip().find('$') == 0 :
                fields = l.strip().split(':')
                current_key = fields[0].strip()
                exist = current_key in self.spectrum
                continue
            if exist :
                if current_key == '$SPEC_ID':
                    # read spec ID
                    self.spectrum[current_key] = l
                elif current_key ==     '$SPEC_REM':
                    # read comments
                    self.spectrum[current_key].append(l.strip())
                elif current_key ==     '$DATE_MEA':
                    self.spectrum[current_key] = l.strip()
                elif current_key ==     '$MEAS_TIM':
                    self.spectrum[current_key] = np.array(l.split(), dtype = float)
                elif current_key ==     '$DATA':
                    self.sp_data.append(l.strip())
                elif current_key ==     '$ROI':
                    self.spectrum[current_key].append(l.strip())
                elif current_key ==     '$PRESETS':
                    self.spectrum[current_key].append(l.strip())
                elif current_key ==     '$ENER_FIT':
                    self.spectrum[current_key] = l.split()
                elif current_key ==     '$MCA_CAL':
                    self.spectrum[current_key].append(l.strip())
                elif current_key ==     '$SHAPE_CAL':
                    self.spectrum[current_key].append(l.strip())
    def data_setup(self):
        # thats all now convert to proper format
        self.spectrum['$DATA_RANGE'] = np.array(self.sp_data[0].split(), dtype = float)
        self.spectrum['$DATA'] = np.array(self.sp_data[1:], dtype = float)
        # calibration data
        # number of cal par (needs to be checked)
        ncal = int(self.spectrum['$MCA_CAL'][0])
        # calibration parameters
        # self.cal_par = np.array( self.spectrum['$ENER_FIT'], dtype = float )
        self.cal_par = np.array( self.spectrum['$MCA_CAL'][1].split()[:ncal], dtype = float )
        # channel numbers
        self.chn = np.arange( self.spectrum['$DATA_RANGE'][0], self.spectrum['$DATA_RANGE'][1] + 1.)
        # converted to energy
        self.energy = get_energy(self.cal_par, self.chn)
        self.cont = self.spectrum['$DATA']


