# Copyright European Space Agency, 2013

from __future__ import division

import unittest
import numpy as np
    
from auromat.coordinates.geodesic import containsOrCrossesPole, course, Location
from auromat.utils import convexHull
from numpy.testing.utils import assert_equal
from geographiclib.geodesic import Geodesic

class Test(unittest.TestCase):

    def testContainsPole(self):
        # a polygon that doesn't cross or contain a pole
        poly1 = [[1,0], [1,4], [5,6], [5,2]]
        assert not containsOrCrossesPole(poly1)
        
        # as poly1 but spanning the discontinuity
        poly1a = [[1,179], [1,-177], [5,-175], [5,-179]]
        assert not containsOrCrossesPole(poly1a)
        
        # a polygon around the north pole
        poly2 = [[85,-135], [85,-45], [85,45], [85,135]]
        assert containsOrCrossesPole(poly2)
        
        # a polygon crossing the north pole
        poly3 = [[85,-90], [85,0], [85,90]]
        assert containsOrCrossesPole(poly3)
        
        #http://testbed.echo.nasa.gov/spatial-viz/interactive_spherical_polygon_coverage?ring_ordinates=0,90,0,85,10,85,0,90#
        # a polygon with a point on the north pole
        # FIXME broken, courses are wrong
        # courses: [ 180.         180.          85.0189306   94.9810694    0.         350.       ]
        # 350 should be 0!
        
        # we are pointing straight to the pole, so the compass should point North 0 deg
        # -> there are different kinds of azimuth, we may have the wrong one
        #  true-north azimuth, grid azimuth
#         c1 = course(Location(90,0), Location(85,0))
#         c2 = course(Location(90,10), Location(85,0))
#         c3 = course(Location(85,0), Location(90,0))
#         c4 = course(Location(85,0), Location(90,10))
#         assert_equal(c1,180)
#         assert_equal(c2,-180)
#         assert_equal(c3,0)
#         assert_equal(c4,0)
#         
#         import logging
#         logging.basicConfig(level=logging.DEBUG)
#         logging.root.setLevel(logging.DEBUG)
#         
#         poly4 = [[90,0], [85,0], [85,10]]
#         assert containsOrCrossesPole(poly4)
        
    def testPoleBug(self):
        outlineReduced100 = np.array([[61.05555555555556, -92.91483406637346],
                                     [60.97222222222223, -93.82646941223511],
                                     [60.97222222222223, -94.73810475809677],
                                     [60.88888888888889, -95.64974010395842],
                                     [60.833333333333336, -96.32147141143543],
                                     [60.66666666666667, -96.89724110355858],
                                     [60.52777777777778, -97.18512594962016],
                                     [60.41666666666667, -97.28108756497402],
                                     [60.611111111111114, -97.4250299880048],
                                     [60.63888888888889, -97.71291483406638],
                                     [60.16666666666667, -97.7608956417433],
                                     [60.611111111111114, -97.85685725709716],
                                     [60.19444444444445, -97.9048380647741],
                                     [60.63888888888889, -98.04878048780488],
                                     [60.111111111111114, -98.09676129548181],
                                     [60.52777777777778, -98.19272291083567],
                                     [60.22222222222223, -98.24070371851259],
                                     [60.27777777777778, -98.33666533386645],
                                     [60.361111111111114, -98.38464614154339],
                                     [59.91666666666667, -98.62455017992804],
                                     [60.47222222222223, -98.52858856457418],
                                     [60.083333333333336, -98.62455017992804],
                                     [59.91666666666667, -98.81647341063575],
                                     [60.47222222222223, -98.7205117952819],
                                     [60.111111111111114, -98.81647341063575],
                                     [59.55555555555556, -99.24830067972812],
                                     [60.111111111111114, -98.96041583366653],
                                     [60.333333333333336, -98.96041583366653],
                                     [59.77777777777778, -99.20031987205118],
                                     [59.25, -100.01599360255898],
                                     [59.80555555555556, -99.34426229508198],
                                     [60.361111111111114, -99.15233906437425],
                                     [60.05555555555556, -99.24830067972812],
                                     [59.47222222222222, -99.87205117952819],
                                     [59.111111111111114, -100.83166733306678],
                                     [58.97222222222222, -101.79128348660535],
                                     [59.05555555555556, -102.75089964014394],
                                     [59.361111111111114, -103.71051579368253],
                                     [59.05555555555556, -102.94282287085166],
                                     [58.94444444444444, -101.98320671731308],
                                     [59.02777777777778, -101.02359056377449],
                                     [59.05555555555556, -100.5437824870052],
                                     [58.88888888888889, -101.50339864054378],
                                     [58.91666666666667, -102.46301479408237],
                                     [59.111111111111114, -103.47061175529788],
                                     [59.13888888888889, -103.6625349860056],
                                     [59.13888888888889, -104.14234306277488],
                                     [59.05555555555556, -104.71811275489804],
                                     [58.77777777777778, -105.38984406237505],
                                     [58.52777777777778, -106.10955617752899],
                                     [58.19444444444444, -106.82926829268293],
                                     [57.861111111111114, -107.64494202319072],
                                     [57.44444444444444, -108.4126349460216],
                                     [57.05555555555556, -109.0843662534986],
                                     [56.611111111111114, -109.7560975609756],
                                     [56.16666666666667, -110.33186725309876],
                                     [55.833333333333336, -110.85965613754497],
                                     [55.416666666666664, -111.29148340663734],
                                     [55.02777777777778, -110.33186725309876],
                                     [54.638888888888886, -109.37225109956017],
                                     [54.25, -108.4126349460216],
                                     [53.833333333333336, -107.45301879248301],
                                     [53.388888888888886, -106.4454218312675],
                                     [52.916666666666664, -105.4858056777289],
                                     [52.47222222222222, -104.57417033186725],
                                     [51.97222222222222, -103.61455417832867],
                                     [51.47222222222222, -102.65493802479008],
                                     [50.916666666666664, -101.64734106357457],
                                     [50.36111111111111, -100.68772491003598],
                                     [49.80555555555556, -99.77608956417433],
                                     [49.25, -98.86445421831267],
                                     [48.69444444444444, -98.04878048780488],
                                     [48.138888888888886, -97.23310675729708],
                                     [48.0, -96.2734906037585],
                                     [48.22222222222222, -95.31387445021991],
                                     [48.47222222222222, -94.35425829668134],
                                     [48.72222222222222, -93.39464214314275],
                                     [48.97222222222222, -92.43502598960416],
                                     [49.36111111111111, -91.90723710515795],
                                     [49.94444444444444, -91.90723710515795],
                                     [50.5, -91.95521791283487],
                                     [51.05555555555556, -92.0031987205118],
                                     [51.611111111111114, -92.05117952818873],
                                     [52.166666666666664, -92.09916033586566],
                                     [52.72222222222222, -92.09916033586566],
                                     [53.30555555555556, -92.14714114354258],
                                     [53.861111111111114, -92.19512195121952],
                                     [54.416666666666664, -92.24310275889644],
                                     [54.97222222222222, -92.29108356657338],
                                     [55.52777777777778, -92.29108356657338],
                                     [56.083333333333336, -92.3390643742503],
                                     [56.66666666666667, -92.38704518192723],
                                     [57.22222222222222, -92.43502598960416],
                                     [57.77777777777778, -92.48300679728109],
                                     [58.333333333333336, -92.53098760495803],
                                     [58.88888888888889, -92.57896841263495],
                                     [59.44444444444445, -92.57896841263495],
                                     [60.02777777777778, -92.62694922031187],
                                     [60.583333333333336, -92.67493002798881],
                                     [61.05555555555556, -92.86685325869652]])
        
        outlineReduced50 = outlineReduced100[::2]
        
        # full outline
        outlineFull = np.array([[61.05555555555556, -92.91483406637346],
                             [61.02777777777778, -92.91483406637346],
                             [61.0, -92.96281487405038],
                             [61.0, -93.01079568172732],
                             [61.0, -93.05877648940424],
                             [61.0, -93.10675729708117],
                             [61.02777777777778, -93.1547381047581],
                             [61.02777777777778, -93.20271891243503],
                             [61.0, -93.25069972011195],
                             [61.02777777777778, -93.29868052778889],
                             [61.02777777777778, -93.34666133546583],
                             [61.0, -93.39464214314275],
                             [61.0, -93.44262295081967],
                             [61.0, -93.4906037584966],
                             [60.97222222222223, -93.53858456617354],
                             [60.97222222222223, -93.58656537385046],
                             [60.97222222222223, -93.6345461815274],
                             [60.97222222222223, -93.68252698920432],
                             [61.0, -93.73050779688126],
                             [61.0, -93.77848860455818],
                             [60.97222222222223, -93.82646941223511],
                             [60.97222222222223, -93.87445021991203],
                             [60.97222222222223, -93.92243102758897],
                             [60.97222222222223, -93.97041183526589],
                             [61.0, -94.01839264294283],
                             [61.0, -94.06637345061975],
                             [61.0, -94.11435425829669],
                             [61.0, -94.16233506597362],
                             [61.0, -94.21031587365054],
                             [60.97222222222223, -94.21031587365054],
                             [60.94444444444445, -94.25829668132747],
                             [60.94444444444445, -94.3062774890044],
                             [60.97222222222223, -94.35425829668134],
                             [60.97222222222223, -94.40223910435826],
                             [60.94444444444445, -94.4502199120352],
                             [60.97222222222223, -94.49820071971212],
                             [60.97222222222223, -94.54618152738905],
                             [60.97222222222223, -94.59416233506597],
                             [60.97222222222223, -94.64214314274291],
                             [60.97222222222223, -94.69012395041983],
                             [60.97222222222223, -94.73810475809677],
                             [60.94444444444445, -94.73810475809677],
                             [60.91666666666667, -94.78608556577369],
                             [60.94444444444445, -94.83406637345063],
                             [60.94444444444445, -94.88204718112755],
                             [60.91666666666667, -94.93002798880448],
                             [60.91666666666667, -94.97800879648142],
                             [60.91666666666667, -95.02598960415834],
                             [60.91666666666667, -95.07397041183526],
                             [60.91666666666667, -95.1219512195122],
                             [60.91666666666667, -95.16993202718913],
                             [60.91666666666667, -95.21791283486606],
                             [60.91666666666667, -95.26589364254298],
                             [60.88888888888889, -95.31387445021991],
                             [60.88888888888889, -95.36185525789685],
                             [60.91666666666667, -95.40983606557377],
                             [60.91666666666667, -95.4578168732507],
                             [60.88888888888889, -95.4578168732507],
                             [60.861111111111114, -95.50579768092763],
                             [60.861111111111114, -95.55377848860456],
                             [60.88888888888889, -95.60175929628149],
                             [60.88888888888889, -95.64974010395842],
                             [60.861111111111114, -95.69772091163534],
                             [60.861111111111114, -95.74570171931228],
                             [60.88888888888889, -95.7936825269892],
                             [60.88888888888889, -95.84166333466614],
                             [60.861111111111114, -95.88964414234306],
                             [60.861111111111114, -95.93762495002],
                             [60.861111111111114, -95.98560575769693],
                             [60.861111111111114, -96.03358656537385],
                             [60.861111111111114, -96.08156737305079],
                             [60.861111111111114, -96.12954818072771],
                             [60.833333333333336, -96.12954818072771],
                             [60.80555555555556, -96.17752898840465],
                             [60.80555555555556, -96.22550979608157],
                             [60.77777777777778, -96.22550979608157],
                             [60.75, -96.22550979608157],
                             [60.72222222222223, -96.2734906037585],
                             [60.75, -96.32147141143543],
                             [60.77777777777778, -96.32147141143543],
                             [60.80555555555556, -96.32147141143543],
                             [60.833333333333336, -96.32147141143543],
                             [60.833333333333336, -96.36945221911236],
                             [60.833333333333336, -96.41743302678928],
                             [60.833333333333336, -96.46541383446622],
                             [60.80555555555556, -96.46541383446622],
                             [60.77777777777778, -96.51339464214314],
                             [60.80555555555556, -96.56137544982008],
                             [60.80555555555556, -96.609356257497],
                             [60.77777777777778, -96.65733706517393],
                             [60.77777777777778, -96.70531787285086],
                             [60.75, -96.75329868052779],
                             [60.75, -96.80127948820473],
                             [60.72222222222223, -96.80127948820473],
                             [60.69444444444445, -96.80127948820473],
                             [60.66666666666667, -96.80127948820473],
                             [60.63888888888889, -96.80127948820473],
                             [60.611111111111114, -96.80127948820473],
                             [60.583333333333336, -96.84926029588165],
                             [60.611111111111114, -96.89724110355858],
                             [60.63888888888889, -96.89724110355858],
                             [60.66666666666667, -96.89724110355858],
                             [60.69444444444445, -96.89724110355858],
                             [60.72222222222223, -96.89724110355858],
                             [60.75, -96.89724110355858],
                             [60.75, -96.9452219112355],
                             [60.72222222222223, -96.9452219112355],
                             [60.69444444444445, -96.99320271891244],
                             [60.72222222222223, -97.04118352658936],
                             [60.75, -97.04118352658936],
                             [60.77777777777778, -97.04118352658936],
                             [60.77777777777778, -97.0891643342663],
                             [60.75, -97.0891643342663],
                             [60.72222222222223, -97.13714514194322],
                             [60.72222222222223, -97.18512594962016],
                             [60.69444444444445, -97.18512594962016],
                             [60.66666666666667, -97.18512594962016],
                             [60.63888888888889, -97.18512594962016],
                             [60.611111111111114, -97.18512594962016],
                             [60.583333333333336, -97.18512594962016],
                             [60.55555555555556, -97.18512594962016],
                             [60.52777777777778, -97.18512594962016],
                             [60.5, -97.18512594962016],
                             [60.47222222222223, -97.18512594962016],
                             [60.44444444444445, -97.18512594962016],
                             [60.41666666666667, -97.18512594962016],
                             [60.38888888888889, -97.18512594962016],
                             [60.361111111111114, -97.18512594962016],
                             [60.333333333333336, -97.18512594962016],
                             [60.30555555555556, -97.18512594962016],
                             [60.27777777777778, -97.18512594962016],
                             [60.25, -97.18512594962016],
                             [60.22222222222223, -97.18512594962016],
                             [60.19444444444445, -97.23310675729708],
                             [60.22222222222223, -97.28108756497402],
                             [60.25, -97.28108756497402],
                             [60.27777777777778, -97.28108756497402],
                             [60.30555555555556, -97.28108756497402],
                             [60.333333333333336, -97.28108756497402],
                             [60.361111111111114, -97.28108756497402],
                             [60.38888888888889, -97.28108756497402],
                             [60.41666666666667, -97.28108756497402],
                             [60.44444444444445, -97.28108756497402],
                             [60.47222222222223, -97.28108756497402],
                             [60.5, -97.28108756497402],
                             [60.52777777777778, -97.28108756497402],
                             [60.55555555555556, -97.28108756497402],
                             [60.583333333333336, -97.28108756497402],
                             [60.611111111111114, -97.28108756497402],
                             [60.63888888888889, -97.28108756497402],
                             [60.66666666666667, -97.28108756497402],
                             [60.69444444444445, -97.28108756497402],
                             [60.72222222222223, -97.28108756497402],
                             [60.72222222222223, -97.32906837265094],
                             [60.69444444444445, -97.32906837265094],
                             [60.66666666666667, -97.32906837265094],
                             [60.63888888888889, -97.32906837265094],
                             [60.611111111111114, -97.32906837265094],
                             [60.583333333333336, -97.32906837265094],
                             [60.55555555555556, -97.37704918032787],
                             [60.583333333333336, -97.4250299880048],
                             [60.611111111111114, -97.4250299880048],
                             [60.63888888888889, -97.4250299880048],
                             [60.66666666666667, -97.4250299880048],
                             [60.69444444444445, -97.4250299880048],
                             [60.69444444444445, -97.47301079568173],
                             [60.66666666666667, -97.47301079568173],
                             [60.63888888888889, -97.47301079568173],
                             [60.611111111111114, -97.47301079568173],
                             [60.583333333333336, -97.52099160335865],
                             [60.611111111111114, -97.56897241103559],
                             [60.63888888888889, -97.56897241103559],
                             [60.66666666666667, -97.56897241103559],
                             [60.69444444444445, -97.56897241103559],
                             [60.69444444444445, -97.61695321871252],
                             [60.66666666666667, -97.61695321871252],
                             [60.63888888888889, -97.61695321871252],
                             [60.611111111111114, -97.61695321871252],
                             [60.583333333333336, -97.61695321871252],
                             [60.55555555555556, -97.66493402638945],
                             [60.583333333333336, -97.71291483406638],
                             [60.611111111111114, -97.71291483406638],
                             [60.63888888888889, -97.71291483406638],
                             [60.66666666666667, -97.71291483406638],
                             [60.66666666666667, -97.7608956417433],
                             [60.63888888888889, -97.7608956417433],
                             [60.611111111111114, -97.7608956417433],
                             [60.583333333333336, -97.7608956417433],
                             [60.55555555555556, -97.7608956417433],
                             [60.52777777777778, -97.7608956417433],
                             [60.5, -97.71291483406638],
                             [60.47222222222223, -97.71291483406638],
                             [60.44444444444445, -97.71291483406638],
                             [60.41666666666667, -97.71291483406638],
                             [60.38888888888889, -97.71291483406638],
                             [60.361111111111114, -97.71291483406638],
                             [60.333333333333336, -97.71291483406638],
                             [60.30555555555556, -97.7608956417433],
                             [60.27777777777778, -97.7608956417433],
                             [60.25, -97.7608956417433],
                             [60.22222222222223, -97.7608956417433],
                             [60.19444444444445, -97.7608956417433],
                             [60.16666666666667, -97.7608956417433],
                             [60.13888888888889, -97.7608956417433],
                             [60.111111111111114, -97.80887644942024],
                             [60.13888888888889, -97.85685725709716],
                             [60.16666666666667, -97.85685725709716],
                             [60.19444444444445, -97.85685725709716],
                             [60.22222222222223, -97.85685725709716],
                             [60.25, -97.85685725709716],
                             [60.27777777777778, -97.85685725709716],
                             [60.30555555555556, -97.85685725709716],
                             [60.333333333333336, -97.85685725709716],
                             [60.361111111111114, -97.85685725709716],
                             [60.38888888888889, -97.85685725709716],
                             [60.41666666666667, -97.85685725709716],
                             [60.44444444444445, -97.85685725709716],
                             [60.47222222222223, -97.85685725709716],
                             [60.5, -97.85685725709716],
                             [60.52777777777778, -97.85685725709716],
                             [60.55555555555556, -97.85685725709716],
                             [60.583333333333336, -97.85685725709716],
                             [60.611111111111114, -97.85685725709716],
                             [60.63888888888889, -97.85685725709716],
                             [60.66666666666667, -97.85685725709716],
                             [60.66666666666667, -97.9048380647741],
                             [60.63888888888889, -97.9048380647741],
                             [60.611111111111114, -97.9048380647741],
                             [60.583333333333336, -97.9048380647741],
                             [60.55555555555556, -97.9048380647741],
                             [60.52777777777778, -97.9048380647741],
                             [60.5, -97.9048380647741],
                             [60.47222222222223, -97.9048380647741],
                             [60.44444444444445, -97.9048380647741],
                             [60.41666666666667, -97.9048380647741],
                             [60.38888888888889, -97.9048380647741],
                             [60.361111111111114, -97.9048380647741],
                             [60.333333333333336, -97.9048380647741],
                             [60.30555555555556, -97.9048380647741],
                             [60.27777777777778, -97.9048380647741],
                             [60.25, -97.9048380647741],
                             [60.22222222222223, -97.9048380647741],
                             [60.19444444444445, -97.9048380647741],
                             [60.16666666666667, -97.9048380647741],
                             [60.13888888888889, -97.95281887245102],
                             [60.16666666666667, -98.00079968012795],
                             [60.19444444444445, -98.00079968012795],
                             [60.22222222222223, -98.00079968012795],
                             [60.25, -98.00079968012795],
                             [60.27777777777778, -98.00079968012795],
                             [60.30555555555556, -98.00079968012795],
                             [60.333333333333336, -98.00079968012795],
                             [60.361111111111114, -98.00079968012795],
                             [60.38888888888889, -98.00079968012795],
                             [60.41666666666667, -98.00079968012795],
                             [60.44444444444445, -98.00079968012795],
                             [60.47222222222223, -98.00079968012795],
                             [60.5, -98.00079968012795],
                             [60.52777777777778, -98.00079968012795],
                             [60.55555555555556, -98.00079968012795],
                             [60.583333333333336, -98.00079968012795],
                             [60.611111111111114, -98.00079968012795],
                             [60.63888888888889, -98.04878048780488],
                             [60.63888888888889, -98.09676129548181],
                             [60.611111111111114, -98.09676129548181],
                             [60.583333333333336, -98.04878048780488],
                             [60.55555555555556, -98.04878048780488],
                             [60.52777777777778, -98.04878048780488],
                             [60.5, -98.04878048780488],
                             [60.47222222222223, -98.04878048780488],
                             [60.44444444444445, -98.04878048780488],
                             [60.41666666666667, -98.04878048780488],
                             [60.38888888888889, -98.04878048780488],
                             [60.361111111111114, -98.04878048780488],
                             [60.333333333333336, -98.04878048780488],
                             [60.30555555555556, -98.04878048780488],
                             [60.27777777777778, -98.04878048780488],
                             [60.25, -98.04878048780488],
                             [60.22222222222223, -98.04878048780488],
                             [60.19444444444445, -98.04878048780488],
                             [60.16666666666667, -98.09676129548181],
                             [60.13888888888889, -98.09676129548181],
                             [60.111111111111114, -98.09676129548181],
                             [60.083333333333336, -98.09676129548181],
                             [60.05555555555556, -98.09676129548181],
                             [60.02777777777778, -98.14474210315873],
                             [60.05555555555556, -98.19272291083567],
                             [60.083333333333336, -98.19272291083567],
                             [60.111111111111114, -98.19272291083567],
                             [60.13888888888889, -98.19272291083567],
                             [60.16666666666667, -98.19272291083567],
                             [60.19444444444445, -98.19272291083567],
                             [60.22222222222223, -98.19272291083567],
                             [60.25, -98.14474210315873],
                             [60.27777777777778, -98.14474210315873],
                             [60.30555555555556, -98.14474210315873],
                             [60.333333333333336, -98.14474210315873],
                             [60.361111111111114, -98.14474210315873],
                             [60.38888888888889, -98.14474210315873],
                             [60.41666666666667, -98.14474210315873],
                             [60.44444444444445, -98.14474210315873],
                             [60.47222222222223, -98.14474210315873],
                             [60.5, -98.14474210315873],
                             [60.52777777777778, -98.19272291083567],
                             [60.55555555555556, -98.19272291083567],
                             [60.583333333333336, -98.19272291083567],
                             [60.611111111111114, -98.19272291083567],
                             [60.63888888888889, -98.19272291083567],
                             [60.63888888888889, -98.24070371851259],
                             [60.611111111111114, -98.24070371851259],
                             [60.583333333333336, -98.24070371851259],
                             [60.55555555555556, -98.24070371851259],
                             [60.52777777777778, -98.24070371851259],
                             [60.5, -98.24070371851259],
                             [60.47222222222223, -98.19272291083567],
                             [60.44444444444445, -98.19272291083567],
                             [60.41666666666667, -98.19272291083567],
                             [60.38888888888889, -98.19272291083567],
                             [60.361111111111114, -98.19272291083567],
                             [60.333333333333336, -98.19272291083567],
                             [60.30555555555556, -98.19272291083567],
                             [60.27777777777778, -98.19272291083567],
                             [60.25, -98.24070371851259],
                             [60.22222222222223, -98.24070371851259],
                             [60.19444444444445, -98.24070371851259],
                             [60.16666666666667, -98.24070371851259],
                             [60.13888888888889, -98.24070371851259],
                             [60.111111111111114, -98.24070371851259],
                             [60.083333333333336, -98.28868452618953],
                             [60.05555555555556, -98.28868452618953],
                             [60.02777777777778, -98.28868452618953],
                             [60.0, -98.28868452618953],
                             [59.97222222222222, -98.33666533386645],
                             [60.0, -98.38464614154339],
                             [60.02777777777778, -98.38464614154339],
                             [60.05555555555556, -98.38464614154339],
                             [60.083333333333336, -98.38464614154339],
                             [60.111111111111114, -98.38464614154339],
                             [60.13888888888889, -98.33666533386645],
                             [60.16666666666667, -98.33666533386645],
                             [60.19444444444445, -98.33666533386645],
                             [60.22222222222223, -98.33666533386645],
                             [60.25, -98.33666533386645],
                             [60.27777777777778, -98.33666533386645],
                             [60.30555555555556, -98.33666533386645],
                             [60.333333333333336, -98.33666533386645],
                             [60.361111111111114, -98.33666533386645],
                             [60.38888888888889, -98.33666533386645],
                             [60.41666666666667, -98.33666533386645],
                             [60.44444444444445, -98.33666533386645],
                             [60.47222222222223, -98.33666533386645],
                             [60.5, -98.33666533386645],
                             [60.52777777777778, -98.33666533386645],
                             [60.55555555555556, -98.33666533386645],
                             [60.583333333333336, -98.38464614154339],
                             [60.583333333333336, -98.43262694922032],
                             [60.55555555555556, -98.43262694922032],
                             [60.52777777777778, -98.43262694922032],
                             [60.5, -98.38464614154339],
                             [60.47222222222223, -98.38464614154339],
                             [60.44444444444445, -98.38464614154339],
                             [60.41666666666667, -98.38464614154339],
                             [60.38888888888889, -98.38464614154339],
                             [60.361111111111114, -98.38464614154339],
                             [60.333333333333336, -98.38464614154339],
                             [60.30555555555556, -98.38464614154339],
                             [60.27777777777778, -98.38464614154339],
                             [60.25, -98.38464614154339],
                             [60.22222222222223, -98.38464614154339],
                             [60.19444444444445, -98.38464614154339],
                             [60.16666666666667, -98.43262694922032],
                             [60.13888888888889, -98.43262694922032],
                             [60.111111111111114, -98.43262694922032],
                             [60.083333333333336, -98.43262694922032],
                             [60.05555555555556, -98.43262694922032],
                             [60.02777777777778, -98.48060775689724],
                             [60.0, -98.48060775689724],
                             [59.97222222222222, -98.48060775689724],
                             [59.94444444444445, -98.52858856457418],
                             [59.91666666666667, -98.52858856457418],
                             [59.88888888888889, -98.52858856457418],
                             [59.861111111111114, -98.5765693722511],
                             [59.88888888888889, -98.62455017992804],
                             [59.91666666666667, -98.62455017992804],
                             [59.94444444444445, -98.62455017992804],
                             [59.97222222222222, -98.62455017992804],
                             [60.0, -98.5765693722511],
                             [60.02777777777778, -98.5765693722511],
                             [60.05555555555556, -98.5765693722511],
                             [60.083333333333336, -98.5765693722511],
                             [60.111111111111114, -98.52858856457418],
                             [60.13888888888889, -98.52858856457418],
                             [60.16666666666667, -98.52858856457418],
                             [60.19444444444445, -98.52858856457418],
                             [60.22222222222223, -98.52858856457418],
                             [60.25, -98.52858856457418],
                             [60.27777777777778, -98.52858856457418],
                             [60.30555555555556, -98.52858856457418],
                             [60.333333333333336, -98.52858856457418],
                             [60.361111111111114, -98.52858856457418],
                             [60.38888888888889, -98.52858856457418],
                             [60.41666666666667, -98.52858856457418],
                             [60.44444444444445, -98.52858856457418],
                             [60.47222222222223, -98.52858856457418],
                             [60.5, -98.52858856457418],
                             [60.52777777777778, -98.52858856457418],
                             [60.55555555555556, -98.52858856457418],
                             [60.55555555555556, -98.5765693722511],
                             [60.52777777777778, -98.5765693722511],
                             [60.5, -98.5765693722511],
                             [60.47222222222223, -98.5765693722511],
                             [60.44444444444445, -98.5765693722511],
                             [60.41666666666667, -98.5765693722511],
                             [60.38888888888889, -98.5765693722511],
                             [60.361111111111114, -98.5765693722511],
                             [60.333333333333336, -98.5765693722511],
                             [60.30555555555556, -98.5765693722511],
                             [60.27777777777778, -98.5765693722511],
                             [60.25, -98.5765693722511],
                             [60.22222222222223, -98.5765693722511],
                             [60.19444444444445, -98.5765693722511],
                             [60.16666666666667, -98.5765693722511],
                             [60.13888888888889, -98.62455017992804],
                             [60.111111111111114, -98.62455017992804],
                             [60.083333333333336, -98.62455017992804],
                             [60.05555555555556, -98.62455017992804],
                             [60.02777777777778, -98.62455017992804],
                             [60.0, -98.67253098760496],
                             [59.97222222222222, -98.67253098760496],
                             [59.94444444444445, -98.67253098760496],
                             [59.91666666666667, -98.7205117952819],
                             [59.88888888888889, -98.7205117952819],
                             [59.861111111111114, -98.7205117952819],
                             [59.833333333333336, -98.76849260295882],
                             [59.80555555555556, -98.76849260295882],
                             [59.77777777777778, -98.81647341063575],
                             [59.75, -98.81647341063575],
                             [59.72222222222222, -98.86445421831267],
                             [59.75, -98.91243502598961],
                             [59.77777777777778, -98.91243502598961],
                             [59.80555555555556, -98.91243502598961],
                             [59.833333333333336, -98.86445421831267],
                             [59.861111111111114, -98.86445421831267],
                             [59.88888888888889, -98.81647341063575],
                             [59.91666666666667, -98.81647341063575],
                             [59.94444444444445, -98.81647341063575],
                             [59.97222222222222, -98.81647341063575],
                             [60.0, -98.76849260295882],
                             [60.02777777777778, -98.76849260295882],
                             [60.05555555555556, -98.76849260295882],
                             [60.083333333333336, -98.76849260295882],
                             [60.111111111111114, -98.7205117952819],
                             [60.13888888888889, -98.7205117952819],
                             [60.16666666666667, -98.7205117952819],
                             [60.19444444444445, -98.7205117952819],
                             [60.22222222222223, -98.7205117952819],
                             [60.25, -98.7205117952819],
                             [60.27777777777778, -98.7205117952819],
                             [60.30555555555556, -98.7205117952819],
                             [60.333333333333336, -98.7205117952819],
                             [60.361111111111114, -98.7205117952819],
                             [60.38888888888889, -98.7205117952819],
                             [60.41666666666667, -98.7205117952819],
                             [60.44444444444445, -98.7205117952819],
                             [60.47222222222223, -98.7205117952819],
                             [60.5, -98.7205117952819],
                             [60.52777777777778, -98.7205117952819],
                             [60.55555555555556, -98.76849260295882],
                             [60.55555555555556, -98.81647341063575],
                             [60.52777777777778, -98.81647341063575],
                             [60.5, -98.76849260295882],
                             [60.47222222222223, -98.76849260295882],
                             [60.44444444444445, -98.76849260295882],
                             [60.41666666666667, -98.76849260295882],
                             [60.38888888888889, -98.76849260295882],
                             [60.361111111111114, -98.76849260295882],
                             [60.333333333333336, -98.76849260295882],
                             [60.30555555555556, -98.76849260295882],
                             [60.27777777777778, -98.76849260295882],
                             [60.25, -98.76849260295882],
                             [60.22222222222223, -98.76849260295882],
                             [60.19444444444445, -98.76849260295882],
                             [60.16666666666667, -98.76849260295882],
                             [60.13888888888889, -98.81647341063575],
                             [60.111111111111114, -98.81647341063575],
                             [60.083333333333336, -98.81647341063575],
                             [60.05555555555556, -98.81647341063575],
                             [60.02777777777778, -98.81647341063575],
                             [60.0, -98.86445421831267],
                             [59.97222222222222, -98.86445421831267],
                             [59.94444444444445, -98.86445421831267],
                             [59.91666666666667, -98.91243502598961],
                             [59.88888888888889, -98.91243502598961],
                             [59.861111111111114, -98.96041583366653],
                             [59.833333333333336, -98.96041583366653],
                             [59.80555555555556, -98.96041583366653],
                             [59.77777777777778, -99.00839664134347],
                             [59.75, -99.00839664134347],
                             [59.72222222222222, -99.05637744902039],
                             [59.69444444444445, -99.05637744902039],
                             [59.66666666666667, -99.10435825669732],
                             [59.63888888888889, -99.15233906437425],
                             [59.611111111111114, -99.15233906437425],
                             [59.583333333333336, -99.20031987205118],
                             [59.55555555555556, -99.24830067972812],
                             [59.583333333333336, -99.29628148740504],
                             [59.611111111111114, -99.29628148740504],
                             [59.63888888888889, -99.24830067972812],
                             [59.66666666666667, -99.24830067972812],
                             [59.69444444444445, -99.20031987205118],
                             [59.72222222222222, -99.15233906437425],
                             [59.75, -99.15233906437425],
                             [59.77777777777778, -99.10435825669732],
                             [59.80555555555556, -99.10435825669732],
                             [59.833333333333336, -99.10435825669732],
                             [59.861111111111114, -99.05637744902039],
                             [59.88888888888889, -99.05637744902039],
                             [59.91666666666667, -99.00839664134347],
                             [59.94444444444445, -99.00839664134347],
                             [59.97222222222222, -99.00839664134347],
                             [60.0, -99.00839664134347],
                             [60.02777777777778, -98.96041583366653],
                             [60.05555555555556, -98.96041583366653],
                             [60.083333333333336, -98.96041583366653],
                             [60.111111111111114, -98.96041583366653],
                             [60.13888888888889, -98.96041583366653],
                             [60.16666666666667, -98.91243502598961],
                             [60.19444444444445, -98.91243502598961],
                             [60.22222222222223, -98.91243502598961],
                             [60.25, -98.91243502598961],
                             [60.27777777777778, -98.91243502598961],
                             [60.30555555555556, -98.91243502598961],
                             [60.333333333333336, -98.91243502598961],
                             [60.361111111111114, -98.91243502598961],
                             [60.38888888888889, -98.91243502598961],
                             [60.41666666666667, -98.91243502598961],
                             [60.44444444444445, -98.91243502598961],
                             [60.47222222222223, -98.96041583366653],
                             [60.5, -98.96041583366653],
                             [60.5, -99.00839664134347],
                             [60.47222222222223, -99.00839664134347],
                             [60.44444444444445, -99.00839664134347],
                             [60.41666666666667, -98.96041583366653],
                             [60.38888888888889, -98.96041583366653],
                             [60.361111111111114, -98.96041583366653],
                             [60.333333333333336, -98.96041583366653],
                             [60.30555555555556, -98.96041583366653],
                             [60.27777777777778, -98.96041583366653],
                             [60.25, -98.96041583366653],
                             [60.22222222222223, -98.96041583366653],
                             [60.19444444444445, -99.00839664134347],
                             [60.16666666666667, -99.00839664134347],
                             [60.13888888888889, -99.00839664134347],
                             [60.111111111111114, -99.00839664134347],
                             [60.083333333333336, -99.00839664134347],
                             [60.05555555555556, -99.05637744902039],
                             [60.02777777777778, -99.05637744902039],
                             [60.0, -99.05637744902039],
                             [59.97222222222222, -99.05637744902039],
                             [59.94444444444445, -99.10435825669732],
                             [59.91666666666667, -99.10435825669732],
                             [59.88888888888889, -99.15233906437425],
                             [59.861111111111114, -99.15233906437425],
                             [59.833333333333336, -99.15233906437425],
                             [59.80555555555556, -99.20031987205118],
                             [59.77777777777778, -99.20031987205118],
                             [59.75, -99.24830067972812],
                             [59.72222222222222, -99.24830067972812],
                             [59.69444444444445, -99.29628148740504],
                             [59.66666666666667, -99.34426229508198],
                             [59.63888888888889, -99.34426229508198],
                             [59.611111111111114, -99.3922431027589],
                             [59.583333333333336, -99.44022391043583],
                             [59.55555555555556, -99.44022391043583],
                             [59.52777777777778, -99.48820471811275],
                             [59.5, -99.53618552578969],
                             [59.47222222222222, -99.58416633346661],
                             [59.44444444444445, -99.63214714114355],
                             [59.41666666666667, -99.68012794882047],
                             [59.38888888888889, -99.7281087564974],
                             [59.361111111111114, -99.77608956417433],
                             [59.333333333333336, -99.82407037185126],
                             [59.30555555555556, -99.87205117952819],
                             [59.27777777777778, -99.92003198720512],
                             [59.25, -99.96801279488204],
                             [59.25, -100.01599360255898],
                             [59.27777777777778, -100.06397441023591],
                             [59.30555555555556, -100.01599360255898],
                             [59.333333333333336, -99.96801279488204],
                             [59.361111111111114, -99.92003198720512],
                             [59.38888888888889, -99.87205117952819],
                             [59.41666666666667, -99.82407037185126],
                             [59.44444444444445, -99.77608956417433],
                             [59.47222222222222, -99.7281087564974],
                             [59.5, -99.68012794882047],
                             [59.52777777777778, -99.63214714114355],
                             [59.55555555555556, -99.58416633346661],
                             [59.583333333333336, -99.58416633346661],
                             [59.611111111111114, -99.53618552578969],
                             [59.63888888888889, -99.48820471811275],
                             [59.66666666666667, -99.48820471811275],
                             [59.69444444444445, -99.44022391043583],
                             [59.72222222222222, -99.3922431027589],
                             [59.75, -99.3922431027589],
                             [59.77777777777778, -99.34426229508198],
                             [59.80555555555556, -99.34426229508198],
                             [59.833333333333336, -99.29628148740504],
                             [59.861111111111114, -99.29628148740504],
                             [59.88888888888889, -99.29628148740504],
                             [59.91666666666667, -99.24830067972812],
                             [59.94444444444445, -99.24830067972812],
                             [59.97222222222222, -99.24830067972812],
                             [60.0, -99.20031987205118],
                             [60.02777777777778, -99.20031987205118],
                             [60.05555555555556, -99.20031987205118],
                             [60.083333333333336, -99.20031987205118],
                             [60.111111111111114, -99.15233906437425],
                             [60.13888888888889, -99.15233906437425],
                             [60.16666666666667, -99.15233906437425],
                             [60.19444444444445, -99.15233906437425],
                             [60.22222222222223, -99.15233906437425],
                             [60.25, -99.15233906437425],
                             [60.27777777777778, -99.15233906437425],
                             [60.30555555555556, -99.15233906437425],
                             [60.333333333333336, -99.15233906437425],
                             [60.361111111111114, -99.15233906437425],
                             [60.38888888888889, -99.15233906437425],
                             [60.41666666666667, -99.15233906437425],
                             [60.44444444444445, -99.15233906437425],
                             [60.47222222222223, -99.20031987205118],
                             [60.47222222222223, -99.24830067972812],
                             [60.44444444444445, -99.24830067972812],
                             [60.41666666666667, -99.20031987205118],
                             [60.38888888888889, -99.20031987205118],
                             [60.361111111111114, -99.20031987205118],
                             [60.333333333333336, -99.20031987205118],
                             [60.30555555555556, -99.20031987205118],
                             [60.27777777777778, -99.20031987205118],
                             [60.25, -99.20031987205118],
                             [60.22222222222223, -99.20031987205118],
                             [60.19444444444445, -99.20031987205118],
                             [60.16666666666667, -99.20031987205118],
                             [60.13888888888889, -99.24830067972812],
                             [60.111111111111114, -99.24830067972812],
                             [60.083333333333336, -99.24830067972812],
                             [60.05555555555556, -99.24830067972812],
                             [60.02777777777778, -99.29628148740504],
                             [60.0, -99.29628148740504],
                             [59.97222222222222, -99.29628148740504],
                             [59.94444444444445, -99.34426229508198],
                             [59.91666666666667, -99.34426229508198],
                             [59.88888888888889, -99.34426229508198],
                             [59.861111111111114, -99.3922431027589],
                             [59.833333333333336, -99.3922431027589],
                             [59.80555555555556, -99.44022391043583],
                             [59.77777777777778, -99.44022391043583],
                             [59.75, -99.48820471811275],
                             [59.72222222222222, -99.48820471811275],
                             [59.69444444444445, -99.53618552578969],
                             [59.66666666666667, -99.58416633346661],
                             [59.63888888888889, -99.63214714114355],
                             [59.611111111111114, -99.63214714114355],
                             [59.583333333333336, -99.68012794882047],
                             [59.55555555555556, -99.7281087564974],
                             [59.52777777777778, -99.77608956417433],
                             [59.5, -99.82407037185126],
                             [59.47222222222222, -99.87205117952819],
                             [59.44444444444445, -99.92003198720512],
                             [59.41666666666667, -99.96801279488204],
                             [59.38888888888889, -100.01599360255898],
                             [59.361111111111114, -100.06397441023591],
                             [59.333333333333336, -100.11195521791284],
                             [59.30555555555556, -100.15993602558976],
                             [59.30555555555556, -100.2079168332667],
                             [59.27777777777778, -100.25589764094363],
                             [59.25, -100.30387844862055],
                             [59.25, -100.35185925629749],
                             [59.22222222222222, -100.39984006397441],
                             [59.19444444444444, -100.44782087165134],
                             [59.19444444444444, -100.49580167932827],
                             [59.16666666666667, -100.5437824870052],
                             [59.16666666666667, -100.59176329468212],
                             [59.13888888888889, -100.63974410235906],
                             [59.13888888888889, -100.68772491003598],
                             [59.111111111111114, -100.73570571771292],
                             [59.111111111111114, -100.78368652538984],
                             [59.111111111111114, -100.83166733306678],
                             [59.083333333333336, -100.87964814074371],
                             [59.083333333333336, -100.92762894842063],
                             [59.05555555555556, -100.97560975609755],
                             [59.05555555555556, -101.02359056377449],
                             [59.05555555555556, -101.07157137145143],
                             [59.05555555555556, -101.11955217912835],
                             [59.02777777777778, -101.16753298680528],
                             [59.02777777777778, -101.2155137944822],
                             [59.02777777777778, -101.26349460215914],
                             [59.02777777777778, -101.31147540983606],
                             [59.0, -101.359456217513],
                             [59.0, -101.40743702518992],
                             [59.0, -101.45541783286686],
                             [59.0, -101.50339864054378],
                             [59.0, -101.55137944822071],
                             [59.0, -101.59936025589764],
                             [58.97222222222222, -101.64734106357457],
                             [58.97222222222222, -101.69532187125151],
                             [58.97222222222222, -101.74330267892843],
                             [58.97222222222222, -101.79128348660535],
                             [58.97222222222222, -101.83926429428229],
                             [58.97222222222222, -101.88724510195922],
                             [58.97222222222222, -101.93522590963615],
                             [58.97222222222222, -101.98320671731308],
                             [58.97222222222222, -102.03118752499],
                             [58.97222222222222, -102.07916833266694],
                             [58.97222222222222, -102.12714914034386],
                             [59.0, -102.1751299480208],
                             [59.0, -102.22311075569772],
                             [59.0, -102.27109156337465],
                             [59.0, -102.31907237105158],
                             [59.0, -102.36705317872851],
                             [59.0, -102.41503398640543],
                             [59.02777777777778, -102.46301479408237],
                             [59.02777777777778, -102.5109956017593],
                             [59.02777777777778, -102.55897640943623],
                             [59.02777777777778, -102.60695721711315],
                             [59.02777777777778, -102.65493802479008],
                             [59.05555555555556, -102.70291883246702],
                             [59.05555555555556, -102.75089964014394],
                             [59.05555555555556, -102.79888044782088],
                             [59.083333333333336, -102.8468612554978],
                             [59.083333333333336, -102.89484206317474],
                             [59.083333333333336, -102.94282287085166],
                             [59.111111111111114, -102.9908036785286],
                             [59.111111111111114, -103.03878448620551],
                             [59.13888888888889, -103.08676529388245],
                             [59.13888888888889, -103.13474610155937],
                             [59.16666666666667, -103.18272690923631],
                             [59.16666666666667, -103.23070771691323],
                             [59.19444444444444, -103.27868852459017],
                             [59.19444444444444, -103.3266693322671],
                             [59.22222222222222, -103.37465013994402],
                             [59.25, -103.42263094762095],
                             [59.25, -103.47061175529788],
                             [59.27777777777778, -103.51859256297482],
                             [59.30555555555556, -103.56657337065174],
                             [59.333333333333336, -103.61455417832867],
                             [59.333333333333336, -103.6625349860056],
                             [59.361111111111114, -103.71051579368253],
                             [59.38888888888889, -103.75849660135945],
                             [59.38888888888889, -103.80647740903639],
                             [59.361111111111114, -103.80647740903639],
                             [59.333333333333336, -103.75849660135945],
                             [59.30555555555556, -103.71051579368253],
                             [59.27777777777778, -103.6625349860056],
                             [59.25, -103.61455417832867],
                             [59.22222222222222, -103.56657337065174],
                             [59.22222222222222, -103.51859256297482],
                             [59.19444444444444, -103.47061175529788],
                             [59.16666666666667, -103.42263094762095],
                             [59.16666666666667, -103.37465013994402],
                             [59.13888888888889, -103.3266693322671],
                             [59.13888888888889, -103.27868852459017],
                             [59.111111111111114, -103.23070771691323],
                             [59.111111111111114, -103.18272690923631],
                             [59.083333333333336, -103.13474610155937],
                             [59.083333333333336, -103.08676529388245],
                             [59.05555555555556, -103.03878448620551],
                             [59.05555555555556, -102.9908036785286],
                             [59.05555555555556, -102.94282287085166],
                             [59.02777777777778, -102.89484206317474],
                             [59.02777777777778, -102.8468612554978],
                             [59.02777777777778, -102.79888044782088],
                             [59.02777777777778, -102.75089964014394],
                             [59.0, -102.70291883246702],
                             [59.0, -102.65493802479008],
                             [59.0, -102.60695721711315],
                             [59.0, -102.55897640943623],
                             [58.97222222222222, -102.5109956017593],
                             [58.97222222222222, -102.46301479408237],
                             [58.97222222222222, -102.41503398640543],
                             [58.97222222222222, -102.36705317872851],
                             [58.97222222222222, -102.31907237105158],
                             [58.97222222222222, -102.27109156337465],
                             [58.94444444444444, -102.22311075569772],
                             [58.94444444444444, -102.1751299480208],
                             [58.94444444444444, -102.12714914034386],
                             [58.94444444444444, -102.07916833266694],
                             [58.94444444444444, -102.03118752499],
                             [58.94444444444444, -101.98320671731308],
                             [58.94444444444444, -101.93522590963615],
                             [58.94444444444444, -101.88724510195922],
                             [58.94444444444444, -101.83926429428229],
                             [58.94444444444444, -101.79128348660535],
                             [58.94444444444444, -101.74330267892843],
                             [58.94444444444444, -101.69532187125151],
                             [58.94444444444444, -101.64734106357457],
                             [58.94444444444444, -101.59936025589764],
                             [58.94444444444444, -101.55137944822071],
                             [58.97222222222222, -101.50339864054378],
                             [58.97222222222222, -101.45541783286686],
                             [58.97222222222222, -101.40743702518992],
                             [58.97222222222222, -101.359456217513],
                             [58.97222222222222, -101.31147540983606],
                             [58.97222222222222, -101.26349460215914],
                             [59.0, -101.2155137944822],
                             [59.0, -101.16753298680528],
                             [59.0, -101.11955217912835],
                             [59.0, -101.07157137145143],
                             [59.02777777777778, -101.02359056377449],
                             [59.02777777777778, -100.97560975609755],
                             [59.02777777777778, -100.92762894842063],
                             [59.05555555555556, -100.87964814074371],
                             [59.05555555555556, -100.83166733306678],
                             [59.05555555555556, -100.78368652538984],
                             [59.083333333333336, -100.73570571771292],
                             [59.083333333333336, -100.68772491003598],
                             [59.083333333333336, -100.63974410235906],
                             [59.111111111111114, -100.59176329468212],
                             [59.111111111111114, -100.5437824870052],
                             [59.13888888888889, -100.49580167932827],
                             [59.13888888888889, -100.44782087165134],
                             [59.16666666666667, -100.39984006397441],
                             [59.16666666666667, -100.35185925629749],
                             [59.13888888888889, -100.30387844862055],
                             [59.111111111111114, -100.35185925629749],
                             [59.083333333333336, -100.39984006397441],
                             [59.083333333333336, -100.44782087165134],
                             [59.05555555555556, -100.49580167932827],
                             [59.05555555555556, -100.5437824870052],
                             [59.02777777777778, -100.59176329468212],
                             [59.02777777777778, -100.63974410235906],
                             [59.02777777777778, -100.68772491003598],
                             [59.0, -100.73570571771292],
                             [59.0, -100.78368652538984],
                             [58.97222222222222, -100.83166733306678],
                             [58.97222222222222, -100.87964814074371],
                             [58.97222222222222, -100.92762894842063],
                             [58.94444444444444, -100.97560975609755],
                             [58.94444444444444, -101.02359056377449],
                             [58.94444444444444, -101.07157137145143],
                             [58.94444444444444, -101.11955217912835],
                             [58.91666666666667, -101.16753298680528],
                             [58.91666666666667, -101.2155137944822],
                             [58.91666666666667, -101.26349460215914],
                             [58.91666666666667, -101.31147540983606],
                             [58.88888888888889, -101.359456217513],
                             [58.88888888888889, -101.40743702518992],
                             [58.88888888888889, -101.45541783286686],
                             [58.88888888888889, -101.50339864054378],
                             [58.88888888888889, -101.55137944822071],
                             [58.88888888888889, -101.59936025589764],
                             [58.88888888888889, -101.64734106357457],
                             [58.88888888888889, -101.69532187125151],
                             [58.861111111111114, -101.74330267892843],
                             [58.861111111111114, -101.79128348660535],
                             [58.861111111111114, -101.83926429428229],
                             [58.861111111111114, -101.88724510195922],
                             [58.861111111111114, -101.93522590963615],
                             [58.861111111111114, -101.98320671731308],
                             [58.861111111111114, -102.03118752499],
                             [58.88888888888889, -102.07916833266694],
                             [58.88888888888889, -102.12714914034386],
                             [58.88888888888889, -102.1751299480208],
                             [58.88888888888889, -102.22311075569772],
                             [58.88888888888889, -102.27109156337465],
                             [58.88888888888889, -102.31907237105158],
                             [58.88888888888889, -102.36705317872851],
                             [58.88888888888889, -102.41503398640543],
                             [58.91666666666667, -102.46301479408237],
                             [58.91666666666667, -102.5109956017593],
                             [58.91666666666667, -102.55897640943623],
                             [58.91666666666667, -102.60695721711315],
                             [58.91666666666667, -102.65493802479008],
                             [58.94444444444444, -102.70291883246702],
                             [58.94444444444444, -102.75089964014394],
                             [58.94444444444444, -102.79888044782088],
                             [58.97222222222222, -102.8468612554978],
                             [58.97222222222222, -102.89484206317474],
                             [58.97222222222222, -102.94282287085166],
                             [58.97222222222222, -102.9908036785286],
                             [59.0, -103.03878448620551],
                             [59.0, -103.08676529388245],
                             [59.02777777777778, -103.13474610155937],
                             [59.02777777777778, -103.18272690923631],
                             [59.05555555555556, -103.23070771691323],
                             [59.05555555555556, -103.27868852459017],
                             [59.083333333333336, -103.3266693322671],
                             [59.083333333333336, -103.37465013994402],
                             [59.111111111111114, -103.42263094762095],
                             [59.111111111111114, -103.47061175529788],
                             [59.13888888888889, -103.51859256297482],
                             [59.16666666666667, -103.56657337065174],
                             [59.16666666666667, -103.61455417832867],
                             [59.19444444444444, -103.6625349860056],
                             [59.22222222222222, -103.71051579368253],
                             [59.22222222222222, -103.75849660135945],
                             [59.25, -103.80647740903639],
                             [59.27777777777778, -103.85445821671331],
                             [59.30555555555556, -103.90243902439025],
                             [59.333333333333336, -103.95041983206717],
                             [59.333333333333336, -103.9984006397441],
                             [59.30555555555556, -103.9984006397441],
                             [59.27777777777778, -103.9984006397441],
                             [59.25, -103.95041983206717],
                             [59.22222222222222, -103.90243902439025],
                             [59.22222222222222, -103.85445821671331],
                             [59.19444444444444, -103.80647740903639],
                             [59.16666666666667, -103.75849660135945],
                             [59.13888888888889, -103.71051579368253],
                             [59.13888888888889, -103.6625349860056],
                             [59.111111111111114, -103.61455417832867],
                             [59.083333333333336, -103.56657337065174],
                             [59.05555555555556, -103.61455417832867],
                             [59.05555555555556, -103.6625349860056],
                             [59.083333333333336, -103.71051579368253],
                             [59.083333333333336, -103.75849660135945],
                             [59.111111111111114, -103.80647740903639],
                             [59.13888888888889, -103.85445821671331],
                             [59.16666666666667, -103.90243902439025],
                             [59.16666666666667, -103.95041983206717],
                             [59.19444444444444, -103.9984006397441],
                             [59.22222222222222, -104.04638144742103],
                             [59.25, -104.09436225509796],
                             [59.27777777777778, -104.14234306277488],
                             [59.27777777777778, -104.19032387045182],
                             [59.25, -104.19032387045182],
                             [59.22222222222222, -104.14234306277488],
                             [59.19444444444444, -104.09436225509796],
                             [59.16666666666667, -104.09436225509796],
                             [59.13888888888889, -104.14234306277488],
                             [59.16666666666667, -104.19032387045182],
                             [59.19444444444444, -104.23830467812874],
                             [59.22222222222222, -104.28628548580568],
                             [59.22222222222222, -104.33426629348261],
                             [59.19444444444444, -104.33426629348261],
                             [59.16666666666667, -104.33426629348261],
                             [59.13888888888889, -104.38224710115954],
                             [59.16666666666667, -104.43022790883647],
                             [59.16666666666667, -104.4782087165134],
                             [59.13888888888889, -104.4782087165134],
                             [59.111111111111114, -104.4782087165134],
                             [59.083333333333336, -104.52618952419033],
                             [59.111111111111114, -104.57417033186725],
                             [59.13888888888889, -104.62215113954419],
                             [59.13888888888889, -104.67013194722111],
                             [59.111111111111114, -104.67013194722111],
                             [59.083333333333336, -104.62215113954419],
                             [59.05555555555556, -104.62215113954419],
                             [59.02777777777778, -104.67013194722111],
                             [59.05555555555556, -104.71811275489804],
                             [59.05555555555556, -104.76609356257497],
                             [59.02777777777778, -104.76609356257497],
                             [59.0, -104.8140743702519],
                             [59.0, -104.86205517792882],
                             [59.02777777777778, -104.91003598560576],
                             [59.02777777777778, -104.95801679328268],
                             [59.0, -104.95801679328268],
                             [58.97222222222222, -105.00599760095962],
                             [58.97222222222222, -105.05397840863654],
                             [58.94444444444444, -105.10195921631347],
                             [58.94444444444444, -105.14994002399041],
                             [58.91666666666667, -105.14994002399041],
                             [58.88888888888889, -105.14994002399041],
                             [58.861111111111114, -105.19792083166733],
                             [58.88888888888889, -105.24590163934427],
                             [58.88888888888889, -105.29388244702119],
                             [58.861111111111114, -105.29388244702119],
                             [58.833333333333336, -105.34186325469813],
                             [58.80555555555556, -105.34186325469813],
                             [58.77777777777778, -105.38984406237505],
                             [58.80555555555556, -105.43782487005198],
                             [58.80555555555556, -105.4858056777289],
                             [58.77777777777778, -105.53378648540584],
                             [58.77777777777778, -105.58176729308276],
                             [58.75, -105.58176729308276],
                             [58.72222222222222, -105.6297481007597],
                             [58.72222222222222, -105.67772890843662],
                             [58.69444444444444, -105.67772890843662],
                             [58.66666666666667, -105.72570971611356],
                             [58.69444444444444, -105.77369052379048],
                             [58.69444444444444, -105.82167133146741],
                             [58.66666666666667, -105.86965213914434],
                             [58.63888888888889, -105.86965213914434],
                             [58.611111111111114, -105.91763294682127],
                             [58.611111111111114, -105.96561375449821],
                             [58.583333333333336, -105.96561375449821],
                             [58.55555555555556, -106.01359456217513],
                             [58.52777777777778, -106.01359456217513],
                             [58.5, -106.06157536985206],
                             [58.52777777777778, -106.10955617752899],
                             [58.52777777777778, -106.15753698520592],
                             [58.5, -106.20551779288284],
                             [58.47222222222222, -106.25349860055978],
                             [58.47222222222222, -106.3014794082367],
                             [58.44444444444444, -106.3014794082367],
                             [58.41666666666667, -106.34946021591364],
                             [58.44444444444444, -106.39744102359056],
                             [58.44444444444444, -106.4454218312675],
                             [58.41666666666667, -106.49340263894442],
                             [58.38888888888889, -106.49340263894442],
                             [58.361111111111114, -106.49340263894442],
                             [58.333333333333336, -106.49340263894442],
                             [58.30555555555556, -106.54138344662135],
                             [58.333333333333336, -106.58936425429827],
                             [58.333333333333336, -106.63734506197521],
                             [58.30555555555556, -106.68532586965213],
                             [58.27777777777778, -106.73330667732907],
                             [58.27777777777778, -106.781287485006],
                             [58.25, -106.781287485006],
                             [58.22222222222222, -106.781287485006],
                             [58.19444444444444, -106.82926829268293],
                             [58.22222222222222, -106.87724910035986],
                             [58.22222222222222, -106.92522990803678],
                             [58.19444444444444, -106.97321071571372],
                             [58.16666666666667, -107.02119152339064],
                             [58.16666666666667, -107.06917233106758],
                             [58.13888888888889, -107.06917233106758],
                             [58.111111111111114, -107.1171531387445],
                             [58.111111111111114, -107.16513394642143],
                             [58.083333333333336, -107.16513394642143],
                             [58.05555555555556, -107.21311475409836],
                             [58.02777777777778, -107.26109556177529],
                             [58.02777777777778, -107.30907636945221],
                             [58.0, -107.35705717712915],
                             [57.97222222222222, -107.40503798480607],
                             [57.94444444444444, -107.45301879248301],
                             [57.94444444444444, -107.50099960015993],
                             [57.91666666666667, -107.50099960015993],
                             [57.88888888888889, -107.54898040783686],
                             [57.861111111111114, -107.5969612155138],
                             [57.861111111111114, -107.64494202319072],
                             [57.833333333333336, -107.69292283086766],
                             [57.80555555555556, -107.69292283086766],
                             [57.77777777777778, -107.74090363854458],
                             [57.77777777777778, -107.78888444622152],
                             [57.75, -107.83686525389844],
                             [57.72222222222222, -107.88484606157537],
                             [57.69444444444444, -107.9328268692523],
                             [57.66666666666667, -107.98080767692923],
                             [57.63888888888889, -108.02878848460615],
                             [57.611111111111114, -108.02878848460615],
                             [57.583333333333336, -108.07676929228309],
                             [57.583333333333336, -108.12475009996001],
                             [57.55555555555556, -108.17273090763695],
                             [57.55555555555556, -108.22071171531387],
                             [57.55555555555556, -108.2686925229908],
                             [57.52777777777778, -108.2686925229908],
                             [57.5, -108.2686925229908],
                             [57.47222222222222, -108.31667333066773],
                             [57.44444444444444, -108.36465413834466],
                             [57.44444444444444, -108.4126349460216],
                             [57.41666666666667, -108.46061575369852],
                             [57.41666666666667, -108.50859656137546],
                             [57.38888888888889, -108.50859656137546],
                             [57.361111111111114, -108.55657736905238],
                             [57.333333333333336, -108.60455817672931],
                             [57.333333333333336, -108.65253898440623],
                             [57.30555555555556, -108.65253898440623],
                             [57.27777777777778, -108.65253898440623],
                             [57.25, -108.70051979208317],
                             [57.22222222222222, -108.70051979208317],
                             [57.19444444444444, -108.74850059976009],
                             [57.16666666666667, -108.79648140743703],
                             [57.16666666666667, -108.84446221511395],
                             [57.13888888888889, -108.89244302279089],
                             [57.13888888888889, -108.94042383046781],
                             [57.111111111111114, -108.94042383046781],
                             [57.083333333333336, -108.98840463814474],
                             [57.083333333333336, -109.03638544582167],
                             [57.083333333333336, -109.0843662534986],
                             [57.05555555555556, -109.0843662534986],
                             [57.02777777777778, -109.0843662534986],
                             [57.0, -109.13234706117552],
                             [57.0, -109.18032786885246],
                             [56.97222222222222, -109.2283086765294],
                             [56.94444444444444, -109.2283086765294],
                             [56.91666666666667, -109.27628948420632],
                             [56.88888888888889, -109.27628948420632],
                             [56.861111111111114, -109.32427029188325],
                             [56.833333333333336, -109.37225109956017],
                             [56.833333333333336, -109.42023190723711],
                             [56.80555555555556, -109.46821271491403],
                             [56.77777777777778, -109.46821271491403],
                             [56.75, -109.51619352259097],
                             [56.75, -109.56417433026789],
                             [56.72222222222222, -109.56417433026789],
                             [56.69444444444444, -109.61215513794482],
                             [56.66666666666667, -109.66013594562175],
                             [56.66666666666667, -109.70811675329868],
                             [56.63888888888889, -109.70811675329868],
                             [56.611111111111114, -109.7560975609756],
                             [56.611111111111114, -109.80407836865254],
                             [56.583333333333336, -109.80407836865254],
                             [56.55555555555556, -109.80407836865254],
                             [56.52777777777778, -109.85205917632946],
                             [56.5, -109.85205917632946],
                             [56.47222222222222, -109.9000399840064],
                             [56.44444444444444, -109.94802079168332],
                             [56.41666666666667, -109.99600159936026],
                             [56.41666666666667, -110.04398240703718],
                             [56.38888888888889, -110.09196321471411],
                             [56.361111111111114, -110.09196321471411],
                             [56.333333333333336, -110.13994402239105],
                             [56.30555555555556, -110.13994402239105],
                             [56.27777777777778, -110.18792483006797],
                             [56.27777777777778, -110.2359056377449],
                             [56.27777777777778, -110.28388644542183],
                             [56.25, -110.33186725309876],
                             [56.22222222222222, -110.33186725309876],
                             [56.19444444444444, -110.28388644542183],
                             [56.16666666666667, -110.33186725309876],
                             [56.16666666666667, -110.37984806077569],
                             [56.138888888888886, -110.37984806077569],
                             [56.111111111111114, -110.42782886845262],
                             [56.111111111111114, -110.47580967612954],
                             [56.083333333333336, -110.47580967612954],
                             [56.05555555555556, -110.47580967612954],
                             [56.02777777777778, -110.52379048380648],
                             [56.0, -110.52379048380648],
                             [55.97222222222222, -110.5717712914834],
                             [56.0, -110.61975209916034],
                             [56.02777777777778, -110.61975209916034],
                             [56.02777777777778, -110.66773290683726],
                             [56.0, -110.66773290683726],
                             [55.97222222222222, -110.7157137145142],
                             [55.94444444444444, -110.7157137145142],
                             [55.91666666666667, -110.66773290683726],
                             [55.888888888888886, -110.7157137145142],
                             [55.861111111111114, -110.76369452219112],
                             [55.861111111111114, -110.81167532986805],
                             [55.861111111111114, -110.85965613754497],
                             [55.833333333333336, -110.85965613754497],
                             [55.80555555555556, -110.85965613754497],
                             [55.77777777777778, -110.90763694522191],
                             [55.75, -110.95561775289885],
                             [55.75, -111.00359856057577],
                             [55.72222222222222, -111.00359856057577],
                             [55.69444444444444, -111.00359856057577],
                             [55.666666666666664, -111.0515793682527],
                             [55.638888888888886, -111.09956017592962],
                             [55.638888888888886, -111.14754098360656],
                             [55.611111111111114, -111.14754098360656],
                             [55.583333333333336, -111.14754098360656],
                             [55.55555555555556, -111.14754098360656],
                             [55.52777777777778, -111.19552179128348],
                             [55.52777777777778, -111.24350259896042],
                             [55.5, -111.24350259896042],
                             [55.47222222222222, -111.24350259896042],
                             [55.44444444444444, -111.29148340663734],
                             [55.44444444444444, -111.33946421431428],
                             [55.416666666666664, -111.33946421431428],
                             [55.416666666666664, -111.29148340663734],
                             [55.388888888888886, -111.24350259896042],
                             [55.388888888888886, -111.19552179128348],
                             [55.361111111111114, -111.14754098360656],
                             [55.333333333333336, -111.09956017592962],
                             [55.30555555555556, -111.0515793682527],
                             [55.30555555555556, -111.00359856057577],
                             [55.27777777777778, -110.95561775289885],
                             [55.25, -110.90763694522191],
                             [55.22222222222222, -110.85965613754497],
                             [55.22222222222222, -110.81167532986805],
                             [55.22222222222222, -110.76369452219112],
                             [55.19444444444444, -110.7157137145142],
                             [55.166666666666664, -110.66773290683726],
                             [55.138888888888886, -110.61975209916034],
                             [55.111111111111114, -110.5717712914834],
                             [55.111111111111114, -110.52379048380648],
                             [55.083333333333336, -110.47580967612954],
                             [55.083333333333336, -110.42782886845262],
                             [55.05555555555556, -110.37984806077569],
                             [55.02777777777778, -110.33186725309876],
                             [55.02777777777778, -110.28388644542183],
                             [55.0, -110.2359056377449],
                             [54.97222222222222, -110.18792483006797],
                             [54.97222222222222, -110.13994402239105],
                             [54.94444444444444, -110.09196321471411],
                             [54.916666666666664, -110.04398240703718],
                             [54.888888888888886, -109.99600159936026],
                             [54.888888888888886, -109.94802079168332],
                             [54.861111111111114, -109.9000399840064],
                             [54.833333333333336, -109.85205917632946],
                             [54.833333333333336, -109.80407836865254],
                             [54.80555555555556, -109.7560975609756],
                             [54.77777777777778, -109.70811675329868],
                             [54.77777777777778, -109.66013594562175],
                             [54.75, -109.61215513794482],
                             [54.72222222222222, -109.56417433026789],
                             [54.69444444444444, -109.51619352259097],
                             [54.69444444444444, -109.46821271491403],
                             [54.666666666666664, -109.42023190723711],
                             [54.638888888888886, -109.37225109956017],
                             [54.638888888888886, -109.32427029188325],
                             [54.611111111111114, -109.27628948420632],
                             [54.583333333333336, -109.2283086765294],
                             [54.55555555555556, -109.18032786885246],
                             [54.55555555555556, -109.13234706117552],
                             [54.52777777777778, -109.0843662534986],
                             [54.52777777777778, -109.03638544582167],
                             [54.5, -108.98840463814474],
                             [54.47222222222222, -108.94042383046781],
                             [54.44444444444444, -108.89244302279089],
                             [54.44444444444444, -108.84446221511395],
                             [54.416666666666664, -108.79648140743703],
                             [54.388888888888886, -108.74850059976009],
                             [54.388888888888886, -108.70051979208317],
                             [54.361111111111114, -108.65253898440623],
                             [54.333333333333336, -108.60455817672931],
                             [54.30555555555556, -108.55657736905238],
                             [54.27777777777778, -108.50859656137546],
                             [54.27777777777778, -108.46061575369852],
                             [54.25, -108.4126349460216],
                             [54.22222222222222, -108.36465413834466],
                             [54.19444444444444, -108.31667333066773],
                             [54.19444444444444, -108.2686925229908],
                             [54.166666666666664, -108.22071171531387],
                             [54.138888888888886, -108.17273090763695],
                             [54.138888888888886, -108.12475009996001],
                             [54.111111111111114, -108.07676929228309],
                             [54.083333333333336, -108.02878848460615],
                             [54.05555555555556, -107.98080767692923],
                             [54.02777777777778, -107.9328268692523],
                             [54.02777777777778, -107.88484606157537],
                             [54.0, -107.83686525389844],
                             [54.0, -107.78888444622152],
                             [53.97222222222222, -107.74090363854458],
                             [53.94444444444444, -107.69292283086766],
                             [53.916666666666664, -107.64494202319072],
                             [53.888888888888886, -107.5969612155138],
                             [53.861111111111114, -107.54898040783686],
                             [53.861111111111114, -107.50099960015993],
                             [53.833333333333336, -107.45301879248301],
                             [53.80555555555556, -107.40503798480607],
                             [53.77777777777778, -107.35705717712915],
                             [53.77777777777778, -107.30907636945221],
                             [53.75, -107.26109556177529],
                             [53.72222222222222, -107.21311475409836],
                             [53.69444444444444, -107.16513394642143],
                             [53.69444444444444, -107.1171531387445],
                             [53.666666666666664, -107.06917233106758],
                             [53.638888888888886, -107.02119152339064],
                             [53.611111111111114, -106.97321071571372],
                             [53.611111111111114, -106.92522990803678],
                             [53.583333333333336, -106.87724910035986],
                             [53.55555555555556, -106.82926829268293],
                             [53.52777777777778, -106.781287485006],
                             [53.52777777777778, -106.73330667732907],
                             [53.5, -106.68532586965213],
                             [53.47222222222222, -106.63734506197521],
                             [53.44444444444444, -106.58936425429827],
                             [53.416666666666664, -106.54138344662135],
                             [53.388888888888886, -106.49340263894442],
                             [53.388888888888886, -106.4454218312675],
                             [53.361111111111114, -106.39744102359056],
                             [53.33333333333333, -106.34946021591364],
                             [53.30555555555556, -106.3014794082367],
                             [53.30555555555556, -106.25349860055978],
                             [53.27777777777778, -106.20551779288284],
                             [53.25, -106.15753698520592],
                             [53.22222222222222, -106.10955617752899],
                             [53.19444444444444, -106.06157536985206],
                             [53.166666666666664, -106.01359456217513],
                             [53.166666666666664, -105.96561375449821],
                             [53.138888888888886, -105.91763294682127],
                             [53.111111111111114, -105.86965213914434],
                             [53.08333333333333, -105.82167133146741],
                             [53.05555555555556, -105.77369052379048],
                             [53.05555555555556, -105.72570971611356],
                             [53.02777777777778, -105.67772890843662],
                             [53.0, -105.6297481007597],
                             [52.97222222222222, -105.58176729308276],
                             [52.94444444444444, -105.53378648540584],
                             [52.916666666666664, -105.4858056777289],
                             [52.888888888888886, -105.43782487005198],
                             [52.888888888888886, -105.38984406237505],
                             [52.861111111111114, -105.34186325469813],
                             [52.83333333333333, -105.29388244702119],
                             [52.80555555555556, -105.24590163934427],
                             [52.77777777777778, -105.19792083166733],
                             [52.77777777777778, -105.14994002399041],
                             [52.75, -105.10195921631347],
                             [52.72222222222222, -105.05397840863654],
                             [52.69444444444444, -105.00599760095962],
                             [52.666666666666664, -104.95801679328268],
                             [52.638888888888886, -104.91003598560576],
                             [52.638888888888886, -104.86205517792882],
                             [52.611111111111114, -104.8140743702519],
                             [52.58333333333333, -104.8140743702519],
                             [52.58333333333333, -104.76609356257497],
                             [52.55555555555556, -104.71811275489804],
                             [52.52777777777778, -104.67013194722111],
                             [52.5, -104.62215113954419],
                             [52.47222222222222, -104.57417033186725],
                             [52.44444444444444, -104.52618952419033],
                             [52.44444444444444, -104.4782087165134],
                             [52.416666666666664, -104.43022790883647],
                             [52.388888888888886, -104.38224710115954],
                             [52.361111111111114, -104.33426629348261],
                             [52.33333333333333, -104.28628548580568],
                             [52.30555555555556, -104.23830467812874],
                             [52.27777777777778, -104.19032387045182],
                             [52.25, -104.14234306277488],
                             [52.25, -104.09436225509796],
                             [52.22222222222222, -104.04638144742103],
                             [52.19444444444444, -103.9984006397441],
                             [52.166666666666664, -103.95041983206717],
                             [52.138888888888886, -103.90243902439025],
                             [52.111111111111114, -103.85445821671331],
                             [52.08333333333333, -103.80647740903639],
                             [52.05555555555556, -103.75849660135945],
                             [52.02777777777778, -103.71051579368253],
                             [52.0, -103.6625349860056],
                             [51.97222222222222, -103.61455417832867],
                             [51.97222222222222, -103.56657337065174],
                             [51.94444444444444, -103.51859256297482],
                             [51.916666666666664, -103.47061175529788],
                             [51.888888888888886, -103.42263094762095],
                             [51.861111111111114, -103.37465013994402],
                             [51.83333333333333, -103.3266693322671],
                             [51.80555555555556, -103.27868852459017],
                             [51.77777777777778, -103.23070771691323],
                             [51.75, -103.18272690923631],
                             [51.72222222222222, -103.13474610155937],
                             [51.72222222222222, -103.08676529388245],
                             [51.69444444444444, -103.03878448620551],
                             [51.666666666666664, -102.9908036785286],
                             [51.638888888888886, -102.94282287085166],
                             [51.611111111111114, -102.89484206317474],
                             [51.58333333333333, -102.8468612554978],
                             [51.55555555555556, -102.79888044782088],
                             [51.52777777777778, -102.75089964014394],
                             [51.5, -102.70291883246702],
                             [51.47222222222222, -102.65493802479008],
                             [51.44444444444444, -102.60695721711315],
                             [51.416666666666664, -102.55897640943623],
                             [51.388888888888886, -102.5109956017593],
                             [51.361111111111114, -102.46301479408237],
                             [51.361111111111114, -102.41503398640543],
                             [51.33333333333333, -102.36705317872851],
                             [51.30555555555556, -102.31907237105158],
                             [51.27777777777778, -102.27109156337465],
                             [51.25, -102.22311075569772],
                             [51.22222222222222, -102.1751299480208],
                             [51.19444444444444, -102.12714914034386],
                             [51.166666666666664, -102.07916833266694],
                             [51.138888888888886, -102.03118752499],
                             [51.11111111111111, -101.98320671731308],
                             [51.08333333333333, -101.93522590963615],
                             [51.05555555555556, -101.88724510195922],
                             [51.02777777777778, -101.83926429428229],
                             [51.0, -101.79128348660535],
                             [50.97222222222222, -101.74330267892843],
                             [50.94444444444444, -101.69532187125151],
                             [50.916666666666664, -101.64734106357457],
                             [50.888888888888886, -101.59936025589764],
                             [50.86111111111111, -101.55137944822071],
                             [50.83333333333333, -101.50339864054378],
                             [50.80555555555556, -101.45541783286686],
                             [50.77777777777778, -101.40743702518992],
                             [50.75, -101.359456217513],
                             [50.72222222222222, -101.31147540983606],
                             [50.69444444444444, -101.26349460215914],
                             [50.666666666666664, -101.2155137944822],
                             [50.638888888888886, -101.16753298680528],
                             [50.61111111111111, -101.11955217912835],
                             [50.58333333333333, -101.07157137145143],
                             [50.55555555555556, -101.02359056377449],
                             [50.52777777777778, -100.97560975609755],
                             [50.5, -100.92762894842063],
                             [50.47222222222222, -100.87964814074371],
                             [50.44444444444444, -100.83166733306678],
                             [50.416666666666664, -100.78368652538984],
                             [50.388888888888886, -100.73570571771292],
                             [50.36111111111111, -100.68772491003598],
                             [50.33333333333333, -100.63974410235906],
                             [50.30555555555556, -100.59176329468212],
                             [50.27777777777778, -100.5437824870052],
                             [50.25, -100.49580167932827],
                             [50.22222222222222, -100.44782087165134],
                             [50.19444444444444, -100.39984006397441],
                             [50.166666666666664, -100.35185925629749],
                             [50.138888888888886, -100.30387844862055],
                             [50.11111111111111, -100.25589764094363],
                             [50.08333333333333, -100.2079168332667],
                             [50.05555555555556, -100.15993602558976],
                             [50.02777777777777, -100.11195521791284],
                             [50.0, -100.06397441023591],
                             [49.97222222222222, -100.06397441023591],
                             [49.94444444444444, -100.01599360255898],
                             [49.916666666666664, -99.96801279488204],
                             [49.888888888888886, -99.92003198720512],
                             [49.86111111111111, -99.87205117952819],
                             [49.83333333333333, -99.82407037185126],
                             [49.80555555555556, -99.77608956417433],
                             [49.77777777777777, -99.7281087564974],
                             [49.75, -99.68012794882047],
                             [49.72222222222222, -99.63214714114355],
                             [49.69444444444444, -99.58416633346661],
                             [49.666666666666664, -99.53618552578969],
                             [49.638888888888886, -99.48820471811275],
                             [49.61111111111111, -99.44022391043583],
                             [49.58333333333333, -99.3922431027589],
                             [49.55555555555556, -99.34426229508198],
                             [49.52777777777777, -99.34426229508198],
                             [49.5, -99.29628148740504],
                             [49.47222222222222, -99.24830067972812],
                             [49.44444444444444, -99.20031987205118],
                             [49.416666666666664, -99.15233906437425],
                             [49.388888888888886, -99.10435825669732],
                             [49.36111111111111, -99.05637744902039],
                             [49.33333333333333, -99.00839664134347],
                             [49.30555555555556, -98.96041583366653],
                             [49.27777777777777, -98.91243502598961],
                             [49.25, -98.86445421831267],
                             [49.22222222222222, -98.81647341063575],
                             [49.19444444444444, -98.81647341063575],
                             [49.166666666666664, -98.76849260295882],
                             [49.138888888888886, -98.7205117952819],
                             [49.11111111111111, -98.67253098760496],
                             [49.08333333333333, -98.62455017992804],
                             [49.05555555555556, -98.5765693722511],
                             [49.02777777777777, -98.52858856457418],
                             [49.0, -98.48060775689724],
                             [48.97222222222222, -98.43262694922032],
                             [48.94444444444444, -98.43262694922032],
                             [48.916666666666664, -98.38464614154339],
                             [48.888888888888886, -98.33666533386645],
                             [48.86111111111111, -98.28868452618953],
                             [48.83333333333333, -98.24070371851259],
                             [48.80555555555556, -98.19272291083567],
                             [48.77777777777777, -98.14474210315873],
                             [48.75, -98.09676129548181],
                             [48.72222222222222, -98.04878048780488],
                             [48.69444444444444, -98.04878048780488],
                             [48.666666666666664, -98.00079968012795],
                             [48.638888888888886, -97.95281887245102],
                             [48.61111111111111, -97.9048380647741],
                             [48.58333333333333, -97.85685725709716],
                             [48.55555555555556, -97.80887644942024],
                             [48.52777777777777, -97.7608956417433],
                             [48.5, -97.7608956417433],
                             [48.47222222222222, -97.71291483406638],
                             [48.44444444444444, -97.66493402638945],
                             [48.416666666666664, -97.61695321871252],
                             [48.388888888888886, -97.56897241103559],
                             [48.36111111111111, -97.52099160335865],
                             [48.33333333333333, -97.47301079568173],
                             [48.30555555555556, -97.47301079568173],
                             [48.27777777777777, -97.4250299880048],
                             [48.25, -97.37704918032787],
                             [48.22222222222222, -97.32906837265094],
                             [48.19444444444444, -97.28108756497402],
                             [48.166666666666664, -97.23310675729708],
                             [48.138888888888886, -97.23310675729708],
                             [48.11111111111111, -97.18512594962016],
                             [48.08333333333333, -97.13714514194322],
                             [48.05555555555556, -97.0891643342663],
                             [48.02777777777777, -97.04118352658936],
                             [48.0, -96.99320271891244],
                             [47.97222222222222, -96.99320271891244],
                             [47.94444444444444, -96.9452219112355],
                             [47.916666666666664, -96.89724110355858],
                             [47.916666666666664, -96.84926029588165],
                             [47.916666666666664, -96.80127948820473],
                             [47.916666666666664, -96.75329868052779],
                             [47.916666666666664, -96.70531787285086],
                             [47.916666666666664, -96.65733706517393],
                             [47.916666666666664, -96.609356257497],
                             [47.94444444444444, -96.56137544982008],
                             [47.94444444444444, -96.51339464214314],
                             [47.94444444444444, -96.46541383446622],
                             [47.97222222222222, -96.41743302678928],
                             [47.97222222222222, -96.36945221911236],
                             [48.0, -96.32147141143543],
                             [48.0, -96.2734906037585],
                             [48.0, -96.22550979608157],
                             [48.02777777777777, -96.17752898840465],
                             [48.02777777777777, -96.12954818072771],
                             [48.05555555555556, -96.08156737305079],
                             [48.05555555555556, -96.03358656537385],
                             [48.08333333333333, -95.98560575769693],
                             [48.08333333333333, -95.93762495002],
                             [48.08333333333333, -95.88964414234306],
                             [48.11111111111111, -95.84166333466614],
                             [48.11111111111111, -95.7936825269892],
                             [48.138888888888886, -95.74570171931228],
                             [48.138888888888886, -95.69772091163534],
                             [48.138888888888886, -95.64974010395842],
                             [48.166666666666664, -95.60175929628149],
                             [48.166666666666664, -95.55377848860456],
                             [48.19444444444444, -95.50579768092763],
                             [48.19444444444444, -95.4578168732507],
                             [48.22222222222222, -95.40983606557377],
                             [48.22222222222222, -95.36185525789685],
                             [48.22222222222222, -95.31387445021991],
                             [48.25, -95.26589364254298],
                             [48.25, -95.21791283486606],
                             [48.27777777777777, -95.16993202718913],
                             [48.27777777777777, -95.1219512195122],
                             [48.30555555555556, -95.07397041183526],
                             [48.30555555555556, -95.02598960415834],
                             [48.30555555555556, -94.97800879648142],
                             [48.33333333333333, -94.93002798880448],
                             [48.33333333333333, -94.88204718112755],
                             [48.36111111111111, -94.83406637345063],
                             [48.36111111111111, -94.78608556577369],
                             [48.388888888888886, -94.73810475809677],
                             [48.388888888888886, -94.69012395041983],
                             [48.388888888888886, -94.64214314274291],
                             [48.416666666666664, -94.59416233506597],
                             [48.416666666666664, -94.54618152738905],
                             [48.44444444444444, -94.49820071971212],
                             [48.44444444444444, -94.4502199120352],
                             [48.47222222222222, -94.40223910435826],
                             [48.47222222222222, -94.35425829668134],
                             [48.47222222222222, -94.3062774890044],
                             [48.5, -94.25829668132747],
                             [48.5, -94.21031587365054],
                             [48.52777777777777, -94.16233506597362],
                             [48.52777777777777, -94.11435425829669],
                             [48.55555555555556, -94.06637345061975],
                             [48.55555555555556, -94.01839264294283],
                             [48.55555555555556, -93.97041183526589],
                             [48.58333333333333, -93.92243102758897],
                             [48.58333333333333, -93.87445021991203],
                             [48.61111111111111, -93.82646941223511],
                             [48.61111111111111, -93.77848860455818],
                             [48.638888888888886, -93.73050779688126],
                             [48.638888888888886, -93.68252698920432],
                             [48.666666666666664, -93.6345461815274],
                             [48.666666666666664, -93.58656537385046],
                             [48.666666666666664, -93.53858456617354],
                             [48.69444444444444, -93.4906037584966],
                             [48.69444444444444, -93.44262295081967],
                             [48.72222222222222, -93.39464214314275],
                             [48.72222222222222, -93.34666133546583],
                             [48.75, -93.29868052778889],
                             [48.75, -93.25069972011195],
                             [48.77777777777777, -93.20271891243503],
                             [48.77777777777777, -93.1547381047581],
                             [48.77777777777777, -93.10675729708117],
                             [48.80555555555556, -93.05877648940424],
                             [48.80555555555556, -93.01079568172732],
                             [48.83333333333333, -92.96281487405038],
                             [48.83333333333333, -92.91483406637346],
                             [48.86111111111111, -92.86685325869652],
                             [48.86111111111111, -92.8188724510196],
                             [48.888888888888886, -92.77089164334267],
                             [48.888888888888886, -92.72291083566574],
                             [48.916666666666664, -92.67493002798881],
                             [48.916666666666664, -92.62694922031187],
                             [48.916666666666664, -92.57896841263495],
                             [48.94444444444444, -92.53098760495803],
                             [48.94444444444444, -92.48300679728109],
                             [48.97222222222222, -92.43502598960416],
                             [48.97222222222222, -92.38704518192723],
                             [49.0, -92.3390643742503],
                             [49.0, -92.29108356657338],
                             [49.02777777777777, -92.24310275889644],
                             [49.02777777777777, -92.19512195121952],
                             [49.05555555555556, -92.14714114354258],
                             [49.05555555555556, -92.09916033586566],
                             [49.05555555555556, -92.05117952818873],
                             [49.08333333333333, -92.0031987205118],
                             [49.08333333333333, -91.95521791283487],
                             [49.11111111111111, -91.90723710515795],
                             [49.138888888888886, -91.90723710515795],
                             [49.166666666666664, -91.90723710515795],
                             [49.19444444444444, -91.90723710515795],
                             [49.22222222222222, -91.90723710515795],
                             [49.25, -91.90723710515795],
                             [49.27777777777777, -91.90723710515795],
                             [49.30555555555556, -91.90723710515795],
                             [49.33333333333333, -91.90723710515795],
                             [49.36111111111111, -91.90723710515795],
                             [49.388888888888886, -91.90723710515795],
                             [49.416666666666664, -91.90723710515795],
                             [49.44444444444444, -91.90723710515795],
                             [49.47222222222222, -91.90723710515795],
                             [49.5, -91.90723710515795],
                             [49.52777777777777, -91.90723710515795],
                             [49.55555555555556, -91.90723710515795],
                             [49.58333333333333, -91.90723710515795],
                             [49.61111111111111, -91.90723710515795],
                             [49.638888888888886, -91.90723710515795],
                             [49.666666666666664, -91.90723710515795],
                             [49.69444444444444, -91.90723710515795],
                             [49.72222222222222, -91.90723710515795],
                             [49.75, -91.90723710515795],
                             [49.77777777777777, -91.90723710515795],
                             [49.80555555555556, -91.90723710515795],
                             [49.83333333333333, -91.90723710515795],
                             [49.86111111111111, -91.90723710515795],
                             [49.888888888888886, -91.90723710515795],
                             [49.916666666666664, -91.90723710515795],
                             [49.94444444444444, -91.90723710515795],
                             [49.97222222222222, -91.90723710515795],
                             [50.0, -91.90723710515795],
                             [50.02777777777777, -91.95521791283487],
                             [50.05555555555556, -91.95521791283487],
                             [50.08333333333333, -91.95521791283487],
                             [50.11111111111111, -91.95521791283487],
                             [50.138888888888886, -91.95521791283487],
                             [50.166666666666664, -91.95521791283487],
                             [50.19444444444444, -91.95521791283487],
                             [50.22222222222222, -91.95521791283487],
                             [50.25, -91.95521791283487],
                             [50.27777777777778, -91.95521791283487],
                             [50.30555555555556, -91.95521791283487],
                             [50.33333333333333, -91.95521791283487],
                             [50.36111111111111, -91.95521791283487],
                             [50.388888888888886, -91.95521791283487],
                             [50.416666666666664, -91.95521791283487],
                             [50.44444444444444, -91.95521791283487],
                             [50.47222222222222, -91.95521791283487],
                             [50.5, -91.95521791283487],
                             [50.52777777777778, -91.95521791283487],
                             [50.55555555555556, -91.95521791283487],
                             [50.58333333333333, -91.95521791283487],
                             [50.61111111111111, -91.95521791283487],
                             [50.638888888888886, -91.95521791283487],
                             [50.666666666666664, -91.95521791283487],
                             [50.69444444444444, -91.95521791283487],
                             [50.72222222222222, -92.0031987205118],
                             [50.75, -92.0031987205118],
                             [50.77777777777778, -92.0031987205118],
                             [50.80555555555556, -92.0031987205118],
                             [50.83333333333333, -92.0031987205118],
                             [50.86111111111111, -92.0031987205118],
                             [50.888888888888886, -92.0031987205118],
                             [50.916666666666664, -92.0031987205118],
                             [50.94444444444444, -92.0031987205118],
                             [50.97222222222222, -92.0031987205118],
                             [51.0, -92.0031987205118],
                             [51.02777777777778, -92.0031987205118],
                             [51.05555555555556, -92.0031987205118],
                             [51.08333333333333, -92.0031987205118],
                             [51.11111111111111, -92.0031987205118],
                             [51.138888888888886, -92.0031987205118],
                             [51.166666666666664, -92.0031987205118],
                             [51.19444444444444, -92.0031987205118],
                             [51.22222222222222, -92.0031987205118],
                             [51.25, -92.0031987205118],
                             [51.27777777777778, -92.0031987205118],
                             [51.30555555555556, -92.0031987205118],
                             [51.33333333333333, -92.0031987205118],
                             [51.361111111111114, -92.0031987205118],
                             [51.388888888888886, -92.0031987205118],
                             [51.416666666666664, -92.05117952818873],
                             [51.44444444444444, -92.05117952818873],
                             [51.47222222222222, -92.05117952818873],
                             [51.5, -92.05117952818873],
                             [51.52777777777778, -92.05117952818873],
                             [51.55555555555556, -92.05117952818873],
                             [51.58333333333333, -92.05117952818873],
                             [51.611111111111114, -92.05117952818873],
                             [51.638888888888886, -92.05117952818873],
                             [51.666666666666664, -92.05117952818873],
                             [51.69444444444444, -92.05117952818873],
                             [51.72222222222222, -92.05117952818873],
                             [51.75, -92.05117952818873],
                             [51.77777777777778, -92.05117952818873],
                             [51.80555555555556, -92.05117952818873],
                             [51.83333333333333, -92.05117952818873],
                             [51.861111111111114, -92.05117952818873],
                             [51.888888888888886, -92.05117952818873],
                             [51.916666666666664, -92.05117952818873],
                             [51.94444444444444, -92.05117952818873],
                             [51.97222222222222, -92.05117952818873],
                             [52.0, -92.05117952818873],
                             [52.02777777777778, -92.05117952818873],
                             [52.05555555555556, -92.05117952818873],
                             [52.08333333333333, -92.05117952818873],
                             [52.111111111111114, -92.09916033586566],
                             [52.138888888888886, -92.09916033586566],
                             [52.166666666666664, -92.09916033586566],
                             [52.19444444444444, -92.09916033586566],
                             [52.22222222222222, -92.09916033586566],
                             [52.25, -92.09916033586566],
                             [52.27777777777778, -92.09916033586566],
                             [52.30555555555556, -92.09916033586566],
                             [52.33333333333333, -92.09916033586566],
                             [52.361111111111114, -92.09916033586566],
                             [52.388888888888886, -92.09916033586566],
                             [52.416666666666664, -92.09916033586566],
                             [52.44444444444444, -92.09916033586566],
                             [52.47222222222222, -92.09916033586566],
                             [52.5, -92.09916033586566],
                             [52.52777777777778, -92.09916033586566],
                             [52.55555555555556, -92.09916033586566],
                             [52.58333333333333, -92.09916033586566],
                             [52.611111111111114, -92.09916033586566],
                             [52.638888888888886, -92.09916033586566],
                             [52.666666666666664, -92.09916033586566],
                             [52.69444444444444, -92.09916033586566],
                             [52.72222222222222, -92.09916033586566],
                             [52.75, -92.09916033586566],
                             [52.77777777777778, -92.09916033586566],
                             [52.80555555555556, -92.14714114354258],
                             [52.83333333333333, -92.14714114354258],
                             [52.861111111111114, -92.14714114354258],
                             [52.888888888888886, -92.14714114354258],
                             [52.916666666666664, -92.14714114354258],
                             [52.94444444444444, -92.14714114354258],
                             [52.97222222222222, -92.14714114354258],
                             [53.0, -92.14714114354258],
                             [53.02777777777778, -92.14714114354258],
                             [53.05555555555556, -92.14714114354258],
                             [53.08333333333333, -92.14714114354258],
                             [53.111111111111114, -92.14714114354258],
                             [53.138888888888886, -92.14714114354258],
                             [53.166666666666664, -92.14714114354258],
                             [53.19444444444444, -92.14714114354258],
                             [53.22222222222222, -92.14714114354258],
                             [53.25, -92.14714114354258],
                             [53.27777777777778, -92.14714114354258],
                             [53.30555555555556, -92.14714114354258],
                             [53.33333333333333, -92.14714114354258],
                             [53.361111111111114, -92.14714114354258],
                             [53.388888888888886, -92.14714114354258],
                             [53.416666666666664, -92.14714114354258],
                             [53.44444444444444, -92.14714114354258],
                             [53.47222222222222, -92.14714114354258],
                             [53.5, -92.19512195121952],
                             [53.52777777777778, -92.19512195121952],
                             [53.55555555555556, -92.19512195121952],
                             [53.583333333333336, -92.19512195121952],
                             [53.611111111111114, -92.19512195121952],
                             [53.638888888888886, -92.19512195121952],
                             [53.666666666666664, -92.19512195121952],
                             [53.69444444444444, -92.19512195121952],
                             [53.72222222222222, -92.19512195121952],
                             [53.75, -92.19512195121952],
                             [53.77777777777778, -92.19512195121952],
                             [53.80555555555556, -92.19512195121952],
                             [53.833333333333336, -92.19512195121952],
                             [53.861111111111114, -92.19512195121952],
                             [53.888888888888886, -92.19512195121952],
                             [53.916666666666664, -92.19512195121952],
                             [53.94444444444444, -92.19512195121952],
                             [53.97222222222222, -92.19512195121952],
                             [54.0, -92.19512195121952],
                             [54.02777777777778, -92.19512195121952],
                             [54.05555555555556, -92.19512195121952],
                             [54.083333333333336, -92.19512195121952],
                             [54.111111111111114, -92.19512195121952],
                             [54.138888888888886, -92.19512195121952],
                             [54.166666666666664, -92.19512195121952],
                             [54.19444444444444, -92.24310275889644],
                             [54.22222222222222, -92.24310275889644],
                             [54.25, -92.24310275889644],
                             [54.27777777777778, -92.24310275889644],
                             [54.30555555555556, -92.24310275889644],
                             [54.333333333333336, -92.24310275889644],
                             [54.361111111111114, -92.24310275889644],
                             [54.388888888888886, -92.24310275889644],
                             [54.416666666666664, -92.24310275889644],
                             [54.44444444444444, -92.24310275889644],
                             [54.47222222222222, -92.24310275889644],
                             [54.5, -92.24310275889644],
                             [54.52777777777778, -92.24310275889644],
                             [54.55555555555556, -92.24310275889644],
                             [54.583333333333336, -92.24310275889644],
                             [54.611111111111114, -92.24310275889644],
                             [54.638888888888886, -92.24310275889644],
                             [54.666666666666664, -92.24310275889644],
                             [54.69444444444444, -92.24310275889644],
                             [54.72222222222222, -92.24310275889644],
                             [54.75, -92.24310275889644],
                             [54.77777777777778, -92.24310275889644],
                             [54.80555555555556, -92.24310275889644],
                             [54.833333333333336, -92.24310275889644],
                             [54.861111111111114, -92.29108356657338],
                             [54.888888888888886, -92.29108356657338],
                             [54.916666666666664, -92.29108356657338],
                             [54.94444444444444, -92.29108356657338],
                             [54.97222222222222, -92.29108356657338],
                             [55.0, -92.29108356657338],
                             [55.02777777777778, -92.29108356657338],
                             [55.05555555555556, -92.29108356657338],
                             [55.083333333333336, -92.29108356657338],
                             [55.111111111111114, -92.29108356657338],
                             [55.138888888888886, -92.29108356657338],
                             [55.166666666666664, -92.29108356657338],
                             [55.19444444444444, -92.29108356657338],
                             [55.22222222222222, -92.29108356657338],
                             [55.25, -92.29108356657338],
                             [55.27777777777778, -92.29108356657338],
                             [55.30555555555556, -92.29108356657338],
                             [55.333333333333336, -92.29108356657338],
                             [55.361111111111114, -92.29108356657338],
                             [55.388888888888886, -92.29108356657338],
                             [55.416666666666664, -92.29108356657338],
                             [55.44444444444444, -92.29108356657338],
                             [55.47222222222222, -92.29108356657338],
                             [55.5, -92.29108356657338],
                             [55.52777777777778, -92.29108356657338],
                             [55.55555555555556, -92.3390643742503],
                             [55.583333333333336, -92.3390643742503],
                             [55.611111111111114, -92.3390643742503],
                             [55.638888888888886, -92.3390643742503],
                             [55.666666666666664, -92.3390643742503],
                             [55.69444444444444, -92.3390643742503],
                             [55.72222222222222, -92.3390643742503],
                             [55.75, -92.3390643742503],
                             [55.77777777777778, -92.3390643742503],
                             [55.80555555555556, -92.3390643742503],
                             [55.833333333333336, -92.3390643742503],
                             [55.861111111111114, -92.3390643742503],
                             [55.888888888888886, -92.3390643742503],
                             [55.91666666666667, -92.3390643742503],
                             [55.94444444444444, -92.3390643742503],
                             [55.97222222222222, -92.3390643742503],
                             [56.0, -92.3390643742503],
                             [56.02777777777778, -92.3390643742503],
                             [56.05555555555556, -92.3390643742503],
                             [56.083333333333336, -92.3390643742503],
                             [56.111111111111114, -92.3390643742503],
                             [56.138888888888886, -92.3390643742503],
                             [56.16666666666667, -92.3390643742503],
                             [56.19444444444444, -92.3390643742503],
                             [56.22222222222222, -92.38704518192723],
                             [56.25, -92.38704518192723],
                             [56.27777777777778, -92.38704518192723],
                             [56.30555555555556, -92.38704518192723],
                             [56.333333333333336, -92.38704518192723],
                             [56.361111111111114, -92.38704518192723],
                             [56.38888888888889, -92.38704518192723],
                             [56.41666666666667, -92.38704518192723],
                             [56.44444444444444, -92.38704518192723],
                             [56.47222222222222, -92.38704518192723],
                             [56.5, -92.38704518192723],
                             [56.52777777777778, -92.38704518192723],
                             [56.55555555555556, -92.38704518192723],
                             [56.583333333333336, -92.38704518192723],
                             [56.611111111111114, -92.38704518192723],
                             [56.63888888888889, -92.38704518192723],
                             [56.66666666666667, -92.38704518192723],
                             [56.69444444444444, -92.38704518192723],
                             [56.72222222222222, -92.38704518192723],
                             [56.75, -92.38704518192723],
                             [56.77777777777778, -92.38704518192723],
                             [56.80555555555556, -92.38704518192723],
                             [56.833333333333336, -92.38704518192723],
                             [56.861111111111114, -92.38704518192723],
                             [56.88888888888889, -92.38704518192723],
                             [56.91666666666667, -92.43502598960416],
                             [56.94444444444444, -92.43502598960416],
                             [56.97222222222222, -92.43502598960416],
                             [57.0, -92.43502598960416],
                             [57.02777777777778, -92.43502598960416],
                             [57.05555555555556, -92.43502598960416],
                             [57.083333333333336, -92.43502598960416],
                             [57.111111111111114, -92.43502598960416],
                             [57.13888888888889, -92.43502598960416],
                             [57.16666666666667, -92.43502598960416],
                             [57.19444444444444, -92.43502598960416],
                             [57.22222222222222, -92.43502598960416],
                             [57.25, -92.43502598960416],
                             [57.27777777777778, -92.43502598960416],
                             [57.30555555555556, -92.43502598960416],
                             [57.333333333333336, -92.43502598960416],
                             [57.361111111111114, -92.43502598960416],
                             [57.38888888888889, -92.43502598960416],
                             [57.41666666666667, -92.43502598960416],
                             [57.44444444444444, -92.43502598960416],
                             [57.47222222222222, -92.43502598960416],
                             [57.5, -92.43502598960416],
                             [57.52777777777778, -92.43502598960416],
                             [57.55555555555556, -92.48300679728109],
                             [57.583333333333336, -92.48300679728109],
                             [57.611111111111114, -92.48300679728109],
                             [57.63888888888889, -92.48300679728109],
                             [57.66666666666667, -92.48300679728109],
                             [57.69444444444444, -92.48300679728109],
                             [57.72222222222222, -92.48300679728109],
                             [57.75, -92.48300679728109],
                             [57.77777777777778, -92.48300679728109],
                             [57.80555555555556, -92.48300679728109],
                             [57.833333333333336, -92.48300679728109],
                             [57.861111111111114, -92.48300679728109],
                             [57.88888888888889, -92.48300679728109],
                             [57.91666666666667, -92.48300679728109],
                             [57.94444444444444, -92.48300679728109],
                             [57.97222222222222, -92.48300679728109],
                             [58.0, -92.48300679728109],
                             [58.02777777777778, -92.48300679728109],
                             [58.05555555555556, -92.48300679728109],
                             [58.083333333333336, -92.48300679728109],
                             [58.111111111111114, -92.48300679728109],
                             [58.13888888888889, -92.48300679728109],
                             [58.16666666666667, -92.48300679728109],
                             [58.19444444444444, -92.48300679728109],
                             [58.22222222222222, -92.53098760495803],
                             [58.25, -92.53098760495803],
                             [58.27777777777778, -92.53098760495803],
                             [58.30555555555556, -92.53098760495803],
                             [58.333333333333336, -92.53098760495803],
                             [58.361111111111114, -92.53098760495803],
                             [58.38888888888889, -92.53098760495803],
                             [58.41666666666667, -92.53098760495803],
                             [58.44444444444444, -92.53098760495803],
                             [58.47222222222222, -92.53098760495803],
                             [58.5, -92.53098760495803],
                             [58.52777777777778, -92.53098760495803],
                             [58.55555555555556, -92.53098760495803],
                             [58.583333333333336, -92.53098760495803],
                             [58.611111111111114, -92.53098760495803],
                             [58.63888888888889, -92.53098760495803],
                             [58.66666666666667, -92.53098760495803],
                             [58.69444444444444, -92.53098760495803],
                             [58.72222222222222, -92.53098760495803],
                             [58.75, -92.53098760495803],
                             [58.77777777777778, -92.53098760495803],
                             [58.80555555555556, -92.53098760495803],
                             [58.833333333333336, -92.53098760495803],
                             [58.861111111111114, -92.53098760495803],
                             [58.88888888888889, -92.57896841263495],
                             [58.91666666666667, -92.57896841263495],
                             [58.94444444444444, -92.57896841263495],
                             [58.97222222222222, -92.57896841263495],
                             [59.0, -92.57896841263495],
                             [59.02777777777778, -92.57896841263495],
                             [59.05555555555556, -92.57896841263495],
                             [59.083333333333336, -92.57896841263495],
                             [59.111111111111114, -92.57896841263495],
                             [59.13888888888889, -92.57896841263495],
                             [59.16666666666667, -92.57896841263495],
                             [59.19444444444444, -92.57896841263495],
                             [59.22222222222222, -92.57896841263495],
                             [59.25, -92.57896841263495],
                             [59.27777777777778, -92.57896841263495],
                             [59.30555555555556, -92.57896841263495],
                             [59.333333333333336, -92.57896841263495],
                             [59.361111111111114, -92.62694922031187],
                             [59.38888888888889, -92.57896841263495],
                             [59.41666666666667, -92.57896841263495],
                             [59.44444444444445, -92.57896841263495],
                             [59.47222222222222, -92.57896841263495],
                             [59.5, -92.62694922031187],
                             [59.52777777777778, -92.62694922031187],
                             [59.55555555555556, -92.62694922031187],
                             [59.583333333333336, -92.62694922031187],
                             [59.611111111111114, -92.62694922031187],
                             [59.63888888888889, -92.62694922031187],
                             [59.66666666666667, -92.62694922031187],
                             [59.69444444444445, -92.62694922031187],
                             [59.72222222222222, -92.62694922031187],
                             [59.75, -92.62694922031187],
                             [59.77777777777778, -92.62694922031187],
                             [59.80555555555556, -92.62694922031187],
                             [59.833333333333336, -92.62694922031187],
                             [59.861111111111114, -92.62694922031187],
                             [59.88888888888889, -92.62694922031187],
                             [59.91666666666667, -92.62694922031187],
                             [59.94444444444445, -92.62694922031187],
                             [59.97222222222222, -92.67493002798881],
                             [60.0, -92.62694922031187],
                             [60.02777777777778, -92.62694922031187],
                             [60.05555555555556, -92.62694922031187],
                             [60.083333333333336, -92.67493002798881],
                             [60.111111111111114, -92.67493002798881],
                             [60.13888888888889, -92.62694922031187],
                             [60.16666666666667, -92.62694922031187],
                             [60.19444444444445, -92.67493002798881],
                             [60.22222222222223, -92.67493002798881],
                             [60.25, -92.67493002798881],
                             [60.27777777777778, -92.67493002798881],
                             [60.30555555555556, -92.67493002798881],
                             [60.333333333333336, -92.67493002798881],
                             [60.361111111111114, -92.67493002798881],
                             [60.38888888888889, -92.67493002798881],
                             [60.41666666666667, -92.67493002798881],
                             [60.44444444444445, -92.72291083566574],
                             [60.47222222222223, -92.72291083566574],
                             [60.5, -92.67493002798881],
                             [60.52777777777778, -92.67493002798881],
                             [60.55555555555556, -92.67493002798881],
                             [60.583333333333336, -92.67493002798881],
                             [60.611111111111114, -92.72291083566574],
                             [60.63888888888889, -92.72291083566574],
                             [60.66666666666667, -92.72291083566574],
                             [60.69444444444445, -92.72291083566574],
                             [60.72222222222223, -92.72291083566574],
                             [60.75, -92.67493002798881],
                             [60.77777777777778, -92.67493002798881],
                             [60.77777777777778, -92.72291083566574],
                             [60.80555555555556, -92.77089164334267],
                             [60.833333333333336, -92.77089164334267],
                             [60.861111111111114, -92.77089164334267],
                             [60.88888888888889, -92.72291083566574],
                             [60.91666666666667, -92.72291083566574],
                             [60.94444444444445, -92.72291083566574],
                             [60.97222222222223, -92.72291083566574],
                             [61.0, -92.72291083566574],
                             [61.0, -92.77089164334267],
                             [61.0, -92.8188724510196],
                             [61.02777777777778, -92.86685325869652],
                             [61.05555555555556, -92.86685325869652]])
        
        outlineReduced50 = convexHull(outlineReduced50)
        outlineReduced100 = convexHull(outlineReduced100)
        
#        from auromat.draw import drawLinePlot, saveFig
#        for o,s in [(outlineFull, 'full'), (outlineReduced100, '100'), (outlineReduced50, '50')]:
#            o = np.append(o, [o[0]], axis=0) 
#            saveFig('outline_' + s + '.svg', drawLinePlot(o[:,1], o[:,0], 'Longitude', 'Latitude', linewidth=0.1))
        
        assert not containsOrCrossesPole(outlineReduced50)
        assert not containsOrCrossesPole(outlineReduced100)
        assert not containsOrCrossesPole(outlineFull)
    
    def testPoleBug2(self):
        outlineFull = [[-50.56250000000001, 113.2087822173735],
                     [-50.56250000000001, 113.1596890767762],
                     [-50.56250000000001, 113.1105959361789],
                     [-50.52083333333334, 113.0615027955816],
                     [-50.52083333333334, 113.0124096549843],
                     [-50.52083333333334, 112.963316514387],
                     [-50.52083333333334, 112.9142233737897],
                     [-50.52083333333334, 112.8651302331924],
                     [-50.52083333333334, 112.8160370925951],
                     [-50.47916666666668, 112.8160370925951],
                     [-50.47916666666668, 112.8651302331924],
                     [-50.43750000000001, 112.9142233737897],
                     [-50.39583333333334, 112.963316514387],
                     [-50.35416666666668, 113.0124096549843],
                     [-50.31250000000001, 113.0615027955816],
                     [-50.27083333333334, 113.1105959361789],
                     [-50.27083333333334, 113.1596890767762],
                     [-50.22916666666668, 113.2087822173735],
                     [-50.18750000000001, 113.2578753579708],
                     [-50.14583333333334, 113.3069684985681],
                     [-50.10416666666667, 113.35606163916539],
                     [-50.06250000000001, 113.4051547797627],
                     [-50.02083333333334, 113.45424792035999],
                     [-50.02083333333334, 113.5033410609573],
                     [-49.97916666666667, 113.55243420155459],
                     [-49.93750000000001, 113.6015273421519],
                     [-49.89583333333334, 113.65062048274919],
                     [-49.85416666666667, 113.6997136233465],
                     [-49.81250000000001, 113.74880676394379],
                     [-49.77083333333334, 113.7978999045411],
                     [-49.77083333333334, 113.84699304513839],
                     [-49.72916666666667, 113.8960861857357],
                     [-49.68750000000001, 113.94517932633299],
                     [-49.64583333333334, 113.9942724669303],
                     [-49.60416666666667, 114.04336560752759],
                     [-49.56250000000001, 114.0924587481249],
                     [-49.52083333333334, 114.14155188872219],
                     [-49.47916666666667, 114.1906450293195],
                     [-49.47916666666667, 114.23973816991679],
                     [-49.43750000000001, 114.2888313105141],
                     [-49.39583333333334, 114.33792445111139],
                     [-49.35416666666667, 114.3870175917087],
                     [-49.31250000000001, 114.43611073230599],
                     [-49.27083333333334, 114.4852038729033],
                     [-49.22916666666667, 114.53429701350059],
                     [-49.18750000000001, 114.5833901540979],
                     [-49.14583333333334, 114.63248329469519],
                     [-49.14583333333334, 114.6815764352925],
                     [-49.10416666666667, 114.73066957588979],
                     [-49.06250000000001, 114.7797627164871],
                     [-49.02083333333334, 114.82885585708439],
                     [-48.97916666666667, 114.8779489976817],
                     [-48.93750000000001, 114.92704213827899],
                     [-48.89583333333334, 114.9761352788763],
                     [-48.85416666666667, 115.02522841947359],
                     [-48.81250000000001, 115.0743215600709],
                     [-48.77083333333334, 115.12341470066819],
                     [-48.72916666666667, 115.1725078412655],
                     [-48.72916666666667, 115.22160098186279],
                     [-48.68750000000001, 115.2706941224601],
                     [-48.64583333333334, 115.31978726305739],
                     [-48.60416666666667, 115.3688804036547],
                     [-48.56250000000001, 115.41797354425199],
                     [-48.52083333333334, 115.46706668484929],
                     [-48.47916666666667, 115.51615982544659],
                     [-48.43750000000001, 115.5652529660439],
                     [-48.395833333333336, 115.6143461066412],
                     [-48.35416666666667, 115.66343924723849],
                     [-48.31250000000001, 115.7125323878358],
                     [-48.270833333333336, 115.76162552843309],
                     [-48.22916666666667, 115.8107186690304],
                     [-48.18750000000001, 115.85981180962769],
                     [-48.145833333333336, 115.908904950225],
                     [-48.10416666666667, 115.95799809082229],
                     [-48.10416666666667, 116.0070912314196],
                     [-48.06250000000001, 116.05618437201689],
                     [-48.020833333333336, 116.1052775126142],
                     [-47.97916666666667, 116.15437065321149],
                     [-47.93750000000001, 116.2034637938088],
                     [-47.895833333333336, 116.25255693440609],
                     [-47.85416666666667, 116.3016500750034],
                     [-47.81250000000001, 116.35074321560069],
                     [-47.770833333333336, 116.399836356198],
                     [-47.72916666666667, 116.44892949679529],
                     [-47.68750000000001, 116.4980226373926],
                     [-47.645833333333336, 116.54711577798989],
                     [-47.60416666666667, 116.5962089185872],
                     [-47.56250000000001, 116.64530205918449],
                     [-47.520833333333336, 116.6943951997818],
                     [-47.47916666666667, 116.74348834037909],
                     [-47.43750000000001, 116.7925814809764],
                     [-47.395833333333336, 116.84167462157369],
                     [-47.35416666666667, 116.890767762171],
                     [-47.31250000000001, 116.93986090276829],
                     [-47.270833333333336, 116.9889540433656],
                     [-47.22916666666667, 117.03804718396289],
                     [-47.18750000000001, 117.0871403245602],
                     [-47.145833333333336, 117.13623346515749],
                     [-47.10416666666667, 117.1853266057548],
                     [-47.06250000000001, 117.23441974635209],
                     [-47.020833333333336, 117.2835128869494],
                     [-46.97916666666667, 117.33260602754669],
                     [-46.93750000000001, 117.381699168144],
                     [-46.895833333333336, 117.43079230874129],
                     [-46.85416666666667, 117.4798854493386],
                     [-46.81250000000001, 117.52897858993589],
                     [-46.770833333333336, 117.57807173053318],
                     [-46.72916666666667, 117.62716487113049],
                     [-46.68750000000001, 117.6762580117278],
                     [-46.645833333333336, 117.72535115232509],
                     [-46.60416666666667, 117.77444429292238],
                     [-46.56250000000001, 117.82353743351969],
                     [-46.520833333333336, 117.87263057411698],
                     [-46.47916666666667, 117.92172371471429],
                     [-46.43750000000001, 117.97081685531158],
                     [-46.395833333333336, 118.01990999590889],
                     [-46.35416666666667, 118.06900313650618],
                     [-46.31250000000001, 118.11809627710349],
                     [-46.270833333333336, 118.16718941770078],
                     [-46.22916666666667, 118.21628255829809],
                     [-46.18750000000001, 118.26537569889538],
                     [-46.145833333333336, 118.31446883949269],
                     [-46.10416666666667, 118.31446883949269],
                     [-46.06250000000001, 118.36356198008998],
                     [-46.020833333333336, 118.41265512068729],
                     [-45.97916666666667, 118.46174826128458],
                     [-45.93750000000001, 118.51084140188189],
                     [-45.895833333333336, 118.55993454247918],
                     [-45.85416666666667, 118.60902768307649],
                     [-45.81250000000001, 118.65812082367378],
                     [-45.770833333333336, 118.70721396427109],
                     [-45.72916666666667, 118.75630710486838],
                     [-45.68750000000001, 118.80540024546569],
                     [-45.645833333333336, 118.85449338606298],
                     [-45.60416666666667, 118.90358652666029],
                     [-45.56250000000001, 118.95267966725758],
                     [-45.520833333333336, 119.00177280785489],
                     [-45.47916666666667, 119.05086594845218],
                     [-45.43750000000001, 119.05086594845218],
                     [-45.395833333333336, 119.09995908904949],
                     [-45.35416666666667, 119.14905222964678],
                     [-45.31250000000001, 119.19814537024409],
                     [-45.270833333333336, 119.24723851084138],
                     [-45.22916666666667, 119.29633165143869],
                     [-45.18750000000001, 119.34542479203598],
                     [-45.145833333333336, 119.39451793263329],
                     [-45.10416666666667, 119.44361107323058],
                     [-45.06250000000001, 119.49270421382789],
                     [-45.020833333333336, 119.54179735442519],
                     [-44.97916666666667, 119.54179735442519],
                     [-44.93750000000001, 119.59089049502249],
                     [-44.895833333333336, 119.63998363561979],
                     [-44.85416666666667, 119.68907677621708],
                     [-44.81250000000001, 119.73816991681439],
                     [-44.770833333333336, 119.78726305741169],
                     [-44.72916666666667, 119.83635619800899],
                     [-44.68750000000001, 119.88544933860628],
                     [-44.645833333333336, 119.93454247920359],
                     [-44.60416666666667, 119.98363561980089],
                     [-44.56250000000001, 119.98363561980089],
                     [-44.520833333333336, 120.03272876039819],
                     [-44.47916666666667, 120.08182190099548],
                     [-44.43750000000001, 120.13091504159279],
                     [-44.395833333333336, 120.18000818219008],
                     [-44.35416666666667, 120.22910132278739],
                     [-44.31250000000001, 120.27819446338468],
                     [-44.270833333333336, 120.32728760398199],
                     [-44.22916666666667, 120.32728760398199],
                     [-44.18750000000001, 120.37638074457928],
                     [-44.145833333333336, 120.42547388517659],
                     [-44.10416666666667, 120.47456702577388],
                     [-44.06250000000001, 120.52366016637119],
                     [-44.020833333333336, 120.57275330696848],
                     [-43.97916666666667, 120.62184644756579],
                     [-43.93750000000001, 120.67093958816308],
                     [-43.895833333333336, 120.67093958816308],
                     [-43.85416666666667, 120.72003272876039],
                     [-43.81250000000001, 120.76912586935768],
                     [-43.770833333333336, 120.81821900995499],
                     [-43.72916666666667, 120.86731215055228],
                     [-43.68750000000001, 120.91640529114959],
                     [-43.645833333333336, 120.96549843174688],
                     [-43.645833333333336, 121.01459157234419],
                     [-43.68750000000001, 121.06368471294148],
                     [-43.72916666666667, 121.11277785353879],
                     [-43.72916666666667, 121.16187099413608],
                     [-43.770833333333336, 121.21096413473339],
                     [-43.81250000000001, 121.21096413473339],
                     [-43.85416666666667, 121.16187099413608],
                     [-43.895833333333336, 121.11277785353879],
                     [-43.93750000000001, 121.06368471294148],
                     [-43.97916666666667, 121.06368471294148],
                     [-44.020833333333336, 121.01459157234419],
                     [-44.06250000000001, 120.96549843174688],
                     [-44.10416666666667, 120.91640529114959],
                     [-44.145833333333336, 120.86731215055228],
                     [-44.18750000000001, 120.81821900995499],
                     [-44.22916666666667, 120.76912586935768],
                     [-44.270833333333336, 120.76912586935768],
                     [-44.31250000000001, 120.72003272876039],
                     [-44.35416666666667, 120.67093958816308],
                     [-44.395833333333336, 120.62184644756579],
                     [-44.43750000000001, 120.57275330696848],
                     [-44.47916666666667, 120.52366016637119],
                     [-44.520833333333336, 120.47456702577388],
                     [-44.56250000000001, 120.42547388517659],
                     [-44.60416666666667, 120.42547388517659],
                     [-44.645833333333336, 120.37638074457928],
                     [-44.68750000000001, 120.32728760398199],
                     [-44.72916666666667, 120.27819446338468],
                     [-44.770833333333336, 120.22910132278739],
                     [-44.81250000000001, 120.18000818219008],
                     [-44.85416666666667, 120.13091504159279],
                     [-44.895833333333336, 120.08182190099548],
                     [-44.93750000000001, 120.03272876039819],
                     [-44.97916666666667, 120.03272876039819],
                     [-45.020833333333336, 119.98363561980089],
                     [-45.06250000000001, 119.93454247920359],
                     [-45.10416666666667, 119.88544933860628],
                     [-45.145833333333336, 119.83635619800899],
                     [-45.18750000000001, 119.78726305741169],
                     [-45.22916666666667, 119.73816991681439],
                     [-45.270833333333336, 119.68907677621708],
                     [-45.31250000000001, 119.63998363561979],
                     [-45.35416666666667, 119.59089049502249],
                     [-45.395833333333336, 119.54179735442519],
                     [-45.43750000000001, 119.54179735442519],
                     [-45.47916666666667, 119.49270421382789],
                     [-45.520833333333336, 119.44361107323058],
                     [-45.56250000000001, 119.39451793263329],
                     [-45.60416666666667, 119.34542479203598],
                     [-45.645833333333336, 119.29633165143869],
                     [-45.68750000000001, 119.24723851084138],
                     [-45.72916666666667, 119.19814537024409],
                     [-45.770833333333336, 119.14905222964678],
                     [-45.81250000000001, 119.09995908904949],
                     [-45.85416666666667, 119.05086594845218],
                     [-45.895833333333336, 119.00177280785489],
                     [-45.93750000000001, 118.95267966725758],
                     [-45.97916666666667, 118.90358652666029],
                     [-46.020833333333336, 118.85449338606298],
                     [-46.06250000000001, 118.85449338606298],
                     [-46.10416666666667, 118.80540024546569],
                     [-46.145833333333336, 118.75630710486838],
                     [-46.18750000000001, 118.70721396427109],
                     [-46.22916666666667, 118.65812082367378],
                     [-46.270833333333336, 118.60902768307649],
                     [-46.31250000000001, 118.55993454247918],
                     [-46.35416666666667, 118.51084140188189],
                     [-46.395833333333336, 118.46174826128458],
                     [-46.43750000000001, 118.41265512068729],
                     [-46.47916666666667, 118.36356198008998],
                     [-46.520833333333336, 118.31446883949269],
                     [-46.56250000000001, 118.26537569889538],
                     [-46.60416666666667, 118.21628255829809],
                     [-46.645833333333336, 118.16718941770078],
                     [-46.68750000000001, 118.11809627710349],
                     [-46.72916666666667, 118.06900313650618],
                     [-46.770833333333336, 118.01990999590889],
                     [-46.81250000000001, 117.97081685531158],
                     [-46.85416666666667, 117.92172371471429],
                     [-46.895833333333336, 117.87263057411698],
                     [-46.93750000000001, 117.82353743351969],
                     [-46.97916666666667, 117.77444429292238],
                     [-47.020833333333336, 117.72535115232509],
                     [-47.06250000000001, 117.6762580117278],
                     [-47.10416666666667, 117.62716487113049],
                     [-47.145833333333336, 117.57807173053318],
                     [-47.18750000000001, 117.52897858993589],
                     [-47.22916666666667, 117.4798854493386],
                     [-47.270833333333336, 117.43079230874129],
                     [-47.31250000000001, 117.381699168144],
                     [-47.35416666666667, 117.33260602754669],
                     [-47.395833333333336, 117.2835128869494],
                     [-47.43750000000001, 117.23441974635209],
                     [-47.47916666666667, 117.1853266057548],
                     [-47.520833333333336, 117.13623346515749],
                     [-47.56250000000001, 117.0871403245602],
                     [-47.60416666666667, 117.03804718396289],
                     [-47.645833333333336, 116.9889540433656],
                     [-47.68750000000001, 116.93986090276829],
                     [-47.72916666666667, 116.890767762171],
                     [-47.770833333333336, 116.84167462157369],
                     [-47.81250000000001, 116.7925814809764],
                     [-47.85416666666667, 116.74348834037909],
                     [-47.895833333333336, 116.6943951997818],
                     [-47.93750000000001, 116.64530205918449],
                     [-47.97916666666667, 116.5962089185872],
                     [-48.020833333333336, 116.54711577798989],
                     [-48.06250000000001, 116.4980226373926],
                     [-48.10416666666667, 116.44892949679529],
                     [-48.145833333333336, 116.399836356198],
                     [-48.18750000000001, 116.35074321560069],
                     [-48.22916666666667, 116.3016500750034],
                     [-48.270833333333336, 116.25255693440609],
                     [-48.31250000000001, 116.2034637938088],
                     [-48.35416666666667, 116.15437065321149],
                     [-48.35416666666667, 116.1052775126142],
                     [-48.395833333333336, 116.05618437201689],
                     [-48.43750000000001, 116.0070912314196],
                     [-48.47916666666667, 115.95799809082229],
                     [-48.52083333333334, 115.908904950225],
                     [-48.56250000000001, 115.85981180962769],
                     [-48.60416666666667, 115.8107186690304],
                     [-48.64583333333334, 115.76162552843309],
                     [-48.68750000000001, 115.7125323878358],
                     [-48.72916666666667, 115.66343924723849],
                     [-48.77083333333334, 115.6143461066412],
                     [-48.81250000000001, 115.5652529660439],
                     [-48.85416666666667, 115.51615982544659],
                     [-48.89583333333334, 115.46706668484929],
                     [-48.89583333333334, 115.41797354425199],
                     [-48.93750000000001, 115.3688804036547],
                     [-48.97916666666667, 115.31978726305739],
                     [-49.02083333333334, 115.2706941224601],
                     [-49.06250000000001, 115.22160098186279],
                     [-49.10416666666667, 115.1725078412655],
                     [-49.14583333333334, 115.12341470066819],
                     [-49.18750000000001, 115.0743215600709],
                     [-49.22916666666667, 115.02522841947359],
                     [-49.27083333333334, 114.9761352788763],
                     [-49.31250000000001, 114.92704213827899],
                     [-49.31250000000001, 114.8779489976817],
                     [-49.35416666666667, 114.82885585708439],
                     [-49.39583333333334, 114.7797627164871],
                     [-49.43750000000001, 114.73066957588979],
                     [-49.47916666666667, 114.6815764352925],
                     [-49.52083333333334, 114.63248329469519],
                     [-49.56250000000001, 114.5833901540979],
                     [-49.60416666666667, 114.53429701350059],
                     [-49.64583333333334, 114.4852038729033],
                     [-49.64583333333334, 114.43611073230599],
                     [-49.68750000000001, 114.3870175917087],
                     [-49.72916666666667, 114.33792445111139],
                     [-49.77083333333334, 114.2888313105141],
                     [-49.81250000000001, 114.23973816991679],
                     [-49.85416666666667, 114.1906450293195],
                     [-49.89583333333334, 114.14155188872219],
                     [-49.89583333333334, 114.0924587481249],
                     [-49.93750000000001, 114.04336560752759],
                     [-49.97916666666667, 113.9942724669303],
                     [-50.02083333333334, 113.94517932633299],
                     [-50.06250000000001, 113.8960861857357],
                     [-50.10416666666667, 113.84699304513839],
                     [-50.14583333333334, 113.7978999045411],
                     [-50.14583333333334, 113.74880676394379],
                     [-50.18750000000001, 113.6997136233465],
                     [-50.22916666666668, 113.65062048274919],
                     [-50.27083333333334, 113.6015273421519],
                     [-50.31250000000001, 113.55243420155459],
                     [-50.35416666666668, 113.5033410609573],
                     [-50.39583333333334, 113.45424792035999],
                     [-50.39583333333334, 113.4051547797627],
                     [-50.43750000000001, 113.35606163916539],
                     [-50.47916666666668, 113.3069684985681],
                     [-50.52083333333334, 113.2578753579708]]
        
        outlineHull = [[-50.47916666666668, 112.8160370925951],
                     [-50.52083333333334, 112.8160370925951],
                     [-50.56250000000001, 113.1105959361789],
                     [-50.56250000000001, 113.1596890767762],
                     [-50.56250000000001, 113.2087822173735],
                     [-50.39583333333334, 113.45424792035999],
                     [-50.14583333333334, 113.7978999045411],
                     [-49.89583333333334, 114.14155188872219],
                     [-49.64583333333334, 114.4852038729033],
                     [-49.31250000000001, 114.92704213827899],
                     [-48.89583333333334, 115.46706668484929],
                     [-48.35416666666667, 116.15437065321149],
                     [-48.31250000000001, 116.2034637938088],
                     [-48.270833333333336, 116.25255693440609],
                     [-48.22916666666667, 116.3016500750034],
                     [-48.18750000000001, 116.35074321560069],
                     [-48.145833333333336, 116.399836356198],
                     [-48.10416666666667, 116.44892949679529],
                     [-48.06250000000001, 116.4980226373926],
                     [-48.020833333333336, 116.54711577798989],
                     [-47.97916666666667, 116.5962089185872],
                     [-47.93750000000001, 116.64530205918449],
                     [-47.895833333333336, 116.6943951997818],
                     [-47.85416666666667, 116.74348834037909],
                     [-47.81250000000001, 116.7925814809764],
                     [-47.770833333333336, 116.84167462157369],
                     [-47.72916666666667, 116.890767762171],
                     [-47.68750000000001, 116.93986090276829],
                     [-47.645833333333336, 116.9889540433656],
                     [-47.60416666666667, 117.03804718396289],
                     [-47.56250000000001, 117.0871403245602],
                     [-47.520833333333336, 117.13623346515749],
                     [-47.47916666666667, 117.1853266057548],
                     [-47.43750000000001, 117.23441974635209],
                     [-47.395833333333336, 117.2835128869494],
                     [-47.35416666666667, 117.33260602754669],
                     [-47.31250000000001, 117.381699168144],
                     [-47.270833333333336, 117.43079230874129],
                     [-47.22916666666667, 117.4798854493386],
                     [-47.18750000000001, 117.52897858993589],
                     [-47.145833333333336, 117.57807173053318],
                     [-47.10416666666667, 117.62716487113049],
                     [-47.06250000000001, 117.6762580117278],
                     [-47.020833333333336, 117.72535115232509],
                     [-46.97916666666667, 117.77444429292238],
                     [-46.93750000000001, 117.82353743351969],
                     [-46.895833333333336, 117.87263057411698],
                     [-46.85416666666667, 117.92172371471429],
                     [-46.81250000000001, 117.97081685531158],
                     [-46.770833333333336, 118.01990999590889],
                     [-46.72916666666667, 118.06900313650618],
                     [-46.68750000000001, 118.11809627710349],
                     [-46.645833333333336, 118.16718941770078],
                     [-46.60416666666667, 118.21628255829809],
                     [-46.56250000000001, 118.26537569889538],
                     [-46.520833333333336, 118.31446883949269],
                     [-46.47916666666667, 118.36356198008998],
                     [-46.43750000000001, 118.41265512068729],
                     [-46.395833333333336, 118.46174826128458],
                     [-46.35416666666667, 118.51084140188189],
                     [-46.31250000000001, 118.55993454247918],
                     [-46.270833333333336, 118.60902768307649],
                     [-46.22916666666667, 118.65812082367378],
                     [-46.18750000000001, 118.70721396427109],
                     [-46.145833333333336, 118.75630710486838],
                     [-46.10416666666667, 118.80540024546569],
                     [-46.06250000000001, 118.85449338606298],
                     [-45.43750000000001, 119.54179735442519],
                     [-44.97916666666667, 120.03272876039819],
                     [-44.60416666666667, 120.42547388517659],
                     [-44.270833333333336, 120.76912586935768],
                     [-43.97916666666667, 121.06368471294148],
                     [-43.81250000000001, 121.21096413473339],
                     [-43.770833333333336, 121.21096413473339],
                     [-43.72916666666667, 121.16187099413608],
                     [-43.645833333333336, 121.01459157234419],
                     [-43.645833333333336, 120.96549843174688]]
        
        outlineHullReduced = [[-50.47916666666668, 112.8160370925951],
                             [-50.56250000000001, 113.1105959361789],
                             [-50.56250000000001, 113.1596890767762],
                             [-50.39583333333334, 113.45424792035999],
                             [-50.14583333333334, 113.7978999045411],
                             [-49.64583333333334, 114.4852038729033],
                             [-49.31250000000001, 114.92704213827899],
                             [-48.35416666666667, 116.15437065321149],
                             [-48.31250000000001, 116.2034637938088],
                             [-48.22916666666667, 116.3016500750034],
                             [-48.145833333333336, 116.399836356198],
                             [-48.10416666666667, 116.44892949679529],
                             [-48.020833333333336, 116.54711577798989],
                             [-47.97916666666667, 116.5962089185872],
                             [-47.895833333333336, 116.6943951997818],
                             [-47.85416666666667, 116.74348834037909],
                             [-47.770833333333336, 116.84167462157369],
                             [-47.72916666666667, 116.890767762171],
                             [-47.645833333333336, 116.9889540433656],
                             [-47.60416666666667, 117.03804718396289],
                             [-47.520833333333336, 117.13623346515749],
                             [-47.43750000000001, 117.23441974635209],
                             [-47.395833333333336, 117.2835128869494],
                             [-47.31250000000001, 117.381699168144],
                             [-47.270833333333336, 117.43079230874129],
                             [-47.18750000000001, 117.52897858993589],
                             [-47.145833333333336, 117.57807173053318],
                             [-47.06250000000001, 117.6762580117278],
                             [-47.020833333333336, 117.72535115232509],
                             [-46.93750000000001, 117.82353743351969],
                             [-46.85416666666667, 117.92172371471429],
                             [-46.81250000000001, 117.97081685531158],
                             [-46.72916666666667, 118.06900313650618],
                             [-46.68750000000001, 118.11809627710349],
                             [-46.60416666666667, 118.21628255829809],
                             [-46.56250000000001, 118.26537569889538],
                             [-46.47916666666667, 118.36356198008998],
                             [-46.43750000000001, 118.41265512068729],
                             [-46.35416666666667, 118.51084140188189],
                             [-46.31250000000001, 118.55993454247918],
                             [-46.22916666666667, 118.65812082367378],
                             [-46.145833333333336, 118.75630710486838],
                             [-46.10416666666667, 118.80540024546569],
                             [-45.43750000000001, 119.54179735442519],
                             [-44.97916666666667, 120.03272876039819],
                             [-44.270833333333336, 120.76912586935768],
                             [-43.97916666666667, 121.06368471294148],
                             [-43.770833333333336, 121.21096413473339],
                             [-43.72916666666667, 121.16187099413608],
                             [-43.645833333333336, 120.96549843174688]]
        #print(','.join(map(str, np.fliplr(np.array(outlineHullReduced)).ravel().tolist())))
        
#         from auromat.draw import drawLinePlot, saveFig
#         for o,s in [(outlineFull, 'full'), (outlineHull, 'hull'), (outlineHullReduced, 'hull_reduced')]:
#             o = np.asarray(o)
#             #o = np.append(o, [o[0]], axis=0) 
#             saveFig('outline_' + s + '.svg', drawLinePlot(o[:,1], o[:,0], 'Longitude', 'Latitude', linewidth=0.1))
        
        assert not containsOrCrossesPole(outlineFull)
        assert not containsOrCrossesPole(outlineHull)
        assert not containsOrCrossesPole(outlineHullReduced)
        