# Copyright 2010 Boris Figovsky <borfig@gmail.com>
#
# This file is part of pybfc.

# pybfc is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.

# pybfc is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.

# You should have received a copy of the GNU General Public License
# along with pybfc.  If not, see <http://www.gnu.org/licenses/>.
"""
Handy wrappers for cached compile() calls

>>> import tempfile
>>> execed = tempfile.NamedTemporaryFile()
>>> execed.write('print 5\\n')
>>> execed.flush()
>>> code = compile_exec_from_filename(execed.name)
>>> code is compile_exec_from_filename(execed.name)
True
>>> exec code
5
>>> evalled = tempfile.NamedTemporaryFile()
>>> evalled.write('6\\n')
>>> evalled.flush()
>>> code = compile_eval_from_filename(evalled.name)
>>> code is compile_eval_from_filename(evalled.name)
True
>>> eval(code)
6

"""

from .cache.func import hashable_non_null_cached

def compile_from_filename(filename, mode):
    with open(filename, 'rb') as f:
        return compile(f.read(), filename, mode)

@hashable_non_null_cached
def compile_exec_from_filename(filename):
    return compile_from_filename(filename, 'exec')

@hashable_non_null_cached
def compile_eval_from_filename(filename):
    return compile_from_filename(filename, 'eval')
