import atomman as am
import numpy as np

def sys_gen(units = 'metal',
            atom_style = 'atomic',
            pbc = (True, True, True),
            ucell = am.System(atoms = am.Atoms(4, prop = {'atype': [1], 
                                                          'pos': [[0.0, 0.0, 0.0],
                                                                  [0.5, 0.5, 0.0],
                                                                  [0.0, 0.5, 0.5],
                                                                  [0.5, 0.0, 0.5]]})),                              
            axes = np.array([[1, 0, 0], [0, 1, 0], [0, 0, 1]]),
            shift = np.array([0.1, 0.1, 0.1]),
            size = np.array([[-3,3], [-3,3], [-3,3]], dtype=np.int)):
    """
    Generates the LAMMPS input command lines associated with having LAMMPS create a system.
    
    Keyword Arguments:
    units -- LAMMPS units option to use. Default is 'metal'.
    atom_style -- LAMMPS atom_style option to use. Default is 'atomic'.
    pbc -- list or tuple of three booleans indicating which directions are periodic. Default is (True, True, True).
    ucell -- a small system (i.e. crystallographic unit cell) from which the returned system is generated from. Default is an fcc unit cell with cell lengths = 1.
    axes -- crystallographic axes to rotate the ucell by.  For a cubic ucell, the dimensions of the rotated system will increase to ensure that all directions can be periodic.  Default is [[1,0,0],[0,1,0],[0,0,1]].
    shift -- box-scaled vector to shift all atoms by. Default is [0.1, 0.1, 0.1]. (The LAMMPS algorithm for generating systems sometimes has issues at boundaries.)
    size -- system multipliers to expand the system by.  Default is [[-3,3], [-3,3], [-3,3]], i.e. 6x6x6 supercell of ucell where Cartesian (0,0,0) is in the center of the returned System's Box.
    """
    size = np.asarray(size)
    shift = np.asarray(shift)
    axes = np.asarray(axes)
    
    boundary = ''
    for i in xrange(3):
        if pbc[i]:
            boundary += 'p '
        else:
            boundary += 'm '
    
    ntypes = ucell.natypes
    pos_basis = ''
    type_basis = ''
    for i in xrange(ucell.natoms):
        pos = ucell.atoms_prop(a_id=i, key='pos', scale = True)
        pos_basis += '        basis %f %f %f' %(pos[0], pos[1], pos[2])
        if i < ucell.natoms - 1:
            pos_basis += ' &\n'
        if ucell.atoms_prop(a_id=i, key='atype') > 1:
            type_basis += ' &\n             basis %i %i'%(i+1,  ucell.atoms_prop(a_id=i, key='atype'))
    
    vects = ucell.box.vects
    
    #Test if box is cubic
    if vects[1][0] == 0.0 and vects[2][0] == 0.0 and vects[2][1] == 0.0:
        region_box = 'region box block %i %i %i %i %i %i' % (size[0,0], size[0,1], size[1,0], size[1,1], size[2,0], size[2,1])
        ortho = True
    else:
        assert np.allclose(axes[0], [1,0,0]) and np.allclose(axes[1], [0,1,0]) and np.allclose(axes[2], [0,0,1]), 'Rotation of non-orthogonal box not suppported'
        ortho = False
        size_xy = vects[1][0] * (size[0,1] - size[0,0]) / ucell.box.a
        size_xz = vects[2][0] * (size[0,1] - size[0,0]) / ucell.box.a
        size_yz = vects[2][1] * (size[1,1] - size[1,0]) / ucell.box.b
        region_box = 'region box prism %i %i %i %i %i %i %f %f %f' % (size[0,0], size[0,1], size[1,0], 
                                                                      size[1,1], size[2,0], size[2,1], 
                                                                      size_xy,   size_xz,   size_yz)
        
    #Adjust crystal spacing for systems to be (nearly) perfectly periodic across boundaries
    spacing = np.zeros(3)
    for i in xrange(3):
        spacing[i] = vects[i][i] * ((axes[i,0]**2+axes[i,1]**2+axes[i,2]**2)**0.5)

    
    newline = '\n'
    script = newline.join(['#Atomic system info generated by AtomMan package',
                           '',
                           'units ' + units,
                           'atom_style ' + atom_style,
                           ''
                           'boundary ' + boundary,
                           '',
                           'lattice custom 1.0 &',
                           '        a1 %.12f %.12f %.12f &'      % (vects[0][0], vects[0][1], vects[0][2]),
                           '        a2 %.12f %.12f %.12f &'      % (vects[1][0], vects[1][1], vects[1][2]),
                           '        a3 %.12f %.12f %.12f &'      % (vects[2][0], vects[2][1], vects[2][2]),
                           '        origin %f %f %f &'           % (shift[0], shift[1], shift[2]),
                           '        spacing %.12f %.12f %.12f &' % (spacing[0], spacing[1], spacing[2]),
                           '        orient x %i %i %i &'         % (axes[0,0], axes[0,1], axes[0,2]),
                           '        orient y %i %i %i &'         % (axes[1,0], axes[1,1], axes[1,2]),
                           '        orient z %i %i %i &'         % (axes[2,0], axes[2,1], axes[2,2]),
                           pos_basis,
                           '',
                           region_box,
                           'create_box %i box' %(ntypes),
                           'create_atoms 1 box' + type_basis]) 
    return script

    
