/*
 * Decompiled with CFR 0.152.
 */
package oracle.pgx.common.util;

import java.io.FileNotFoundException;
import java.io.IOError;
import java.io.PrintStream;
import java.io.UnsupportedEncodingException;
import java.lang.reflect.InvocationHandler;
import java.lang.reflect.Method;
import java.lang.reflect.Proxy;
import java.nio.charset.StandardCharsets;
import java.util.Arrays;
import java.util.concurrent.atomic.AtomicLong;
import oracle.pgx.api.PgxFuture;
import oracle.pgx.common.util.CcTrace;
import oracle.pgx.common.util.PrettyPrint;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public final class InterfaceTracer<T>
implements InvocationHandler {
    private static final String STDERR_TOKEN = ":stderr:";
    private final Printer printer;
    private final boolean printStackTraces;
    private static final AtomicLong GLOBAL_REQUEST_COUNTER = new AtomicLong(0L);
    private final T base;
    private final String name;
    private final AtomicLong requestCounter = new AtomicLong(0L);

    private InterfaceTracer(T base, String outputFile, boolean printStackTraces) {
        this.base = base;
        this.name = base.getClass().getSimpleName();
        this.printStackTraces = printStackTraces;
        this.printer = InterfaceTracer.getPrinter(outputFile);
    }

    private static Printer getPrinter(String outputFile) {
        if (outputFile == null) {
            return new LogPrinter();
        }
        if (outputFile.equals(STDERR_TOKEN)) {
            return new PrintStreamPrinter(System.err);
        }
        return new PrintStreamPrinter(InterfaceTracer.openPrintStream(outputFile));
    }

    private static PrintStream openPrintStream(String cctraceOut) {
        try {
            PrintStream printStream = new PrintStream(cctraceOut, StandardCharsets.UTF_8.name());
            Runtime.getRuntime().addShutdownHook(new Thread(printStream::close));
            return printStream;
        }
        catch (FileNotFoundException | UnsupportedEncodingException e) {
            throw new IOError(e);
        }
    }

    static <I, E extends I> I getTracerFor(E base, Class<I> interfaceType, String outputFile, boolean printStackTraces) {
        InterfaceTracer<E> interfaceTracer = new InterfaceTracer<E>(base, outputFile, printStackTraces);
        Object proxy = Proxy.newProxyInstance(interfaceType.getClassLoader(), new Class[]{interfaceType}, interfaceTracer);
        return interfaceType.cast(proxy);
    }

    @Override
    public Object invoke(Object o, Method method, Object[] objects) throws Throwable {
        RequestInfo requestInfo = this.logCall(method, objects);
        Object result = method.invoke(this.base, objects);
        if (result instanceof PgxFuture) {
            return this.traceFuture((PgxFuture)result, requestInfo);
        }
        return this.traceResult(result, requestInfo);
    }

    private <E> PgxFuture<E> traceFuture(PgxFuture<E> result, RequestInfo requestInfo) {
        return result.thenApply(e -> this.traceResult(e, requestInfo));
    }

    private <E> E traceResult(E e, RequestInfo requestInfo) {
        long responseTime = System.currentTimeMillis();
        long relativeRunningTime = responseTime - requestInfo.startMs;
        this.printer.printResult(responseTime, this.name, requestInfo.globalRequestCounter, requestInfo.requestCounter, PrettyPrint.prettify(e), relativeRunningTime);
        if (this.printStackTraces) {
            this.printer.printStackTrace("RES", this.name, requestInfo.globalRequestCounter, requestInfo.requestCounter);
        }
        return e;
    }

    private RequestInfo logCall(Method method, Object[] objects) {
        long request = this.requestCounter.getAndIncrement();
        long globalRequest = GLOBAL_REQUEST_COUNTER.getAndIncrement();
        long startMs = System.currentTimeMillis();
        this.printer.printRequest(startMs, this.name, globalRequest, request, method.getName(), this.getArgumentsString(objects));
        if (this.printStackTraces) {
            this.printer.printStackTrace("REQ", this.name, globalRequest, request);
        }
        return new RequestInfo(globalRequest, request, startMs);
    }

    private String getArgumentsString(Object[] objects) {
        if (objects == null) {
            return "[]";
        }
        return Arrays.deepToString(objects);
    }

    private static class RequestInfo {
        public final long globalRequestCounter;
        public final long requestCounter;
        public final long startMs;

        public RequestInfo(long globalRequestCounter, long requestCounter, long startMs) {
            this.globalRequestCounter = globalRequestCounter;
            this.requestCounter = requestCounter;
            this.startMs = startMs;
        }
    }

    private static final class LogPrinter
    implements Printer {
        private static final Logger LOG = LoggerFactory.getLogger(CcTrace.class);
        private static final String RES_LOG_STRING = "{} RES[{}${}, {}] = {} ({} ms)";
        private static final String REQ_LOG_STRING = "{} REQ[{}${}, {}] {}{}";
        private static final String STACK_TRACE_STRING = "ST {}[{}${}, {}] {}";

        private LogPrinter() {
        }

        @Override
        public void printResult(long timestamp, String interfaceName, long globalRequestCounter, long localRequestCounter, String result, long executionTime) {
            LOG.debug(RES_LOG_STRING, new Object[]{timestamp, interfaceName, globalRequestCounter, localRequestCounter, result, executionTime});
        }

        @Override
        public void printRequest(long timestamp, String interfaceName, long globalRequestCounter, long localRequestCounter, String methodName, String parameters) {
            LOG.debug(REQ_LOG_STRING, new Object[]{timestamp, interfaceName, globalRequestCounter, localRequestCounter, methodName, parameters});
        }

        @Override
        public void printStackTrace(String prefix, String interfaceName, long globalRequest, long localRequest) {
            StackTraceElement[] stackTrace;
            for (StackTraceElement stackTraceElement : stackTrace = Thread.currentThread().getStackTrace()) {
                LOG.debug(STACK_TRACE_STRING, new Object[]{prefix, interfaceName, globalRequest, localRequest, stackTraceElement.toString()});
            }
        }
    }

    private static final class PrintStreamPrinter
    implements Printer {
        private static final String RES_FORMAT_STRING = "%d RES[%s$%d, %d] = %s (%d ms)\n";
        private static final String REQ_FORMAT_STRING = "%d REQ[%s$%d, %d] %s%s\n";
        private static final String STACK_TRACE_FORMAT_STRING = "ST %s[%s$%d, %d] %s\n";
        private final PrintStream out;

        public PrintStreamPrinter(PrintStream out) {
            this.out = out;
        }

        @Override
        public void printResult(long timestamp, String interfaceName, long globalRequestCounter, long localRequestCounter, String result, long executionTime) {
            this.out.printf(RES_FORMAT_STRING, timestamp, interfaceName, globalRequestCounter, localRequestCounter, result, executionTime);
        }

        @Override
        public void printRequest(long timestamp, String interfaceName, long globalRequestCounter, long localRequestCounter, String methodName, String parameters) {
            this.out.printf(REQ_FORMAT_STRING, timestamp, interfaceName, globalRequestCounter, localRequestCounter, methodName, parameters);
        }

        @Override
        public void printStackTrace(String prefix, String interfaceName, long globalRequest, long localRequest) {
            StackTraceElement[] stackTrace;
            for (StackTraceElement stackTraceElement : stackTrace = Thread.currentThread().getStackTrace()) {
                this.out.printf(STACK_TRACE_FORMAT_STRING, prefix, interfaceName, globalRequest, localRequest, stackTraceElement.toString());
            }
        }
    }

    private static interface Printer {
        public void printResult(long var1, String var3, long var4, long var6, String var8, long var9);

        public void printRequest(long var1, String var3, long var4, long var6, String var8, String var9);

        public void printStackTrace(String var1, String var2, long var3, long var5);
    }
}

