#!/usr/bin/env python
"""Tests for config_lib classes."""


import io
import os

from absl.testing import absltest
from grr_response_core.lib import fingerprint
from grr_response_core.lib import package


class FingerprinterTest(absltest.TestCase):

  def testReasonableInterval(self):
    # Check if the limit on maximum blocksize for processing still holds.
    dummy = io.BytesIO(b'')
    fp = fingerprint.Fingerprinter(dummy)
    big_finger = fingerprint.Finger(None, [fingerprint.Range(0, 1000001)], None)
    fp.fingers.append(big_finger)
    start, stop = fp._GetNextInterval()
    self.assertEqual(0, start)
    self.assertEqual(1000000, stop)

  def testAdjustments(self):
    dummy = io.BytesIO(b'')
    fp = fingerprint.Fingerprinter(dummy)
    big_finger = fingerprint.Finger(None, [fingerprint.Range(10, 20)], None)
    fp.fingers.append(big_finger)

    # The remaining range should not yet be touched...
    fp._AdjustIntervals(9, 10)
    self.assertEqual([fingerprint.Range(10, 20)], fp.fingers[0].ranges)
    # Trying to consume into the range. Blow up.
    self.assertRaises(RuntimeError, fp._AdjustIntervals, 9, 11)
    # We forgot a byte. Blow up.
    self.assertRaises(RuntimeError, fp._AdjustIntervals, 11, 12)
    # Consume a byte
    fp._AdjustIntervals(10, 11)
    self.assertEqual([fingerprint.Range(11, 20)], fp.fingers[0].ranges)
    # Consumed too much. Blow up.
    self.assertRaises(RuntimeError, fp._AdjustIntervals, 11, 21)
    # Consume exactly.
    fp._AdjustIntervals(11, 20)
    self.assertEmpty(fp.fingers[0].ranges)

  class MockHasher(object):

    def __init__(self):
      self.seen = b''

    def update(self, content):  # pylint: disable=g-bad-name
      self.seen += content

  def testHashBlock(self):
    # Does it invoke a hash function?
    dummy = b'12345'
    fp = fingerprint.Fingerprinter(io.BytesIO(dummy))
    big_finger = fingerprint.Finger(None, [fingerprint.Range(0, len(dummy))],
                                    None)
    hasher = self.MockHasher()
    big_finger.hashers = [hasher]
    fp.fingers.append(big_finger)
    # Let's process the block
    fp._HashBlock(dummy, 0, len(dummy))
    self.assertEqual(hasher.seen, dummy)

  def testSampleDataParsedCorrectly(self):
    for fname, expected in self.SAMPLE_LIST.items():
      path = package.ResourcePath(
          'grr-response-test',
          os.path.join('grr_response_test', 'test_data', 'fingerprint', fname))
      with io.open(path, 'rb') as f:
        fp = fingerprint.Fingerprinter(f)
        fp.EvalGeneric()
        fp.EvalPecoff()
        result = fp.HashIt()

        self.assertCountEqual(result, expected,
                              'Hashing results for %s do not match.' % fname)

  # pyformat: disable
  SAMPLE_DATA_1 = [{
      'sha256':
          b"\xf2\xa7\xccd[\x96\x94l\xc6[\xf6\x0e\x14\xe7\r\xc0\x9c\x84\x8d'\xc7\x94<\xe5\xde\xa0\xc0\x1ak\x864\x80",
      'sha512':
          b"-\x04\xe5*]N\xaa\xab\x96\xbefQ\xb7\xa0d\x13\xe04w\x84\x0fg^\xe7>$7a\xb2\x19\x19\xe5&\x8ew+\x99\xe2\xfc\xb5\xb9`\xd7\x89\xc5>\xbc\x1b\xff\x83`\x1d''2\xd4KaeS\xe1tj\x0f",
      'name':
          'generic',
      'md5':
          b'\xb5\xb8\xb5\xef.\\\xb3M\xf8\xdc\xf8\x83\x1e54\xfa',
      'sha1':
          b'\x98\xcc;\x8f81\x1a\xbf\xdfW{\xe4\x03\xff\x9ft\xaaU\xf7E'
  }, {
      'SignedData': [(
          512, 2,
          b'0\x82\x1c4\x06\t*\x86H\x86\xf7\r\x01\x07\x02\xa0\x82\x1c%0\x82\x1c!\x02\x01\x011\x0b0\t\x06\x05+\x0e\x03\x02\x1a\x05\x000h\x06\n+\x06\x01\x04\x01\x827\x02\x01\x04\xa0Z0X03\x06\n+\x06\x01\x04\x01\x827\x02\x01\x0f0%\x03\x01\x00\xa0 \xa2\x1e\x80\x1c\x00<\x00<\x00<\x00O\x00b\x00s\x00o\x00l\x00e\x00t\x00e\x00>\x00>\x00>0!0\t\x06\x05+\x0e\x03\x02\x1a\x05\x00\x04\x14\x9b\xd4D\xd5\x8bY\xcc\xa82\xbb_\xc9\x11\xf8\x1flf\xb4\x0f\xcc\xa0\x82\x17\t0\x82\x04\xbb0\x82\x03\xa3\xa0\x03\x02\x01\x02\x02\na\x04\xcai\x00\x00\x00\x00\x00\x080\r\x06\t*\x86H\x86\xf7\r\x01\x01\x05\x05\x000w1\x0b0\t\x06\x03U\x04\x06\x13\x02US1\x130\x11\x06\x03U\x04\x08\x13\nWashington1\x100\x0e\x06\x03U\x04\x07\x13\x07Redmond1\x1e0\x1c\x06\x03U\x04\n\x13\x15Microsoft Corporation1!0\x1f\x06\x03U\x04\x03\x13\x18Microsoft Time-Stamp PCA0\x1e\x17\r070605220321Z\x17\r120605221321Z0\x81\xa41\x0b0\t\x06\x03U\x04\x06\x13\x02US1\x130\x11\x06\x03U\x04\x08\x13\nWashington1\x100\x0e\x06\x03U\x04\x07\x13\x07Redmond1\x1e0\x1c\x06\x03U\x04\n\x13\x15Microsoft Corporation1\'0%\x06\x03U\x04\x0b\x13\x1enCipher DSE ESN:A5B0-CDE0-DC941%0#\x06\x03U\x04\x03\x13\x1cMicrosoft Time-Stamp Service0\x82\x01"0\r\x06\t*\x86H\x86\xf7\r\x01\x01\x01\x05\x00\x03\x82\x01\x0f\x000\x82\x01\n\x02\x82\x01\x01\x00\xaf\xf8Z\xf64\xb1\xa2\x0f\x8e\xac\xcf\xfc\x18\xf0\xad%\rq\xbb_[\xb7;7A\x13$\x1c\xea#\xc5\xae\xac\x99\xc0\xdc\xa6#\xf0\xca\xa9302j/\x9a!B\x02\t&Y\x9bP\xe9Wq\x8bg\xeb\x97\xe8\x04s\x8bx\xb1\x98@F\xa3Z\x01(\x00\x16\xa0\xda\xc5N\x8b\xa9\tT\x911\x9c\xff\xc4\x90\x95E\x07\x1d\x80(\x05\x06q\x80\xcb\xac8ga\xec\xc2_\x16\x9d\x7f\xb0?\xd1*\xf72os=|n\x986\x9a\x86\xee\x008\xee\x17\xecOU3F\x81z@\xdc\xa5\xf5\xd2#\x86]+3O$H\x93\x1a\xb1\xb4~\xcf\xdf3\x1b\xbf\xd5[\xc6/\x13\xf3\xe6\x84\xb7N\x93\x98\xbc\x00\xeb\\\x1di\xc5\xc8\xc8\xd2\xc4<\xb0\x00\x14\xe3\xb2\x9c\xf9\xa1\x1e\t\xdc\xd5\xf5m|213\xb2\x8fY\x9b\xfe\xca\x1f]"N\xba\x12\x1d\xf0g\xc3\x04\x89K\x7f\x03\xfd}\xd7T\x81\x03]I\xf5\x1e\xab ~\xd2e\xb4q\x19\xe8L}\xc6\x94\xbd\x85\x91mPZ\xc3\xa9\x02\x03\x01\x00\x01\xa3\x82\x01\x190\x82\x01\x150\x1d\x06\x03U\x1d\x0e\x04\x16\x04\x14\x9e\x8f\xa8t\xc1B\xee\x18\xcd=\xbd\xe9\xd6\xee\'\xb0\xc754g0\x1f\x06\x03U\x1d#\x04\x180\x16\x80\x14#4\xf8\xd9RFp\n\xed@\xfbv\xfb\xb3+\xb0\xc35\xb3\x0f0T\x06\x03U\x1d\x1f\x04M0K0I\xa0G\xa0E\x86Chttp://crl.microsoft.com/pki/crl/products/MicrosoftTimeStampPCA.crl0X\x06\x08+\x06\x01\x05\x05\x07\x01\x01\x04L0J0H\x06\x08+\x06\x01\x05\x05\x070\x02\x86<http://www.microsoft.com/pki/certs/MicrosoftTimeStampPCA.crt0\x13\x06\x03U\x1d%\x04\x0c0\n\x06\x08+\x06\x01\x05\x05\x07\x03\x080\x0e\x06\x03U\x1d\x0f\x01\x01\xff\x04\x04\x03\x02\x06\xc00\r\x06\t*\x86H\x86\xf7\r\x01\x01\x05\x05\x00\x03\x82\x01\x01\x00g\xa5\xdf(Q\xdcH\x83\xfd\xc6e\xf9]\xfaf\xaa\xb2\x8d\xa7V\x9d\xa2H\xbe2\xacm\x97\x0fO\x0cD\xbb\xbb\x18\xf4\x90\xe7\xe2}\xc5t\x00\xd6\x86\xf3\x91\xc0\xd0\xd1\x05\x89^b\x8b\xfah\xf7e\xcf\xe6S\xa9\xdft\x12!\xad\x8c\x15\xb4\xa6\xcaC\x93\xc9k\xe1J\xb2\xed\x0f\xce\xb4\x8a:\x7f\xf7\xb5\xcc\xbc\x12\x9a{\xcf\x04\xc4?\xd9\n,u<\x92\x9e\xcf\x066b\x0e\x02\x17\xe6\x99\'\xf7\xfbq\xef\xffZzaU\xba\xba\x98\x87\xebq\xacv\xf5\xe8\x86\x9f6gPAx\xfc3\xa4\xd5is\x01\xc8p#k|\xa3W\xc8h\x07\xe4[A\xc1\xf8&\xfdiI\x03\x99\xd6\xaf\xc9?\xa4\xf5\x9cL\xf9\n\x9d\xd2\xcdK\xbf}\xfa\x16\x90\xec;qI\xfe\x04\x15\xca\xe2j\xdai\x0b\xb4\x93>4\xfe\xca\xec\x80I\xe4\xb2t\x18\xcc\x91\xe7>N\xb7{\xf5@\x8fd[\xd4P\xd59MS\xcd\xbe\xd4^G\x82[\xd1Zz\x99.Fpk\x07\x85\rl\t\x13c\x0fu0\x82\x05\x960\x82\x04~\xa0\x03\x02\x01\x02\x02\na\x01\xc6\xc1\x00\x00\x00\x00\x00\x070\r\x06\t*\x86H\x86\xf7\r\x01\x01\x05\x05\x000\x81\x811\x0b0\t\x06\x03U\x04\x06\x13\x02US1\x130\x11\x06\x03U\x04\x08\x13\nWashington1\x100\x0e\x06\x03U\x04\x07\x13\x07Redmond1\x1e0\x1c\x06\x03U\x04\n\x13\x15Microsoft Corporation1+0)\x06\x03U\x04\x03\x13"Microsoft Windows Verification PCA0\x1e\x17\r081022203922Z\x17\r100122204922Z0\x7f1\x0b0\t\x06\x03U\x04\x06\x13\x02US1\x130\x11\x06\x03U\x04\x08\x13\nWashington1\x100\x0e\x06\x03U\x04\x07\x13\x07Redmond1\x1e0\x1c\x06\x03U\x04\n\x13\x15Microsoft Corporation1\r0\x0b\x06\x03U\x04\x0b\x13\x04MOPR1\x1a0\x18\x06\x03U\x04\x03\x13\x11Microsoft Windows0\x82\x01"0\r\x06\t*\x86H\x86\xf7\r\x01\x01\x01\x05\x00\x03\x82\x01\x0f\x000\x82\x01\n\x02\x82\x01\x01\x00\xdc:\xd3D\xf4n \x9f\xdd\xa4\x0e\x82N\xc7\x86^c\xcc\xca\xe5BSK\x85\xfa]ql\xcfv\x0c\x18\x8b\xa6\r\xf5J\xf7\xfe\x17\xf2\x90\xccb\xc7$\xad\x9b\x9a\xe1E;a\xd8\r\x05i\xc7\xcd\x88*\xb8\xb9\x18\x1e`\x10_\x88\xc6\xd2\x82NmI\xc5\xbe\\\x12\x86H\x85\x89\x91\x81\xcd\x1b\xad\x1f\xb7-gy\xf1{\x9f%\x87\x14v_\xe3\x0ed\xa1ra%\xe5ui\xc5\x14\xf1_\x07V\xa4\rp\x06#\xa7l\xdd\x82\xae\xd9\x9bG\xa4\xa5l\x08\xb0X\xf1SjO\xda\x85a\xcb\x02{I\xaf\x1f\xbb\xe0\xd7\xb9^\xdbs\x89v\xc1:\xbb\r\xf5\x97\xf0\x88]iw\x80\xcf\xf1~\x03\x9fsm\xde\x05\xb8/w\xb5TUE\xd0\xd28\xbd\x96\xe3\xf7\xea@\xe5\xac\x19\xfcq\xcb(\'\xaaq\xa1r\xb5\x12\'\xc1Q\xf66\xc5\xc0\xc7{::\x937\x04\xcc\xee\x0bixduA\xb6x"\x0fw\x84\xf7K\x8dFe\x92[MVku\x04F?\x0b\x1b\xb4\x19\xbf\x02\x03\x01\x00\x01\xa3\x82\x02\x0f0\x82\x02\x0b0\x1f\x06\x03U\x1d%\x04\x180\x16\x06\x08+\x06\x01\x05\x05\x07\x03\x03\x06\n+\x06\x01\x04\x01\x827\n\x03\x060\x1d\x06\x03U\x1d\x0e\x04\x16\x04\x14\xa1\xe6\xc3e\xd0\xe6\xe8(b\xc2\xf3\xc2#\xa6\x1cI\x82\x0b\xd5S0\x0e\x06\x03U\x1d\x0f\x01\x01\xff\x04\x04\x03\x02\x07\x800\x1f\x06\x03U\x1d#\x04\x180\x16\x80\x14\x90\x8b\x11\xa5p\xed\xe0\xf9\xa9\xc0\xac\x08\xc7\xb5\xf4\x82\xb1<\xc5J0{\x06\x03U\x1d\x1f\x04t0r0p\xa0n\xa0l\x864http://crl.microsoft.com/pki/crl/products/WinPCA.crl\x864http://www.microsoft.com/pki/crl/products/WinPCA.crl0R\x06\x08+\x06\x01\x05\x05\x07\x01\x01\x04F0D0B\x06\x08+\x06\x01\x05\x05\x070\x02\x866http://www.microsoft.com/pki/certs/MicrosoftWinPCA.crt0\x81\xc6\x06\x03U\x1d \x04\x81\xbe0\x81\xbb0\x81\xb8\x06\t+\x06\x01\x04\x01\x827\x15/0\x81\xaa0@\x06\x08+\x06\x01\x05\x05\x07\x02\x01\x164https://www.microsoft.com/pki/ssl/cps/WindowsPCA.htm0f\x06\x08+\x06\x01\x05\x05\x07\x02\x020Z\x1eX\x00C\x00o\x00p\x00y\x00r\x00i\x00g\x00h\x00t\x00 \x00\xa9\x00 \x001\x009\x009\x009\x00-\x002\x000\x000\x005\x00 \x00M\x00i\x00c\x00r\x00o\x00s\x00o\x00f\x00t\x00 \x00C\x00o\x00r\x00p\x00o\x00r\x00a\x00t\x00i\x00o\x00n\x00.0\r\x06\t*\x86H\x86\xf7\r\x01\x01\x05\x05\x00\x03\x82\x01\x01\x00s_\xca\x80\x1c`Fo\xb94\x9d\x88\xe3\xbe"\x8c\xfa\xe6X\x9a\xab{\x1a\x97\xfd\xed.9\xccY[\x1dz\x06\x8a\xbbC\x93{\x1e\xa1\x88S\xdfD\xf8S\xa9\xea\xf6g\x1b:x\x84\x11jo)G\x90\n\x0c{"wNo\xb8d)\xdf\x06\xc7\xc8s\x84\xd6f\xa0\xca\xd9Z&\x82W\xf9\xe3O9\xaf.\x8e\xb1\x06[r\xf272\xaeN\xce<}\xb0\x12+\x9e\xa5u\xe3C\xa6\x12\x8b\x06\x14\x98w\xe3X2%`\x07\x8cYq\xa7qA\xb3\x06\x8d\\\xef\x9c\x7fZ"m\xb7\xd3\xd9\xf5\xa6\x1bR\xde\xf5~v|\xfe\xf4\xc8#\x1aK%\xeb\xe4\xee\xaf\x10\x0bU\xc3\xd8\xc1\x17\x85ao\xd3?\xb6\xe9\xec\x84\xa5\xeem\xb2\xff\xe8l\x95\xab+^\xc8\x85\xc3\x11`\xac\xfa\x02\x05\xf1{\xda\xc3iI\x96\xa5p\xf9efF\x10\x8d4\xe9!\x94<\x0fqJ\x1c\xea\x1f\xf7#\xa6\x87`4\xe9\x14\xe1\xde\x03Y\xb4\x02\x1d:\xaf\xe3U\x05\xf5\xed\xc1\xf4\xe4]\x0e\xd3\x970\x82\x06\x070\x82\x03\xef\xa0\x03\x02\x01\x02\x02\na\x16h4\x00\x00\x00\x00\x00\x1c0\r\x06\t*\x86H\x86\xf7\r\x01\x01\x05\x05\x000_1\x130\x11\x06\n\t\x92&\x89\x93\xf2,d\x01\x19\x16\x03com1\x190\x17\x06\n\t\x92&\x89\x93\xf2,d\x01\x19\x16\tmicrosoft1-0+\x06\x03U\x04\x03\x13$Microsoft Root Certificate Authority0\x1e\x17\r070403125309Z\x17\r210403130309Z0w1\x0b0\t\x06\x03U\x04\x06\x13\x02US1\x130\x11\x06\x03U\x04\x08\x13\nWashington1\x100\x0e\x06\x03U\x04\x07\x13\x07Redmond1\x1e0\x1c\x06\x03U\x04\n\x13\x15Microsoft Corporation1!0\x1f\x06\x03U\x04\x03\x13\x18Microsoft Time-Stamp PCA0\x82\x01"0\r\x06\t*\x86H\x86\xf7\r\x01\x01\x01\x05\x00\x03\x82\x01\x0f\x000\x82\x01\n\x02\x82\x01\x01\x00\x9f\xa1l\xb1\xdf\xdbH\x92*|k.\x19\xe1\xbd\xe2\xe3\xc5\x99Q#P\xad\xce\xdd\x18N$\x0f\xee\xd1\xa7\xd1L\xadt0 \x11\xeb\x07\xd5T\x95\x15I\x94\x1bB\x92\xae\x98\\0&\xda\x00k\xe8{\xbd\xec\x89\x07\x0f\xf7\x0e\x04\x98\xf0\x89\xcc\x1f\xcb3$\x87\x9d\xf2\xf4g\x1c,\xfc{\xe7\x88\x1d\xea\xe7N\xa3\xa1\xc1#S\xca\x8d\xfaE\xcf\t\xd0^\xaf\xd0\xb0B\x04\xa2\xf9\xa6l\x93g\xd7(\xdcFS\xb0\x86\xd0\xe5(F.\'\xac\x86OUR\x0c\xe4\x03,\xfbj\x90\x900n\x87\xf3Y0\x9d\xfa~\xd6\x97\xb3\xe8!\x97~\xf8\xd2\x13\xf3\x08\xb7SmR\xb4E\x90\x9fH\x00JGf\x11\')f\xa8\x97\xe4\xd3\x06\x81J\xa2\xf9\x84\xa7\x11G\x14\t\x82\x9f\x84\xedUx\xfe\x01\x9a\x1dP\x08\x85\x00\x100F\xed\xb7\xde#F\xbb\xc4-T\x9f\xaf\x1exA1w\xcc\x9b\xdf;\x83\x93\xa1a\x02\xb5\x1d\r\xb1\xfc\xf7\x9b\xb2\x01\xce"KT\xff\xf9\x05\xc3\xc2 \x0b\x02\x03\x01\x00\x01\xa3\x82\x01\xab0\x82\x01\xa70\x0f\x06\x03U\x1d\x13\x01\x01\xff\x04\x050\x03\x01\x01\xff0\x1d\x06\x03U\x1d\x0e\x04\x16\x04\x14#4\xf8\xd9RFp\n\xed@\xfbv\xfb\xb3+\xb0\xc35\xb3\x0f0\x0b\x06\x03U\x1d\x0f\x04\x04\x03\x02\x01\x860\x10\x06\t+\x06\x01\x04\x01\x827\x15\x01\x04\x03\x02\x01\x000\x81\x98\x06\x03U\x1d#\x04\x81\x900\x81\x8d\x80\x14\x0e\xac\x82`@V\'\x97\xe5%\x13\xfc*\xe1\nS\x95Y\xe4\xa4\xa1c\xa4a0_1\x130\x11\x06\n\t\x92&\x89\x93\xf2,d\x01\x19\x16\x03com1\x190\x17\x06\n\t\x92&\x89\x93\xf2,d\x01\x19\x16\tmicrosoft1-0+\x06\x03U\x04\x03\x13$Microsoft Root Certificate Authority\x82\x10y\xad\x16\xa1J\xa0\xa5\xadLsX\xf4\x07\x13.e0P\x06\x03U\x1d\x1f\x04I0G0E\xa0C\xa0A\x86?http://crl.microsoft.com/pki/crl/products/microsoftrootcert.crl0T\x06\x08+\x06\x01\x05\x05\x07\x01\x01\x04H0F0D\x06\x08+\x06\x01\x05\x05\x070\x02\x868http://www.microsoft.com/pki/certs/MicrosoftRootCert.crt0\x13\x06\x03U\x1d%\x04\x0c0\n\x06\x08+\x06\x01\x05\x05\x07\x03\x080\r\x06\t*\x86H\x86\xf7\r\x01\x01\x05\x05\x00\x03\x82\x02\x01\x00\x10\x97\x8a\xc3\\\x03D6\xdd\xe9\xb4\xadw\xdb\xceyQM\x01\xb1.tq[m\x0c\x13\xab\xce\xbe{\x8f\xb8.\xd4\x12\xa2\x8cmb\xb8W\x02\xcbN \x13P\x99\xddz@\xe2W\xbb\xafX\x9a\x1c\xe1\x1d\x01\x86\xac\xbbx\xf2\x8b\xd0\xec;\x01\xee\xe2\xbe\x8f\n\x05\xc8\x8dH\xe2\xf0S\x15\xddO\xab\x92\xe4\xe7\x8dj\xd5\x80\xc1\xe6\x94\xf2\x06/\x85\x03\xe9\x91*$"p\xfb\xf6\xfc\xe4x\x99.\r\xf7\x07\xe2p\xbc\x18N\x9d\x8ek\nr\x95\xb8\xa19\x9cg-\xc5Q\x0e\xeab\\?\x16\x98\x8b ?\xe2\x07\x1a2\xf9\xcc1Jv1=+r\x0b\xc8\xeap=\xff\x85\n\x13\xdf\xc2\na\x8e\xf0\xd7\xb8\x17\xebN\x8b\x7f\xc55+^\xa3\xbf\xeb\xbc}\x0bB{\xd4Sr!\xee0\xca\xbbxe\\[\x01\x17\n\x14\x0e\xd2\xda\x14\x98\xf5<\xb9fX\xb3-/\xe7\xf9\x85\x86\xccQV\xe8\x9dp\x94l\xac9L\xd4\xf6y\xbf\xaa\x18zb)\xef\xa2\x9b)4\x06w\x1ab\xc9=\x1em\x1f\x82\xf0\x0b\xc7,\xbb\xcfC\xb3\xe5\xf9\xec}\xb5\xe3\xa4\xa8t5\xb8N\xc5q#\x12&v\x0b<R\x8cqZFC\x14\xbc\xb3\xb3\xb0Mg\xc8\x9fB\xff\x80y!\x80\x9e\x150f\xe8B\x12^\x1a\xc8\x9e"!\xd0C\xe9+\xe9\xbb\xf4H\xcc,\xd4\xd82\x80L&*H$_Z\xeaV\xef\xa6\xde\x99\x9d\xca:o\xbd\x81\'t\x06\x11\xeev!\xbf\x9b\x82\xc1\'T\xb6\xb1j=\x89\xa1va\xb4n\xa1\x13\xa6\xbf\xaaG\xf0\x12o\xfd\x8a2l\xb2\xfe\xdfQ\xc8\x8c#\xc9f\xbd\x9d\x1d\x87\x12d\x02=-\xafY\x8f\xb8\xe4!\xe5\xb5\xb0\xcac\xb4xT\x05\xd4A.P\xac\x94\xb0\xa5x\xab\xb3\xa0\x96u\x1a\xd9\x92\x87\x13u"/2\xa8\x08n\xa0[\x8c%\xbf\xa0\xef\x84\xca!\xd6\xeb\x1eO\xc9\x9a\xeeI\xe0\xf7\x01eo\x89\x0b}\xc8i\xc8\xe6n\xea\xa7\x97\xce1)\xff\x0e\xc5[\\\xd8M\x1b\xa1\xd8\xfa/\x9e?.U\x16k\xc9\x13\xa3\xfd0\x82\x06\xa10\x82\x04\x89\xa0\x03\x02\x01\x02\x02\na\x07\x02\xdc\x00\x00\x00\x00\x00\x0b0\r\x06\t*\x86H\x86\xf7\r\x01\x01\x05\x05\x000_1\x130\x11\x06\n\t\x92&\x89\x93\xf2,d\x01\x19\x16\x03com1\x190\x17\x06\n\t\x92&\x89\x93\xf2,d\x01\x19\x16\tmicrosoft1-0+\x06\x03U\x04\x03\x13$Microsoft Root Certificate Authority0\x1e\x17\r050915215541Z\x17\r160315220541Z0\x81\x811\x0b0\t\x06\x03U\x04\x06\x13\x02US1\x130\x11\x06\x03U\x04\x08\x13\nWashington1\x100\x0e\x06\x03U\x04\x07\x13\x07Redmond1\x1e0\x1c\x06\x03U\x04\n\x13\x15Microsoft Corporation1+0)\x06\x03U\x04\x03\x13"Microsoft Windows Verification PCA0\x82\x01"0\r\x06\t*\x86H\x86\xf7\r\x01\x01\x01\x05\x00\x03\x82\x01\x0f\x000\x82\x01\n\x02\x82\x01\x01\x00\xc5\xac\x93N\xe6J\x11\x9e7\xd05\xd2\xb0\x06\\\x83\x14\xa5a\x92\xfc\xfe\xb3\x89\xc1\xe6\xcd\xc8\x1f1\x84\t\x914F\x92,\xb8\xee,\xc5 s\xc4\xe8\x1a\xd8\x8d\xcezh\xb5fx\x8b\xe5`\x15\xa2\xf6)Z\x1d^]\xfcb\xd3\xc1\\)\x89\xfb3\xf8\x16\x956\xdd\xb1FtirS\xd5\xaa\xe8\x8a\x99\x1a\xd8\xf7g\t\xd9\t "8z\xd6\x03#\xd7\x89\x9f\x1c\x01\xb7Q\xdf\x98f*\x02\x8b\x06\xec\xe4)\xef[B\x17\x15\x97Q\x8d}%\n\xaf%\xe8\xde\xdc\x9bg\x16\nVs\x9d\xb3\x1d\x85\x83\x0b~3+b3\xce\x1c \x81K^\xd3\xc6I\xb8\xf6.\xd3N\xb0qDFd>\xdeCf\x04\xb9\xcc\x83H:\xc56z\x04H\x0b\x89\x02=c\xa2\x01v)\x97u\xe9\x01\xe6\x00\x97\t\x92\xf8\xe2\'\xf0)gCw\xc3P\x96S1\xe1\xb6q\x8b\xecw\xc7|1H\xd5\xb8%"\x8c\x00\xf7(8z\xbd|\xc7?\xcd@&w\xdd\x00\x00\x11\x9a\x95\xbe\x1f\xdb\x02\x03\x01\x00\x01\xa3\x82\x02:0\x82\x0260\x10\x06\t+\x06\x01\x04\x01\x827\x15\x01\x04\x03\x02\x01\x000\x1d\x06\x03U\x1d\x0e\x04\x16\x04\x14\x90\x8b\x11\xa5p\xed\xe0\xf9\xa9\xc0\xac\x08\xc7\xb5\xf4\x82\xb1<\xc5J0\x0b\x06\x03U\x1d\x0f\x04\x04\x03\x02\x01\xc60\x0f\x06\x03U\x1d\x13\x01\x01\xff\x04\x050\x03\x01\x01\xff0\x81\x98\x06\x03U\x1d#\x04\x81\x900\x81\x8d\x80\x14\x0e\xac\x82`@V\'\x97\xe5%\x13\xfc*\xe1\nS\x95Y\xe4\xa4\xa1c\xa4a0_1\x130\x11\x06\n\t\x92&\x89\x93\xf2,d\x01\x19\x16\x03com1\x190\x17\x06\n\t\x92&\x89\x93\xf2,d\x01\x19\x16\tmicrosoft1-0+\x06\x03U\x04\x03\x13$Microsoft Root Certificate Authority\x82\x10y\xad\x16\xa1J\xa0\xa5\xadLsX\xf4\x07\x13.e0P\x06\x03U\x1d\x1f\x04I0G0E\xa0C\xa0A\x86?http://crl.microsoft.com/pki/crl/products/microsoftrootcert.crl0T\x06\x08+\x06\x01\x05\x05\x07\x01\x01\x04H0F0D\x06\x08+\x06\x01\x05\x05\x070\x02\x868http://www.microsoft.com/pki/certs/MicrosoftRootCert.crt0\x81\x80\x06\x03U\x1d \x04y0w0u\x06\t+\x06\x01\x04\x01\x827\x15/0h0f\x06\x08+\x06\x01\x05\x05\x07\x02\x020Z\x1eX\x00C\x00o\x00p\x00y\x00r\x00i\x00g\x00h\x00t\x00 \x00\xa9\x00 \x001\x009\x009\x009\x00-\x002\x000\x000\x005\x00 \x00M\x00i\x00c\x00r\x00o\x00s\x00o\x00f\x00t\x00 \x00C\x00o\x00r\x00p\x00o\x00r\x00a\x00t\x00i\x00o\x00n\x00.0\x1f\x06\x03U\x1d%\x04\x180\x16\x06\x08+\x06\x01\x05\x05\x07\x03\x03\x06\n+\x06\x01\x04\x01\x827\n\x03\x060\r\x06\t*\x86H\x86\xf7\r\x01\x01\x05\x05\x00\x03\x82\x02\x01\x00%1\xa1X\xeaR\xe5\xe1\x17\x0c\xe6\xf1?x\xa3?r\xaf\xa7W\x10S\x89\x10^2\x9c\xb6p\xc3\x15+M\x1504\xe8\xc0j\xe4\x1c\xd3. eH\xd7\x1b\x98b!\xbaE\x9fJ\xec\xdb/\t\x19Q\xe5\xed2\x19Q/\xe1\xdd\xfb\xc6R\xfd\xeb\xc6\x82%B\x03\t\xa68\xb66\x1f\xcc\xc9\x80\xbbZi\x181\xc3\xb3\xa0\xb3gG\xbe\x9d\xc7\xe2?\x96\xb3\x88\xf8\x19\xbe9\xb9\xe9\x95\xce\xfc|\xaf\xa8\xcd\xd0A\x90\xe0\xd5\xb3\x1c/h\xbb\xdb\x0flj\xdd\xf2\xaf\xde\xf2\xb5\xde\r\xb6\xa6Z\xf0\x86\n\xb9m\x99K?{-\x01\x84l\x8f\x87\xdc\x7f\x8f\xab\x14\x88\xd0\x06\x914\xbe\x1b\x82"\xa4\xbcU\x8a\xad\x9b\xfcs\x14\x10\xc4\xc9\x19\x1e\x07}\x9b\x0e\xc0\x95&]\xc6\x1f\xac\xb4\xf2~\xba%pJ{\xd7\x8e\xd1\x9d\xa0\x13Iz\xb0\x02RR$\xf4\xaf\xdd@-\xe5>2X\xb3Jj\xdd\x11Y\xaa-\xbc\xa4\xa0s8\xf9@wk4\x19W\xcd8h\'\x82\xf8\xd1o\xeb#\xc0?R\xf3N\xd5\x02>j\x9a+\xc1\xf51q\xdbAM;\xde\xef\xad\xaf\x1f\x88eC\x1bQ\xb7\x9au\xca\x8eiI\x10\x8fx\x8atE\xb9\t\x8esw\x072JK\xd7h+\x98\xc5\xbaT\xea?\xcb\xa2\x00\x8c\xbb\xd8\x10X\xf2\xdb\xdc\x9b\xcd\xd8\xeaHC\xe2J~e\xb2\xdc\xf5-N%g\xa8\xe0\xb5\xba\xa7\xdd~^\xc1L\x02t\xc9\xb3n\xe3\xf8\xf0\x0b\xed\xfc\xb9)\xc5[\xc96Q\x90\xdbx}\xb92\x0f^v\xd2\x15\\;7!\xc6\xdb\xc9\x19n\xedt*\\,\x0bQIES\xb0\xb2\xb3#\xd4\xa1\xb0_\r\x19\xcd\x14\xa7\xe3<\x9b\x97r\x94\x14\xdf\xff\xc1\x90\x1b\xa5\xdf\xf5\xa9\xf3\x1b\x17\xda\xb5\xfcD\xe0\xe8\xe2<\xa2z\xbb\xbbe\xe6M\xb1\xb5\x15\xa1\xd9g;\xb0\x0c};\xe9\xeeQ*G\xf5\x15\x0f\x8c\xad].5\xdf\xf4\xa4.\xf6\x137Z+\xe8U\x9aI,\x97\xce\x9d\x01\x9e\x97F\\\xd9-\xbc$Z\x95YoM\xca\x9d\xd6W&1\x82\x04\x960\x82\x04\x92\x02\x01\x010\x81\x900\x81\x811\x0b0\t\x06\x03U\x04\x06\x13\x02US1\x130\x11\x06\x03U\x04\x08\x13\nWashington1\x100\x0e\x06\x03U\x04\x07\x13\x07Redmond1\x1e0\x1c\x06\x03U\x04\n\x13\x15Microsoft Corporation1+0)\x06\x03U\x04\x03\x13"Microsoft Windows Verification PCA\x02\na\x01\xc6\xc1\x00\x00\x00\x00\x00\x070\t\x06\x05+\x0e\x03\x02\x1a\x05\x00\xa0\x81\xba0\x19\x06\t*\x86H\x86\xf7\r\x01\t\x031\x0c\x06\n+\x06\x01\x04\x01\x827\x02\x01\x040\x1c\x06\n+\x06\x01\x04\x01\x827\x02\x01\x0b1\x0e0\x0c\x06\n+\x06\x01\x04\x01\x827\x02\x01\x150#\x06\t*\x86H\x86\xf7\r\x01\t\x041\x16\x04\x14\xcd?\xf6\x88\xbe1\x7f\xaduK\n\xf5B\x99\xc6\xab\x81\xc92\x190Z\x06\n+\x06\x01\x04\x01\x827\x02\x01\x0c1L0J\xa0$\x80"\x00M\x00i\x00c\x00r\x00o\x00s\x00o\x00f\x00t\x00 \x00W\x00i\x00n\x00d\x00o\x00w\x00s\xa1"\x80 http://www.microsoft.com/windows0\r\x06\t*\x86H\x86\xf7\r\x01\x01\x01\x05\x00\x04\x82\x01\x00\x9d\xe7\x0f`\xe4\xc2\t\xb35?\x93O\x08\xf6A\x9f\x0eiu\x89\xa17\xf3\x80\xa6\x9c\x90w3q9#\xa6>)\x8e\xbf\x98\xf8=\xee\x8ae\x83\x1b|\x0b{\xd7\th\x12}\x03~\r\x1b/K\xec=Y-\xfc\xc2\x8b8\x90v}\xdek1\xb57\x88\x90\x96\x83\xb51!\xa9\xd7\x15\x87\xfa\x12\xc1\x84\xf0\x91\x80\x0e#rH\x8a\x04\x97R\xba\xfe\xc3\xeb\xa1\xdb`\x08<1\n>*|\xae\xeex\xf4\xcd\x02b7\x00\xcf3/\x13>R{\xdf\xa0\xcb\x84\xa6cRg\xc8b\r\xb0|=:\xdf5M&\x0c*N\x83\xe9-\xa0 \xee)ZJ\xadD\xb95\t\xa4\x0f\xa2(c\xe9\xdb\xd0;\xcb0\xdd!{}\xec\x8a/\xe3\xfa\xdf9\x1aa\xf8EF\x9eC\xd4H\xbaT>4\x9f\x88si]\x85\xae\xdev\xf4{\x968\x0b\r\x11\xa4^/qN\xf1\xd9n~\x00\xa4J\xc0\x88\xe4\xf0\x02\xae\\\x94\xd6\r\xbf\'\xdfSQ\x08\xd2$\x18\xd0\xba"\x91\xe7\xc4\x0b\xa1\x82\x02\x1d0\x82\x02\x19\x06\t*\x86H\x86\xf7\r\x01\t\x061\x82\x02\n0\x82\x02\x06\x02\x01\x010\x81\x850w1\x0b0\t\x06\x03U\x04\x06\x13\x02US1\x130\x11\x06\x03U\x04\x08\x13\nWashington1\x100\x0e\x06\x03U\x04\x07\x13\x07Redmond1\x1e0\x1c\x06\x03U\x04\n\x13\x15Microsoft Corporation1!0\x1f\x06\x03U\x04\x03\x13\x18Microsoft Time-Stamp PCA\x02\na\x04\xcai\x00\x00\x00\x00\x00\x080\x07\x06\x05+\x0e\x03\x02\x1a\xa0]0\x18\x06\t*\x86H\x86\xf7\r\x01\t\x031\x0b\x06\t*\x86H\x86\xf7\r\x01\x07\x010\x1c\x06\t*\x86H\x86\xf7\r\x01\t\x051\x0f\x17\r090714014555Z0#\x06\t*\x86H\x86\xf7\r\x01\t\x041\x16\x04\x14\x80\xd1\xdb\xd8\r\x9b\x91a\xc3\x9c$\xb5m\xcd\x85"b\xf5u\x1f0\r\x06\t*\x86H\x86\xf7\r\x01\x01\x05\x05\x00\x04\x82\x01\x00\x86\xec4$\xf6\x9d\xc4\xe93\xc6e\xa3\x02\x1b4+\x8f1G\xda\xf8\xb1\xd9\xcd[\x0b\xb2T\xef\x1c\x93\xb8g\xb1\x0e?@\xfam@\xa5\xbf\xbf\r"%\xfd\x9a\xa7S\x04\xfbk\x9b\xd3?K\x00\x91\xe2a\x024\xd3\x9b\xa9\x85\x16BE\xfc\x8b~Z\xc1u\xe8L\xb4\x0f\xbe\xe6M\x94\xdc\xaf\xe2\xa9\xe8\x18\x8aF|\xcf\xeaa\xaaU\xc1\xdd\xb4\x8fj\xa6i\xa4\xd13n%(\xbe\x96\xf5\xfds\x88\xe1H%\xc3|.1\xe6\xf7G\x85C\xc8\xa7;E\x0e\xe7\\[\xef=\x0b\x7f\x91\x96\xca\xba\xf2\\&\xa8\xb7\xa9&\xc79\xf7x\xff\xdd\x9d\xf0\xf2C\xec\xe2\xffq_\xe9\xcd\xb3\x85\xb4\x14\xf4\x99\x81\xdf w\x1c\xabM\x02~\xeb\x95\x94,\xe4\xfd!\xc9\xff\xc7\xea\xae\x9f"W\xc5\xbdd\x04\xd0\xe1\xe1\x1f\x8f\x8a\x9e\xbd\x04\xa3\x07o\xcb\xb1\xd6\x97\xd8-L\xf05u\xbc\xe7\xb2\x88\xcb\xa2n\xd0\x11m\xde\x9e+\xf4\xa7\xd3y\xb0\xb6\x8cX\xa6\xa4K\x14j`0\xff\xa0\xff',
      )],
      'sha1': b'\x9b\xd4D\xd5\x8bY\xcc\xa82\xbb_\xc9\x11\xf8\x1flf\xb4\x0f\xcc',
      'name': 'pecoff',
      'md5': b'q\x03\x14]B\xe8\x19h\xd1\xa77\x136\x05[-'
  }]

  SAMPLE_DATA_2 = [{
      'sha256':
          b"\xa1]\x00\x99\\\xfc\x00'\x0e6\xc4i\x11\x93\xda\x84\xaa\x10\xf8\xcb\xc3\x1b\x9eI\xf3\xe63\x8c\xd2\x7f\x862",
      'sha512':
          b'^\\\x8d\x97\xb3o=0\xf2\xfe\xc5\x1c\xc8\xf5g|\xb1U#\x9d\x9d%\x82>}A\xe2\x0f\x81\x14\xb48\x15\x88o\x11|SO\xf8)3\x87\xee8\x88\xb6X\xc4\xfa\xb2&q\n\xe0x\xbcW:!A\x11\x05\x05',
      'name':
          'generic',
      'md5':
          b'&\x087]\x0b\x15\xf6a\xfbo\xbd\x7fe\xb6\xae)',
      'sha1':
          b'\xfcY\x7f~\x84\xd2=\x1db\xd5@\xa5\xfd\xac\xc4p\xd8%\xfat'
  }]

  SAMPLE_DATA_3 = [{
      'sha256':
          b'\x05\x0b\xbe\xb6\xb9\xaa@Ba\xb2\t\x892\\hC7\x086z\xaa\xedN\x1d\xff=$\xae)\xa5-*',
      'sha512':
          b"\xf8\xc0`\xcc<.D\xdbL\x04\x0e|\x9c\xfb\x05B\x8a\x17\x92\n'\xc4N%\x8f\xe7]\x11\xe3l\xe6V\xa7\xd0\x18\x86\xe6F\xfe\x8e\x1a\xa7\xb4\x9a\x16Yw\x0c!*\xb4\x91IB5e\x06\x16\r\x1f\xack\x16r",
      'name':
          'generic',
      'md5':
          b'{CV{L2\xadz\xde\xd57\xcd;\x13B\xb9',
      'sha1':
          b'\x83"\xf1\xc2\xc3U\xd8\x842\xf1\xf0:\x1f#\x1fc\x91!\x86\xbd'
  }, {
      'SignedData': [(
          512, 2,
          b'0\x82\x153\x06\t*\x86H\x86\xf7\r\x01\x07\x02\xa0\x82\x15$0\x82\x15 \x02\x01\x011\x0e0\x0c\x06\x08*\x86H\x86\xf7\r\x02\x05\x05\x000g\x06\n+\x06\x01\x04\x01\x827\x02\x01\x04\xa0Y0W03\x06\n+\x06\x01\x04\x01\x827\x02\x01\x0f0%\x03\x01\x00\xa0 \xa2\x1e\x80\x1c\x00<\x00<\x00<\x00O\x00b\x00s\x00o\x00l\x00e\x00t\x00e\x00>\x00>\x00>0 0\x0c\x06\x08*\x86H\x86\xf7\r\x02\x05\x05\x00\x04\x10\xe0(n\x9az\x8d\xf0\xd5\xd9\xea\x88m\xea\x8e\xe6\x00\xa0\x82\x10\xea0\x82\x03z0\x82\x02b\xa0\x03\x02\x01\x02\x02\x108%\xd7\xfa\xf8a\xaf\x9e\xf4\x90\xe7&\xb5\xd6Z\xd50\r\x06\t*\x86H\x86\xf7\r\x01\x01\x05\x05\x000S1\x0b0\t\x06\x03U\x04\x06\x13\x02US1\x170\x15\x06\x03U\x04\n\x13\x0eVeriSign, Inc.1+0)\x06\x03U\x04\x03\x13"VeriSign Time Stamping Services CA0\x1e\x17\r070615000000Z\x17\r120614235959Z0\\1\x0b0\t\x06\x03U\x04\x06\x13\x02US1\x170\x15\x06\x03U\x04\n\x13\x0eVeriSign, Inc.1402\x06\x03U\x04\x03\x13+VeriSign Time Stamping Services Signer - G20\x81\x9f0\r\x06\t*\x86H\x86\xf7\r\x01\x01\x01\x05\x00\x03\x81\x8d\x000\x81\x89\x02\x81\x81\x00\xc4\xb5\xf2R\x15\xbc\x88\x86`)\x16J[/K\x91k\x87\x91\xf35TX5\xea\xd16^bMRQ4q\xc2{f\x1d\x89\xc8\xdd*\xc4j\n\xf67\xd9\x98t\x91\xf6\x92\xae\xb0\xb5v\x96\xf1\xa9JcEG.k\x0b\x92NK+\x8c\xeeXJ\x8b\xd4\x07\xe4\x1a,\xf8\x82\xaaX\xd9\xcdB\xf3-\xc0u\xde\x8d\xab\xc7\x8e\x1d\x9alL\x08\x95\x1e\xde\xdb\xefg\xe1r\xc2I\xc2\x9e`<\xe1\xe2\xbe\x16\xa3cxi\x14{\xad-\x02\x03\x01\x00\x01\xa3\x81\xc40\x81\xc104\x06\x08+\x06\x01\x05\x05\x07\x01\x01\x04(0&0$\x06\x08+\x06\x01\x05\x05\x070\x01\x86\x18http://ocsp.verisign.com0\x0c\x06\x03U\x1d\x13\x01\x01\xff\x04\x020\x0003\x06\x03U\x1d\x1f\x04,0*0(\xa0&\xa0$\x86"http://crl.verisign.com/tss-ca.crl0\x16\x06\x03U\x1d%\x01\x01\xff\x04\x0c0\n\x06\x08+\x06\x01\x05\x05\x07\x03\x080\x0e\x06\x03U\x1d\x0f\x01\x01\xff\x04\x04\x03\x02\x06\xc00\x1e\x06\x03U\x1d\x11\x04\x170\x15\xa4\x130\x111\x0f0\r\x06\x03U\x04\x03\x13\x06TSA1-20\r\x06\t*\x86H\x86\xf7\r\x01\x01\x05\x05\x00\x03\x82\x01\x01\x00P\xc5K\xc8$\x80\xdf\xe4\r$\xc2\xde\x1a\xb1\xa1\x02\xa1\xa6\x82-\x0c\x83\x15\x817\n\x82\x0e,\xb0Z\x17a\xb5\xd8\x05\xfe\x88\xdb\xf1\x91\x91\xb3V\x1a@\xa6\xeb\x92\xbe89\xb0u6t:\x98O\xe47\xba\x99\x89\xca\x95B\x1d\xb0\xb9\xc7\xa0\x8dW\xe0\xfa\xd5d\x04B5N\x01\xd13\xa2\x17\xc8M\xaa\'\xc7\xf2\xe1\x86L\x028M\x83x\xc6\xfcS\xe0\xeb\xe0\x06\x87\xdd\xa4\x96\x9e^\x0c\x98\xe2\xa5\xbe\xbf\x82\x85\xc3`\xe1\xdf\xad(\xd8\xc7\xa5Kd\xda\xc7\x1b[\xbd\xac9\x08\xd58"\xa13\x8b/\x8a\x9a\xeb\xbc\x07!?DA\t\x07\xb5e\x1c$\xbcH\xd3D\x80\xeb\xa1\xcf\xc9\x02\xb4\x14\xcfT\xc7\x16\xa3\x80\\\xf9y>]r}\x88\x17\x9e,C\xa2\xcaS\xce}=\xf6*:\xb8O\x94\x00\xa5m\n\x83]\xf9^S\xf4\x18\xb3W\x0fp\xc3\xfb\xf5\xad\x95\xa0\x0e\x17\xde\xc4\x16\x80`\xc9\x0f+n\x86\x04\xf1\xeb\xf4x\'\xd1\x05\xc5\xee4[^\xb9I2\xf230\x82\x03\xc40\x82\x03-\xa0\x03\x02\x01\x02\x02\x10G\xbf\x19\x95\xdf\x8dRFC\xf7\xdbmH\r1\xa40\r\x06\t*\x86H\x86\xf7\r\x01\x01\x05\x05\x000\x81\x8b1\x0b0\t\x06\x03U\x04\x06\x13\x02ZA1\x150\x13\x06\x03U\x04\x08\x13\x0cWestern Cape1\x140\x12\x06\x03U\x04\x07\x13\x0bDurbanville1\x0f0\r\x06\x03U\x04\n\x13\x06Thawte1\x1d0\x1b\x06\x03U\x04\x0b\x13\x14Thawte Certification1\x1f0\x1d\x06\x03U\x04\x03\x13\x16Thawte Timestamping CA0\x1e\x17\r031204000000Z\x17\r131203235959Z0S1\x0b0\t\x06\x03U\x04\x06\x13\x02US1\x170\x15\x06\x03U\x04\n\x13\x0eVeriSign, Inc.1+0)\x06\x03U\x04\x03\x13"VeriSign Time Stamping Services CA0\x82\x01"0\r\x06\t*\x86H\x86\xf7\r\x01\x01\x01\x05\x00\x03\x82\x01\x0f\x000\x82\x01\n\x02\x82\x01\x01\x00\xa9\xca\xb2\xa4\xcc\xcd \xaf\n}\x89\xac\x87u\xf0\xb4N\xf1\xdf\xc1\x0f\xbfga\xbd\xa3d\x1c\xda\xbb\xf9\xca3\xab\x840\x89X~\x8c\xdbk\xdd6\x9e\x0f\xbf\xd1\xecx\xf2w\xa6~o<\xbf\x93\xaf\r\xbah\xf4l\x94\xca\xbdR-\xabH=\xf5\xb6\xd5]_\x1b\x02\x9f\xfa/k\x1e\xa4\xf7\xa3\x9a\xa6\x1a\xc8\x02\xe1\x7fLR\xe3\x0e`\xec@\x1c~\xb9\r\xde?\xc7\xb4\xdf\x87\xbd_zj1.\x03\x99\x81\x13\xa8G \xce1s\rW-\xcdx43\x95\x12\x99\x12\xb9\xdeh/\xaa\xe6\xe3\xc2\x8a\x8c*\xc3\x8b!\x87f\xbd\x83XWou\xbf<\xaa&\x87]\xca\x10\x15<\x9f\x84\xeaT\xc1\nn\xc4\xfe\xc5J\xdd\xb9\x07\x11\x97"|\xdb>\'\xd1\x1ex\xec\x9f1\xc9\xf1\xe6"\x19\xdb\xc4\xb3GC\x9a\x1a_\xa0\x1e\x90\xe4^\xf5\xee|\xf1}\xabb\x01\x8f\xf5M\x0b\xde\xd0"V\xa8\x95\xcd\xae\x88v\xae\xee\xba\r\xf3\xe4M\xd9\xa0\xfbh\xa0\xae\x14;\xb3\x87\xc1\xbb\x02\x03\x01\x00\x01\xa3\x81\xdb0\x81\xd804\x06\x08+\x06\x01\x05\x05\x07\x01\x01\x04(0&0$\x06\x08+\x06\x01\x05\x05\x070\x01\x86\x18http://ocsp.verisign.com0\x12\x06\x03U\x1d\x13\x01\x01\xff\x04\x080\x06\x01\x01\xff\x02\x01\x000A\x06\x03U\x1d\x1f\x04:0806\xa04\xa02\x860http://crl.verisign.com/ThawteTimestampingCA.crl0\x13\x06\x03U\x1d%\x04\x0c0\n\x06\x08+\x06\x01\x05\x05\x07\x03\x080\x0e\x06\x03U\x1d\x0f\x01\x01\xff\x04\x04\x03\x02\x01\x060$\x06\x03U\x1d\x11\x04\x1d0\x1b\xa4\x190\x171\x150\x13\x06\x03U\x04\x03\x13\x0cTSA2048-1-530\r\x06\t*\x86H\x86\xf7\r\x01\x01\x05\x05\x00\x03\x81\x81\x00Jk\xf9\xeaX\xc2D\x1c1\x89y\x99+\x96\xbf\x82\xac\x01\xd6\x1cL\xcd\xb0\x8aXn\xdf\x08)\xa3^\xc8\xca\x93\x13\xe7\x04R\r\xefG\'/\x008\xb0\xe4\xc9\x93N\x9a\xd4"b\x15\xf7?7!Op1\x80\xf1\x8b8\x87\xb3\xe8\xe8\x97\x00\xfe\xcfU\x96N$\xd2\xa9\'Nz\xae\xb7aA\xf3*\xce\xe7\xc9\xd9^\xdd\xbb+\x85>\xb5\x9d\xb5\xd9\xe1W\xff\xbe\xb4\xc5~\xf5\xcf\x0c\x9e\xf0\x97\xfe+\xd3;R\x1b\x1b8\'\xf7?J0\x82\x04\xbf0\x82\x04(\xa0\x03\x02\x01\x02\x02\x10A\x91\xa1Z9x\xdf\xcfIef8\x1dLu\xc20\r\x06\t*\x86H\x86\xf7\r\x01\x01\x05\x05\x000_1\x0b0\t\x06\x03U\x04\x06\x13\x02US1\x170\x15\x06\x03U\x04\n\x13\x0eVeriSign, Inc.1705\x06\x03U\x04\x0b\x13.Class 3 Public Primary Certification Authority0\x1e\x17\r040716000000Z\x17\r140715235959Z0\x81\xb41\x0b0\t\x06\x03U\x04\x06\x13\x02US1\x170\x15\x06\x03U\x04\n\x13\x0eVeriSign, Inc.1\x1f0\x1d\x06\x03U\x04\x0b\x13\x16VeriSign Trust Network1;09\x06\x03U\x04\x0b\x132Terms of use at https://www.verisign.com/rpa (c)041.0,\x06\x03U\x04\x03\x13%VeriSign Class 3 Code Signing 2004 CA0\x82\x01"0\r\x06\t*\x86H\x86\xf7\r\x01\x01\x01\x05\x00\x03\x82\x01\x0f\x000\x82\x01\n\x02\x82\x01\x01\x00\xbe\xbc\xee\xbc~\xef\x83\xeb\xe07O\xfb\x03\x108\xbe\x08\xd2\x8c}\x9d\xfa\x92\x7f\x19\x0c\xc2k\xeeBR\x8c\xde\xd3\x1cH\x13%\xea\xc1cz\xf9Qe\xee\xd3\xaa;\xf5\xf0\x94\x9c+\xfb\xf2f\xd4$\xda\xf7\xf5\x9fn\x1996\xbc\xd0\xa3v\x08\x1e"\'$l8\x91\'\xe2\x84I\xae\x1b\x8a\xa1\xfd%\x82,\x100\xe8q\xab(\xe8wJQ\xf1\xec\xcd\xf8\xf0T\xd4o\xc0\xe3m\n\x8f\xd9\xd8d\x8dc\xb2-N\'\xf6\x85\x0e\xfem\xe3)\x99\xe2\x85G|-\x86\x7f\xe8W\x8f\xadg\xc232\x91\x13 \xfc\xa9#\x14\x9am\xc2\x84Kvh\x04\xd5q,]!\xfa\x88\r&\xfd\x1f-\x91+\xe7\x01UM\xf2m5(\x82\xdf\xd9k\\\xb6\xd6\xd9\xaa\x81\xfd_\xcd\x83\xbac\x9d\xd0"\xfc\xa9;Bi\xb2\x8e:\xb5\xbc\xb4\x9e\x0f^\xc4\xea,\x82\x8b(\xfdS\x08\x96\xdd\xb5\x01 \xd1\xf9\xa5\x18\xe7\xc0\xeeQp7\xe1\xb6\x05HRHo8\xea\xc3\xe8l{D\x84\xbb\x02\x03\x01\x00\x01\xa3\x82\x01\xa00\x82\x01\x9c0\x12\x06\x03U\x1d\x13\x01\x01\xff\x04\x080\x06\x01\x01\xff\x02\x01\x000D\x06\x03U\x1d \x04=0;09\x06\x0b`\x86H\x01\x86\xf8E\x01\x07\x17\x030*0(\x06\x08+\x06\x01\x05\x05\x07\x02\x01\x16\x1chttps://www.verisign.com/rpa01\x06\x03U\x1d\x1f\x04*0(0&\xa0$\xa0"\x86 http://crl.verisign.com/pca3.crl0\x1d\x06\x03U\x1d%\x04\x160\x14\x06\x08+\x06\x01\x05\x05\x07\x03\x02\x06\x08+\x06\x01\x05\x05\x07\x03\x030\x0e\x06\x03U\x1d\x0f\x01\x01\xff\x04\x04\x03\x02\x01\x060\x11\x06\t`\x86H\x01\x86\xf8B\x01\x01\x04\x04\x03\x02\x00\x010)\x06\x03U\x1d\x11\x04"0 \xa4\x1e0\x1c1\x1a0\x18\x06\x03U\x04\x03\x13\x11Class3CA2048-1-430\x1d\x06\x03U\x1d\x0e\x04\x16\x04\x14\x08\xf5Q\xe8\xfb\xfe==d6|h\xcf[x\xa8\xdf\xb9\xc570\x81\x80\x06\x03U\x1d#\x04y0w\xa1c\xa4a0_1\x0b0\t\x06\x03U\x04\x06\x13\x02US1\x170\x15\x06\x03U\x04\n\x13\x0eVeriSign, Inc.1705\x06\x03U\x04\x0b\x13.Class 3 Public Primary Certification Authority\x82\x10p\xba\xe4\x1d\x10\xd9)4\xb68\xca{\x03\xcc\xba\xbf0\r\x06\t*\x86H\x86\xf7\r\x01\x01\x05\x05\x00\x03\x81\x81\x00\xae:\x17\xb8J{U\xfadU\xec@\xa4\xedIA\x90\x99\x9c\x89\xbc\xaf.\x1d\xcax#\xf9\x1c\x19\x0f\x7f\xebh\xbc2\xd9\x888\xde\xdc?\xd3\x89\xb4?\xb1\x82\x96\xf1\xa4Z\xba\xed.&\xd3\xde|\x01n\x00\n\x00\xa4\x06\x92\x11H\t@\xf9\x1c\x18yg#$\xe0\xbb\xd5\xe1P\xae\x1b\xf5\x0e\xdd\xe0.\x81\xcd\x80\xa3lRO\x91uU\x8a\xba"\xf2\xd2\xeaAu\x88/cU}\x1eTZ\x95Y\xca\xd94\x81\xc0_^\xf6z\xb50\x82\x04\xdd0\x82\x03\xc5\xa0\x03\x02\x01\x02\x02\x10\t^ \xa8\x0bT\xa4]6\xf6\xc0k\x9b\x02\xf9:0\r\x06\t*\x86H\x86\xf7\r\x01\x01\x05\x05\x000\x81\xb41\x0b0\t\x06\x03U\x04\x06\x13\x02US1\x170\x15\x06\x03U\x04\n\x13\x0eVeriSign, Inc.1\x1f0\x1d\x06\x03U\x04\x0b\x13\x16VeriSign Trust Network1;09\x06\x03U\x04\x0b\x132Terms of use at https://www.verisign.com/rpa (c)041.0,\x06\x03U\x04\x03\x13%VeriSign Class 3 Code Signing 2004 CA0\x1e\x17\r070627000000Z\x17\r090626235959Z0\x81\xa01\x0b0\t\x06\x03U\x04\x06\x13\x02US1\x130\x11\x06\x03U\x04\x08\x13\nCalifornia1\x120\x10\x06\x03U\x04\x07\x13\tCupertino1\x130\x11\x06\x03U\x04\n\x14\nApple Inc.1>0<\x06\x03U\x04\x0b\x135Digital ID Class 3 - Microsoft Software Validation v21\x130\x11\x06\x03U\x04\x03\x14\nApple Inc.0\x81\x9f0\r\x06\t*\x86H\x86\xf7\r\x01\x01\x01\x05\x00\x03\x81\x8d\x000\x81\x89\x02\x81\x81\x00\xc0\xcc\xa0\x8c\x1b\x94\x94\xe0\xd7c\xcej6M\xa6\\Z7\xd1\x93i\x05\xb1\xf2V\x00\x14\xb0)s4\xb5\x11T\x0c\xe3JI\xf5\xad\x07\xc0\x90\xb7\xc0?GF\xd8\xeb\'\'i\xd7#\x8c\xb8\xb04^"l\xb0\xff\x98^\x0c\x03,J\xa0x&98!\x91\'1\xc0\x9b\xb7\xe6\xfa\x84G\xd0L(%\xd1\x075E\xbb\xa8\xf0\xa60\xdb\xf1\xd6\x8b\xfa\xc8\xa2\x87\xc3f\xfa\x90\xc5W\xccD\x01\xbd\xac\xe2V\xa6\xbaG\xf9E\x9cP\xcf\x02\x03\x01\x00\x01\xa3\x82\x01\x7f0\x82\x01{0\t\x06\x03U\x1d\x13\x04\x020\x000\x0e\x06\x03U\x1d\x0f\x01\x01\xff\x04\x04\x03\x02\x07\x800@\x06\x03U\x1d\x1f\x0490705\xa03\xa01\x86/http://CSC3-2004-crl.verisign.com/CSC3-2004.crl0D\x06\x03U\x1d \x04=0;09\x06\x0b`\x86H\x01\x86\xf8E\x01\x07\x17\x030*0(\x06\x08+\x06\x01\x05\x05\x07\x02\x01\x16\x1chttps://www.verisign.com/rpa0\x13\x06\x03U\x1d%\x04\x0c0\n\x06\x08+\x06\x01\x05\x05\x07\x03\x030u\x06\x08+\x06\x01\x05\x05\x07\x01\x01\x04i0g0$\x06\x08+\x06\x01\x05\x05\x070\x01\x86\x18http://ocsp.verisign.com0?\x06\x08+\x06\x01\x05\x05\x070\x02\x863http://CSC3-2004-aia.verisign.com/CSC3-2004-aia.cer0\x1f\x06\x03U\x1d#\x04\x180\x16\x80\x14\x08\xf5Q\xe8\xfb\xfe==d6|h\xcf[x\xa8\xdf\xb9\xc570\x11\x06\t`\x86H\x01\x86\xf8B\x01\x01\x04\x04\x03\x02\x04\x100\x16\x06\n+\x06\x01\x04\x01\x827\x02\x01\x1b\x04\x080\x06\x01\x01\x00\x01\x01\xff0\r\x06\t*\x86H\x86\xf7\r\x01\x01\x05\x05\x00\x03\x82\x01\x01\x00\x88\xd9Hf\r\xd5\xfa\x0f6\xfd\x87\t\xe2[\xda\x1e\x9c\xb9\xdcs\xbdbM\x1f\x84[\xa1\x88\xf2A\x06\xa1\xe0\xed\x14N\xc4\xa23q>\x9d\x87|\xfc\x0c\xf2\x91\x18\x8d\xb4\x1d\x07x%\xa9\x82\x8aX\xef\xc2?g\x89\xfc\x0f\xaa\x0fMB\xb3o\xbb\xef\x1c\xde\xaf&\xdc\xd6\xad\xf7\xcc\xe7\xce\xbfK<\x02>5\xf7\xef\x14\x8f\xb8\x06"\x95~\xc3,\xa4%\n\xd0\xd8\x01:\x83\x1e\xc1\x87)\xfe]\xc1\rW\xa6G\xde,\xfe*x\x1f\xc6\x99\xa1r\xe8\xd3\xadL\x93\xd6~U\xa8Q\x05\x99\x8aP\x16\xedY]m@\x04\xb9R\xf3\xaa\x8cZme\x96\x0f\x08\xcc\xb8\x15:n#\xbdAz\xb6\xf1K#\x05-\x17<9\n\x8a!@\xe6on&\xab\xed\x91\xcf\x96\xf5\xe8[k\xea\xf2\xde_\x92\n\xce\x1e\xbf\xb5\xb5*\xd5\xac\xdfv\xb9\xc9\xd6\xbe\xca\x15\x8e\xab5;\x98\xea/\x8e\xdc\x084zef\x11V0\xd1\xeb"\x98\x10\x8c\xe34\x93\xda\x10\x90\x18\xf5a8CI\x181\x82\x03\xb20\x82\x03\xae\x02\x01\x010\x81\xc90\x81\xb41\x0b0\t\x06\x03U\x04\x06\x13\x02US1\x170\x15\x06\x03U\x04\n\x13\x0eVeriSign, Inc.1\x1f0\x1d\x06\x03U\x04\x0b\x13\x16VeriSign Trust Network1;09\x06\x03U\x04\x0b\x132Terms of use at https://www.verisign.com/rpa (c)041.0,\x06\x03U\x04\x03\x13%VeriSign Class 3 Code Signing 2004 CA\x02\x10\t^ \xa8\x0bT\xa4]6\xf6\xc0k\x9b\x02\xf9:0\x0c\x06\x08*\x86H\x86\xf7\r\x02\x05\x05\x00\xa0\x81\xba0\x19\x06\t*\x86H\x86\xf7\r\x01\t\x031\x0c\x06\n+\x06\x01\x04\x01\x827\x02\x01\x040\x1c\x06\n+\x06\x01\x04\x01\x827\x02\x01\x0b1\x0e0\x0c\x06\n+\x06\x01\x04\x01\x827\x02\x01\x150\x1f\x06\t*\x86H\x86\xf7\r\x01\t\x041\x12\x04\x10\x85\x90\xd4\xd2\xd3{w\xde\xf3[\xe7E\x88w\xc7+0^\x06\n+\x06\x01\x04\x01\x827\x02\x01\x0c1P0N\xa0,\x80*\x00A\x00p\x00p\x00l\x00e\x00 \x00S\x00o\x00f\x00t\x00w\x00a\x00r\x00e\x00 \x00U\x00p\x00d\x00a\x00t\x00e\xa1\x1e\x80\x1chttp://www.apple.com/macosx 0\r\x06\t*\x86H\x86\xf7\r\x01\x01\x01\x05\x00\x04\x81\x80\x9d\x80\xe8L\xe3P\xae\x81\xbb{\xa5\xe6\x14\xf3\xc6f\x8e\x80\xde\xfa\xceM^\xea\xf7\xad[n\xedFx>;o\x85\x0f\xba:\xd9\xbfA\xf5b\xc02\x96o\xdbBe\xd7\x84f\xbb\xb2\xccJj$\x01\xa0=\xf0\x03f\xdd\x0b\xdb\x8cY\xda\xb3-\x89\xe3\x7f\xfd\x1d\x97\xe3\xab\x15\\b\xc1\xcd\x0f\xdd9\xab\xa4r0\xdd <\x8b=\xf1\xe1\xb8+\xbc<\xf5C\xf7\xf2\xef\x9a5\xcd\xca\xd0\x85R\xca~\xe4F2\xe9KN\x03\n\xcb\xc0\xa1\x82\x01~0\x82\x01z\x06\t*\x86H\x86\xf7\r\x01\t\x061\x82\x01k0\x82\x01g\x02\x01\x010g0S1\x0b0\t\x06\x03U\x04\x06\x13\x02US1\x170\x15\x06\x03U\x04\n\x13\x0eVeriSign, Inc.1+0)\x06\x03U\x04\x03\x13"VeriSign Time Stamping Services CA\x02\x108%\xd7\xfa\xf8a\xaf\x9e\xf4\x90\xe7&\xb5\xd6Z\xd50\x0c\x06\x08*\x86H\x86\xf7\r\x02\x05\x05\x00\xa0Y0\x18\x06\t*\x86H\x86\xf7\r\x01\t\x031\x0b\x06\t*\x86H\x86\xf7\r\x01\x07\x010\x1c\x06\t*\x86H\x86\xf7\r\x01\t\x051\x0f\x17\r080725222153Z0\x1f\x06\t*\x86H\x86\xf7\r\x01\t\x041\x12\x04\x10\xcc\xfeN\xc7\xf2\xca\x9e[\xfdH\xa69\xb4m\x03\xb40\r\x06\t*\x86H\x86\xf7\r\x01\x01\x01\x05\x00\x04\x81\x80\x89\xc0\xf3\xc9\xfc\xe5R]3\xe5\x88E\xf3M}+\xc7\xcc\xe7\x87[\xc6\x99\xb6\xb8\x11\x8a\xf0\xbb:\x9c\x90\xc5\x1c\x0f\xfe\x1d.\xf8D\x87\x887\x12\x8f$%\xd7Lx\x15*\t\xf5\xb7\xa2~\xb0\x8d"y\xa1^\xc5\xd0\x91\xf72\x9cj\xa6\xb7\x809p\xce\xb9\x14\x9b\x87\xa8\xc5C\x14^\x88\xc2\x93Yw\xe9b\x86\x98\x95\x83r\xda\x19*NlIg\x1cX\xf1\xbf"\x83\x9d\xcc\xd2\x1d=\xe8\xb5M\xc2SW\xe7|T\xf4{\xc5\xc9\x00',
      )],
      'sha1':
          b'\x97\x8b\x90\xac\xe9\x9cvHA\xd2\xdd\x17\xd2x\xfa\xc4\x14\x99b\xa3',
      'name':
          'pecoff',
      'md5':
          b'\xe0(n\x9az\x8d\xf0\xd5\xd9\xea\x88m\xea\x8e\xe6\x00'
  }]

  SAMPLE_LIST = {
      'pciide.sys': SAMPLE_DATA_1,
      'simple': SAMPLE_DATA_2,
      'SoftwareUpdate.exe': SAMPLE_DATA_3,
  }
  # pyformat: enable


if __name__ == '__main__':
  absltest.main()
