# Copyright (C) 2014/15 - Iraklis Diakos (hdiakos@outlook.com)
# Pilavidis Kriton (kriton_pilavidis@outlook.com)
# All Rights Reserved.
# You may use, distribute and modify this code under the
# terms of the ASF 2.0 license.
#

"""Part of the service module."""

# NOTE: unicode_literals causes problems when using instantiation with 'type'
# so use 'str' which is a no-op in Python 3.x and converts arguments properly
# in Python 2.x
# TODO: This is a nested daemon factory class. Maybe we can enhance it?
#       This is also known as CRTP: Curiously Recurring Template Pattern.
from __future__ import absolute_import
from __future__ import print_function
from __future__ import unicode_literals
from __future__ import generators
# Python native libraries.
from enum import Enum
import os
import io
import re
import imp
import inspect
import types
import errno
# Python 3rd party libraries.
import jinja2

import crapi.misc.Utils as Utils
from crapi.service.DaemonNamingError import DaemonNamingError
from crapi.service.DaemonContractError import DaemonContractError

__CLASS_NAME_PATTERN = re.compile(
    "^[_a-z][-a-z0-9_]{4,30}$",
    re.IGNORECASE
)


class ACTION(Enum):
    INSTALL = 'install'
    UPDATE = 'update'
    STOP = 'stop'
    DELETE = 'delete'


class WindowsDaemonFactory(object):

    """A class that allows creating templated Windows services."""

    #FIXME: Use unassociated attribute for users who do not wish to use our
    #       daemon library for daemon creation but want to control it instead.
    #TODO: Use uuid and anonymous groups with dictionary lookups.


def instantiate(
    name='crapi', display_name='CRAPI: Common Range API',
    description='Dynamic runtime templating engine of Daemon services.',
    timeout=0, action=ACTION.UPDATE, action_file=None
):

    """A class that enables implementation of Windows services."""

    class_name = name.replace('-', '_')
    r = __CLASS_NAME_PATTERN.match(class_name)
    if r is None:
        raise DaemonNamingError(
            'Windows service class names should start with an underscore '
            'or letter followed by alphanumerics or underscore. The '
            'minimum length of the service name should be 6 characters '
            'and the maximum should be 30 characters.',
            'name',
            class_name
        )
    template_class = 'WindowsDaemonTemplate.tmpl'
    class_docstring = '\n    =================================='
    class_docstring += '=========================================\n'
    class_docstring += '    This file is autogenerated. Do NOT modify!'
    class_docstring += '\n    =================================='
    class_docstring += '=========================================\n'
    class_docstring += '    Part of the service/rt (runtime) module.\n'

    top_frame = None
    for frame in inspect.stack():
        top_frame = frame[1]

    cwf = os.path.realpath(top_frame)
    cwd = os.path.dirname(cwf)

    if action_file == '.':
        action = imp.load_source(
            'crapi.service.rt.' + class_name + 'DaemonUserFunctions',
            cwd + os.sep + class_name + 'DaemonUserFunctions.py'
        )
    else:
        action = imp.load_source(
            'crapi.service.rt.' + class_name + 'DaemonUserFunctions',
            os.path.realpath(action_file)
        )

    action_run = None
    action_advance = None
    action_bootstrap = 'def bootstrap(self):\n    pass'
    action_preprocess = 'def preprocess(self):\n    pass'
    action_postprocess = 'def postprocess(self):\n    pass'
    action_cleanup = 'def cleanup(self):\n    pass'
    if hasattr(action, 'run'):
        if not isinstance(action.run, types.FunctionType):
            raise DaemonContractError(
                'The run method signature is not a function!',
                'action.advance',
                type(action.advance)
            )
        action_run = inspect.getsource(action.run).strip()
    else:
        if not hasattr(action, 'advance'):
            raise DaemonContractError(
                'One of the advance or run methods must be implemented!',
                'action.attributes',
                dir(action)
            )
        elif not isinstance(action.advance, types.FunctionType):
            raise DaemonContractError(
                'The advance method signature is not a function!',
                'action.advance',
                type(action.advance)
            )
        else:
            action_advance = inspect.getsource(action.advance).strip()
        if hasattr(action, 'bootstrap'):
            if not isinstance(action.bootstrap, types.FunctionType):
                raise DaemonContractError(
                    'The bootstrap method signature is not a function!',
                    'action.bootstrap',
                    type(action.bootstrap)
                )
            else:
                action_bootstrap = inspect.getsource(action.bootstrap).strip()
        if hasattr(action, 'preprocess'):
            if not isinstance(action.preprocess, types.FunctionType):
                raise DaemonContractError(
                    'The preprocess method signature is not a function!',
                    'action.preprocess',
                    type(action.preprocess)
                )
            else:
                action_preprocess = inspect.getsource(
                    action.preprocess
                ).strip()
        if hasattr(action, 'postprocess'):
            if not isinstance(action.postprocess, types.FunctionType):
                raise DaemonContractError(
                    'The postprocess method signature is not a function!',
                    'action.postprocess',
                    type(action.postprocess)
                )
            else:
                action_postprocess = inspect.getsource(
                    action.postprocess
                ).strip()
        if hasattr(action, 'cleanup'):
            if not isinstance(action.cleanup, types.FunctionType):
                raise DaemonContractError(
                    'The cleanup method signature is not a function!',
                    'action.cleanup',
                    type(action.cleanup)
                )
            else:
                action_cleanup = inspect.getsource(action.cleanup).strip()

    # Boilerplate code time :P
    # Make sure that the service's python class file path is correctly set
    # since there is no __main__ entrypoint (BTW this is desirable).
    src_file = Utils.Environment.getSourceFilePath(name=__name__)
    src_file_no_ext = Utils.Environment.getSourceFilePath(
        name=__name__, extensionless=True
    )

    top_md = __name__.split('.')[0]
    proj_base_path = src_file_no_ext.split(top_md)[0]

    base_dir = os.sep.join(src_file.split(os.sep)[:-1])
    class_file = class_name + '.py'
    class_module = '.'.join(__name__.split('.')[:-1]) + '.rt.' + class_name

    dst_filepath = base_dir + os.sep + 'rt' + os.sep + class_file

    # NOTE: Workaround jinja2's weird caching behaviour :/
    try:
        os.remove(dst_filepath)
    except OSError, e:
        if e.errno != errno.ENOENT:
            raise
    tmpl_env = jinja2.Environment(
        loader=jinja2.PackageLoader(class_name, base_dir),
        trim_blocks=True,
        lstrip_blocks=True,
        keep_trailing_newline=True
    )
    template_file = tmpl_env.get_template(template_class)

    if not os.path.isfile(dst_filepath) or action == ACTION.UPDATE:
        with io.open(dst_filepath, 'w') as fdout:
            if action_run is None:
                print(
                    template_file.render(
                        class_docstring=class_docstring,
                        name=class_name,
                        display_name=display_name,
                        description=description,
                        timeout=timeout,
                        action_advance=action_advance,
                        action_bootstrap=action_bootstrap,
                        action_preprocess=action_preprocess,
                        action_postprocess=action_postprocess,
                        action_cleanup=action_cleanup
                        ),
                    file=fdout
                )
            else:
                print(
                    template_file.render(
                        class_docstring=class_docstring,
                        name=class_name,
                        display_name=display_name,
                        description=description,
                        timeout=timeout,
                        action_run=action_run
                        ),
                    file=fdout
                )

    return (class_module, proj_base_path + class_module + '.' + class_name)
