from imagehash import ImageHash, phash

import numpy


class ErrorThresholdReachedException(Exception):
    pass


# thanks to https://github.com/JohannesBuchner/imagehash/issues/20
def _hex_to_hash(hexstr):
    """
    Convert a stored hash (hex, as retrieved from str(Imagehash))
    back to a Imagehash object.
    """
    l = []
    if len(hexstr) != 2*(16*16)/8:
        raise ValueError('The hex string has the wrong length')
    for i in range(0, int(16*16/8)):
        h = hexstr[i*2:i*2+2]
        v = int("0x" + h, 16)
        l.append([v & 2**i > 0 for i in range(8)])
    return ImageHash(numpy.array(l).reshape((16,16)))


def card_image_to_id(card_image, preferred_hash=phash, max_error=None):
    """
    given a PIL Image object, find the closest Hearthstone card match and return its card id.
    :param card_image: PIL Image object
    :param preferred_hash: one of imagehash's phash, dhash, whash, or average_hash
    :param max_error: percent error calculated by (diff(input, closest) / diff(input, furthest)). if exceeded,
        raise ErrorThresholdReachedException
    :return: card id of closest match found
    """
    card_image_hash = preferred_hash(card_image, 16)

    sort = sorted(db["cards"], key=lambda x: card_image_hash - _hex_to_hash(x[preferred_hash.__name__]))

    if max_error is not None:
        percent_error = (abs(card_image_hash - _hex_to_hash(sort[0][preferred_hash.__name__])) /
                         (abs(card_image_hash - _hex_to_hash(sort[-1][preferred_hash.__name__]))))

        if percent_error > max_error:
            raise ErrorThresholdReachedException

    return sort[0]['card_id']

"""
when the module is loaded, load our database file into memory. the database is essential because it contains the
different hash values for each card.
"""
db = {"cards": [{"dhash": "468f087c05ef1c98bb605f0fbedc7401c1624084b4827d1ddb3a66560c87d13e", "average_hash": "f921fb03f80ff80ff00fc42ffc7efc7d9833b80018101816980798030108011c", "name": "Pompous Thespian", "phash": "c3f3735ada4bb6a5b7951dcf6ef992d1499266ce818c186609c960ba4ce40ca7", "mana": 2, "whash": "f9a1f987f80ff84ff40fe43ffc7ffc7fbc33bc101c10981698079803010e013c", "card_id": "KAR_011"}, {"dhash": "cae3ccce111c0a3135cf584bb0b6404d86b2314de3aa7c05dd390a708485652a", "average_hash": "779cffb8fbb8dcb18c79e425b56465506d404100c10458465806080101020100", "name": "Cyclopian Horror", "phash": "77c76731ef6173867e98cbb86cb218988c8c1cc40e93cbc473846be49ca1736e", "mana": 4, "whash": "7f9cffb8fbb8ddb98cb9ed75edf665516f414944c154d946780e448301820320", "card_id": "OG_337"}, {"dhash": "88810d1f30c4030074c2703d83fa3ff575c26f84df08fe3fe02f0c7d5bf8ffc7", "average_hash": "0f7c47e0e381c300000000000000f800fc00f801f019f87f807fa03ff077f8cf", "name": "Headcrack", "phash": "f3f53e443d027328c7e18c92032b8cf80b0b3a954b50933c7bec9ccaf39e36f8", "mana": 3, "whash": "ffff47f0e383c330002000801000f803fc01f801f91bf9ffa07fe07ff0fff8df", "card_id": "EX1_137"}, {"dhash": "fcccad39d2f3044679c086de09015356aeac5c1d8932714923b086e06947139e", "average_hash": "07110b632be7384fbccf87dbc343c3734316230723c313cf10c310c212061618", "name": "Trade Prince Gallywix", "phash": "9989bf0ec673fe983e0416911bc9462601e192a1ebeb5bd07c2c36b5db1ac76c", "mana": 6, "whash": "0f530f6f2be7386fbcdf87dbcb43c2534307031f239331df10c3384212879e1c", "card_id": "GVG_028"}, {"dhash": "feb8e84520be54eda3522f93d4c38d171b4d829a04a43f58d923b05748a1f16f", "average_hash": "07c6038e034fc04fc00fe00fe01fc00fc30df001f001f400fc00fc1039001900", "name": "Poisoned Blade", "phash": "7bd35d48ce2636335d1167ec36e4b1b61be784eda22c36b40df926c93324d620", "mana": 4, "whash": "07c60feef7cfc74fc10fe0bfe09fc28fe38ff001f001fc00fc00fd5039007ba4", "card_id": "AT_034"}, {"dhash": "3cc0ec3f217f4afeb5f853f1afe259c7b78e7f1dc62b3c4730b7e16f8389053e", "average_hash": "070103808101f007e007e00ff00ff00ff01ff01fe01fe01fe01fc01f801f811f", "name": "Wisp", "phash": "79f8d6508e172f819e810688290001a0129293e1ebf0d3e17f89ffd9ffcabeed", "mana": 0, "whash": "0f110ba093a3f087e007f00ff05ff05ff05ff01fe03fe01fe01fe01fc11f811f", "card_id": "CS2_231"}, {"dhash": "08f20c8ec1598bc5363a4d76ba0c74f1d992b36548cb18163734ea6884813137", "average_hash": "371f9b9dc1ade835783cec3e7c3e1c3e7c3e6c3d183918280020000000002102", "name": "Clockwork Knight", "phash": "e9976b96fe205f4b4b6887a08f0441acc3b086f0329c3a6cbe61dce1cf83ebe4", "mana": 5, "whash": "ff1fdfbdc3bde8b57c3cfc3efc7e3c7efc7f6c3d182918281020000001032107", "card_id": "AT_096"}, {"dhash": "f88ffdff20f00f00fc06904b30b6447cfbd0e6019827326c67c00c0e1f58f8b3", "average_hash": "0700e303f31f807f807f022c06640e003e090e300678067f023f263f1c39f83f", "name": "Inner Rage", "phash": "adf05612f9e5bf8806135e7034220651494804ac4b164fd95b623eb77bcdf73f", "mana": 0, "whash": "0700e747fb9fb57f807f127c06640e003e480e7906f806ff223f263f1c39f83f", "card_id": "EX1_607"}, {"dhash": "4afc8d641291040e387c72f88de85ad1b702e98406ab3d44c2338c6f5a8c8130", "average_hash": "f703f76ffbcd7049784078c0fc02fc00fc03d036000010033017a01ea01c0110", "name": "Baron Rivendare", "phash": "e369f7dafd199be03b16038d96e00958928c28699aa6d34d371936339b27b34d", "mana": 4, "whash": "ff13ff6ffbed705f7c40fcc0fc42fc40fd03d83600217013381fe03ea11e8114", "card_id": "FP1_031"}, {"dhash": "98ffed3c82f104e0bb0b673788d8536db7f263c5fc0ab01508322f64cc899133", "average_hash": "6707c367934ff04f004f00961c109001f801fc01f818301f801fb813b8033101", "name": "Ragnaros the Firelord", "phash": "cbad1dcadd906369b6430ca46f8c13965898c3c926d9ceedcd711e370ca3ccc9", "mana": 8, "whash": "ef17c37f93cff04f045f24be1c50b851f809fc01fc18701fa01fb817f9133903", "card_id": "EX1_298"}, {"dhash": "e83ffde3b091af6f1b9a9e003f63fc80c8d503fd7f02fa0606c81e33f57fec7f", "average_hash": "07008307f11f781efa13f80b726e767c006c00603c61306d10607e007e7df87f", "name": "Burgle", "phash": "e34a7b651fc1be382c121680930916ec8c46c738e6660cd363f933c71fc3ddbd", "mana": 3, "whash": "07008307f11d795efe1bfc0f7a7e727c007c2060b461b06d30607e007e7ffa7f", "card_id": "AT_033"}, {"dhash": "1adc6c6fe1864a79b5d6526da49c4f71ba6235c5ca0abf077037846c1899213e", "average_hash": "071183804389400050015003b809bc0b7c03fc0bf808f83bf83bb03f201f011e", "name": "Eater of Secrets", "phash": "eb693ec63c14b749deba993166e082e6a98866202a72b3ecce889b49bf817ea6", "mana": 4, "whash": "8f13cfa1c3a940a05009500ffc4bbc4ffc47fc0bf80af83bf83bb03fa11f211f", "card_id": "OG_254"}, {"dhash": "521cb933a6e778cfd1f4a388c7119c233047678c9e04b919de7346c68d0c737f", "average_hash": "07e001c200c600c293c3d38f9387338e33983389b383f887f0cff18b3702377f", "name": "Keeper of Uldaman", "phash": "d761dc3ea6fea43f6e1a33db3f9126c921bb8680119b08c431cb46d419338fb6", "mana": 4, "whash": "0de001c200c620c29383d38f8387b38e339c3399bb83f897f8dff14b7702f77f", "card_id": "LOE_017"}, {"dhash": "566389d7440cfb1876cbacc65cbcb8295072a0cc07011f0a9c34b8eb725363be", "average_hash": "635e21dfe09fe03e40384b38083818382d383d3838383c2c28a4603e600fe01f", "name": "Wicked Witchdoctor", "phash": "69573d2f59aec77bef6bde8eefc23e4f3036a3400c675b0e21d213c731281040", "mana": 4, "whash": "e15e31dfe0bfe0bf48b84a381a781cfc2d783d383c383c3c68a460bee02fc03f", "card_id": "KAR_021"}, {"dhash": "caf48ccd11828b7d373b4c22b8cc713df3f2868599cb411686376e6edc983931", "average_hash": "f70f3b1df91948186c1ccc1ccc3884382c383c087c03382138383819b8197900", "name": "Enhance-o Mechano", "phash": "639d7f4ee389d758fef039b05ae609a824de184426c69311fbc49b26cf027b2c", "mana": 4, "whash": "f71f799df999d8386c3cec3ecc3884382c38bc0d7c36bc2578393839f8197900", "card_id": "GVG_107"}, {"dhash": "0a81cd3f70fde3e4dfb9b7c27e0cfb01e0038007180f339f423e8efc34f0fccf", "average_hash": "07180340c387e007e007e01fe83ff87ff07f703330232072307218720060e867", "name": "Mark of the Wild", "phash": "b97e7b99fe63e6f52fcb8da14a18c49041380acc0bc132225b137e24fbd27e23", "mana": 2, "whash": "071c0360c387e107e20fea5ffa7ffc7ff87f707730722072307218720860f86f", "card_id": "CS2_009"}, {"dhash": "f280fd3f006e0f9c19b2b3600ce139c2d78c8e891c1031277302c6078957fabf", "average_hash": "070073407b84fc46fc4cfe0cfa00f829f271b273ba331803980310031003f00f", "name": "Soul of the Forest", "phash": "e300778edcffbe31fd581e471f3319f821ce0c5c03261ab44bd5135963a8fe04", "mana": 4, "whash": "070c73607b86fc47fe4dfe0ffa08fa29fa71fa73ba333a13980318031007f80f", "card_id": "EX1_158"}, {"dhash": "68cacd26121809e1fbd3b4ece111863f144719cac115b048c1bbd8f6ef0e9f9f", "average_hash": "f71fff76fbf5c8f140f85efa82efb3ce3f821b8202a0000000d800ce820f841f", "name": "Alexstrasza's Champion", "phash": "f9a1b7a63bbbf652db7068e82a8c06cc20e9321039d196016f417f11df13efb1", "mana": 2, "whash": "f71ffb76fbd1d0f144785e7b9effa1ce3f861b820ba0009000d800de828fc41f", "card_id": "AT_071"}, {"dhash": "0cfc0d09130004637896a21c45b086cb1bd37caef144c5089b3126e74c4cf3bb", "average_hash": "f703fb3ffb49f84038c01bc03fc02cc02d802d86258e188e98cc385cb801f603", "name": "SI:7 Agent", "phash": "c32bb68cbbe14916b38c4ce3db648c6d12c6a6b453b6b6e1798c9692e1ad94b1", "mana": 3, "whash": "f713fb5ffb49f8403cc03fc03e802ec02e822d862f8ebc8eb8ceb85cba01fc0b", "card_id": "EX1_134"}, {"dhash": "82c0ed77e0c7039f8c9f17243890402886f05dc1624a02cc37986c21d871feff", "average_hash": "f7bf21fc798e7c3efe0e7e1e3e743e601c60c878681830403c603c21b860fc7f", "name": "Competitive Spirit", "phash": "c7de4be6f9f9fcffbd6796b8694e1a648e522a7109d34be44b1452005b105680", "mana": 1, "whash": "f79f63f4799c7c3efe0f7e1e3e543e401e508c78681838403c603c01b860fc7f", "card_id": "AT_073"}, {"dhash": "00838d1fe0ecc1d919f805e087006f095c68bcd01921e0230ecc03800d82f02f", "average_hash": "f7ffc3e1e183fc03fc01fc01fc03dc478e279627f4030606632019001800fc0f", "name": "Enter the Coliseum", "phash": "935431ae6681bcc4cd4eed0123998e81091642f57b544ad753ad73685bebf39d", "mana": 6, "whash": "ffffc3e3f183fc03fe01fe01fc07fe4f8e27c627f6031607622018003800fe0f", "card_id": "AT_078"}, {"dhash": "28ccad18432b0446f81ba6f6490c92fb21dbc73610c56c0820b1cde603403fac", "average_hash": "d73bcf7ba373b467d4c24fd04fd30ff2bff1e7f1cfe00dc004c000c006800600", "name": "Shado-Pan Rider", "phash": "972cbb29fae83b772f904e8c27c630f040d18b06b9bc5799e6247c64d322d673", "mana": 5, "whash": "ff3bcf7bab73bd73d4e34fd06fd30ef2bff0e7b1cfa00fc004c004c002800620", "card_id": "AT_028"}, {"dhash": "8a7f4df8f3863f1ad965d08301c3203c60fa9e4cb91b6036d4e96f938ce43f4f", "average_hash": "0700c71f837fc0ffe3ffa1f30301c7000300390319e701e7036783c720c3f94f", "name": "Shiv", "phash": "b90a2ff5d9ebdede66f51eb0c39c32b92c4294c20bd4a47cbb019e084f24fc84", "mana": 2, "whash": "0700c39f837fc1ffe2ffe3f34301c70007003d0719e703e703ef83c721c3f94f", "card_id": "EX1_278"}, {"dhash": "0a005d1ef0e90cc3392c735ae684cd7192a30b4716be791c7638c8f0b1f1ffe7", "average_hash": "077c07c0038003c420c020c0a0c100e100e000e080e000e600e300f340f9faff", "name": "Conceal", "phash": "b5aa5ef5afd49ee62fd9939c633224ecc9601c9353506e344bc238c8cbc43c2d", "mana": 1, "whash": "ffff0ff087c083c420c630eca0e188e100e051e099e01de781f341f3e1f9ffff", "card_id": "EX1_128"}, {"dhash": "0adc2cf1d1fa8b0535324eecbcc87111a3324c6548cb9297a93676692c92f13c", "average_hash": "f713b3976b9e083f582adc38fc3df839ec195c191c090c20d838180408042104", "name": "Dragonkin Sorcerer", "phash": "23d7fb4cde349f9c33a433e34e6288e8318c99d009e196a47e496666778e6792", "mana": 4, "whash": "f713fb9f6b9e083f5c3edc38fc3df879fc1bfc1b1c290c20d83898142904211c", "card_id": "BRM_020"}, {"dhash": "4af88ce111870b5a36244ec4b84873f1e4b25965868b38174d36d36e209d153a", "average_hash": "f70ff11e791df81ef83cf8181c301c3adc36d81ed8104813c018c01e001a011a", "name": "Murloc Tidecaller", "phash": "eb67e3f0fb387b86dfe43ce01a333126985089a4329af6886b8358dbddc13690", "mana": 1, "whash": "f71ffb9efb9ff83ef83ffc381c381c3efc3ad81ed815c813c018c01e001a001a", "card_id": "EX1_509"}, {"dhash": "88f82d60c283450ebb2976de8d3c707966e2c1049b8b94177937f06e6a9c9533", "average_hash": "e70fe37f01df305e30d80c981c181c30383878387c3858385818801a8019a111", "name": "Hemet Nesingwary", "phash": "adaddf8a5b3263c6bf951c061240626083c4f274dba84e691c1eb639cfe3d3a9", "mana": 5, "whash": "ef1fef7f03ff707e3c780cd81c593c783c387c387c385c38d81a801aa11ba113", "card_id": "GVG_120"}, {"dhash": "ea83fdfdc01e86e31f443ff370cc7736d8cce4d91bf3810610c963b00d39fbff", "average_hash": "070003003b1e9c0b9c7d3e7fee71d677127f027a927f1e6e065996401670f27f", "name": "Execute", "phash": "afe0ce5f76bffeeb7cf03ebe215a8c51894393a2130c4e0cc9706d8403c27383", "mana": 1, "whash": "07000300391e1d0b9c7d3e7dee7dd677127b127a127f1e6f065b96401270fa7f", "card_id": "CS2_108"}, {"dhash": "cac2acfc11fe8bcf370f4f32bee07001c00ee4f5c08b0d17b7346a6f949c3939", "average_hash": "f700f1813b19c81ff83ffc3ffc3ff83f6018103818381830180d081d0819111c", "name": "Bloodsail Raider", "phash": "e999f3367e89b6b176e8639a4f6638c818660926cf02fec0ed44cf02ff016f12", "mana": 2, "whash": "f714f991bb9fc83ff83ffc3ffc3ffc3f7039103818381830382d181d1019111c", "card_id": "NEW1_018"}, {"dhash": "ea87fdff60a58f791a07341e68fcd0d8b1f179e3c34287870b0e1e19fc64f2df", "average_hash": "07200100f31a4c1e8e531c511c501c580c583c5c3c1c1c3e1e7e1c6cf04ff04f", "name": "Reincarnate", "phash": "4f2a3e3dfbf06e65f2c28e9283b911c20a6661ea4f503a8d0f07fb017f63b305", "mana": 2, "whash": "07300340fb1e6c1e9e5b3e511c501c584c583e5c3e3c1e3e1e7e5c7ef04ff05f", "card_id": "FP1_025"}, {"dhash": "c8f80d5302c6c901fb3ab7c5e877838f1c47231a463498c820b987f03701ffaf", "average_hash": "f73f7f7ffbffe5ff0cffc6dda6c363c263c8a3c883c002c000c000c003800623", "name": "Siege Engine", "phash": "7d68e9868b94ff62fb449ea00b812c4382dcb604b9f93e9173867686cdb99e61", "mana": 5, "whash": "ff3f7f7ffbffe7ff8cfbc6dd26c363c223c883c883c005c004c004c007a00623", "card_id": "GVG_086"}, {"dhash": "aacb2c27919d4233b5e073e6a79c5373bc4e4015886acbc5ffb935722b86472c", "average_hash": "d797efa3fba1f5a1fc40f8019d438943015301180114005c804e8007c1064104", "name": "Wailing Soul", "phash": "7bf3b7057f566cac9f84d94e26a441e0016190b00d736cc3de866f13f708e7ca", "mana": 4, "whash": "dff7efa3fba1f7a1fce0bcc19dc789c74b538150015401dc804f858fc1864325", "card_id": "FP1_016"}, {"dhash": "663e18f001e097012f0e5e1c8c7839a5714aa2c4d499213306660cc0f887ff3f", "average_hash": "cff7e39f803f003f181f005f0038d01cd03c801c000e000020002000e006fd3f", "name": "Forgotten Torch", "phash": "e9f7d98adb0c963237f3b688970db6b38576cc8ea518a632a746a200a6ddaa15", "mana": 3, "whash": "cfffe39fe33f013f197f117f017df17cf13dd01d881e200220002100e107fdbf", "card_id": "LOE_002"}, {"dhash": "8ae40ccd41308b4cb6b84c87bd7a7b85827a6495cdea33954532fa64e6891933", "average_hash": "f71fff99eb9b603b203040110438403d8001000000117c137810f81098111911", "name": "Tentacle of N'Zoth", "phash": "a3ad797cf395dbd87bd313c29ef112b2cce00918cc2c1b2c3b4863076722be63", "mana": 1, "whash": "ff1fffb9ebbb6dbbc43244310578517f814931509d517c537c11f81099113911", "card_id": "OG_151"}, {"dhash": "ccd86d60938d0d3376ec2cd15906e7fc1adb673182cc3c0b413cf4e0c345072f", "average_hash": "ff3f7f7e3b7c1df81cf0aff187f107f067f8f7d86fd01de01ce000c004800620", "name": "Void Crusher", "phash": "a7bceb81ca03636e79509c949314716084c31c69f9f1839b5c3be7d1cbc27cbc", "mana": 6, "whash": "ff1f7f7e3b7c1ff81cf0aff187f146f067f8f7d86fc01de01ce004c000800600", "card_id": "AT_023"}, {"dhash": "7a80fd3f00f701c09bbe375f78e451f8bfe0e0c1170b87f40f0ffc12e06fc07f", "average_hash": "070013403f01fc030604661c7e38be107c0c3838b0067042fc60fc10fc1cfc3f", "name": "Quick Shot", "phash": "63ee1ebdff77fc438dd137ac3ce028e01a4a0374235c4b8c5b039e255f0c6712", "mana": 2, "whash": "070813407f85fd030e44661d7e38fe18fc0d783ab826f042fc60fc51fc1ffc3f", "card_id": "BRM_013"}, {"dhash": "2ac4255a42f6c92c7fb3a45c5979b2cc64d2c9e61fc57c0af1309ae334476b38", "average_hash": "87100d68011340bd40de82f3d2f573fe7d327db1ffa1fca1dcc1688260084600", "name": "Brave Archer", "phash": "a7cebdf95ecab757eef41ba32791704c08f261101c918ac19b045f896f6293cc", "mana": 1, "whash": "87100b78035340bdc0fe8af7dab7f1fe7d327fb1ffa1fca1fc83688262084600", "card_id": "AT_059"}, {"dhash": "5afe24f0d1c40b39b7e44c83b9165209849a40f5a7cbe1167235786b2e92353c", "average_hash": "8703e58fc99fd83fc03fe03b8037601d000a0000802030281800c80968043114", "name": "Secretkeeper", "phash": "b9fd19655b97d6f376bcb2e8b33010c346a126e78e84c640bd09de8c3be29823", "mana": 1, "whash": "cf13efafcbbfd8bfc43fe43fc57f615f014a004091303c683c18d82d6905391c", "card_id": "EX1_080"}, {"dhash": "88f80cf2010c8b1136e34cccb9187637ef0e441d98ea6f4410b8e1634b869738", "average_hash": "f71ff39ffb9fd83fe831983390339036803b001b0002000100000004000c0110", "name": "Mind Control Tech", "phash": "f9b5eb244b1adbccefe016cab60d30c833a65a7299e11bcbcde366e061099664", "mana": 3, "whash": "ff9fffbffbbffcbffc339c339d779177d17f015b0142094100000084818c0118", "card_id": "EX1_085"}, {"dhash": "8affedff33628f853c41780cf890d67e81f9eac171130741638fc41f12fc7cfe", "average_hash": "0700010ef33f487f387718793078007c8e1d3e3e3a1f320fb80b0c0f0c12907f", "name": "Wrath", "phash": "89e8fdba7ade5b402e372ef9a6c006de30924cace32116654943eb72f3d07629", "mana": 2, "whash": "07000347f33f787f787f18791c78007e0e3f3e3e3e1f320fb80b0c0f0c16d87f", "card_id": "EX1_154"}, {"dhash": "cade6c7181460ab8360f57dcbff053a1ac02fb25e4cb46979d373a6d64968939", "average_hash": "370023835b0e180608073819381ff813dc17c01f483f683ee83ef83c501d1119", "name": "Evolved Kobold", "phash": "69c7de307ec97f36bf69a7a42c1930f282a4c4d061f247c2be84196573863ccb", "mana": 4, "whash": "a7122b8b5b8f38060817b81bb81ffc57dc17e01fc83fe83ee83ef03c501d511d", "card_id": "OG_082"}, {"dhash": "4cf08cc681990b33377e4cf8b9006229cc529be5318b4f12df34f46cdc997932", "average_hash": "f71c7b9cbb9838381838fc39fc33fc323c203c20382018203830383030137918", "name": "Emperor Cobra", "phash": "a7757f2179a15e127336b3e4338b0ca92d72999433639af93f879e391882a318", "mana": 3, "whash": "ff1fbfbcbbb838b83c38fc39fd73fc723d603c2039201820383038303933f919", "card_id": "EX1_170"}, {"dhash": "18b02d467299fd03723f846fc19d987b1fd779acc24005891e30bee07c88311f", "average_hash": "076f81ee83cd80d000d003d08bc01fc00bc06fa0ff987dc878c078c07b003f10", "name": "Shielded Minibot", "phash": "572f1df2e3c0597c9fc6c3826f60c29488c093251be6bac9df09a7853b1adb31", "mana": 2, "whash": "877f8bee83dd80f800f003d08fc81fc02f826facff9cff987cc079c07b007f10", "card_id": "GVG_058"}, {"dhash": "eaf20dc552080d70fcc8e1bd0701e43a0de5621e07781cc07239c1f20f449f1c", "average_hash": "f73fff7feb7dfdf914e203c203e303e41bc413c0e3c187c104c000c000800600", "name": "Druid of the Saber", "phash": "f7b3eb912b797bbb4f8298481ec0c4e002d136187e0c5f217e8e3c84bf03dbc3", "mana": 2, "whash": "ff3fff7ffbfdfdf914e207e60fe70af41bc41bc0e7c187c104c004c204800620", "card_id": "AT_042"}, {"dhash": "d8e4adc9c29005333b6e6ed89808709184e200c5d00b3907f6b67869bf96092c", "average_hash": "c71fe37fe1db987d9879bcbd343e583f8803f0017830383000300000c0084908", "name": "Captain Greenskin", "phash": "a99b39297e9bb6285b4a4d9067911a498c04c7e66ab2339b9f8976e69e234db2", "mana": 5, "whash": "8f1fe77ff3fb987f9c7dfcfd7c7f587f881ff801f830383000300001d1084908", "card_id": "NEW1_024"}, {"dhash": "8cf80d61d28f8d2ffb7de6e10984f7443f1df87ef0f9dbc3273f67f4ce481910", "average_hash": "7706134e0b4c081c209ef4bfc723a017e09f40be013e103d98bf881998181818", "name": "Dark Arakkoa", "phash": "4909ff68e683cd90f3a4933966e0712401d83e3626463b89efb944e77b1b3e66", "mana": 6, "whash": "77161b4e0b5c085c80fef4bfc737e4b7e0bf42bf01be103d98bf887d98181818", "card_id": "OG_293"}, {"dhash": "01c0cf1fb0b80af638e4519b9f0636c9601a933006e69c0ee0df437e8718fe33", "average_hash": "ffffcbf0f1c3f823e803f0077d1f0d1ec530072006400660023806f80ff00f38", "name": "Holy Fire", "phash": "2d7cd303b5093cc24fb6cd08d35824ac23060faee94197f553f0791b73ee39a5", "mana": 6, "whash": "ffffcbf1f1e1f863e803f807fcbf0d06c534079006c006c002f807f80ff00fbb", "card_id": "EX1_624"}, {"dhash": "9af8add302c63d187f75cec698097107623ecced1153038b723287e11c0c7330", "average_hash": "470a215e215eb01e80bcc33fcb3fc13f033b033b23be3198088018cc32003200", "name": "Archmage Antonidas", "phash": "ed0d39ef46eade7cbfcc11db3f100cc7107143829b86336d15db13654f225614", "mana": 7, "whash": "470f235e235eb07ea83fcbbfeb3fc33f033b033b23be39b818c318cc3b003300", "card_id": "EX1_559"}, {"dhash": "08f7ac0c6197423cb1f85669acc25aa5b7422485db2a91456cbaf46403839738", "average_hash": "c710c39943816009e019e01ef01af81bfc1f3809e00630132002a00200068110", "name": "Southsea Captain", "phash": "736977ca9e9696c58f3771a078b0031a23cd74cce2e4dbbe9b368c9b828918b4", "mana": 3, "whash": "cf96cfb963a17589f419e01ff15af95bfd5f7d59e1567153a012a092a1068110", "card_id": "NEW1_027"}, {"dhash": "dadc243181800a3f351b5ca6b95c63f9c4f201c5028b3f16b9361b6f769e7938", "average_hash": "37936db739ac18280c184c100c301c721c6018001c20dc20dc38783c791c791c", "name": "Bog Creeper", "phash": "e7df9efc6b9996f3dfa83986b63048889263cc612e4036926f01db44db226ec8", "mana": 7, "whash": "3f977fb73bac1db80c384c300d701d721d701d401d60dd60dc3c78bc79be791c", "card_id": "OG_153"}, {"dhash": "e8e1cdcb9216846939c7703c8ff85c69a382d04463cb4e148c315873b085413a", "average_hash": "37003b787bd8f840fc403c82d81ef01ff8177038780c780478047000e0000100", "name": "Eydis Darkbane", "phash": "e3ce8f2136616e98f93859368e13925cd33079c22ce6d394d6996cdb0cad09e7", "mana": 3, "whash": "3f1c3f787bf8f860fc423cc0dd7ef97ffd1f7d30790c78447c04f001f1009100", "card_id": "AT_131"}, {"dhash": "aabfddfc31852f105c00b03d2767ecdc80b11be32d5273a48c8f2482911dec2f", "average_hash": "0700870ff33ff07f807f0262466646640e601e606e616e1ef27830086400f00f", "name": "Commanding Shout", "phash": "bf4abdf26b7d179d16de7e6890868633213e269c61614e0c9b093cd24ba49f31", "mana": 2, "whash": "0710c75ff3bff1ffc07f026346664e644ee01ee0cf716e5ef668b0086600f00f", "card_id": "NEW1_036"}, {"dhash": "9cce8d1d023bc464b9a3716183da4715b4320e753cead9c5bbb163608787072a", "average_hash": "ef50ffe1bfe1f5c1ecc0ffc29f41d7c18f400701874181c584c1c1c305830320", "name": "Sylvanas Windrunner", "phash": "973be728ff189f816e6411614e94024c218c12f83bf3cecc5fe7ce636c632627", "mana": 6, "whash": "ffd1fff1bfe1f761fcc0fdc29f41dfc38f400741874181c584c7c58305a20320", "card_id": "EX1_016"}, {"dhash": "68dfec2e91f14b9eb7f84ca5a966560999f233e5c78a8f153536de6c689bf93e", "average_hash": "0700818093813010e018c03bc017e007f80ffc0cfc1cf8187831b812b816f91c", "name": "River Crocolisk", "phash": "e3b73cdcae35479b5e62948429eb0c60659846b486449ec137493f9b7742fe2c", "mana": 2, "whash": "0f108fa193a130b0e038c03be457e84ff84ffc0efc1dfc18fc3db812b916f91e", "card_id": "CS2_120"}, {"dhash": "0383cd3e70e891d03f014e4c9c98392173c28904030d261b1036626ecd98fa3f", "average_hash": "ffffd9ffe3cff81ff81f7dbe391c19381118410040004000000000000100e183", "name": "Resurrect", "phash": "61df192a399e96e4cdf9a6e9636c2c3c4b3222914b945e955bc97ba05ba25b28", "mana": 2, "whash": "ffffdbfff3cff91ff87f7dbe39bc19b9199d719040804180008001800180f18f", "card_id": "BRM_017"}, {"dhash": "9cdf2c3ee1fd4ac3b7804f53ba3e7971c2c20485730acb14163b68767088c136", "average_hash": "c713e7b783afc02f703e003c043f1c7f783978037804780c701878107910c110", "name": "Silverback Patriarch", "phash": "63dd19443e06d38a9f852d8d5eb2e2e021499be618eb36fbedb36d25522a6360", "mana": 3, "whash": "cf97cfbf83afc4aff03e003c057f1d7e7d79794179447c4c781878107910c110", "card_id": "CS2_127"}, {"dhash": "b88d6f3bc0ed38b37344e71886e11e436996022cc8d090892332cfe5998d3310", "average_hash": "47e043e243c1b8c1f8dd3bcc3bccebbea3bd439d0399019800c011d6131f1719", "name": "Twilight Darkmender", "phash": "dd9537c696e03c0c3f1446462ec9a19921314a2756ba2e19ff373a856b639ed5", "mana": 5, "whash": "4fd04be043c1b8c5f8dd3fdd3fccebfea3bf439d0399099800c011de131f1318", "card_id": "OG_096"}, {"dhash": "1cd9ed3e022344ccb91963f19f06772fa46ae0dd1babeb571637266ccc98f937", "average_hash": "e707e366bbc7f043e0c64084e033a037801b80187024703e3838183018100913", "name": "Illidan Stormrage", "phash": "fba45b2a87076f129e01259072cc06c3192c921cab49c7b9dfd48dadf9b8b35b", "mana": 6, "whash": "ef17ef7ebbf7f443e44744e5e173a17781378019f126707e3c38383019104913", "card_id": "EX1_614"}, {"dhash": "c8fc8d6103c74c007b26a01d4cf799022705089e1174afc8cb3326e4c443bfb4", "average_hash": "771f3f7f737f707e18f813c073c8fbd983db03c223c003c018d008c002840604", "name": "Dark Iron Skulker", "phash": "dda9eb58cb9afe0c3b00268116309c8521e5720a66e65f72ffa639e73632cd32", "mana": 5, "whash": "ff3f3f7f7b7f747f9cfd17c077d8fad983df0382238001c01cf00c500a8c0600", "card_id": "BRM_008"}, {"dhash": "6ce44998923905f33a84650a9b7c6a91c742f704e84b9897e037136f2c9c1938", "average_hash": "930f130b99179813b017b00fcc06f802f803f00be80df81fc81fc81ff81f701e", "name": "Icehowl", "phash": "c9ef96b0cd813b49f6480d7e6b1899b1c26463361e7e1b356e1263cb93191d96", "mana": 9, "whash": "870f115399179853b817b81fcc0ff802f803f80be81ff81fc81fd81ff81f701e", "card_id": "AT_125"}, {"dhash": "8af905e202800532bbc86c119b0044e7980e3c1d002a8cc4c8b181636383373c", "average_hash": "f70ff15ff15fe03fe01ff03fe03fe00fe00fe00fc007c003c000400020042104", "name": "Baron Geddon", "phash": "79fce9a967b7726acf5236d69a90661834698e842b9613c6d624dab03c09772b", "mana": 7, "whash": "ff1ffb7ffbdff07fe07ff0bff87fe01fe00fe00fe007c003c00060002104210c", "card_id": "EX1_249"}, {"dhash": "6afb4cf491e38b05363f4cb2bc747ec9f092c125a6c90e937936e36c86991532", "average_hash": "d702f31d191e383e383c78387c386c3c6c3ecc3ecc3ce831c833881380112000", "name": "Chillwind Yeti", "phash": "2935cfc2dae133f26f6833cd36d81826c2980cce9b649ae4cb249e37b6619e66", "mana": 4, "whash": "f712f99d199f383e383c7c387c3a6c3e6c3eec3eec3cec31c833c833881b2010", "card_id": "CS2_182"}, {"dhash": "7a00fd1e30ffe3ce076f0cbc143e6b608ec01c001b80f732df8f3cf3fde1d087", "average_hash": "073013803b003423fe1dfe09fe0b7e033e033603c003e003f065f070f800fc07", "name": "Ancestral Spirit", "phash": "c3700e55f4fffcea8f2d65821a4f2db00938384dc9f0c26c9ec7c7036b017727", "mana": 2, "whash": "073013c03f003d23fe1fff1ffe0ffe073f073e03c003f003f065f071f810fc0f", "card_id": "CS2_038"}, {"dhash": "eadf8527027c3d187ba2c5608c9971735f46b14d62934648bcb1d8e2ff07fb2f", "average_hash": "f703f163f941f045f0b87b9c8b1d0301b32b4300438063c060c021c07b00f30f", "name": "Flame Leviathan", "phash": "738b33f7f3f8d6dc9f59ccb92fc4086611301e031a6a6b0ac43c660c3ca39b79", "mana": 7, "whash": "ff13fb63fbc1f06df2f9ff9c8f5d3b01a32b4b00438063c020c021c07b08ff0f", "card_id": "GVG_007"}, {"dhash": "eaceac1d41f30ac4b7914f61b99e661dd972f2a5fd4b3297f5b682690496c92d", "average_hash": "1710038303830007001dc01e803780278c3c0c3e2c3fe83fc03fd00fc807c105", "name": "Midnight Drake", "phash": "9997fe2cf6c9cf9d763371e26ca446e222b2c24c99e106c66958d6d8bf803989", "mana": 4, "whash": "1f100fb303a300b7803dc03ec47f84678c7f8c3fac3fec3fe03fd00fd90fc107", "card_id": "OG_320"}, {"dhash": "cce08c8111230aceb57c5793ae645df1abaa3015410a9a45a6b94e76b9842139", "average_hash": "f71ffb9ff99b780278083c1d2c1d881dd00f7008700020013001300870006100", "name": "Sunwalker", "phash": "e3d9c331f705fb8039861998661861e018e61666de0c93bc7d73d9cbed81f38c", "mana": 6, "whash": "ff1fffbffbbbf8827c3cfc1d2d5f8c5fd14f704e710830411001301971007108", "card_id": "EX1_032"}, {"dhash": "3adc393042c3cc86b91c77718ac274bf593aa4f4c08b87170d37fb6cb6903f21", "average_hash": "cf7e4f7f0fef05ee24cc63fc037803fc03fc0d3e1d781df80c780db005a00320", "name": "Deathwing, Dragonlord", "phash": "4dfe5ba29f0c8f473fd326e566f1207f247b26f32ea773c107811a411b0087e1", "mana": 10, "whash": "cf5f4f7f2fef05ee24ee65f403f003fc03fc0f7a1f781df80c7805b005a00320", "card_id": "OG_317"}, {"dhash": "161c817124e77908f218e661cc93196735ce669ccf30974938b361e49f00437f", "average_hash": "fbe73bc700c6c4c603834380c380e3806380338833886088e0c801c10300077e", "name": "Medivh's Valet", "phash": "7529d9ae89eef33dff8a36dbffc8b60121ee326301f762448cac227888990e11", "mana": 2, "whash": "fff77be730d6c4dec7cb4780e389e388e38833887398f0c8e4c811c123804f7e", "card_id": "KAR_092"}, {"dhash": "ca1ffd7fe0b003c3dd9a37fe7fb8e540fe11f1a343468e0f04ce38900130fb66", "average_hash": "0700e107f103fc03f8007e007e75fc797c7e0c7e247ce07c207e003e00442006", "name": "Frost Shock", "phash": "a1aa92fd7a752dfe8f1d6c9226ea0c66a3e30eb90f5d0e8562cc362259c3ec44", "mana": 1, "whash": "0700e307f103fc03fc007e407e7dfc7f7c7e2c7e247c607d607e007e005ef06f", "card_id": "CS2_037"}, {"dhash": "0af86de292c40d00ff23ec0e0cf5f9c2331d007e09d8f082873a1ef7784ec314", "average_hash": "e71fc37fcb5f807f00fe0afd1b3864bcc29d028801800904188450dc600c0004", "name": "Klaxxi Amber-Weaver", "phash": "6987692bebf2dfce7b48c7389e9010b008d462e08ef1bb81b386567c7c41fe64", "mana": 4, "whash": "e71fcb7fcb5f807f00fe0eff1f7c4efcc2bf028d0380092e38d450dcf01c800c", "card_id": "OG_188"}, {"dhash": "12163d7924cb6894f13987364c7990e327c64d18c3440c5b3bb260a2814d86ff", "average_hash": "c561c1c1808d608d428c63847a907890799173926298c090d081c003c203e47f", "name": "Tunnel Trogg", "phash": "7391ce9e19fab67de7ecb2ab2d86fa9805f36646a9e20e76254c6632146106c1", "mana": 1, "whash": "cf41e3e1e2edf48de28d6b9c7ad079d079d37392ea98c098d081e003e203e4ff", "card_id": "LOE_018"}, {"dhash": "0af038884111c323b64e4dffb91a6325a4cacc350b0bd446a9b10f633987d53c", "average_hash": "f79f6fbf07bf01b10431043001730173207070387128004c000420043104211c", "name": "Sea Giant", "phash": "eddc9dc55b024b4a7b2819e667d662fe10b393efb1d973f8272113609b809360", "mana": 10, "whash": "ffdf6fbf3fbf05b504f1057007f301f323707178797849ec004c34843586233c", "card_id": "EX1_586"}, {"dhash": "4ae68dc802040d2cb8c575d08f0a723740eebddcfb6b0cc70836536c28891130", "average_hash": "f71df77ffbdbf0d860c03c800411003400200028c03ec03ee010781838103900", "name": "Twin Emperor Vek'lor", "phash": "e3e9fbd963697942ded31bb24bb24618761659a26ec44626cf646c849904776c", "mana": 7, "whash": "ff5fff7f7bfbf47a6c413ce41559117019a0012ad13ee87ee830781879103900", "card_id": "OG_131"}, {"dhash": "2cc32c8841428b36b6c44c01b0da6375cfca703d81aa22c75fb43b6836812d26", "average_hash": "fffeffbfffb7edf974f83ff01ff007f0177f03fe03f001f000e00de00d800b20", "name": "Ironfur Grizzly", "phash": "adf473855b0759e2df4216422761497868f00e6f0ca3d9f2bfd0f67c6848d2ca", "mana": 3, "whash": "fffeffbffbb7eff174f935701bf001e0177f01fe03f001f0007005a00d800320", "card_id": "CS2_125"}, {"dhash": "28decc3801f70a6cb58c5379a6825c25b10a7365c68a3d955936c96c1499613e", "average_hash": "871f838f010760076003700cf00cb01fb01cc01dd818f817f817f813f802711e", "name": "Ice Rager", "phash": "6b2dded89f441d236d33c3b06e6230c889ec33654bda7676ae698dc999a82632", "mana": 3, "whash": "c71fcb9fc9876007700f700ef004b05fb01cc01ff81af81ff837f817f912791e", "card_id": "AT_092"}, {"dhash": "6af9ccf091e78bcf37975940a69848e5b71a6e65b0ca6114833932730c863928", "average_hash": "971e1f9e1b9f983ff81ffc07f403f001f401ec0f9c07dc061c00980018001900", "name": "Faerie Dragon", "phash": "cb67d964ce415fe77ee806f20ba6316020a21c10bc08ff98bf03ee0c7f8cbe2d", "mana": 2, "whash": "9f1e9fbe1bbe98bffc3ffc07f447fc45fc41fc0fdc07dc071c00980419001900", "card_id": "NEW1_023"}, {"dhash": "8ad6a42c41c18ab435e85b11b6026d9da232d8f599cb36974136996463818725", "average_hash": "f79fedbff9affcadec3ff41f404f406f083104300d330c3304384010c1004105", "name": "Goldshire Footman", "phash": "b9dddbdb73b1afb57ed26be0cf8426a0248046426e8433319b45d9523b86ce89", "mana": 1, "whash": "ff9fefbffbaffcbfec3fec1f554f416f095105700d730c730038401081004105", "card_id": "CS1_042"}, {"dhash": "0a87dd78f060c34b3c8e62796f80bc36c7fc3b33b62a0cdc09e3e292e538f9ff", "average_hash": "077e07e3038f1b261c462c02a60002200262c6f6c5ffc0c3d0001a000220f27f", "name": "Gang Up", "phash": "df66de1cefc939e6bd1826d9113e86a6a0460db161e03e48cb853e075b0aff0c", "mana": 2, "whash": "4fff87f333de1ba61cc62c02a6c2826802e2c7fec5ffc5c3d1001b000b30faff", "card_id": "BRM_007"}, {"dhash": "cace043b013e0af035285670a82c5ed9f9b2f745e94b26970c364c6c128d7538", "average_hash": "f711f981f980f801b80c7c0cfc0058107c387c3c3c3e383e281f601f201e011e", "name": "Abusive Sergeant", "phash": "63dfb3f3fb79bfd43ee39ea44d30249999c1902412c656466c604c1bce10ef64", "mana": 1, "whash": "f711fba1fba0f801f81c7c1cfc527c5c7c387c3c3c3e383e281f601f211e211e", "card_id": "CS2_188"}, {"dhash": "ea3ff8ff61c0ef011f5e3c3e7a64e4d889b1f367e7821f0dfe1bf03f0060187e", "average_hash": "03008107e00f303c183808726c7468647824783c783cf819f81ff81ff007f81f", "name": "Ironforge Portal", "phash": "e18edcfcba7173df1e9eb6eca6ed8cf3e6616438691ab2660906190293a502f0", "mana": 5, "whash": "05008107e01f307c187c08766c76686678647c7c7c7cf819f83ff83ffc0ff81f", "card_id": "KAR_091"}, {"dhash": "7ad94cf6d1f30b513700524eacf85d39b2b247453a8a7c17f83780680093112f", "average_hash": "8703c187fb1ff81ff81ff80ebc08fc0fbc09c801d807e007f007f003b0062107", "name": "Pint-Sized Summoner", "phash": "935f3fa5fcb43ec69ec91e6096a002600978d3e0b68d6e30fb009bd9bf09e633", "mana": 2, "whash": "8713e98ffb9ff81ff81ffc0fbc08dc0fbc09d803d807f007f007f00bb00f2107", "card_id": "EX1_076"}, {"dhash": "8af30dc59283053b7a46a4895826b14b48f780ce8735094b52b0b6e67f49b7a2", "average_hash": "771e3f7d1b7f187b98fb5bf907f807fc07f00be033f011f000d000d002002602", "name": "Southsea Squidface", "phash": "eda56b8bebea6e127b1e6de463d031ce107c222c52ae3e4439a592869ba6d2f0", "mana": 4, "whash": "7f1e3f7d1b7f397b9cfb5ffd37fc06f80fec0bb033b011f000f0007002a02602", "card_id": "OG_267"}, {"dhash": "3cc02d20028fc4f8bb9367419c9670bdc542811c822b184770b4816b0f87373c", "average_hash": "870f07740be08458c0fe08fe847f857e803fc03ae03ca039802b000e010e0114", "name": "Onyxia", "phash": "3d363f8ade0ab3c0af1499106b814be633513e303aa8cf9b4f934ddd298953ce", "mana": 9, "whash": "8f570f740bec847cc47f0cff857e857ea93fe13fe13ca17d803f008e018e011c", "card_id": "EX1_562"}, {"dhash": "e8d90c3211f08a6035455b88b4b86867d68ecb3d86e938c323b6076cd6982d3b", "average_hash": "f71ffb87f987f8079c032c09046030603020e03080310030003800300019011b", "name": "Blood Knight", "phash": "b95dd342db85b1b4fbd04cf1199b068cc9681c733ca6de5cc633396b94861ca5", "mana": 3, "whash": "ff9fffaffbaffcafbc873c2d156931687172e17081710171003804b001bb011b", "card_id": "EX1_590"}, {"dhash": "b8b76d4e8395047bfbe4a6984d41b006616f841e89b530cb45b609e717484fb0", "average_hash": "c718eb6ddb4fd073f0f9b3fb23fb82f803fc03be03ba00be00f6005c001c0604", "name": "Shady Dealer", "phash": "2da1b34b5ada7cd62e824d8ccf01663986f62e64cee4cf999f99d87149234472", "mana": 3, "whash": "e718cf6ddb6ff073f4f9b3fb22fb82f803bc03be03be00be00fe007c029c0604", "card_id": "AT_032"}, {"dhash": "0a80cd7f30fa43e097c37f0e35b3ec0cdaf3a0c74e6f3a9f4d5e9abcf07b0ee4", "average_hash": "0700030081014007001f081f002e006d0c7c0c7c00780c7b487efa7fe07ff47f", "name": "Soulfire", "phash": "a9de5ef9b3a5b66f4f5acda68c7d29860dd83190c683192d63013333d39059e1", "mana": 1, "whash": "073803c0c3034017003f083f007f046d0c7e8c7c147a4c7fd87ffa7fe0fffc7f", "card_id": "EX1_308"}, {"dhash": "00c0cd3fe086817daf7b1fc3bca4d10832036cf8d9e18115896b12d6a42ce919", "average_hash": "f7bf03e001804018e81dec0ffc3b8c59049f001f060f9c5e49de595c5b58da1f", "name": "Duplicate", "phash": "4989b43c66c3dc2c3d99a7860b1068c10beda7c8d8dd4f6753f25889d3f7456a", "mana": 3, "whash": "f7bf02f0518c4018e81ebc0ffc7bce5b041f021f0e0f1e5e485e595c5b58da1f", "card_id": "FP1_018"}, {"dhash": "929e0839a1fe1cbc39f157c62b8147129b2468fdd082c31d44ba0c633f95f17f", "average_hash": "e7a1f3a3f107f847e807e807f807e007d80f900618023818000c200e1102817f", "name": "Huge Toad", "phash": "d3de33d6ec54ce79bfad279c2dad1a66219bc6f848a206e105920ef025a4cf16", "mana": 2, "whash": "efa1f3a3f6a7fc47ec07e847f847f847f84fd0473846385c201c200e1102c3ff", "card_id": "LOE_046"}, {"dhash": "0008ddfff0ffe36e940c2ad081f03fa1eca313872f48ffd1ff27ff0fc217bcbf", "average_hash": "073003c0030fe00ff007f001f803f803fc67f801fc01f400fc01fc039c07dc0f", "name": "Lava Burst", "phash": "835ffc800c07e66cad0306e8623c2782a9f49a39e38dadc46bfcda0f7be41c37", "mana": 3, "whash": "0f7807c0839fe11ff007f803f803fc03fc67f801fc01f401fc03fe0fbc0ffc0f", "card_id": "EX1_241"}, {"dhash": "03c00d33f0e4d981bd07531fbc3e78f9d072b6c708ce331c66314c6a8190ff27", "average_hash": "ffff0bfe03ce003e006e01fe01fc1dfc1dfc1dec00e708c380c3a0e101e0f1ff", "name": "Embrace the Shadow", "phash": "cd5f4d22f3e497ad679737b983b023da034db0f05b485b255b505b905b025eaa", "mana": 2, "whash": "ffff0bf603ce003e006e01fe01fc0dfc1dfc1dfe08e708c380c3e0e101e0f1ff", "card_id": "OG_104"}, {"dhash": "96392162648cf9497390a620dde10bc717077e0ca8305c48f8b1e3e7ef0f0b3f", "average_hash": "c3c7e1cfe0cfc08ac39a639a6398e39be38be38fc387c087c0838083c31ff71f", "name": "Ivory Knight", "phash": "7945656e89fad09aab19d43eef44bf0700698c11849f1252d68c7bcd9e0f63f2", "mana": 6, "whash": "c1c7e1cfe0cfc08ac39a6398639ce39be38be38fc387c087c083c083c30ff71f", "card_id": "KAR_057"}, {"dhash": "bac0ec37c1640ac8b4da53f9a5a242c7bf2e0edd58eae297c932866509820d34", "average_hash": "47944fb47ba6dda64442c043e341f140f1579103110308478c17901781070104", "name": "Bloodsail Corsair", "phash": "dbe677d60f97dfe577b139980ec819e10586c2188b613343fbc05618a7093bac", "mana": 1, "whash": "4fd44fb07ba6dfa644c3c543e3c1f1c0f357914b11470de78c579497858f0326", "card_id": "NEW1_025"}, {"dhash": "c8c30d2662ccc531bbe46441819e5e6173c2c894950b4b175e36986c92993533", "average_hash": "7700fb60c3c4c04cb4f9a4d384433c08383b303c3834783c703e603220132113", "name": "Gruul", "phash": "e73d37e3764ce794ce495360879028ed63591ee21eb238ec7f0d4e633c0b908c", "mana": 8, "whash": "7f10ff70e3e4c47df479a4f38547bd5a393b383c39347c3c783a603a21122113", "card_id": "NEW1_038"}, {"dhash": "6af04c2091d00aa7b7184c7fbf7c71c9e002c10500aba5467f32b46ec89f3138", "average_hash": "d71fd98fd9851827303fc834fc3e783f6c383010100010005012b012901b3119", "name": "Lowly Squire", "phash": "e9f75bd3dbf3dee26b3f39643c300828698c99d153320e337602dbd0251929c3", "mana": 1, "whash": "df1fdfafdbad18a5343fec36fc7efc7f6c783832102010005016b012b11f3919", "card_id": "AT_082"}, {"dhash": "0cc9093742c0c433fbc7a43c194966c6889e356169c7c38a97393861644e9b3f", "average_hash": "77065372c3c7005920f81ab058324264400c4f0f4d8c7c8c7cce68c41010321c", "name": "Lord Jaraxxus", "phash": "6f9ebe64e7295b1e3ec0c7268f613c6c41d83cc74bae92c1599e93f13908c524", "mana": 9, "whash": "ff165f72c3e70479b4781ef25c724264460c4f0f6f8c7cad7cce6cc410b0361c", "card_id": "EX1_323"}, {"dhash": "9ccd6d1a823708ce78eea311663387c6270e4cdf1834f648a8390fe37846b731", "average_hash": "a73bef723be379e37cc7bac7b3c723c38101c38163c030c400c400c402000600", "name": "Stablemaster", "phash": "d7889b0c2e6e3e473e11cd986b881137887ce66c1ee79bd1cfd947066c3a1899", "mana": 3, "whash": "af7bef727be77fa37ce7fec7b3c733c381c3c3c173c031c400c404c406a02620", "card_id": "AT_057"}, {"dhash": "0ae2858c423bcd66fecf2cfe5838dd7833e7c6c89c85610bc33e17e6884c1333", "average_hash": "371c1f780970007004b00eb01c300cdc18189bb398b3183f1cffb81f901b8017", "name": "Voidwalker", "phash": "8dad76ba7a7adbf16f9b49ca93d4234948e0991092b4e6044d197d24feca1393", "mana": 1, "whash": "371c1f580b7000f004b01eb01cb00edc1cbdbbb39db718bf1cffb8dfb89b8217", "card_id": "CS2_065"}, {"dhash": "0a80cd3330ef01fef7e47f599c526eb4dd4c86b038016703ce063e0db898e02f", "average_hash": "0700030033002000e006300fb42fd40efc0ff623fe07fe17fc3ff81ffc0ffc0f", "name": "Tinker's Sharpsword Oil", "phash": "e3707c68ef8f9ea01f4be9f09fbc318809a683568ed8e32149b293e4b3436336", "mana": 4, "whash": "0700030033002100f00f700fb42fd42ffc1ffe23fe07fe17fc1ff81ffc1ffc0f", "card_id": "GVG_022"}, {"dhash": "0080cd39e0c6930d275acefc98df63b446228dc192c317c16c86987d7163cf0c", "average_hash": "ffbe03e01186ec0cd8087838f930ff021c331e120e107c107c12c45ec246f20f", "name": "Polymorph: Boar", "phash": "43c6bc0979b28f2c9f21a73464990dc6035979828b6df3d8cd6ddba4dab51923", "mana": 3, "whash": "ffff03e13186ec0cdc187838f930ff329c231f120e507e14fc12cc5ec646f20f", "card_id": "AT_005"}, {"dhash": "ca1ffdfb00c00326d62c3c5979f2f2c4f681ed33d9cba1074b04921c663fccae", "average_hash": "0700c107f11f703f30323831fc70fc3a7c3b7c3fbc3fbc3fdc07cc02cc02ec0f", "name": "Rockbiter Weapon", "phash": "abfcfcb95e76dbeb36ea36338e99268c2e45a41e658327c84b6563044390db04", "mana": 1, "whash": "0700c107f11f703f30323831f870fc3a7c3b7c3fbc3fbc3fdc07cc06cc02ec0f", "card_id": "CS2_045"}, {"dhash": "0280a51e60e581df3f677f3cfcdaf985ff8bbf671f8e0c7c201880640780fc1f", "average_hash": "173c01e00182000038043c1cfcfaf4fdf4ffe0fff87ff87fd63300180000380e", "name": "Arcane Blast", "phash": "a1ffde7a5e1d39ece5eba7e14b1f2c9883d1c2865b60427549e1138853ba1b20", "mana": 1, "whash": "77be01e001820000b8243c5efcfef4fdf4ffecfff87ff87fd63b00180000380f", "card_id": "AT_004"}, {"dhash": "e8dcad20434ccc37f8fea1c943bddffa37a74f0980b431491cba80e1f347272e", "average_hash": "171f8b7f0b6404e01ce01ae11bc1dadffbdffbdf039100d800d000c0c087e607", "name": "Darkshire Councilman", "phash": "9f81cf037e567b251b859e619ed193e10c8c0c7cf6f8dbc86ca653e4666284d4", "mana": 3, "whash": "1f1f8f7f6b6404e014e03ee11fe1dedffadffbdf83910098c0d100c0c087e607", "card_id": "OG_113"}, {"dhash": "8087edfbe0e7738f0f5a1a641c987970f6b8c96193c33e077d0d7258c09008a7", "average_hash": "070001c0711ffc3f7e3c5e1e9e191e185e3a5e324e3aec3bf83ff033c023e003", "name": "Effigy", "phash": "2b454f20f098bd0c96211e012e042790cd708e774b7b9bed4bcd7b9ccff37b18", "mana": 3, "whash": "0708c043711ffc3f7e3e7e1e9e191e185e3a5e324e3aec3bf83ff033c023e001", "card_id": "AT_002"}, {"dhash": "c081bdfde0b30128b7d006a3046b3cdefdbc3b70f4e52d03d386f40dc91784bf", "average_hash": "073063c061183c181c18461e420e4206e247fe07fe1ffc07b807f403fe03f807", "name": "Lightning Storm", "phash": "4b206c0ca63d97b27c821e420def07f383274f13eb413f6c69f8399433d794ad", "mana": 3, "whash": "073063c075187d1e9c19461e420e6206e667fe07fe1ffe07b807fc03fe07f80f", "card_id": "EX1_259"}, {"dhash": "fafffdf3e3870f3e1ee4998b3f1e6238c4e159c1108301069f8e2f0d000cf83b", "average_hash": "07004102791e7c78fe71be771e7f1e7f1e7f1e7f1e3e1e1c3e1c1e1c0e180008", "name": "Bolster", "phash": "8f20f17e7adfcef6dcc81c30b2e43790347b0e6969984e4cd3123e04cbc73e09", "mana": 2, "whash": "07004106791e7c78fe71be771e7f1e7f3e7e5e7f1e3e1e1c1e1c1e1c0e080008", "card_id": "AT_068"}, {"dhash": "c283fd7fe0fcc3f0cfc39f813c173bbe6d3cf278ccf999e327876d0cd310e62f", "average_hash": "07180340ff87dc1f8e3f063f073f023f823fc01fc01f863ba219e205e001fc07", "name": "Swipe", "phash": "1b119feaf6bef46ffc62ec980d682ab209f002fc4bef33dd4bb2928c5b003340", "mana": 4, "whash": "071c8360ff87dd1f8e3f0e3f063f033f823fc81fc41f873be219e205e001fc07", "card_id": "CS2_012"}, {"dhash": "9af0a52552ca8c373b72667498e86b51d29284a4694bc296bc37676ef49fd934", "average_hash": "e70f616f29c51848585858bc583078097c0c2c242c212c240839b839d81fd90f", "name": "Hogger, Doom of Elwynn", "phash": "6bdff7aa7933fe1c7be62d651aa609412348064196114c92f3191b33f634f329", "mana": 7, "whash": "ef1fe97f2bc5184858795cbd5c387c297c0c2c242c212c2c083bb83fd91ff90f", "card_id": "OG_318"}, {"dhash": "e8f18ce741ee8b87363a4ff0b0184879b3d26505ff0a6c448933c36f668f0d3e", "average_hash": "070c130c811fc01f783c783cfc0d38083c187c08e81fc007c006001f881fc01f", "name": "Abomination", "phash": "49976fd018a65ee2c64986c8a394a78c706019303e7c9b3ded2df6a4d69273de", "mana": 5, "whash": "071e1b8ec19fe03ff83efc3dfc3d38083c197c1cfc1fc00fc017001fc81fc11f", "card_id": "EX1_097"}, {"dhash": "caf88de412890d117aa7e45e197dead211c5220ecc193e033c3e92f4ec49f913", "average_hash": "f71ffb7fbb5d987d88f00af03b3efcfef889f28839b07938609018d018103000", "name": "Wildwalker", "phash": "e721d933fbf0734adb488de48eb414f311d28e6172b09fe18ead4464ee02de58", "mana": 4, "whash": "f71ffb5fbb5f987d88f48cf83f3bfebbfa88fa8039b0583860b8105018183000", "card_id": "AT_040"}, {"dhash": "0883fdfc20f80ff41c42f80cf138ce61800b001789ee31d7ff3e7079c1f0fc27", "average_hash": "073ec3e1e7be247fe07fb07fb877387f10f000c000400070807080610061f02f", "name": "Savage Roar", "phash": "bdfa4b003b5576a5d64bb542da441898e9540cec4372acfc6b534e8d5bc39696", "mana": 3, "whash": "773ec7f3fffff7ffe27ff07fb87f387f107000c0004000709070c0610061f02f", "card_id": "CS2_011"}, {"dhash": "3a7f70bca750ce4888b319af674bc81e80f901b3d3c723eea718cf9f7f3ce060", "average_hash": "8100d9139c750e7f8e23ce477e4f1a4e1e402e401e781e600e603e7c3e7efe3f", "name": "Everyfin is Awesome", "phash": "8faa38e7b9b7d63f72e5f883727c7e381c42e9a856c887c23cb21726e0300d1c", "mana": 7, "whash": "8100c9119c778e7f8e63ce675e4f9e6e1e402e401e781e600e603e7c3e7f7e3c", "card_id": "LOE_113"}, {"dhash": "888f1d3bf0d6e378d4e107d2d9e6b7097f177e2ffc5ee865c1ca809703260ed8", "average_hash": "070003004107400384038203e003e003e00fe00fc00fc01fe25fe24fe67ffe7f", "name": "Nourish", "phash": "3908f6d5cf6059a2275bb6208e310b6826d530c6e38469c95ad61b38d9f773fe", "mana": 5, "whash": "071c03634187e103c603e203e003e007e00fe04fe00fc35fe25ff67fe67ffe7f", "card_id": "EX1_164"}, {"dhash": "38fc6d1cd270848539fa73e5879e4927b7424e45308ad14463318f60da83053e", "average_hash": "87139b6f9bc3c8476cc1c0c7f04ff80bfc039c03880000000000000001000100", "name": "Tinkmaster Overspark", "phash": "93ab9b83ac049f31fb3179607ccc221cc64cc2ed99b45bf656c7b6cda6094c38", "mana": 3, "whash": "af539fff9be3de67ec43c5c7f54ff9cbff43fd038d4009400c428c802180032c", "card_id": "EX1_083"}, {"dhash": "0200fd2f70f98ff71fccb3f061e40011487af224fe89c02387076a4c419afebf", "average_hash": "073003c0b300bcf1fcff787fe03fe80ff00ff80bcc07180f180c00040400e01f", "name": "Lava Shock", "phash": "2bdfb5287ef15ed77b740d8c8d8c24b881c31b366328ecd04b672701db017f86", "mana": 2, "whash": "0f7803c0f380bdf1befff8ffea7fe81ff00ff80fdc07180f181c80040414f01f", "card_id": "BRM_011"}, {"dhash": "0ce08dd082233d4f728cc6d898b521e3580eec3cded1074bb332a4e4b1890312", "average_hash": "f70f735e3bdc38d838983b980f3003a1e30803100389019480d401d203060703", "name": "Herald Volazj", "phash": "ef25e32049a079723b5a11161b84616c33d09bdc88be2ba3dfdbda7496b75511", "mana": 6, "whash": "ff1f7fde3bdc38da3eb83fb80fb017a1f3880392038f01f580d401d6031f0707", "card_id": "OG_316"}, {"dhash": "ea9ffdff60869f2c9fdd1e303fc67e3ea448f0810783050683283c78c180feff", "average_hash": "07008100e11fec3ffe7ffe7ffe7fee7f3e7f3e7c0678127c303c000c0000c00e", "name": "Feign Death", "phash": "296a7b2c72ffeef19ea07ab1a7600e5881e43cdc411acaa53b480a1b7b837e6c", "mana": 2, "whash": "0700c100f11fec3ffe7ffe7ffe7ffe7f3e7f3e7c0678107c301c000c0000800e", "card_id": "GVG_026"}, {"dhash": "388eed32927fcc607983e35c0f10d0662e8d94de787885807c3bf3f66c4a931c", "average_hash": "07100b600b70e0c1ccc71fc7ff4732d372cbc3ef33ce3d004cc818d4b084361c", "name": "Ironbark Protector", "phash": "d729bf0eaec7be42afd007a75a3011c1287ce2f12b5b66c1edad04a53e069b64", "mana": 8, "whash": "0f100f600b70a0c3c4c79fc7ffc736d3729fe2a7338f3f00dcc858d4f894361c", "card_id": "CS2_232"}, {"dhash": "ca34c9c3828f1c17317e66fc8cf8517120624bc519829f1c3d3972e2c697f97f", "average_hash": "f74b7bcb79c8b8c878887c18fc187c183c003c003c0018009800c8000900f17f", "name": "Brann Bronzebeard", "phash": "c3ab63a839d8962af93c3adcfb981e1519101ed339288eb1f39c2ed1635c6e54", "mana": 3, "whash": "f74ffbcf7e5cfcd87c88fc98ff587f587f003d003c409c40cc00c8000900fbff", "card_id": "LOE_077"}, {"dhash": "fe1ff0e3c1360f5f30fec730bacced319e86586ca0534287842c2959fee33f7f", "average_hash": "07e007800d200cf01c603814600cc86f801f00170117080704278061f067fc7f", "name": "Dart Trap", "phash": "d9be764d375adebcd96e7e5a67b194a8856634cb259680bda436006f245a41d0", "mana": 2, "whash": "07e007802f601df03ce038347c3cc86f913fc91f035f8d678467e063e067ff7f", "card_id": "LOE_021"}, {"dhash": "188c2d106262c888f861a3915f33b366648ec11b82f5034b7fb6e7ec1c4b313c", "average_hash": "873f8f6703e3008004c006c002f803fe013e03bc03b800b800fa18b77a3f7e1f", "name": "Starving Buzzard", "phash": "eda41e4ad39abd65eb8089a52688d3a26068699334e572dadbf2770c8ee3b1ac", "mana": 5, "whash": "cf3f8f7703e307a004c006c003f803fe01fe03fc03fc01f800fa18ff7abf7e3f", "card_id": "CS2_237"}, {"dhash": "2ae78dce12bd895176a3ac0618a5b10946efe39c0f313ec8e83090e3b04e21bd", "average_hash": "070c0318815bc0bb883f08bf0038003800bef199e001e083f887f007f80ff01f", "name": "Siltfin Spiritwalker", "phash": "312d2c37c3cb766a6eda46562e9442b210b4324bd6b4d9c51ef8c725bca99e27", "mana": 4, "whash": "071c9959b15bc0bb88bf0cbf0e3a00ba01bef19be18bf083f887f00ff80ff01f", "card_id": "GVG_040"}, {"dhash": "4afe857c8299052339c6727e8c085903b6624895906a38d537b15e6ef18f013f", "average_hash": "f713b77f9bff985b1cc99ccc7c4e381f981bb01380100000200070107118191c", "name": "Skycap'n Kragg", "phash": "ebe99bebffd2bef0eb24665e1e92a2e10809228d2be019261d2376183682d7ce", "mana": 7, "whash": "ff13ff7fdbff9c5f5c498ccc7d5fb95fb91bb11381110150200170107118191c", "card_id": "AT_070"}, {"dhash": "8af104e681cb0b9437184ee1bce6780dd27a64c5c98a9317043719686684cd3a", "average_hash": "771ef99fd99f603eb03fd03c603ea03f001b181b1818081f0018000000001900", "name": "Gadgetzan Jouster", "phash": "69ef69b4d696dfe9f7e875ece7c01268c918d31c66643e6436039b1199009b26", "mana": 1, "whash": "7f1ffbbfdbbff1bff83ff03c607ee07f087f1c1f181f1c3f001c000801001900", "card_id": "AT_133"}, {"dhash": "48e76cc8d18cca7bb5b55369a4c24b07a78e001d1b2a265418b8317342838f2d", "average_hash": "17bc1fbc0bb885f884efe74fe3cbf3cf734763427342f5c1c44045c005800320", "name": "Jungle Panther", "phash": "7f5b4d410e07c7c4ef8163e2cbf0c8b093f172e066f8cef8db304672c58cd8c8", "mana": 3, "whash": "1ffc0fb80fb887f884ebe5cfe3cff3cf734763c373c2f5c244c0c58005a00320", "card_id": "EX1_017"}, {"dhash": "6af6cdfc9229859b7b0ea6d94826bdf913c386267185c70abf3181e63c4d7b9e", "average_hash": "f70fb11ddb1fa83e909f7299f0812c0cbc083c211d2c18a8e886000a50067014", "name": "Goblin Auto-Barber", "phash": "4b33bb3bfbe31e5b33d83069f6d0986c01760c0ef31cf3804d0719c63e61dba1", "mana": 2, "whash": "f71fb91ddb5fa87ed0ff71b9f4892cacbc893ca11cac1cace886104a5007f014", "card_id": "GVG_023"}, {"dhash": "80c7ef3fe0c03b960d2d531cb6b3386d03b2e6e4cc9d016a01962f7c7f1cfe21", "average_hash": "ffffe9f7f99f3c3e3e3e390c390c37382d011c000821002006300e380f383f38", "name": "Power Word: Tentacles", "phash": "4d59d90a396630e5b7ed649087030e180b70c6e843e6739f5b1d7bf05be2d33a", "mana": 5, "whash": "ffffebf7f99f7c3e3e7e3f0c310c37102f011c081c200a2006700e780f783fb8", "card_id": "OG_094"}, {"dhash": "ca0ff5ffe0f1836016063d627a4fe19ce379feb1294286260d9cd9388067fcdf", "average_hash": "073041c0f91bfc3f9e3f9e340e3c0e7886702e7d7e5c7e7c7e203c000400f80f", "name": "Evolve", "phash": "0ffcbff5f27ff6fa3ceb368649928eb48b41490869192c8743224e095b433ac4", "mana": 1, "whash": "0730c3c0fd1fbd3f9e3fbe344a7c0e780e702e7d7e5c7e7c7e203c000400f80f", "card_id": "OG_027"}, {"dhash": "c8f38d6404c13983730ee0669ccd2c1b1126264c4c98d08890b127e3bd84430d", "average_hash": "37ceb3febbde18de18fe0bee8bbe8bfc9b8c9b8c0b88018800c801c803000300", "name": "Dark Cultist", "phash": "4d53c931d238565bbb58347cbe8442c208fc6dad0cf572ce5edb6bd269b30845", "mana": 3, "whash": "3fdebfffbbdf18fe18fe0ffe8ffe8bfd9f8c9b8c0b88018800c801c803000300", "card_id": "FP1_023"}, {"dhash": "c08fddff705c83a024814b2a1fd6390876f8e110c921344340263749febf686f", "average_hash": "0730c3c3f397f807f807fc03fc3cac1b0c1fcc1f4419c4130030c020f03ff01f", "name": "Spreading Madness", "phash": "3b55d702f5823ea5cd282c7246a886528a4c23866ed38bd4eb6cf3bc5ba37ccc", "mana": 3, "whash": "0738c3c3f39ff907fc07fc03fc3cec3b0c1fec0f461bc413c0314024f03ff01f", "card_id": "OG_116"}, {"dhash": "fa8ffdff00ff035e1eb09d7118e374c4c31407691e92fc25d88af09be72f0efc", "average_hash": "0700e103f90ffc07f837f833fe10fe00fe00fa01da41da03e601c603e60ffe2f", "name": "Wild Growth", "phash": "b300e3fddcfb9e2b6cf83cf484073c7804d21f0306d94bbdc50f7f80df815340", "mana": 2, "whash": "0700e103f90ffc0ffc27f833fe11fe00fe00fa41fa41fa03e601c603f60ffa2f", "card_id": "CS2_013"}, {"dhash": "c8e4adf9538385067b5fa49e49f981f3128764eecbc5a78b0f331ee6fc4cf399", "average_hash": "b71f3b7f3b7e38783cd81fd01ff01be01fe80ff80ff81df818f838f87a3cf61f", "name": "Ogre Ninja", "phash": "ed09ff086bf89f4e3bc086a31f21266800e126b00cb65fc4df6759e4cc6b1d6d", "mana": 5, "whash": "b71e3b5f3b5e38783cf81ff01eb01ae01ea805a80fb81cb818f818787a3cf41f", "card_id": "GVG_088"}, {"dhash": "0ae60ce811cd8b91362649d4b0ac6b49deb2fc45818b44977932876c1e88f132", "average_hash": "f71ffb9ff39da83c983d982d4c292807042f483f5816c8148813001100100110", "name": "Aberrant Berserker", "phash": "e99f731377317bccfb88124923c289981832469819896d704fd2cf653dc7e6cd", "mana": 4, "whash": "ff1ffbbffbbfa8bd9c3fdc3f5c692c67046f583f581ec8168813001501100110", "card_id": "OG_150"}, {"dhash": "88df6c70c1cc0bb8b7a45ee3ba8c7d19fa4234056a2af8c761b4896b3396c53d", "average_hash": "0703c387c18ec00f801fc01fe00fe82f9c1f880fc00f801f800fc00fe00fe107", "name": "Validated Doomsayer", "phash": "19df27337e849ec9a79c998c3ce024a0894966ca96cc6f6ceee4d6f28d8932cc", "mana": 5, "whash": "8713cba7d3afc01fd01fe41fec0fec5fdc1f880fc00f801fc00fc00fe10fe10f", "card_id": "OG_200"}, {"dhash": "02c0c5ff30b0cb0c3f087f18f634ec01f1fb0286011c44334c35d2eafdd9fabf", "average_hash": "ffff01f8e1ff707e78fe20ff20fe20ff80fd807c40000000400000000000f00f", "name": "Noble Sacrifice", "phash": "edffeda876c536f5bd3b5c98c3760dbb09c042444b64b9315b8153a45b0958e0", "mana": 1, "whash": "ffff4bf8f1ff71ff7cff38ff20fe20ff80fff4fe40044000400000001080f0af", "card_id": "EX1_130"}, {"dhash": "5ad90cf0d10c8b3935e04f06be0c712db21220f5d2aa4f4406b8d873a3871d3d", "average_hash": "b796ffb603be0db8983bcc3f8c3f0579051d001e110eb0074001400281089118", "name": "Twisted Worgen", "phash": "5977ef34f726dba06d3193b48b92024a82a9c6026c483d197f34bc41fba1efc3", "mana": 2, "whash": "bf9fffb603be0db89cbfcc3f8d7f057f855d015e114eb1474003c08281889938", "card_id": "OG_247"}, {"dhash": "e001fdff71a4e71b8cf1980e30114826a4cdc9391b5333eccf4ffc9bcb313fe6", "average_hash": "0730238079317c787e780e7bbe6912500260227200620062187fa33f923ffe0f", "name": "Healing Wave", "phash": "ad00272d5bf5fc955e801f031f4a2d2406e3ada1a15e99972b6d9bf163b0569b", "mana": 3, "whash": "073023c07b317d787e788e7fbe6d1270026022722062027e987fa27f9a3ffe4f", "card_id": "AT_048"}, {"dhash": "00c04d41b09c233bcc60988d33db4db29a46678cbf3c7e47f20da89bf027014c", "average_hash": "f7bf03f009c80878822182010240d349994de807e087f66ff26ff24ff04ff04f", "name": "Blessed Champion", "phash": "f1432c0fe94c69bcde2bbb503ab40b250bc24bf44c370bcb7acd5b095e10dbb4", "mana": 5, "whash": "f7ff0bf219d80878822182010240d24d994de907e407f66ff66ff24ff24ff04f", "card_id": "EX1_355"}, {"dhash": "8af6adcc129b4900f96cb299f4f4bd7341c782ea35442b0a523ca5e7cf08afb5", "average_hash": "e71fe35db3dd705e3048300938a839bc39a829b03c14bc27a4a1801d00110001", "name": "Kor'kron Elite", "phash": "e3397f4a3bbfff367ec4397386883ce483724b843cf4974169a1b6640903c630", "mana": 4, "whash": "e71feb7df3df70beb04abc493ca839ba39b82db03e94bc27a4e3c0bd82110001", "card_id": "NEW1_011"}, {"dhash": "88f6accfc11b8b3136e64c8cb8387081e602c9f5b5696357d6b7ac6e599d513e", "average_hash": "371d2b9da1bdf8319830183118313830183240362035303c303cb03eb01e311e", "name": "Pit Fighter", "phash": "edb537896b197c9c5e46c7998e88186626a2b3d839849be4fbc166360db299c9", "mana": 5, "whash": "2f1d2fbca3bdfcb99c391c391d713d701d724172217d307c303c303eb11e311e", "card_id": "AT_101"}, {"dhash": "ea8f8d3512ef3cf07be3e78ec819857332c67c8cc151870b38330fe47808c319", "average_hash": "770079603bc0b8c0e8d3cb9b9ba739813b833b9f3bb8f9b8389838dc12080208", "name": "Sorcerer's Apprentice", "phash": "f709b39a26a2e77cdec80cc53f44402c00c866324c333b47afad7e1dff84ee25", "mana": 2, "whash": "770079617bc1b8c0e8d7c99f9ba7398b7b933b9b3bbcf8b8389c38dc2a000219", "card_id": "EX1_608"}, {"dhash": "fe7f58c6a16c6f339f7736fc7cd8f92385c61a896f96dc0c01d07300fc377eff", "average_hash": "03c0e30ef413d009fc79fc59fc79f870bc407b40fac09cc01c400e00e203fc3f", "name": "Curse of Rafaam", "phash": "a3a379775c54de87e71ca44e5dab9e18a4b3b0e806eb90b9067da08d06f890fc", "mana": 2, "whash": "0340e30ff413f80ffe7bfc79fc79fc78bc407f40f8c09ec01e000e00e243fc7e", "card_id": "LOE_007"}, {"dhash": "cae0ccc5119b8b27370f4c1cb0345bc9a6c20905778ad80503b2986167804f24", "average_hash": "f71f7b9f3b9d383f1c3f3c3f7c0b7c183c167c00480510050000000000000100", "name": "Kezan Mystic", "phash": "c31deb097c717be0776cf3645b9608e689a4198e725893714d2676c59b919a9a", "mana": 4, "whash": "ff1f7fbfbbbfb8bf3c3f3c3f7d7b7d5a7d567c426d4d18471000000001000100", "card_id": "GVG_074"}, {"dhash": "0cc36d3ec27dc4eef9e0b3f107834326ba4ef4398967788bf93f01e406491f32", "average_hash": "0710036003c08047c0c7e2c7e247c243800183310db18cbfccffc0fb00121a12", "name": "Wilfred Fizzlebang", "phash": "99add6426e662f19b64139644fa4489009d3326469edce31766d5793f3ce616e", "mana": 6, "whash": "1f500f6003e08667e4c7e6c7e343c243a241a335afb98cbfccffc4fb00b71e16", "card_id": "AT_027"}, {"dhash": "08879d7ff0f1e37f19fc36e03f8cfe32f149c49999336c44f28dc893632780dc", "average_hash": "0700030001011000f801f003e007f01ff01ff03ff83ff00ff00fe00fe00fd00f", "name": "Cobra Shot", "phash": "f3609e875e5aeff01825330e8710e7e04ec10d869cdc3a7063982b9bdb4c9bed", "mana": 5, "whash": "0700030001031003f803f807e817f81ff01ff07ffe7ff23ff40fe25ffa1ff01f", "card_id": "GVG_073"}, {"dhash": "03808d7f70c0911f36c74c0c911834e16d42f884f02d8d1b127724cad997fa2f", "average_hash": "ffff0bf0c3cf001f803991bf11be31363137011e001e800f802c00203100f10f", "name": "Convert", "phash": "e9753dad37ca5732d7f4b7560965a41929b3488149465b645b405b625ba9b39a", "mana": 2, "whash": "ffff0bf0c3df011f803db1bf51bf31b631b7019e009e809f80ac01e03186f58f", "card_id": "AT_015"}, {"dhash": "46b0297dc6de3889d312a7a14c5998e33984731ae2a4c44889b193e77b5f773c", "average_hash": "f323f941c8cdd8cfd28cf09c009c108cf21c638c408e008e4086001ec00fe01e", "name": "Silverware Golem", "phash": "694ab78bb276d69eeb199c657f607b8493f68cd2644c1b248ea496d66d092f08", "mana": 3, "whash": "fb63f965d88fdccdd6def4dc02de589ef31ee28e408e008e4086001ee03fe03e", "card_id": "KAR_205"}, {"dhash": "8acd8c67d1988ac3363e4dceb0b463e9be92df65a8c9e0160f363468039fdd3c", "average_hash": "77183fb2e388d8293834bc303c317c606c306c3f183f0c3e003e001b001c011c", "name": "Master Swordsmith", "phash": "af9ff7907c6359cc76a886a18d9160b0468e96c939992f0777017f125f32e664", "mana": 2, "whash": "7f183fb2eba8d8a93c36bc303d717d617d706c7f1d3f1c3e183e801b011e011c", "card_id": "NEW1_037"}, {"dhash": "8a8ddd7b30c6814837914e27994c7a99e1008225cecb10145327364c7898f0a7", "average_hash": "073847e6f38ff81fe81fcc3fcc3fcc3ffc3ce430c03918181c1098306030f01f", "name": "Hunter's Mark", "phash": "69775bcefee136bbb7f28e7d997ca4340b2522054b4943c049c172e05363bcd0", "mana": 1, "whash": "071c47e6f38ff91ff81fec3fcc3fdc3ffc3cc030c07918183c1098307030f01f", "card_id": "CS2_084"}, {"dhash": "98e02dc3c2bf3d70f2e5c1ce92d56cb37f66de4db1d3c288703388e3078c8719", "average_hash": "770e636e29d0b8d980810f003f209f289f1f9f3fcb9f49ce00cf81cd03080718", "name": "Vol'jin", "phash": "7f0fa33070a8d986eb6254505f9219b0016173708b396a297cb73edc77f29ce5", "mana": 5, "whash": "771f6bde69d0b85982b10fa03f209fa89f1fdf1f4b9f49ce08db81c503090718", "card_id": "GVG_014"}, {"dhash": "82c3ed7fe08c032c1c5a1ea03c1e71f8e7e01ff3e117cf07de1f0c3e303ce060", "average_hash": "0508c140f10c78007c017e201e381e383e3c700e383c7e7efc3e7c3e7c3c3c3c", "name": "Divine Strength", "phash": "63cc36ef78ffd9323cf32d2d86d8d3cc8b4c4c9559646b244b475b801b603519", "mana": 1, "whash": "070cc160f10c78017c017e601e381e383e7c724e383e7e7efc7e7c3e7e3c3c3c", "card_id": "OG_223"}, {"dhash": "0238194724da7ca0d320a601cf1994f329cd431aa124065870b0bf83c316a6ff", "average_hash": "f76ff3dff1cff0cd72dd829c039e139ef39df3806280208000c0c1040600847f", "name": "Fierce Monkey", "phash": "79b0fb29bb7ef24c89079c372d0c334389e9cad81198b219b3b19e247366ccdc", "mana": 3, "whash": "ff7ff3eff2bff4dd729d829c03de13def39df384628020800080c10c0601c4ff", "card_id": "LOE_022"}, {"dhash": "0a80cd3ef0f941fbb5f64bbb97b427497332ed40de290d533a2ef479c0a68005", "average_hash": "073c07e083800000900ea40ff00ff807e81bd81fd00fd027e033f07de03df00f", "name": "Snake Trap", "phash": "3b77de2c7eba3fe7638da598c9e6216882c44b322bc43b8cdb211b065f233384", "mana": 2, "whash": "67be07e083800304940fb42ff82ff807f81bd81fd00fd027f073f07de07ff01f", "card_id": "EX1_554"}, {"dhash": "b8c92d3743ccc413fbbfa63c41e19387273f08cefc958d0902b318e22540cbaf", "average_hash": "471247608364c044b0d87bd87bc0d3f3c7d31fc99ffe99de30da40dcc2060606", "name": "Defias Ringleader", "phash": "7d67b626de5ebf9f9ac4e7c14f4124c939641282c6d91b02798dc641de81fe10", "mana": 2, "whash": "4f124f6083648064b4d87fd87fc0d6f3c7931f899fbeb8de20fa505dc282860f", "card_id": "EX1_131"}, {"dhash": "68d3cce001870a5c35c85f9cae304d49b2f27d05912b22d40c36c769f88b6530", "average_hash": "d71ffbbffbbef81ef81bfc1f3c1d284d480b781f781320000000000000000100", "name": "Crazed Worshipper", "phash": "eb5d4b413e369fa45f820e992ba0894c342cc32067e7dee47c42f339ce7198cd", "mana": 5, "whash": "df1fffbffbbffcbffc1bfc1fbd5d394e696b785f793304402000000001020100", "card_id": "OG_321"}, {"dhash": "08d96cfe11008a203401498bb606480d805a00b5236aded471b9836007810724", "average_hash": "fff7ffffdfff8fe30ce30fe007f807f007e007c007e007e004e305e005a007a0", "name": "Fen Creeper", "phash": "1f5efb05fb857ba06f20672066909620942836689b211bfbfb9399e729a34be6", "mana": 5, "whash": "ffffffffdfff8fe30ce30fe00ff807e007e007c007c007e004e205e007a00760", "card_id": "CS1_069"}, {"dhash": "9a317cc86096d32fb654ace9dc85b95b62fc8cb2e3ed1f1b3f26165c60918166", "average_hash": "078c030c031e003b803dc03cc0390039003c200e2c3cfc39fc3bf838f8107004", "name": "Gladiator's Longbow", "phash": "69dd5ce9d736d71e5e02a3d356f2e1b69d8d84b481ec12648c21523c87793399", "mana": 7, "whash": "07ce479c075e017fa03dc03dc03d003902bd601e7c3cfc39fc3ffc3af934792c", "card_id": "DS1_188"}, {"dhash": "fcf92dc6d2148509389f309687245f39bcc6e119c0f733cb413ec7e41c0039b8", "average_hash": "071a7b7e0bff447e4cd62ec63e420d5f0f1e033d03f800b00cf018f81b801e00", "name": "Iron Juggernaut", "phash": "8d0f7b2a66c3bd605b42870c3650420b88f41cc30bb99e74cdb53e9fdbe3878d", "mana": 6, "whash": "0f5a7f7e0bff4c7e4c7e2ec63e460f5f0f5e033d03b904f20cf81cf80b801e00", "card_id": "GVG_056"}, {"dhash": "0c8f0d116236fd087234a4c9c99789bf147749ac005097c9ef367fe3fc8d731b", "average_hash": "f765f3e361c2e0d100f083f083f003e803d00fb0038003d000e018d8fb1ff71b", "name": "Mysterious Challenger", "phash": "7525b50a63e8de5efb602c133fb00ca700f04a4923fa7ea57e4bced68ca15ed0", "mana": 6, "whash": "f775fbfb63c2e0f1a0f083f0a3f083e803b00fb00f8003b008e879d8fb1ff71b", "card_id": "AT_079"}, {"dhash": "8081ddff70e2eb41de8e143d07f230006300c6149c79fde77b47079c0408fc1f", "average_hash": "073843c0f39f7c7e1e7e1e7e7e1efe31fc31e01000000f103e181f180f10fe1f", "name": "Shadow Bolt", "phash": "2f05fdba5d0a369733918f48cc468ca12398666e63743e9d89f11e4973eb4ae4", "mana": 3, "whash": "073843e0f39f7d7e1e7e3e7e7e1efe31fc31f01010000f103f181e180e18fe1f", "card_id": "CS2_057"}, {"dhash": "48c68cf9c1c30b3536ce4d79b0447c89c31ae635e0cb09173336ce6cdc98913b", "average_hash": "b71d9383318f783c1839903ec8224838803d803f0c3d0839003b8030b019311f", "name": "Junkbot", "phash": "a9d55742d3a91c197b3236e60b3220b32dc9678edbe673726d166d4d93269249", "mana": 5, "whash": "ff1d9bb33baf78bc3c3db03edc66487c807d803f0c3d0c39003b9035b119311f", "card_id": "GVG_106"}, {"dhash": "929ab87961a35a46bbc85639ab7256e5ac4a589583a30d5dfaba40600196f57f", "average_hash": "e7a663a76107b40b301d74163456741f701e301e7006f010f007c0030100e17f", "name": "Fossilized Devilsaur", "phash": "fbfcdf1c3f5ff656ed1616e767864e90213106441921266011e4cadcc924ff94", "mana": 8, "whash": "efa763a776aff40b741d745e745e745ff05e305e7046f050f007c0030100e1ff", "card_id": "LOE_073"}, {"dhash": "28cb6cbed1ff8270378075f8bb3866018c121f25306a2094d139ff73fc8f213f", "average_hash": "47100580c387e01ff01fb817f807f807e807c0038000000000003800f80fe107", "name": "Echoing Ooze", "phash": "937f3f657fc53ee7db8486e38cb04220300c73421dc31f11ce017b279b23f939", "mana": 2, "whash": "cf12cdb2cbbfe0bffc1fbc1ffc67f947e947c843c10304000c007804f90ff117", "card_id": "FP1_003"}, {"dhash": "0cf00cc401990b6637cc4c88b90263058c3a32f5048abb54e038806119848733", "average_hash": "f71ffb9ffb9f783bb83f383304330063000f800d800070030003000001000100", "name": "Argent Commander", "phash": "99d96909630173025e9293809385189299d8966d8fcdfbecdbe6766f694a59f2", "mana": 6, "whash": "ff9fffbffbbffcbfbc3fbc3f2d73056f014f814d9d40f1432003008001800100", "card_id": "EX1_067"}, {"dhash": "828ffd7f30e06704dc13b8a6620d0dba8354b6ad791af334d3710243128efc1d", "average_hash": "0738c3c3f38f107e067e0ae00be8020c834421623140390c180e080e0c0cf61f", "name": "Fist of Jaraxxus", "phash": "cd0abd646bf7d2313e680e87a5780c1e09e16cf8812e46d3e3d54bbe33c1332d", "mana": 4, "whash": "0f7cc3e7f3bf15ff06fe0efe0fe8034c83c4236331c0390c180e098e0d0cf73f", "card_id": "AT_022"}, {"dhash": "6cf8ccc091130baf36524d86b04c6091985a42b591ca2e94b3386772da81bd26", "average_hash": "f79fbbbfbbbffcbffc3d5c3f1d731d60045004100d100c000c00080001000900", "name": "Reckless Rocketeer", "phash": "875fe9c1fb015e207e20134e298430a03292921c9bcd79e16fb3165ff6e3bbf4", "mana": 6, "whash": "ff9fbfbfbbbffebffcbf7d7d7df31d60075005500d500d401c000c8001800b20", "card_id": "CS2_213"}, {"dhash": "e8d0cc82910d0bdb36274c8cb16867b598d26525d84a27950d36fe6c94913126", "average_hash": "771e7b9eb9ba9830bc343c30182218023c083c180c1a4c001c20583308133106", "name": "Tauren Warrior", "phash": "c74767b3fb846f687bc896862f3c2978329298e4a7c336db2de14263f12c8461", "mana": 3, "whash": "ff9e7fbebbbe9cb4bc3c3c381d621d4235483d5a1d1a4c401c3058333933310f", "card_id": "EX1_390"}, {"dhash": "0a0fdd7ce0e1c39b37b50ddc7d21f344a6f11c81703be0c7ff0c095310bceb7a", "average_hash": "072081038107c40ffc3efc3ff83f787f043fc007c0070477ac67e80f601af01f", "name": "Primal Fusion", "phash": "99743ef77e7b9cea6e8da78c6666225f2db064c40b522cbc1b4d8382dbc0b105", "mana": 1, "whash": "072083078107c40ffc3ff83ff83f7a7f043fe007c0270477ee67e80f601af81f", "card_id": "OG_023"}, {"dhash": "8a1f3dfe6100c742dc1e3cca7880d000e1e18143670e0d175a1ca43e1b70fccf", "average_hash": "0720c187c33fe07ff07ff27ffc7df87fd83c782478503030104000730070e01f", "name": "Forked Lightning", "phash": "a9b23bf75efd6ef67ecf8639668086d48c64a43c410a2fd429a16ec45b520e85", "mana": 1, "whash": "0730c387c13fe17ff07ff27ffc7ff87f5839782c78783860105800730070e01f", "card_id": "EX1_251"}, {"dhash": "03808d1ee4ed9159247b4ff09cc1bf0dcb13b60c7024649b8b771a481891f02f", "average_hash": "ffff8bf183f0c418c00de00ff107f19fe9eff967d0130406501260003100f103", "name": "Shadow Word: Horror", "phash": "71573d23ce219da0f19b63cdcb64431a49eb2a642b65573659e4f171139a52ea", "mana": 4, "whash": "ffff8bf181f0c418c00df80ff10ff1bff9eff9e7d2930486509a61003100f183", "card_id": "OG_100"}, {"dhash": "0880fd1be0ebe10702bc8d3917733fce7efcb88071008a01344fa2bb4f6efefa", "average_hash": "071803c07f803c003e003700330e630e7e0fbe073e07be198218026c147efc7f", "name": "Deadly Shot", "phash": "4f00f69435af1b0dd9447c00a7418f902bcd4c3cc3b36ce9fbd636f45b6097a6", "mana": 3, "whash": "071c03e07f803d147e003700330e630e7e0f3e077f07ff19c61c226e36fefc7f", "card_id": "EX1_617"}, {"dhash": "a8cd6c98d1200aedb0325325a442438f9c6ab7dd6fbb9104bd314b63f48cf93d", "average_hash": "5f93cfb70bb30ca0cc46c4404340c14007401168336c394c186d1c8819987b3d", "name": "Micro Machine", "phash": "c7cf9d248f85cfd13eac128e63f00cd2060ae661ac598e85ef24b9c37f227336", "mana": 2, "whash": "5ff3cfb70fb31ff0c4c645c0e7c2c3c0074013e833e839c818e91d885db87b3d", "card_id": "GVG_103"}, {"dhash": "1692007205a01e80bd224819bcf670bd654a969429200f5d3cbae8549388653f", "average_hash": "fba7f9a7fc634c6804b8073803f8c7f8f3b43121710070407400640165000b3e", "name": "Avian Watcher", "phash": "e7fc8157d34ff3b0fb3776b86f981a9839b6364c0998128629a626cd099216d3", "mana": 5, "whash": "ffa7fbe7fde70c6a04e8077807f8c778f3743321770074c07440744165810b3f", "card_id": "KAR_037"}, {"dhash": "caffedff70308fe01e003d4870f0e610dda13343678eef0d3f093c100220802f", "average_hash": "0700c107e31fe033f077f075f070f07c786c780e584ff85ff80ff803f801fc03", "name": "Explosive Trap", "phash": "a192bcf47a7d5eeb66d2b6b88ce006f689703664866566044bd37b835b13db0a", "mana": 2, "whash": "0700c107e11fe037f07ff075f071f87c786c780e784ff81ff80ff803f801fc03", "card_id": "EX1_610"}, {"dhash": "48d8ec31916f0a9b35664e8ebd187d7191c2206547ca88963935776a9c966139", "average_hash": "37130b831b861804980f8819d83d383fb80ff80ce80cc80c8809d801680c7108", "name": "Bilefin Tidehunter", "phash": "4b4bdfa4b6b45f1eefa8b1e1330812298870618632c37e08bf01ef49eea2df66", "mana": 2, "whash": "3f171ba71ba658a5982f8c3bfc7d3c7fbc5fbc0eec0e880c8829d801e90c710d", "card_id": "OG_156"}, {"dhash": "0afd4dd08283090e7f593c3cd973b2a7781fc4a201c54808b13f46e0cc0f19bf", "average_hash": "f703f11f195a3818783c683e703ef03e603f04bf0810080c18091801980f901f", "name": "N'Zoth's First Mate", "phash": "e9e5636c9aeb1e1ffb74b3c60a9c165c6190f2a08ce54f8986249d86cca4e7e8", "mana": 1, "whash": "f71bfb1f1bdeb838783c7c3e703ef1bf60bf0cbf0c18180c182f1801980f901f", "card_id": "OG_312"}, {"dhash": "08f00cf0d1c50bc336064f9eba3c74e9e0b2814516894313df37bc6e4c9cf13c", "average_hash": "f70ff11e290e181f182f183e3c3f7c3e7c321c301c301830383cb81cb81cf01c", "name": "Explosive Sheep", "phash": "cb5759ca7a21b6e573f869b896d211e263c899c03b488e613e067e12fb96de12", "mana": 2, "whash": "f70ff91eab1e181f182f183f7c3f7c3e7c3e5c301c303830383cf81cb81cf01c", "card_id": "GVG_076"}, {"dhash": "6cf84ce381868b9d36364c4cb8907189f2f2e585cf4b9e97bc37106c26819d26", "average_hash": "d71f5b9f7b9c783c383c9838f838f8307c383c387c38ec3cf81fc013c0002100", "name": "Squirming Tentacle", "phash": "63d76f815a824bc8b6611b622f5398258c9c3678a9ed36663b6d6373ad234aca", "mana": 3, "whash": "f71f5b9f7b9c783cb83cbc38fc38bc707c387c387c38fc3cf83fc013c0002100", "card_id": "OG_327"}, {"dhash": "dc1f3cfe60fc9d7d26ff2f0ad904baf160c1c762b68d681bd827375738a2416e", "average_hash": "0780c300e11ac01fd837f83ffc3f7c3e783efc3fdc1f980fc80ed807f007c003", "name": "Coghammer", "phash": "e3373f007e0a1eb1e6401736138198f04cc0460a28b8f3e98fe5be25ffbc7e3e", "mana": 3, "whash": "0780c300e11ae00ff837f83ff83e7c3e7c3efc3fdc1f981fc80fc80ff007c003", "card_id": "GVG_059"}, {"dhash": "4ce48d89621b4526bc786bc19d96610fc67a1c85702a83d516b7bc7669899333", "average_hash": "771f337d23d0107114e080a9803f803b8033940b300f001c0018f01ef0139103", "name": "Mekgineer Thermaplugg", "phash": "a9793f9a67296f289b51690edad843669c8df3301bcb36a4770892f15b2e9624", "mana": 9, "whash": "7f1f3f7d23f935793461c4f9817f817f8933950b310f015d0438f09ef1929103", "card_id": "GVG_116"}, {"dhash": "caf0accf11a20b06b67c4cc9a7d24d259b4a6495d92af245ecb939675296253d", "average_hash": "f71ff39efb9bf83f7030f011b00cb00fb00fb00370006008600ee00fe00f4107", "name": "Jeeves", "phash": "f3f9bf257f06dfa99bf369f226a491a008900990c8601b99bb0ddec4be01ee4e", "mana": 4, "whash": "f71ffbbffbbff83f7030f011f04cf04fb04fb003f001600c700ee00fe10f610f", "card_id": "GVG_094"}, {"dhash": "7267c9ab0694fd2877d6aa20544da8b3534fa08c47c19f9a7737c7a68d491a3f", "average_hash": "f148f95bf8ddf0d9dab9dbbb48fe58f831f811f801f808bb809b90991218103e", "name": "Swashburglar", "phash": "2d49f3ce9abde63fee723d4f1f56b67130b966c260fa7230639806938c811642", "mana": 1, "whash": "f948b97bf85df0b958b9dfbb46fc58f830f811f801f808bb80bb9099121b103e", "card_id": "KAR_069"}, {"dhash": "c8e10ccf111b0396373c4b70acec52b1a166728d6f2aded405373b6e7898813c", "average_hash": "f71c7b9c3b99383c781cfc0cec14dc16b8003808000cc00c683c383c701c211c", "name": "Raging Worgen", "phash": "639be741ed987ec8df2429a658d2e7d1382619f28e35333a29e34ccf9929b218", "mana": 3, "whash": "ff1c7fbc7bb938bd783cfc1cfc56dc56b8503848000cc40d6c3c783cf11c291c", "card_id": "EX1_412"}, {"dhash": "80c30f672408397cda10770fc82cb2f150b3876413cd66fa8c9e0f617adcf4ef", "average_hash": "fffffbfff1ddf811f252e4983cd038f078f87838783068302ef8348c38b0f8f8", "name": "Shadowform", "phash": "65ff39a05901cd16cd5f4e7899446f48b38482350bf922fd59e1331edbc2f18a", "mana": 3, "whash": "ffffeafff1ddf055f250e4987cd018f078f85838782028302ef0348838b078f8", "card_id": "EX1_625"}, {"dhash": "68e0cd8712bc09687c132eccf13083410ef3180a23c44c08f138def1bc075bbc", "average_hash": "f71f3b7cfbd1f8dbdc61dc303cf03dc00d823f800200180008c010c252025418", "name": "Orgrimmar Aspirant", "phash": "c7c7e30979e17e386f188e122cc2186618dbd36061b93be733c77386f3981c9a", "mana": 3, "whash": "f71f3b7cfbf9f9fbdc61def03ef13dc01f803f820f801c8018c418c252825c1c", "card_id": "AT_066"}, {"dhash": "08daad2c12478be07e04bf0cf0f980330f4f06da38b4694896b8adf16f02d7b4", "average_hash": "f73dbf75fbeffdfe7cff3eff3efc5bc093c103c103c002c000c000c003800620", "name": "Armorsmith", "phash": "ffb0eb177b1edea236412da9298009a82294728636487f49ed49df44df26df82", "mana": 2, "whash": "ff3fbf75fbeffdff7cffbeff3ef45bc013c103c103c000c000c004c007800620", "card_id": "EX1_402"}, {"dhash": "0afb0cf6d1ad8bc3370f4f38beb070ede3d2cfc59d0932137d36fb6cec899937", "average_hash": "f703f116c11fe81f383f3c3c1c3c3438bc30fc317c3178307811d811f811f80d", "name": "Gnomish Inventor", "phash": "a3b55d727ba3b6ac76720ce1c7d0938d607830d8398e1626395adcc91dc33b1b", "mana": 4, "whash": "f717f11ec11fe81f383f3c3c3c3c343cfc307c317c317831f811d811f811f80d", "card_id": "CS2_147"}, {"dhash": "f87f7dfbe1108f7f19c283967e00d400483cb3e3cc8653028e08b41582386c1f", "average_hash": "0700c303f13f3c7bfc4bfa2ffc7fdc37482e8e4d1e3f3e003c003c042800f803", "name": "Beneath the Grounds", "phash": "830ab9c87e6d6e713c16d6b01b83164a067484b965b2ee54ead432e57b3bb5c6", "mana": 3, "whash": "0700c30bc13fbc7bfc6bfa0ffc7ffc37482e9e4d3e2f3e097c0038043802f807", "card_id": "AT_035"}, {"dhash": "98cd2cff018c0b30b7485617bf2874f9a0b25165670adf1426b5586a3394e739", "average_hash": "e7116390f199f039a03d001f081f183f7c1a7c0cfc0c78083000200000000118", "name": "Kvaldir Raider", "phash": "63dd39a53e2467961e869c800f8a2448a1c10db27b823365bb6ddddb6dcb6e39", "mana": 5, "whash": "ef13efb1fbbff4bfa43f041f1d5fd97f7d5afc4cfd0c7c487000200021000118", "card_id": "AT_119"}, {"dhash": "eacfc5f2029ec520faff3078078d7c1a7766c3c887171bcb073edee4344be13c", "average_hash": "17019563c15860794079a481e001f83018381a3c78b8603838b07810781c701e", "name": "Cho'gall", "phash": "6787bf695efcafe5cb5296385a508f952c8d698c63a84b31561146027ec65467", "mana": 7, "whash": "1f13db63d3dc607d4479f681e041fa709838bbbcf8b8f03878f07830781e781e", "card_id": "OG_121"}, {"dhash": "b65b69bd8462db05b67a4ce5b8027bedf15ac2950d23129406b96954b3806d3f", "average_hash": "4bf649f74cf744f774fcd6f8dd782d78e77831783d780c480400000001004b3d", "name": "Medivh, the Guardian", "phash": "adbeb1a4da86df3bff67566aefc66630690ecc6ca9a01a6aa94c46d109137210", "mana": 8, "whash": "4ff659f74cf744ff74fcdff89f782f78eb783d783d780c480440004001004b3d", "card_id": "KAR_097"}, {"dhash": "389d6d3ed2ecbc917966e208cdd39acb20b6b36c67d1cd88893753e6d6089b19", "average_hash": "07400be009c0d8c408c033c013c003d04ba02fac1fac1fce4cfc08f98319831d", "name": "Flamewaker", "phash": "5d0f96a2a9e9cf56be2024e95f70269d00c6bb2541d6fae5aaf94add372002d3", "mana": 3, "whash": "0f500be049c158c50cc1b7c017c00fd04fe46facbfac9fee5cfe09f98b19971d", "card_id": "BRM_002"}, {"dhash": "4cdc8cf901e44b00b7ac5331a6424887b48a7b15e42a82450dbafe60dd87b33d", "average_hash": "f797fbbfe39fc03eb01e701f704ee04ee00e700420040010001010003105b11d", "name": "Ancient Harbinger", "phash": "7bf07b449b111684fb0196c08b9984e02c6612e679e67bce3ecbec2db629ee34", "mana": 6, "whash": "ff97ffbffbbfd4bfb43e701f714ef14e714ef1442144015000101081318db11d", "card_id": "OG_290"}, {"dhash": "0080cd9fe0310ee0b89f613482c004193b7ef6fc08eb69e097c6eb891737fe0c", "average_hash": "ffff03f0d3f13cf100e20cc200040004010109010f600f00afa17f233f7eff0f", "name": "Mirror Entity", "phash": "1f54f0ab2f003f115a575f944b6a9d0851988d4e23d74b359bd43be5d9e174a1", "mana": 3, "whash": "ffff0ff0dbf13df102e20cc21184010401050d010f620f40efe17fa33ffeff8f", "card_id": "EX1_294"}, {"dhash": "fabfcc6020e794ce279c2f12db24bc59e4a48c7bd8e7739f052f8a583e80316e", "average_hash": "8780c30f730c300e782f783f280f043f003b007b003f003f083f001eb00f300e", "name": "Glaivezooka", "phash": "c9dc5f7276733ce9b6650b9c93d8846c85b408b8466a66392da4a7d4de6a5e21", "mana": 2, "whash": "8781e78f730c790e782f7c3f2c1f4c3f003f807b003f283f283f003eb81ff90f", "card_id": "GVG_043"}, {"dhash": "3ad8e43041e3cad6b4895939b7f27e87f91e93a56c4b97c768b6d26c8990373f", "average_hash": "0fb60fa60ba604e604660267e366f3fff37ead6d0d6d81f2c03200b20580033c", "name": "Murloc Raider", "phash": "fdf6dfb69ea58fb76bf2b3e41e090cf0216096480f13de60c9c1d312f9206c48", "mana": 1, "whash": "0ff60fa60ba607e604e6054763eef1eff37eaded0d6d85f2c07205b205a0033c", "card_id": "CS2_168"}, {"dhash": "eaf004e1018ecb1f37714eceb81c70f9e072c7c5e00b5f1736374c6c9080713f", "average_hash": "b70f711e511e001cf03d981d9c3f1c3cdc383c3c3c3c381db01f201960006000", "name": "Grimscale Oracle", "phash": "69ff7de27ac84ff396e693795bda328830b04cc099213ac36e869384a724e762", "mana": 1, "whash": "f71f799f719e201df83dbc3fbc3f1c36fc383c3c3c3c383db01f301970006008", "card_id": "EX1_508"}, {"dhash": "c082ed7f408c831847049f013cfe63a4f9cc9090022309c2f7048c4c0899fb37", "average_hash": "77bf63e1f98f8c1ece0c46380638377cb73f773e172217213e3018300030f63c", "name": "Ice Barrier", "phash": "2f70590bd28fd7717dc0a4f83e0008070b7a9c2f4be52c95d3e3432a5be3363f", "mana": 3, "whash": "f7bfe3e1f98fcc1eca1c463c0638377c277f773e372207313c3018300030763e", "card_id": "EX1_289"}, {"dhash": "c8f40dbaa2654c86f90ce6990cf3d9663acd658ec7380d439a382df35a46a71d", "average_hash": "f71ffb7f537fb05e30cc32d8335832d812c80398e39c3330b0c428c420040607", "name": "Addled Grizzly", "phash": "e530ebe19b1e3b263f89e6386790401802f07ab21ea7aee1bbad10cd8d0dbedc", "mana": 3, "whash": "f71fff7f937fb0fe70fc32d833d832d91298029ce39cb33830c428c4208c260f", "card_id": "OG_313"}, {"dhash": "0c85ed0f02e07c96796ce310cec99b9b3756da2cf1b0c34900b301e63f086710", "average_hash": "970fc363f1c1f0c030c13b845bd8c99acbbe83bf039f639e00de00da43106200", "name": "Faceless Summoner", "phash": "79299353b2d819643f474ce1cb10883920706acd5c3c6386bbffe3f1b7617ac4", "mana": 6, "whash": "e71fcbe3f1c1f0c130c139c45bdcd99fcbbf83bf039f619e00de00db43146300", "card_id": "OG_207"}, {"dhash": "fa9ffd7f63805e4e9d49329f64004438ba704ce101c3f94523880430c867fbdf", "average_hash": "0700e107fb07fc63fe687e693e611e001e101e031e201e301e0006000248605f", "name": "Slam", "phash": "87aaf35e787f9ef55c856ea0a5ca346825e504138378ad851ed65b42bb257f00", "mana": 2, "whash": "0700e307ff2ffcf3feeb7e6d7e7b1e441e501e431e601e313e400600024cf47f", "card_id": "EX1_391"}, {"dhash": "8a8ffd7fa0f8d3f4476f3fcd78026138ca4186811f2321e74688f9198370fec7", "average_hash": "070003009b07cc1fee1ffe3ffe739c61bc703e70d67006702e3006680660fc27", "name": "Revenge", "phash": "aff07bdd7eeefeac86d22ee82b800c1803f41cf6438de60073431e075b01ff30", "mana": 2, "whash": "070003409b07ed1ffe1ffe3ffe739e61be703e7096712270267006680660fe6f", "card_id": "BRM_015"}, {"dhash": "18f02de062c44580bf006e4f98127047409a85340aeb24c659b7466ee999b735", "average_hash": "e71fe37f03df807e00fe00be003e003e0038403100308038803800180015c101", "name": "Chromaggus", "phash": "69fc49225b0bb64af35847601b9608f453529634abe75ef65ee5dd91c608d698", "mana": 8, "whash": "ef1fef7f83ff847f847f00fe017e217f013e41394139c178803800ba819fe10f", "card_id": "BRM_031"}, {"dhash": "e23d894496b139dd7306e70cce4d1cbb3c1671ac66c4c919b67340e2ad0cdb7f", "average_hash": "a7e3f1cef8c338da9b9f9b9e0b8c098e038e038e838db08db0cd11c91208c67f", "name": "Animated Armor", "phash": "5d63d72eb3e8f2793e6647db3db3dec581b08e49848d06119769a2d908925f12", "mana": 4, "whash": "95ebf1cef8c338da9b9f9b9e0b8e098e038e038c838db88db8cd1149130ac77f", "card_id": "LOE_119"}, {"dhash": "00007de1e1ffff0b5e1f943934e660dce3e1ffcbe6320d067a9cd333fc3c02e0", "average_hash": "0700010001000016025a3a7076387e383e5cfa7ffe3ffe7ffe7ffe7ffe7ffe7f", "name": "Elemental Destruction", "phash": "2b00fc359ef75bd766913640ad40992b0cac2bd5e1f623932ed5b1c20e4bd9ab", "mana": 3, "whash": "07000000000000120250227022386e383e5cfa7efe3ffe7ffe7ffe1ffe7ffe7f", "card_id": "AT_051"}, {"dhash": "6adb8d2612d90ca2794c93180c295ac28166378d78168d08c231a4e37943b3be", "average_hash": "c706f367f34ff04ff04fbb8f321f781ef802b909b987b88010c0004000000400", "name": "Xaril, Poisoned Mind", "phash": "d33afb9b3ef6f7672e442c648f69c26c04f4408c22e912015fc67761b621de36", "mana": 4, "whash": "ef16fb6ffbcff04ff84fff9ff25f7a5efa02b98bf987b88010c0004022000400", "card_id": "OG_080"}, {"dhash": "c8f78ccb11100b4436804d08b8347ac9e502c905836904d708b6236cc7980539", "average_hash": "f71ff19ffb1f78377837b8320438203c003e003e403e403c003c0018001f010e", "name": "Ship's Cannon", "phash": "69f5f360db8979c24e6a249b9eb482a4c89826cc3e4c1b676f027b123f13fe8c", "mana": 2, "whash": "f71ff99ffbbf7837fc37bc376c38347c083e003e403ec03dc03c003c001f011f", "card_id": "GVG_075"}, {"dhash": "a8f94de552cb8d967b682cdc4234f9d22527cfe8dc85000b733ee6e68c4c5939", "average_hash": "771e7b1c7b1d381d383d8831a00d78bff817d813989b08301890181838181018", "name": "Usher of Souls", "phash": "a3817b9e6ee09be67b6416601a61016404e663599bf8be069f679f65ba9861b9", "mana": 5, "whash": "f71e7b1c7b5df87d383d8c31b01d5cbff81fda93d89b1c3118b1183838181818", "card_id": "OG_302"}, {"dhash": "0af7edc692090d13785ce6a40fc9f4f25985e70e8058aa815f3a26f6fc4d5b1f", "average_hash": "f71ff35dfb5db81c38845882db3cd824f8a8f8b8291808180010185080098008", "name": "Mire Keeper", "phash": "63a3f3c378797d3c3bd21ac33e5cd278425282a123e633c9c6ac92a74e83eb31", "mana": 4, "whash": "f71ffb5ffb5df87c3cc4dc92df1df8adfab8fab829980818189818508009c00c", "card_id": "OG_202"}, {"dhash": "0a80fdefe19c8f1d1e723ccd6103c33886e30d3c1e132c66e08d990c2112f32f", "average_hash": "073801c047b84c780cf8d0f990fb80fb20f39033c802c002e402e6226200f00f", "name": "Power Overwhelming", "phash": "a5ecadbf5ed45a57d9993de88963a6fc098e271089c4e6913b443a786314d626", "mana": 1, "whash": "073c07e0d3f85df86cf8d0f9b0fba4fb38f3b07bca16c807e602e6227200f20f", "card_id": "EX1_316"}, {"dhash": "1a802d27d2c88b137eaea1946e7d98d060a6476f9fc4660999327ae5c842033e", "average_hash": "e71e0364e301b8bd1cf01ac01ad87bb879387b984f97dc86d884d8a5f8013600", "name": "Houndmaster", "phash": "6787ff3bfb727b966b4c0be28f60a1f001f4290558c95a4033927318b3a5bcad", "mana": 4, "whash": "e71e0b64e345b8b91cf01ec01ad87bf879387b987f975c96f880d8b1fa01360c", "card_id": "DS1_070"}, {"dhash": "26f448cac50c9f413ff25748ac935c07b14c649ceb22f85d80bb015403888f3f", "average_hash": "933f191f983f883f881f841fe01fe01fa01f001fe00fe00f800f00100000013a", "name": "Arcanosmith", "phash": "79398f67a647f3d0fe197b3efb1a96c5c93432368891927829328e5961447686", "mana": 4, "whash": "dbbf99bfd83f9c7fcc7f8c1fe05fe05fa01fa01fe00fe00fa01f80140100013f", "card_id": "KAR_710"}, {"dhash": "aaf94df6924109897e032d2eda8cb45963b2c46f98c5248b593723e4c4480f39", "average_hash": "f70fd97f9b1f583f1cff1e3f1cbc19bd0c3d0dbb0dba08300898081800100608", "name": "King's Elekk", "phash": "0d87dbf2fae0ff2c3fd824a24e6001d820986e006761ba11ff21df01bf837fe3", "mana": 2, "whash": "f71fdb5f9b5f583f9cff1e3f5cbc1dbd5d3d0dbd0fba08b008b8089800100008", "card_id": "AT_058"}, {"dhash": "2803fdfc72c76f3fdd46929c68f137384f6cbcdb683431e3e3c4831f551efa20", "average_hash": "07000300011c203812691b610bd18373882786431601072317261f3e063ef63f", "name": "Backstab", "phash": "9fc0f42f6ff01a74b11f0f24b6c14942b3542c98c2ae82b5837bad547f895a5b", "mana": 0, "whash": "07000300011c21fc36691ef35ad1a673da279de797a347231f2e1f3e173ef63f", "card_id": "CS2_072"}, {"dhash": "0cf00ce001cd4b32b43d5069a6c24d05b30e621d1aaab74464bb0f6d339acd36", "average_hash": "ffbfffbf7fbc3dfc34e8df40efc0e7c02340035003c011c0045205b005b40334", "name": "Ironforge Rifleman", "phash": "7ff77b436995dbc07ba613869b62a1b1c94863d829cd72c3bec2169ee6901224", "mana": 3, "whash": "ffffffbf7fbc3ffcbce0ddc0efc0e7c023c003d003c015c004d205b005b40334", "card_id": "CS2_141"}, {"dhash": "0ae00dc9922085133c8e633996f26805d05284341bcb63969f347e689887313f", "average_hash": "f71f3b7f5bdb08703c603c8c300c342814000020082018203c203800380f390f", "name": "Gormok the Impaler", "phash": "c75b7debfb0d3ec67b786378595412f81236388463805b9637a5b314330c590f", "mana": 4, "whash": "ff1fff7f5bfb1c783c603cec756e756d152805220d203d603c203881390f390f", "card_id": "AT_122"}, {"dhash": "ea87fd7ef088c17b8fc71931776ecf589c27708f307e24e4c98f74bd973c3f20", "average_hash": "07000300830c88088c0016220664006f986f887f007f007e027e807f827ffe3f", "name": "Rampage", "phash": "a9e2567f2fd5f5aedb922cd8a664892b0a671c9806c98b07c3d8af0437233fa8", "mana": 2, "whash": "070003009b0f8d088e021e26067e407f987f88ff607f427e867f807fde7ffe3f", "card_id": "CS2_104"}, {"dhash": "cade8c3891f30acf37964ffcbfb073c1fe12ccc5d889e3978f373f6c58986138", "average_hash": "670131033101180678023828b82ff837fc3fb83f183f203e381c381c781ef80e", "name": "C'Thun's Chosen", "phash": "ebb5967076a23dbdb3d865931ec803a629e692840e436e72b681ce89f6383bc3", "mana": 4, "whash": "67017103b1817806f802382ab82ff83ffc3fb83f183f343e383cf81cf81ff81e", "card_id": "OG_283"}, {"dhash": "6af0cdc3128705593b026c0c98786fa1d14280849b0b27479cb0f873638cc53f", "average_hash": "f71f3b7ffb5ef87cf83e1cbc1c3abc3c9c3c3030301020202000600080008117", "name": "Old Murk-Eye", "phash": "e9e7efa37b8af6489f5996d09c94323c3132324636cc1625792313a1db0633cf", "mana": 4, "whash": "ff1f3f7ffbfef87efc7e1cbc1c7abc7ebc3e3832103220206010f0048100011f", "card_id": "EX1_062"}, {"dhash": "9add2c2bd1d00ae1b5bc516baa925627acfe185d612ad944b6b94c731384673e", "average_hash": "6fb16fa7eba73deb1ccec7e7c7d583f60356634633c431cd044121c805804320", "name": "Arcane Nullifier X-21", "phash": "ff9ebb8667ccf7983fa439f10b928ca88c61a64933c91b32692273916e0366e2", "mana": 4, "whash": "6ff56fa7efa73feb1ceec5f7c7c583d6035623c63bc421cd04c1258805804320", "card_id": "GVG_091"}, {"dhash": "eafb0de1928e051b7b602c095c34f55062e7c1988365028b273e1ce47048e33e", "average_hash": "f70ff11f711cf81cf81d901d303c083c083e003e00bc003838bc601c4000c00e", "name": "Flame Imp", "phash": "69c5f3f373eefddc9ed41e3c3a5113c209490644eca09e815c984ec16e46e75c", "mana": 1, "whash": "f71ffb1f7b5cf87cf83dd03dbc3c18be083e10be00be083838be781c5006c01e", "card_id": "EX1_319"}, {"dhash": "0ae84dd392bc0d137f1eecf81824f3704085d36e31996402d338a4f0fb43bf17", "average_hash": "f71f7b5e7b5898393cbc7cb83f393c301cb008b0092000000080004000060007", "name": "Savage Combatant", "phash": "a505e3695b68fe42dbd473783954005319d39bd339d93aa036cb0583ef99ce58", "mana": 4, "whash": "f71f7f7e7b7998f9bcfc7ef87ff97efc3eb00eb00ba20b2000c000c000860607", "card_id": "AT_039"}, {"dhash": "e8dd2cf341a0cb4336bc4c55b8ea73c5e01ac3351eca7694cf38cc73bc81793f", "average_hash": "071263bfe1bfa033383a00301030407860388430c4006806780778037803f906", "name": "Madder Bomber", "phash": "e3443d31d34c5e1e2fc1c6c80ee28629c9a216e3acb1beb87b3ab6dbed034623", "mana": 5, "whash": "0f96efbfe3bfe4bf3c3a343031700178617c8570c5446d467c0778837903f90e", "card_id": "GVG_090"}, {"dhash": "6ac2ccfc912b0a9e35445bacb7184fb190627c85f1ca87151d3173726285c53e", "average_hash": "97969fb1bbb9f804fc047c037c433c4a9803180600065800c800c000c100c110", "name": "Sunfury Protector", "phash": "c3e7cfb13f97dce59ee113e64ec4022263a243706b407e617c011e0daf89bb93", "mana": 2, "whash": "9f9fdfb3bbbdfda4fc277c07fd473d4e9d431d4e094e4d44c800e082c180c110", "card_id": "EX1_058"}, {"dhash": "0080dd1ff0e3833e3deecadcbf8f33396362c644c89b30e7b7ce4d1fbb1cfeaf", "average_hash": "773e03e00380201c002b9037881f8c1f981bb80198110b102f7c3e7c3638fc3f", "name": "Innervate", "phash": "bf19fe266e663e811b792f80894403180b426c244b98baa57bf61bd9db4abc6f", "mana": 0, "whash": "777f07e003802134002b903fe83fec1f9c1bbc1598110b502f7d3e7c3e3afcbf", "card_id": "EX1_169"}, {"dhash": "8283ed3be0dc01300f80774a8a1e308c410e9f1026e109f278b4b1687b58e697", "average_hash": "fffff3e1fb80bc09de0f0c9e841f02200020022082208030c4b0e4b0e7b0f6df", "name": "Cone of Cold", "phash": "3756db4b73adb454fde83db409b30e1683494bf1495863b5469c76a4d3c61322", "mana": 4, "whash": "fffffbe1fb80fd09de0f0c9f869f42684020032082210030c2b0e6b8f7f0ffdf", "card_id": "EX1_275"}, {"dhash": "9e3f686ee1dcd3b0367b2df2d2c4b90b6700ae32c02448db11a3aa5566a68d2c", "average_hash": "0780c382c309c03dc03fe83fe01fe83ff03be00bc41f800d8005c004c0002000", "name": "Death's Bite", "phash": "39973fc4de425e60b76d87b44c9ca6fe84b120b240f9345b8db1292d5ae85e33", "mana": 4, "whash": "87c0c78ac7cbc17dc83fec3fec1fe83ffabbf00fe41fc00fc40fd405c1002908", "card_id": "FP1_021"}, {"dhash": "48f40cd391a50b8336144f2ebc8c7039e042d0a5174938964130876e7e99f939", "average_hash": "f71ff39f791ff83f383e1c3fdc3e1c1c1c041c20ac200c200c000000f800f919", "name": "Stormpike Commando", "phash": "833f69c073819e3c73c11c2e3312ccf003993116bcf19bb6fee9330bcca4e3f2", "mana": 5, "whash": "f71ffb9ffb9ff83f7c3e9c3ffc3e3c1c3c043c30bc200c330c000800f801f919", "card_id": "CS2_150"}, {"dhash": "0af92ce611c80b92b7204d59baec6c39e8925165a38a061558328c689b811533", "average_hash": "f71ffb9ffb9df83df03c1c3c0c3ec83f083e481e481470000000000000000100", "name": "Twilight Summoner", "phash": "6b377b33772173ec9fc4dee4cec480c800b013a8182066313f9333c3f389ffb9", "mana": 4, "whash": "ff1fffbffbbdfcbdfc3c3c3c3d7ecd7f8d7e495e595478540000000001000100", "card_id": "OG_272"}, {"dhash": "f67f88e001594ea23d065b193c66e39c05617cc66008c310023f363c1cf0fc67", "average_hash": "e1c7fb3fbc3dbc3f9c7f1c3f0e7c46729c98380c380c380c14f810780070f07f", "name": "Cat Trick", "phash": "89fb9b56f99672793b0e3caccbdb94e5136cb49ce5b8065b84a114bcaec904c0", "mana": 2, "whash": "c5c7f93fb83fbc3f9c7f9c3f0e7c46729e9a380c380c380c10f810780070f87f", "card_id": "KAR_004"}, {"dhash": "08c6cc0d011b4a34b46851d1a6a2495787ae2c5d71aac244a6b9ed725b85973c", "average_hash": "ffffbffdbff975f104f147e547c903c903c103c003c201c004c005c025a00f21", "name": "Imp Master", "phash": "ffd93b91af11df617f0939c81b8092a421c8636463c8f3f6db96662cb8885a9a", "mana": 3, "whash": "fffffffdfffd77f164f547f147c943c903c003c203c201c004c005c025a02f69", "card_id": "EX1_597"}, {"dhash": "4cfc8cb401710b8437604f47ba9c7cf1e882f105a7095e133c36726e808c013b", "average_hash": "e30ff11fe117f017f039a03d983c383e7c3c783c783c783df81ef81ff81ff00b", "name": "Saboteur", "phash": "69959d34da616792cf620d633cd8224c866c93d11af61b9efc386c7c678b22cb", "mana": 3, "whash": "e30ff007e117f007f019a03dd83c383e783c783c783c783df81ef81ff01bf00b", "card_id": "AT_086"}, {"dhash": "bacf2d3f726c48f0d181a3040719de033827c4ecbc91f10b013782e4244c69b0", "average_hash": "8701c303e1c7e087f00fba8fda07f80ff89dd995b93718bf089e081e001c0000", "name": "Fireguard Destroyer", "phash": "db07b3383eaeafc7ae898cc38ed16cd920f1823076e24d11c7386524dbea3aa6", "mana": 4, "whash": "8713e143e147e087f08fb88ffa87f88ff99ff997b9b71cbf08be085e081c0000", "card_id": "BRM_012"}, {"dhash": "f89fec7920ff95ff23f62dc2de10b9797210e762d2cfbf1f072f1c5f709ec17a", "average_hash": "078083003300f001f807fc079c310c390c3ccc380c38fc38783c783cf81ef01f", "name": "Doomhammer", "phash": "237f9691fae02c5c1c10c791932563b60cc90c2986a9e3704ebc6e0d9fcbff73", "mana": 5, "whash": "0780e3807300f90bfc0ffc079c3b0c399c3ccc3acc39dc3a7c3c783cf81ef91f", "card_id": "EX1_567"}, {"dhash": "2a8e251802227c3ef35ca6fcccb9b4634387d60c30319dc97f3242e6008ec318", "average_hash": "c741e1e3e1c360c030c1fbd8fbf9f3fc7ba073b013a001d1d0c150c003080708", "name": "Vilefin Inquisitor", "phash": "f781b3fe1eea9dfdef5233ab1e61c2d400d06c2103a666245f926b619e2dd250", "mana": 1, "whash": "c753ebe3f3c370d030e1fbf8fbf9fbfc7ba073b233a00191d0c170c03308070c", "card_id": "OG_006"}, {"dhash": "0afd856002d90d073b6266c8981879717ac2f304c74b2a9794360b69be83f13d", "average_hash": "f707f16ff1cff85ff85df899b8199839381c781c6016a8062800080000000109", "name": "Shifter Zerus", "phash": "e3bbfbe9fb3e7b367f4c96343b8446c409110e043a3156317741661b7f026698", "mana": 1, "whash": "ef1ffb7ffbdff85ff83df8bbbc39983b7c1c781c681ee806880408000800c10d", "card_id": "OG_123"}, {"dhash": "0280e53fe0c16338dc6092fb31e57302cc3bdee7af9f3f3478ece41e393900e2", "average_hash": "071801c00100003c82608273e233e0734031007fe07ff07ff03ff43ffe7ff47f", "name": "Arcane Missiles", "phash": "b1fa5e8f47f3d0ff3e9cf3048d2a0cf223caa3184a706bd6321d9b605ae031aa", "mana": 1, "whash": "071c01e00180003c826c8273e26be2734031007fe87ff07ff03ff43ffe7ff67f", "card_id": "EX1_277"}, {"dhash": "fcb1cd4382be84c1394f720496a0603f42f29ee43f0b7396c7362e6d00801121", "average_hash": "070e1b6efb40f8431c4b7881182000301c303c209c3760377837b81f80010101", "name": "Cairne Bloodhoof", "phash": "87d75f4a5b29cd943c50c9805e1e032b78ac1827c3b672e6b3a576784e8b8797", "mana": 6, "whash": "0f1e1f6efbe9fc437c4f7ce13d6001703d303d209d3774777837b83f81038101", "card_id": "EX1_110"}, {"dhash": "f8cf2c3f61f6ca04b5797e93bfe67367e44e8a9d35ab734687b87a632d87593c", "average_hash": "07108780c183c00fc019e019603fe03ff03fb03fb0073007f005d0077006710e", "name": "Sludge Belcher", "phash": "f1f03c0c0ec797f1af3c06804e811032ccc849ec6dc877e6bbc9766dadec66d4", "mana": 5, "whash": "07108fa0c3a3c0afe03fe01de07fe07ff07ff03ff02f70077007f007f106791e", "card_id": "FP1_012"}, {"dhash": "08ff0dfe92c0a9017706ae0c1cd1a06d47bf8c4cb091cccb13b66fe4db40b7b3", "average_hash": "f71ffb5fd9dfc8bf18bc1abc0b3880a101a101b203b800bc00b8005800000000", "name": "Thunder Bluff Valiant", "phash": "2d054b8373aafb74b27066696f4086dc20903e32ffa659491db996971dcbaa9c", "mana": 5, "whash": "ff1fff5fdbffd9bf9cbf1ebc0fb886f105b309b203bc02bc00b8007802900404", "card_id": "AT_049"}, {"dhash": "8aff4dff7082e3348c6b19d890207f4fc088a30d5f9becac021bc4f01b61ff4f", "average_hash": "0700414fe1bfc07de47f24296013e01f6008f068c86e80efb0f32043ac60fe6f", "name": "Living Roots", "phash": "b96a7bdd9fb7fc6b6ef59abc074226782924342623d9461b4b843944d9685684", "mana": 1, "whash": "07084377c1bfc07de47fa43f6013e01f6008f068d86ae8efaa53b043ea62fe6f", "card_id": "AT_037"}, {"dhash": "bcf02d6792b84546ba296c969d6c76d9e822b12463cbc49703375f68bc807938", "average_hash": "670f637efb5c105fb057ccbb4c3f6c3e7c3e3c3e0c3c083c083c181818001900", "name": "Justicar Trueheart", "phash": "e98f7b20f6014f527bc41c4c3ed082c049a499648be517c7370f3639f739d3e3", "mana": 6, "whash": "ef1f737efbde907fb037ccbf6c3f2c3e7c3e3c3e0c3e0c3c083c181818001900", "card_id": "AT_132"}, {"dhash": "e67f78d3f3806638cd659acb3c9f7982a364c4f93f92ff04f00f60d1d7273f7e", "average_hash": "07c6031f407e007d8279803880381e1802180070f800f803e00fc007e007fe3f", "name": "Explorer's Hat", "phash": "29696c69d99ed6ae8ea5d2b5b6739c9a275bd29aa4de30ae04ee806eb0740865", "mana": 2, "whash": "07e6039f437f227da379a339a039de38c2180270fb03fe03e01fe00fe207fe3f", "card_id": "LOE_105"}, {"dhash": "ca81dd3f10f8c3e4bde17b12fe4ed81f87bdc23904f26ec4bd3972038b4cf1bf", "average_hash": "07180140f183e047c067806fc0fd80f800fe0a7c0e604e44cc49f00ff811fc1f", "name": "Claw", "phash": "39aafcfcd2d52ec6ed656dfa0e9699110dea2c2529ca5b80db7127c74b821229", "mana": 1, "whash": "071c6360f183e147c067c0ffc0ff80f882fe8e7c0f604e64cc49f20ff81bfc1f", "card_id": "CS2_005"}, {"dhash": "3cf96cc241cbcb1eb7004cf9b8c26707ca2a38dd70ea8144833904720b841d2c", "average_hash": "c71fcb9fc3bee03ef03f303cf031d02f800f000e000f10061000000000000100", "name": "Boulderfist Ogre", "phash": "69f369856f0f5b4e9ec693e1c6a000388198d998e6b8a6b72cc3dbf3b2a332b8", "mana": 6, "whash": "cf9fcfbfc3bee4bef4bffc7cf177d16fc36f014f014f91471c04048001800120", "card_id": "CS2_200"}, {"dhash": "28f62dfc42e2c90cfe37a46ac9c4b388261efb6fcec5188961378fe7fc4ce33d", "average_hash": "c70f834f811f001e00f8ea18e498c092c016e99ec999c81b009f101ef01ff21f", "name": "Tundra Rhino", "phash": "2925be22cb989fdb7b428661270164a498c437834d70daa4ffa71f457eeb98e1", "mana": 5, "whash": "c71f8b5f811f003ea0f8ee38ecb8c59ac016e99ecc99c81b80bff01ed01ff01f", "card_id": "DS1_178"}, {"dhash": "02cfcd7f20e9633c9e403f932d26760cdc18f1a06647c8888107024e0480fe1f", "average_hash": "671ec1ffe19ff03ff67ff63fe61fe60fcc0fcc0e840c0c0f080c080000000000", "name": "Eye for an Eye", "phash": "497ef1f8f6cd5cd776ec8e891f38a6664bc036386b744a8d73d05a041b861668", "mana": 1, "whash": "f79fc1fff1bff03ff67ff67fe63fe63fd41fcc1e8c0e0c0f080c080000000800", "card_id": "EX1_132"}, {"dhash": "e21f18f070c3d72623014c0893131f473c7c60e0fd0887217823cec7008cff38", "average_hash": "0df043cf020f241f241f00000000001e031e0e1fbc1fe01ec01fe01ff01ffd3f", "name": "Sacred Trial", "phash": "497db6bef5de5f0bb5a7f4af87f70dbc252ee13d87522c00811788056714c801", "mana": 1, "whash": "0ff063cf830fa51f251f01050106011f031f0e1fbe1fe31fc01fe01fe11ffdbf", "card_id": "LOE_027"}, {"dhash": "0280051fe4fef1eca7d94ff29ac3348d493ad3fd3cfc4070882703cf2600fb27", "average_hash": "ffff0df081c2c003c403e511e1b5c5f4c53581f9d9ff1f9f07be03fc0b20d707", "name": "Light of the Naaru", "phash": "3dd7b1fedf21cd23e5f66d9e43c929490be8c902713352955be01fe0518a5298", "mana": 1, "whash": "ffff0bf08182c003c001c515e1b5c5f4c5bd81f897ff0f9f07be03fc0300d787", "card_id": "GVG_012"}, {"dhash": "8a8ffd7fb0e3c1c737265b089e703498e1b0c0c5f7093ff38e8d021e8c31f90f", "average_hash": "0700c101f107fc0f7c0e3c0e3c3e5c1e5c1e3810381c900ef407b6011802fc0f", "name": "Cold Blood", "phash": "cbf67bbeb6c93c6d3c872ef14e7b06984946cc7423a8db844b49d20ca5a48b94", "mana": 1, "whash": "0700c301f107fd0f7c0f7c0f3c3e7c3e5c3e3c143c1eb42efe0fb6113803fc0f", "card_id": "CS2_073"}, {"dhash": "52a6e84ca1c95ebbb936564b2d955279a4b24f25b242e65d89bb5b662394c57f", "average_hash": "87ab83a99109904cd00ef05fa05b98590818c816c016401e401e400e6106c17f", "name": "Wobbling Runts", "phash": "792f9751775a9e84bf3d76cdd92642462dcb4236992122cc1d689a3133a68ced", "mana": 6, "whash": "cfab13a996a9944cd8def45fab5b9b591b588d56c056405e401f401ec107c3ff", "card_id": "LOE_089"}, {"dhash": "00800f1fe4bc31f0e4c7cfe498413b91463a07ceec0c98d130b6e1c8ceb1ec0f", "average_hash": "ffff0bf041c080018007891fd98bc183c1c38183c081c18bc021e1e179e1f997", "name": "Thoughtsteal", "phash": "b1574d82e71035f9cd4eaf544b01813259cd334043f77bcc79e45acc5b6a762a", "mana": 3, "whash": "ffff8ff0c1c0c1038007099f9babc1abc1c38183c381c18bc1a1e1e1fbe1f9df", "card_id": "EX1_339"}, {"dhash": "8aec0dd1028e0b187f3eaeec5831b3e07186030f1a3496c87cb9e1e20345673e", "average_hash": "f713fb737b5838b818f87af83eb83bbcf13ce3926380208120c1808000000600", "name": "Infested Wolf", "phash": "656daf2d9abaf38ebb749d9192c011630991d6d813746a506f8a568abea52da5", "mana": 4, "whash": "ff13ff737bf878b83cf87ef83ef87ffcf9fcf3926380608160c180c002800600", "card_id": "OG_216"}, {"dhash": "c8fdcd8093190c3f78e328544298cff01ac715083f044208983910e2be476336", "average_hash": "771bfb7fbb7838c07cc01ee19dc01ec07cc17fc1f3c1788038c000c020800600", "name": "Floating Watcher", "phash": "c7838703ed7073e8ebf01e423b947198424816e41eabeeccbdb14f657a2ef070", "mana": 5, "whash": "7f1bff7fbb703fe1fce11ee19fc05ec07ec17f81ff81fc8038c004c030802604", "card_id": "GVG_100"}, {"dhash": "cae78ccc11990b02341c58a1ab4846d5b8ae6d1d532a8ac576b8e9735986872c", "average_hash": "f79fffbffbbdbcbd3c3c7c304d6641464040600a010600412001200741010100", "name": "Spellbreaker", "phash": "ebf86b61bb197d897fc69eb4e62c62680cc109e2098436cd6e62539acd61f332", "mana": 4, "whash": "ff9fffbffbbffdbf3cbc7d706dd641c64348615a214e01412001348351810320", "card_id": "EX1_048"}, {"dhash": "429b886601dd1aba3977528ead9859f181e27fc5dd823a1cf4396663c196e17f", "average_hash": "f700f102f805f80df801fc01fc003c019c00f8007811b8019001e001f001e17f", "name": "Naga Sea Witch", "phash": "835b63727970f694efa496b96d2c32568c33c68c39a68233358c8ee555987ee2", "mana": 5, "whash": "ffa2f9aaf805fc0ffc0bfc41fc417c41fc41fc017811b803d803f005f103e1ff", "card_id": "LOE_038"}, {"dhash": "b23ce161e40ff9f37294e706def93d077e36dc6cfed1cc0b017600e4110ae37f", "average_hash": "01c101c400ccc0801a87038e038ff19f839fc38fc38fe8bff09ff0804200460f", "name": "Babbling Book", "phash": "5915dc7ba6bab9bfbe4be6dd9b46de27a1e5b2c054382414797263489905ce00", "mana": 1, "whash": "01e101c400cec0c51b8fc38e0b9ff19fc39fc39fc39fe8bff89ff0804308c72f", "card_id": "KAR_009"}, {"dhash": "80815d6e3051e3a71c500baad67ef9d4af353a2947100026bc4f42810c3cfd7f", "average_hash": "073043c0838c20257025082e042a96c9cac6fa6efb387b00737d130c1360fb7f", "name": "Drain Life", "phash": "eb021eddff5177041d220ecda3048b0291a80e75a95c9e276bc1aff15bea33e5", "mana": 3, "whash": "073843e0a18ca1667426082f042bbecbfad6fa7efa387300737d130d1368faff", "card_id": "CS2_061"}, {"dhash": "9ad42c3b617c4b11b64648cfa1b457cbac2adfdd98eb225761b6cc684b83873c", "average_hash": "07950fa343a6c5b790739441b54129618150013b013bc031003b803ba10d811d", "name": "Mogu'shan Warden", "phash": "bdf5dfe65b97edc4679bc69889c188c02083c1b4c9d856b26d04cdd867836f21", "mana": 4, "whash": "0fd50fa3c3a7c5b794f39541bdc129c18350017f017b45f1047384bfe58fc33d", "card_id": "EX1_396"}, {"dhash": "f8f18cc5114a0b003675484bb90e6799cc22e17508cb7c17ff36e66d01931d3e", "average_hash": "471fbf9ebbbf983dcc398430843414260832c0180c000837c037f0074006210c", "name": "Garrison Commander", "phash": "89b76fc9db80dfc56f6493f42c68304c88c61ec629919e0dd7087e26bd81cf8d", "mana": 2, "whash": "ef1fffbefbbfd8bfcc398430847d54661876c03a0c220c37c037f0376106310c", "card_id": "AT_080"}, {"dhash": "88e70ccf81080b6236644cd9b9206741dc4aba1d382b00c607b08f680387c73e", "average_hash": "f71ffb99fb9bf839f83db033f0332037803f802f002f00070001000000000104", "name": "Stampeding Kodo", "phash": "a9f563096e2b594a7b4a96c84bd022d20692c6cc336cbe24db265f366e32bee6", "mana": 5, "whash": "ff1fffbdfbbff8b9fc3dfc33fd73f977a17f806f012f0007000300000100010c", "card_id": "NEW1_041"}, {"dhash": "4af98d7693c905877b6ea7905c30a3734ce7f80ee2d5870b1c3620e487489bb9", "average_hash": "f707f90ef94f385d78dc3b9c3cb818c618a718be01be40be609c401800080000", "name": "Bladed Cultist", "phash": "e9a3f3bb33fb79769e74376c2f41409812848c603c997b01db58d70137249dd9", "mana": 1, "whash": "f717fb1ff95f787d78fc3fbc3cb81ca61aa718be01bec8be60fc605800180000", "card_id": "OG_070"}, {"dhash": "8a870d7f30d00180651acef09c0c6f0171f29104c6093c5670ac877a8e95fcbf", "average_hash": "f73fe7f3e38fe10fc00af808e80f380f9817dc0dc400c021c00388210810fc1f", "name": "Mark of Y'Shaarj", "phash": "7b5f7b8a3fa03fb0cfb7ac6827ec0c4689904e5c4b103e814be05f8073cbbc2c", "mana": 2, "whash": "ffffe7f3f38fe30ff00af808f80f380f9817fc0dd41ac061c04388291810fc1f", "card_id": "OG_048"}, {"dhash": "ccf70ccfd1160b6d36024c9cb17046d9b1026f851b0aa6150835c16a3397dd3e", "average_hash": "f71cfb9cfbbd583a783b1c383c207c00fc18fc1078007800000000200014011e", "name": "Disciple of C'Thun", "phash": "634d63897d1073787372d6f09ef0a6e133ee528c9c370f877963783a9920d330", "mana": 3, "whash": "ff9cffbcfbbf7cba7c3b1c383d60fd40fd58fd58fd407d4004200020011c811e", "card_id": "OG_162"}, {"dhash": "1c952c786080d4ef23fe2721d0c4ae837f0dda32f0a6c45933af6f50daa7bd6f", "average_hash": "07c30fa3874a05407806f883fc01e004c38fc017801f001f84498501a901f921", "name": "Eaglehorn Bow", "phash": "5b7a77449e191d60371633d9c3c492b44cc10c2a14307b59f3e1ce2cdf3c3e3f", "mana": 3, "whash": "07c38fe387ca07407d86fc83fc83fa84e39fe017801f801f845ff543e901fba3", "card_id": "EX1_536"}, {"dhash": "66e0c88485681f933e0e5d68b8906521cb6284c53aa30d5d7abae1530787cd3f", "average_hash": "dbbe793c383c9c3f9c37fc79fc75bc7598235021101010008003800f010cc13f", "name": "Menagerie Magician", "phash": "a9f7e1176c0776e57b5b373b676699cf99b148333132c6788909963490219b19", "mana": 5, "whash": "d9bf79bc383e9c7f9c77fc7dfc75fc7798735821503890108003800f010ec13f", "card_id": "KAR_702"}, {"dhash": "1af12d405299c963ffc6ac8959d3a63e001e7bff86346e48f8bbe1e60347073c", "average_hash": "c73fcf7e03fc00b910fb86fb82f783ee010283848393c0c4c0cfe0cf820f0606", "name": "Scavenging Hyena", "phash": "7d33af496bcf73b2de52998683a0936c10e892100c9466d2ff2d6d037e9af661", "mana": 2, "whash": "ef1fcf7f03fc00bb14fb96fb86f783ee0186c3848392c084c0cfe0dfc28f0606", "card_id": "EX1_531"}, {"dhash": "3acf392c6272c4fdb88af37587a34d679a0e351d5a3afc44d8b981736387c73f", "average_hash": "cff28fe107e105e044c147c043c0a3c123c18340c3c2c3c7c4c7e5c7e5efc7bf", "name": "Deathwing", "phash": "7fea87882f161e60af9943f41ae740fe99ffc26faae74e6447811b811b0017c8", "mana": 10, "whash": "cff28ff10fe107e0c4e147c0c7c0a3c523c0834083c2c3c7c4c7e5c7e5afc77f", "card_id": "NEW1_030"}, {"dhash": "2cfe8d01d24504fb39b2734487085e19b022c31da6ab044719b493682385c73c", "average_hash": "f71e3f7f1be79944dcc7dcc7dd47cd478c00c020813000300010001801088108", "name": "Malygos", "phash": "19e6cb22ed186cd07b4b196d3bd2664366a366f332b3662636661266938716f6", "mana": 9, "whash": "ff5e3fff9bef9d64dc47ddc7dd47dd4f8f58c134c1704170003004980188832c", "card_id": "EX1_563"}, {"dhash": "0883cd3ff0e0c18cb729585220bd757ddbdcb6b30fc63b88f78ef83fa13b066e", "average_hash": "07180340c383781e0c3c04200800042036612e788e70bc03fc27f63fee3ffe7f", "name": "Recycle", "phash": "2b5c6e077be4bf213f089a950e1a2970c2594b2547c0db143be51b5cd9f1d39b", "mana": 6, "whash": "071c03e0e387790a2c3c04240a400c2036610ef8ae72fc07fc67f67ffe7ffe7f", "card_id": "GVG_031"}, {"dhash": "c8e28dc4122d094f7c14b3f0e0ec83db2eb7dd0ab845e389843b37f66e02d9b9", "average_hash": "371fb35d39dc381e786638008881c9c16182f8a73837241e00ce800ef805f801", "name": "Ravaging Ghoul", "phash": "cb23ef686b734d1c37c50c1b8b90d3f658ec9b8609cd6eb1ccbc53c39036198c", "mana": 3, "whash": "371fbb7db9dc38be78663c208ca1c9836996f9b72c9f241e001fa88efa05f801", "card_id": "OG_149"}, {"dhash": "cce08dc10283853c3f7769528eac783173e2c6c49fcb64173b36ca6ca699693b", "average_hash": "f70ff37f7b5a7058f838fcac7c1c3c089c381830dc301c30183108381818191b", "name": "Bolf Ramshield", "phash": "a3cf7b38f9089f309b095b385b94305419c14a1cfa88de34dbf493a72d2fb1a5", "mana": 6, "whash": "ff1ffb5ffb5f7058f8387cad7c1d3c081c395c30fc301c32583148381818591f", "card_id": "AT_124"}, {"dhash": "0a804d1e90b90172b78cfe53b0ec00f31bcee7074f494af00740740ef11cea37", "average_hash": "077c07c083819109b03ba03d6071e0007818780e6d03ed01ff00ff28f638fe1f", "name": "Sabotage", "phash": "e359bc2b3fe0d7de6dc6273c099ee36143611cd2034a21b1cb2d36d69849f68c", "mana": 4, "whash": "0ffe07e08381930bb03ba07d607de0207808fc4e7d03fd01ff00ff28f738fe1f", "card_id": "GVG_047"}, {"dhash": "ca810d2ef0d880fb37a34f569e642ccd739ac424b3096c92c06798cf21987eb0", "average_hash": "077867c0538019009c079c07ec3fec1c641a6412e024e026003e003f001ff01f", "name": "Blade Flurry", "phash": "6b57160bfda1f6f03fe80d3e93978ced036309538e8d9c906b8396f0c960b334", "mana": 4, "whash": "4fffe7e073808b039c07dc3fec3eec1ce43e6412e036e03e203f003f803ff01f", "card_id": "CS2_233"}, {"dhash": "88d80c33514c8a9034614fe1b94c721ba7b2cf6592cb28975f37c36cac99b937", "average_hash": "f717fb87e987e006a0018021c033c032c003d813d803d833f839783ab819711f", "name": "Silver Hand Regent", "phash": "b3277b106b99f3a47be0a721b2782289269812763be4deec6ef3333399249ac8", "mana": 3, "whash": "f717fb8ff987e825a0059029c033c072c813d80bd803d833f83b783bf81ff91f", "card_id": "AT_100"}, {"dhash": "c2eb008e85391f62be4c5d1cbe915de6b49a80657f03d8de123ecd6d36984932", "average_hash": "f11af119f031f8333033303c383c101e205f6063c86f803f8039083d583e9934", "name": "Arcane Anomaly", "phash": "e9f9f3693237395f6f266cdbe7dbd60c993189c3a61e2068184236cc6c183781", "mana": 1, "whash": "f9bbf119f031f8333033383a387c101e707f607bc82fa03f803d283d793db936", "card_id": "KAR_036"}, {"dhash": "d23fa94c1690f86b735fe5f8c2811f072e16792cd8f465d89ff120e2010dd77f", "average_hash": "67eff1cff0cbc0c173d07b88fb84038703820381038300c2c0cdc1c18300c77f", "name": "Ethereal Conjurer", "phash": "5529e31929fa987eff1463126f866e060cec0ee415af06449d09aad993f3df64", "mana": 5, "whash": "efeff1eff0cbc0d1fbd07b80fbc653c703820381038300c7e0cbc3c98300c7ff", "card_id": "LOE_003"}, {"dhash": "7cd8ac7741ff42ecb5c85f91bf207f01a002480df72a8e473036406894807d39", "average_hash": "07000380618cf00ff00ff01ff83ffc3ff81b1804e01ef019b019000100006100", "name": "Alarm-o-Bot", "phash": "eb9ddb843e0326338e85999973828ee323f919e6dbf2b63939cd4866a6812469", "mana": 3, "whash": "87110ba463aef0bff01ffc1ffc7ffc7ffc5bf816e01ff039b03f000901006100", "card_id": "EX1_006"}, {"dhash": "caf04dc5d29485737e472c9c5170a7815963c7c61c85378b9b3620e4d448299b", "average_hash": "b70fb11fd918983818b11cb13c207c201c389c301c3838b0b8b8981948190018", "name": "Buccaneer", "phash": "a385db2a78a97b72dbd8c3989ed4199359d0d20c2769bfc11e4973cc3284c74d", "mana": 1, "whash": "b71ffb1ffb58987a183b3cb13ca07ca09cbcdcb81cb87cb4f8bc98196819001a", "card_id": "AT_029"}, {"dhash": "c2b0884101865e0c381a50ffa2995d32b34401cd68a23f5dfeba50618396c57f", "average_hash": "3faf7baf79ae74400c400240b841b845f85a280030483050f007c0010100c3ff", "name": "Tomb Spider", "phash": "e3fbc75cd95ccb70fb331bb3bcbc3e7629a20699119a22e405a00e9213870f6d", "mana": 4, "whash": "bfbf7faf7eae74400cc04340fb41fb55fb5a2b4031483050f597c001c521c3ff", "card_id": "LOE_047"}, {"dhash": "78deadf8522229c97615adea16b0ad6573dbc42498c1300b4136b3edcc4b09b7", "average_hash": "c703e347f1cf7896c8a668b6bb0df02fe4bffdbbbd3318b30892880008030007", "name": "Tuskarr Totemic", "phash": "99157b6a9ea6bf2d7b581ec28e44269618c48685e6b677942da45c719bd32ac9", "mana": 3, "whash": "c713e34ff14f7897589468b6be2df0aff5bffdbfbdb318b30892080108030007", "card_id": "AT_046"}, {"dhash": "0afe0cfc01a04b00b7345e41bcdc5931b0e2ecc5d92ba3d64e311b626480193c", "average_hash": "f71ff19ffb9ff01ff01fd01c881c981b18181018101a000a080a180618031100", "name": "Dragon Egg", "phash": "eb697bfddb96df9cbfa30da26ef006c226699b842c1c3a60bb0193616e821e0a", "mana": 1, "whash": "ff1ffbbffbbff0bff03ff43e8c7e985f18583838103b100a0c0e180e99031100", "card_id": "BRM_022"}, {"dhash": "eabffdff01100f601ed03c007800f000c5c193832f737ce4b219640fc838bf67", "average_hash": "0700e10ff33ff07ff87ff87b787838701c608060e061f041fc01bc3180319e21", "name": "Renounce Darkness", "phash": "a132f9ec5b91765a8ed6e6b9306e8cf3c6e8a41027cd9c23cb5433244f406f8f", "mana": 2, "whash": "0700e30ff33ff87ff87ff87ff87838701c60c060e061f041fe41be318031fe31", "card_id": "OG_118"}, {"dhash": "88b02d6752f33d047312a56cce998c3b38f76eecd150a089e1329fe7728c4f13", "average_hash": "f74f63ffe1dff8dff0df9bcd9b8c8b8c0b88038b0380088200c740d463100700", "name": "Tuskarr Jouster", "phash": "5d8dd95aa3ecdd1c3f5218c22f4121ac08c03339999c3ec5bfb39a7938ae26d3", "mana": 5, "whash": "f75f69ffe1dfb8dff8df9bcd9b8c8b8d0b8c0b8b3389008208c740d463104f00", "card_id": "AT_104"}, {"dhash": "cac6859d126609597e862dfede28b1d16033f74282450c8b1a3b31f6660cd9b8", "average_hash": "f701f111f903f816b832383c7c3d3c3cdc3cdc3ec83c881cc818c008c808f008", "name": "Warbot", "phash": "2327f3b4f2783b7b6fc92c996bd01c274630590461587e46dcb0db00b7d337c9", "mana": 1, "whash": "f719f911f953f836b8303c3c7c3d7cbcdc3cdcbec83dc83ce81dc019c808f008", "card_id": "GVG_051"}, {"dhash": "c8fcade9c2d24d267e4ee0901161f7c62e7dd0beb4494b82863f59f73b4ce310", "average_hash": "3717337f215f307d70f87ae3736342f302bf02be0db40d3560bc60dc7010e000", "name": "Druid of the Claw", "phash": "ed9cdf8163789f337a60d3c0877141e12091aee3ebc0db811fab34b43c0e3b5c", "mana": 5, "whash": "37173b7f235f307d78f87ae3732342b302bf12bc0dbe0d3f60fc605e6018f000", "card_id": "EX1_165"}, {"dhash": "c8ef0cdf018a0b36366448c8b0944091833a097590ea26957f324773f8861538", "average_hash": "f799ffb9fbb9fcb93c383c389d400d4005400100050004500c300808090c010c", "name": "Fencing Coach", "phash": "273de32163017ea45ee2066b9eb062c83cb406de3be7fae2ee784e4e66a221e4", "mana": 3, "whash": "fff9ffbdfbb9fff97cf8bd789fe00dc0074001c007e005d00c700d980d8c032c", "card_id": "AT_115"}, {"dhash": "fd3f6297897b36a24d009f7c2e1d79d8807050e0ef1b26e6c90f36415c909e7f", "average_hash": "07e0e693f837f80ff82f187cb83d3c7e9c3e1e1c7c3e423a1a3e980df007fc3f", "name": "Anyfin Can Happen", "phash": "2b56b50832b03e2417bd1c227d48469b9d4cb33b8c93b426853f8c96fe97a977", "mana": 10, "whash": "07e0e69bf837f80ff82f1a5cb83f1c7e9c3a1e1c7c3e420a081e980de007bc3f", "card_id": "LOE_026"}, {"dhash": "daf9ed62129d04f279c5e3360f486f82d70cbb1d703ac240a4bb48f2114e471e", "average_hash": "2703616bfb4db849f0c7f8877f0ff80ef227e207c107610e608d60456004000c", "name": "Malorne", "phash": "fbca9f6feef9ff6cbf66474c1e869130085006820799cf04cd614361db02dbc1", "mana": 7, "whash": "6f13e37bfb4df84df88ffe87ff0ffa0ff227e207c187600e608d60476004000c", "card_id": "GVG_035"}, {"dhash": "0ab84d52808d3923734fe61a8c6714cf009e073c3b50c6cb9c337be6ec88b121", "average_hash": "f7fe5bee0bde08d80ad803d827dc23cc43806382638761e444d8f9d8fb107b00", "name": "Shifting Shade", "phash": "770f9d63a3635176ff302933db90336900d0d39118ed22305f17da0cef27de15", "mana": 4, "whash": "fffe5dfe09fa00d80af80fd807dc23c843c06380638765e468f8f9d8fb187f00", "card_id": "OG_335"}, {"dhash": "0cd90c360168cad0b50d73f7a66e4dff929ae5259b493816fb3896723487e93c", "average_hash": "f713f3a2e382800700070004c007c00f700b7c3b6c33ac21fc01f8007804790c", "name": "Dancing Swords", "phash": "f3ce1c34f727a36d7bcc1ae38d94c3ec99c91c1eb3e15adabc4c29f90006d240", "mana": 3, "whash": "ff93ffa2e3a6808704070005c147e14f714bfd7bfd3bac21fc03f8807904f90f", "card_id": "FP1_029"}, {"dhash": "80bf8dff30980f007c047d81fb26e0ddc01b33e7048e191ce63bc8f783af3f46", "average_hash": "073c03ffe1bfe07fe07fe07e007f407f806f806c884118412043a043807f8e0f", "name": "Spellbender", "phash": "b93a69405f0096ab0e46c48c0633343a87c899876b7ced7563996a957bf3d9ec", "mana": 3, "whash": "379c82ffe1bfe07fe07fe07ea07f407f807f807d8841b0412043a063807f9e1f", "card_id": "tt_010"}, {"dhash": "98d46d29c21fccfeb9e077018ca6590ddf32f7e4c80b99173a36766dc093393e", "average_hash": "6700036003c02040a0c180878409c009c40d9c0fbc39bc39fc39f83df80ff91f", "name": "Harrison Jones", "phash": "2b5bde88e784a7497ec43195cb604382233ce29822e45e3fe7c59e634f0b7dc2", "mana": 5, "whash": "6f10076003e02040a44584d7c559c159cd1c9c3fbd3bbc39fc39f83df91ff91f", "card_id": "EX1_558"}, {"dhash": "3afc64f0d1860b9135315f66acee59e5b39a1f956d0ade173837706c6080cf3e", "average_hash": "c707c19f491fc81cc00c401c801dc019e001f00df80ff819f819f800c000c11c", "name": "Angry Chicken", "phash": "73ff9d96dfc7d3f83af89c8dcfe232b0273012038fb13614ce041b09db017c48", "mana": 1, "whash": "e717c9bfcbbfd83cc81c401cc01dc44de45df40ff80ffc1dfc39f811c100d11c", "card_id": "EX1_009"}, {"dhash": "cae1ccc311044b48b77f4edabd107e31f862f04560cb00950fb0ff629987113f", "average_hash": "371e199e7b9820300038bc3cbc3fbc1f1c3f1c3e0c1c08040000300030071107", "name": "Gilblin Stalker", "phash": "c97d6f65f6c45ec6cbe35b780c13c0f01cf0cb90d310d31463107703fb867f26", "mana": 2, "whash": "3f1f1fbefbbc20b80038bc3fbd7fbc7f9c7f1c3f0d3c0c1c0000f0033107110f", "card_id": "GVG_081"}, {"dhash": "e8f5cc0b11bf037736ca4c06b21c63f9e4c2c9058e0970961f3a2976d085fd3b", "average_hash": "770c7b8e7982381bf833dc379c313c03fc107c30383000306810281800080106", "name": "Demolisher", "phash": "c39dd3c17c8066e05f761c8eb9d623a98ba10c7937fe3cf39344c6c726a22631", "mana": 3, "whash": "771c7bbdfba1f83bfc33fc379c373c73fc31fc383c30083c68142818010c7107", "card_id": "EX1_102"}, {"dhash": "58feacf941e64b04b6484e99a8627097c14a07353cead8953139c7728a83253e", "average_hash": "87978fbfe3bfe0bff03f703b70196030d420f40084018c010c01080301030102", "name": "Frostwolf Warlord", "phash": "bbbd69249d0d16b36ee286891b39836025a6709636639ee1af87be1cf6a8c662", "mana": 5, "whash": "8f978fbfe3bfe4bff4bff43b7159e978f560f540b5418d410c010c830183012a", "card_id": "CS2_226"}, {"dhash": "6af2c4cc91110b23b6fc50d1a7024e87fa7efcfd912b0046deb831736386e53d", "average_hash": "979f1fbd9bbf9cb93430f401c147c1464177213f313600204000c0004100c100", "name": "Tournament Attendee", "phash": "fdf389f16fd3cde159ae527aa6b8a06c843646321b019e4973624932bc29f34c", "mana": 1, "whash": "9fdf1fbd9bbf9fb934f1d541e5c7c1c7437f217f397411e0c040c4804580c320", "card_id": "AT_097"}, {"dhash": "0ad96c6251c50a0ab7104f2da6524ca7b9cef79db03a42479cb5016ac79b0d36", "average_hash": "d717c3ae6b8ef01c601e003e901eb00e103cb03f201fe02ec00fc00000020100", "name": "Infested Tauren", "phash": "69f26f6636063b9cdfa532e28da4206d22c846e23369a4e56e99da29e722cf8d", "mana": 4, "whash": "ff97efaefbafe4bc343e803e915ef14e117fe17f215fe16ec00fc00801060100", "card_id": "OG_249"}, {"dhash": "f838dcc000c11f9e274e2c22d970f699d930e3428e87351fcf261c5568ab416f", "average_hash": "c787730f331e381c383c3c3e1c32cc34c83cd818581218173006f000e001e007", "name": "Tentacles for Arms", "phash": "eb774f38db381bc437069b34894c210b19d2140386a19669b764f3b29fbb5e1f", "mana": 5, "whash": "c787778f3b1e383c383c7c3e1c33cc35cc3dd83cd83a5817b817f006f007f007", "card_id": "OG_033"}, {"dhash": "08f0edc1d225bd49722d805ec1fd8ae355879b0ce5110e0bc83630e443897f12", "average_hash": "b75f1bff19de08fa0cf00fe03fc07fc0ffe0ffa07bb070b000f000d003000700", "name": "Warhorse Trainer", "phash": "e725c90b98e8f1d0db5ad4383e10389c23e4c3a5c3fd32e67f6542f2c2b323d4", "mana": 3, "whash": "b77f2bff1bdf18fa0cf02fe03fe07fc0ffa0ffa47fb071b000f000f003100700", "card_id": "AT_075"}, {"dhash": "caf185c0128305027e4d2c9a90b0638946f2b76d0f93d88ae1351ae7e84cbb31", "average_hash": "f70ff15ff95f781e383c3c323c313c318c20e001c001c001080c480c2007f003", "name": "Acidmaw", "phash": "e39deb6c79e27bca3b589c5893d04c3404753209628c1e5757c3ed99262e9673", "mana": 7, "whash": "f71ffb5ff95ff87e383e3cba3c31bc318c21e921e801c801582d580c381ff007", "card_id": "AT_063"}, {"dhash": "08cb2d36c2d904e03bed67f39e844311866eb89c506b27c6c731bf73f88fe13e", "average_hash": "f703e363c14c704b2055e89fec3fc80f0007002f003200021006d803f81ff91f", "name": "Nexus-Champion Saraad", "phash": "d92e379a73339e603b2603856d90cac48e8923cd6be25be65fa30d333cd3cc38", "mana": 5, "whash": "ff13e363e3ed704b3c7ffc9ffc7fc84f0007002f003a00021806d807f91ff91f", "card_id": "AT_127"}, {"dhash": "c23fb8ffc0ce35796b825e2eb9146609ec32d2e18dc33807f7abfc5f01b00c7e", "average_hash": "01c0e107f00ff81ff81fd83ffc37fc274c350c31a8319811b01ff00fc007f00f", "name": "Firelands Portal", "phash": "abbf71077868b4b8965edd7086a57ceee6e3b318cc690c966d1e8c20a7a78064", "mana": 7, "whash": "01e0e187f00ff81ff81fd83ffc3ffc274c354c31a8319811d01ff00fe007f81f", "card_id": "KAR_076"}, {"dhash": "02c0e526e40901f88e745f9b9a6620856bdc8ee0fd01f71380a31d4d7f9fe03e", "average_hash": "ffff81fcd19cfc009e00c6216430fe00f7203f03be07fc0f7c0e7c047d06ff0f", "name": "Holy Smite", "phash": "43f535ab4da3dd29bd943dfe9bd886906348c3a44a1d2b9d31c15b2b5968f890", "mana": 1, "whash": "ffff81f0999cfc009e00c6116434fe00f7203f03be07fc0f7c0e7c047c06ff0f", "card_id": "CS1_130"}, {"dhash": "c287cd3f30ae6178fe80dc03bfdb7820c17d8a9b1467588ee71c9b3860f8e8c3", "average_hash": "471ce1e3f183f013f017e003c82c86700070327092f10c71647cf87e4078207f", "name": "Repentance", "phash": "b1f0f3757bed9df2cc730e9e0e9de12528782b53c3c10cbc5360539859c358c8", "mana": 1, "whash": "771ee3e3f183f003f017f003cc2ca671027032f196f18e716c7df87e407c607f", "card_id": "EX1_379"}, {"dhash": "baf0edee9299490277e5accc1913fa2d7637c9cc8091f10b27371ae4344b63bc", "average_hash": "0714014c13dfb0bfb0b98abb831b00b901bb89b71bb018bc189c585070186018", "name": "Whirling Zap-o-matic", "phash": "ed859f3166eaced73e0a1a092611a2844878f2e09ba1d90d3f876d09fba1beb9", "mana": 2, "whash": "071c0b5c13dfb0bfb4bd8ebb87bb82b909bf89b51bb018bc389c785a7018741c", "card_id": "GVG_037"}, {"dhash": "4af48d4912d308367efcb590f921b7c36e37d16ab48541090f3a1ef6fc00b3bf", "average_hash": "371b330f3bc9380d3862f03370b361bfe1b7e8b5981758167890781838003001", "name": "Screwjank Clunker", "phash": "e385ff58ee797fa73bd0a6ec8ab022b208b4c3c124cc52446d9bb649f64c99a4", "mana": 4, "whash": "371f3b4f39cb382f3872f8337cb371bfe1bfe9b7d89738167810789838003000", "card_id": "GVG_055"}, {"dhash": "0ae60dc892914926766dac82081591f926c70f0c103066c8ecb5c9e283411fae", "average_hash": "f71fff7f9bdb38b93cb93e931fd10eb03190e1826382028200c3004300010600", "name": "Totem Golem", "phash": "f709ebc9bbcb7b662b229b4c6334633404540e026fe357c15e891f0d3f26be82", "mana": 2, "whash": "ff1fff5fdbffb9bfbcb97e931fb92ef03dd0f182638222820083004302830600", "card_id": "AT_052"}, {"dhash": "e230c96184c13e23514ca6984cb11f23d2c6208ae1049e1938736082611482fd", "average_hash": "934f110ff807b88b380b38193098b01f300d3984781c781cf09df00dd00d807f", "name": "Desert Camel", "phash": "c999dbc119c67b0efd3239592f8cd65999692263b4313391a9cc93f0c7446ed4", "mana": 3, "whash": "9f4f190ff807b88f3a8b3a993899b01f300d3984789cf89df09df00dd20dc47f", "card_id": "LOE_020"}, {"dhash": "a8f70dfce2834b267f50ae4f58bda0694782ef0ff0f5fd8bfb3306e7bc4b6130", "average_hash": "e70ce31de11f301d7838103c183c1c283c26e033c01e800f901f981fe80fe001", "name": "Metaltooth Leaper", "phash": "69016fc97bca7b6636168d033cc0a2c486f84e0fb9f1d9945cf29b0d8cadd9cc", "mana": 3, "whash": "e71dfb5fc11ff03f783c103c183c3ca83c26e9b3c09ea01f981fd81fe80fe003", "card_id": "GVG_048"}, {"dhash": "5af6acec51898a1a35b458c9b1827f0dce7a9ab5b06b27d50e3200726180dd32", "average_hash": "879dafbdfbbdfc0dfc0cf002b04380618424202604362413f011f000c1008100", "name": "Flame Juggler", "phash": "733b7b9b3f031d9d9c92b2e8c66404a6188996c4c7007e44ef04bf91ff60ee66", "mana": 2, "whash": "af9dafbdfbbdfdbdfc29f402f143a165856d216625776552f811b090c180c100", "card_id": "AT_094"}, {"dhash": "68cdec9ec1a50a17316c58fbb170631b8ea27d15fe2a00d56338b66f68889130", "average_hash": "d790dfb0fbb1f8987c483c203d63d9737c41b410911f80151c06b83f91118100", "name": "Antique Healbot", "phash": "a76567443f2719a43e8496605ed846e2268219e5cf243679db307ea637b1cbcd", "mana": 5, "whash": "df90ffb0fbb1fcb87c083c203d63d9727d43b554115f81571c07bcbf91918100", "card_id": "GVG_069"}, {"dhash": "acdd0c3f01edca9b37206cb8b1426685d80a8f3d9aeb304706b61c68f99cf33f", "average_hash": "77107387c18fe00ff83ff83ff027f022c020e022403300112010300030006118", "name": "Shade of Naxxramas", "phash": "79765b064e99becc76c6a6e0a98d24e1693b53f2799c3c790cc3c6c7c3326032", "mana": 3, "whash": "771effbfdbafe0bffc3ffc3ff467f063e168e023c0330031201070007110e11c", "card_id": "FP1_005"}, {"dhash": "bac02488413b4bf4b4864ff5bd027a05e3facedd704ba0965f361a6f809c3738", "average_hash": "071f0f9803b1203180038022c033a039003b98378c3f0426f832f839001c011e", "name": "Hungry Crab", "phash": "adfd96d86f9d9fc66de9a3e45c72838158202d19b384dcf0273959429e4133e4", "mana": 1, "whash": "8f9f4fb203b1a4b584038033c17be178817b99778d7f0576f833fcbb01bc011e", "card_id": "NEW1_017"}, {"dhash": "2aca6cb7d1440a99b4754891bfe65b4d87923a05410aee17b235606f4e8c9939", "average_hash": "f71fbb97dbbff82fe024e821e418645df447fc073800b80c8005800f80010101", "name": "Mechanical Yeti", "phash": "7b1f7fc69fc3dfa1ffac39a8066092b201924cc90b499342fbc833c16921de72", "mana": 4, "whash": "ff1fbfbfdbbff8afe424ec23e45a645df447fc077800b82c880d801f91010109", "card_id": "GVG_078"}, {"dhash": "6ce08ded829305873fee6d0098087edb7c32916412cb86079f33fe6fdc8f013a", "average_hash": "870ab348395d781d383c3c3f003e083e003e401e000600021019b81ff81fc11f", "name": "Ysera", "phash": "c9ada719f3893e623d6686515b6423d12cc492e61ae893cdd93ccdc93c26f4e8", "mana": 9, "whash": "8f1fb37c795f787d7c3f3cbf143e083e083ec01f400e000a101df81ff81fc11f", "card_id": "EX1_572"}, {"dhash": "d27829cd223a7cc4d168a7854c039912223f1c7e6cc4d0d801f173a28747eeff", "average_hash": "af7be37d31df60cf73cfe3db83cb03d103810380038900c000c00040062086ff", "name": "Tomb Pillager", "phash": "5d89fb3929fe6e1fbf61b66d9b51d694919c9290919896f00d8d46c445e0eec0", "mana": 4, "whash": "af7ff37d76ff64dff7dfe3df93db03d303c103c10b89008004c002000620ceff", "card_id": "LOE_012"}, {"dhash": "0add8dba12c3093c7e793cc2f81cb1f143c3dfa2ec05910b233e22e0840279b6", "average_hash": "e7017306f3567019f031e839cc383cb07c20fcbdbc3c1c381838180008000000", "name": "Magnataur Alpha", "phash": "a395fb2cdee2ef53ded20c8b2939b29c86e40c090996334973329fa49c875fc2", "mana": 4, "whash": "e717fb07fbd6703bf871ec39ccb93db0fca0fdbdfc3cbc381c3c182408000804", "card_id": "AT_067"}, {"dhash": "d0d82d3152cfac977bffaf2c1c7871e1e0f2d1c48101130b063608e55040abb4", "average_hash": "c70723466146784eb839f83e7c387c3c7c3e7d3c393c38bc389c280c00000000", "name": "Hallazeal the Ascended", "phash": "e317bb265eaacf1caf6409653741129e10540a4979ef58665de64765ac4eb699", "mana": 5, "whash": "e70761476146f84fb839fcbf7e3f7c3e7d3e7d3c3dbc38be389c381c00000000", "card_id": "OG_209"}, {"dhash": "0cb300ce059d9f3c3eeb5cc2bc844389837a60a5cd02bd5d73bbc65c9892353c", "average_hash": "ff3fff0ff81bfc3bec39fc39ec79c41c80032800340010001010301000000108", "name": "Arcane Giant", "phash": "23293301cfd0dee05e826da02714929498e1643ca46dd2efe0e6f6fea4cfb477", "mana": 12, "whash": "ffbfffbff87bfc7bfc79fc79ec79e45e8007a800bc009051903130302100113c", "card_id": "KAR_711"}, {"dhash": "78c62c38e1e0cb1f372e5611ace65839b3e2041d672a184566b8d973e0878d3d", "average_hash": "8fb58fa1c7a304befc7ffc596d58dd58bd5d7144e100b04130405800818f812f", "name": "Grim Patron", "phash": "7fdb3f959e17b6901f93862939a043828cb12d62ece8cc71364bd274c786199b", "mana": 5, "whash": "8f958fa1c3a307befcbffd596dd8fdd83f5d7146e140314130404c80818f832f", "card_id": "BRM_019"}, {"dhash": "426bc90594a92ade75b48f6e5f3db44360c6d328e7459c9b293356a40c49faff", "average_hash": "f71ef9deb881380038849b26a83e78b609b3a1b9c09cc89c989f9811d000f47f", "name": "Rumbling Elemental", "phash": "e317c732f3ea7dedffbc89ae3976bee241dc625089381e46119462d011470cd1", "mana": 4, "whash": "fd5e794bb8803884b88c9e27ba3e79bf09b1a11dc29dc89ed89e9813d000f47f", "card_id": "LOE_016"}, {"dhash": "e8f0cc8b01b20b44b77750bfa43e4e6993d220c5888b09573eb6a16c06895d24", "average_hash": "37977fbd7bb96cba047c045b054b2d4d3d43bd41192010700430003009000920", "name": "Azure Drake", "phash": "cfdd4bc86e84f7b1db1a58e00ce30c344834cc0633cd92edd7a646b32de3bcc1", "mana": 5, "whash": "3f9f7fbd7bbb6fba04be055f0dcb7dcdbf43bd5119601570047004b009a00b20", "card_id": "EX1_284"}, {"dhash": "8cda0c3f11cb0aa7355a5b78a6e04c39bac27e85b10af7158033866c2989c134", "average_hash": "f707f387f907f80d780d780efc0efc073c03bc05bc0678023800100000000100", "name": "Priestess of Elune", "phash": "c39773903c009f253fb068e00f8684e1309833663b31fdcc9f33bbb3f9c0ced9", "mana": 6, "whash": "f717fb87fbaff82ff80f7c0efc0efc477c07fc07bc067c1f381c101010140104", "card_id": "EX1_583"}, {"dhash": "849f6df5d0820b65728aef349ccd393066c0c1046628c6fff81e332100c6fc8f", "average_hash": "0730e3c7eb97f8bb789f381edc3c9e3f2e1ff00cf00c40004c64c8054000e007", "name": "Dark Bargain", "phash": "6b87ab85b69a367a1f700dc2e0e2cc688f0d26c127936c3ecb0992b173db9e56", "mana": 6, "whash": "0738e3c7eb97f9bb789f789fdc3c9e3f2e1ff80cf00c6000cc66c80d400cf00f", "card_id": "AT_025"}, {"dhash": "f83f9c6d00d9153323622e90dce0b3c17111fd02e0050019012286552ca05d6e", "average_hash": "e787f30ffb0ff80dfc15fc1bfc19fc1efc0ffc07b80398001800180000000000", "name": "Arcanite Reaper", "phash": "43ff7399bc425ce07d440ee5161412061ca3042005c13ecddf69de26efb77e3f", "mana": 5, "whash": "e787f78ffb1ff91ffc1ffc1bfc19fc1efe0ffc0ffc079c011c00180001000900", "card_id": "CS2_112"}, {"dhash": "caf18d47929a0c75789e21584394cc297b53b6e06fc59b0b203fc062324d7d34", "average_hash": "770e711cfb0d78097803fc01bc00bc00bc20bc28dc09fc397cb8000e080e7814", "name": "Mistress of Pain", "phash": "c31bf3b86ce42f7b7e30596c83869c6c03e0184c6e62271333633f93fb4499ce", "mana": 2, "whash": "771e791cfb4d78497807fc01fc01bc81bc28bcb8fca9fc39fcba083a081e781c", "card_id": "GVG_018"}, {"dhash": "3ac7ac0c11584a60b4834871a3f64d0d9b1a2235516aaed77836636cc8909535", "average_hash": "c79fefb3f3a1e4a1442214208444e54de44de405e500e410c431a03101018101", "name": "Mechwarper", "phash": "33dbdf528fc573265b9b0ce1cce48820290852ca8f11b7917f02ff047f26dec9", "mana": 2, "whash": "ef9fefbbfba1e4a1c4a31560c545e54de74de5456540e550e439b4b111a19321", "card_id": "GVG_006"}, {"dhash": "e887fd5f60fc83b20ee459c1b7066b85de08cf71186833826603d944849f293f", "average_hash": "07006300fd03ec11fc21f003e403e407f603e413f4037c03fc03f803f801f00f", "name": "Force of Nature", "phash": "934d3c726e93dfac7ce21a4eccc031810f120eb10a1c6b25d3f6d7ccb94a7f89", "mana": 5, "whash": "0708e341fd83fc11fc29f007e40bf427f633f613fc03fc03fc03f803f813f80f", "card_id": "EX1_571"}, {"dhash": "0cf20cc581c00b11376256ccb0187d39d2428cf539cb911667305c60b8807526", "average_hash": "f71ffb9ffb9fd81f181f981b9c239c3b8c3b04030c0318003000100000000100", "name": "Coliseum Manager", "phash": "895959836f26ffe43f10b3711a8808a86dca5646f2f6b6353b4bf3909ca7d248", "mana": 3, "whash": "ff1fffbffbbff8bfdc1f9c1fbd639d7f8d7b04730d2318203810100021000100", "card_id": "AT_110"}, {"dhash": "caf10c07611ac2e7b5fd58cab1b04f61988267559b2a36176cb2dd614b839734", "average_hash": "779e7fbe23a585a17033fc21bd41bd477c447011791348132003a00681048104", "name": "Twilight Drake", "phash": "f3bcefc97c166fcc6fb38338a3cc60e010c498611e7116e2d386db906d02df72", "mana": 4, "whash": "7f9e7fbe03a585a1f4b1fc21bd41fd477d4471517d536d53a401a485a185a124", "card_id": "EX1_043"}, {"dhash": "9ce3adcf4238c482bb6672c8849a497f97822405590a83976e30f069f487493a", "average_hash": "071ce37debfbb44794d8b8c9904d184e7c037c03f900f800f8005800d900d918", "name": "The Black Knight", "phash": "734b7d383f86ff5c4bc24c932c9182c04474c8729a92d3bd4f0c6f6fec8831a7", "mana": 6, "whash": "2f1def7debfbf4679478bcc9914d194afd0b7d01fd02fd6078005880d980d908", "card_id": "EX1_002"}, {"dhash": "12b5686ac19c9f39399356a6afec5301bcc270058702369d483a7967ee9ff978", "average_hash": "e7a3c18de00de81cd808cc025c12fc03f80d701c500060020800c001d11ff17f", "name": "Gorillabot A-3", "phash": "636f3bcf39537e98fb9c96e6dfe41a33931819e565b049cc32248a1411b0cee1", "mana": 4, "whash": "efabf1adf0bdfc1ddc08dc427c5afc43f85d781c780868420800c019d11ff1ff", "card_id": "LOE_039"}, {"dhash": "8aff2dce129805247e48ac8159b3a1464b1b276648c1708a85373ee4f449df91", "average_hash": "f71ff11ff35ff87ff8bff3b3b0b120a1408084010d000ca008b0001800188010", "name": "Patient Assassin", "phash": "a19dfb586bef1e5e3a70a4a413d220ec80d08c183ba57fc0f7059605ff416fb5", "mana": 2, "whash": "f71ff91ffb5ff87ff8fff3fbf2b332a166a18c810d801ca00cf0007802188019", "card_id": "EX1_522"}, {"dhash": "48fb0cf211e50bc0b6806d2db21a6833ece69fc5f30ba44741b7076e3f983530", "average_hash": "f71ff59f9b9ee03f603e0036100818001834182e003f003f003e003c001c0118", "name": "Haunted Creeper", "phash": "c9e7d3667b3d9ba1b69334a899b44892308833096e423d23ae07dfc3f36166f2", "mana": 2, "whash": "ff9ffdbffbbfe8bf743f003e156c1968196c187e113f803f003e003c013c0118", "card_id": "FP1_002"}, {"dhash": "18f00de2029c05bd3b706cc099c07701e2329144aa8b7cd70736fe6fe09d1938", "average_hash": "e70ff35ff15ff01ff01df831f831e03340134000c03080307830f818f01ff01f", "name": "Mimiron's Head", "phash": "29c75b8849299e46db4c27141ed203ca18d81933e6ce1c7d77e54f9b4d13d7cc", "mana": 5, "whash": "e70ff35ff15ff05ff83ff831f831e037c033c000c03080387830f81cf81ff81f", "card_id": "GVG_111"}, {"dhash": "6cf0cc3001ec8adbb147531ea6bc7f7984223925c2cab095613b437696950d3c", "average_hash": "970f1b8f1b868006800f380fbc07fc3ffc0ffc0f8c0b0c0b001b0819800d010e", "name": "Windfury Harpy", "phash": "cb4bdb907c902f6de39689895a7819d8396434334a2cc6b6ad8dd96273b992e6", "mana": 6, "whash": "971f9faf1bae8887840f3c0fbc47fc7ffc4ffc0fdc0b0c0f081f0819810d011e", "card_id": "EX1_033"}, {"dhash": "4cfd097b02e00dd47b73ef8018106138fb6084c5089bc3008e3b18f7644ecb1c", "average_hash": "e703f347f14ff81ff81ff83f9d3b8c219c3918311808300c700c6006c004c00c", "name": "Aviana", "phash": "e307bb297ae6fe18be511f644f0101b540983e2e8eadabc1597f50dafe028785", "mana": 9, "whash": "e707f34ff94ff85ff81ffc3fdd3b8c219c3998393918380c700c7046f00cd00c", "card_id": "AT_045"}, {"dhash": "cadd8c3b11e20a84307c53e1a5425a8f9b3a3eb5e54ac39606392d72f8857137", "average_hash": "771f7d9ffb8ff8067c04f001f001e000c009e00c2c0f3c0e2c04300030003902", "name": "Frostwolf Grunt", "phash": "f3d8f3c5c744499cbde536ec0be066320c2605343a823b437f10fe23ff109f93", "mana": 2, "whash": "ff9f7fbffbaffc87fc04f401f141e140e149e94c3d4f3c4f3c04380039023902", "card_id": "CS2_121"}, {"dhash": "0a992d7152f6bcfd7b4ae490c40f9a330146669e0130fa49c0b38fe41b087f36", "average_hash": "f75665e62bce00cf88ffa3fbc3d983c893800381038003d600fe01de03100718", "name": "Fallen Hero", "phash": "5d29b34aabeef636366221e43b81899909e076647dcdb601ee12ee057f85364d", "mana": 2, "whash": "ff57ebf66bee00efcaff8bfbc3db93c883c0038103c003d600fe01fe0312071c", "card_id": "AT_003"}, {"dhash": "c8fffde761980f307ef098c233214642cc4d914b8e937c66d84c21d310cefebe", "average_hash": "0700830fe13ff079f879f873f867f867b063fa619a71c2748224000e0006e007", "name": "Powershot", "phash": "b10237494cfd5ee2d21c5e163685a6e9666224c967123af623cd3e964bcbd935", "mana": 3, "whash": "0700c34ff13ff079f879f873f867f867b863fa61da71d27c826e000e0016f00f", "card_id": "AT_056"}, {"dhash": "0a81fd83e014806f57de983d30f326c67d9c97393df2f2e4e1cbc70ecf353e0c", "average_hash": "0700030041000c001c185a00f200f200ee21ee31ef43ff07ff0fbe0fbe0ffe0f", "name": "Shield Slam", "phash": "0340e4f74cff3b9c77b6cf611be809194a44966822f439904bcc798da9b57f64", "mana": 1, "whash": "070003006b101d001e18fe00f620fe20ee29ee31ff43ff07ff0fbe0fff0ffe0f", "card_id": "EX1_410"}, {"dhash": "e8fe0d5192a22dc1732aa7d4ce6d90db29375a4ce01003c8cfb1b0e689810736", "average_hash": "f74ff3fff1df58df18de9bdd0b9fc39fcb9c4b860b86208220c280c203000700", "name": "Cobalt Guardian", "phash": "5d87698bb268d36c3f1631380e9082e300bc9ccd4fe73ecc3ff3eab4289c0e59", "mana": 5, "whash": "e75ff1fff1dfd8dfb8de9bdd0b9fcb9fcb9f0b8f0b86008220c280c003000700", "card_id": "GVG_062"}, {"dhash": "0080cd3fe0efc34d9eb11c42189d6778c3e3bf0f671e18e0b087ed0ff2bd08e0", "average_hash": "271c01e001800c105e320e300f309b60386078f478fc9b098e0be61fe07fe03f", "name": "Holy Wrath", "phash": "ad1486a479edd78bf5131f102dc401fe8641291f0eb59a537ba879f252c41deb", "mana": 5, "whash": "f79f03e001800c105e360e308f309b60b86078fcfcfcdb0f8f0be61fe07ff07f", "card_id": "EX1_365"}, {"dhash": "aac0a533c25d8cef799ee72f0c1cf4f040e181c63718ec00d03be0f1f947131e", "average_hash": "071007600b6038405ccfcef6ff7a3cfc1ee43fa0ff80fb0030c010c200878607", "name": "Forbidden Ancient", "phash": "c7b7bfefb8b8f69b8fd169e03ba49cc409d1c309a6313a0079a249909c05fbd4", "mana": 1, "whash": "0f110f600b6039e05ccfdffefffe3efc1ea43fa0ff80fb8070c410c2008f860f", "card_id": "OG_051"}, {"dhash": "823f21d844b4594bb2946621b9107321e64ac4bda1637bde463c8c481a91753e", "average_hash": "7bfcf9dbe8df44def4fe76fa197805730031003021343077343330323120293c", "name": "Moroes", "phash": "edfc31eb530ebd9a3e9b934c6fb27661992466962c4b8a2e66129359890632d9", "mana": 3, "whash": "ffebf9ffe0dfc4def4de74fa1b780b7103700130213430773423303231302b3c", "card_id": "KAR_044"}, {"dhash": "08f6acecc1d34b61b6964927a64c59f9a3120125380ab7d40cb9f173c687192e", "average_hash": "f79fcbbfa3bf70bf1c375c3e4c4f5d58fc42ec000c0000006000c00001000100", "name": "Argent Horserider", "phash": "1b4fc981f991720867037bcacee4b3a6e3a4ccc9679873d8cd27c6edb00c9a64", "mana": 3, "whash": "ff9fffbfebbf74bf1cb75c3f4d5f5d58ff43ed400d4241406400c48001801120", "card_id": "AT_087"}, {"dhash": "4a3b994622ac1cb1b3d6e73c8c5070614ee2b9154222e3dd0e3b3462099cf17f", "average_hash": "f746f1c6f04ad8c9d099d81ef81af83b7806f8061000201c301e10081108c17f", "name": "Arch-Thief Rafaam", "phash": "e36333a23d33f638bb06ef4d3d829ed31d1c365923224a381598e671396cbec4", "mana": 9, "whash": "f74ef3c6fe4afc89d49dd89ff81ff83b7806f80e504a205c301e100d1108c3ff", "card_id": "LOE_092"}, {"dhash": "4af68d4a12143db1784de5f2cac1800724b6483c9fd12b4bc6b20de57f02df34", "average_hash": "f75ffbfffbfb98e38ce16fe06fe093e003b003b003b203f400f401c003000700", "name": "Dalaran Aspirant", "phash": "fd95f34be3eae931ff4071b03b1c0c8c18d11a7345b46e04ba666a6137299e1d", "mana": 4, "whash": "ff5ffbfffbffd8f78ee1efe0e7e093e003f003b007b303f400f401c003000700", "card_id": "AT_006"}, {"dhash": "0080dd37b0cf223d3ffe5d70bac16586893a3e45008a00d60db8a2415300fcbf", "average_hash": "1f7c07e01384d828d879f077e87de07fc077887f800f0040400004000400f00f", "name": "Sacrificial Pact", "phash": "31d2af034ecddea939302794898d0f2821420e5d4b514673cbe09ba56fffb63d", "mana": 0, "whash": "ffff07f01384db28fc79fc7ff87fe277c07fcdffc07f4040404004000c00f00f", "card_id": "NEW1_003"}, {"dhash": "4618893006423c8c70b0a5e04a9d85331847738c9404671a7837c6a4094bc23f", "average_hash": "f347f947f8c7f0c7fa84c384c285d085318cb18631867083908380830200841d", "name": "Ethereal Peddler", "phash": "d133f3ababecd36dbbb2b98b7f644e2765d88244397342d0cd4c16f121011b19", "mana": 5, "whash": "f967f947f867f8c7fa84eb84c287d2cdf98db98eb986f083c08390938205803d", "card_id": "KAR_070"}, {"dhash": "76b4c061851d9fc7387857c6bc2c49993a1241e40602f8dd0f3afb51c687993f", "average_hash": "8bafb9aff82ffc73fc04dc4dec7bfc09fc0c080004000010e000f805d90f993f", "name": "Pantry Spider", "phash": "13dbc958d913fcb4d95b3bcf8f895306e6249239a53c0c87339346464dc1d778", "mana": 3, "whash": "8fafb9aff87ffc77fc48dc4dfd7bfd09fd08490005000010e800f805c90f993f", "card_id": "KAR_030a"}, {"dhash": "aacb0df60300cf0cf6f12de34b8cd71a27676ec9d834b14862b9c9e39b47372e", "average_hash": "7712df78cb7c04f404f047f087f38ad38bdb9bdb93df80cf00c700c700870627", "name": "Summoning Portal", "phash": "9da527a7df5efb69af05927c676096cc009cc898cd98491a590e5dc46b067367", "mana": 4, "whash": "7f16df788b7c04f404f047f08ff38ad38bdf1bdf93df81cf00c700c700870607", "card_id": "EX1_315"}, {"dhash": "0cde2c3c41c84ab4b5405229ac1250b7aa6e40dd00aafd44e7b13a62f3811d2e", "average_hash": "f793efa3e3afe4afe06fe05ff35e514053511100110010410000000081810124", "name": "Silithid Swarmer", "phash": "fbf33b113f1e9685b7a533693f6908c989c896b066369be9cc8c98c681837962", "mana": 3, "whash": "fff3efa7e3a7e7efe4efe55ff3dfd1d5135111c011c011c10040058085832b24", "card_id": "OG_034"}, {"dhash": "9cd42c4161984a79b59e5635bbaa64d39d823b35f26ae457ccb5d96f728fa13e", "average_hash": "c717638163800008000a101410140804600ee00ee00e603e601f601ff81ff11f", "name": "Am'gam Rager", "phash": "69c6b681bb99734c3b938db41b6109698ce21c937e726eb0f69276236d26de60", "mana": 3, "whash": "ef976fab63a804a8101a101415741964714ef14ee14f607f703f603ff91ff91f", "card_id": "OG_248"}, {"dhash": "0a80d57fb0f9418e9f0b0f1e9cbc79f9ffe0d000ac010017602e434e36987c2b", "average_hash": "07180140818c001f401e001e380c3c39fc3ffe3fb83f003f003f083f083ef81f", "name": "Wisps of the Old Gods", "phash": "69dd76fa7eeebff5979fac4d7bd00dc409288b45c3414b8259c15280732219d8", "mana": 7, "whash": "071c0360118e911fc01e601f380cfc3ffe3ffe3ff83f003f003f083f483ff81f", "card_id": "OG_195"}, {"dhash": "08f9cdd1128305067a0dac3e10fd5981e702300d47f2df0b3c3560e0814107b2", "average_hash": "f70e735e395e383c283c28bc7b28fc11fc2109084188708cf0ad8003c0038002", "name": "Al'Akir the Windlord", "phash": "e30cefa2d9e4734bb36433d66c4418dc219452cc4eebd75699b453a7ce689053", "mana": 8, "whash": "ff1e7b5e395e387d383c7ebc7e28fc11fd216d08f38878acf0adc043c003c003", "card_id": "NEW1_010"}, {"dhash": "ea0ffdfe613903f71eec9d407383e0a6b609c15b8cb331e6400c03f1eca6fb0b", "average_hash": "07208381f331fc31fc31f277f27e36721678027802721673066200700048d00f", "name": "Ancestral Knowledge", "phash": "a520f3755adf9cec6ce99c2e7306248f87f8847c2b166b40b3443e0b6ba11f35", "mana": 2, "whash": "073093c3fb33fc33fe33fa77f27e36fa16790278067216730e7200700078f02f", "card_id": "AT_053"}, {"dhash": "38cc6c3a41e6ca18b57052c9a10e413f83fe1e1d7d2ad0c483b97f63db843d3b", "average_hash": "dffbcfbbefae75ee34c89fc187c103c0034003c083c101c104c035c81d831b33", "name": "Spiteful Smith", "phash": "9fd93b66eb453e869f811cb11b8126310c98493063b29a96ef8196edfb69b319", "mana": 5, "whash": "cffbcfbfefae67ee34f89dd187c103c00bc043c083c105c104c035c81dbb1f3b", "card_id": "CS2_221"}, {"dhash": "fab7fdf91146cba39e5138917122e74ecc9d383893a00347eee89111783dffdf", "average_hash": "0700f3493bbeccfd66fe067a00636067006662060606326060620e022270fa7f", "name": "Tracking", "phash": "adcc2fff67f35ebdda2bb6fa2df81c2c21c21e950b07a4210f40e124c7c0d831", "mana": 1, "whash": "0718f3eb3bbecdfd6efe26fe00676067026662160617326060741e622272fa7f", "card_id": "DS1_184"}, {"dhash": "02830d3eb0fce3e82c23530c943e3a81725aed27f359c2031966762d09f8fecf", "average_hash": "77bec1e1c38bc00fc847c82fc41b741ff01f6c7f683e4b1e001480130008c40f", "name": "Arcane Explosion", "phash": "795fbbc29eb1e70c67688ef58d1e897243c26b0803797a414b13178e5ba9f823", "mana": 2, "whash": "ffffc3e1c38bc00fc84fc82fd41bfc1ff01fec7f683e493e001c801b000ccc0f", "card_id": "CS2_025"}, {"dhash": "78deed3992703ce979c4873a8365641b4976db6db4930d09d33276e7ec8f9b1c", "average_hash": "8703f363fb43f843f8837b837f00cb008f009b001b80198018d018d81b1c1f1e", "name": "Tirion Fordring", "phash": "d787bb58c9e996b73b12040c4ff00c2f00f466c6e6fcee307f2d5bc621cb44c4", "mana": 8, "whash": "cf13f363fbc7f843f883ff837f00cf008f009f001f801fb01cf019f81b1e1f1e", "card_id": "EX1_383"}, {"dhash": "f8ff3dff63a0cf40dc10b07b6c634f961cc4f70cc96322a7d11844b7b228efcf", "average_hash": "0700f307f93fe47fe67fe24ee244724ef80ff00fb21bcc038e0400584000e00f", "name": "Brawl", "phash": "39a039743e7f8ed5f24076a23e180c480dc2a56c05bb169fca657a96f98f3f61", "mana": 5, "whash": "0700f307d93fe4ffe67fe24ee25c726ef80ff81ffa1bcc039e0c00506211e00f", "card_id": "EX1_407"}, {"dhash": "08e22d9042ec45f1bb6f66039fcc7f796cc2e104870b3c4678b09073cf871d3c", "average_hash": "870fc347c153c01b800fe007ec3ffc3ffc3e7c3ef031e002c003800380070119", "name": "Fjola Lightbane", "phash": "396f9fab9e216766ef199a506bc0485249d938b3666c5326b5b5d9d86c26068e", "mana": 3, "whash": "871fc357c353c05fc00ff89fec3ffc3ffc3e7c3ef037e026c003c0038007011f", "card_id": "AT_129"}, {"dhash": "6ae4ed4393f504607988a6994ca19142313f56febb35864b0030cba1a44f03b0", "average_hash": "871f1b3fdb4de84e20cb83d923dd22c183b001bbe3bff0bf00c00842001f0604", "name": "Master of Disguise", "phash": "d989fb4bcfe6bbc12e91d6b959580cec1064494c06e63b8373029e91ebc95ee4", "mana": 4, "whash": "871f9b5fdb4ff87e64dba7cb22df22e1c2b801bfe3bfe0bf00c208420a1d0604", "card_id": "NEW1_014"}, {"dhash": "8cef0cf811e20b0c37124e6dbd82490db7aa345d40aac34764b481694f963d38", "average_hash": "f71bfb9ffb9f783f583a1038803c001d000f000f0007803b003f0024001c0118", "name": "Thrallmar Farseer", "phash": "69e96304f314dbc07f29343c26e54ec2245846b8eed66dd9deb65a4e2513dab0", "mana": 3, "whash": "ff1fffbffbbf7cbf5c3e1c3e917c055d054f104f0107803f802f0024011f1118", "card_id": "EX1_021"}, {"dhash": "cccfad3ad2d54477b904721b84ce61d783e26304ce0a1c443fb9b87363851f2c", "average_hash": "1ff00ff00fe605eb84ef07c827681b61dfc1ff50ff580dc834c005c485832728", "name": "Maexxna", "phash": "776a9f089f142727b7034c50276103e182910ce363ec5e6fdf73e1ddd58e1127", "mana": 6, "whash": "0ff00ff00fe607eb84ef07e8a7e81be0dfc0ff58ff580dc834c005c405b3372c", "card_id": "FP1_010"}, {"dhash": "0280cd7ff0bb03c63efc5d1fbe4400a1436efbd8c6a19d063311664cd8b3fd2f", "average_hash": "5f7e07e02383383b207dc17f607ef038f0001e001e003f403e0038200820fc3f", "name": "Demonfuse", "phash": "e7f17dac9f9c96e179562dd209aa90b723692f0d29c09b065b601e096f817ed8", "mana": 2, "whash": "ff7f07e0239b3b7b307de07ff07ff038f0009e001f003fe03e203e301c20fc3f", "card_id": "AT_024"}, {"dhash": "88f30c8c01218bd2360f4f7ab8847349e7b2f06581c91a9327364e6c1898f130", "average_hash": "f70ef31db903b8338827383dec398c3e4c3b0c390c3c0c383838383830180100", "name": "Tinkertown Technician", "phash": "a91ddb32f2e16dcc3f0a65a06e92cbe41ec66c303365becccee2c93919139358", "mana": 3, "whash": "f71ffb9fb99bf83398377c3dec3bac3e4c3f8c3d0c3c0c383838383830188110", "card_id": "GVG_102"}, {"dhash": "0af924e341c20afdb52a5305ae7a5985a6229fddf2abe446db33b86d0e87b33c", "average_hash": "f71fed9ee99e682cd00fd00f10095011d000d004d012100c300ec007200f211d", "name": "Argent Squire", "phash": "79f673ce5b1c7366b3ed99e899b113e626c024c91e8836cb330dd3611c03db34", "mana": 1, "whash": "ff1fffbeebbe74bcd41ed00f115d7151d140d044d13a105c183ec01f310f211f", "card_id": "EX1_008"}, {"dhash": "9ad9182d45984a33b76456d9a482591fa40a087534ebe00647398d700b89153e", "average_hash": "ffb7ffb5cfa18de1b4f9b758a3dd83d303720560036019f20c5225c025901b3c", "name": "Faceless Behemoth", "phash": "bf793bd04f1496c67fa4632c6df3b07c89ef907d585a866d9b0016c04f213b90", "mana": 10, "whash": "fff7ffb5dfa185f1b4f9b558a3dc81db037207e003e019f20c52258025b00b3c", "card_id": "OG_141"}, {"dhash": "0ad20c24419bca30b5e35003a34e4e3d87da72bdbfea7555803226646d8c9738", "average_hash": "f7bf9fbd8fad85e984698c450742054083418140f110f1d4183010b801b80318", "name": "Zealous Initiate", "phash": "7df3db9a6d95e3f67fca4ba673e6202892207164160cd3103b0c6c0fff2218ce", "mana": 1, "whash": "ffffbfbf8bad87e984e98dc187c207c08341a3c0e3d0f5d41c5c15b805b80338", "card_id": "OG_158"}, {"dhash": "90802f3e80fb7925f342ecd89ab7256f6ed6d08ca1404e8988309ce37186a31c", "average_hash": "47c023e081c1d0df40d903bd83a5e3a3f9aff9bf7d9c7986788621c403040304", "name": "Cabal Shadow Priest", "phash": "f591b5c90e624326bf5014066e8441c5016c7be989e39b17ff31ba61b33f5a75", "mana": 6, "whash": "47c023e081c150df50f903bd83a5e3afffaffbbf7f9c7d8678c631c403040304", "card_id": "EX1_091"}, {"dhash": "01c0ed37600e81d90ca31b7c1c7931f3770cdff03e007918e034c0cf81fc3ef3", "average_hash": "f7bf7bf0fd84cc00ce46122e091bf809f217d607ec07e007f003c007001bf037", "name": "Forbidden Healing", "phash": "1b546786f4be53b1ed41f84989700c4013f01a3d23dc3be54bd35b69db26b7a4", "mana": 0, "whash": "f7ff4bf0fd84cc014e46423ec81ff81bf237d617ec07f007f003c007001bf07f", "card_id": "OG_198"}, {"dhash": "f4dcad7f02bcc543da80ad001edb7c3fe16e84fd3833c148c0b48fe73a4e37bc", "average_hash": "07012365a34fc05fd0f782bf037e823e813d03210384028400c40044000c260c", "name": "The Mistcaller", "phash": "3d92bb2156a8964dae480486169092b721f129295eef5dc5c9d9d747e7e267b2", "mana": 6, "whash": "0f51af67f3efe67ff47f8eff877e827e817d03270785068400c4044c228c265c", "card_id": "AT_054"}, {"dhash": "68eeadd9422785da3a876d2e9ddc63b1c0c2dd24b64b1a9687311872f48c9533", "average_hash": "c70ff77f7bdbf876fc467ca29c309c3618306830e43c282b3c08480840003103", "name": "Feugen", "phash": "a36fdb267a087dd8bf814e4339303282925229376eb6f6c9dd4c4f2f398624b1", "mana": 5, "whash": "cf1fff7f7bdff877fc727caabc70bc763831e834e8382c293c08400821103103", "card_id": "FP1_015"}, {"dhash": "3adac5608203049c79fe77108e6e4cf9b9126335c6cbfc9781b28d713f8ee730", "average_hash": "e707df671bcc384018c018de005e481eec0eec1cc40c843f001c0006011c8100", "name": "Chillmaw", "phash": "49b6c3e2b6d93b7e7b36599c3bd9220cb21166dce638693863194cc19e8336c3", "mana": 7, "whash": "ef57df7f1bec3d601c402cde055e4d5eed5eed1ecd3c8d7f001e0087019c8120", "card_id": "AT_123"}, {"dhash": "72918d23008c7831f166e4d98bb716cf0f1e1d3c7a7024c9c8b091e12383472e", "average_hash": "4ff63fec3bec14c007d103d023c563c6c3c3c3c383c243c2c0c241c243824702", "name": "Spawn of Shadows", "phash": "5f66e59bae6ef166ff4e45b62f8110e100c46ab010d83a846f676a14fe8b3b45", "mana": 4, "whash": "7ffe3ffe7bec14e087f187d127c563c6e3c3c3c383c343c2c0c241c247824722", "card_id": "AT_012"}, {"dhash": "9888fdffe3efefbe1e41310c7000c0ec8ff9f845601b93152322665e40c0fc9f", "average_hash": "07000300013efefffeff8effd67f0064c4677c37183d0019900800080000f00f", "name": "Blood Warriors", "phash": "a982af047e79fe955a663e2129888714136a24c9e3563e3dd969de365bc893e1", "mana": 3, "whash": "070003403f3efffffeffbeffde7f0266c4677c7f187f001d981880080000f00f", "card_id": "OG_276"}, {"dhash": "2ad9cc3251e582e8b5d14bafbf107501e012c2a508c97716d0b26074c78d9f3b", "average_hash": "1706d386cb86e007c003c027f83ff83f781c9c398c2458260017801980038101", "name": "Voodoo Doctor", "phash": "b977db665e66ef96afb9aca096d21c2861ac07582d19c3867e48db9133076c60", "mana": 1, "whash": "7717dba6ebafe807e407e027f837fc7ff83bbc3dcc265836a017801981038109", "card_id": "EX1_011"}, {"dhash": "48fa2ca5d1620bcd363a51f1a6825925965a203540ead6973b36a66c9c8d7139", "average_hash": "f71ff3afebaff836781e700fe00f804b800b0408040000201830381038183118", "name": "Shattered Sun Cleric", "phash": "739c7396e71536783b69b3e166d2266489e59a863b9673ce3e9b26cd34021948", "mana": 3, "whash": "ff9fffafebaffcb77c1ef00ef14f814bc14b054805400d601c3038b839983918", "card_id": "EX1_019"}, {"dhash": "ecdbadee02c14d82fb1fe62608f9d21231656dceb398e701c63a86f70c4f9b1c", "average_hash": "6710336cb35f105e00de4ade1b1c388c188c18823986391f388638ce100e100c", "name": "Druid of the Flame", "phash": "cd233d09bada964cb68237846fc44ce428d0ebf1a6ed2fcd738e5964bc20e6d0", "mana": 3, "whash": "6712bb6cbb5f907e04fe4ede1f1c3ede1a8c1a823b86b91f38ce38ce180e180c", "card_id": "BRM_010"}, {"dhash": "80c10f7a9484794bee947c31f3f6ec8ff7900a716476c88e308f601cec309803", "average_hash": "739ec0e2c09f001a50326476646ee67eef7fe767dc4dce7de639d67890701030", "name": "Velen's Chosen", "phash": "2162d864e4411b1bd51c27d8c696b7202b8c6261587c2fbf4bfd73c759c29b9b", "mana": 3, "whash": "f59ec0e2c09e801250326474e47ee27eee7fe667cc4fce7de679c67890701030", "card_id": "GVG_010"}, {"dhash": "c8f00cf811630b83360f4d36b9ee7299ec3a81754aca3c15c7379e6c30987130", "average_hash": "f71e7f9f3b9f383e1c370c378c33cc32c43c44284c001c00383e381c00000110", "name": "Youthful Brewmaster", "phash": "edf7d990cb19674b37cd93259682009201b216626c488d81bf25de69df52ffa5", "mana": 2, "whash": "ff1e7fbf3bbf3cbe1c370c37cd73cd73c57c4c784d001c40383c383e09180110", "card_id": "EX1_049"}, {"dhash": "ea1ffdffc0c083390fd21c0c3818f060e1c1e001fb83e707863f3c6e40d8f0a3", "average_hash": "0700e107f91fdc3fde3fde3fbc3e3c3e3c167c18381e380e381c201c00000001", "name": "Windfury", "phash": "cbf51bc77cdff63cf630ac321cc1240c86b60cf225cf8e25db04cf0c6b097f01", "mana": 2, "whash": "0700e307ff1fdc3fde3fde3ffc3f3e3e3c367c183c1e3c0e381c201c0018b005", "card_id": "CS2_039"}, {"dhash": "0080ed7f60f881e13f00cc00bea6fdcd803b01e4d77bace313804748fd90fe3f", "average_hash": "ffff1ffcffdfcc3fd83f01b701fc01fc01e005400460863806080e000600ff0f", "name": "Frost Nova", "phash": "6d5ff98a5380b720f7be2c81a32492de094129744b3936a5d3c45adb49dbd35a", "mana": 3, "whash": "ffff0ffeffdfcf3fd87f01fd01fc01fc01e805400660073807080f000700ff0f", "card_id": "CS2_026"}, {"dhash": "ea8ffd7f609f806f0dfd02f809f037e17f02de04f00982176cb8c0408899f8bf", "average_hash": "0730e1c3fd03fc03fe03fc03fc01fc03fc03fc07f81ff839404000000000f00f", "name": "Mortal Coil", "phash": "f37d37fa7cebfdfe7bf524b0c9061ea425c00e581b60261d4b035ea6c3015681", "mana": 1, "whash": "073cf3e3ff8ffd0bfe03fe03fc03fc03fc03fc07fc1ff83bf05080000000f00f", "card_id": "EX1_302"}, {"dhash": "0180ef1fe0b7c1dc87310bc31486a8ccd319273348e6b498d930b3656cccf88f", "average_hash": "ffff0bf80dc2ee03e601e780e78007c087e08fe00ec00c800c801d801180f18f", "name": "Circle of Healing", "phash": "177e3b00e9078d2ffd616f01335033820bd40ae45b5c5b6d5b077ab95b2a56ba", "mana": 0, "whash": "ffff0ff80fc2ef03ee09ef89ef8807c08fe0cfe14fc00fc00d801d8019c0f18f", "card_id": "EX1_621"}, {"dhash": "eaf90ce511028b1436004d3fb8fc6369cc92c125b7c92c977f36fc69f493893e", "average_hash": "771e799e7b9f283d88310438cc30fc277c307c300c300c303830382018001900", "name": "Recombobulator", "phash": "a79779b8dbe1c78c73e83c702ce712849230180ea7857f08f988d3c33f07af27", "mana": 2, "whash": "7f1e7fbe7bbff8bfac310438fc78fc677c727c304c300c303c307820f900b900", "card_id": "GVG_108"}, {"dhash": "c70fb8ff298b1716ad375c2c807000616186201c7868d0d10da7f045ec8f3f3f", "average_hash": "0ff003d0007e817f817ed97e393eb91fe107e001e001000040008100800ffdbf", "name": "Entomb", "phash": "e97fad2ab50f1608df9fc6a7a547b42fa51a8c3aa56c8987a7762a012c146a74", "mana": 6, "whash": "0ff003f8037f817fe1fef9fe7dbfb91fe18fe003e00161007000c100c10ffdbf", "card_id": "LOE_104"}, {"dhash": "0a8f390ce2f84c70b8e270c7879a59ef978a3f7c488a6644bcb0d1732183872e", "average_hash": "ff1fe767c7c3d0cbc0c380c180410041e009e00ff803f004e005e003e1078103", "name": "Y'Shaarj, Rage Unbound", "phash": "33fa3f893b0473c23f5d4e7ccb97416e64f9c6ed29db57b1868538901e80064c", "mana": 10, "whash": "ff5fef67c7e3d5ebc4c384c1854181d1e1c9f10bf943f145e007e083e1838127", "card_id": "OG_042"}, {"dhash": "8cfc0de092490cbb786e289952a3c5641bff20212dc4cc88393963e686419d33", "average_hash": "f71ffb5f9b5ed86c94a1b2a1202022c024083b083c840c0488c1880100018000", "name": "Anima Golem", "phash": "d31beb9099834b0afb709138e39831d818ce46c6662e6e67377736a32c2718f7", "mana": 6, "whash": "ff1fff5fbb7fd8ecdca1b6e1a4e026c036c87f883f849c8c8cc188c188818611", "card_id": "GVG_077"}, {"dhash": "ea89d537b08f037714bc28f81300a6c4fd896723de4e76ad889b017fa73cfe7b", "average_hash": "07300744f384ff20fc03f802fc03f000fe00ec00ec40e064607c0c780e7afe7f", "name": "Crush", "phash": "e3fcf3f5adeb3cab1fe9ee5c932c0e85cb90866d33503c0443039c7143c81ba4", "mana": 7, "whash": "073867e4f385ff20fc03fc02fc03f800fe00fe40ecc1e074e07c0c780f7afe7f", "card_id": "GVG_052"}, {"dhash": "c0cfdd37e06609ee7d10a331c6c44187db7daed11eef659e83217a7680e8fe8f", "average_hash": "573ee3f7f387fc0778cf308c040c8041c2e1d263dc639c601c0048014040f00f", "name": "Mark of Nature", "phash": "33d27b10eb2c9f633c014e9bc950a540431b969c4bd43cf2fb51e6a4f1ea3adc", "mana": 3, "whash": "ffffe7f7f387ff87f8ff308c448c8041c2e1d273dc639ce15c0048014050f00f", "card_id": "EX1_155"}, {"dhash": "98f42cc941a68bc6340cce38bdf077c7e04ec595b929634702b6166dfd9b313e", "average_hash": "c71fe39f639f603e202f383a3838703e70323033303f103e103410381006011e", "name": "Nightblade", "phash": "e9f47b0a5b8fb9d0378226a39b1996a0499259d22669d6266fc3dbf44c589ac9", "mana": 5, "whash": "ef1fefbf73bfe0bf303f383a3878707e70763033303f103e103c103c1116011f", "card_id": "EX1_593"}, {"dhash": "e89fdd6c30b84078b1c166b3c96c90b95f66bec4c3997153cf20b43eb875f40f", "average_hash": "0720f7c7f30bd10be003641b6417f800f809f80ff80378027c02bc00e042f80f", "name": "Assassinate", "phash": "d36f3fb87f70c7932f1e0eac58640c28020734264f39ecf0278932ccd9d8b3f3", "mana": 5, "whash": "077cf7e7f389f38be407e41b641ffc03f809f80ff80378037c02bc48f042f80f", "card_id": "CS2_076"}, {"dhash": "400e9df930ff43529fac3d1b7ce4f9b0e761cc83fc0b4107875e2cbc597980e6", "average_hash": "0700030021078037803fe03c203c903bf07bf87fb87fb87ff077f073b877f877", "name": "Hex", "phash": "a9317e4dde149f4a164e66980d92b32039cf8b7166de2b06ab710ba666989fb5", "mana": 3, "whash": "070003000103803fa03fe03c203cf03bf07bf87fbc7fb87ff077f073b877f077", "card_id": "EX1_246"}, {"dhash": "badd6c3351820a80b46b5311a7465e9d803219453e8a114466b9c9732383572e", "average_hash": "cfb36fbffbb7fcfb444d744fe746cd464d600d40194311c10041008205860322", "name": "Mana Addict", "phash": "dfce7b8cfd049fcd7f6659e88ec904c3a4c90363232093c19b81de117e027ea6", "mana": 2, "whash": "cff36fbffba7fdfb44cd7d4fe7c6edc64f401dc01bc111c10041058205860322", "card_id": "EX1_055"}, {"dhash": "fa19fcf3a0ffd3fe26cc2f00dc61a6ff61d9d0c20005c71f0c2c7052f2a0ad63", "average_hash": "078003800300b03ff03ff83f703e603fb83ffc3e3c0e7c2e6000c001e000c901", "name": "Light's Justice", "phash": "e97eddf0368f06e25f046c6e0376137999c1c8a025e40f3cdbc97668273396a5", "mana": 1, "whash": "078207804354f13ff83ff83ff03ee03ffabffc3e3c2e7c3ff400c401e100e901", "card_id": "CS2_091"}, {"dhash": "cafc4d6392ef0d877b2ee67c0cd8d2b02361cdc63299ce029d3f69f692483119", "average_hash": "b709fb5c7b4cf81e789c3c981f105c101c90dca0d924680c609c281828080000", "name": "Keeper of the Grove", "phash": "e38bf36378983bd69e612d991ed0853910d41a4298b19e498eb954b4ff21be67", "mana": 4, "whash": "b719fb5efb4cf87e7cdc3c983f187c905eb1deb0dda47c0c60bc287c28182008", "card_id": "EX1_166"}, {"dhash": "9a9f6d3320647870f387e40f8bbd37634f36b0ec0151728be9b697e67d88f312", "average_hash": "47e4e5e663c600c782c303c21be03be30fa70fe30ba049b000f011f87310f710", "name": "Shadowboxer", "phash": "d5b5d99a62aa1ddeef12a4a53e92087918d0e60126e3cac2ef65f2c03d293bd5", "mana": 2, "whash": "4ff4e5e753e7c0c682d303e21bf03fe30fe79fa34fa049e000f031f87310f710", "card_id": "GVG_072"}, {"dhash": "4af085c0129f253a7ae0b0810007621ded72c8ed1353268a5d36ffe1f04f73be", "average_hash": "f70ff17d395c787ef87bd2838301002009301d300bb00cb00cb02840e00fe41f", "name": "Neptulon", "phash": "af1febfb6bfaee577bf812981b44930f00441a02efac5a1057807591362a9b8b", "mana": 7, "whash": "ff1ffb7f3b5c787efc7bc683874102600d301d381fb80cb00cb02846e80ff41f", "card_id": "GVG_042"}, {"dhash": "0218a16064d11ca6b1486501a2835e27b44a4935b262acdc38b9115263809d3f", "average_hash": "fdffc9efb4cfbccff6c777c7834383c383c2039003d040c8044001c005000b20", "name": "The Curator", "phash": "d9fad99bc95b6de7bf27f3e8eb8cc3a48169866b8d88c2c8698c739061107250", "mana": 7, "whash": "ffffebeff7efaccff5c7f7c7834383c3834203d007d045c804c005c005800b78", "card_id": "KAR_061"}, {"dhash": "f6b828e005c45ff8bb975727aada7499c112bfe5f483db1dc73f1a4e3090013a", "average_hash": "c12fe10fb01f801ec01fc01fd00cf834b820c83c183c181f781f781fe00f811e", "name": "Netherspite Historian", "phash": "69f77972d91cc79fb657dbcc1db3386c946c46a6246280f9a5b9194c18c31643", "mana": 2, "whash": "c9aff12ff01fc01ec01fe01fd00cf834f821c83c3c3c181f781f781fe11fc13f", "card_id": "KAR_062"}, {"dhash": "c28eed3f60c31f8e9e11342870d8c472b9207261e6ce8d192730ce2d081bf2be", "average_hash": "073c21e7f9a7fc7c7e7c3e7c167c185c085c081c081c380c1c021c0600062006", "name": "Shatter", "phash": "cdb2b5bad2745a8f5c938e682dc92eec23ea2cb203d7e3314b07da237b017f00", "mana": 2, "whash": "77be3be7fda7fc7c7e7e3efc16fc1adc1a5c085c085c3c0c1c021e060006b00f", "card_id": "OG_081"}, {"dhash": "1613294ce4b8d8feb0b56701af6678cde092012507624cdcff3921570080053f", "average_hash": "cfe1c3c1c4c1c4c3e4c7f79e775e633f6d7a2d1005400440e445000005000320", "name": "Barnes", "phash": "7dbe39e3b653b69eed4d69637632b3358912dec849e9a67ca16452b20c30034a", "mana": 4, "whash": "cfe1cbe1c7c1c4c3e4c7f7df775e67ff6f7e2f1027c004c0e4c505c005800338", "card_id": "KAR_114"}, {"dhash": "e70f98ff68bb975e209ccbf99da03bc5d780e7238f493a32dce431096493cc3f", "average_hash": "0bf001802008200ef0067802f003f023f86bfc33fc31f801f204ff00e800fd9f", "name": "Excavated Evil", "phash": "f3dddc284c33b5899d4dd61695dd1fc6cd33a63d2d18804e2f76283607962a31", "mana": 5, "whash": "0ff001802008700ef906f902f983f1b3fd7bfe73fc31f821f204ff04e902fdbf", "card_id": "LOE_111"}, {"dhash": "00829dfef1c14f9ab86171dde231de47fdb9c127131a7ce5f3db0f9f3f34e279", "average_hash": "07200300013c007c007c004d20073047607f4c76e802c217f65ffe3fff18ff1f", "name": "Demonwrath", "phash": "69bafc3b2f41ba25c60ddb89166c25ca0920e71489f43fc169df5760b9d66989", "mana": 3, "whash": "073003c0013c00fc007c004f20473047e07ffc76e052c657fe5ffe3fff1cff3f", "card_id": "BRM_005"}, {"dhash": "eac4ed0d933f44fff8faa3a145138b36146d2d8e5a11bd4842b171e7fb43078d", "average_hash": "17300f600b6015e054c0d3cb83e393eb93c91bea07ea03cb00c880ddc68fe60f", "name": "Anub'ar Ambusher", "phash": "bf49e6e22f9eee259b11dde9334444e340f099198cac1b416e217364bd93d6d8", "mana": 4, "whash": "1f700f703b60176054c0d7cb87e393fb93e80bea07ea03ea00f885ddc2afe62f", "card_id": "FP1_026"}, {"dhash": "361da17e66947c68f0e0210747f49bab37177c4ccdb43819e23380a1314bde3e", "average_hash": "c323c147e04ff0c7c283c3874c82fc82b983e985918b9087408700060206c43e", "name": "Deadly Fork", "phash": "d964336ea8e6ff4f2e27de663ec19e87936566c4648c9398e94973644c126780", "mana": 3, "whash": "cb63c167e0cff0c7c283c7c75ec3fec2eb83e987d19b908f509700868206e43f", "card_id": "KAR_094"}, {"dhash": "6acfe4dea1b94372b6246c49b83241a7a14e1f9d342a2144cab9d461398b7132", "average_hash": "0fb08fb197b9b5b9b4bf34bd037d33d8f350b341334231c0304030a031903b10", "name": "Undertaker", "phash": "edf12db59ec7d6c25e8e8cc84d08a190e6c04e806f11e6a1ee847b61fd396d69", "mana": 1, "whash": "0ff08fb193b8b7fdb4ff35fd03fd73d8f350e3c131c231c03040358035b03b30", "card_id": "FP1_028"}, {"dhash": "02b2087661eddbc3bf9f5d7ebf887c31e3e2cce5f183a61f433f8eee0c9d0178", "average_hash": "8304810c800e001e001e1806781e183d183b183f183f183f981fd81ff01f807f", "name": "Anubisath Sentinel", "phash": "c9a776727a72894db799c393c79e986dac884b9b430e1a66a4e14e705526cf4c", "mana": 5, "whash": "fd8e918c801e801e001e182ef81f383d183f183f183f983f983ff81ff11fe17f", "card_id": "LOE_061"}, {"dhash": "f8fffdff01000f581cf038b6f166e385870981337e47f88cc3181c33297ceaf3", "average_hash": "07004300f91ba070c060c071e073f07af076e060cc7dcc6fbc6ff44e7038f07f", "name": "Starfire", "phash": "a16ab605ced15f1a66c45da89486766289322421869cc9cd0bb3af9d5b7adbed", "mana": 6, "whash": "0700c200f91ba070c072e071e073f07ef07ee460ec6dcc6fbc6ff44e707ef07f", "card_id": "EX1_173"}, {"dhash": "8cf30cc081bf0b0c36da4c30b3647689a44249a5b64aed4512b34c6c9580ad32", "average_hash": "f71ffbbffbb9f83bfc39fc337c22ec10f81260002006200a1019103000000900", "name": "Skeram Cultist", "phash": "e36573847fc15b707e9619d05b92008409214c3ee6d0de69ffb4363d360b9eb3", "mana": 6, "whash": "ff1fffbffbb9fcbffc39fc337d60ed52b95269402946b05a1018103901100900", "card_id": "OG_339"}, {"dhash": "f887d9ff20180f787e80fb01708fc11b836c44f908860304072eac7f60f0bcce", "average_hash": "0700e341f31bf071c07bc07d40fc087902790279127c105c0064007e807fc02f", "name": "Tree of Life", "phash": "a98b33ccd2754e811e09cc2e5ce22c60862f8eb667d07cf8c3c9322d7be5d1f2", "mana": 9, "whash": "0708f363fbbbf47fc07fc07f40fc0879027d027d127c127c187c007e807fd03f", "card_id": "GVG_033"}, {"dhash": "dc3f3cf720c81580270c2e38d8d1b2e36784cd02c927ba5ac4a3045602a0cd6f", "average_hash": "0783c307e31ff01f703d303ef038b032f03b703b6019201f800f800300000000", "name": "Ogre Warmaul", "phash": "697d7f201e121bc836048e3004982698199026c933e1fb6c3769ce37ffbd7e5f", "mana": 3, "whash": "8783e78ff31ff11ff83f783ef83cf83af23ff03f7039e01fa00f800f00004901", "card_id": "GVG_054"}, {"dhash": "f63f30e00f049c3d98db70b77760c0c489997f7220c4cf0d96330053f0a8fc17", "average_hash": "c1e7f99ffc7f7cf06cc17ce07ee03ec044e06c941e84bece384d004c00000000", "name": "Purify", "phash": "07aea14ef9575933fc75de58de85b4dcb4c906a225f6a47e06fa88148b69a482", "mana": 2, "whash": "c9f7f9bffc7f7cf06cc17ee07ee03ec066e06e951e843ecf380d004c0020d017", "card_id": "KAR_013"}, {"dhash": "5332b8e4040138bf7070cfc09b29009b30664ccccbb0b7596a72dec4b98de37f", "average_hash": "e5efb3cdb8df28c0fbc1dbb80bbf038203800380039010d010d033c01700877f", "name": "Museum Curator", "phash": "f5e5519b297ad479bb5e31bf2993f6d210eec2c81dac00e4312c86c493f049db", "mana": 2, "whash": "edeff3effcdf28c0fbc5dbb80bff43c203c0138023d030d011d033c01700c7ff", "card_id": "LOE_006"}, {"dhash": "28df2db0920009167771a9c61139c2c13c177bfcf710ed89d03526e14842d7b6", "average_hash": "f71fff7ffbd7f89078b00aa81b063ac22180018cab84a084a08c9044a004e600", "name": "Vitality Totem", "phash": "7336bbcdbbfa7be7bb8409c22e5120d9225cc2201ca75c016708ff24fea0eea6", "mana": 2, "whash": "f71fff5ffbf7f8b0fcb08ea81f823e822980818cbf8ea286a88eb044e285e401", "card_id": "GVG_039"}, {"dhash": "c287bd7d60ea83d11fa73f3cbc4870b1e166864d389264e6c80d013a7226eccf", "average_hash": "0730e3c1f381fc1dbc1e3c3f783f183898709961db63c2628e600c404440e807", "name": "Darkbomb", "phash": "0b47ebb17eb8b642a4cb0decc9c4861a09764c86cbd93af0cb0d7e39f3619e06", "mana": 2, "whash": "0738e3c1f781fd1fae1ebc3ffc7f987c98709a619b63c6628e600e404c40f80f", "card_id": "GVG_015"}, {"dhash": "7235e9c30434194832d86d219f137007405c0cf51922735ce0b989530382873e", "average_hash": "fbeff9fff8fbfcf24492e0bfc03f203f00270003000300030003000201000128", "name": "Prince Malchezaar", "phash": "d9ec61a7761bd3cbed5279fa3ec9966e9c4c42b629a4b654a51c364d28814e42", "mana": 5, "whash": "fffff9fffcfbfcf74cd3e5ffe33f237f032f010701038043004300060100033c", "card_id": "KAR_096"}, {"dhash": "8aec8dcb92973d2172e684cec99db33b2f77c0cc81510389803202e56b8ab715", "average_hash": "f75b7bfa79ddf8d818d21fd39fb19bb18fb11fb01b90088008c000c003000700", "name": "Murloc Knight", "phash": "c785e3696ae8f35cbb5010389f4c08e41064735221f73b937b92bad3e6a35e71", "mana": 4, "whash": "f75b79fa79ddf8fd1cf21ff39fb19fb18fb11fb03fb008b008d000c403040700", "card_id": "AT_076"}, {"dhash": "eac2cd07931f043c797827e24880c100133f3ef0dfc40809593ae2e5c4410332", "average_hash": "f71c7b24fb60f840fcc07e90fc109cc104018381ec87c801c0c1d000f0014000", "name": "Wrathguard", "phash": "638be7d3ebf259bb5c7029b0691c1818145b931b2d13bf497f043f41ae03365e", "mana": 2, "whash": "f71cff5cfb60f9e0fc80fef0fcb09ec906818785ef87dc81c0e1f0c3f0816200", "card_id": "AT_026"}, {"dhash": "48dc0cf8c1600a1f35065ae1bc026179c68288351d8b7ac6a5b96b689e9f313f", "average_hash": "f797ffa7bbbf382fdc2bf019d47988712032203000210001000100017913391a", "name": "Mukla's Champion", "phash": "bbdd5b61db04de14e7a432e323e12284180a8cc36cf2e3ec3e2737e73931b331", "mana": 5, "whash": "ff97ffb7fbbfbdafdc3bf419d579097921722170016101438001008179b3391a", "card_id": "AT_090"}, {"dhash": "e01ffdfeb00107a41ed83ce0f084e70eed5893c102032707ccfc107933c0fe8f", "average_hash": "0720c38ff31ff877f075e072e871c075e676fe7e7a3c7036002600640000e007", "name": "Totemic Might", "phash": "a996bb9ddec94f201f32b40ee60924a480b486998b59ae65e9266a8d6bdbff81", "mana": 0, "whash": "0730c3c7f13ff977f875f87ae871e875e676fe7e7a7e7036001200640000f00f", "card_id": "EX1_244"}, {"dhash": "faefad9c5238cc78f8e3b2864faddd6217c76c88f935f44b813f10e7a34f5f3c", "average_hash": "07108758837b0051c0c1c2c36007badbb00ba387e0bfc0bf00dd805c001e021e", "name": "Succubus", "phash": "3989b72f6e76ef35cf0966a212c410d200c14cf4c698db443f89df017f13ef74", "mana": 2, "whash": "0f10af58cb7b00f3c481c6c36cc7badbf28be38fe3bfc0bf00fd80fc009e021e", "card_id": "EX1_306"}, {"dhash": "5cf10c6b41d70a8c35405a89bd1a52b5af7a58c5b8abd1d66731066e0c901934", "average_hash": "f71ff38e639f783ff83f300f14131412040704030023000a080e181418000100", "name": "Dalaran Mage", "phash": "c9dd731867127fd69ea41e219b3003a623e2a6c59bed996c3c9a364b29b09ba3", "mana": 3, "whash": "ef9fffbf63bf7cbffc3f343f1d7315530557955701530468180f181419000104", "card_id": "EX1_582"}, {"dhash": "3adb2c2861d0ca78b1c5528bac9a4935b3627445efcbdc173837e06041838336", "average_hash": "c796cfa783a780038003840384419049981c181cd81dcc1de43be007f107e107", "name": "Ancient Mage", "phash": "1bdb9e1cfb346f87eea133a48fa4c0e021631262e46c6e643b42b949cf23bb33", "mana": 4, "whash": "cf97cfa783a78483840384038541914999581d5cdd5fcd7fec3fe087f187f107", "card_id": "EX1_584"}, {"dhash": "88f70dde9384056b7adea4244941920624ff7e9e1fb40f48a1b1c8e26b4597ae", "average_hash": "f73dff7ffb7bfd787cf27bf63bf203d003d003cc03c603c300c100c006800600", "name": "Undercity Huckster", "phash": "f7a4e36733da7b9a2e10d390cb9064b024f90e224ec33f01df265fc1bf09de24", "mana": 2, "whash": "ff3dff7ffb7bfd787cf27ff63ff217f007d0038c07c603c300c004c002800620", "card_id": "OG_330"}, {"dhash": "acefc97c82d384017993836e1304664bc4f288a53903124861b17de0fb4707bf", "average_hash": "670fd363bb4ff84fcccecfc6ef662b660f233d222f8300c000c08040c2018603", "name": "Anub'arak", "phash": "9b0eebc23bd9f6c9bb547c266f840e1e4ccc5398ced839d14e8c51b1c68b11cc", "mana": 9, "whash": "4f1fdf6ffbeff86fdccecfcefe666e660f233f222f82008000c00040e2858607", "card_id": "AT_036"}, {"dhash": "6cff2cecc1910b06b7194872befc67698202840d186a20540eb8fb7163863739", "average_hash": "f79fffbffbbf383f3c3e6c3ecc7ff47ff833f821400300004000400001003110", "name": "Dragonhawk Rider", "phash": "fb5dc9017e3133e1b3913ee22e666222c6b09978bec78687dbcc78d88e0b30e4", "mana": 3, "whash": "ff1fffbffbbf3cbf3c3e6c3ecd7ffd7ff973f861c10300404000400001003100", "card_id": "AT_083"}, {"dhash": "fa3ff5ffe3032f541fa73f467818f8bcf2f10023a34204870f0d399ae0787b83", "average_hash": "0500c102f91f7c73be793e7c1e7e0e7e8e7f7e7f4e7c4e7c7e7cfe3822707000", "name": "Sinister Strike", "phash": "2b2adbfff2f57ede9ce53e324e902718064884c6c3dc8d464937e7101b43bb80", "mana": 1, "whash": "0500c102f81f7c73fe793e7c1e7e0e7e0e7f3e7f2e7c4e7c3e3cea3802301000", "card_id": "CS2_075"}, {"dhash": "eaf78d48122328ce710eaabc16d8aca17bcbe79ccf411f0b783662edec4bd1b3", "average_hash": "f70ff11f794f3886380e38801800980030b831b8f139f890f8b1d801f8109001", "name": "Dust Devil", "phash": "e313cb4afbe279f93efa16872f583c90123406069e90d6249e68478ddb9669f8", "mana": 1, "whash": "f71ff95f794fb887788e3c80bc20bc0030b8f5b8f5b9fcb0f8b1f801f810b003", "card_id": "EX1_243"}, {"dhash": "faf7fde0a101679adc30326f27de6e3c99e5009bf33067c30e2688cc0090fe2f", "average_hash": "070071483d3c3c781e6c8642866ca67ebe7e7ade721e261e780370030000c003", "name": "Naturalize", "phash": "cf602f7bf2dfeb8479f23e725e20968d09722c70c3cd1e19c9211e364b99c341", "mana": 1, "whash": "0708f37d3dbc3d7c1e7c8e62866ca67efe7e7e5e731e661e780770030001c007", "card_id": "EX1_161"}, {"dhash": "6afaedf412c329845748ac181831d4e16187eb0cf631ccca98b591e2234547ba", "average_hash": "870fc14ffbcff09ff0bd72bc3b1038907898f99cf3bee08ec08ec00600020000", "name": "Flametongue Totem", "phash": "6996cb9999ea7b768e2d16e42ed022cd20d8cb20d6a4d744dd225d89bf23be89", "mana": 2, "whash": "c71ffb5ffb5ff0bff8bd7abc3a9038907998f9bcf3bce08ec08ec04600020000", "card_id": "EX1_565"}, {"dhash": "c8dfcd39920728c97142e3c4073ccc391cf3602c81c03788ec30fbe67e4df3bf", "average_hash": "f713fb63fbc7f883fc807e801f001c803d8c1d980f800c8048806840f800760e", "name": "Earth Elemental", "phash": "c70ff3c3f3e0bc6ce9703c1c0e51632101d096247dcc4745dcac76e7d9326b9a", "mana": 5, "whash": "ff13ff43fbe7f8a3fc817e801f881ec03dce1d9a0f801e806c80e840fa80fc1f", "card_id": "EX1_250"}, {"dhash": "fad96c26d15c0af135e672c5a59a4f578da2724ded8bda862f34e469dc87c13f", "average_hash": "0797cfa6dba09981980380039041984108078801900df82ff827f807f90ff90f", "name": "Zombie Chow", "phash": "9bd39ee7ef74ef94ffa499a03969802c0d8a86413e499e44be04d6e599899b84", "mana": 1, "whash": "0f97ffa6dba398a19c0380038141994109458941f94ff86ffc07f897f91ff90f", "card_id": "FP1_001"}, {"dhash": "fc32fc48e0af9d7f267e29f8dc90bbc326006a02d025665a9da07155c2a0c56f", "average_hash": "07800388030038107c33fc2bfc3bf83be01fa00f0006c006e000e001e001e901", "name": "Perdition's Blade", "phash": "3335cf013e1a5c20391933d88752031865e6846119a17e66cffc9e399ffb7e36", "mana": 3, "whash": "07c0079907d079707d37fcaffc3ff83bfb9ff81f3006c006e001e541e901e9a3", "card_id": "EX1_133"}, {"dhash": "c8d90c32519c8a31358f581ea06443c987f23ec5478a89155e3bbc606089c136", "average_hash": "7791fbb6fbacc8298c3be411784078403c403803fc01fc00fc00f800f900f900", "name": "Coldlight Oracle", "phash": "e3279dd06f23c7a1ee90e6a43bf199688cf033d3ce9966b872a6ece4cc489848", "mana": 3, "whash": "ff95ffb6fbacccad8c2bec137d4079403d403d43fd41fd40fc10f800f900f900", "card_id": "EX1_050"}, {"dhash": "08f0cdc512934d26bb1476d9887673c96c02d87403eb8615d8b6256643889d34", "average_hash": "f71f3f7f3bdd30dd30fc30d8304878327c36043c040004100014001801100100", "name": "Nat, the Darkfisher", "phash": "edb9eb89ff3b9bc6db0653604c7008640304720c6a305f253705fb86ef0ddf17", "mana": 2, "whash": "ff5fbf7f3bfd35fd34fd30dc397879f27df60d3c05580d500434049801900110", "card_id": "OG_338"}, {"dhash": "faf7f9c1c26345bebb7076219c027087451a9a94342be2c4c8b7d36f7f99e736", "average_hash": "0f14077f07da01fc20ff80dfc03f403ee022802000000000003c003ec11fc11f", "name": "Yogg-Saron, Hope's End", "phash": "2db40f894f2a9648bb162dcd5a7bc0fe645ee6f726b3d2b45f4486a42f013780", "mana": 10, "whash": "0f500f7f0ffe35fce4ffa0ffc17fc1fee1e281210142014c003c00bee1b7c13f", "card_id": "OG_134"}, {"dhash": "ead8ed31d2f384ef39c3710d83104e6d19b203643ccbfc976137d76e0c9c893b", "average_hash": "070707670bc70047e84fbc879c07040f0c0dcc01c003d803d83b983b9819191f", "name": "Millhouse Manastorm", "phash": "9befdfb4ed8436e64e3483645be423c9295826298e64c681dd1076497b036f8f", "mana": 2, "whash": "0f170f671be71047fc4fbcc79d47144b8d0dcc01c503dc03d83bb83fd919191f", "card_id": "NEW1_029"}, {"dhash": "8a03fdff608e831b5e731dff38f0e1e0cf051f0be0430492382ce15906a7f89e", "average_hash": "07204180e100f830f838fe3cfe75fe73fa77fa6fe078c03080200064000ed00f", "name": "Lightning Bolt", "phash": "235637f5dcfdfcbcdbe88efe6fcc892222658c7883384a852d001aa1d3922f45", "mana": 1, "whash": "073043c0e110f938fc78fe3cfe77fe73fa7ffa7fe278e03080300066000ff00f", "card_id": "EX1_238"}, {"dhash": "eab3f5e7201c6739985f34a629f95b3c91c8c6309d4168cfde1b34b72b2cf64c", "average_hash": "0700810cf91c5c7c0e7b7e101e00fe41be007e306e201e162e9c365e3a4efc6f", "name": "Bestial Wrath", "phash": "0728bedff9ff7eb4b25b5e20969c1ce80d369cc003fec623230c6b4cb9611a0b", "mana": 1, "whash": "0700c34df93c5d7c4e7a7e503e40fe40be007e306e30ae163e1c265e3a4efc7f", "card_id": "EX1_549"}, {"dhash": "c8d70da843000d667780ae29581bf37640ff91c8ba05ff48feb900e3794a8736", "average_hash": "f717ff7feb6b34f13cfb02f803f802f613f01bf01ff034d8e0cf00ce00a60617", "name": "Imp Gang Boss", "phash": "edb4b3895b1a73929e8669322f6106e7c1a8c4343ee67e76cc665964cb4c344d", "mana": 3, "whash": "ff17ff7feb7bb6f11cb902f803f802f612f01fb01fb03498e0ce00ce00a60617", "card_id": "BRM_006"}, {"dhash": "0ac60d189234a9c97023e30e0c94d9a941f384641f40208847307fe3f24fc799", "average_hash": "f717fb73fbd3d883cc878e8d0f580cb00da00d800f80008000c00040c00cc61d", "name": "Dunemaul Shaman", "phash": "170fb385fbe0f76d2b787e785b10429600342924c6ec5d4b99a0cdf4ceac2ba7", "mana": 4, "whash": "ff1fff77fbf3dcb7cc878e8f1f983ef01de01f804f83068000c00440c28cc61f", "card_id": "GVG_066"}, {"dhash": "00808d9fe07feff1dc91316662decfe83f1adfe53f0b6116782c72d8f400e017", "average_hash": "77bc03e001b0807206670ee2cec0eec0e406fc27fc63fc31f060f8307e10f80c", "name": "Vaporize", "phash": "b3769c28ec85f3555a2926ab9b4889066960233999476eb50bdddff0dae9f220", "mana": 3, "whash": "f7be03f001b0c0f28e620ee38ec0eec0f806fc27fc73fc31f870f870fe10f80e", "card_id": "EX1_594"}, {"dhash": "0cf04de0d2808b0f7e1f2c3ed844a08920f281e70b841308263bdce6a0493336", "average_hash": "f71f1b5e091c083c0cf84e382cbc1dbc0c1a3d803d803c003890781038123216", "name": "Savannah Highmane", "phash": "cd2d298bfba0d34bdb70331093d031659878730421dc1ad63bedd793b3732c8d", "mana": 6, "whash": "f71f1b7f0b5f08bc0cfc4e7c6ebc3dbc0d3e3d807f803c803c9078903a163616", "card_id": "EX1_534"}, {"dhash": "0af30d2453c4c5017e6320ce5a1ce53842f380e03fc57e8a3938e3e086450f3e", "average_hash": "f71ecf7c036f05f60cf80ec19db11ef11ce01fe01de00ca088c188c3c003861f", "name": "Voidcaller", "phash": "9f9f6f9279a067c2cfc013996ec4326424a431e4b1b1da64db43d740ee1ab5b9", "mana": 4, "whash": "f71ecf7c0b6f80fe0cf80ee39cf11ef91ee01fa01fa00ca48cf388c3c08b861f", "card_id": "FP1_022"}, {"dhash": "68ce6cb951d3832437ff4c58b8a27345ec9a903529c946961934b3696492c73c", "average_hash": "979b8bb34bbf7dbd0c3dfc3ff4307571253c042c0d230c60cc20c8248104c110", "name": "Eldritch Horror", "phash": "2f9fdb895a239e4effa0b3260e6d09f22569caf826604bf29bc73cc34c8a9224", "mana": 8, "whash": "9f9b8fb34bbf7dbd2c3dfc373571b579657c056c0d610d60cc20cca48184c118", "card_id": "OG_142"}, {"dhash": "03c04d7ec0c89b91bc27761ff8e883cdb68b419703643718f633c16706cf7cd6", "average_hash": "ffff19f8c3ff887f807f80f831f010da4cde24d23280318038821c830cd64cce", "name": "Shadow Word: Pain", "phash": "c5ff29139b17f60047ecc64f76d0097e499b0e2031380b975bc15bf0db4bdb06", "mana": 2, "whash": "ffff49f8c3fd887f807fc0f871f011da4dde25d22280318038c21ccb0cd65cce", "card_id": "CS2_234"}, {"dhash": "8cfd0d72d2208dc27b85e41c197ff692682dd11e8cf9194323be46f4094e2718", "average_hash": "f717fb6f4b5f18760cfb0efe033600be02be02be01b90139009980dd000c0010", "name": "Mech-Bear-Cat", "phash": "cd84d306534afbc87bc663c8266944300274ba30b6b4de921fdb4c6fbd253ecd", "mana": 6, "whash": "f71ffb7fcb5f18770cfb0efe07f606fe02be02be03b9013900fb80dd009c0210", "card_id": "GVG_034"}, {"dhash": "f63fc87120c3158d33122f3cd974a2a57c90b36266849b196427985d42922569", "average_hash": "c783e307f30f300fd80f981e7c1efc06240e6c0d880df80bf813e037e0082008", "name": "Charged Hammer", "phash": "4b1fdf80fe083b481e5cb3a40cc33cd6c7ec02fb24bc67780df3466c23787c32", "mana": 4, "whash": "c787e30ff30f390fd81fd81ffc1ffc0ef40ee80fc80df80bf817e037e0092008", "card_id": "AT_050"}, {"dhash": "a8f90c8641084ab9b5744bc7b2186f25f47aeec598cb611703375e6e7489c132", "average_hash": "c717c79fc39dc008000800250023103f941f9c3f9c371c3f1c38d818f800e114", "name": "Argent Watchman", "phash": "89777cd076068fa5793886c9d2d4226b4c6892c6b69477327cb0cf83bf07be0c", "mana": 2, "whash": "ef17cfbfc3bdc0b88008042d8863107f947f9c3f9c371c3f1c39d819f90ae914", "card_id": "AT_109"}, {"dhash": "78d8ad36e2614881f9e2b1c5674b9e96390e731fcc34b8cbe1b700ee0748ff3f", "average_hash": "070303668107c00780c38247828fc38fc00fc189c399c03f80bfc0bf001ae61f", "name": "King of Beasts", "phash": "19a3d6728fa4390d66028d843689262989f436c949d99e757bedf691b9ed4669", "mana": 5, "whash": "87138b6783c7c087c0c3c2c782cfc3cfc10fc38dc39dc0bf80bfc0bf021fe61f", "card_id": "GVG_046"}, {"dhash": "48f0cce491980b21360f4c96b8046269edb2f6e5c9c911130734a269c49a1920", "average_hash": "f71ffb1ff91f983d98377c312c300c004c242c3d3c3918301820182008001000", "name": "Lil' Exorcist", "phash": "8b976b107be169f87a78d626af4c18969266986929cd3b9e9bc95a69c892938c", "mana": 3, "whash": "f71ffb9ffb9f983f9c3f7c393c339c02cc3d6c3f3c391c311c30183018111900", "card_id": "GVG_097"}, {"dhash": "6cf08ce0c19f8b03368f4cdcbe387171e942d32583cb6497c3370e6ed89f813b", "average_hash": "070b131c011c001b203130303839783c383c383e4816081f183fb81ff01fc01f", "name": "Archmage", "phash": "e9b576405a295f666bd387c99224a13018c433c90dcd79e67ea679962e1b538f", "mana": 6, "whash": "871ebb1e011c403f20333830383d783d7c3cfc3fd837981f183fb83ff01fc01f", "card_id": "CS2_155"}, {"dhash": "38b82d40e2017d47f07883fdc49f83333e2774ccca909d0922b30ce63185671c", "average_hash": "c767e3fe03da00d000c083c1c3c3fbc3fb8ffb9fdb9cdbcd20c820cc03040704", "name": "Quartermaster", "phash": "5da19d222c3e6947bb4613903fc2e69092c493e539f938816e6f9ecfa9c37251", "mana": 5, "whash": "c7f7ebfe03da00d000c083c5c3c3fbc3fb8bfb9fdb9cdbdf20cc21cc03040704", "card_id": "GVG_060"}, {"dhash": "9adb2d24d273cd857b6be6fe108d76826d94cf2f9e597e83b83c70f6844d8b1c", "average_hash": "c706e3640341004d08cc8891cb21c835c031e0b3e131e130f888f8d3b019981c", "name": "Fandral Staghelm", "phash": "3387dc1ecfeb336ebe23c3422e41016c48f2923093cc1e311ccdd2f1bb0bdec7", "mana": 4, "whash": "ef16e3670355006f08ac8ab9eb21ea35c231e2b3e1b1f831f8baf85bf819d81c", "card_id": "OG_044"}, {"dhash": "3afc2de06080f904f209e4b1897117876c3ed8ec30c1608bc03281e483892713", "average_hash": "c70fc1dfc1dfe0dfe0bf63bfe19fe187c1a709a709a700b2009200d203030300", "name": "Lightspawn", "phash": "d995796b3ceb9f57db425068ae5484d400d043870cf12ab0f78ddac4bc335b75", "mana": 4, "whash": "c7cfc1dfc1dfe0ffe0bfe3bfe39fe387c1a709a70da700b3009201d303030301", "card_id": "EX1_335"}, {"dhash": "8087cd3f20f801f03fe07f008f003c417882c304e348b633498ee218f7818c1f", "average_hash": "f7bfe3e7f18ff01ff01ff81ff01f701e380e68066007481380318400c803f807", "name": "Seal of Champions", "phash": "5b55f99abf230e044e54ad601b0c045aef144a544bf11b854ff85bbd4bee53e4", "mana": 3, "whash": "f7bfe3f7f18ff01ff01ff81ff81f781e780e680e6807c81380518401c803f80f", "card_id": "AT_074"}, {"dhash": "28c64c8c911f833836d14ca6bbe47ec9f0f2c1cd8f89141641b40069ff801d3f", "average_hash": "f71ffb91fbb1f830fc306c304c186c3c0c3e183cf831d0200020003000000100", "name": "Force-Tank MAX", "phash": "2797330db261f9c14d2b0c6cb3c418b4495a734ae7e79ee759b6d6d20c839c07", "mana": 8, "whash": "ff9fffb1fbb1fdb1fc31ec305d786d7c0d7e397cf971dd60102000b001800108", "card_id": "GVG_079"}, {"dhash": "68e68d1d427bbc84713d8349c6f99ceb3b1735ecc830130b9630f0e2a185471a", "average_hash": "c749b3f9a1d120cd68ec33cc13ccbb86ef8bcf8db38c31a830c080ca03030702", "name": "Aldor Peacekeeper", "phash": "5529a79bbc6c3119bb6544e36ed01b5a0296d8a45afb9b5cf7313ab6288ec2c4", "mana": 3, "whash": "e7599bf921d120c768ec77cc338ebbdeef8fcf8db39db0a050cc80ca83030702", "card_id": "EX1_382"}, {"dhash": "03808d0f64f771ee67dccb929f37736dceff10d94d10bf28ca959cc3f996e03f", "average_hash": "ffff6bf063c07000360023062133a37325e773e9fb88ff80720071007101f1b7", "name": "Mindgames", "phash": "f7500dae62e73783bd9e99344b01c3da89f24b8149d552fa5b323b68d9c9528c", "mana": 4, "whash": "ffff6bf0618070003a006306213b23f3e7e7f3e9fb88fb80f2807180f181f1bf", "card_id": "EX1_345"}, {"dhash": "4ade042401084a5cb4f851f3a3864f1d9f0a3c1d792a404490b834616382873d", "average_hash": "f793ffb7f3adf4a0f440e403e543c543c143c043c10700448000800001000100", "name": "Young Priestess", "phash": "33faf3b56eb39de39b899bb8e38c86600c2136c43b191f196f801e439f81f3c0", "mana": 1, "whash": "ffffffbffbadf7e0f4e0e547f7c3e5c3c343c143c16781c48040858005800320", "card_id": "EX1_004"}, {"dhash": "28f04cced19e0361b61e68f5a40a4d3dbb720e65b68acc0418b97876f18be32e", "average_hash": "e71fdf9e4b99c939103e503b100b00058405cc03c801e809e000f010f014f10e", "name": "Nerub'ar Weblord", "phash": "33ff9dcd63063337dbb072c8cb46813409323381b2083b9bef0cdc10df28dee7", "mana": 2, "whash": "ff9fdfbfcbbdc8b9143f543b154b054d9555cd43d913e841e000f010f114710e", "card_id": "FP1_017"}, {"dhash": "fce7accd61934b26b75c4e9dbc306071dd06f81de0abc6470db2186865999f37", "average_hash": "0718039823982038303c703f383f387ff83f003f003f401e401e4000c0001111", "name": "Nerubian Prophet", "phash": "69dd6f0936255c8696031686cdc0646022290c3ccfdc7bc7fecbbc716fc29bb1", "mana": 6, "whash": "0f980fb823b974b8303c703f3d7f397ff97f117f417fc07fe01fc000c1101113", "card_id": "OG_138"}, {"dhash": "0afcccf211248b4136834c24b9787a61e082d3a5cd4993932c37376cdc9b4136", "average_hash": "f70371073b1e58138832183a1c3e7c3e7c3e243d3039603cc83ab835f817f016", "name": "Annoy-o-Tron", "phash": "e975df64d2a37fd8d39007e73c6c9294214c93911c63e784e7213b64ee11be24", "mana": 2, "whash": "f703710ffb1ed8170832183a1c3e7c3f7c3e2c3d303b603cc83ef815f817f00e", "card_id": "GVG_085"}, {"dhash": "eafff8e007000e81b90f76f1e6045c3878c2c107416fb1fc811f0f1040a2ff4f", "average_hash": "0300813f807f807f007f407dc07d807c003e207f007f007f807ff0742060f00f", "name": "Protect the King!", "phash": "a9ba5fcdb655fc346e5436d2d6c9e62c0e3716323782a69c67c41cd293b00cf0", "mana": 3, "whash": "0740813fd07f88ff00ff787dc07dc47e003e20ff807f807f807ff07c2064f81f", "card_id": "KAR_026"}, {"dhash": "8a9fedfef0c143029fc51ecc39187266e4d4d3b98c4379049a897c1e04340360", "average_hash": "0700c103c10ffc1fbc1d3e3e3a3b30703870fc788e718e6eb65cfa7ff87ffe3f", "name": "Call Pet", "phash": "ab20daf57ade3cbab4849c20067425dc46b8a1494bd14976bff47a219f46bf0c", "mana": 2, "whash": "0500c103c10ffc1f3c193e1e3a3b30702870dc700e708e6ea65cf27ff87ffe3f", "card_id": "GVG_017"}, {"dhash": "9a990d32e04cf8e1f1c2e71d8f2b385772cec0bc11706e488cb380e093812716", "average_hash": "87c7c1e6c3c680c1c0c4039f03bf03bf83bf63bf638e618fc0c601c003020300", "name": "Shrinkmeister", "phash": "7da595e3167a733eaf61d9e04b98847102c1ce180c399b99ce136e246f8b7bc6", "mana": 2, "whash": "8fd7c9e6c3c680c1c0cf03ff83ff93ff83bfe3bf638ee18fc0c701c203020300", "card_id": "GVG_011"}, {"dhash": "6af04d709305044b79b3a3264f04984b3497722ee4550d8b1bb257e4984143b8", "average_hash": "f71ff92f1b5e986618cacfcecdce2cde23de019e09be80bec0d8185100000600", "name": "Cutpurse", "phash": "492fdb62b2d1fdb16b98336c1b16828934f08e847ae25f128f036c61ef257ecc", "mana": 2, "whash": "f71ffb4f1b5e98661cca8fcece8e2e9e2a9e299e0bbf84bec0fa10594a084400", "card_id": "AT_031"}, {"dhash": "0ac0853f60c3598fce048319367b1de4f200e03b20f7cdec2fca1f802516ca9d", "average_hash": "f79f01fce1cf700eb00e320e370c771ff73f20fc027c02de006e50265e0c9e0c", "name": "Avenge", "phash": "6d50f3b892ffbdae3be18ec80c21a3760bad4ef4d01c7b49dbd4590853281c1b", "mana": 1, "whash": "f79f01f0e1cf742eb45e360e370c771ff77f20fc027c02de006e50265a0cde0c", "card_id": "FP1_020"}, {"dhash": "cafb05e592cb05373b486c9a902465d94332afc4c80b33976e34cd7092810527", "average_hash": "f70e715e195d385d383d1cb86c202c38fc33dc3b1c180812c803600240030103", "name": "Dr. Boom", "phash": "abdf6db3daa07f66bb5e76349e6123c3214466301bc33826cc311c4b598af2d8", "mana": 7, "whash": "ff1e7b7f3bdf387d3c3d1cbc7c286c38fc3bdc3b1c390832e8036003c1038103", "card_id": "GVG_110"}, {"dhash": "0dc0cb3fe4ff13d06e00d181a6374c6fc88e131b2736cc4c98306268d8f13ccf", "average_hash": "ffff07f00297ec3fe07f814f814fe3c7e160e3e0e2c0f0c03ce018e180e1e0e3", "name": "Mind Control", "phash": "f15a490085956624df589fdb7b6aa153633e999f4bf5537e43a15aa859c11740", "mana": 10, "whash": "ffff0ff0029fec3fc07f814f81cfe3c7e1e0e3c0e2c0f0c038e018e180e1f0e3", "card_id": "CS1_113"}, {"dhash": "ca3f35e7e18087303ce7191f3626611cc0d891b1272b8f46168f641c8931fea7", "average_hash": "0700070de33c287808710065047d007f087e247c247c3478243824380002f00f", "name": "Sprint", "phash": "add8bf9f52e56ffbdbe6b6f99392866423418c4c63100e0013c21ab63ba93d26", "mana": 7, "whash": "072c87ffe3bc2b7808790c77067d047f0c7e2e7e347cb478b43834380432fc1f", "card_id": "CS2_077"}, {"dhash": "b8f36dce92118dc2781de32200d9c0220345069e0978b2c0ceb999f563429718", "average_hash": "e71fcf7ddb7fdcf7fcff7fcedfc4bec193c183c103c103c200c200c400880600", "name": "Druid of the Fang", "phash": "5d2b2b436dcefec92b9146264e4c50908034ba36b3a4dec1e7ad51e9fd236a56", "mana": 5, "whash": "ef1fdf7fdb7ff9f7fcfe7fceffc4bec193c18381038103c200c200c000880608", "card_id": "GVG_080"}, {"dhash": "6adacd21929608fcf9a0b3c3e61d87a71c5f32aa4c5431887a38e1f5c6032fb7", "average_hash": "f717ff677bed78c1f4c7f2c7e2c3f3c3b3c48f81828300c084c088c10b010601", "name": "Sparring Partner", "phash": "93b3fbc2ef86ee635f6219261830193260f8b64c248c6e837f43df4c7d0c3f05", "mana": 2, "whash": "f717ff6f7bed78e1f447f6c7eac3f5c3fb84af81ab810c808cc088c10b810401", "card_id": "AT_069"}, {"dhash": "f27fc8ffc17507080f115c6abac6b40c6f18d2618fcb20d77f0ffc3e00f80008", "average_hash": "0301f903f817f81fe81fec15e434c43084370c34cc311c187e1efc3ff87ff00f", "name": "Moonglade Portal", "phash": "2bbff614d3498e45b41277b8bc0f990a4e69b1e66e789ce839b42690aca59378", "mana": 6, "whash": "0101f907f81ff81fe81fec15e435c43084370c34cc311c187e1efc3ff87ff01f", "card_id": "KAR_075"}, {"dhash": "7cd9ed2ed35b04a37800a1c0448b8f3e1ef320ae4140be88fd31bce3f04583af", "average_hash": "0f33df66fb61bde1bcc193c003c083c11fc01fc02fc06fc07cc0fcc0f681e627", "name": "Kidnapper", "phash": "d70a9f03abf4ce063e007ce01e04036309f03818cce9778d4fbff2c9786ec77d", "mana": 6, "whash": "0f33df76fb63fe63bcc18fc007c003c11fc01fc02fc06fc07cc0fcc0f281f627", "card_id": "NEW1_005"}, {"dhash": "0ab00d6460c07801f100e21d8fe31ca7314e2d1c1af020c8c6b2ade75b8ba31c", "average_hash": "f7eff7efe3cfc0c302c103c063c473cc1388c382c38301c000c021c02314371c", "name": "Hooded Acolyte", "phash": "7f39595aab9a996febc2449cba81906d00e17b8610de1e919e613b24eb3a7e94", "mana": 4, "whash": "ffffffefe3efc4c713c303c073c473d813cac3c2c3c303c000c001d073943f1c", "card_id": "OG_334"}, {"dhash": "82c3fd0f601e03ff0ce2130830006480cb0887192cf3116624cc5099673efe67", "average_hash": "671ee1e1f981fc21fe2ffe0ffe67fe61c621c221826100610260026006785e3d", "name": "Blessing of Wisdom", "phash": "b3d0d3ff6cffbcad1f6fcda0c9083cc027d4263112f60a994b095b01d3649b98", "mana": 1, "whash": "779ee1e3f981fc21fe2ffe4ffe67fe61c661c261826102610260026006787e7d", "card_id": "EX1_363"}, {"dhash": "eae6acd8118f0b38b5604f6fbcd07181e702cd159a2b30578eb46160c3813d33", "average_hash": "379dfbbbbbbbf839f009cc3dfc38f878f832503b003b00300020000001000100", "name": "Armored Warhorse", "phash": "a57d7b4bde199fccffe40e9a0dd0128a42c2226069718c93eda07c21d6a3fb4e", "mana": 4, "whash": "bf9fffbbbbbbfcb9f439dc3dfd78f970f97ad17b617b01700020008001800100", "card_id": "AT_108"}, {"dhash": "023ec979c693396e93b12cc243349c793ce27062c38484082b31ffe246440f3e", "average_hash": "f963f94ff8cffcdffebffcb63e9f3c9f3f1e1f9e498818880080c00000000020", "name": "Malchezaar's Imp", "phash": "434fb3bdb2e2decbcf92b6746ee6733621f132dd8dbd82f22136141469401e40", "mana": 1, "whash": "f963f94ff8cffcdffe9ffcbf3edfbc9f3f1e1e9e4c881c880080c08000000020", "card_id": "KAR_089"}, {"dhash": "4cf0ccc0918f0bb337464c9cb0785aa1a47218e5748ad7146c39d373ac85613a", "average_hash": "f71f3b9f3b9ff83ffc3f3c3f3c023c123c001c001c001c0040005008f80a6118", "name": "Sideshow Spelleater", "phash": "c367c93179405e32fe30b0488f98313031a4837336f29739efb9fc8d8f731cc6", "mana": 6, "whash": "ff1fffbfbbbff8bffc3f3c3f3d423c423c403c401c005c004002d80af90a711a", "card_id": "AT_098"}, {"dhash": "3acf6d3ed2eccc79f996e43d09c3d2066d75deeeb1190783ec3c83f50c4ab913", "average_hash": "c710c161c347c047c0c3e2c3f312f09780b39ab539bc793268a718d618101801", "name": "Anodized Robo Cub", "phash": "b3a73b620ec32f9726a1662467c91ce718d067206b923e40dfa25c86ff45ef4c", "mana": 2, "whash": "c710c961c347c067c0e3e2d3f352f2f7c2b3dab77fbd793668e718d618101801", "card_id": "GVG_030"}, {"dhash": "0af76c09d1fa0ad1b5a35606ab3c4679e092ff3596eb2cd7f3b4a0637184cb3d", "average_hash": "47908fa95ba0182c0c1e0c178c478c436c606430c430c031f425f002f109f900", "name": "Dragonling Mechanic", "phash": "3397cf8f77c137f4bf0c9b280da4c16843920ce3c7a4b69cc628b0cc3ce4766e", "mana": 4, "whash": "ef908fa95ba0ddad0c3e0c178d578d476d61e570c570c571f425f482f188f90d", "card_id": "EX1_025"}, {"dhash": "ca8bcd7730cd738acf779aff3146737c84f81831b3e36dc6f888b290f761e79f", "average_hash": "07008100f30cb81c3e0c3e697e735e733e033e425f60cf60ce40e600e601f603", "name": "Demonfire", "phash": "0760ec3378fb9ee6df8c35cc8d584698097328edc9e5b3512c817f886b4673c4", "mana": 2, "whash": "0700c3c2f31cbc1c7e2cbe695e73de737e033e425f60df60ce40e600ee41fe07", "card_id": "EX1_596"}, {"dhash": "0cd72ceec1998bb336494bc6bc8c7979e0e2c18587c94f179f361a695492a92f", "average_hash": "e701f380d9809811883b8c198c380c3c1c3c3c387834f834f834782478000901", "name": "Scaled Nightmare", "phash": "a3973f09f200e7e47c6461823a0882c889988ccd2df137e95ccadf3ce7ccfec9", "mana": 6, "whash": "e711fba0fba89839ac3f8c3d8c3d0c7e1c3c7c3c7c3cfc3df835f82479000901", "card_id": "OG_271"}, {"dhash": "226ccdc82693fc0650c9a6e04d8f931b7e64f8c2e485c81b01b303e60f58feff", "average_hash": "d51b191b1019000f6086300390839803183f18be183c08be009e001e001c007f", "name": "Reliquary Seeker", "phash": "c9e3b3e97cf86de793c39a6da7844e96119e329319f18c648363367071169b84", "mana": 1, "whash": "df5fdb7b92ff108f6a87308392cb98c31fbf1ebe18bc08be009e001e021c80ff", "card_id": "LOE_116"}, {"dhash": "f6a1201b05f05de6bb04564ab71462e1eccadf153003a39f183f216e4490993e", "average_hash": "030e710bc003300ff00fb017f80f30267033f433a43ff03ff03da8399818811c", "name": "Zoobot", "phash": "e97dd34c6c0bbdccfe26661b6d339bd1ccc4dc12199396c60e86133361461f13", "mana": 3, "whash": "09af791bd803700ff00fb81fe80f38267837f433943ff03ff83de8399909813c", "card_id": "KAR_095"}, {"dhash": "8cf18cc5119b0b3737664c7eb9ec76d9ecb29b653749e2929f37366dcc9cb139", "average_hash": "730e731c391c381838381830dc364c366c366c367c272c2eb83cb81cb81df01d", "name": "Flesheating Ghoul", "phash": "6317dec46bc95b1e7bc60d269a6824e3790c134bb6e1cc34db69c1d9394a23b3", "mana": 3, "whash": "770e711c391c383938381c38dc366c366c366c32fc2eec2eb83cf83cf81df81d", "card_id": "tt_004"}, {"dhash": "4cff0cfc91814b03b7064ec9bcb27c61d18ae0d5088b011637326e6df69ff93e", "average_hash": "f71ffb1ff91fb03fb03d183c301d101d001d00100000000000000000d80ef80e", "name": "Grand Crusader", "phash": "e9f179615b06160c7b0a4698e66034c0783819d8b6bc5fc7dfc62d873e98dada", "mana": 6, "whash": "f71ffbbffbbff0bfb03fbc3dbc7fb07d105d00381020180018001810f90ef91e", "card_id": "AT_118"}, {"dhash": "88b92d4502c87d1f7b9fe53cde5994e32086030c1a70e5488eb319e47300b726", "average_hash": "777e7bee33ec10dc00fe1bfefbfe3ffe7ff46fc0038003c020c001c003000700", "name": "Twilight Flamecaller", "phash": "4d02c9e3be68f61ceb1649c39f4426ac08f05ee369d91695a6f7d2d43d2c7264", "mana": 3, "whash": "7f5e7fee3bec10fc12fe1ffefffe3ffe3fd46f8023c003c020c001c003000700", "card_id": "OG_083"}, {"dhash": "0cf04ce401808b00b73b4e07b87c61d98892706533eaf605b93bc270f88f053e", "average_hash": "f71ffb9fd99fe83f083e043c0c3c0c3e6c0f2c0760066807e8099000c003011c", "name": "Blood of The Ancient One", "phash": "699f6d91fb61d3881f1e97613d8052382b8cd2c6bc313bdcf3da89cdcc826226", "mana": 9, "whash": "ff1ffbbfdbbfe8bf083e043c0c7c0c7e6c4f2c0760076807f81b9002e107811d", "card_id": "OG_173"}, {"dhash": "e00f3d7f70f463c8cf91992733de6ea0d958b22176c3cc07991f523cb461e8c7", "average_hash": "07008301c10fc41fc63ec23ffe2ffe2ffe2ffc2ffc2cd81cd878587878607846", "name": "Far Sight", "phash": "2b423a74f48fa633d610ac806c36ac400ef187660bfd3a8e4a4f679b6b7db792", "mana": 3, "whash": "07008201c10fc41fc63ec23ffe0ffe0ffe2ffc2ffc2ed81cd879587078607844", "card_id": "CS2_053"}, {"dhash": "2afccdf102837d0ef339a662cacd9c1b3137406c7750ca09b0b263e10b82172e", "average_hash": "d74e3bff39df30dee0dcebf8cffcc39c039c03868387818100c180c103000700", "name": "Selfless Hero", "phash": "5db9e9ec92f8dae36f4239d7374430b220f0721386c466857399ebc463335ad1", "mana": 1, "whash": "f75e3bff3bdf30fee0ffeffccfbccbfc039c03864387818100c580c503000700", "card_id": "OG_221"}, {"dhash": "1af12cc6d1b98b02371b506ea49c4c799a02310566ebd5c389b7536ea99cf739", "average_hash": "671fe39ff99d983f7c1f7c0cfc047c0c7c08040c000c002c003c003820183118", "name": "Ancient Brewmaster", "phash": "695fdb1669257cd2d7f856ceda48006cc1c819349372b3c466cb7659ad897c98", "mana": 4, "whash": "f71febbff9bdf83ffc3f7c0efc4efc4d7c4c140c800c003e003c00386118711c", "card_id": "EX1_057"}, {"dhash": "bcc96c33c1fc8b3fb41050c1a7fe4b0db53a48e5318ac7149a3b746bdc963939", "average_hash": "07900fa00ba0083c9001c400004be44be407bc077c06f806f80cf80df90ff91f", "name": "North Sea Kraken", "phash": "5b7f3e04ef81dfa6bb2c76d8cee001a1212149cc8c997e7373e6f269cc88ccc0", "mana": 9, "whash": "0f900fa00ba00cbc9429c400014bed4bf547bd477d47fd47f80ff88df90ff91f", "card_id": "AT_103"}, {"dhash": "bc3f5cffa09915e2230c2e6cdcd9f933f360848309079e1a38a7a05c429a8d6d", "average_hash": "4780c30bf31ff01f381f381f9839d839d878783c782cf819e01b8015800d800d", "name": "Cogmaster's Wrench", "phash": "29615fe05e9c374c3e021e25c69884349361021381b13c5b6fa6796d6dffefd9", "mana": 3, "whash": "4780c30bf31ff91ff81f783f983fd83bf879f83c783cfc39e01b8015800d800d", "card_id": "GVG_024"}, {"dhash": "7cdf8d3842400463b9e4739f8d305ec739ea871472eac045a8b2b170ef8f1f3e", "average_hash": "37519f7fdbff0d6f34ee04cf3d43315e853ff103c504004000000000810e311e", "name": "Gazlowe", "phash": "df7a9f893f0136c2eb118601432110706c915626e64c1dcd7cbff9e4e76465e7", "mana": 6, "whash": "3f519ffffbffcf6f346f84cf3d4a315f877ff503c144094000000480818f313e", "card_id": "GVG_117"}, {"dhash": "00fffdff20c36f009c8d32096412c0a4cc089b3338e7874e3cc960341348ee97", "average_hash": "070881cff13ff47f367fb67fa67f9267d027c463c473e478e058001004000010", "name": "Lay on Hands", "phash": "3978c91a668d36d79e97760b0d488684e34224fc21676372cbc77b915b995ccd", "mana": 8, "whash": "071c80fff93ff47f367fbe7fa67fb267d267c463c473e479e058005006100010", "card_id": "EX1_354"}, {"dhash": "0082cd1c70f04bcfbcbf7360f78cbe3c0fba0322dbcdb0196033dc7682ed3cdb", "average_hash": "77bc03e10193807ef06ef866cc73843dc40fe003e00370031803005b00dbd04f", "name": "Flame Lance", "phash": "916f3d12ec8956953d91870c63464b864df229381b07cbc573dcc3c9fbe8183f", "mana": 5, "whash": "f7bf83f181bb807ef06ef867cc73843fc41fe807f8037803188304db00dbf04f", "card_id": "AT_001"}, {"dhash": "f8eff99f61362ee01c0031e4709ac1fc80d1631b0fb209250fc806904f32ff67", "average_hash": "0700e303f923fc637ee13ef09ef80ef80ef82678067c3250126010600c20de2f", "name": "Shield Block", "phash": "a788b3cdda557e200c178c236fb0061a53a80cd7b35286f64a7aee89af8dbf92", "mana": 3, "whash": "0700e303f923fce3fee17ef89ef81ef84ef86678267c3278126010601c70fe2f", "card_id": "EX1_606"}, {"dhash": "b8f3ecec41b84b42b6b44fc3be1e79e1e02a875dd28b2417cb36946902991d32", "average_hash": "0716839d819d803f803ec01ec01f243f603ed01f481e003ed03ec01f001f0113", "name": "Darkscale Healer", "phash": "29f53f999611e7c66fc809824ea0226084744f9c33cb3cf3e634d3f3ce09f69c", "mana": 5, "whash": "071e8b9f819f803f803ec03fc03f3c7fe03ed03ff83e803ed03fc83f401f2117", "card_id": "DS1_055"}, {"dhash": "30896f2ea4fd78e2f311e0e38487130f0e3e586cb0b0624825b0c9e13380b725", "average_hash": "0ff00fe087e084c1c7ff43cec3c1c3c303c78be383e303c000c101c227812721", "name": "Shadowfiend", "phash": "5d8c1d63269a8c87bf0444cc2f11c92409f0bbb112f356dafff7aad9692b0ac5", "mana": 3, "whash": "0ff00fe087e0c7e1c7ffc3cec3c3c3c3c3c78be3c3e343c240e345c227c12761", "card_id": "AT_014"}, {"dhash": "0280cd7f10c86330c7078e1f342f401e26f8fdc8f13143c2000119067a9cc6bf", "average_hash": "f7bf03e0338c001c021f023e0e7e0f5a1f023f17bf3fff0f0e0f460ce000f005", "name": "Blessing of Kings", "phash": "4b50eca3723b9790b7e63e0ea9511a740be92193c9b573764b6453a95b6ab350", "mana": 4, "whash": "f7ff0bf0338c101e023f023e0e7e0f5a1f023f17bf3fbf0f2e0f460ce000f00f", "card_id": "CS2_092"}, {"dhash": "80870d3c30d8033035e04ec19f803f017f02fe0cf439d077a8e9f4f78b3f080f", "average_hash": "0718c343c1878001800180018003c003e007e007f007f00ff00ff007f81ff80f", "name": "Moonfire", "phash": "3b713c330f88fbcc0f240e484e04092046cc0d144a7167cc5b7377f95bffdfba", "mana": 0, "whash": "673ec3e3e38fc11fc00fc007c007e007e007f00ff00ff00ff00ff00ffc5ffc1f", "card_id": "CS2_008"}, {"dhash": "ca871d7e30806360dc88b1c377beed389361ec8bdd13fe87fc3f30f800e41ddb", "average_hash": "0700c100c107c0100060024006608e698c79b07bf07ff07ff87fe07f807b803f", "name": "Upgrade!", "phash": "a9aa56f4f8f7ebff1e7b1f9ae694946007cc2c0cc33018294306ce28dbe29d89", "mana": 1, "whash": "0700c301e107e0334062024086698e799e79b07bf87ff87ff87ff07f807fe07f", "card_id": "EX1_409"}, {"dhash": "28cd6cf881a1cb1ab771568ba4365941868a3b1d64aa89441a39a7634c839130", "average_hash": "c793cfb38bbf80bee039c41f445d6d59604ea00680048000e000980181018101", "name": "Fel Reaver", "phash": "39392dcd9f27d6c6d78c13c939c161a0262c0d9949935968ee8d3737cfa17ac8", "mana": 5, "whash": "df93cfb39bbf87bee4bdc55f65dd6dd9634ea14681449140e040dc809181b321", "card_id": "GVG_016"}, {"dhash": "02c68d3f70e0c2903921c7c3bede03394ff8b8d0e305de0318077c0ef098808f", "average_hash": "071c03e3c187e007e005e005e001fc03fc07fe07fe07fe07fe1bfc19f808f804", "name": "Hammer of Wrath", "phash": "5357fcb2fc2bcfc496ed272d4e9242b282ab235869b17b524b41539959d079a4", "mana": 4, "whash": "771e01e3c18fe007e005e005e001fc03fc07fe07fe07fe0ffe1bfc19fc09f80d", "card_id": "CS2_094"}, {"dhash": "f89ffde13288cfff1bd01fbd6102c8feef70b8010492e700cf01f803300cf0ff", "average_hash": "0700e30779081c007e414e7ff62fb210fe27fc0fbc43fe07f80078026000f00f", "name": "Bouncing Blade", "phash": "03403f80fe5dda9778503e001d2007fe06b416ade9f73e568d0fe62f63c333f8", "mana": 3, "whash": "0700e30779081d40fec1ce7ff66ff210fe27fc2ffe43fe07fc007803f000f00f", "card_id": "GVG_050"}, {"dhash": "7af82ce77188cb00b6114c33b8c67297e70ac275b0cb301683b5896f6e9c8d30", "average_hash": "071f059f83bfc03fc03fe03ee03ec03fc03f803f003f00370009003e801dc119", "name": "Loot Hoarder", "phash": "39b57d645e1fded3ffa3c668868808e093e029043610b3217f02bf49bf87e7c1", "mana": 2, "whash": "0f1f0bbf83bfc0bfe03fe03fe07ee07fd07f803f003f0037001f003ec11fc11d", "card_id": "EX1_096"}, {"dhash": "03808d09e0dff83fa03e4fec91e627d95e32fb4c80e933917d37074c0490f807", "average_hash": "ffff0bf003c018041c001f0091804de04fa009fc017c253ef9bbfdbd3da6f987", "name": "Mind Blast", "phash": "e75776aac1a9ad8cb7d6336dcbe18b924b0c42b669c34b947bc81b2c5b207622", "mana": 2, "whash": "ffff0bf003c000041e001f0093804dc04fe009fc43fc29bed9fbfdbd3fa6f98f", "card_id": "DS1_233"}, {"dhash": "cad98c6291810a76372d57b6b87c7bf1f64246858d0a13c47433ff6eac981935", "average_hash": "6707f186930a180838140c1c4c1b783af81ff81fb81b70017801f81bb013311c", "name": "Wild Pyromancer", "phash": "e36d5f76de70dbd8eb841c8c5e9801b0648896450e0d3ec1ef01be70ff04b747", "mana": 2, "whash": "6717f186b38b9808381c3c1d5c1b783af83ff81fb81bf0017819f81bb813f91f", "card_id": "NEW1_020"}, {"dhash": "0af82cf0d1c28b85343b497ebeb879e1e382c9059f093e12e934926ab08d4137", "average_hash": "f70fe30f1b0f182e08260c203c383c38fc3b7c307830e822c8225002c0138007", "name": "Twilight Guardian", "phash": "cb57cf92d8c0f3e423b9136b369236ac092665c14dc293679e03ae25674bfb49", "mana": 4, "whash": "f71ffb9f1b8f182e0c268c24bc3cbc38fc3b7c38f832f832f822700be013c117", "card_id": "AT_017"}, {"dhash": "caf295c1229ffd307767cdca9a85355963b2ca4c15932a8947323be4728aef14", "average_hash": "f70ff15df15ff00f8cb1af31af31af311f391d300bb0019000d001c003000300", "name": "Prophet Velen", "phash": "af9ffb7978e8cdeeaf9a511c1f92131e10505a1803862a267f4e0ec4ad391e95", "mana": 7, "whash": "f71ff3dff1dff05f8cb1afb7af35af353f3d1f3c0bb0099008d001d003000300", "card_id": "EX1_350"}, {"dhash": "0cff0dfcc3c18dc3771caf705867d8882537df60f604084961b8d063ae411b3f", "average_hash": "f70bf30f831ff81ff83fe83ee039ec98cc144c26889f801100c300028003b003", "name": "Felguard", "phash": "69a56b82da519e06366476c86b304ec666e93639f6d8becc6d2dddc4aec18191", "mana": 3, "whash": "f71bfb0feb5ff87ffc3fe83ff031fc986c145eb68c9f8013008780038003b003", "card_id": "EX1_301"}, {"dhash": "888f7d78e0c1a3eec419093711e676dcbf7926136e26594c80bd1879e1e8fec3", "average_hash": "0718c3c70f8f3c2e3e3f0e030e020e00ce49f6c9c4e5e061007040f000f8e07f", "name": "Call of the Wild", "phash": "3dc85f54edef73fb3fa936240b402c20cb042434c6782ed85bd3ce985bc619a7", "mana": 8, "whash": "071cc3e31f8f3f2e3e2f0e070e024e00ced9f6c9f6eff0670070c0f880f8f07f", "card_id": "OG_211"}, {"dhash": "1ad22ce0d1c48b0937234cc4b83a71458f0a261558ea83c517328c6cf899f137", "average_hash": "f71fffafcbbec83ecc3d9c3d001100200001800100012000380038307811f919", "name": "Boneguard Lieutenant", "phash": "e39d19c6e334de32fbb0a6b0199903212120ac843e607e486f06bf677f27fbcc", "mana": 2, "whash": "ff9fffbfcbbfedbfdc3d9c3d017901600141814901493150381078b07991f91d", "card_id": "AT_089"}, {"dhash": "08a6a749608cf8b0f133e56e83991ac7349e012c0cd1fb0a023144e4bc8f0b10", "average_hash": "f7ec83eb81c280c000cc8bc483c463d26398e3803b8039c038e819e01b1b1700", "name": "Upgraded Repair Bot", "phash": "7717393a89c2e3587fc401333b92288840c8db8622722bc7fef76b6c72b64bd9", "mana": 5, "whash": "fffdaffba1e884c002dd8bd487c463da63e8e3a86be03de138e819e01b1b1700", "card_id": "GVG_083"}, {"dhash": "88c40c8cd13283e337cf6d36b02c6019d8e2e4d58b0b3b1736366f6ccc819923", "average_hash": "77197f901bb119b11c3e0c3f8c3fcc630c2830383c303c303c30183198019901", "name": "Spectral Knight", "phash": "879d5d8cf6e166b873425b0e49606ce1361b938db9b0d9584f9ed3692d29984b", "mana": 5, "whash": "7f997fb81bb118b91c3f0c3fcd7fed630d6871783d703c703c3118b199019901", "card_id": "FP1_008"}, {"dhash": "08c70c2ed1188a6334e659ccbc98694d969a2df57fcafb15ff330c7630880133", "average_hash": "77102f8003818803980388018819c01960007402fc08fc0ff81ff81ff803f903", "name": "Unstable Ghoul", "phash": "b35d9cc3f3846ff476ec292306220324c16033639919390cff8bbce4ef27e743", "mana": 2, "whash": "ff102fa003a198a39c038c038d69e1696549f443fd0bfc1ffc1ff81ff903f903", "card_id": "FP1_024"}, {"dhash": "f8caedfcc21d85317ac2b2e68104732fce7e3c399fb6fc49d1bb04f20304cfa9", "average_hash": "0f508f713bf89d7f9cfb8eebcf438f738763074043c7c3c780cb01c807808620", "name": "Grommash Hellscream", "phash": "9f2baf896fb17eb07e541256128606c999d50b8649a96e706cfde46d666693c0", "mana": 8, "whash": "0fd08f703bf89f7f9cf38efbcf438ff38763074047c7c3c780cb04c807a08620", "card_id": "EX1_414"}, {"dhash": "8af80ce011c942b8b52258c7bc4c7f31bcc2d085e18bd31787372c6f589ca130", "average_hash": "f71ff39fb39f900cc00d80040002000a201c001c101c301e383c301cb01d2118", "name": "Crowd Favorite", "phash": "e91b7b42fb19cb9cff8c93cc1ecc01f208ce06e18ec4d6b06e321b13dd20bf2c", "mana": 4, "whash": "ff1ffbbfb3bf91adc00f840c0452985e385e381d383c383e383d303cb91d211d", "card_id": "AT_121"}, {"dhash": "cad6cd6992f508b77b0eb014eba99c7339e702ea4dc49308263a98f0fe0cc1b3", "average_hash": "3713bb615bcb384ff84ffa477a86398839983b807e003c0018c048c0fa11c011", "name": "Axe Flinger", "phash": "c783dbf0ba78ee593f31deb4039061bc88b0430c4ce463677e6c33c14c5b79cc", "mana": 4, "whash": "37123b61dbcb78affc7ffe477cc6bdc83f9c7d807e803c0018d048c0fa11c811", "card_id": "BRM_016"}, {"dhash": "28f2ecf111ee8b1137074cccb97870f1e062c5c5938927136d360c6c3898713e", "average_hash": "c704111c3b1cc81e883e1838183e183e183e183e383ff83ff83f783e381ef01e", "name": "Ancient Watcher", "phash": "e9f51e135a9ab3f8fef087d93631088c3102930c73646788f782ff11df00df31", "mana": 2, "whash": "c704111c3b1ee81e883e1838383e183e183e183ef83ff83ff83f783f781ef01e", "card_id": "EX1_045"}, {"dhash": "2cf04c03c14c0ab3b56c4fd1b8e640fd8f023b05fc0a1215ec30f26fc8800136", "average_hash": "f71f439fc38c982090296038601c6004f806f807f80ff003e807f81ff0032100", "name": "Magma Rager", "phash": "73df6f423f41db647f2639882d899361967212b38ea6336469b132de966293d0", "mana": 3, "whash": "ff1fffbfdbacb8a414296439647c6044f846f807f80ff803f807f81ff1032100", "card_id": "CS2_118"}, {"dhash": "2a0ffdfbd0b5236b5f829b607e99f83cf331a0e10108c7533e8f74900130ffff", "average_hash": "07000302cb0bf83bf83ffe3fde7f8e7d0e7f0e3e24002018503cf0010200f83f", "name": "Betrayal", "phash": "2b519f4ff6ec7efc1f9b34b24376868c21d8163d21672e42c3986c82db60b706", "mana": 2, "whash": "07000302cb0bf93bfe3ffe7fde7f9e7d1e7f0e3e2c002018527cf0012200fc3f", "card_id": "EX1_126"}, {"dhash": "bcf72d8c6235c5c8b81177678c5e413d86722865fccb3997e733fc669896012d", "average_hash": "c70cc37de3d3e057e04e409c001c80010c030c008c111c19381ff01ff00ff90f", "name": "Mogor the Ogre", "phash": "197f3ca67b00860b1ec369c16c925236c672b3b0ee25d6fc9f27664fb498858d", "mana": 6, "whash": "cf1cc77fe3f3e077e06ec09e407d84438c030c008c3b1c393c1ff01ff90ff90f", "card_id": "GVG_112"}, {"dhash": "3acf2c0a61bcca60b4c3534dafb25bc7b40a6b5df62a04c548b8b3724380872c", "average_hash": "8791cfb3c3a3c4a9c443804fa55fa15fc15fc11ec11fc05fc00ec00781000120", "name": "Polluted Hoarder", "phash": "3967bf0db6177fe5cfb14bc82da1126204c891b10d9336f1c988d664e784afc9", "mana": 4, "whash": "8f91cfbbc3a1c5bbc483854fb5dfa15fe35fc15fc15fc15fc00fc08781800320", "card_id": "OG_323"}, {"dhash": "fcf98dcc12bb05603b86762e8c9679f9e6b2c104a70bde971236856c788cf339", "average_hash": "070bf35fb15bb05b804f009f081c0c3c5c327838683c203c183c081c1018011c", "name": "Emperor Thaurissan", "phash": "29ad9b48fe20279436064c2679d293a4991c2ed9a2a573e99e2c96dbdbb293e6", "mana": 6, "whash": "8f13f37fb3dbb05b805f0cbf1c5c0c7e7c3f7c3c7c3c343c383e083c1118011c", "card_id": "BRM_028"}, {"dhash": "8adfecffd1a08b1837494832b7ec4c19bf227b65e68a98153b31766ee4868139", "average_hash": "27008107f90ff806e8016801ec04fc01fc0fdc0fdc0dd809d80df801f009e009", "name": "Puddlestomper", "phash": "034fdc1cfcc1cfe37bf8868c2c9880e00620238c4ee6ee68b7863b1bff247f0f", "mana": 2, "whash": "2700f98ffb8ff81fe80f6c07ec0efc0bfc0fdc0ffc0dd80bd80df801f009e009", "card_id": "GVG_064"}, {"dhash": "4662898064c1bfb3154d22b04ec017c7000e7b38706585c83fb3ffe7d55e033f", "average_hash": "b97fb3df901f0cfc0cfd6ecc7394611f4104838603ae0088c080d00ff21fe63f", "name": "Kindly Grandmother", "phash": "795bcd7ba969dc63fba0e38b0f48874e2cba49563076726a8afa9934b8d11248", "mana": 2, "whash": "b97fb3ff909f0cfc0ebd6ecd77b4f11d41048386038a0088c080f00ff21ff63f", "card_id": "KAR_005"}, {"dhash": "828bfdff60bb4340df00be077ddbfb36f6ec0cd931f2e227450e8c1d193f3260", "average_hash": "070003008103e013701010101e78167e167f3a7f7c7f7e7ffe7ffe7ffe7ffe7f", "name": "Hellfire", "phash": "8b00fe7f721edf9fcc612e6026ceb9e009f809ee82472bf14bf14b054fc073a1", "mana": 4, "whash": "070001008100e003600010101c78167e127f1a7f7c7f7e3f7e7ffe7ffe7ffe3f", "card_id": "CS2_062"}, {"dhash": "9cdf2d3a42e04486b90b731d84324001854a3ef573aa864459b190720781772f", "average_hash": "ef54efe7ffff05cf24cff6c737413d4013400100354251c0404000c005800320", "name": "The Beast", "phash": "dfeadb803d062e23260127c03bc018480c73c639abac1ec71f8777fdf6c9d6d8", "mana": 6, "whash": "efd5effffbff877f64cffde73f413fc01b400342374251c044c2458005a00320", "card_id": "EX1_577"}, {"dhash": "e007fd7f50d60f3138fef0d8e781ff06f0790c8730daf3ee879d09333e60f8cf", "average_hash": "073043c0f901f8619841f043f043e02f607838720060246e3e6f1e7e0e40fc0f", "name": "Bloodlust", "phash": "a18ab2cd4f45ce343e614bb892618ca349980e46c3cc67a42bd39b35cbf736cf", "mana": 5, "whash": "0f3043c1f901f9e1f861f043f043e06fe07038720c642e6e3e7f1e7e4e74fe0f", "card_id": "CS2_046"}, {"dhash": "e803fd0fe074e299810706fe1b3c27700ce072c4f90982130727dbceb310ffbf", "average_hash": "0f7ee7e9ff80df268e1c1e183e3f3e073e063c0118020000000000000000fe0f", "name": "Shadowstep", "phash": "0f550f227c0af4c9c3c02e880d300c30096a8306c3d636e6e3ef3ef979f9ffbb", "mana": 0, "whash": "6fffeffbffcfdfa68f1d1f1c3f3f7e073e063f033d060900010000000200fe0f", "card_id": "EX1_144"}, {"dhash": "ccce8c34916902e7359a5334a6964b39aee2b8c5f60b5b179637116f24980530", "average_hash": "b711b383bb8718035807c807e001ec03b807dc0cbc1fb807e01dd01c401c0104", "name": "Scarlet Crusader", "phash": "d3d397217cb0e3b1ee9931984c0c0dc69e1926cd66e3c6938fa3797cc94683c3", "mana": 3, "whash": "bf11bba3bba71883dc07c807e841fc43b8475c0efc3ffc2ff43fd03e411c0104", "card_id": "EX1_020"}, {"dhash": "7cf2edcc02bbc5003a1f6cec9b7a65b14b429c84312be754c8b9b96916930137", "average_hash": "871f937db3d9a07f08f818b01c38dc3cbc2f382f3007c406e006e02368000100", "name": "Toshley", "phash": "89762d8976095b6aafd33446cee49294582c43b499f8d3e737d1466d0d8fc668", "mana": 6, "whash": "8f1fbf7dbbf9207f0c781cf81d7ddc7cfd2f3c2f3d07c406e406e02369000100", "card_id": "GVG_115"}, {"dhash": "4af08cc791990b43b6845019ad327a65e5ced01dab6bd64638b7656e83822729", "average_hash": "371f119e9b9f983fb03f300f3007303a70307038603ce03cc03d201d00090108", "name": "Dire Wolf Alpha", "phash": "e9f9cf211dcb67cec69cd870d2c6c398b13019cc3ba15b43bf8c7606d7083a86", "mana": 2, "whash": "3f1f1fbe9bbf99bfb03f300f314f317f7179703c613ce03ce03d201d010d0109", "card_id": "EX1_162"}, {"dhash": "0203ed3e10ed23f894c24fb09371fec2fc11e3ad8c53d10760f802bf0908fef3", "average_hash": "073083c09303e003c007f037f80f700ff83ef83fb93fbe79902f023c0218e01f", "name": "Stormcrack", "phash": "ab75f2973ef10f336d5eccc8938d2ce443eaae08897f2f44cbc4d2850fc15a02", "mana": 2, "whash": "073083c09303e103c007f037f80ff80ff83efc3fbd7fbe7dd02f027c0238f01f", "card_id": "OG_206"}, {"dhash": "2883ed7ff0bbc3de3f110c643fc07c24fd827267e60b06a66818bb3b4c37fa2f", "average_hash": "07000340830fe01fe41ff81ef00ed00cf01fc81ef81fe0004006a024303ef81f", "name": "Ball of Spiders", "phash": "6b763f219e097e8eeff3362023440914192880714b8463ccbb5b1ee773eb9bb9", "mana": 6, "whash": "07080340939ff11fe43ffa1ef41fd01ff01ff81ef81fe0056207e826303ef81f", "card_id": "AT_062"}, {"dhash": "cab9ad5312a77d15f92ae6ddd9b9a2e33c965b2ea55042899d3176e60c0c3318", "average_hash": "0700236831de10de00c813a8b3b479e27b987f9c3f9c798c58c818d813180210", "name": "Demented Frostcaller", "phash": "e721bd689e6a671ebf3412c31e9009a912ca99a40db36e81aef19e471d877ba5", "mana": 4, "whash": "075039ea31de10fe50eb11e8f3f479a27b987f9c7f9c7d8c58c898d833182318", "card_id": "OG_085"}, {"dhash": "8af10dc2129cbd1d731b8664ccd99bf33447538ce450d808a3330ce239877f1c", "average_hash": "f74ff1ff7bdc48d8e8d85bd81b889b9ffb97bb85db8c888f00cf00c203001700", "name": "Argent Protector", "phash": "470b61cff0dcf9b0fb70986d3792303c99bc1c620689db846726dfd1b3896e90", "mana": 2, "whash": "f71ff1df7bdc48dae8d85f981f889b9ffb97bb85cb8c088f008f00c203007700", "card_id": "EX1_362"}, {"dhash": "2af1cce611dc0a0935374ee4b6945969e2924d25d3ca349449308360cc87393f", "average_hash": "f70ff98cfb8cf80ddc08fc20fc0c2c196c13e4036c1078020800080018003104", "name": "Lance Carrier", "phash": "a3cf73c3ffc379ecb9b81e636362100c982c09c95b90e344e710db49f3283f87", "mana": 2, "whash": "f71ff9bffbbef83dfc39fc3cfc1cbc5bec13ec13ec18fc020800180019003904", "card_id": "AT_084"}, {"dhash": "0ad0c43581e30a87370e4c38b07040e989c21d95538a86042f37fc6eb0984139", "average_hash": "771e35853187180e783e3c3e3c3f700e780770077002700070107038f018f118", "name": "Southsea Deckhand", "phash": "e3e5dd927fe5d6f0bfb8258c6f4c29a463c2584e18429652e721b330b5013bcd", "mana": 1, "whash": "7f1f3ba733a7188f7c3e3c3e3c7f785ef847f0077002780070387038f118f918", "card_id": "CS2_146"}, {"dhash": "e2cfe57740c50f881d191b36344c68c0e10848b9974366840c017b9ab620e09f", "average_hash": "0718f1e7f90ffc7efc7efe3e7c7e5e3c2e3006102e320e4c0e084e007e007c00", "name": "Blessing of Might", "phash": "6346e973f2df3c39bef10ecf4f383e840c5a876c48d363111b144ed159e2931c", "mana": 1, "whash": "071ce1e7f91ffc7efe7e7e3e7e7e7e7c3e3006102e720e4c0e004e007e007c00", "card_id": "CS2_087"}, {"dhash": "18ff2d7c028c4500fb76b66d0c1951e472cea79dcf13f2090436b0e540428324", "average_hash": "e707e36fe14fe05fc059d29c921831183018f108f393e01f70be900f00050601", "name": "Dreadscale", "phash": "79297b6a1b9ef34cbe100725368006e304fc9a8d6b724bff47e556cc086336b9", "mana": 3, "whash": "ef17e37ff34ff05fc059d29c925831583118f108f39bf09ff8be909f02050605", "card_id": "AT_063t"}, {"dhash": "68e6cccc019b0b36376653bdac7073c1a4024c05bd8b63144432896c2e887936", "average_hash": "9719bb99bb99b839b839b81dfc1bfc337c103c02380738030000080008001100", "name": "Recruiter", "phash": "e3e93b613c11df8c77641bc249c22418309c970c6ee66ee35dc2de766e263666", "mana": 5, "whash": "9f19ffb9bbb9fcb9bc3ffc1ffd5bfd537d527c433d0738432000081009001900", "card_id": "AT_113"}, {"dhash": "5ccfac3dc1738a9d35b853f1bfc27f0fec3ad8e5a08b4e4639b5636c83912737", "average_hash": "0790afa1eba1b804f8006004e04fc07fc43f043e083ec02dc029c03301036100", "name": "Arcane Golem", "phash": "3997c720b60d0db3bda433c94e2286ac33e3cc604fbe7efa67c68ce1b9212623", "mana": 3, "whash": "0f90efa3fba3bc87fc04e406e14fc17fc57f057f057ec17fc039c0b381836100", "card_id": "EX1_089"}, {"dhash": "f887ddff2000637898ed011b17763ec46618be6038d1e022c345068b3917fb1e", "average_hash": "07107341f19f440006003603360f7e1ff607ee07be079e071a0e32045206c20f", "name": "Feral Rage", "phash": "5b5037b8ec839b876b48ce109e0c368026c12ce4287edb76b3395bcf5bf15e23", "mana": 3, "whash": "071ce363fd9f651086003607fe0ffe1ffe07ee07fe07be071e0e12045206f20f", "card_id": "OG_047"}, {"dhash": "88b00d65c2c2bd0573faa4b7d87fb50343979c6c65c19208353068e7058e9f1c", "average_hash": "774c33ec09dc08dc48de43fee3b3fbb1bbb32ba57fb97c9908c180c9031c070c", "name": "Scarlet Purifier", "phash": "4d9feda64cbad71bfb7033c16f4c18a602d023c5d8d89b853bcfeac4822c183d", "mana": 3, "whash": "671c33fc01dd08dc48fe43fee3bbfbb5bba30fa57fb95c8908c100c9031c070c", "card_id": "GVG_101"}, {"dhash": "7af2ece491890bc5b77a4ef6bc147339a6fa2e25d8cba0848137036c4e843d30", "average_hash": "c717cb99bbbc783c183e1c3f8c7b887b041bcc0f8c0f0807001c002400080110", "name": "Cult Master", "phash": "096f6be176d25b6abf2999859b4640e0e4b0932613c3dbe9cc306d167692ee69", "mana": 4, "whash": "cf97cfb9fbbc7cb81c3e1c3f8d7f897f855fdd4f8d0f885f001d003401080110", "card_id": "EX1_595"}, {"dhash": "7cc6ec3d41e1c2ceb53b5223a10c58b1ff6ab0f5c1cb0f969e363f6d769ae13e", "average_hash": "87908fa103a305a6044ecc06cd400550317c3d3c1d3c2c103434f83ce91ce11e", "name": "Drakonid Crusher", "phash": "4f169e707b84861177a136a40f0661e06c78068fd9e0f37cbf8c9ecdebc859d8", "mana": 6, "whash": "8f908fa003a307a7048ecc4fcd420558357c3d7c1d7c0d7c3c34fcbcf9bce13e", "card_id": "BRM_024"}, {"dhash": "0082cf7fb4c00b032e06581ab06e60c5c09a01f337664ecc9c98797187b0fe27", "average_hash": "ffff81f3e19ff03ff87f3c7e2c3a6460046006600640066006600e600e601c20", "name": "Holy Nova", "phash": "05fee905f90156a8165e061e87478683e3c16ba16b595b9073f053e87bee59ae", "mana": 5, "whash": "ffff81f3f19ff83ff87ffc7e2c7a74784660066006e0066006600e600ff01da4", "card_id": "CS1_112"}, {"dhash": "9cd94d39126604ce38386d7183ec47d18bc23ca533cb681695307a6904838934", "average_hash": "6706736739473046f046f8a0fc00fc00fc017c03fc06f8047800d80198000100", "name": "Alexstrasza", "phash": "c386f9e4dc809b193fd936c6362613664ea21a26aee88ee1372763ec538a733a", "mana": 9, "whash": "6f177b677bc73146f866fcb0fc507c41fc01fc03fc07fc07fc04f801d901d900", "card_id": "EX1_561"}, {"dhash": "3ad76c2ed15502eb359c536ca7d04d03bb7a40e58f8a5e4431334060c683bd24", "average_hash": "c798cfb4cba1f9a038427c00f846b94790479c11fd01f001f001080001040100", "name": "Faceless Shambler", "phash": "f3f37f633e5633c31c3361e063e00c69b1e0538226313be233839f393c99bb91", "mana": 4, "whash": "cf98cfb4fba5fda4bcc2fd42fdc4b9c79b579d51fd41f141f8510c8001840320", "card_id": "OG_174"}, {"dhash": "6ae7c48891990b3f37024eceb8146631b8b2c14513cb6c96db36b66d6c99dd30", "average_hash": "971df199b99dd81df83f383bbc39bc2c1c1e3c36180688009830c830c010f110", "name": "Leper Gnome", "phash": "a3b74bd933b1def26ff93c737e58298442121e128610cc00cec2c736f3a4bead", "mana": 1, "whash": "971dfb99f99df81ffc3ffc3fbc39bc2c3c1e3c361c069c029830c830e010f110", "card_id": "EX1_029"}, {"dhash": "02f80dfc70c0ee23fc6733077f04ec79f003c2170e8e041888731272e4c4fc8f", "average_hash": "ffff01fe81ff807f806fc67fc0ffe0fff0f8f0f88090c080404800480000c007", "name": "Hand of Protection", "phash": "79fb9d9ade4966ade7b513f612f4a648635a185a69807be11b253846c3d0529a", "mana": 1, "whash": "ffff03fe81ff81ff806fc67fc0ffe0fff0fdf0f8c0d2c08040c800c80000f00f", "card_id": "EX1_371"}, {"dhash": "6ac6ad3d52f089457f703f91fe46899912b3034a0e04f8087439f7f0e00567be", "average_hash": "d713f563fbc3f8e37cf03efd44fc4ddc0d88df80fe819801e0c0f0c042002400", "name": "Cruel Taskmaster", "phash": "338bbbddb36492631cd22dd92bb186e44c645a1279c21f49db317704e707bc3c", "mana": 2, "whash": "f713fb63fbc3f8b37c707e7d4efd4ddc0f88df80ff81dc81e0c0f0c042802000", "card_id": "EX1_603"}, {"dhash": "0180cf3fe0f3d9e6ad895b78b8f36086cd4c1b9a2c3441e886970d43f28efc3d", "average_hash": "ffff0bf0e1c7782f3c6f1ffeb7fce3f0e3f0a7c00384038427c02784018cf9bf", "name": "Lightbomb", "phash": "7d70a10fc9cf7e203527cd78c9410338a10c7b9553715ac753ec53e2537ad9b8", "mana": 6, "whash": "ffff0bf0e18778073c7e1ffea7fce3f0c3f083c00384038027c02784018cb9bf", "card_id": "GVG_008"}, {"dhash": "02866d3fe0f113f31ffc9f111e6638c1f086e00edf19a11346368ee91693798e", "average_hash": "771e01e1c183fc7ff07ff07ff03ffc3c703c781c301f301f101700060806d807", "name": "Equality", "phash": "e95537289e87ccfd668fbdf44971a41223780b4663f05a3b7ba45b255f009bc0", "mana": 2, "whash": "771e01e1c183fc7ffa7ff07ff03ffc3ef03c783c301f301f101700070807d807", "card_id": "EX1_619"}, {"dhash": "e8cf8c3901c00a90b766510dacb05bc7f69acc25f34b1297ff33ee67cc87892c", "average_hash": "6717f3837987000d8035100f3009701b603b643a681e080fb81fb81ff80f910f", "name": "Gurubashi Berserker", "phash": "cbd9fe94db13670c7bc087213c9239623c904c81ce395e4eebc2dba48e87b3b1", "mana": 5, "whash": "f717fba37ba7600d8035100f3859705b603b643e681e082fb81fb81ff90fb90f", "card_id": "EX1_399"}, {"dhash": "8af90ce6118c8a5935f34cc4bd887931a6b2e8659789200763b0046049801721", "average_hash": "f71efb9efb9cf820d810d833d83b983998185c3e5c3318101000000000000100", "name": "Maiden of the Lake", "phash": "27677b667e52cf8c6f8cd9c89b8402a614688749e1ccd2ccbb899dc96c82f396", "mana": 4, "whash": "ff9effbefbbcfca8dc30dc33fd7b9d79dd5c5d7e5d3b18721001000001000100", "card_id": "AT_085"}, {"dhash": "30f86d60428acd1cba21640e98bc707172c2ed84910b63968c37116f67989d31", "average_hash": "c70f037e0bde00de84fc04bc0c3c38383c1a3c1b3c36642e403e003c00180100", "name": "The Boogeymonster", "phash": "6d3ddfa27e02db48a7492cd69a9401e963d80bb19b8d57d87791b3ed63035422", "mana": 8, "whash": "cf5f8f7e0bfe047e847e0cfc0d7c3d783dba3d3b3d37647e443e003c01380100", "card_id": "OG_300"}, {"dhash": "1af439c082094d92bb646619998277455c3a8484322a6495c93892710084073e", "average_hash": "e70f877f87dc80dd80dda09fe03fa03fe03ff807f407e4164802000000000100", "name": "C'Thun", "phash": "19be29a93e0b9b4e9b5892ccf27c92db299ff2fc82c73c9d5b319604260067e2", "mana": 10, "whash": "ef1faf7f8ffe857da47da0fff17fa17fe13ff807f507ec164802100001000100", "card_id": "OG_280"}, {"dhash": "0a988d7612e3bcc47b3fe748cc14b93b72f6c4acc9d1108b4732dae4760be91f", "average_hash": "f707f166b14670ce38ceb99ebf9f09b919b81b9b09b91899188808c60817c21f", "name": "Snowchugger", "phash": "c90df372f2e2b6dd3bd016170bc1428400c9ba9c7379bb257b56db419b217e31", "mana": 2, "whash": "f707b146b3c670ce38debd9ebf9f09bb19b81b9b09b9189b188808c60a17c21f", "card_id": "GVG_002"}, {"dhash": "4cfc3ac1c192030d7752cc24bec06337df72cc8da829134724b46b6c9399e533", "average_hash": "b71f769f4f9d483fd83fd83fd03e8037803f983f003f00330021003120106112", "name": "Mountain Giant", "phash": "a9c54b43660c9fb1d3603672e3e78393907cecf607c32f8d9e981ed227a16e0a", "mana": 12, "whash": "f71ffabf4fbd49bfd83fd83fd83ec07fa03f983f803f00338031003121106912", "card_id": "EX1_105"}, {"dhash": "08f1ece7d1ca0b97372e4c0cbb10677dc83ad56510cb4c17bd36c06e1e893932", "average_hash": "771e5b9c2b9c183c1c3c1c3c1c3e0427402388311c2058307838003b201a3111", "name": "Wolfrider", "phash": "cfc74f927b2c36c6fa608764a7a329ec72d01c2c271e63a5fa68731c2929e3c2", "mana": 3, "whash": "7f1e5fbe3bbe18bd1c3c1c3d1d7e046f5063c8395d201c30bc3a083b391b3913", "card_id": "CS2_124"}, {"dhash": "fafc6dd2d2d089eb7f933766f288aeff51b7aa6a9515ef8a953f06f4fc0cf9bb", "average_hash": "0707831f83df801f487f583edc2e00aec9ac60843824282c181c18087800f813", "name": "Bloodhoof Brave", "phash": "c987996cfbe9fe18bf540e83029426cc38f8966406b137c596b17d6132d897c7", "mana": 4, "whash": "0717ab7f83df90bfc87fdc7edcae01bf49af498c3884682c187c188c7a087813", "card_id": "OG_218"}, {"dhash": "fafffdf163c38f0d1d363e787020e0c6e2ed0d7910c28b25f6c9c1171e3c7c64", "average_hash": "07000107bf3e7c7c7efc3e7c3c7e0078007042503e003e50a64f0e3f067f306f", "name": "Battle Rage", "phash": "ad2865f75b5f5eadf8e49e5927e21c82348b04fc1b30a4344f237b802f459f62", "mana": 2, "whash": "07000307ff3efcfc7efc3e7c3e7e0078027242503e403e50864f0e3f067fb86f", "card_id": "EX1_392"}, {"dhash": "42c2ed7fe0e003010e869d99b935f321e642c9c5863f010f003efc7ce0f99f21", "average_hash": "f7bfa1f5fb9ffc2f96331031207b207b307b903e907b007800788079c03dc01d", "name": "Seal of Light", "phash": "a975d39a5bdbe56b75283cad8ed098f006cc09664b7869c7d3897a0ddb80f380", "mana": 2, "whash": "f7bfa1f7f99ffc2f9e331035b07ba87b307b107e907a007800788079c03dc01d", "card_id": "GVG_057"}, {"dhash": "82c3ed1f44303b431c9e51a0c2a7051f9ffcf8dbe3b7eecd9d9e3f437c8cc139", "average_hash": "fffff9f1f9d3bc731e6059c00180070007c207e646fc66fe7efe7e1eff18fb1a", "name": "Power Word: Glory", "phash": "67e496bfe32d7de25a596d9e64c24fa28d700a1989d073cd139e73c05ba01738", "mana": 1, "whash": "f7ffe9f1f993bc711e4018400080030007c207c646fc66fe7efe7e5c7b18fb18", "card_id": "AT_013"}, {"dhash": "6a9fcdb8126009d37ea62c78d984b1107ee6708f05740189ee3791e5274edf31", "average_hash": "e707f167fb33f822bcf03eb0fcb039f9183e319c138808000080001000188210", "name": "Steamwheedle Sniper", "phash": "a325b3a7bbe65b5a3959346c06b04ce20c4716036b046e05bf85ef09fb9c73e3", "mana": 2, "whash": "e717fb67fbd7f8b2fcf23ef0feb13ff9193e3b9c138c0c80408200b0021c9610", "card_id": "GVG_087"}, {"dhash": "4cf10cc611890bb437694e56bc097837e0ced31dacf950c623b4af77fe843139", "average_hash": "f70ff31ff91e783c783fbc3ff83f103d303ca0340035002500002818b8083900", "name": "Master Jouster", "phash": "e9916b60532f3cf0ef600b4b189c60389cd089c37634933b1def9367799ab8d9", "mana": 6, "whash": "f71ffb9ff99ff83ffc3fbc3ff83f187f303ca03c003d002580016818f8083900", "card_id": "AT_112"}, {"dhash": "867d03c0440039677646ac98d9612693496f338cc4101c4898b021e34306877c", "average_hash": "f3dff1dff0df78b33bb13bb33bb053a4d3acd384e388c080808001840304070c", "name": "Nightbane Templar", "phash": "7515e1ad19aa796b7f02d49ea709de3304d6e69c24bc1268ed4976d2710e1b72", "mana": 3, "whash": "f9dff1dff0ff78f37bb33fb33bb0d3b4c3acf38cf388c080808001840304071c", "card_id": "KAR_010"}, {"dhash": "8ae42cdbd1ac0b3b36e64ccfb9387641cb3a86e5400abf977937596efe9d813b", "average_hash": "f71e799a6b99983880398033b833b8358031880b3800581fc83bc838e01be11b", "name": "Novice Engineer", "phash": "a96dffe06f6c4e4afb9889c4d230307269d2672219033ba06602fb911f8fbb6d", "mana": 2, "whash": "f71e3bba7bb9d8388c399433f833b87db079880b3008581fc83bc839e11bf11b", "card_id": "EX1_015"}, {"dhash": "fa83dd7ef07023c34ccc9bb333c3643e91d86e2185723a85c40e383de37b9cf2", "average_hash": "07000100d30388230a232267c667c6618e516e59e6512e730674407dc07ff03f", "name": "Blood To Ichor", "phash": "bbb0de63f8f776bd3febbcd2e3980c888e6239944b440925329179b043489f83", "mana": 1, "whash": "07008300d303c923ba232a67e677c6659e516e5fee516e734674c07dc07ff07f", "card_id": "OG_314"}, {"dhash": "cae704e7118e0b3836404c99ba7e7f4de3828c8d110b3704e43b40711e84bd2f", "average_hash": "770cf99ef9bdf838f83b00332030683d2c333823782070003006000300007100", "name": "Lightwarden", "phash": "e3dd63f57773d7ee7f989cf896c098c86328240b6c203e901b836b0279493b59", "mana": 1, "whash": "7f1effbffbbdf8b8fc3f10332179787ffc77782378217801781600030100f100", "card_id": "EX1_001"}, {"dhash": "289c2d30c238fc7379a0e24fcd9d92330ed6182c7d50d388033012e2c405393b", "average_hash": "876783e7c1c3c0c3b0c383c38bc29bc33fc33fc00f8d0fcf18c019c00b000303", "name": "Spellslinger", "phash": "9709b93888cccbd9bf4446273998868981fc198f66e63e3c98cfe6d1772c4b45", "mana": 3, "whash": "cf57cfe7c3e3c0c3f2c383c3afc2bfc33fc23f800fcd0fcd1cc019c00b800f03", "card_id": "AT_007"}, {"dhash": "0af044cc91b10b67361a4cecbdd87339cf7280e5610bd3970c3738680e94f939", "average_hash": "f70ff11f991f181f783ad8389c3c9c3fbc3f1c0f3c0c7c0be838e0083808b00b", "name": "Captured Jormungar", "phash": "cb5ff9e57ad85b9a7bf3357266e84c8c990c421218634ec93b638c955848e7a1", "mana": 7, "whash": "f70ff91f991f181f583adc389c3c9c3fbc3f3c0f3c0f7c0bf838a00a3808b00b", "card_id": "AT_102"}, {"dhash": "a8f0ec47119d0a32b46658c9b02664bdcf72b3d5679bcd040b392ff6dc8ce931", "average_hash": "f7977fbcbba899bb946184610561056005643164994058483c081818d911f911", "name": "Venture Co. Mercenary", "phash": "87533d707b85c7423db40c8c93a19838191693c1fb705b8e7f63e77ccf0a78c2", "mana": 5, "whash": "ff9f5fbcfbaabfbb84a39d6105e105e00764396439603948384838987991fb31", "card_id": "CS2_227"}, {"dhash": "38d38c27614dca9ab5654fc9b0327be7c6ce9c95292b7346c6b6d86d3181e53f", "average_hash": "07940fa4c3a084a0a041b029307131783077303f31273007703270326100e118", "name": "Questing Adventurer", "phash": "f3f41e133e86cf94efc1668c1b83088c386246e629e5527ab97266a75969b6d2", "mana": 3, "whash": "4f940fa4c3a085a0e485b579f5e071f8337f317f39773163707274b27180e339", "card_id": "EX1_044"}, {"dhash": "02800d3f30fce3e2ffcdff333626080c7078e0c08101030206b40c5ef93fe02b", "average_hash": "173c03e08183800f807fe07fe67fe21f0c3f1c3e3c3f3c26300030107c3cfc2f", "name": "Polymorph", "phash": "e95c3ca3f6c3c6352f0ff7b889560cad189323c8caf109f79b0c538a46e55b90", "mana": 4, "whash": "77be03e08183800f807fe07fe67fe21f0c3f1c3e3e3f3c2e300030107e3cfc3f", "card_id": "CS2_022"}, {"dhash": "cafb996702dc85383a252c4a92c6670d467a88e10187230adf3e36f74c0af99e", "average_hash": "f70ff76ffb4df85c8c318cb18420803604330c211c203c201c30181c181c181e", "name": "Varian Wrynn", "phash": "83b3733863a3c768d974b634963c186d39ed7243687a5e4c47855f923b9916e0", "mana": 10, "whash": "ff1fff7fff4df87c8c79acb1a4208536053b1d2b1ca03c201c78989c1a1c181e", "card_id": "AT_072"}, {"dhash": "b8f12cc3c19e827d31b25265a55a5af584023d5d5e2abec4c03983733c877d3c", "average_hash": "079f4fbc63b8cdb9ec4ac046dd427552f1405042d101e04324061806090fc11f", "name": "Injured Blademaster", "phash": "536e27870d1dde98efc1cf2439a0c6c2333c8c718ce1dbc6e6d84c9c9946267b", "mana": 3, "whash": "0f9f4fbc6bb8c7b9ec8ad546ddc27552f7405142f141e1472c061c86098fc33f", "card_id": "CS2_181"}, {"dhash": "88d0ba20412f4aceb114527da5384b7183d216b5a5ea6d551ab82561f3827d3d", "average_hash": "fff3fff03fe03de03ccc3fc01fc01fc01fc00fc00fc01dc004c005e005800f30", "name": "Molten Giant", "phash": "d770e7003b143f943f8d12e18be5d8daa92fb987bf05de41ff007f006f007b20", "mana": 25, "whash": "fff3fff07fe03fe03cec3fc01fc01fe01fc11fc00fd09dc024d005e045a00f70", "card_id": "EX1_620"}, {"dhash": "cadd8c811153029835275e4ea8ec40f98f723ec5700af957fd38d273e68f8d37", "average_hash": "f71fff9f7ba8382c1c0c9c181c000c00ec00b801b8009025d807c003c01fe117", "name": "Ravenholdt Assassin", "phash": "c3f7e7f979c4f7f077bc1de7cdb1116858864cd2c3082d90360b1841dc187ac6", "mana": 7, "whash": "ff1fffbf7ba83dbc9c3c9c3c0d544d40ed40b941b9009065d80fc007c11fe117", "card_id": "CS2_161"}, {"dhash": "eab2cdc48083393f72dae7269cdd3ef97c32d36ca681840b183608e695886b17", "average_hash": "070c81cc11d810d8188a5b8c19b69dbeddbeddbcd9bc78bc00bc009c031e030f", "name": "Holy Champion", "phash": "4d87f73bdca865379b32d4185bc608e6893caec40cb66e84fb65ca01b3cc7a45", "mana": 4, "whash": "074c91dc11d818dc188a5b8e1dbebdbeddbeddbcd9bc78bc00bc00dc031e031f", "card_id": "AT_011"}, {"dhash": "3cd6ed6c93d90fa07c4b2b964c3dd3402e831420ab641ac9e0b849e0974d6736", "average_hash": "c71fcb07994f987f082f4a8d101178836c234503e081e00300c3400100100014", "name": "Dread Infernal", "phash": "9b899b53393333196f2436007c940ccc4932e361ccf9a369f3c7d64dba6674ce", "mana": 6, "whash": "c71fdb4fdb6f98ff0cbf4ebd5c95fec37ca34783e585e01300cb40412092001e", "card_id": "CS2_064"}, {"dhash": "eaffa53342c80411b9e076c78d6a5d958c2a30d5452a26c5dcb3786381849b3d", "average_hash": "071ca56fe1efe84dd0d9c0d1d059d15de00e300c310060006008200811000100", "name": "Rend Blackhand", "phash": "7bfa1bab9f7deffedfe3364c8e814e11268026896b885e347704328793809720", "mana": 7, "whash": "1f5fbf7ffbeffd6ff479c4fbd559d15de34e310c314061406008608911801129", "card_id": "BRM_029"}, {"dhash": "0080fd7f60c0037e9e983f39fc407089c7a29f47330e6c17da3e047ff895e82b", "average_hash": "17be03e0f187fc19fc0bfc7e7c7e3c709c63d867f026e036f0347018f01ef00f", "name": "Dragon's Breath", "phash": "e1f71f087e01bca69fc92d3813c806c38a11625c43870b376ef57be0dba85b9e", "mana": 5, "whash": "77be03e0f18ffc01fc0ffc3a7c5c3c709c63d863f066e016f0367018f00ef00f", "card_id": "BRM_003"}, {"dhash": "2af1c5e9129309247c4ea68cd958a2e00fc2972f29c41a09f43aa2e704416b36", "average_hash": "f70ff15f791f781f38493830183018b03007e18168817801f883d80740047000", "name": "Timber Wolf", "phash": "e399edf97bee7b9afe760c2736801972c6f064043165ca186b84d6069e0bccc9", "mana": 1, "whash": "f71ffb5f795f783f38ed3e393cb01db23807e98178817c01f883d88750047006", "card_id": "DS1_175"}, {"dhash": "02f648fc81811b3137c25c1cb8396431c264eccd9802215d02bab061c3970d7f", "average_hash": "f30ff11fd81ff81d083f183f383f383730139013901b101200008000800f017f", "name": "Murloc Tinyfin", "phash": "89f14bd37b5bb261db5256168da53699268c12183732ce70cd648be517a7de4c", "mana": 0, "whash": "ffbff99fd83ff83d183f183fb87fb83fb817b833b01b101300108000810f817f", "card_id": "LOEA10_3"}, {"dhash": "88f40cc0c1928b2d37564c4cb8906301c6b29875b1e942d707360f609886c139", "average_hash": "771ff39f099d483fd83f983fbc3c9c370c230032003e003c0014300010080118", "name": "Illuminator", "phash": "e9dd6b097e09de9cd630c3306be312cc19ccc7d8e63066f23e0d3cc764a3a6a4", "mana": 3, "whash": "7f1fffbfabbf48bfdc3fbc3fbc7c9c7f0c670037003e003e203e3000110c0118", "card_id": "GVG_089"}, {"dhash": "88814d2e90b8216036ffcd7c9e8931674e9abff4c1888313662c8f5a1edcf0b1", "average_hash": "673ec3e0d380b01b001a9007d01db0302028441d781c383a307b383e383cf83f", "name": "Healing Touch", "phash": "e9553ea267a537683792a5484bac0c58ac5c4fa159594795d3f21c9e1a8f7cca", "mana": 3, "whash": "f77fc7f0d390b31b001a9007f81d98316028cc1f7c1c783a707f383f383cf83f", "card_id": "CS2_007"}, {"dhash": "2ade2ca0d1200b7cb7f14ecbb8846039a0bacce5398a57042c3bc46018812729", "average_hash": "f79fffa7fbb33830a4398c3d8c3f8c7d80124c33381638099801380001000100", "name": "Stormwind Knight", "phash": "bbbd3bb5f6c4dbc61ff21b3329d8286c2461c660070763c36d613c5b7c087664", "mana": 4, "whash": "ff9fffa7fbb3bcb0e4388c3dcd7f8d7f855e4d733956394d3801380201000100", "card_id": "CS2_131"}, {"dhash": "9c981c38a0d0143223d626accdd396077915f6cad9f4a0db47a68c553aa3f56c", "average_hash": "e7c7e7a7f747b841a803d883f002b006a30d000f001f003a00160416010e692c", "name": "Sword of Justice", "phash": "db913341bf9e3d847310b324194d1c0b0d31cc3087e14f433fcd58e6bf6d6f1a", "mana": 3, "whash": "e7c7ffe7ffc7bf47b903f883f003f086a38f800f001f003b04170556190e6bae", "card_id": "EX1_366"}, {"dhash": "46e3c0ce85e11f033e965d6cb29841e19e023945e283ce9c3d3b4a561c89913f", "average_hash": "bbbe993fb83fb83f98369c36f84c3c003c021c0608040808180018080100113e", "name": "Violet Illusionist", "phash": "c973e97369195ecbf77a76cb7dd2d664cf4c461849895334a1149634318c3709", "mana": 3, "whash": "fbbeb9bff83fbc7fbc7ebc36fc5c3c403c461c061c2c1c181c1818183114113f", "card_id": "KAR_712"}, {"dhash": "cad38cf5d1c50ba2375c4eb5b9c47381e60ac73d9f6b0ec71936306b7386f538", "average_hash": "171eb39dfb8eb83e381cd01bec31f431f038e0386033c031c020400ce00cf11c", "name": "Defender of Argus", "phash": "637ddf214f139cb07f3c96a54cc84cc885710c1ce62c7346dbc46c466f92cfc6", "mana": 4, "whash": "171ebbbdfbbfb83a3c3cf43bec71fc71f078e038603bc031c020e00ce11cf11c", "card_id": "EX1_093"}, {"dhash": "8af98cc0118f0b13374e4cbeb9647ae1fc02c3658ecb9b1776366c6cf898d131", "average_hash": "771f719e3b9e383c18396c307c32fc38783c683808383838381820185018c118", "name": "Twilight Geomancer", "phash": "e36dcb64db315fd87bee93cc1ecc089221e092c4330867847f02be01ff11ff98", "mana": 2, "whash": "771ff9be3bbe383c183d7c387c3afc7a7c3c6c38383838383838201c7118e11d", "card_id": "OG_284"}, {"dhash": "4290c86181051a773c8e5812bbf457a5a74a48959062a35d4cb819614396e57f", "average_hash": "bfad1bac19a8186018326c602570b473f013f0136053745a20000000010083ff", "name": "Ancient Shade", "phash": "e7b5c7974c53fb79b9317638731c33f799618631353262ce198c9a3101a13e1c", "mana": 4, "whash": "bfbd1fac1eba1c703cf26f602b60b373f37ff353e153705a258200000520c3ff", "card_id": "LOE_110"}, {"dhash": "f8c82d77429ec43f3af070e08d0852ff87327f0dbc6b60d6a0b007763d88f331", "average_hash": "871b776aebfee45bfcd3fcc3d4739810ec01e801c02244000000101031103100", "name": "Foe Reaper 4000", "phash": "b3ef7b2aec581e5e8e030c61b3d003090e4dade518e75a5b5d23e5b323b1339c", "mana": 8, "whash": "ef5b7f6afbfff47ffc73fcc3fd539950ed41e901c16345600400109031903130", "card_id": "GVG_113"}, {"dhash": "0080dd07707e4378b4e2bd9d773dc4739a863c0dc01399ef10d867907f27cf7f", "average_hash": "073003c00380403080010021007f30fd307f117f037f83ff86f78f63ff1fff7f", "name": "Twisting Nether", "phash": "bd80f6fd3734fd23cb089c1a09e90a6421b3638d8946eb91c6507aa7dbd81aed", "mana": 8, "whash": "073803c003a0c13080110029007f30fd307f107f027f83ff8effaf63ff5fff7f", "card_id": "EX1_312"}, {"dhash": "8280ed3780c7011b07ac4cb8b942d78dfc1bf155e4b7836f0fdf1cbe7974f6e0", "average_hash": "f7bf2be03b8478081818080000000006006e003c003c02fc127c367c36fcfeff", "name": "Consecration", "phash": "6df2d207a7286d931d7efc283b8d29720bc70bf183535bb9da115b30d3a11b18", "mana": 4, "whash": "ffff7bf03b84fd0c3c1978006400040600fe007c027c02fc12fc367c37fefeff", "card_id": "CS2_093"}, {"dhash": "2cc8cc25011f4a7eb5c85001abd256459d6a261d592a3244f9b800614383e73e", "average_hash": "77951fb07ba8fda1f443f443f557e55fe15ff14b514101428003000001800120", "name": "Corrupted Seer", "phash": "f3b8e3012e164e834da11ca693a033e831986694e3e84e6e7f9bf3d92d2b394b", "mana": 6, "whash": "7ff51fb07ba8fee1f4c3f543f7d7f5dff35ff149514101c28043058405800320", "card_id": "OG_161"}, {"dhash": "f8bffdff000003f01ef33984f308fe79e8e3e083810f031f0660fc856e5a9a3d", "average_hash": "0700f303fd1b7c019e341c37087f007f087f307f307b307030308004d005f80f", "name": "Unleash the Hounds", "phash": "89d4136e7213f429598a9c468c600e92acdc0cb3c66d43ff4b652bd3794ff34b", "mana": 3, "whash": "0700f343fd1f7d01be76dc3f087f007f087f307f387f387a3030a004d80dfc2f", "card_id": "EX1_538"}, {"dhash": "d8bff9ff20ce6378deb03d4362804420db40ae9d1d333f043f0f781cc000809f", "average_hash": "07000300e30ff01ff207f677fc61f060b020f021f011fc01f019f011f001f00f", "name": "Bash", "phash": "33405d245ec5ae6d861a379a0f20b0704e63129f29bd1bdce9f45ee25bcf530b", "mana": 3, "whash": "07000300f31ff11ff23ff677fc67f060f061f023f831fe11f419f031f801f81f", "card_id": "AT_064"}, {"dhash": "9af66dfed2e485057b1c3e3098e06047749e80e1b9c7e70b193e66e6cc0891b8", "average_hash": "e709c17dc95f685e783e78bc703c202320380c203c32183c983d981998181018", "name": "Malkorok", "phash": "29ab7bce3ba39f1fbe588399c6d0a698488cd3841cce6fc1ed4d9620c9a2de60", "mana": 7, "whash": "ef19cb7dcb5fe87e783e7cbc7c3e612b20380d203cb29c3e983d98b998189018", "card_id": "OG_220"}, {"dhash": "caf08ce111cb0b2337c64c04b93c72d9cd929225610b02061cb72e6cdd9e633c", "average_hash": "f71ff19f7b1f783fb83f1c3f1c3b0c306c26680f60020000000000007015611c", "name": "Bluegill Warrior", "phash": "e99dcb657bf17e1a7f1e96638be19130241826989841db847e06e7849e23fb61", "mana": 2, "whash": "f71ffb9ffb9ff83fbc3f9c3f9c3b2c7cec2e682f681f200000008000701d711c", "card_id": "CS2_173"}, {"dhash": "0af80dc092082902764ce99e0030c1213e6348cc3f80708b03343fe0dc4fb39b", "average_hash": "f71ffb5ffbdc389738a13ea13f003c801d801d801f8008b018a01840a01bb013", "name": "Windspeaker", "phash": "c7317be279aa7fc67b6816581b5414d010d49281c7a71d4d36a37cd0ae926eeb", "mana": 4, "whash": "f71fff5ffbfff8bf3ca13ea33f813e801d801d901f800eb01ca01840a09bb413", "card_id": "EX1_587"}, {"dhash": "88dc2d21c25244bcb96073c987824b1db3420f34c4cb98076936466088814532", "average_hash": "070327672bef7947f0c3d0c1b44bc009c019e003e001c833d031900009020100", "name": "Kel'Thuzad", "phash": "bbfbdf88ce862f166e210fe1329192f104c64ecc6ae387c5365b19dbd1098599", "mana": 8, "whash": "2f572f67ebef7647f443f4c1f55bc14bc95be107e101ed73d431f08029070900", "card_id": "FP1_013"}, {"dhash": "8cdc9ab141a28b0d35db5aa6bf6a7cfdf13207c51c8ab004e333d266a2914527", "average_hash": "f7133e833f86581a0c08cc0b441c443cfc38bc1398039803d803c01b40020102", "name": "Clockwork Giant", "phash": "83cd6f98de0093b16fb038e69263a32b642e58cec9e1f7707f381f328f21c718", "mana": 12, "whash": "ff177eb73fbf58ba4c39cc1a457d447cfc78fc1f99039803d807c01349064106", "card_id": "GVG_121"}, {"dhash": "ca03ddff30fa03269fcc3c1961f0c7c8bf3370cde613cd65328b48763d29f09e", "average_hash": "07204100f109f00ff07b707f3074e003e001c81cf81efa1ff40d7c0f7e06f80f", "name": "Earth Shock", "phash": "e3a8bc9ddef57e733696a5f8ad638ccc1b19c666cc980f34b1c4238148217639", "mana": 1, "whash": "0720e380f109f10ff07b707f7074e007e005c01cf85efa1ffc1f7c577e06f80f", "card_id": "EX1_245"}, {"dhash": "0af88d20029b48a0f97fb4f4e8e187d30c87114aeb94f04937bae4f404081bbc", "average_hash": "f71fff7ffbedf1c900c0fac0f2c1fbc16bc44b844b81109490d390c1cb006600", "name": "Ancient Shieldbearer", "phash": "f38afbdaab79dbd67b4699252ec41a06c97032c1b3c81f93e38c16434c333ca2", "mana": 7, "whash": "ff1fff7f7bedf1e944c0fec1f6c179c56b8563854b81109590d390c183807400", "card_id": "OG_301"}, {"dhash": "3a7afce4c04113032636396cf318e4b3ecc0f9821b07d25f40ae04514aaebd7c", "average_hash": "8787c30fc11fb83f18369031982438263836583f607f303f103e001e000f600f", "name": "Gorehowl", "phash": "c9f717b35b1b7d8c5e021473b3dc8627c7b4529a00c2b331533cac4866e56eb6", "mana": 7, "whash": "8787c38fe11fb93f18369873dc3d386e3c36583f607f307f103f001e800fe01f", "card_id": "EX1_411"}, {"dhash": "ee3fb8f7416ab79f4f14d968b2d074a1e1c2c3c18b0bcf17802f00def89cfc79", "average_hash": "0140c107e037f07ff83ff83cf83cf83cf83e783c783ef03e701e101c081cf81f", "name": "Raven Idol", "phash": "e9d7bf321ee9deed9e776ef3b74f8e09b13e0cb6849a000ca434889227c88225", "mana": 1, "whash": "0140c107e037f07ff83ff83cf83cf83cf83e783c783ef03e701c381c081cf83f", "card_id": "LOE_115"}, {"dhash": "fadda533126f3cde79ac86d09b2166035886e11d06334c4890b1a3e643831724", "average_hash": "07007162f9c6f846f8c6fb95fb37fb2e732e633cc3a283c080c000c003000700", "name": "Eadric the Pure", "phash": "f736e3ae1efebcefef5815993f9126290040594628ba0e819f254b67618a1cd6", "mana": 7, "whash": "0f107b62fbc6f846fec7ffb5ff7ffb6e7b2e633ec3a283c080c001c003800700", "card_id": "AT_081"}, {"dhash": "02800d7ff0fce1e3771f4f3c9c3830c0412266ccb5b969438b06366dcc92b82f", "average_hash": "ffff01f0018e001f001f181ffc1f7e1e98098001a00436261e309c341001f00f", "name": "Redemption", "phash": "c9d75d6dbcfeb6b6b3c9a7e44b321893ebc03b9023615b215902de0059e6338a", "mana": 1, "whash": "ffff0ff0038e011f001fb81ffc1f7e3f9a099807f2047f341e309c3c1803f04f", "card_id": "EX1_136"}, {"dhash": "16a3604ea49e5d59fbb054c7adfc7b33e6444c1c18223a5ce1b98b4f37984d31", "average_hash": "eba6c998c059f058f81aec181d5fb87b98738013800300430003003b0138013c", "name": "Book Wyrm", "phash": "b9ed63276c56da90af179f1e7e9665e4d382492dcd9852da31147a188985c783", "mana": 6, "whash": "ffafc9b8f4d9f458f8daed581f5bbb7fdb77e117810300470043003b0138033c", "card_id": "KAR_033"}, {"dhash": "eaf30dffc2c009027f10a66660cfbc3960fec3ef8e850c4813ba67e74d408b3f", "average_hash": "f71efb7ffb1ff8be7cfc4edcc2cc85bc003809b803bc009c00c0008000000601", "name": "Fiery Bat", "phash": "6de763eef3e8fff6bed21eea2c8084218cb19a115c5018449d281f0d670bf6f0", "mana": 1, "whash": "f71eff7ffbfff8be7cfc4efcc6ec8ffc053c0bbc03bc409c00c4008002800601", "card_id": "OG_179"}, {"dhash": "9c762ce96012932c26403dc0f39960b398648c231e07f3dfceaa115464bae97b", "average_hash": "8785c30de31f603ff037b079907d9067d06038009c036036701c181008000008", "name": "Powermace", "phash": "a9d639985e8d5668564225b28451a6c5d2a49a364db06d2c372dda29d7b67fc7", "mana": 3, "whash": "8787c79fe35fe13ff83ff07dd07ff067fa6078209c236436f01c1c1059101918", "card_id": "GVG_036"}, {"dhash": "0af48dc0123105413b8a64c69f2c715942a290c4358b43040cb3b86023870d3c", "average_hash": "f70ff15ffb5ff85718175c3a1c3f0c3f0c390424700478000000000000000100", "name": "Nat Pagle", "phash": "cb9feba97328f3d03d5c694c52a212f2800dcb062eb2760bb685be07cf09ef89", "mana": 2, "whash": "ff1fff7ffbfff85f5c7f5cfa5c7f1c7f1c3f5c25780678047000000001000100", "card_id": "EX1_557"}, {"dhash": "3cdc2c38e1b4ca60b4e15189bf12783ff0e2e59dc36b0b171c36266b188ee13c", "average_hash": "8793cfa343a3c003c0038003803100783c38303c703c783c3c30300d591ce11e", "name": "Hobgoblin", "phash": "69dd9e441b06e7013f8396b1739406f9922664e8b8728f8f9b79fce4c3c8464e", "mana": 3, "whash": "8f93cfa3c3a3c6a3c487c027817101783f78317c717c7d7c3c32108d599ce13f", "card_id": "GVG_104"}, {"dhash": "faf1edeb02170d00f89f3039479acc24377f6ce8001487c8deb97de3e044c33b", "average_hash": "071e1f7f7b7f7cff04c676c317cc16cc07db8b9f0bcb20c060c070c0e0808610", "name": "Fearsome Doomguard", "phash": "dfe9cfee6fd69fcf9fe176f21e44990c14b0420661b23b08c78866802b221c9f", "mana": 7, "whash": "0f1e1f5f7b7f7dff049636c797ce16cd06df8b9f0b8b208074c074c0e0809611", "card_id": "AT_020"}, {"dhash": "42188d3224e778d8d180a359c0bb9be32e8d597a83e404997b72ce813107e6ff", "average_hash": "ff67b9e7b1c2f0c2f2c3b2c033c01bc043826380638050d018d039820602c4ff", "name": "Obsidian Destroyer", "phash": "f3e4f372897ef6ffbd5cc6b939ba3a8421e76246219a2e7489218ec139320270", "mana": 7, "whash": "ff67bbe7b6e6f4c2f6c3f2c1f3c01bc263c26382e29078901cd039024602ccff", "card_id": "LOE_009"}, {"dhash": "ca83cd2f92750c837976e3c807fccbf817e12cc67988e300cf39bcf16843530f", "average_hash": "f710fb60fb41f8417cc0bec1df41fcc3fc83fc837d803d003cc038c018000000", "name": "Ancient of War", "phash": "c72bf3e1fcf2e7bd3bf46cb22e8541a1201c3c0649c19b01e389709c7c86cfc7", "mana": 7, "whash": "f714ff60fb61f8c1fcc0fec1ffc3fec3fe83fe837f813f003cc038c018800200", "card_id": "EX1_178"}, {"dhash": "3cde6d31c286441fb9f475338ac84031bbe25545e78acd151a33346f6c8fd93e", "average_hash": "0700036201c00040d0c0f0c6780098003818781c781c781cfc1cf81ff91ff91f", "name": "Majordomo Executus", "phash": "63a79e88b9046e0bbf0d17746b801a91867936c66bde4daddd31527b66c22587", "mana": 9, "whash": "0f140f6003e3044ad460f4c7dd64b9403d187c1cfd1c7c5cf81cf81ff91ff91f", "card_id": "BRM_027"}, {"dhash": "82803dff70f6e38c9c1900681c1b08b0d36c6fd3380643106661c84f3199fe3f", "average_hash": "7f7e67e06bff6f7f663e7e1ebf081e0038403ec03ec02800300030000000f00f", "name": "Shadowflame", "phash": "674c2d8cf99c96ab96636e510ba80de069d68cf681e9be3c4bf116987bc17e80", "mana": 4, "whash": "ffff6ff06fff6fff7e7e7f1e3f085f087e407ec03ec02800300030002000f00f", "card_id": "EX1_303"}, {"dhash": "fcd6ad4f020d457ebeac61fe81f043d18fa21b253e4be094e03100771b80f724", "average_hash": "07012b61ebd0f070f0f1fcb0fc40fc01fc03ec01ec01ec000806000800000100", "name": "Hogger", "phash": "b33ee7a0fc901b0e1fc7cc4033900cf1cc6032790b674e6639e5db3c670b97ce", "mana": 6, "whash": "0f556ff9ebfaf671f471fdf0fd41fd41ff43fd01ed43ed442c06048801800120", "card_id": "NEW1_040"}, {"dhash": "08e40d4153da84e57b97a73c48e1b06b03d706ce4f95190824314ae0b5404fbd", "average_hash": "f71ffb3ffb4f7845ccde7fde3fd060f123803980db80288000c0004000000600", "name": "Undercity Valiant", "phash": "f389fb3afbd8f3593b5019ec4b640946109832c2a6c17e187f023f43ff017761", "mana": 2, "whash": "f71ffb5ffb4ff86dfcfe7ffe7ef366f1be80b9a0dba0388000e0006002000400", "card_id": "AT_030"}, {"dhash": "927409c842b3bda65649ae02540d8d7272e7cd8e1105661b8eb394a6694886fd", "average_hash": "e30fc11fc01fe01fe09fe88f880b180c1839383628a66096701ef01ee00cc07f", "name": "Unearthed Raptor", "phash": "490d990b79faeea69e4b536b97166b118de9561229ad32b49d26564a192736d9", "mana": 3, "whash": "e10ff15fc01fe09fe8bfe8afc88f188d18b938bc28a6609ef01ef01ee00cc07f", "card_id": "LOE_019"}, {"dhash": "9a333046eee059fd93f2af09df630c867f4dde123c2d70d9e02341d6d6a71fff", "average_hash": "efcee7cee3824182f08790076006430cf007b0078007848384038103c903ffff", "name": "Cursed Blade", "phash": "dbdd66669f2f5bb86f425e1fc5f59b556fee9e5d643691216422334124801180", "mana": 1, "whash": "efeee7cee7c27182f087f087e206630df00ff007c0878407c5038943c903ffff", "card_id": "LOE_118"}, {"dhash": "eaf1fdefe094e327137c3e936c2cc203fc9f91610ec2f1ce8e092b80ff38f8df", "average_hash": "07006308fb1c3c3c3e1c2e5f867f8847047e0f7c0e79047e0c7c18000e00e61f", "name": "Multi-Shot", "phash": "0fa86bcdf655bc7eb826dca91b580e90c9b4967729cba9fcd3c1330a4b403d24", "mana": 4, "whash": "07086348ff183c3c3e1e2e5f8e7fc847047e0e7c0f7d2c7e1e7c1e000e40f63f", "card_id": "DS1_183"}, {"dhash": "d4b02fc1c080b910726fe4ce9cfd313b6376ceec8491940b3932f3e4e6899b1f", "average_hash": "a74f63cd71dd18d808b09bb909b889b99999899bd999d898c891c9d053003301", "name": "Temple Enforcer", "phash": "350d4d18f2c8714adb58300e8bd09e3400788ee7097d8ae7fff39ab4bca766d1", "mana": 6, "whash": "e74f62df79dd48dc08b01fb90fb88bb999b989bbd9b9d898c891c9d053003309", "card_id": "EX1_623"}, {"dhash": "889a2d2442c9bc207b79e4eec913826f1fd636cee990bf0b6f3776e47c08f11d", "average_hash": "2700236481c000c808c0099083800180238133813b98399bf8bff8dffa1ffa1f", "name": "Soot Spewer", "phash": "e7311e1e2ba2cf50fb109d642f4c068d08ec666133d39b4d0eff2ef79ba11891", "mana": 3, "whash": "2712a3e689c428c808d00dd09781018023a1b3813b98b9bff8fff8fffb1ffb1f", "card_id": "GVG_123"}, {"dhash": "b80d7c7e60f9d4f5a382a799cf099e63780de952f0a5025982a2c854f8affd6f", "average_hash": "07c00fe087c9814fc08fc08fe01fb00ff38fc00fe00ff00734001d408903f92f", "name": "Hammer of Twilight", "phash": "5bed1f60b65b26906f04c6b421800669c1f0c82c27a036316fb35edb3f9fff0d", "mana": 5, "whash": "07e00fe087c787cfc18fd08fe08f928ff38fe00ff80ff80734003d409907fbaf", "card_id": "OG_031"}, {"dhash": "eadf8c3f417ccaf4b4004bc5b3f06701ef02d91db42b78d7a4b7c36702860d38", "average_hash": "0710ef83e3a3f023e003f000f000f007e037e0338017801f403f801f000e010c", "name": "Young Dragonhawk", "phash": "b3b6b3967f4ddce5cee70ce12ee109289188db10a6706e68372273923b03decc", "mana": 1, "whash": "ef13ffa3f3a3f0a3f423f400fd44f847f977e037a137c017c03f801f010f810c", "card_id": "CS2_169"}, {"dhash": "e61fe8ff81f316862f585eb4b804200943328e651c8b93072e2bf8d601b80a7e", "average_hash": "01e00186f00ff80ff81ff83ff83fe839e830f83138313c50f818f00ff007f01f", "name": "Silvermoon Portal", "phash": "a3dfdd175ce0966abc1675e987ed782f666890b3219e87b8a917a611891a2ad0", "mana": 4, "whash": "01e00187f80ff80ff81ff83ff83fe839e831f83938313c50f818f01ff007f81f", "card_id": "KAR_077"}, {"dhash": "0ce18dda92d509037e483ebedc30b1416af3d62281c5070bffbc08f0530eafbc", "average_hash": "f70ef31bf91ff81f783f3c3b3c393c3d0c3e043f04357830700c600c4004001c", "name": "Bloodsail Cultist", "phash": "e99dcb617a62fb741e5863580e04583c63c21eb649f93fc65c6ed3b0b4274c38", "mana": 3, "whash": "f71efb1ff95ff83ff83f7c3b3c397cbd0c3f04bf0431f830700c600c4005001c", "card_id": "OG_315"}, {"dhash": "3af8ec80d1318beb34df49f0bfc07c09e31a5e25dc4b6097d037616f438c9535", "average_hash": "870f811f830118023803fc03fc07fc35e431ec19cc1b0c16000f801f801f801f", "name": "Captain's Parrot", "phash": "993fd746dc813cfe3bb9d990c74c02888cc089a55bc0b624fe497699df329f60", "mana": 2, "whash": "871f899f9387d8337823fc03fc27fc35e433ec19cc3b8c3f801f801f801f811f", "card_id": "NEW1_016"}, {"dhash": "da34997d22e25c0fb1b6e3c80199587221c4030dae22585da4b88b636197f57f", "average_hash": "176133d905cf44cef080ba81bb019a113810f800f141804140004000010683ff", "name": "Reno Jackson", "phash": "f3f1cf695d5e3699af31c631b33166640dc418e6190eca5889c1abd553c54e5c", "mana": 6, "whash": "37533fe917ef44eef481bbc1bb439bd1bb50fb41f9418041458145040526cbff", "card_id": "LOE_011"}, {"dhash": "0284ed3b60ff03cc8f3306691d84781f5268e8d8b52399c236854f0fd4989821", "average_hash": "773f51e23b83fc13ee1cf61c8e1ecc2b0e07161216300639043bfc3dd81df80f", "name": "A Light in the Darkness", "phash": "0b51f72ef19f9cd0dee66d5b6e2029b4115a2119c3359ab1cf505a035f23df42", "mana": 2, "whash": "f79f43e23b83fc13ee1cf61c8a0ecc2b1e07161216280639843ff63dd81df80f", "card_id": "OG_311"}, {"dhash": "fafffdff208d1392c939bee73d9ceb0082e1132be8521087a748499f9024f94f", "average_hash": "07000107f31fbc3ff20ce668fe7ffe7ffc5ffc4f4e7f0e72280118290018400e", "name": "Misdirection", "phash": "2b0a7b77fed59e9c3de0a6483e6646998a342d6103da1c726fc17b81db015e8c", "mana": 2, "whash": "07000107f91fbc1ff20ce668de7ffe7ffc5ffc5f4e5fae72380118690008400e", "card_id": "EX1_533"}, {"dhash": "6cf88df342cc051efb61e01e1ffc7cb8e910d385854b0883f63e3ded6a4ad316", "average_hash": "8703335c715e701cb01d00019d34fc3c7c3c3c3c0d3c00300c3cb034b016f016", "name": "Cenarius", "phash": "69377fcada602e9fa3639e109e49069929cd79b20d969684733e51ee632249e6", "mana": 9, "whash": "8713335cf15e785cf01dac83bd3cfc3cfc3c3e3c2d3c08300cbcb874b016f81e", "card_id": "EX1_573"}, {"dhash": "8a894d3f90e401c937004f90be017f0ccc3310e66286dd10ba61604ee09ffe3f", "average_hash": "0738c3e2f387f807f007f007e027c13f807f8c0f9808f809f00de0038003f00f", "name": "Bear Trap", "phash": "db77ffa67ee5276b1cbca49803f9089c49a22e410b4c3a995b807b107bc65e2a", "mana": 2, "whash": "073cc7e2f387f90ff00ff007e027c17f887f981fd88af80df80de003e00ff00f", "card_id": "AT_060"}, {"dhash": "88d90ca291450bbb35734be4b6905c6df1e2e28d918907171b36666cd4990933", "average_hash": "f717f317790f180c1828cc2dc40c201c303e383a383838181818181018130113", "name": "Silver Hand Knight", "phash": "c36573d29ac4e724792c1ac2134da42148384c923cd39fe7fe788f1df2e15e36", "mana": 5, "whash": "f717fbbf7bbf383cbc28dc2dd43de85eb03e383a38383c381c38183019130913", "card_id": "CS2_151"}, {"dhash": "3a7730fec1c34f261f9a76ec6d18f931e063c3871d7f6eeef10c2a18c67ebcdf", "average_hash": "0100c105e03f703f783ff85bf80ffc7f9c70bcf1b870dc74de27ca608071f85f", "name": "Kara Kazham!", "phash": "a1bb4add7e9c36b6746f1e47f42bb4a84fc4f098cb592c0aee0e0d4c92968e90", "mana": 5, "whash": "0100c115e03f783d783ff85bb83ffc3b1c70bc713870de70de23ca618071f85f", "card_id": "KAR_025"}, {"dhash": "7abcec616093950622f9acc8d935b16b6215dc628384265bcba6bc5502a2ed7c", "average_hash": "8783038f331f301f781eb83fb037b83b78310816481850021006600601048905", "name": "Stormforged Axe", "phash": "c93ddf395e7a1ee69f1390f2048586bca6a420ed81e1b67127696e0a67c8ee49", "mana": 2, "whash": "87c3078f375f315f781eb83fb83ff83bfab51817681a58021c0665470105e9af", "card_id": "EX1_247"}, {"dhash": "ea9fcdff309d631894e509c117460edc77e4e1191232bee67f8c6119cc329007", "average_hash": "0700e107f10ff00bf001f600c4104218fc1ffa0fee02e663f661fe61fe20fe00", "name": "Corruption", "phash": "3340687fb7b3dfee4eef3cd65e34a79318c82c2ec34491704a86e64119cb36a4", "mana": 1, "whash": "0700e107f11ff00ff421f621c6184218fc1ffe0ffe02e763f661fe61fe60fe00", "card_id": "CS2_063"}, {"dhash": "02800538f48200ff24fecf31be87e2053d0af81488eda09b4337566e0c81b81f", "average_hash": "ffff09f801c600000000e13fe1ff81ff853f803f803f001f981f981b99044007", "name": "Flash Heal", "phash": "d9f7749ad6cb9de9753ae167699c03c50246262221445b91dbe05e8279e91eb6", "mana": 1, "whash": "ffff0bf801c600000000e13fe1ff81ff85bf81bf80bf809f989f999b9984f187", "card_id": "AT_055"}, {"dhash": "fa81fdff40c703020f7c1ce0338860916f02df4421b1c063e2cfcd8f1100f83f", "average_hash": "07100140791ffc3f7c38fc31f025d020c8248c1fc00fd202361f361f6e00f00f", "name": "On the Hunt", "phash": "2b75a7ffce6cdcfe3696366232e80d9c6463324109181a964be11e29c98c77ce", "mana": 1, "whash": "07184360f99ffd3f7c38fc31f825d020c824cc17c00fd20f363fbe1ffe01f00f", "card_id": "OG_061"}, {"dhash": "8ae824d9d1b20b433486483cb3384675d88a81b51b6aa6941139676af89ff13f", "average_hash": "f79f7dbe7bbf1dbf1c331c221c442946346404220501cc001c0038007921f91f", "name": "Injured Kvaldir", "phash": "c7fffdcc7b8d3e7a7bb04693396086c0392298108b613e869b011c847722ffb3", "mana": 1, "whash": "ff9f7fbe7bbf5dbf1c371c221d643d46356405620541cd401c003c8079a0f91f", "card_id": "AT_105"}, {"dhash": "ecc88cb101c3432635785e46bc9e599de33296e528abcb56b0b5506f63908722", "average_hash": "779f3fb73bbf31bd3c3884198059c558cc3188259125f04c000c801e81080101", "name": "Acolyte of Pain", "phash": "3d99efa1df34d3c43e8538631a9208cce6c865e1d8d333333bac5c4ee524d296", "mana": 3, "whash": "7f9f7fbf3bbf35bd3c388c3981590559cd71cd65c565f54c802d809e818a0101", "card_id": "EX1_007"}, {"dhash": "0ac7ed1dd2f30804ff39bcebe185951b0b77b3ea66c49d88683980f23e05fbba", "average_hash": "e71e836163c178dd207cc271e2d1e98189a48bacda0cf80be8cb00ca120a301a", "name": "Arathi Weaponsmith", "phash": "330fbf98cf6cdfc31ed2326d29e44c34419c639873d85b82cc92fd14f43c1c83", "mana": 4, "whash": "e71f8b617bc178bd207cc671ecd1e9c18b848bacda9efc9be8cb00ca328a380b", "card_id": "EX1_398"}, {"dhash": "808ceddfe0001b623ec5848b3dbe39b87428dbf8a2c33787451e8139932d3ef2", "average_hash": "070003c3e91bfc31bc3118111e191e381e3c0e3e0e3f0c7f0c7f04ff067f027f", "name": "Ice Block", "phash": "8d4c7225f11e5f864ed05c18be010ca4a3560f9907f37965f3433bfd53ea79a0", "mana": 3, "whash": "070800e3e93bfc31bc311c191e191e391e3c0e3e0e3f0c7f2c7f06ff067f027f", "card_id": "EX1_295"}, {"dhash": "00808d1f70e7e1bc34e74f829f1439396e4adc942c090b127764fe9fd03f0400", "average_hash": "ffff07e0038003008001c00380038003b007b007900d8001c003f07fe03ff01f", "name": "Sense Demons", "phash": "99591e806f26eea2e9689b0a0399211866c52d1ccae67964d3db3add59bb3a75", "mana": 3, "whash": "ffff0ff003800380c003e003a01f800bb117b1bfb52f8107c927f3ffeafff01f", "card_id": "EX1_317"}, {"dhash": "8ce18d8b02274516bb3c66c9843259d7922a27954e6a9f5468bbdd739b85513f", "average_hash": "f71e7f7e7bf2787874f8f4f81548114941088008e108e040e4027001310b111f", "name": "Nefarian", "phash": "67e9e7a0fb047b06dfc35f483b3332361894ca980ae3d7e433959465938cd4a9", "mana": 9, "whash": "ff5f7f7e7bf37d797479f4f8155811495348c108e148e540e4027083318b113f", "card_id": "BRM_030"}, {"dhash": "eae3ccc991b30bcd37be4f50bf647ef9fcf2f0a5f1c9e1178637086f739ee53c", "average_hash": "770ef10ff90378027804781c7c1e6c3e7c3f3c3e3c3e1c3e083e001c001c001c", "name": "Violet Teacher", "phash": "c917f3c17c818f717df86cb247e360980cce4cec2c31c3993a081f46bce3fb98", "mana": 4, "whash": "f70ff90ff90b780b781c7c1e7c3e7c3e7c3f3c3e3c3e1c3e283e003c201c001c", "card_id": "NEW1_026"}, {"dhash": "808c2d33e0c08b1fcfd316271bce64b8d93027ecdc9131038f7cda772189f802", "average_hash": "77be03f6418f8c0fcc0bc612ce3c8e7c0e7c1c3c1b3f1a3748007012081ac00f", "name": "Blizzard", "phash": "0b473f3ad0accd016d5037e4834160e8025a8f1542dd0ef773f1728bfbf572cb", "mana": 6, "whash": "f7bf03f6418f8c0fcc5bce5ace3c8e7c0e7c1e3d1a3f1a3768417016181a500f", "card_id": "CS2_028"}, {"dhash": "0283f53fe0f9c3d98e311f6311ce633cc45808b11f66f889c13f045a3bfe0452", "average_hash": "070081c0f807fc07fe17fe3bfe24fe23fe603e40fe03ec07380e206ff43f6016", "name": "Flamestrike", "phash": "037e3fd7e8bfd6a9f6647d420ed9ac680b841b94cc5b23f4d3410ba15e0671a2", "mana": 7, "whash": "05008140f807fc07fe17fe1bfe24fe23fe603e40fe03ec07b85f206ff47c7016", "card_id": "CS2_032"}, {"dhash": "ca8f8d3f12e83c387bf0e5e0ca819d030206f71e9e313c4ac0b281e5030b0713", "average_hash": "3700f360f943f8cdf8d3f99ff389e1890381c380c3b3c0b7c097c0d300138211", "name": "Water Elemental", "phash": "f1a5e75ba2e8fed46e581cb42e644c3900b86c310cb15a94bafc3b51d3a1bec5", "mana": 4, "whash": "7700f940f9c3f8cff8dbe9bffbabe1890381c381c1b3c0b7c097c0d7821b8213", "card_id": "CS2_033"}, {"dhash": "6adbccb491290b53362450f8b0c07705e13acf5d5eab1c863130e372e485c53a", "average_hash": "d712ffb3fbb1b835bc38bc00dc23c423c431c031d039c011c801c800c000c100", "name": "Shieldbearer", "phash": "b377c7b1ff3139e13efc1ece4ee0c4600cb0c14086825e6c6e89e7c17d60b339", "mana": 1, "whash": "df92ffb7fbb5bcbdbc3cbc18dd63c563c571d179d179c1518811c880c180c100", "card_id": "EX1_405"}, {"dhash": "0ce60d8902104d63ba8e6d5c9a386cc3130260ecd88a2745cfb8987321862728", "average_hash": "f71ffb7ffbdf00f330f918bc783c781c600900180008100630060006000c010c", "name": "Soggoth the Slitherer", "phash": "e9a9a3891b2a5b686b5a43d22e933cbcd23c6e485ad65ac67316437eb382d394", "mana": 9, "whash": "ff5fff7ffbff1573347d3cfc7d7c795c718d0918015a11463006008e018c010c", "card_id": "OG_340"}, {"dhash": "08f40d7842a089617706ae6c0891c3af7947f6ec80c10f0a7b35fce7784fe19a", "average_hash": "e70ff30fe14fc09f981f989bb000a001809c019c1938f880f880f816f81ff00f", "name": "Mana Tide Totem", "phash": "632119cbdbcc6e1e7bd0b6208f8149d41252de8c3eb967c78cb9d9987272a6e1", "mana": 3, "whash": "e71ff34ff14fc09fd89f989bb201a08180bc019d39bcf880f882f80ef81ff01f", "card_id": "EX1_575"}, {"dhash": "fa9ffd7f60e20f841e68b8d07081f11af37686c92d03cb16079d0c300024fc9f", "average_hash": "07000307fb0f6c76647002790079807988791c383e3c3c7c347c304c0800f80f", "name": "Cleave", "phash": "ad9abf8c5ae6cff33803bce82790266021fc06b643660c4f5b529e217b8b7e16", "mana": 2, "whash": "0700c747ff8ffd77667c2279c079807988799c783e3c3c7c3c7c344c1808fc0f", "card_id": "CS2_114"}, {"dhash": "8abf7dff60d0c7609bc132027504cc7a88ccd93913332e6618cc7939d3f064eb", "average_hash": "0700c10fe31fc45f4c5f0e7f046f006f406662726270e238e238623c2023e00b", "name": "Snipe", "phash": "794c19ff73f57604dcd70e72066986bc0cc224fc23908b1d6b43fe19eb633b01", "mana": 2, "whash": "0700c34fe31fc57fce7f0e7f066f006f406762726278e278e278627c203be03f", "card_id": "EX1_609"}, {"dhash": "08f82cf201eccb18b7034c4eb83c73e184822105870a1e04f4b9c8730187952c", "average_hash": "f79fffbfebbee4bf8c3d0c380d70397079587808790078417001600001000100", "name": "Blackwing Corruptor", "phash": "675f69275b32f3c8e7a0968836810160c118d3b47be45ed66f167363338bb3c9", "mana": 5, "whash": "ff9fffbfebbfe5bf8cbd8c780d707d707b78794879407941f001e08001800120", "card_id": "BRM_034"}, {"dhash": "aa01fdcee0350f0e1e71003e879168ea99a437586f20d213bc275acf60befe3f", "average_hash": "072007c01b381cf87cf80a19e80cb064f86cf20873093c01c008602c801ef01f", "name": "Deadly Poison", "phash": "6bdbbfadbdf576d95b321e3a1bd02cce85ba8c30634d0e05a3499c2063b83186", "mana": 1, "whash": "077c07f03bf85ffc7ef83e39ec0cb468fa6c7208fb097d01c009601e801ff81f", "card_id": "CS2_074"}, {"dhash": "ccfa8da39245091f762cacd81b61b1cd6713d32c8651008af234f9e006414bb6", "average_hash": "771ab37e73de78bc38bc3ab03b3944b8e5bce1b4e3b020a00080004200020602", "name": "Thing from Below", "phash": "e534e309dea27f183bc28608a700133604d02ae5e7fb56d43cae74ede9ea9b8e", "mana": 6, "whash": "f71abb5e73ff78be7cbcfeb27fb966b9e5bdf1bce3b022a000a0004200820406", "card_id": "OG_028"}, {"dhash": "76f440c7851c1f21becc58b3b1445ec9a0025914f7e2ac5da33b4657b888613d", "average_hash": "ebbf793ff839d83bcc3b643bcc1fc81fc0078003e006e003f009301c71046138", "name": "Moat Lurker", "phash": "29cf2905ef51da337f1e5bd8ff1c16d1918713363363929069499e198d39174c", "mana": 6, "whash": "ebbf79bff83fdc7bcc3b643fcc5fc81fc807c006e00eb003f009301c7104713a", "card_id": "KAR_041"}, {"dhash": "c8f00c8301300ae734ce4b8cb7386a3190e26c95c9ca1094b33966625c8cf931", "average_hash": "771f7b9e7bb238033807bc079c2f184318003008040908001808180818007100", "name": "Doomcaller", "phash": "c349db6467947ee87f8889d31b9840e996d29c0c9bf1b6f13bad73cca520624c", "mana": 8, "whash": "ff9fffbefbb23cb3bc07bc27bd6f3d4f394931480d491c441c08181839007910", "card_id": "OG_255"}, {"dhash": "3ac0ec9f117f4af8b4c151c1a7867d2df9b2e25dc5290bc73fb6326cf9990533", "average_hash": "07900fb0e3a5e027e007d007c40f846f843d003e003c0038003c003001330111", "name": "Mini-Mage", "phash": "39fd5302f6056cb19e9b0ee9eec480a4115831ec63c21b91afb1db433e21cfbd", "mana": 4, "whash": "0f980fb8fbb7e5a7e4a7f547c5cfcd7f877d017e057c0178007c04b001b30331", "card_id": "GVG_109"}, {"dhash": "28dccd3703d345c4fa79a3e34e9f913e02773ece7c9485096233c4e28c4f3bbe", "average_hash": "d7177b26fb47705d60e6c3cd83dd80c98d809d859986b88238cb104390079a04", "name": "Shadowcaster", "phash": "dba3fb8cebe4c7251e133c693b86049c60bc99b10b9b334373c7db98b2304c36", "mana": 5, "whash": "f7177b47fb47707d60f6c3df829d82c98e809d851f86b88a38cb10439a0f9c0e", "card_id": "OG_291"}, {"dhash": "1cf80cf20180cb17b6094c4bb8b07375c4ee018d3d8adf17b337046d669ecd3c", "average_hash": "e70feb9f411e003e303f083104306033303638003800d839981dc81cc01ec91e", "name": "Piloted Sky Golem", "phash": "e9a5f99c7946d34373c4311b9299c690b1d93c3c0e332de5d9cc9399b323cc62", "mana": 6, "whash": "e71ffb9fc3bf003e303f2c3d8c397073703e38003c01fc399c3dc83cc81ec91e", "card_id": "GVG_105"}, {"dhash": "98f90cc46100ca07b57d56dfa9385365a6ca0c154b2af6c49c39136626909d2f", "average_hash": "ffb7ffbe87bf04f804d8055837d033d237707568734861ce4c5e4db809a00320", "name": "Faceless Manipulator", "phash": "6f7cff211b0673c4d721432473a0066cc82496c2198cd9b83de99b2ffdb236d3", "mana": 5, "whash": "fff7ffbe03bf07f804d8055837d031d2375037e073c871ce4c5e4db80da00b20", "card_id": "EX1_564"}, {"dhash": "fc87fd3f20ff01fe87000c601ce037506a10f1a4ccc193032e07c80df01cfcbf", "average_hash": "0700f301ff01fc03fe077e003c10dc096804ac002c003809f80120044003f00f", "name": "Thistle Tea", "phash": "635d3360fc603c1b1c868e83cc607c3419c81c7d8e39c69c638e3363cbe3bffc", "mana": 6, "whash": "0700ff81ff03ff07fe1ffe04fe10fc1dec17ac0eec037c09f809600c6017f00f", "card_id": "OG_073"}, {"dhash": "127b29f6a2803921d272a4cc5939b3f364e4d1caa3053f1b1e32ece4195df2ff", "average_hash": "c507c35fe01ff01d72989810f8903830783078b078303890389030103008a07f", "name": "Dark Peddler", "phash": "e3a539ee797ac3b69f569661b775264981dc46dc319b02d219bb4eb01904ee30", "mana": 2, "whash": "cd4fc37fe2fff09df298b8b9faf078f07b307ab078907890389030103208e0ff", "card_id": "LOE_023"}, {"dhash": "cafc84fd41a28b0034274b5ea4b443c98fe2f8c5200b4b17f4b4f961028e9534", "average_hash": "3707f19ff99fe81fc8083c283c007c003c071c071c060c018007c007c003e101", "name": "Core Hound", "phash": "db3fcfed79b1dbe4d7e866d026d398c23698360649309968cb0064c89e239ee3", "mana": 7, "whash": "b717fbbffbbfe83fcc383c283c087c407c47bc079c07ec07e007c007e103e115", "card_id": "CS2_201"}, {"dhash": "ccf80cc291840b3c36f34942b6c4798df19283259ce90092833466611c81f92f", "average_hash": "f71ffb9ff91bf838f831fc2dfc3c4c3c0c386c3004320c301820100010003100", "name": "Kodorider", "phash": "23577325d221ddb479585b92339012b22d8dcce043cb971bb9f0b2d63d8d3ce9", "mana": 6, "whash": "f71ffbbffbbff8b8fc39fc2dfc7cec7ccc7cec380c3a0c301820180019003100", "card_id": "AT_099"}, {"dhash": "e8d2ec6821934a64b6a054cbb17647c9961a8d25626bb4d7c1b63f6ffc9ce13d", "average_hash": "879f8bad33af301f0017800180016c43e440646040200030003cf01cf91ef91f", "name": "Flying Machine", "phash": "7bdd9e1049913e46738e1ca1bea0026688ace3cc9cd967ec6d69c6cee32cdcb0", "mana": 3, "whash": "8f9fafaf33af34bf84178011a5436d43ed416d6041600570043cf0bcf9bdf91f", "card_id": "GVG_084"}, {"dhash": "88eccce211840b2034fc59e9be7261e5c4c2a9859fcb3d129b30f673648ed93c", "average_hash": "f719f18efb0ef8018800e400f03c70227c007c20fc30f821f800f80cf81cf11c", "name": "Mad Bomber", "phash": "e377ffc8798397d56df830f83c9c060b92c129c16319f3c8fb08f684d9006e63", "mana": 2, "whash": "f71dfb8efb8ff8338c20e400f43cf4227c027c22fc33fc21f800f81cf81cf11e", "card_id": "EX1_082"}, {"dhash": "48f88d4013c10c13796e24984260d92027f3f9c0cf859b0b373e2ce44041913c", "average_hash": "f70ff90fbb0d180c180438017c083c801c103c183c981c3938bdb01c1004000d", "name": "Darkshire Librarian", "phash": "c3b1f3e2f9706b7d7e6c739c0bc018a40c81424e63989eccb731d7013f23f69c", "mana": 2, "whash": "f71ffb0fbb4f187d18847e917c887c883c903e983cbc3c3d38fdb03d781c000c", "card_id": "OG_109"}, {"dhash": "1a9e2d7c40f878f073e0e7c48f8917132f067c4cf810c009f033c0e58089031c", "average_hash": "e307e1c7c1c7c087c087c187c187c187c187e187f18ff08fe087e087c1038301", "name": "Lightwell", "phash": "d98711eeaefae17ebe4301a46fd010a400f06b0508f4be05ff197b05fe0b7b55", "mana": 2, "whash": "e707e1c7e1c7e0c7e087c387e187e187e187f187f18ff08ff08fe087c3038301", "card_id": "EX1_341"}, {"dhash": "4cf08dc382950bad7e1e2cbcd8d8bc317162c0e60f85bf08773b6ee4cc41032e", "average_hash": "f70ff31ff9187838783c7c387c3ebc3c9c381c381c1018001018900090000000", "name": "Carrion Grub", "phash": "e3c36b8a7aa03f391f72d9a08e0c89b909b9d3446cc472cc376f77637ca4c4b0", "mana": 3, "whash": "f71ffb5ffb18f838787e7e3e7cbefcbc9c3d1db83cb03c109818d001b0020002", "card_id": "OG_325"}, {"dhash": "02c0fd7fe0e75f029c5430bd241c49d89290e7730fe09d2070c78d0f6b16d82c", "average_hash": "671e01e0f1bffc7e3e7c3e783e781e784e78c679e600fc01e208260c1c269826", "name": "Holy Light", "phash": "275aa5abf8763aad8dcf4e89b2612db023b9037639980b77dbc599015b895e42", "mana": 2, "whash": "779e01f0f5bffc7e3e7c3e783e781e784e78ce79e600f6016208260c3c2c1826", "card_id": "CS2_089"}, {"dhash": "8addadf012c749b0d701a6ce0c09d1173037c1ecbe51448a8a3065e2d94727bd", "average_hash": "c717e17ff3dff09fe0bfaa9fb31d12b4099009b00fbc089400800040200b0609", "name": "Eternal Sentinel", "phash": "598dfba49bf89f369ec19aa00fd0844c40c6c3d066ba4f056e26fdc1df61fb08", "mana": 2, "whash": "e717eb5ff3fff1bff4bf2e9df79d529519ba0db00fbc0e9600800040008b4409", "card_id": "OG_026"}, {"dhash": "8287ad7f50e0918037034e8e983c315962b2c6c48089f713ce2418cf219ffe3e", "average_hash": "77bea1e7f38ff81ff81fdc3f1c3f3c3d3c391819180c380e700660020000700e", "name": "Unstable Portal", "phash": "c9dd39b9f662f662a69d86e60cc32cad49944e981b614b647b077b805ba17e98", "mana": 2, "whash": "ffffa3e7f39ff81ff81fdc3f3c3f3c3f3c3d1819180c300e700660024000f00e", "card_id": "GVG_003"}, {"dhash": "4affd8b081470b9e36604cc2bb047d7df2f2e0c5ce8b9b1767378c6e989b7130", "average_hash": "f70ff71ffa07f804f8202430043104382c383c381c383c1fb81b381f301ff00f", "name": "Frost Giant", "phash": "833d33b2db80cb693c6c18bc32fba63c739e4cdbe2649e2c7cc4de404b803fc2", "mana": 10, "whash": "f71fff1ffe1ff804f820ac300c3104382c383c383c38bc1ff81f381f301ff01f", "card_id": "AT_120"}, {"dhash": "fc8ff93b04e6019e1ffc1df03fe37b04b40a6048cc3033436006864c009cf83f", "average_hash": "0738f7c3ff87fc01fc01fc07f80ffc1bf6173b18131b861f040710030000f00f", "name": "DOOM!", "phash": "6345a708de521ea03d2a3cf62cd38cbf097e8c5f4bf766bfc394060dc3831ea0", "mana": 10, "whash": "073cf7e3ff87fd01fc01fc07f81ffe1bf61f7f181a1b921f060730030000f00f", "card_id": "OG_239"}, {"dhash": "0acc8c93019e0a2cb1b04cc9b9126205b89a07656bcade160437b96c648a9934", "average_hash": "f713f383f38070096001a021a02100000410ec007c00fc097c187816d8057905", "name": "Piloted Shredder", "phash": "635fbf68fb85434e9cb29d3193d818ae6cc843c2391c29f3cea452067b22ecb5", "mana": 4, "whash": "ff13ffa3fbb074b9e40ba021a171c1400558ec417d08fc697c3cf815f915f90d", "card_id": "GVG_096"}, {"dhash": "e8c1cc3f91f60bbc376049c6b6cc7819e432cbc596493f923436c16df08b6130", "average_hash": "170073007902f80ff82dfc25dc38cc381c301c309c30ec31e831f816e007e003", "name": "Murloc Warleader", "phash": "a317cfc3fa282e258ee436961cd3cdb23919c96626679bc9e672999c698e188c", "mana": 3, "whash": "17107b80f98af82ffc3dfc2ddc3ddc381c381c381c30ec31e839f816f817f113", "card_id": "EX1_507"}, {"dhash": "fcc76d0c025e84b4b87ce7d98bb25de783ce7f9dff2b83478c36786de09b0130", "average_hash": "8700e3607940e8406040709030083006f003f00ff01ff03ff03ff03fe01fe10f", "name": "Nozdormu", "phash": "f3b8b60a3e0f3b639e616d90ef391a850cc6c9634e735699c785cec1c630c578", "mana": 9, "whash": "8f10c761fbc0e84070607091305c705ef00ff00ff01ff03ff03ff03ff11ff11f", "card_id": "EX1_560"}, {"dhash": "c8ff2cf041888b39374b4e36bccc7829e15296a5e96b1b976d38c377448cb93f", "average_hash": "f71ff39fe19fc81dc81bf81ffc3cbc388c308039003c0819c803c80e9005210f", "name": "Stranglethorn Tiger", "phash": "295f7b40db20be30bf3043c669c892c396646ce35a928ec36d245bf39b8e6ea6", "mana": 5, "whash": "f71ffb9fe39fc83dc83bfc3f7c3fbc7888308839003c281bc803c80f9017210f", "card_id": "EX1_028"}, {"dhash": "ec998d3702ec4c80f97ce3c107bccc581be17c86c118ba015cbbb0f6434d971e", "average_hash": "3706f367f147f04770c0f0c0bf077c817c83388339002100408a805e800e801e", "name": "Volcanic Lumberer", "phash": "d32af3b27972ff4c9f853c266e3084c906cd2e3edbf09a8126bb44b9990cda50", "mana": 9, "whash": "f716fb67f347f047f0c1fec1bf07fe817e833e833d88610a40cac05e801f801e", "card_id": "BRM_009"}, {"dhash": "12182970c6e6dcd8b1ef0396092c93d906b1f98ed7412e08593670e0e2496b32", "average_hash": "e16749c700c744c28c804f812f831cc15fc07ff07ff8fc98f8b0e8804810fe32", "name": "Enchanted Raven", "phash": "170fd9baf9f2e97e9f9e939d6f6474d601f38839e1b61286699984d410231b20", "mana": 1, "whash": "e967c94740c744c28e804f813fc11ec17fc07ff07ff8fc98fcb0e8a06a11fc32", "card_id": "KAR_300"}, {"dhash": "c09f9dfc30334f4ddcd4b2fb6186c31c0672fce5f90f829b00611f0e9cf7f8ce", "average_hash": "073003cfb39f407fd0e2f07be0cbc0430c031c017c38281808080808386ef84f", "name": "Forbidden Ritual", "phash": "896bbde04d164e9513993608676286461864a5c62b8dcd786b699d73739ddb7a", "mana": 0, "whash": "073883ffb3ff51ffd0fbf0fbf8dbcc430c031c117c38383c08180828386ef84f", "card_id": "OG_114"}, {"dhash": "88817d3cf0f383ce3f093b7c7e82f8a4e15b8387204f443c1339fc7561f3fced", "average_hash": "073c07e01387391f7c7e7c7c347c107800f800f000e000e000c080f0c0ecf0ff", "name": "Animal Companion", "phash": "e5fa4b45fb06b6aa2df136a0031a0316cb69c950cbf15cd81be51e8b91e67978", "mana": 3, "whash": "ffff47f1178f3b3ffc7f7c7c367c127800fc00f400e000e000e0c0f0c0ecf0ff", "card_id": "NEW1_031"}, {"dhash": "60c0fd9f60ff03c61f1b1e603e10f040e0b9c1e383860f0d765b48b4b320f07f", "average_hash": "071823c07913fc3bfe7e7e7c9e7f327f007e0c783c58f8507850205000400029", "name": "Avenging Wrath", "phash": "a5280b16f259562dbc525f8007432cccf1d88b6909d7731be3f44f995b3679e1", "mana": 6, "whash": "671c32f0f9b3fc7ffe7e7e7e9e7f367f007e0c7c3c5878507850207000400069", "card_id": "EX1_384"}, {"dhash": "ecf8cc8701870b1834f34e26b90c7239e472c1e50f8b2e16fc34fb6de69f4d3e", "average_hash": "170f339efb11f8184808d8335c361c381c381c387c30fc206c20e836c81fd91f", "name": "Goblin Sapper", "phash": "a307fe315b08a650cda19a94cbcb0cce64dbb3ac63c2bbed93f33c99b6824648", "mana": 3, "whash": "371fbb9ebb91f838d828dc3f5c361c381c381c387c30fc306c20e836e81fd91f", "card_id": "GVG_095"}, {"dhash": "c33a22f7ccc439097332c68c883d11fb40f6c5ec0ec11918aef11cc08187e37f", "average_hash": "fbe0f1c4e0dcd8de1b9c139803992b805ba07bb01ba0088060c071c00700073c", "name": "Onyx Bishop", "phash": "775d412771bad23eff1af2f9af23f7e609c2864239ed0cc3b38b0c10c3207c14", "mana": 5, "whash": "f9e1f1ece0dedcde9bde1b9d13993b885ba07fb03fb01cc060c071c00700877d", "card_id": "KAR_204"}, {"dhash": "0a8c1d3b70e4434f969dce330bc8fe80f90907a4f80bed171a2a2477c8def2bf", "average_hash": "073807c2038684066002f002f806f836f07fe40b300ef00ef006f800b019f01f", "name": "Heroic Strike", "phash": "73665fccde399f7325cb23a6635824d809f30e9313736938e5885e90db463d22", "mana": 2, "whash": "073c07e30386c7066003f003f806fc3efc7ffc0bf80ff91ff807f812b01df01f", "card_id": "CS2_105"}, {"dhash": "b8ce6d3fc2f8c8e979bba34e62258c991c323b6742449b097fb09fe17c47e12f", "average_hash": "87104362c341c08f48cedac73acf7dc44d064d820f80088070c07880f800fa02", "name": "Ram Wrangler", "phash": "538e1d8ff3e03e3d0706c69089a044660cf266c9d9d93b073b67372767ece4b8", "mana": 5, "whash": "cf104f63c3e3c0af4ccfdecfbec7ffc4cd86cf822f800c8070e07880fa80fe06", "card_id": "AT_010"}, {"dhash": "028c0d7af0e00b85bf207e03fc33fa43801f807fdbb7a6cf843e197a23e4fec9", "average_hash": "173c01e6039f007fc07fc07f80ff80ffe07f007f407f407f007e007c00fc887c", "name": "Frostbolt", "phash": "e9ceda955673d6c4d77b65f623e72099039c0d228e513a917b803ec1db2c5b18", "mana": 2, "whash": "779e01e6019f007f807fc07f80ffc0ffa07f007f407f407f407e00fe00fc88fe", "card_id": "CS2_024"}, {"dhash": "4cf6ccd811b38b46371e4f78ba987421f962f2c5e489d3170c37146e0898f139", "average_hash": "f30ff30ff91f781b781bf81df81c7836b83fb83f383d383e781e381838181008", "name": "Target Dummy", "phash": "e915db407e4959901eb263926e9a2408180cb6843b63cf3cefb3d686fbb63e4b", "mana": 0, "whash": "f30ff00ff91f781bf81bf81df83c7837b83db83f383d383ef81e381828181008", "card_id": "GVG_093"}, {"dhash": "2aee0dd8d230a94056c2acac1f19f9e372c7c19c8531070b0e3636e66c4cdbb9", "average_hash": "f717cb57c3d3e893c09102901b38389c31bc31bc33bc30bc309c105c10180018", "name": "Flamewreathed Faceless", "phash": "e9c1bb2e9beefb94efe346e32ed022f40170238859b214151ca85c632ee3aee3", "mana": 4, "whash": "f71ffb5fcbdbe8b3c09302901bb83abc39bc31bc73bc78bc30bc105c18181c18", "card_id": "OG_024"}, {"dhash": "828f5d7e60c00f243cd059ee961068c7e7dac8819f6576c2ff9cfc4300b1fc8f", "average_hash": "ffffc3eff18ffc7d4071c03cd81d70687c723e7066306420740074020000f007", "name": "Mirror Image", "phash": "b3d639cb5bfb97bc5b675a6ba4b4a62ca4cc24d0439b1a165bc456405b271ba0", "mana": 1, "whash": "ffffc3eff1cffc7d4071c038f81df068fe733e70767164207c0074032000f007", "card_id": "CS2_027"}, {"dhash": "68d98d2dc34a8ef3788c303b46e0d1863edb6de1992452498fbbe4e539486338", "average_hash": "17171f66836d89e5bce336c27fc03ac005d06fd833c000c430d830de7092fe18", "name": "Void Terror", "phash": "c7b89f9ab9d77ce12e04c7600640996c1ce3d39c23eb3ebcd9b0a3cd190e03d9", "mana": 3, "whash": "1f171f76836dcfe58ce336c07fc03ec007d04fd837d400e470f830de70b2fe18", "card_id": "EX1_304"}, {"dhash": "dc9f3c7f00f6158ea3182e70f8c5e113cf6c9c8373278c5f70accf58229ba17e", "average_hash": "0780630073067007f01ce838e030c8719077703ff03ec03f003f603e001e001f", "name": "King's Defender", "phash": "e9d476885e2d9d601f03341c63008cb41cc122a709d83e35cf6c8ef97ffafe65", "mana": 3, "whash": "478067807306710f701ef83cf030e871d277f07ff03fe03f003fe03f381ee91f", "card_id": "AT_065"}, {"dhash": "68d70cec41f08b19370350c6a01c4cf9d1629fe50fc91a160c347f68c091c13f", "average_hash": "c797ffb9ebbdc8bfcc3fdc085c403c401c601c200c202c200400f821f9001100", "name": "Booty Bay Bodyguard", "phash": "335f79817b01b7d4d6a112a00d8409e3cc60dc361ce69ee6bf231c6faf933cb2", "mana": 5, "whash": "cf9fffbdfbbddcbfdc3fdc0add403d401d601d600d603d600c20f8a1f9807100", "card_id": "CS2_187"}, {"dhash": "f8cf8dbb42d0bd057b23c6ee9d5960930126026d7e92dcc9a3b207e468089f31", "average_hash": "2710bb71fbff78ff7ed81fd89f7adb209b000b000fc003d800f009f083908711", "name": "Rhonin", "phash": "3f07e34eb9b8f6d4be7285751e18460f61d07e4e8bcc7aa145fc62b077084326", "mana": 8, "whash": "2f10ff71fbf77c7f7ed81fd8df7adbe097100f000fc003f000f409f083900731", "card_id": "AT_009"}, {"dhash": "aacd6cc7d1980b7036364c8cb8f47381e60210155c0a975464b3c0694d96373c", "average_hash": "f717fb9ffb9bf839f83ffc3ffc31fc31f8207008240140034001000000000118", "name": "Refreshment Vendor", "phash": "f3edf3077c055ee8773492f00ce10ca2a18c8638268933e33b937b9b7d4a9e34", "mana": 4, "whash": "ff1fffbffbbbf8bbfc3ffc3ffc71fc71fc727408a407c00b4001000001000118", "card_id": "AT_111"}, {"dhash": "0280ed3fe0ff11fe19f883e5fe60dc01ff93bc27e30f087e2098dc7081e17ccf", "average_hash": "ffbe01e03380fc03f60fe00ff04ff0f7c07fe07fc0ff80ff00fb44730043004f", "name": "Flamecannon", "phash": "f9ea5711defdad3ef99e37e06942096c830903f64bb0d89d63155a04db203ec8", "mana": 2, "whash": "f7ff03e01380fc03f60fe00ff04ff057c07fe0ffc0ff80ff80fb40f20043004f", "card_id": "GVG_001"}, {"dhash": "6ac624bcc1688ad7b4af5377a59c5831b7c269255fcaa804c638e46b9891312d", "average_hash": "97948fb10ba00da40466044e0d469d41bd59fd48fd0ff8463004300331033100", "name": "Worgen Infiltrator", "phash": "d73f1d97bcd473e1a7e99f20793649c8c6921b036c32cbe0b981360baf244ea2", "mana": 1, "whash": "9f948fb10bb00fa40ce60d4e8dc69dd1bf59fd48fd4ffd463c4434a339813320", "card_id": "EX1_010"}, {"dhash": "5ada8c3901e7cadeb57b5127bfc84017ab3e51dda42bcad61f39f46ca191633b", "average_hash": "b794bfb533a7203ee00fcc07cc1ec04dc005d01430250008e018f01071002100", "name": "Sen'jin Shieldmasta", "phash": "73f3df36e63cff91bf8133670f9233a42c094cd8276416ccb3c433039cc04d39", "mana": 4, "whash": "af94bfb53baf64bee4bfdc07cd5fc14dc155f15531650568e818f0b071803120", "card_id": "CS2_179"}, {"dhash": "c201fd7f60fec3491f07b6e0691074b8e95cd6c1accb0de60fcedb143739fe77", "average_hash": "073003c00304840fdc0778181e1f043cb8311c300c3c00344e304e124e32fe3f", "name": "Crackle", "phash": "2fb4d26b1f4cfcf3af906ea4a1730389995208f48bd8e205e3496ea65b22bf91", "mana": 2, "whash": "0f3003c02304850ffe07fe199e1f5c3dfc391e350c3c8c344e307e136e36fe7f", "card_id": "GVG_038"}, {"dhash": "caf695c902b37d73f2c0cc9c98116127075e0ebd3872f3488cb199e33387e71c", "average_hash": "771dbffd3fd9bcfabff31fb9033903a10303830103c003c000c001c007800720", "name": "Confessor Paletress", "phash": "ffb9a9f947fa525aff5a45962f9093b64064762628a66a265f061200f78c5754", "mana": 7, "whash": "7f5dbffdbff9bffabff39ffd0ff903f10343830103c003c000c005c007c00760", "card_id": "AT_018"}, {"dhash": "601efdf9c1ce97016f32be44649be026814424f1d9ca1305268e7f7ecef8fcef", "average_hash": "073003ce3b3ffc7f987eda7e827d127c0260007000780038001800300030f01f", "name": "Ancestral Healing", "phash": "ad0b690252fd96fcd66036912904a4660312ad9d29c8a7da4b66ec35ebb57edd", "mana": 0, "whash": "0f7803df7bfffdffd87ffafee2fd327c0268027000780c3800380c380038f83f", "card_id": "CS2_041"}, {"dhash": "e880fd3fa01f00fe04fc09f817e02dc07f98ed40dab1fc031027004c0098fc3f", "average_hash": "f77f7fe0ff80ff00fe00fe00ff00ff01fe01ff00ff00ff03b80100000000f00f", "name": "Starfall", "phash": "7354c7b09cb33dc7798c9c180b202c640b500e3443670efd4bd31ed75bcbfe9a", "mana": 5, "whash": "ffff7ff0ff88ff00fe00fe00ff01ff01ff01ff01ff01ff03f80100000000f00f", "card_id": "NEW1_007"}, {"dhash": "aab4454312be3d63727e80fcc8c99f933c27714ce4908a8b1f362ce4c08c8319", "average_hash": "e70ff14579c0f8d038803b80d980d983d98ec19c039f00bef89ff898701d320d", "name": "Guardian of Kings", "phash": "638bb7eef9f879d61bcb33371c906169193438a533313a12df41630cec60c6cd", "mana": 7, "whash": "e70ff14d79c078d038c0bb80dd80dd83d98ec19c039f00bff8bff0d9f01d330d", "card_id": "CS2_088"}, {"dhash": "e67f09da06c43b86777e8c641008e61b6cf58aee15c17a0aa234dfedac50192f", "average_hash": "f341f15bf83a78be38bd7fbedd620ce00ab428b02920382020a178b038001000", "name": "Menagerie Warden", "phash": "a117690b72e9b63dbe6793c2ff936d910c99964b9836da866d9616cc05619e60", "mana": 6, "whash": "fb61f97bf87a78bf3cbd7fbfde634ce28af628b0292038a028a178b138203028", "card_id": "KAR_065"}, {"dhash": "6adb852f039b4ce6f984b3494e73e7e60487681903743e09ddbb32e2c3448733", "average_hash": "d730bf74ff63f4ebfcefb7c637cc33e63fc223d063c069c040e000c000800621", "name": "Blood Imp", "phash": "f7b8ebf23ff77eee8e644de26e800c2003c432493cc99a0c5b0a7d86ab8993c9", "mana": 1, "whash": "ff30bf74fb61f7ebfced3fce37ec72ee3fc223d063c069c048e004c000a00621", "card_id": "CS2_059"}, {"dhash": "caf18cc7911f0a85343c515cba3877e1e7c2ce2598c9309607347e689c91313f", "average_hash": "f71e7b9c7b8cf8077802fc003c303c30fc31fc31fc3018203800380038013100", "name": "Dread Corsair", "phash": "e39d63327cc83df03ffdc6e278ec133808c690c423b413923e23e662ff88f634", "mana": 4, "whash": "f71ffbbe7bbdf887fc07fc003c707c70fc73fc31fc313c203c20380039013902", "card_id": "NEW1_022"}, {"dhash": "12b6384d61b35a3cbbc2560d28db4d36b36c669dcea299dc00b94162d396e57f", "average_hash": "cfadc3a9e5bb645be45bf65fd358925d924db0488141804100000000010083ff", "name": "Eerie Statue", "phash": "39f973dc3d1e76db7f931ed799a6c6f121aa669a4524c2f019e01a3c118426c3", "mana": 4, "whash": "cfbdc7afe6bb747be4dbd75fd35d935db34db348814980410580000005218bff", "card_id": "LOE_107"}, {"dhash": "0af26dcc92032917776fecb41868b4e97093e1644381ae0bdc3638ec1048abb4", "average_hash": "e70423481bd838b038b83c985f3c6c3c6cbc7dbc7d2878b470bc681c00101000", "name": "Draenei Totemcarver", "phash": "6795bf29feb07f5e3b7a298687d1607c40f402864fe654a449281ca5a729bbe2", "mana": 4, "whash": "e7143949195838b838b83cb95ebd6cbc7dbe7dbc7dac78bc58bc787c10141004", "card_id": "AT_047"}, {"dhash": "1adb6c2ed1cc8a1f37324cecb8b879e19b227705c6ca7c14a9395f63cc8f313f", "average_hash": "e702c184cb85c809f838fc38fc18fc19fc0dfc0cdc04dc07e0037801f803f11f", "name": "Mana Wraith", "phash": "63ff3f16dcc45be6ebb0a62c25246328218c0749de48df487f08cf01e703ff38", "mana": 2, "whash": "e712c984cb87c809fc38fc38fc18fc59fc1dfc0dfc05dc07e003f801f80bf11f", "card_id": "EX1_616"}, {"dhash": "0afeace111800b48b7764cbdb0485b91b522dfc5f28be50781360e6c589cb13f", "average_hash": "e717a38ff389301810188019d008f8105808d006183e183e183e38143811511f", "name": "Frigid Snobold", "phash": "ebe53f48fb19db9633891e9719e286b02cecc66066626699634234a5764a6ece", "mana": 4, "whash": "e71fab9ff38f303a1018981dfc0af84af818d806583e983e183eb81e381ff91f", "card_id": "AT_093"}, {"dhash": "02808d3f34e1018627b05e6f9f5c34a173f283263d4df298cc371c6ec190f02f", "average_hash": "ffff09f08187e00fe01fd01fd93fdd3fc13ff9af6887440fe01ff01830008003", "name": "Power Word: Shield", "phash": "795d9da2f6e9179ea7f32df5830e61b14bccc38461315a6559805278796b9c28", "mana": 1, "whash": "ffff0bf08187e00fe01fc01fd93fd93fc93ff9af6887c00fe01ff018b0008003", "card_id": "CS2_004"}, {"dhash": "dad8a46f41c50ab1b50e531da0f07f61fb42e0458b6b16d72d32f3628c85213b", "average_hash": "0716259681adc00c100f38067c00bc2fbc3f1c3e403b401d4408980bb8031901", "name": "Clockwork Gnome", "phash": "cbdb5fb2fee53dcd8731d6ac7e84896308909341266e23c19991dc189e07cee4", "mana": 1, "whash": "0f1e2fbebbadd48d140f3c063d40bd6fbd7f9c3e413b407d641c980bf9031901", "card_id": "GVG_082"}, {"dhash": "78ce8d7912f63c007b4b86d48891520b35f638ed639286083931f4e6b1816712", "average_hash": "e701f363fbc9f8df48db3bd99b1103000b0003044bc45bc0f0c061c043006700", "name": "Bolvar Fordragon", "phash": "7789ab0933f4d7963e1234312f840c2c00665926cbf2638b779f765ae62bd6c5", "mana": 5, "whash": "ef11ff63fbe9fcdffedb3fd99f510bc00f00070c4b847bc0f0c0e1c043806700", "card_id": "GVG_063"}, {"dhash": "f88ff9ff00dd6ff79ffc39f07880e5828b0c96394662bcc7f98d791cf23a3c7d", "average_hash": "07000300f10df07dfe7ffe7ffc77f065c005e041e621fe30fe387c3de83ff83f", "name": "Infest", "phash": "2390fea9cc6d3e645e820fd41f600c980fc1b09fe57a0bb74bc94b68e3945be3", "mana": 3, "whash": "07000200f10cf07dfc7ffe7ffc77e025c005e040e620fe30fe3a7c3dc81ff81f", "card_id": "OG_045"}, {"dhash": "01c08f3fe0ce0983bd145c88d130a4651b9a3516c32d98db70f5856102963c0d", "average_hash": "ffff0bf8e1dff83f707c30f031a43085218fe08fe00de03f801f8087010f810f", "name": "Silence", "phash": "f959a7a01d07390cd6589db165c1436269802bd21b673b9d73f8d8695bd4dc72", "mana": 0, "whash": "ffff0bf0e1dffc7f707c30f031a631c5218be18be089e03f809f8087018d818f", "card_id": "EX1_332"}, {"dhash": "08d6aca801534a34b5c3560cbd20721dc7f2df05e60b0c1730b6c1688b910f3f", "average_hash": "f715a3bdabb9b00fe003cc1bbc3fe0398031f83dfc1cc8358033800200030102", "name": "Big Game Hunter", "phash": "b97ddb99de982793ef809c2443261262a49062c26f629fe76e6735ccb3921663", "mana": 5, "whash": "ff15afbfabbbb08fe403ec1bfd7fe07b8075f83ff91ce8358033800201030103", "card_id": "EX1_005"}, {"dhash": "7e3f88fc00fb15f027002c02f004e179cbe0c6c339c7e71f5f2f945e309b4176", "average_hash": "c707e30ff21ff81ff81fe837e0231021183108310037d83ef83ff81ff01fe00e", "name": "Truesilver Champion", "phash": "e9351b187b249ef03e5896761cf6a3bd56f784a426e92aa596b09c788cb87b20", "mana": 4, "whash": "c707e30ff01ff81ff81fe83fe823182118310031803fd83ef83ff81ff01ff00e", "card_id": "CS2_097"}, {"dhash": "2cfb2dd042e0c900f681ac031743d8b5374f4e0cb011214b06b48de03341e3be", "average_hash": "c70fc31fe1dfe09fe03fe0a7e21ff01eb08fd183013200b36090600000000000", "name": "Fire Elemental", "phash": "792419232e8a9f634e6366089e5492d208f16e0c73bedd95c6f636f79b29caa8", "mana": 6, "whash": "c71fcb5fe3dfe0bfe0bfe2b7e29ff29fb19fd98783b7a0bb6090604020000000", "card_id": "CS2_042"}, {"dhash": "8283fd7fe0e083805d369eef3e1e086ce0b0f0a3e147830f0e988d6818dff8a7", "average_hash": "f73fc1e1fd8ffc1ffe4c3e7d9e3d1e1c2e0c2c7c247c207c20780820003c201e", "name": "Humility", "phash": "6d5e33ebf0fff5ac1ef7aee12dc08e964b78097313510b9c534c4830538359a0", "mana": 1, "whash": "f79fc3e3fd8ffc3f7e7c3e7dde3d1e1c2e0c2c7c247c227c20780820003c000e", "card_id": "EX1_360"}, {"dhash": "03c08d07f406013b70eac7949d693f9378029384646dc99a3b25fe4a4097da3f", "average_hash": "ffff0ffe03de80184009618f018fc18fc1b48185c184a188b983f9838186e18f", "name": "Shadow Madness", "phash": "7957ad8867202532d9d6d3cd5bd8a332234b32e2db89389473e158f9dbac1c62", "mana": 4, "whash": "ffff0ffe03de83184009418f118fc18fc1b5818fa185e589fd83f9838186f18f", "card_id": "EX1_334"}, {"dhash": "889959f7a0cccf18df673e1d60d1c9b2bf4550cb0006955f7f3bfcb881672ec0", "average_hash": "07004300411e007c00781e783e601248927f987c104800f8007fe07fe0ffe47f", "name": "Kill Command", "phash": "adea7e0579defe25d31086680649390524b9526a875229b1d3f35a0d7d46dbf2", "mana": 3, "whash": "07004306411e407c00783e783e683648b27fda7f10e810f8b0fff0ffe0fff4ff", "card_id": "EX1_539"}, {"dhash": "f8dfccf311a68b4c36994922a2c44f2996726b85fc4a3097c3370f6f788cfd38", "average_hash": "c707fb0ff91ff83ff8266c06ac018c0304009c10841f0437181f081c001cd11c", "name": "Light's Champion", "phash": "4b3ef34469819de136639e9a363426269cc63c3927a1fcec6d8e466ba6a3ccd0", "mana": 3, "whash": "e717fb9ff99ff83ffc36ec06bc058c434402dc11ac1f04371c3f283c201cf11c", "card_id": "AT_106"}, {"dhash": "e891cf6f80fd38e07392e5248b111c633f867dcc9090a0894333efe7818f031c", "average_hash": "37c453e0f1c0f0c7d8c7c386038603803380139e038f018f809fe1cf830fc30f", "name": "Darkshire Alchemist", "phash": "dde7e7e21bdaa459fe1804309b04c1b40018eb3131747b347f1f1e31ff8c1ad5", "mana": 5, "whash": "37c553e5f1c0f8cfd0cfdb8603860380339c939f038f819f80dfe1cf830fc30f", "card_id": "OG_234"}, {"dhash": "58f3ace441c8cb90340353bca7604b59a8e2e0659dc918960130f470fb87e72f", "average_hash": "c71feb9fe3bff03ffc1f7c0c7c4e7c433c219c305c301c200c000000c103e107", "name": "Ogre Brute", "phash": "9b5f69a13911a6819bc8c6486f8624e293b4d9e46e62d6d9e6e9b1e6d9c01bca", "mana": 3, "whash": "ef1fefbff3bff4bffc3f7c0c7d4f7d433d611c305d301c200c000000c103e107", "card_id": "GVG_065"}, {"dhash": "e8742cd211244b08b6106cd1b88a7037e36ec69d142a81c51ebb3d62db808721", "average_hash": "d79fefbfebbff5be74383038037d017c03788178013c01580050000001800120", "name": "Nerubian Egg", "phash": "2df569c55b4e5b2b9f934c98469022a024c259929f83fe81ff03ff01bf099ea6", "mana": 2, "whash": "ffffefbffbbff7be74fc35fc07fd03fc03780178017801d80050059005800320", "card_id": "FP1_007"}, {"dhash": "68f3cdcc929105df3b0476029c4c7e9195726284e0cb03173930c26a4c82113d", "average_hash": "c70df35c795f385ff81ffc1f9c36c83e8c1e9c1f001e181898019801d8003108", "name": "Leeroy Jenkins", "phash": "4be3692af6c13ec93e5216c66bb013134c488e939e99d6c9799c13cf2da798f0", "mana": 5, "whash": "c71dfb5ef95f385ff81ffc9f9c3ecc3e8c1e9c1f901e1819d801d801d8007108", "card_id": "EX1_116"}, {"dhash": "02c00d1ff4ef09fefffd7b30c6003c01c042309eeb3cd1db80b20d4f1b9afe35", "average_hash": "ffff0ff003c2f07fe07ff5fff1df61bf01ee018c008c009e008c000c012ce1ff", "name": "Mind Vision", "phash": "fddfb728b6b45ed9e777699653a806daa1706b1553c15b6053525a9459a11428", "mana": 1, "whash": "ffff0bf001c2e0fff07ff9fff1ff61bf01ee018c008c00de008c008c019ce1df", "card_id": "CS2_003"}, {"dhash": "e0009dfef0f9e3a3cf4d9e33353f4af8d0ecf6d9b92317c68f0c93992f3f5c60", "average_hash": "0700010001000018003c207e0e7e0e7e1e7f3e7f3e7ffe7ffe7fde761e7e3e7e", "name": "Feral Spirit", "phash": "8d00fc7d7677f3818e201706ed1829812a4d0fe3393b3b57e2864f393ef4ebc0", "mana": 3, "whash": "0700000001000010003c207e0e7e0e7e1e7e3e7f3e7ffe7ffe7fde761e7e3e7e", "card_id": "EX1_248"}, {"dhash": "9ce02dc362e6fdfcf311ef23da43bc87600ec9dea231014a26b69fe43e08f135", "average_hash": "c70fe37c61d0e0d8e0fe43be03fe03be03be03be03be01b000b038d03b007a00", "name": "Coldarra Drake", "phash": "6d84390eb28a9c59cb438cb35bc249a710d043f379f496d1ebfd9ac5d381699c", "mana": 6, "whash": "c75fe3fc63d0e0fce0ff43fe43fe01be03be03be03be01b000f038d07b10fb00", "card_id": "AT_008"}, {"dhash": "d286384f21be5a76bbe4548929915332a7644ccd9882331d6eb8f862e397c57f", "average_hash": "efa0e7a877e0b451b051b651bb51bb539b5099501950385060004080c521c3ff", "name": "Summoning Stone", "phash": "b76db7397d5ee6c73d275658adb11ad3012116b40d8226f01da08ed415e46ece", "mana": 5, "whash": "efa8f7a877f0f471bcd1bf51bbd1bb539b529b501950385065804000c521cbff", "card_id": "LOE_086"}, {"dhash": "6aff855002c98c917b4fe7f80891f16223c542ce2f38fb4084b9d9f2b64f0312", "average_hash": "f71fff7ff96ff87d3cdd7efc3f791af93ef023903388330c00c400c2009b0400", "name": "Ancient of Lore", "phash": "e7b9ebbbbbe6fbf2af65cfe46f90048d0090b3c13e1c6b04c6e244609f023251", "mana": 7, "whash": "f71ffb7ffb4ff87d3cfd3efc3ff91ef93eb02a903388310c20c400c200990000", "card_id": "NEW1_008"}, {"dhash": "4cf90dc003084534f849a2e55c5f81bf3d63418ee614c90b1a3612e4344c67b8", "average_hash": "f71ff31fe15e407dc0c943c143f118d03d9c3980398c10bc10fc0858001c060c", "name": "Blade of C'Thun", "phash": "69a3fb2a7b7ecb73db60b9c47bd0128d01d1b2c89cec4d057e2b56c4acc11998", "mana": 9, "whash": "f71ffb7ff35fc07d60e943c142f15ad03f9c3f883f8c38bc10fc0878021c060c", "card_id": "OG_282"}, {"dhash": "fc3fdcff80bf156627d86ef2dfc8b9337341e482db076f1f982f605400a90170", "average_hash": "0700c30df10ff81ff81ffc1ffc1ff81ff01bf01bf013f03fe01fe01bc0030000", "name": "Rallying Blade", "phash": "636d5f005e127ed01e011d978654946825a5822603297e96b7b4ee39cffaf6f5", "mana": 3, "whash": "0700c20ff10ff81ff81ffc1ff81ff81ff81bf01bf01bf83fe81fe00be003c000", "card_id": "OG_222"}, {"dhash": "8ae684c941928b21364e48b9b3d473b99b0266d588ab714487392c77528c8139", "average_hash": "f79fffbf7bb9f83b1c311c302450e472984dc0398031100e3804200001180110", "name": "Elven Archer", "phash": "e3f9b3cd6f9c7bf23f6946b23cf00989cb02342c2c044fc3f610ad046d29d69b", "mana": 1, "whash": "ff9fffbf7bb9fcb93c313c302d58e573994dc95b8571114e3804308429980118", "card_id": "CS2_189"}, {"dhash": "0af0a5f1024485bb7ec72d8c18197132606ef29de4231949363745ec8c4cb131", "average_hash": "e70c215c395e083eb8371c37983b9039903cb03c903d3019a09d182918198011", "name": "Gahz'rilla", "phash": "e9b1cbe65e9e6f7a3e4393cd93f0d2d232722e833912462619847920d37039b7", "mana": 7, "whash": "ef1e335c3b5e087eb8371eb7983bd939903cb13c90bd301df0bf38b918198019", "card_id": "GVG_049"}, {"dhash": "0ac6ed7fb0c96396462899e232006511e8828b7c9ab8a75338a71142e6a6f81d", "average_hash": "073e83f1fb8df83df63dfb6cfb67cc230030c032101bf018c01c88088c0ce00f", "name": "Poison Seeds", "phash": "2b53b7805f7bbc6d9cde170a6b802c3429f00ef75967e59243c01b8a7319de8c", "mana": 4, "whash": "e73f87f1fb9dff3cf63dfb7cff67cc660470c0301019f01c801d80088c04f40f", "card_id": "FP1_019"}, {"dhash": "38e26cccd1970b7f34c05919b3c66a1df772c8c5a08bdf17b73b6e667c8ce13c", "average_hash": "871d0b9c1b98181a8c0700070022841b0c3b9c3c183c183c181df81ff80df91c", "name": "Darkspeaker", "phash": "4b4f1c4063830f937b301898f3d0930c33a63660c7a6ff6c9ccd1bd9f366ce4e", "mana": 5, "whash": "8f1d0fbc1bb819ba8c3b04078062c47b0c7b9c3c1c3c1c3cbc1df81ff91ff91c", "card_id": "OG_102"}, {"dhash": "8ad18c6391df0ace34984f33bf6c6799fc32f965e2c9cc972136836f0e9e1938", "average_hash": "f706f106790c780e7806603f0c3f4c1fcc1f8c3fcc3fcc3f081f081e001c000c", "name": "Hungry Dragon", "phash": "4997f3d0f6b03bb77eeb386cb070c824536209a32c5c38e176436e1a3b196f93", "mana": 4, "whash": "f716f986798c782e7826603f4c3fcc1fcc1fcc3fcc3fcc3f083f083f001e001c", "card_id": "BRM_026"}, {"dhash": "e8f48cc1c18b0b5f36b64c7cb3c04685b532cb65068a9d156a338e60588e2139", "average_hash": "f71f3b9f3b9c383cb83efc35f826f006c012c8301c003c081802380238191118", "name": "Salty Dog", "phash": "e3676994cf995f483880934013c399240c61471836e6e7fddea63ce7f9060fe7", "mana": 5, "whash": "ef1f7bbf7bbc38bcbc3ffc37fc76f856c85ac830dc003c081808380239191918", "card_id": "GVG_070"}, {"dhash": "da9ca5796427789af167c01a81950b4b3ab6e44cce719d890332c9e4d9893317", "average_hash": "0ff025f121d804ec83ee9fc33fc02bc00bf80bf983b8c1f860d821d023002700", "name": "Shadowbomber", "phash": "fd3dddfaaef82dcdff98922c3b418cc100425e3246902a44de61ce4c6dac6e71", "mana": 1, "whash": "0ff02ff123fc04ec87ebbfc3bfc02bc00bf90bf983f8c7f860d821d033903710", "card_id": "GVG_009"}, {"dhash": "08e26de492c10906ff7eaee558d3b187621dc55a80f5758a29bf06eea10d4f9a", "average_hash": "f71de31d095e083c203850389038c0bc40bc00be003a003e803f003c001e001f", "name": "Warsong Commander", "phash": "2de5779b5b3b9b1a3b503361866064f8095ace728af3cfc6e7f0d9a1843834c1", "mana": 3, "whash": "f71fcb7d1bdf08bc2078727890b8e1fd43be01be02be083e80ff90fe023e801e", "card_id": "EX1_084"}, {"dhash": "e8cccc1351ec8add37034f0cb8f06065980a027526cadf167637e96cc8990136", "average_hash": "771b7b82f982f80efc1f7c3e3c3ff027f004200018007808b819b813b0039107", "name": "Bomb Lobber", "phash": "c3376fc69f0576c8fee089990cc908ac490a9be32c30636e3ea3fb9cd3c22c67", "mana": 5, "whash": "771b7ba2fba2f80efc3f7c3f7c3ff867f04460001c007c08b839b813b9079117", "card_id": "GVG_099"}, {"dhash": "c087fd6940dc03fd9dfb7fc6ff16fb69c6f3dee9b1f304276a4e181e303068a3", "average_hash": "0700030001008821c417e00ff05fb87ffc3ff83ffe7ffe7afe7f6e7e72327033", "name": "Bane of Doom", "phash": "a9045efb7e400f22d0197f2832a4381c0be7ad47c91316694b7ade9c6ee5f3e1", "mana": 5, "whash": "0700010001008021c017e00ff01fb83ffc3ff83ffc7ffe7afe7f7e7e623a7033", "card_id": "EX1_320"}, {"dhash": "98dc6db2d2c909377ffd3c9efa28a75b0cb73142fbe405885a38edf0c20785bd", "average_hash": "07174377c3f638f83cf98cb77ef73be70d8f4f8de686fc02ecc1a8c1aa01041c", "name": "Frothing Berserker", "phash": "9f0eed863e23df66dfc4939829e00619c4e9e6e106b31797597639e190a571a4", "mana": 3, "whash": "07174b7fc3d628bcbc7b8c777cb729e70d8f4d8de686fc02fcc1a8c1aa01001c", "card_id": "EX1_604"}, {"dhash": "48f8cdd112a24d22bbc574db83b07d65b2dac4e4df0bb9d76236c46cc9993327", "average_hash": "f70fb37f79de305f805ba497b40ff03f70183838bc1aa01ba01bb01120032101", "name": "Sneed's Old Shredder", "phash": "cb79cbc2ff21e346f39219301a3624f138c8dbc1461a5eec67a7178fb82d5648", "mana": 8, "whash": "ff1ffb7f7bde305f945ba497b40ff03f7018383abc1aa01ba01bb03331032101", "card_id": "GVG_114"}, {"dhash": "c8839d2f60d081e1271f5e34b049509ea17802c786e08d815b00a68f4914fe3f", "average_hash": "ff7ff7e5f38fec03bc0b5d72d97843181e501cc01e009f009f0002000000fc0f", "name": "Demonheart", "phash": "0747fb0cfb181eb06da2240787678c210b3c3c0c43c727dcfbf1feb14b0b1f27", "mana": 5, "whash": "fffff7e7f38fef07fe0b7c6adb7843589f401cc01f809f009e0002000208fc1f", "card_id": "GVG_019"}, {"dhash": "38c2ec3c91e90a03375268b9b0e675c1e69a39354d2a8f8516326564c68fa539", "average_hash": "87978fb19ba7f9bc7c3c1c200c3065306438601c300e1c1f8c11c019811d011a", "name": "Mad Scientist", "phash": "6dffbfd31b235b293eac96b5cb40694889a449604b02db40df246f0ef6c496e4", "mana": 2, "whash": "8f978fb1dba7fcbd7c3c1c200d706d70657a615c314e1d5f8c31c499c11d411e", "card_id": "FP1_004"}, {"dhash": "fa8fcd1e92f1bc3779eee3ccc1b1930324060b5e5730dc4830b3d1e48b013732", "average_hash": "27707fe0fbe3f8c3fcc1ffc1ffc1ffc193c083c0c380c3c080c001c003000700", "name": "Goblin Blastmage", "phash": "f729c7cbacf87c76ee3447bd2e10048c00d84b980cb13a01abf7cad59f21deb4", "mana": 4, "whash": "2ff07fe0fbe3fcebfec1ffc1ffc1ffc1dfc0c3c0c3c0e3c0c0c081c003800700", "card_id": "GVG_004"}, {"dhash": "68f8ad00527f44f8b8e073d3878e4f1fbeae7f1cd8eab14566bb88763380cf21", "average_hash": "8f16277203e08141c0c3c0c7c04fc14fc01fe00fe00fb01b601b601a41000100", "name": "Loatheb", "phash": "f96c9fb2e614db215ed249cc1b3322c68b3132e28ba5a7b1d616da69e702cd2c", "mana": 5, "whash": "cfd66f7a2be0c761c4e3c4efc15fc14fe15fe10fe14fb15b705b649a41800120", "card_id": "FP1_030"}, {"dhash": "02808d1ff4fff1f4efcf43141f003c67418c82f00dcd171a6cb6396f60f4c88d", "average_hash": "ffff09f0c1c3e007e007ff8f7fbe37fc07f807b00cb078906080e08cc1c8e18f", "name": "Inner Fire", "phash": "75dd9fa071efa4fd2d5363fec37c1b52e11029035bc453c45b597860d3031c8a", "mana": 1, "whash": "ffff09f0c1c3e007f0077f8f7fbe3ffc07f007f005b078906080408cc1ccf18f", "card_id": "CS1_129"}, {"dhash": "c8e0ad01425b05b47a248cce13996d339a6265cdca8284890d331be2324c7798", "average_hash": "370f337b3b603875b0e59ba03f6018481f1c1d1c5d9e5c9c4cdc284c20183618", "name": "Edwin VanCleef", "phash": "4f0bbbb0fc3c4fcfee611d0e1be1193901911bc8ef9c5ecc5ec33445c23259a2", "mana": 3, "whash": "3f1fbf7a3be07875b4659fe03e601e683e1c1f1c5f9e5c9c4cdc284822183618", "card_id": "EX1_613"}, {"dhash": "9cc6ec6901de4a3035c25281a71e5e79f8e28dc5238bcf17b737fe6dfc9b1134", "average_hash": "071803807389f00dc807d003c0033c063c3c3826182cdc3c703ff01ff0077106", "name": "Frost Elemental", "phash": "cbdd3e087f106605e62438201ee689a30c327973ecd863fbe6c9d36cb34db949", "mana": 6, "whash": "0f180fa073abb08dcc0fd007c0477c463c7c3c36383cfc3c783ff03ff11f7107", "card_id": "EX1_283"}, {"dhash": "03808d1fe0ff197727f84f139d2c7a21e0430cb5d0ec2f08f8d30166438dfc3b", "average_hash": "ffff21f01380dc007801e01fe91f3d7f35f925d9243afe0ee28f0084010ae83f", "name": "Divine Spirit", "phash": "537da78e56c36de0ad3f599629540ca843c30e5759875be119f1fb08d2ba1c68", "mana": 2, "whash": "ffff09f01180dc007801f00fe93fadff25f921d824bafe8ee28f008401caf8bf", "card_id": "CS2_236"}, {"dhash": "eaf04c6391a3023f370c4fdab0246dc9be32f345e0cb029715368569fc930924", "average_hash": "f70f7b8f7b0e380eb83fbc3d7c2d7c0d6c06d83dcc1d6c103830182018001900", "name": "Blackwater Pirate", "phash": "c3dffb61fe617f8e1bce253839b191e0122c1e4aa15173646c6499c19b83db58", "mana": 4, "whash": "f71f799ff98e380eb83ffc3dfc3d7c4dec0fdc3dcc1d6c103c30182418021900", "card_id": "OG_322"}, {"dhash": "fcc9ec3f417fcaf0b5e053b9ad825145aea27c156c2a9314efb180634c87d93c", "average_hash": "07910fb063a0e4a3e44bf40df45771536855700f100f4007f00790069106991c", "name": "Blackwing Technician", "phash": "fbfdb7048e33de8c6f8513a00dc98828238c9b6963727bde8e99f36866223362", "mana": 3, "whash": "0f910fb06ba1f4a3f40bf41ff55ff1536955714f314f414ff007b0869186991c", "card_id": "BRM_033"}, {"dhash": "b2b16dce82383de35284a59e5b79b0e24aef94de652483580eb338a0434ee6ff", "average_hash": "0f6f896cd9dad8d39bf30bf00bfe1bb203bd03ac338c30c020c0c000060084ff", "name": "Pit Snake", "phash": "5d858fa73bfaf25baddeeebccdb4f69091dc82c489a60ef0312186241964cb48", "mana": 1, "whash": "0f6f8b6dde7adcf7aff72bf00bfe1bf203fd038c338c309860c0c200060086ff", "card_id": "LOE_010"}, {"dhash": "b61bf27d6cdaf90072ffc7dc8d0113330c4e40ecbbc1f71bfd71e0c3098ee37f", "average_hash": "01c041c5e0cdb09b0380e38ffb9b9b9303830381cb81f8abf88ff1872303831d", "name": "Priest of the Feast", "phash": "db0d5cbe64ba9487bea09307be47de770527ae6d61796148590b46969c863293", "mana": 4, "whash": "01e041c5e0cfb09f8381e38fffdb9bd783830381cb81f8abf88ff1872303871f", "card_id": "KAR_035"}, {"dhash": "00800d3ef0fce19987815fe7bfdc1e21740eb0f9e0774f4410982f74e680f8af", "average_hash": "ffff03e083808006861b041e0c1f8c1f801f830f266ec67fc401fc007c00f80f", "name": "Counterspell", "phash": "59725c24b685b792c94d07bd4d90224dbb818b444b2e7ab669b23ed9334bb6ec", "mana": 3, "whash": "ffff03f08380810f861f051e0c5f8e1fc01f838f36eec67fc641fc40fc00f88f", "card_id": "EX1_287"}, {"dhash": "caa78d4c121bbc3c70a0e3cccd73bbef40c6fe0ee991424b04b728e619083730", "average_hash": "f701f369f1c9a0c0e0c0038033da31bff9bbfb9f73bf70bc20bc00dc021c020e", "name": "Ethereal Arcanist", "phash": "6d19931b3ebae9c59f646ce45e86262782f4c9a0446c6e4b8e9c69d4ffa14e05", "mana": 4, "whash": "f701f169f9c9f8c1a0c00188339e31bfbfbbfbbf73bf70bc20bc00fc021c020e", "card_id": "EX1_274"}, {"dhash": "2ae24484d1098af434315143b69e77fdec5ad185082bc347af377a668c883933", "average_hash": "d79fdf9fcbb7ec01ec058400846004662436283e00372038303c181811103112", "name": "War Golem", "phash": "addf1b9b7b29e9c199f41ccb5a684b29c19092586b483e345ba4cd862d0edbb2", "mana": 7, "whash": "df9fdfbfcbbfedb1ec0584048564056625772d7e217f3178183c189811b03116", "card_id": "CS2_186"}, {"dhash": "ca630cce11bc0bb437484e91b86e7019e06ac89583691ed331374d6cba98153c", "average_hash": "f70ff118f918f839603d003f003f883f003800380038003c003f003fa01f201e", "name": "Lost Tallstrider", "phash": "29fdf3c1732ae661f62699d86cd888d809933336ccec9e61668c9b25ec80fec8", "mana": 4, "whash": "f71ffb99fb99f83df43d403f003fc87f1c3a1438043c043c003f203fa01fe11f", "card_id": "GVG_071"}, {"dhash": "289d8d32024b3cb579f2e3dcc6f986430986123ea7504f897432c8e49005433b", "average_hash": "f773fbe2f9e078c078c09fc03fc473c67bc473847b847bd038d101d103004700", "name": "Servant of Yogg-Saron", "phash": "d782d352386cfbb4fb50092d2e428089a3c872bc1c9c3e47abf3faf577281a15", "mana": 5, "whash": "ff53fbe3fbe078c03ec01fc03fc47fc57bc47b847f847fd038d101d103004700", "card_id": "OG_087"}, {"dhash": "c8b7ad5f12a23c0473ff807ec7bc1af932c3672c8dd01088253148e3f0842b27", "average_hash": "3760f3e8f9ce70c330d03bc0ffc1ffda7fd83f983b9019c138c008c003000700", "name": "Dragon Consort", "phash": "d70de122bcd26dc92f61164bbe912c9000c99aa4dbbc9b4cdfc7ea6568733345", "mana": 5, "whash": "3770fbf8fbce70c638e03fe0ffc3ffda7f98bf98fb901d8138c008c003001700", "card_id": "BRM_018"}, {"dhash": "529e203ce5f85ce5b9ec5619ad035ad7a7341fccf622efdd003b0054e3817d3f", "average_hash": "8da181e1c440444362c3b7c333d313d8cb87cb47f3dff05ffc5901c00503cb2f", "name": "Runic Egg", "phash": "9bef16dbbe5f2d37be17e639dbb196dc2989ccec25cc92cc64b0068264244640", "mana": 1, "whash": "8da183e1c5c0444374c3f74373d31359cb47cb07fb1ff85ffc5d04400583c33f", "card_id": "KAR_029"}, {"dhash": "b898ad2142d6c889f9e0a21763af8e383576674fe8c4ce891d3338e6624c8f31", "average_hash": "07378f6703ee008f80cc42c742c3abc1bd08df981fdc5ccc4cfcc0f8c218861d", "name": "Giant Sand Worm", "phash": "5d8bdf982d94ef07bf9198a48f80216382f93e8993ecea64ee34770366b36422", "mana": 8, "whash": "0f178f6f03ec00af84cc46c746c3abc1bdc9df981f9c5cdc4cfcc0f8c298861d", "card_id": "OG_308"}, {"dhash": "fa9f5dbfb37247bd38e010807966c4ccf9f3fa2f07530c84700c9b986678deaf", "average_hash": "0700c307f323f073f07be00fc066c036607e307fd87dcc120c1748068e00fc0b", "name": "Mortal Strike", "phash": "b926f9d1be4dce2e4eb25e260b7926d925f8a4b6cdd01e58634b3a21c3249da4", "mana": 4, "whash": "0700c307f333f173f47bf00fc066e076607e387fd87fce3a2e1f48068e00fc0f", "card_id": "EX1_408"}, {"dhash": "a8d8cca3018d0b1ab73d4e83bcfc7951e3b2c84513c94796fe37fa6eec9f393a", "average_hash": "4703438431003008201864380c38f839fc317c307c203c20783cf83bf01ff91f", "name": "Cult Apothecary", "phash": "e35d7e72d9a096863b810d8c4e523924921399616686dff3b3b86bb9bc291c9e", "mana": 5, "whash": "77177b867184b8182038f43c0c38f879fc317c307c203c207c3ef83ff81ff91f", "card_id": "OG_295"}, {"dhash": "0a80dd1ef03ce2db1df0b3cd6a9b503c9fd380a367061e444108a7184a20fc8f", "average_hash": "773e07e003818103cc61c201f211f851bc493e607e60de610e000e000400f00f", "name": "Savagery", "phash": "977bf9f17ccc87fee7fb3ea129c4073e13633204cb949a585b81668063c8149a", "mana": 1, "whash": "ffff07e003818303ce61e243f2d1fad1bced3ee07ff0fe639e000f400640f80f", "card_id": "EX1_578"}, {"dhash": "68b3cd4c92933da77b6fe71accbd917302a61c6c39006309b2b08de30e025b2c", "average_hash": "e70fb1ee3bdb38df38dd3bbf3ffc7df17b817b803f803d8000c020c433001300", "name": "Cult Sorcerer", "phash": "d709a9cb3872d776bfc4193e2f15528600f08f1d5c0ceb019b9ffec47f205e01", "mana": 2, "whash": "e71fb9ee1bdf38dd38ff3dff3fbe7da17f817b803f803d8000c000c433001300", "card_id": "OG_303"}, {"dhash": "123a10f4a6ac290d53daaeb1de25351a7c60b082e145e48bc11b4a509084857f", "average_hash": "c107810fc00fc01fe01fe01fe01ff00ff80ffc0ffc07fc07f80bd801c0010000", "name": "Spirit Claws", "phash": "5b4f7968b4ef8732df13f299369c92cfcd87b2b40de8a064d1c672681369220d", "mana": 1, "whash": "e187c10fc01fc01fe01ff01fe01ff80ff80ffc0ffc07fc07fc0ffc01f801e900", "card_id": "KAR_063"}, {"dhash": "c26019d422a06d4ef20a85e358c1bf0636ad7a9e0c30b34dd8b370e0c107d2ff", "average_hash": "f74e7158101e009e00bf631cc01fc0bf829fc09fe10fe013e08ce001c007e07f", "name": "Jungle Moonkin", "phash": "79a1ed2de67ef37abb5b630ef3ac5a91097c9a66096b8a249e1a09d311a4062d", "mana": 4, "whash": "fd5e715c105e00de809f439dcabfc2bf829fc09fe30fe093e089e005d007e0ff", "card_id": "LOE_051"}, {"dhash": "0ab0fdffd04e23b9d4f01b133f602c25e6db1b25e6facc8489093a97732efedd", "average_hash": "07000708433ec12cc007c02f603f203d3036e814c84cec0c7e0c724c3e4cfe3f", "name": "Journey Below", "phash": "6bd6be7d4ff876afed39aecc13679620825126124b840f8cb9e137836318dc88", "mana": 1, "whash": "072007c0e33fc32fd02fc03f603fe03d303ee854f84cec0c7e4c724c7e4cfe3f", "card_id": "OG_072"}, {"dhash": "0af90cd65188cb12b7024e0fbc386151d2e20c25b0cb6297cf369b6f229e6538", "average_hash": "f71ff99fe39f803f803f143e183c3839181a3802182608360036403e601c611e", "name": "Amani Berserker", "phash": "e975fb797b26d68673c90dc68ee230c880980d9387306e20bb44f71c6f09ff31", "mana": 2, "whash": "ff1ffbbfa3bf80bf943f143e1c7e3879187838021c260c360836c03ee11ce11e", "card_id": "EX1_393"}, {"dhash": "0880991ef0e34115f6eb681eff36926924b449dbf6aecda1274dcc1b387f7afc", "average_hash": "072007c0030000000000080000102c12081212d61ede1f7f7e7f7e7ffe7ffe7f", "name": "Fan of Knives", "phash": "cd2436357bda5fc22f24831c07f31304231d49f0f7c09b171a7e3790f8a1f976", "mana": 3, "whash": "0f7e07e0038003000000080028122c16081a1ade1ffe1f7f7f7f7efffeffffff", "card_id": "EX1_129"}, {"dhash": "8af104cf119c0b0037304e46bebe796df21ac4e59449239306368d6c5a92ed3c", "average_hash": "f71ef91ff91df83ff83c043c003c203c24380438043820382038201020000800", "name": "Stormwind Champion", "phash": "697d63c3f3c2fbfc5eda4c920ca61626c148d30038a37b38be84264f678ef698", "mana": 7, "whash": "f71ef9bffbbff83ffc3e8c3c043d2c7d243b84380c3f243f203d203429041910", "card_id": "CS2_222"}, {"dhash": "c8dfcd6812c1048e7990b66a0c9554a82a16f1ec8d4782881cb979e2f146eb3f", "average_hash": "e707f367f94f784ffc4ee68a821d81158006c91ceb8de008e088e000e000f600", "name": "Princess Huhuran", "phash": "732a8f23f3c0f918bfe43c302f040cc646d446160fa35e67d7317761bb63d370", "mana": 5, "whash": "ef17fb6ffbcf784f7c4bee8ea61d891d8916891deb8de088e088e080f200f600", "card_id": "OG_309"}, {"dhash": "f8ef6cdcd1188be0b7534f1bbc3670cdf0fac1c586090dc73836796ce8988121", "average_hash": "07138399c99fc033c03f803f203f203f043e1c3cd83c203c00388019c001a100", "name": "Silent Knight", "phash": "293d992db6016667df24ed0866d29ac9ccccf38c87943be7c9c352ce694c3692", "mana": 3, "whash": "0713cbbbc9bfe83fc43fc43fa43f247f0c3edc3ef83ce03c2038c839e111e100", "card_id": "AT_095"}, {"dhash": "fe3fe8fd87d83d305b29bdd25284e548cb919663adc75e0fc31dfc1bf16400dc", "average_hash": "0100e103f007f81ff81fe835ec25ec210c350c250c35dc3bbc3cfc7ffc0ff00f", "name": "Maelstrom Portal", "phash": "819ef2917a69beff1e7cae57ae854cfa64e915ec44ea05e2e49e06ea818604a8", "mana": 2, "whash": "0100e103f047f81ff81fe837cc276c210c350c250c35dc3bbc3ff87ffc1ff01f", "card_id": "KAR_073"}, {"dhash": "8af08de112c3059e3943728c85704b45c402b91430ab62c7c3369f6c049c0d38", "average_hash": "771f717f3bdf387e9c4f98cf1847f003e023c026000600360016081608180118", "name": "Lorewalker Cho", "phash": "c9aec3da6d997eb4734893491cc9030e203cceb04ef23f84df183d399b26d666", "mana": 2, "whash": "7f1f7f7fbbff397e9c6f9ccf9d47f143e927c0260107007604360816093c0118", "card_id": "EX1_100"}, {"dhash": "eadffdff00ec1f107c449f9024234f0ef01880f119c043861989ff33c343061e", "average_hash": "0700c347fb3ffc7fa07f027f027f027f067e06700e000f600e40ce43e607e207", "name": "Power of the Wild", "phash": "1d08e9fd53f7067f7ad49c8acc813c3c2c7213bc27c1a7c4cb81d215fb097e21", "mana": 2, "whash": "0708e367ffbffd7fa07f027f027f027f067e06700f000f600e40ce47e607e20f", "card_id": "EX1_160"}, {"dhash": "4cd80ce0d1800b03378f4c26b07866f9fca2c5c58b4bb59608302968fe9e9933", "average_hash": "f71ffb8ff99e183d183c0c3c0c244c22783e3c3b7c387c182c00082008101111", "name": "Mogor's Champion", "phash": "cb5759027bb17360bb841e320b29a181606836ce9edb6ba97dd23c7d2d0b67a3", "mana": 6, "whash": "f71ffbbffbbf383f1c3c1c3e1c264c66787e3c3bfc387c183c00082089301911", "card_id": "AT_088"}, {"dhash": "48e68cd8d13d8b6236814e70bbbe676dc8d294a53acbc11603372e6f4c9cb939", "average_hash": "f719bb990998003f8c3e443e803fa03fbc3f3c37bc0f3c0c1800181c181c191c", "name": "Corrupted Healbot", "phash": "c95d3d246601c761fb903c803be60320939859d2be846edb7b8bd6ce6d8eb639", "mana": 5, "whash": "f719bb9b099a083f8c3f443ea03fa07fbc3f7c3fbc0f7c1c3c08181c181c191d", "card_id": "OG_147"}, {"dhash": "fa9f55ff70fc8f031dcc7621626c409c6299dc33316340c43199ef33597ef6df", "average_hash": "07008305c11fe47ffc7ffc7f644e7c007c332e370e650660060026012020f01f", "name": "Whirlwind", "phash": "0bfcf96f78739ef8c694666a0fb0831c266d3c1209d02c7de3c572215bda1e70", "mana": 1, "whash": "0700c307d11fed7ffc7ffc7f6e5c7c427c333e370e65066006402603b025f81f", "card_id": "EX1_400"}, {"dhash": "08ff0c3241e64accb72c4f59beb05c65b22aa75d4daaba4421bb4e769885213a", "average_hash": "f71fe19fe38ff00f703f701db01db01d701dd009c0092008000a100200020118", "name": "Beckoner of Evil", "phash": "69da7bb6de643bc32ff34fb926b182c060c81218b6c6d6406f093c43bf236e6c", "mana": 2, "whash": "f71ffbbfe3aff0aff43f7c3fbc7db85df05dd02fc00aa00e100e10060102011a", "card_id": "OG_281"}, {"dhash": "3aef3978c2c24cadb952702883504ca5326a615c83aaba4506bb7c72f987d32f", "average_hash": "ef53ef7befff78dc5cccdcc20d46214c111a10184110205830103008310ca107", "name": "N'Zoth, the Corruptor", "phash": "7f681bcabb183fa99b72967892b666e368fe669ee6b64c65371146006b801782", "mana": 10, "whash": "ef53ff7beffff5fc5cfcdcc22d4621ce11da11184950215830183088b18ca12f", "card_id": "OG_133"}, {"dhash": "5ce069c8922c05597a92ac6519497792402ec67ddcf3ba8b013613e6a44cff39", "average_hash": "f71feb79db5cd876c8f282b2c231c33a8130833003b900bd18be98980818661d", "name": "King Krush", "phash": "290773224b4b7f1b3b5919473b98926759d4598683b36ef647c7cd849398848f", "mana": 9, "whash": "ef1feb79fbfcd87ed87286b08271c37f8130833003b904bd18fc989c0a186e1d", "card_id": "EX1_543"}, {"dhash": "e8f3ccef119303b4351852b0b8d873e1ef9284656b8bbf14c13b376f689c813c", "average_hash": "171ffb9ff999f81c780ff819700038302c300c001c01d8031808b81cd00cc10c", "name": "Spider Tank", "phash": "e35bf3907b039fc79eb01ce19ecc4c1888e443f266d833e7af19d6c679224c18", "mana": 3, "whash": "171ffbbffbbff8bdf83f7c19fc5838702c780c221c01dc035c08b81cd10cd10c", "card_id": "GVG_044"}, {"dhash": "8ab9057004ca38967346e7788ec118737fc6c28c3110e6098d331ae7348e631c", "average_hash": "f70ff1cf79ce38cc388c398c399c1998b19e398f7986788e588e488c430c430c", "name": "Northshire Cleric", "phash": "4397d3f93bee91f4bb741d679f5010190449560cc19cea806e271e47b766e374", "mana": 1, "whash": "f70ff1cf79ce38cc388c398c399e9998b99e398f7986788e788e489c430e430c", "card_id": "CS2_235"}, {"dhash": "e29ff57fe0c99b889e991f352df246c8881321a7f24e34884b02ee7c181922bf", "average_hash": "05008141f90ffc3ffe7ffe7ffe7efe7e7c66fc707c784c1e3c133c0320014004", "name": "Ice Lance", "phash": "83fe7d93daddc67ff6da9fa02cec96984a56258c23654a06494336815ba15932", "mana": 1, "whash": "07000101f90ffc3ffe7ffe7ffe7efe767c6e3c627c784c1e1c123c1330010000", "card_id": "CS2_031"}, {"dhash": "8083ed3720cfe2b41d4318ba27c6522cedc8b2331dee6890dc26a3ccc091fb7f", "average_hash": "f7bfe3e4f986fc2dee653e63fe72be179625e469e063fc06e80288000003c007", "name": "Stand Against Darkness", "phash": "3312779afa476d371de18e25bdc83c890b986acc614cca71c3915a9c43737be4", "mana": 5, "whash": "f7bfe3e4f986fc6dee65be63fe73be179e27e468e063f806e800c8000002c007", "card_id": "OG_273"}, {"dhash": "c89f4d3692e4088d793ab2a4664d859818f6301f5334ac48d4b98be38743172e", "average_hash": "f730ff62dbe6ddce5ccc5ec94fc80fc4054433ced3c2e2c340c140c002860622", "name": "Forlorn Stalker", "phash": "57c6e3c3bf74f370ee703698238844c828b91c256ef99a193ec7d78c619b08ec", "mana": 3, "whash": "ff31ff63dbe6ffae5cce5ec14fcc0fc405c473ced3c2e1c340c244c402860622", "card_id": "OG_292"}, {"dhash": "6adcc43141c28a3c35fb5206bd5c72f38fe67dcdff0ad344a6b78862358de93a", "average_hash": "97039d87f98e680c6c09cc099c1218103803f017f80ff00f300f700e5802d902", "name": "Cogmaster", "phash": "c3df1f9ffff5dbf19ea59fe913692cc8341846022c924e826f03260736609ac1", "mana": 1, "whash": "9713fba7fbae688cec0bdc0d8c523852f843f81ff80ff00f701f701e590ad902", "card_id": "GVG_013"}, {"dhash": "6480fd1fe0ff01f9178e8e1c996164c2c9a6f11de6b38c4f43ba0c7c80b0fc07", "average_hash": "073803c07f80fc00fe1f7a3f78366064606c507c007c807a107c00740062e00f", "name": "Siphon Soul", "phash": "e9d647059e693ceaf41aad24090119921be9275663e7b6657b929e4c9179da95", "mana": 6, "whash": "1f7c17e07f80ff01fe1ffa3f783e7074706c507cc27e807e147e00740072f00f", "card_id": "EX1_309"}, {"dhash": "4af00ce111834b8237144c2bb94672a9c49a91f53d09734264b6cb6cbd9b8334", "average_hash": "f71ffb9f7b9fb03fb03e903e04360c3204300430043000330031003000100100", "name": "Silvermoon Guardian", "phash": "a9f4492b7b89b7d8963699a44b322693386893663c9cf8e01e932d2fd90cf68e", "mana": 4, "whash": "ff9fffbffbbff4bfbc3f9c3e2d760d7205700d7005702173003100b051b00110", "card_id": "EX1_023"}, {"dhash": "c8fe8cf501c08b8435594b86bc1c6a71f6f2cc6583490c941139e7769c81312e", "average_hash": "b70df31ff91fe81e680fec0f9c3f1c1f381f7c33dc318c008800180338033103", "name": "Gnomish Experimenter", "phash": "cb1f6918f23036b19fa51e63369e98987c7a8f732f27d9607392189badc88603", "mana": 3, "whash": "f71ffb9ff99fe81e682fec0f9c3f9c1f383f7c37fc308c008800180338033107", "card_id": "GVG_092"}, {"dhash": "2cdeec6411c38a8f357b6bd6b08c64f9c9729785280963d79e367f6cf68dd939", "average_hash": "e717fb871989580c78087c081c201c201c209c272c233034183838189819d91d", "name": "Deathlord", "phash": "879f5fda79a19db41b0366a47bc0230f64e29e8e3df36e73e6c88cc9c1806648", "mana": 3, "whash": "f717ffbf3bad788c7c0c7c283c601c601c609c373c2334341838383cf91df91f", "card_id": "FP1_009"}, {"dhash": "b8ff7dfec3c08f0b3c777088e0b04a65fdc3fa87d23f0f1c3a78d0f8a1e17ccf", "average_hash": "0700c30fdd3f5c7f987cb87db83934083006207eb07ef078707060620042e00f", "name": "Dark Wispers", "phash": "e97b0f131fc6fca05a38b65c240a440a66e02c6ced1c1a9dabd17a796173fda6", "mana": 6, "whash": "0700c35fdd3f5c7fd87eb87fb839b408305fa07eb07ef078f070e0630043e04f", "card_id": "GVG_041"}, {"dhash": "5a36b97d225b5cb4b17a63a106e34d4e3b1ce4dd9403fe1d1f39f660c097717f", "average_hash": "8741a3c1e043f0c1f081f00bf00fe00de00d9011f838f01cf808f803f107f17f", "name": "Elise Starseeker", "phash": "73db5e9ab73fae2df99a5a5a2d6596d5611848de85620ac851d826644cc6c2dc", "mana": 4, "whash": "a741e3c1f643f481f081f08bf00fe00fe00df013f879f85ef80af803f107f1ff", "card_id": "LOE_079"}, {"dhash": "0086bd3fe0f493e92f0347e41cc83b01f7222c05d248a2930527f54e0a18f436", "average_hash": "173c03e3e183cc07ec0fec1fc01fc01be03fe00ff01f681a3c18bc1c28043804", "name": "Arcane Intellect", "phash": "0b5f198ec610f4e0fce02f314bc60f12230f8ab46bc97ae45bb366ec5b6a78b2", "mana": 3, "whash": "77be83e3e183cc0fec1fec1fc01fe01fe03ff00ff01f781a7c18bc1c2804780e", "card_id": "CS2_023"}, {"dhash": "0a82fdddd0a7af1f1d5a32b87ef1e16cc45508b33ca07342e4c1190f3312ce2f", "average_hash": "070003003b1078787c787c78387c307f387fbc37be073e077e077e067e20fe0b", "name": "Astral Communion", "phash": "e788a52b7e479ab1bc706cd803fb2766094c25d6492f67d38b69d331637053a2", "mana": 4, "whash": "07000340391078787c787c78387c307f387fbe779e173e077e077e267e00fe0f", "card_id": "AT_043"}, {"dhash": "9cf429e1428ac537fb9934278e4c44b88332f66126c7c88899397763ce4e3338", "average_hash": "e71fe37d635c004c305e5e864c020c001c001d241cbc0c0c8ccc000c301c1018", "name": "Mal'Ganis", "phash": "4d9b7b9879381b9bb365194e8b8493411269d8b0b6ce436757a6e584ea9ac3e9", "mana": 9, "whash": "ef1fe77f63fc805d347ffec65c5f8e515c081f341dbc0c9c8ccc00cc309e1018", "card_id": "GVG_021"}, {"dhash": "9af92de2428ccde0fbc2a1154e39c0821b1726e969c46e0811ba8ee4934e6739", "average_hash": "071fc75fc35fe0ffc0ff12df101d7ae08c098f890d82008610c020c000080208", "name": "Pit Lord", "phash": "590ad9226d6c5f262e19c6e09bc4916811f246de06bb9991e76cc7c4bb233df3", "mana": 4, "whash": "af1f8f5fe37fe5ffc4bf16ff1add7aec8ac98f8b0f82948610c020c000b80608", "card_id": "EX1_313"}, {"dhash": "388f7c3aa0b41c79a2e6256ddb73a4077799ccf27905b1db4a26b755c8a7017c", "average_hash": "4780c78047400000000000068007c00062130023a003a037c437f81ff90ff91f", "name": "Assassin's Blade", "phash": "9b6436c27fc3b74c674813b6cc1080e9242c42990ca6d9386fa9765abff9db1d", "mana": 5, "whash": "47e04fe0cfc047c0818080869087e281e3bb8423f82fa837e437fd5ff90ffbbf", "card_id": "CS2_080"}, {"dhash": "8af80cb1114e0ac136044940b2366ef5b88ac135130b66469cb56d6ef49d193a", "average_hash": "f71ff39f7b9ff827b82738000000000c601c60183004300620002019101b111e", "name": "Burly Rockjaw Trogg", "phash": "e994db16fb6573ec7de81e78963021c949b218833659b64c6e9a4c866c86dea6", "mana": 4, "whash": "ff1fffbffbbff8a7bc372c258560005c607c703879343026201c7039591b111e", "card_id": "GVG_068"}, {"dhash": "4abc8de7429c7d30703be266c60988a301465abea4911f0b7132c2e41c00bf29", "average_hash": "371a13ecebdcf0dde0cfdbcfdbce3bfaa380039403bcf3b818d300d303002300", "name": "Wee Spellstopper", "phash": "fd01796a8ee8efb5fe5093677bd098b801e06c9c03e53201ae3dca347f805e1e", "mana": 4, "whash": "175a19ede9dcf0ffe0efcbcfdfcebbfaa380039403b4f1b818fb00d303002300", "card_id": "GVG_122"}, {"dhash": "8cf30dc712ca8db07b4fe6180c31d2623cc5608e815806817f3eeff5bc4b211f", "average_hash": "770ef35c795c601e089d388b3b123806308c2080200000007810781f781f781f", "name": "Grove Tender", "phash": "e38d7f326bcc7636cb6097a41e4009991876ba346e74e760c3b893e18f813e67", "mana": 3, "whash": "f71efb5ffb5fe01f18dd389f391a3896388c30802990681078117877781f781f", "card_id": "GVG_032"}, {"dhash": "5a1cbd31e2c15cf2f18ce7192de358a6130c7f9dfc2289dd16b913622796d57f", "average_hash": "876385c787c7b0c4b0c702d7e346e200e201d019b15fa04fc00fc003d100e3ff", "name": "Sir Finley Mrrgglton", "phash": "797b9fbc9dde197bb72d9edccda996f51118666981a4227019c86a5419124b1c", "mana": 1, "whash": "874387e7a647f4c1b087d397e35ee3c0e300d159b15fa04fc58fc007d100e3ff", "card_id": "LOE_076"}, {"dhash": "6ae1e4e2c18d4b3bb7624e87bc2c7995e26ad495a96b10972736006cb899e533", "average_hash": "071a8d9e03bc003c803f803f043f00380039203d603d0c396c3100384010c110", "name": "Stonetusk Boar", "phash": "2dfd5f7b7627ffdc7fe20fe9338020e086422c884b909e617f8073928d2133a6", "mana": 1, "whash": "8f9a8fbe8bbe85be843f843f8d7f09790179a17d617d0d797c3100b861b0c910", "card_id": "CS2_171"}, {"dhash": "3ade2ce0e198cb23b64ccc99b83271cfee3ed13d0eab904703b6286cc7801f30", "average_hash": "cfbfcfbf87bd05b910f9323103700171037d817cc17d11f8103800b005800320", "name": "Dark Iron Dwarf", "phash": "add99b995e0679467f9686a483c001c381e918686cd27cf23e9cc661e7637de6", "mana": 4, "whash": "cfffcfbf83bd07f994f9257103f101f9437d81fcc3fe11f8107805b005b00320", "card_id": "EX1_046"}, {"dhash": "03c0cdfd64fa1f353c497002f236e759ce33866f1cf73b8637048789fe930224", "average_hash": "739e01f060981c7c487f006700e300700878087d0c7f8e3fe63ff01ff80ffc07", "name": "Mass Dispel", "phash": "899ff4aa73612d41e677db001bcec9a34fd049f20c795bc95abd53167aa053a0", "mana": 4, "whash": "f59f01f07098147d107f086700f308700878087d0c7fce7fee3ffa1ff80ffc06", "card_id": "EX1_626"}, {"dhash": "0c80d9ffe49173e2ce6c9f5d3e837986f32e01591e27fcc330a00048008ff9bf", "average_hash": "071807c0021e003f027f827fe27fc07fc07fc07fc47fe03fe007e00000000000", "name": "Pyroblast", "phash": "39408d08461533fee52333302d9286eb89dfa6fec17d69df1bf15a854bd05ba0", "mana": 10, "whash": "779e07e0029e007f827fc27fe27fe07ff07fe07fe67fe43fe007e00100007006", "card_id": "EX1_279"}, {"dhash": "2af80de41288c9b4f76caa8f1c01c31314ef7d7c9f1121c8003021e2f24507ba", "average_hash": "f71ffb7ffbdfa0bd00bda29993f382c3018053907f93328000c0084000000600", "name": "Master of Evolution", "phash": "7d096b9b6bbaf3e6bbe692602bc40ccc10d41b2027fc70491d0f39612f8bae8f", "mana": 4, "whash": "ff1fff5ffbffa4bd24bda6bb9fbb82c301c0719a5fbf768014800c4002810600", "card_id": "OG_328"}, {"dhash": "0a80fd7fa0ed4183877c1d02f2cfcfffdb6ce5f318f37706b30c661f8c3ffa7c", "average_hash": "07000340010214060014c027c0628007f003fe217e203e219e05be3d1e7efe7f", "name": "Arcane Shot", "phash": "0bd2fcff4fb3e7c9796e1e92a7dc21b10bf1cc6033c813c4130d4ec16481772a", "mana": 1, "whash": "070803400b8e95071414c427c0638007fe43fe397e713e21bf0dbe7dbe7efe7f", "card_id": "DS1_185"}, {"dhash": "08fb2ce041ef0bf2b70c4a01b4925225a9e23645e98a144469b99472f185573f", "average_hash": "f71febbffbaff83ff43f741fb41cb0049801a8016001d0001002000000000100", "name": "Spawn of N'Zoth", "phash": "fbf379105f461e87be93e6602bc8138889e336626e6636666e86cce4cc613663", "mana": 3, "whash": "ff9fffbffbbffcbffc3f743fb57cb1549941d94df141d1401002200001000104", "card_id": "OG_256"}, {"dhash": "fa9fed3c92f308467918a6806ba395162f6e549fe9340e4910b8cfe3dd402326", "average_hash": "0710cb67bbe7f88f7ccf72c302c583c181138399639ea08400c100c060006604", "name": "Core Rager", "phash": "d389db0fee66bcf73ed33cb867a020cd00f649410cd19a31ef2c96418cc3dee4", "mana": 4, "whash": "0f10ff677be7f9af7cdf7ed702c783c381d79399a39be08480c120c172807604", "card_id": "BRM_014"}, {"dhash": "6af8c5e7938e0d7976b32c6c59d0f2027067e488cf05020b4c3ea6e45849733a", "average_hash": "f703f10cf90df818f830dc32f83bf03850382038603a303ed09a301830000018", "name": "Possessed Villager", "phash": "6165b3f3debaedd89ed099259f840e0c30e4f3103b8b8e991b899921ce9833fc", "mana": 1, "whash": "f717f90cf94df878f830fc3afc3bf8ba503830b870be703ef0beb01c300c5018", "card_id": "OG_241"}, {"dhash": "a8cf8d3703ee0ccc7932af895c23fc86700ff7786455d18bd9bbb3e7064f1f3c", "average_hash": "77007300f904784e600e029d003d00bc003c023d00bf003fc09fc01fc01ff01f", "name": "Doomguard", "phash": "6982e60c93266e62a64019a58e9019c144cc4b9e33e1be6c2ff37fb3ee9c53fc", "mana": 5, "whash": "77107b00fb4cf86e60ae82bf00bf02bc00be83bf88bf803fc0dfc03fc01ff81f", "card_id": "EX1_310"}, {"dhash": "808f8d7f10c00114374a4c9c98d037816f028904244d40f2a0e4a7e94b1afeaf", "average_hash": "fffffffff3dff81ff81bf81bf803f807e007c00080000000000004000700f48f", "name": "Solemn Vigil", "phash": "f355e9084d983ca613e686c1a4492e04294c0e314b676ed45bf85ef15bbf77b8", "mana": 5, "whash": "fffffffffbdffb3ff81ff81bf983f907e187c18380800000000002000380fd8f", "card_id": "BRM_001"}, {"dhash": "0082dd3f60f383c40f991f361cea788cf7c8eff0c0c381070327064efc14880d", "average_hash": "77bc03e0f1877c0f2c1ecc3e4e1c461a063fc63fee3f1c1e1c181c1cfc0df00f", "name": "Cabalist's Tome", "phash": "0b5cf70cf2d3bd393dc0268323260d508e6923dc4ac71373c9157beccf32db98", "mana": 5, "whash": "77be83e0f1877c0f2c1e4e1e4e1c461a063fc61fee3f1c1f1818181cfc0ff00f", "card_id": "OG_090"}, {"dhash": "28f60de0d3c985037b0ca4e3489cb133463f3cfe40c4d10823b147e3ac4e739d", "average_hash": "f71fdb7fcb7f887d38fc13f807f10bf103e303cf07c010c000c118400a08061c", "name": "Iron Sensei", "phash": "0dcb79936bd39b533b0096606fd0c6e881d98a9919dc7eccee2e49b54603d6e4", "mana": 3, "whash": "ff1fff7fcb7d907d3cfc17f807f11ef103a3078d07811de018c118481aa8061c", "card_id": "GVG_027"}, {"dhash": "0a84fdf9e0a1e3079e0f373262ccc04092b735ef46b289c53f1f637ccca3b14d", "average_hash": "071803403b9c1c7e1e7c0e7c067c007c007800489a48b639fe79f879b82ffa0f", "name": "Bite", "phash": "8d8b5e2cd351b6f5785e37b84f6080cf33f0c87321ee4a79c78573125ec0d306", "mana": 4, "whash": "071c03603b9c1d7e1e7c0e7c06fc007c007800689a68be79fef9fc79b86ffa0f", "card_id": "EX1_570"}, {"dhash": "00c00d1f707083c0bce7790df332e7654fcbbe936031e17202e404c9c19bbc37", "average_hash": "ffffcbf0e1d3e023e023b42334213422742ff60ff02fe02f302390218021e03f", "name": "Divine Favor", "phash": "f97037ae3c0759004fa886e149428b18631c3e73cbf11bd6d3d91a7d7b2358b9", "mana": 3, "whash": "ffffcbf0e1d7e123e023f42334213423740ff60ff22fe02fb02790218031f03f", "card_id": "EX1_349"}, {"dhash": "78ee6dcce2197d70fac3c5249fcd399b7336470d9e107c49b0b364e7c10f0b1f", "average_hash": "0700834101598011c013d317cb1fc93df939fb18f380c081808d80df801fc21f", "name": "Anomalus", "phash": "39a3be8c9cb866136f4f5c12d36482a910d1f69c66ba6e85d6eddac57f022c11", "mana": 8, "whash": "07018343015bc057c01fd19feb1fe93ffb39fb98f390c08580cdc0dfc21fc31f", "card_id": "OG_120"}, {"dhash": "caf38cc511830b0e373c4e58bcb07079e1f2c4e58fc91f172136166c2c985930", "average_hash": "771cf31d311c381c781cf81cbc3c3c3c3c3c3c38383878381830181010001000", "name": "Tournament Medic", "phash": "e31d6b38fe60bf9c7e9c5392334884e40cd614588c71cb81fe10dc66ff84b639", "mana": 4, "whash": "771ff99fb99fb81cf83cfc3cfc3c3c3c3c3c3c38fc387c3d183c183018101110", "card_id": "AT_091"}, {"dhash": "4cdccc2991530aa334e6517ca6f06d21fb02d245b2ab6c4318b7616c83990f27", "average_hash": "f707bb8739835801b8003c007c00fc019c2000300034803c8039c03bc017c107", "name": "Stoneskin Gargoyle", "phash": "a3d7d7c899412db49b2c6b396e26386d27c373e636c94ede9b6d32d361088205", "mana": 3, "whash": "ff97bfa7fba3f8a1fc013c007d40fd61dd70cd700134803c803dc03fc13fc10f", "card_id": "FP1_027"}, {"dhash": "0080cf1f14e0719ee72d4ee799dd3361ff82de4a2c2d90b965e601493286fd39", "average_hash": "fffffbf1f1833006b00de410dd009f21ff99cbbfd05d001d080f07060188e109", "name": "Shadow Word: Death", "phash": "73d5e320fc489d8b37070e6e29336918c909c8635bc672bc53f8f3415ba9333b", "mana": 3, "whash": "fffffbf0f1833006b00fe710dd01bf81ffd9dbbfd05d00150a0f07860188e189", "card_id": "EX1_622"}, {"dhash": "eadfad21d30f8cff786a22d44588cb1010f721e9e705dc0b00ba7ce0c1472734", "average_hash": "071067203b607840fcc1fe83fc01fcc11c007b00fcb8fc3c00d0004000010000", "name": "Fel Cannon", "phash": "a78ff786fcf46c709e4013e92395d6e00ce3216386e08c683f4777006b1af7dc", "mana": 4, "whash": "0f106f707b607ce0fce1fee3ffe1fec91ec0ff80ffb8fcbc30f040c000810605", "card_id": "GVG_020"}, {"dhash": "889fddff00e823d95fe0bb003f817c0edb18bc63fec3f887319e69b9e17202ec", "average_hash": "07000306c10fe01ff01fe04f805f807f807f807fc07fc07fc07fe07fe03ff02f", "name": "Mulch", "phash": "3902ce2166f41e3fce18368886472398ac7129744ec51bfd734f73324de353fa", "mana": 3, "whash": "07000307c10fe01ff03fe04f807f807f807f807fc07fc07fc07fe07fe03ff02f", "card_id": "AT_044"}, {"dhash": "fabc2c6ae0cd54ff22bf2f3adfe4aee9fc10ef339cefb19b0327365f5ca8717b", "average_hash": "07800308030400046003f839f81eec1e6c2e6437843b1c1b1c38183c381e701b", "name": "Fiery War Axe", "phash": "cb3d7ed2fec39f5d7314469391e9a4b406ce98b109e266681b90ae4c9f627710", "mana": 2, "whash": "0781078f03068107f033f83bfc3ee81fee7e647f843b1c3b1c3c183c381ef91f", "card_id": "CS2_106"}, {"dhash": "a8815d7eb0dd81a4374b5784ec1ac974beed7563e9c6c221076718dc7e38f167", "average_hash": "07000740d304f90f7c0d1c0f0c0936437e007e403e8838143838003c3270f27f", "name": "Shadow Strike", "phash": "c7891eebfd147604bfe00c7c031e258cdbc08c781b65e78719c7d63839f236a6", "mana": 3, "whash": "072cc7c0d305fb0d7c0f1c0f1c4f36437e407e487e4c3e143a3c287c3a7cf27f", "card_id": "OG_176"}, {"dhash": "0a00cdfff130e7473ca5793ef259cc949b2167c7d80de370d00c8c3b1ceef09b", "average_hash": "070003000338087a0c7c0c6ed86ed06cd84ffc3ff83ff81e3e063c463818f81f", "name": "Eviscerate", "phash": "c9e3bccdfe3d5bd172cb5e32033d8d9c6b06a56299c1498559102e3179e09f4c", "mana": 2, "whash": "070003001338097e047c0c7f586fd06cdc5ffc3ffc3ffc1e3e063c4e3c1ef81f", "card_id": "EX1_124"}, {"dhash": "e8fdec83110d4bd8b421574fac9443299952389da16a4bc7a6b6c96c1298a530", "average_hash": "0799cfbffbbdf0b1c43c841c0d580d4f014f000621042031203128380138011c", "name": "Bloodfen Raptor", "phash": "6dbf7b63378ddf944e4a0ccd0ca008a692d0b6a4e7a0a6c97e13df10be02767a", "mana": 2, "whash": "0f99cfbffbbdf5b5e4bc8d5c0dd80dcf034f014f2144217160712cb829bc033c", "card_id": "CS2_172"}, {"dhash": "1a90e520844b38f17166e3c98403190b307631ec6f10fd09da3324e11d82bb1c", "average_hash": "e7ef85e791c7b8c19ac193c1b3c103c003800b801b8011c0a0c021c113007700", "name": "Twilight Whelp", "phash": "d7b399fa89de5d3effde59b0ebc008ca0004731823192a60ffc30ec3be0b3a51", "mana": 1, "whash": "efffefe79be7bce39fc1b3c1b7c103c003c00bc03bc03bc0a0c4b1c573807f00", "card_id": "BRM_004"}, {"dhash": "00800f1ff0b8e1653efb5500b81405611adaa464f30d0fdb1a3433e22198fc27", "average_hash": "ffff0ff083d1891b6e57ff97bdc3bd8b31093d081c3e8028c00050004000e107", "name": "Muster for Battle", "phash": "37559d883ec66c32cf1933724906628c695a09c5db693a515be5daac73da1e96", "mana": 3, "whash": "ffff0ff083d183dbfed7ffd7bfe3bfc71188388818bea038a00040004400f90f", "card_id": "GVG_061"}, {"dhash": "a0ddcd7a92802d3073ad84528fad7e7355f622ecc6401d893e3278e6e88c5b18", "average_hash": "4702d343f14bf849a899a3022f009b01ff01ff01ff80fc8078c0f8c05b003700", "name": "Ragnaros, Lightlord", "phash": "e30bb96cfce093cfbec0941c3e34382904984ec3669e7e74ff3132e50c3b415d", "mana": 8, "whash": "4f02d347f94ff849a89deb82af02bf01ff01ff81ff80fc80fcc1f8c07b003f00", "card_id": "OG_229"}, {"dhash": "122ed95392af2c1b72f685084ef38986131d632edec0349d7d32cae29105e6ff", "average_hash": "a7691b73bfd3fec0fec0fbc073c8f2c8e388c3884b0008c058c100c3060884ff", "name": "Mounted Raptor", "phash": "7769e72389b8be2c7d58f6983f3636151186664861b26671b39866f4c4b8cee0", "mana": 3, "whash": "af691f7bfe71fed1fed0fb803bc8f2c8f3c8e7884b80489058c110c30608c4ff", "card_id": "LOE_050"}, {"dhash": "28c74cbe91408a83b55c5339b4f07143ee8e87f56a8b99166e33a46c08893536", "average_hash": "779c9fbbdba74c2d344c700c7064307110300039983c3c19bc1fb81721020100", "name": "Stonesplinter Trogg", "phash": "eff37fc2db358fe5bea4c3306e58282800184692789e7fc87b02bb04bb21c7a3", "mana": 2, "whash": "7f9cbfbbdba7ccaf3c0d600c7964a17111784179997c3d59bc1fb89731920100", "card_id": "GVG_067"}, {"dhash": "0af1acc2c11c4b60b6954f61bc827105ce3ab065c84b10976336c76c1f99f130", "average_hash": "571f839fc39cc03f203fc03fc03fa033802f0c2d0c3900380033003300102110", "name": "Oasis Snapjaw", "phash": "a9fd5b627e259ecbd7d29cb04628c26890d4b3ed23cb6c92b33096246c08fbe9", "mana": 4, "whash": "771f8fbfc3bfd0bfa43fc43fe47fa077807fcc3d8c3b08390031003b31107110", "card_id": "CS2_119"}, {"dhash": "08c9ecf7d1e60b0936304834b2cc7839f96260c5478ac7159b37646dcc84313b", "average_hash": "f71fe187fb9bf83cf838fc087c003c381c3c181c18083800b81c981db00be11f", "name": "Murloc Tidehunter", "phash": "635b7346bb31ceb13cec3092726409c38134d2486bc9dfcc9f21e729f70cf334", "mana": 2, "whash": "f71beb87fb8ff83dfc38fc187c203c783c3c3c1c380c3800b83c981db00ff10f", "card_id": "EX1_506"}, {"dhash": "88f82d6502db058239ad61709282650dcef2f6c4ed0b92164f353c6cc0881931", "average_hash": "e707e36ff95ff04db84c6886f001a4018c21b83df83d783e380eb00180010100", "name": "Blingtron 3000", "phash": "fb9b7b8aff21ab483e8152046f4608310c44b3d9e3b01fe65f34db392c2336b3", "mana": 5, "whash": "ff1ff37ffbcff05fb86c6ca6f445a4010c25f83df83d783e381ab01391010100", "card_id": "GVG_119"}, {"dhash": "02e408ce81301b4d3e30598ea6785be5a6e25045ed829a1d67399ee3fc9fe17f", "average_hash": "f589fb11f81178107826cc017809781a78037800700018011802f800f10ff17f", "name": "Jeweled Scarab", "phash": "e3ed27f57175369973785a79391a12168de4c2e92159183b1c926bf244997b4c", "mana": 2, "whash": "fdbbfbb1f839f831783eec21f8497c5efc037800780018017803f801f10ff1ff", "card_id": "LOE_029"}, {"dhash": "c8fcad7152eb3d0c72d2e424d92d86db0e367fccf311008b183226e6d80e3338", "average_hash": "3703336639de78dfd8fadbb20fb001804386cb83f3bd61be089010d8230c020c", "name": "Kirin Tor Mage", "phash": "cd259b2ccc38fb30de5894e53bc513cd40a463999799da446bfb1ad5dcc06316", "mana": 3, "whash": "271733fe39def8ff58faddf20fb001804b86c383fbbd71be88d810d8230c030c", "card_id": "EX1_612"}, {"dhash": "3cdcac3cc1d74a9f377c49c1a4024a7196e227454c8a9144c7b93c73c18d3532", "average_hash": "07918bb38baff82dfc38f005f44138487840f80938091008100c400c01130117", "name": "Coldlight Seer", "phash": "e3d933d01c841ecb9f0c63a033b123c2e4e29e6d7bda9bf9fb98cc6326280c63", "mana": 3, "whash": "af91cfb39baffdadfc38fc3df5437d497948f94939491948100c408c019f0117", "card_id": "EX1_103"}, {"dhash": "8a98c56182c33c057b1fe78eccfc99f33fc6770cf29000090030c8e3d9012b37", "average_hash": "e7067146394e18cf18dfd99d3b9efd9ff99ffb9ff98ff887388010c000000202", "name": "Mana Wyrm", "phash": "c38b69fb9efaf3f6bbf426b53b0418a982a11b614c2073016b72361d9d25cec1", "mana": 1, "whash": "e706714e394e18de18df789d3f9efd9ff99ffb9ff98ff887388010c000000202", "card_id": "NEW1_012"}, {"dhash": "08f14d6612cb0d8c7b38e7420e14d8f824f151a625081ac024385df2b640691d", "average_hash": "f71ef97efb5cf85ef8de7ade3f1c1cd80c941c802d800100008068c078007800", "name": "Darnassus Aspirant", "phash": "638e693373c6be36af651c28cbc0519488f40b12c931db11fb981663ff23df43", "mana": 2, "whash": "f71efb7efb5ef87ff8de7ede3f9e1ed81e943e802f80050044c068c078007800", "card_id": "AT_038"}, {"dhash": "48f6edc8421185383a736de49ec86993d33a66f5004b03940438f973b2876d2f", "average_hash": "c70fe36bfbf9fc7dccf1ccb8d839d8398019841d0c0e2c0024000000c0034903", "name": "King Mukla", "phash": "3bdbb9b1d38cde3839c316584b8618b20294cd666eb473fdc725d6e4b3189438", "mana": 3, "whash": "ef1fef7ffbf9fc75dc71fcf9dd79d979813d051d0d0e2c40240080004103490b", "card_id": "EX1_014"}, {"dhash": "8af805c15a0a81877ffea4d2580cac28121acf6d9cc52b8b583691e5424b0736", "average_hash": "f70f755f611f081738b8f81d8ca58c8c041885b189b7e833c893c01780170208", "name": "Webspinner", "phash": "a9bdebdafbecfb667e5a13c2838120e4066099939693f24c4e4a7919a5a4bca4", "mana": 1, "whash": "f71f7d5f635f881738f8fe3fdca58cac051885b189b5e8b3c8b3c097c0170208", "card_id": "FP1_011"}, {"dhash": "0afa0dc492980d30fa0c2cf25c6cf3906e67dae0a4c5098b773fcee7bc4f713f", "average_hash": "f70ff31ffb19f81970107c302c3818b8083e003e00bc0838389b781ff81ff81f", "name": "Dreadsteed", "phash": "eb253733fbe99cb35b46c64c2e30d9e400e04cc6332826637f92d721bb499979", "mana": 4, "whash": "f71ff91ff959f839f0387c302c381cb80c3e043e00bc0838389b781ff81ff81f", "card_id": "AT_019"}, {"dhash": "f8d28c8b01b48b4a37a44cc3b2b864f59d0afb25628b9ad72736ec6cc8890932", "average_hash": "a71dfbac7ba7d033fc39b03d1c277004f80ef836483c70383830383018101910", "name": "Earthen Ring Farseer", "phash": "e3f57b58be256fdade82c6382c9b232032c81ccd9c74daec67e1929cccc00e87", "mana": 3, "whash": "af1dfbacfba7d0b3f839bc3d1c67f844f84ef83f781c74383830383019111910", "card_id": "CS2_117"}, {"dhash": "c181cd1f70e0a1ec93f70cdf1b3e3e607ff0ff849f197cc701ac3159e0b0f0e7", "average_hash": "7ffef3e0f9802c002e001e003e013e003e00fc13f833c031e070e0703030f067", "name": "Forbidden Flame", "phash": "a3545e29f40ab187cfe13de009c40e3c0c180a1d433f9ad5c3c3f3e17beef6b8", "mana": 0, "whash": "fffff3e0f9802d002e003e03fe033e00be01fc13f833e573e070fc703070f8ef", "card_id": "OG_086"}, {"dhash": "0af025c6c2988563bf766e2c9c5870b56622fdcc866b0107c2b61f652180452c", "average_hash": "e70c717e895c4879303e90bfd03ed03a503e083f803f3033001e601e00000100", "name": "Troggzor the Earthinator", "phash": "69e76feb769adbb87f6663709b1149c8a60c320838c316cd33b11c33c34a5cd9", "mana": 7, "whash": "ef1c7b7ecbdd4979307edcbfd87ed03ef03e983fa03fa83b003e701e01040100", "card_id": "GVG_118"}, {"dhash": "18dfacee619d8b60371b4ce2bb047e6dcad2d1e5a7c91317cd37326ccc99f937", "average_hash": "47104380a119c019e81ff830cc33043c0c2c04343c377833d838d839f815f91e", "name": "Psych-o-Tron", "phash": "a3973e58f3218ed45eb00687d3a40892696c93b9396796f1366bec98ede0133b", "mana": 5, "whash": "e7104391e199e03be83ffc33ec33043c0c2c24347c377c33f83cd83df815f91e", "card_id": "OG_145"}, {"dhash": "6ad9c53012e70dcf7932e6080c1df2326c21d07e80f94a83853f1ff73c486318", "average_hash": "9716d97e394f785ef8debade3f3e1cfe1cbc02bc01bc0d3c0cbc18dc501c401e", "name": "Knight of the Wild", "phash": "4da7f37afbeebedbbfe08f317e5041c10244e2203c489f849b8b50d1390867c7", "mana": 7, "whash": "b716d95e394f705ef8debcdebf3a1ebe1eb402bc01bc0c3c08bc187c501c401e", "card_id": "AT_041"}, {"dhash": "e2f0c8e5819b1b003e8e5e7eb9f84af1a3221c8d7102835d26bb4c669990e17d", "average_hash": "37bd399c381d381d18100c181c187c0ed8139803200c200830183018310c817f", "name": "Djinni of Zephyrs", "phash": "e3d987976dd97b589b966c3c3c0e96c591b8429a91a11298299e4e9a96c1afe5", "mana": 5, "whash": "7fbd3bbf3ebdbc3d1c381c781c7c7c5ef853b843b00e205830183018310cd1ff", "card_id": "LOE_053"}, {"dhash": "fad3ec2b11970a0fb53f5a7ebcfc7bf183e24e459d8a381647341f6c689e2138", "average_hash": "47107da03b80780078007c00fc10fc38fc10fc11fc033c033c20f830b8182918", "name": "Duskboar", "phash": "e71f7ef27c849bb52fa107661a8c20300cc83121b6849f89ff43cf49bb21fecc", "mana": 2, "whash": "4f907fa07ba07da07c007c00fd70fd78fd50fd53fd5b3c633c20f83039383918", "card_id": "OG_326"}, {"dhash": "3af16cc3d1b48b4d379a4d73bce27085c13a90453feb8cc47ebbcf611880f127", "average_hash": "c71fc79ff3b8d83ecc3e643e403ce038c0304820402c0000200b380538001104", "name": "Kobold Geomancer", "phash": "69d639e7dba95ed9bfd359b84c86828618a4c6024dc2f6c07b123704ef01fb66", "mana": 2, "whash": "4f9fefbffbbaddbecc3e643e657ce17ce1704960416c0558201f380739003904", "card_id": "CS2_142"}, {"dhash": "6af0edc1121b05767a04adfc5225a5d912636fceb915408b82360de410487bb4", "average_hash": "371f311e3b59b879b8b71ba5f8a1fce0fc91b893f9bf98bf08b4081000100000", "name": "One-eyed Cheat", "phash": "e399eb496cf8696d3e485cda19d092250256b390b299b6cdef4e9604b329f764", "mana": 2, "whash": "371b391e3b58b87db8f73da5fca1fca0fc91fcb3f9bfd8bf08bc087000100010", "card_id": "GVG_025"}, {"dhash": "4adbac3011cf4a9eb5e65351a402543febdacef539cb07148f393866228d353e", "average_hash": "9793cfa623ae302cd009b00f1047804680312c333c3f7c0c7c0cc00e6106711c", "name": "Crazed Alchemist", "phash": "dbdf9f717f068f15bcb452a2838930d006e31c60e698c6196f09fe08b60efee1", "mana": 2, "whash": "9f97cfae33ae34aed40bbc0f5547815681712d733d3f7c447c0cc80d6107711c", "card_id": "EX1_059"}, {"dhash": "e23ffdff30c083801f003f1c1e6078d0f2b4a56949538386760def1b3c36f8ec", "average_hash": "0700c30ff31ff01ff81f181f3c3e5c384e386a3c3a283c283e041c0edc0efc2f", "name": "Ancestor's Call", "phash": "cb5879b5db33f60836ce86f3a61e0cc626f8667bc3942bc48acd2b816b336321", "mana": 4, "whash": "0700c307f11ff81ff81f181f7c3e7c386e386a3c3a3c3c283e241c0e5c1efc3f", "card_id": "GVG_029"}, {"dhash": "0a80dd3ff0fd83ed3782ed04cf09f8b190cfb79ffe3efe651c0e22d9c0b01e3d", "average_hash": "071803c00383801fd01f80070017000a104c806fe07fe01fc07fc03d800df00f", "name": "Flare", "phash": "d9dfde643fd125bae66b3985c9a411d24b640e3c2956da2963d336807b41b730", "mana": 2, "whash": "673e07e08387cb1ff81f881f001f004f104f807fe07fe27fc27fc03dc03df01f", "card_id": "EX1_544"}, {"dhash": "0cf3ecc81181030e37774ef8bcc87039e31286e50ecbf9960836d76d88917120", "average_hash": "f71cfb9c3b9b38389838bc387c3c9c3c3c3d8c188c0dac07cc31f018300c1100", "name": "Gnomeregan Infantry", "phash": "6b736f41fa801b9a6f9221a61a3224399cb0dcf836cccbbce3e18673792cd2c6", "mana": 3, "whash": "f71cfbbc3bbf3838bc38fc387c3ebc7c3c7d8c389c0dbc07cc31f018310c1100", "card_id": "GVG_098"}, {"dhash": "c289edff20cf03341ed29f993f00f900e4094ef33847d28f615ec63919f266e0", "average_hash": "07000100f91efc1dfe1ffc3ffe1ff03bf073e073fc23cc798073107f207e403c", "name": "Fireball", "phash": "a9ac43236cdeb60df47c7da00f5e96e0268d4bd843f329bdc3111a874b695b8c", "mana": 4, "whash": "05000100f81efc1dfe1ffc1ffe1ff01bf073e073fe63cc7b807b007f207e003c", "card_id": "CS2_029"}, {"dhash": "8ce10cc311990a06373e4eecbd584309b4926ca5cb4a975742b60c68bc837136", "average_hash": "771f7b9e799c380e3814bc3a9c3a5c1e0c1f4c1e241a30103000300038003104", "name": "Lord of the Arena", "phash": "c36d6b86f39079cc1b8e836139040802c930da611dd2bf577f9e76d667a73c2c", "mana": 6, "whash": "7f1f7bbf7bbcb8bf3c14fc3bdc7bdc5f8c5f4c1f241b301a3010300039003904", "card_id": "CS2_162"}, {"dhash": "3ad9443241c74afcb5c85301a4824077ba0af6f54d4aba95c5311f6b7c84c13f", "average_hash": "c797cfa6e3ae6026e00fe00f804200410041003800080017000ff80ef100e103", "name": "Grotesque Dragonhawk", "phash": "d9fedff6fbd64fb73fe31be03b8923c8928826082c0827b0d988d9d099006f62", "mana": 7, "whash": "ef97cfafe3ae65a7f42fe01f854e81410141017d094e295f680ff88ef981c11b", "card_id": "OG_152"}, {"dhash": "d88ead1962335c64f2d8a525cb1b96672ccf481c9570728b65368fe13f836f1e", "average_hash": "2761a3f321c320c300c303c703c703d73396239703b703b700f700d703074706", "name": "Steward of Darkshire", "phash": "dda4f738235aa51d7f420cc73fc546e800d4dac963fc7b817fa7cac4008932d5", "mana": 3, "whash": "67f1abf3a3f330f700f703d783c703d723962397039703f700f701f703074706", "card_id": "OG_310"}, {"dhash": "acf00c8061fbcbff37ff5f09b2024c1df83ab0fd604a8395023905701a80352e", "average_hash": "ef9d1fbc03b905be8c5ff83fb16f054f016e096e0d0f0c4e0c00080001000100", "name": "Harvest Golem", "phash": "cd5b69813615160efed2d3902b9929a20ca69c9cbcd0b6d35bbb5a49bc58e6f0", "mana": 3, "whash": "ff9d5fbc03b907bfdcbffd7fb5ff05cf036e096e0d4f0d4e0c400c8001800320", "card_id": "EX1_556"}, {"dhash": "ca990d7390ee38997126e158802103031e8e695cf430c1c926334fe63e8ceb11", "average_hash": "f7c7f9e7fbc7f8cef8ce3bce73c263c003800380038001c000c009d06318d300", "name": "Wyrmrest Agent", "phash": "f7a5d3f223ea3c3cbb4c843c3bf00ca710c0630c12586e01df937eccbf077f44", "mana": 2, "whash": "fff7ffe7fbeff8cefadebfce7bc273c203c0038003c001c000c009d06318d710", "card_id": "AT_116"}, {"dhash": "81c16f41e0089b1f3e345d71b072256548c2b1044fec981962f62f68b681ff3f", "average_hash": "fffffbfffdfc1c70cef067f863f071f031ec11e800880080008001800100f10f", "name": "Forbidden Shaping", "phash": "655725a65b27584bfb5039901b480c728bb6068dc9ccce154bbd72ec73ef7a22", "mana": 0, "whash": "fffffbfffffc1df4def0e7f873f073f031ec11e801880880018001800180f18f", "card_id": "OG_101"}, {"dhash": "eafccce191990b3737fe4c9cb91877f1ec62cac5870b0d173c301860e0856136", "average_hash": "670d711f7b1f3819383bfc13fc07fc27fc33f831f831f8007800780010000000", "name": "Acidic Swamp Ooze", "phash": "e3ddc9897cd87ff29e96138e38a48488045206533e8a3f89bf01ff81f702736e", "mana": 2, "whash": "e71f799ffb9fb83b383bfc3ffc17fc37fc33f831f831f8107800780018000100", "card_id": "EX1_066"}, {"dhash": "8a9f3dfff098833516c02900bec078c1ff02ff85fc2bfac7fbff06120c34f861", "average_hash": "07008103c30fc83be83be80fc00fc001e003e007e007f07ffc7fbe7f1e7cfc3f", "name": "Freezing Trap", "phash": "a9737e9dcfe0fcbe5eb28de24294076121cc265909652e845b0c7d927b40df9a", "mana": 2, "whash": "07008303c31fc83be83be80fc00fe003e007e007e007f07ff87f9e7f3e7cfc3f", "card_id": "EX1_611"}, {"dhash": "f299ed67804d38f0f0c8e197832f07131e6e788cd1d0af891f337ee6fc8c7118", "average_hash": "070401e6d1c4f0c70087038703838183818f318f219ff09df899f9ddfb1ff31c", "name": "Auchenai Soulpriest", "phash": "d9a59c9667a24d7ffe46148dae8cc98c00d1a30903732bc377432ec5fe685e4d", "mana": 4, "whash": "074401c4d1cef0c7008703870383c183818fb18f719ff89bf89df9ddfb1ffb1c", "card_id": "EX1_591"}, {"dhash": "f838fdf300e747cc9f30334324884919b016c0ee06ddf8b63fe7030f6058f8bd", "average_hash": "07203bcc3f1e3fde0e7e867fc07ec05f00180010002008cfe0313c3c8c1df01f", "name": "Preparation", "phash": "492b4f20cfb4f6177a4139c81c402ea099cc0c36b3740d3b63599bf6cf8e367f", "mana": 0, "whash": "077c3ffc3f9e3fde0efea67f827ec05f0018081088a009eff831183c9c1df01f", "card_id": "EX1_145"}, {"dhash": "fa3d0cff0088178227203f93fc7ef231c0e0a2831dcee71ffc2f305c009c0170", "average_hash": "8503e10ff30ff01ff83e603d003f787f1c743c783806780ef83ff03ff01fe00f", "name": "Argent Lance", "phash": "e97f7e987ba5d6229e4a2ddc1618a4b187a602a53ce93c689fd0692597a95e42", "mana": 2, "whash": "8703e107f10ff01ff03ee03f003e787f1c7038783806781ef83ff03ff01fe00f", "card_id": "AT_077"}, {"dhash": "8a8ffde9e3d3cf7d1cbf1832310462d8c4b9fd61e1808b123627e04c85197836", "average_hash": "0700c3017f383c7cde77fe77ee326c3240004018381c3808b81190030813c007", "name": "Lock and Load", "phash": "a33c2fbefaf17ac97ac69a352f528da445d80cb66198064dcb413e045bc37e23", "mana": 2, "whash": "0708c3437ffc3dfcfe7ffe77fe776e3248004c593c1c3808b81990138813f00f", "card_id": "AT_061"}, {"dhash": "8aefcccbd1c40b2d37f24ee4b9987325e45aca45988b39174337466ef99cd339", "average_hash": "371c139fcb1fd83d6818983e9837b037f037fc3798179c1b881b0018a0188118", "name": "Ogre Magi", "phash": "a9757f866ecb9fe5fb308ea98f2666d82148c0700e923373ee88d34877063e26", "mana": 4, "whash": "371e199fd99ff83d783a983f9837f837f037fc379c379c1b981b00188018a118", "card_id": "CS2_197"}, {"dhash": "08cb0cd26184cb26b67b4ceeb998793996420e8578eae114c339a267449b1936", "average_hash": "f71c7396419a0038203c0c30dc11fc19bc19bc030007bc079c06880780078107", "name": "Razorfen Hunter", "phash": "83276dc87c86d36273e183a778922c9a635b38e93ef38fb33c0db0e4c3307c86", "mana": 3, "whash": "ff1c7bbe43ba00b8243ccc39fc71fc79bc5bbc038407bc07dc06c8178937a107", "card_id": "CS2_196"}, {"dhash": "0a00fdff600ca339dc7c9fa3390d797ac684c4012003cb849f0f763eec70f8cd", "average_hash": "07000300c31dc030c020e2196e3f3e3b7e7b3e7b1e2c1c0c1e6cdc38d048f84f", "name": "Sap", "phash": "2b49be247ef3ceb3d38c2ecc4a2224c882d20c7199147bcaab313e1b7ba5de56", "mana": 2, "whash": "07000300c31dc130c068e279ee3f3e7b7e7b7e7b1f2c1e1c1e6cfc38d868f84f", "card_id": "EX1_581"}, {"dhash": "cacf8c9f51358beb36d44d28b092638d863af9459fcb3c977536e96c5299b13a", "average_hash": "7710f191fb91f831fc3b983f80328002c0020830c831c839e03be83bf01ef11e", "name": "Doomsayer", "phash": "a375d7716339368c5e784cb00dc2868042cc3634ee807e267b46fd19ff037f63", "mana": 2, "whash": "7710f991fbb1f833fc3bf83f9032804240021831c839c83be83ff83ff81ef11f", "card_id": "NEW1_021"}, {"dhash": "a8d50d6b02c6c800f727acd80f21c1570c8f1b3c7ed0f00b8cb578e7d24d43ba", "average_hash": "d70f734771cd408c00841a80331800801084e186c18750aff09ce05fe01fe01f", "name": "Unbound Elemental", "phash": "7982eecb3baafbc6bb2086419e641c274cd419486f3ac7c596b374b4698c3aa9", "mana": 3, "whash": "f71ffb47f3cdc0ad009c1e80379802841184e186d38770aff0bfe05fe81fe41f", "card_id": "EX1_258"}, {"dhash": "828bedffd07e03fc5cfb3d467f1cd96066c1ddb3f3632606082dbc72c88c983b", "average_hash": "07008300e305f817f00ff807f83f3c3b6c1364377c3e7c3e3818f80cb00bf00f", "name": "Imp-losion", "phash": "e36cbe827e33ef6c9ee9272379c690388ecd61e3e0b01a8463236ed959b6120b", "mana": 4, "whash": "07008340e105f937f00ff807f87f7c3b7c137437fc7e7c3ef838b00cb00bf00f", "card_id": "GVG_045"}, {"dhash": "3cc36d2cd27988e279453b8ee3708b331647209a5af4d5482b3bdef494091bb2", "average_hash": "c717cb71cbe1b8c3bcc31ee11ec34fc333c31382038110c018c278c31b031601", "name": "Shieldmaiden", "phash": "93c93b12bb857ee66f0407c62ef0d16443725b0c0c75cfe1399c7ca7d1e99960", "mana": 6, "whash": "cf17cf71cbe7fce7bce39ee13ec34dc3bba31382038314811cc278c31b831c01", "card_id": "GVG_053"}, {"dhash": "02800d0bf0ec60f3e76e4f3d9ef928617a8204040e0df95ae6ffc0ef3783fe2f", "average_hash": "ffff07f003800000100ff00ff80efc2f7d3f30008001800360f3007f4d1eff0f", "name": "Echo of Medivh", "phash": "db5f7e033ca4b42c074ee9ce4b79a9f083d01ab673655b1d93f052e059821976", "mana": 4, "whash": "ffff0ff003800100100ff00ff91f7d2f7d3ff9018001808360f301ff4dbeff8f", "card_id": "GVG_005"}, {"dhash": "ecf88cc1018b0b1637654c3ca1187837e7cacc35a9e911932736de6c24997930", "average_hash": "f7177b9f3b9a303c383c3c1f9c0790301033243300380c381c38583818180118", "name": "Twilight Elder", "phash": "ad65d3306f25bfd4dae013261e1a28b824b33c631a3e9bbcfcc42665332fce24", "mana": 3, "whash": "ff177fbf3bbe3cbd3c3c3c3f9d4799703173253301380c381c38583819381918", "card_id": "OG_286"}, {"dhash": "acff2ce0413dcb56b6a84ee1b1c25311840690355c6bf855e1b383654b82d739", "average_hash": "d71feb9febbcb03ff037b03bd03fc817c003c00080000010000e00060004810b", "name": "Ironbeak Owl", "phash": "b935b3154f095eca1f8349c2b659c2989bac06332c61ecf9ceac66ce66a253ce", "mana": 3, "whash": "ff9fefbf6bbfb5bff43fb43ff57fc957c143c16081400154001e001e0184810a", "card_id": "CS2_203"}, {"dhash": "aae22dc4928b05d33b266e199ce678d961b2e6c4898b17076eb42c6fd28e2d39", "average_hash": "e70fe15c13581858983c34bc343c0c3c0c3c1c397c3878303030203c801d011d", "name": "Bloodmage Thalnos", "phash": "6db95f227b2ab3cc19ce7130c3c419c252466986b6645bc17f0659c3ef06ff0c", "mana": 2, "whash": "ef1ff77c5bfcb878bc7c3cfc347c0c7c0c3c5c397c3878313030a03c811d011d", "card_id": "EX1_012"}, {"dhash": "088eddf830e0ee9fdd7f3ac801fc76f6fbfcf7512f824ec4998ff313a3274e4e", "average_hash": "07000300010200040800fe00fe00fe20fe7dfe7dfe2dfe00fe607e0e4e4e5e0f", "name": "Explosive Shot", "phash": "0742fcbd5c6e6793a9cc362413b0654a24dbd844ad3219a863d9b7e663b55a97", "mana": 5, "whash": "070003000106002c1860fe60fe12fe31fe7dfe7dfe7dfe05fe60fe0e5e4e5e0f", "card_id": "EX1_537"}, {"dhash": "08d34c2e917a02f4b5f04f0cb9f47281e72a8ee57d4bb09601371e6c7883c13c", "average_hash": "f70efb8ff903f807f007dc07d402e432c0139003b807a80b081a70107004910c", "name": "Master of Ceremonies", "phash": "f37773826f929f0d7ff22c64193162e0316af3388cb11a3b3b8d70ccf828e2e0", "mana": 3, "whash": "f71efb8ff987f807f807dc1ff433fc32d03bb80ff80fa82b483a70087004910c", "card_id": "AT_117"}, {"dhash": "28f9ccd201250b8036ac5c5dbd346069e3e2cfc5b60b01175f33fc6efc8ff13f", "average_hash": "f70fd30ef107e01780118011343928003c003812383f381eb81ff01ff81ff81f", "name": "Raid Leader", "phash": "e3659e447b40cfe7f34a3c1819a132a01e1a3990bbf0b33cfb92d6b32d698999", "mana": 3, "whash": "f70ff10ff10ff01780155011b43938203c32f812783f3c1ef81ff01ff81ff81f", "card_id": "CS2_122"}, {"dhash": "f63df0e7a69b297b5546aa14dc2837c15c8caeea41153888e59a0c510286fd7f", "average_hash": "03a801df805fb03bd83f782f7c357d36701ea80fd8016801280734000000f9af", "name": "Fool's Bane", "phash": "c9cdfdc17c3b7e32ee5ed6de9b479a4b6d05263783b03498c5e29806018866a3", "mana": 5, "whash": "03ac03dfc05ff03bf8bf78af7cbd7dbff01eac0fda096a05680724000000f9bf", "card_id": "KAR_028"}, {"dhash": "08fe6d7882dd05a2bb616c1e9b9c7421e0c20664f9cbd3973f37706cc2908539", "average_hash": "f707e367914fb04fe05f0cbfdc3fbc3d3c312001180f381dd839c801c001c119", "name": "Stalagg", "phash": "a99bdb203e48c640b61405865c8c3258c371c3e33be3d6dddfd9ede36c4e4d06", "mana": 5, "whash": "ef07e75fd14ff05fe03f0cbfdc3fbc3d3c3d2001181f3c1dd839c811c001c119", "card_id": "FP1_014"}, {"dhash": "9cfe2de02203452dbacbe09487694f3393f23ea56c0adc97293783688081053b", "average_hash": "4709e37f03de007840c348837801980bb80bbc0ffc0fcc1fcc3fc00740030103", "name": "The Skeleton Knight", "phash": "9bcd6c887c242b33de09318486cc92302ced6663e6329edddea7cdcd4c23d6c9", "mana": 6, "whash": "4f1de77f03fe0478407348e37941f849bd0bfc0ffd0fcc3fd83fd007c1030103", "card_id": "AT_128"}, {"dhash": "f8affddea1a8ef53dca53c4a733e45dcc63999f129e37ee6d18c3339e761decf", "average_hash": "07008301c111cc7c3e663e731e771e605e621e600e618e718e60ce60ce60fe46", "name": "Charge", "phash": "9722e8357873bed14648be10cf385642691424de2cb3336f27cd9641cbd9db66", "mana": 3, "whash": "07008303d139ec7cfe66be771e771e605e621e611e618e71ce61ce60ce60fe47", "card_id": "CS2_103"}, {"dhash": "bcce6c1981fc4ab7b54c5b01be6878b5b75221e5480ab314e7b9de6f698dcd38", "average_hash": "87118b93cb83f02ff02ff80f783fd83ffc1fb40a180808033009b01e801ec919", "name": "Gadgetzan Auctioneer", "phash": "cb699f943e21de283f27cd80020861b034912cf3c6f83b3cc7c336dbcf26ced9", "mana": 6, "whash": "87158bbbcba7d0aff02ff80f3c7fd47ffc5fb40e18082c033009b01e811fc919", "card_id": "EX1_095"}, {"dhash": "eadc6d83d3670c9b772c2ef8588cf1f800d307016f049c083f39fee26c41b13d", "average_hash": "e701610afb007846b838b8b87c301cb07c007d00f880f800f8c0f800f800f801", "name": "Tiny Knight of Evil", "phash": "e387bdbcfbe25ece8e9c2ca53911c36188c09807738c5603ff439e113e26f338", "mana": 2, "whash": "e7137b5efb42f86efcbabeb87cb01eb07c807f80fc80fc00fcc0f800f881f801", "card_id": "AT_021"}, {"dhash": "03c00d18e0f0dbc2fc07d11f82fc7cf5a37a0ecd1af747ce99382360fcc978bf", "average_hash": "ffff0bf903cc043e067e03fc05ec03f401f99dffcbe7eee0fcc019801981f9a7", "name": "Confuse", "phash": "9d7f2d88d3cd33abe1cd663209528b6921f0caf44b7673d87a817b227e929308", "mana": 2, "whash": "ffff0bf303dc047e027e03fc07ec03f401f185ff82e7eee0ecc018801980f9a7", "card_id": "AT_016"}, {"dhash": "f83ffdffe17fc3df34a38dcc7b90d7213fc3e6069d0fe03c903f087e11f9eee6", "average_hash": "07000300f91ffc37fc17fc17fa17f00ff01ff03fe07fe01fc03fc03f6036e00f", "name": "Vanish", "phash": "e9b417a13c4e7c4ade491ea08724203c187294c4afd10e1ccb7ba0bf6bedff4a", "mana": 6, "whash": "07000300f91ffc37fc17fc17fa17f00ff01ff03fe07fe01fc03fc03e4036a00f", "card_id": "NEW1_004"}, {"dhash": "6cdccc3891d90ab335e6538ea7184eb1986233c5cc8a1914f338c6739985f732", "average_hash": "b7137ba71ba798079c0f9c0f9c0fb80fb80cbc0dbc099c011800100000000100", "name": "Volcanic Drake", "phash": "c373db90bc84e7216f2498243b2d203b02b38c6d0b9bfcb0db339e1db989ceb9", "mana": 6, "whash": "bf977fa73ba79c8f9c0f9c0fbd4fbd4fbd4cfd4dbd49bd411c01108801800100", "card_id": "BRM_025"}, {"dhash": "1adeac2d41facae5b54e5239a7425e17b9eee6dd99a97a477bb6846ce1990733", "average_hash": "07900fa183a100a7004f600370422077907d10391033103b90339032c1114113", "name": "Evil Heckler", "phash": "b9f5df1a3e17aeb16622e364cbcc08e389c11c8ce3e033c9fe2433436c08cbf3", "mana": 4, "whash": "0f900fa183a544a7048fe5437147617f937f11791173917fd073f4bac1b1c33b", "card_id": "AT_114"}, {"dhash": "567629cd84927f35d2cfa4ca5934022120d6c6b809c55b18b33004e1d840733f", "average_hash": "fb7ff9fdd8bbf4fbbe99beb93fb31d000d1007b00fa038803880188012000638", "name": "Cloaked Huntress", "phash": "8719399979dbf3dbfe3ec3333d30bcb4c3eb960e8d548264286e12418b079c94", "mana": 3, "whash": "ff7ff9ffd8bfd4ffbef9bebd2fb31d000d3007f00fa03c803c80188012000638", "card_id": "KAR_006"}, {"dhash": "84dd4d6bd29e0c693bfa76f08864539927220c04f80aa04512b8a5734387873c", "average_hash": "f703fb67fbc9fcd1fcd0fcda7c51ec11dc031803000100008000000001000100", "name": "Mukla, Tyrant of the Vale", "phash": "b38fe38cfc129e409f942614cb218260202526bc92c847ed77e77ef36f39363b", "mana": 6, "whash": "ff53ff6ffbe9fef9fcf0fcdafd5bedd1ddc3dd0399410140800000800180012c", "card_id": "OG_122"}, {"dhash": "48faec6c21914b66b7cc4a89b4126827c64e0c1d48ead04423b80670cb83b72c", "average_hash": "9fbe8fbfb7bfb5ffb4ff337f33ff33f81373036b03e901e0044001c005800320", "name": "Knife Juggler", "phash": "dd79c9714f175e87ffa6462209829064242526f19b48db407f127f03ff01bf6c", "mana": 2, "whash": "9ffe0fbfb3bfb7ffb4ff317f33ff11f8137303eb03c901e00440058005800320", "card_id": "NEW1_019"}, {"dhash": "6cdf4d01d24a0894fb48b4b7e63e8bfb1ce5798ae314c3892e3a8cf11c037bae", "average_hash": "f717fb6f7bc068c4205c024a06c839ce3b8e3b8e7a8f200d30c770c23a063806", "name": "Sea Reaver", "phash": "cb8a3f0bb3f45b23cf623c7808e0c930243966981e97769d3cafd6b46c694749", "mana": 6, "whash": "f717fb6f7bc178e5245c064206cc3dce3b8e3b8e7b8f208d28c770c23a863c06", "card_id": "AT_130"}]}

