"""
Based on django templateloader
"""
import re

from django.conf import settings
from django.contrib.staticfiles import finders
from django.template.base import TemplateDoesNotExist
from django.template.loader import BaseLoader
from django.utils._os import safe_join

class Loader(BaseLoader):
    is_usable = True

    included_files = []
    processed_files = []
    PLACEHOLDER = '<!-- INCLUDE_JS_HERE -->'
    JS_RE = re.compile('^\s*//= include ([^#]*)(#.*)?$', re.I)

    def __init__(self, *args, **kwargs):
        self.included_files = []
        self.processed_files = []
        super(Loader, self).__init__(*args, **kwargs)


    def get_template_sources(self, template_name, template_dirs=None):
        """
        Returns the absolute paths to "template_name", when appended to each
        directory in "template_dirs". Any paths that don't lie inside one of the
        template dirs are excluded from the result set, for security reasons.
        """
        if not template_dirs:
            template_dirs = settings.TEMPLATE_DIRS
        for template_dir in template_dirs:
            try:
                yield safe_join(template_dir, template_name)
            except UnicodeDecodeError:
                # The template dir name was a bytestring that wasn't valid UTF-8.
                raise
            except ValueError:
                # The joined path was located outside of this particular
                # template_dir (it might be inside another one, so this isn't
                # fatal).
                pass

    def load_template_source(self, template_name, template_dirs=None):
        tried = []
        for filepath in self.get_template_sources(template_name, template_dirs):
            try:
                file = open(filepath)
                try:
                    return (self.process_template(file.read().decode(settings.FILE_CHARSET)), filepath)
                finally:
                    file.close()
            except IOError:
                tried.append(filepath)
        if tried:
            error_msg = "Tried %s" % tried
        else:
            error_msg = "Your TEMPLATE_DIRS setting is empty. Change it to point to at least one template directory."
        raise TemplateDoesNotExist(error_msg)
    load_template_source.is_usable = True


    def process_template(self, content):
        self.find_js(content)
        indent = self.get_include_placeholder_indent(content)
        js_libs_include = ('<script type="text/javascript" src="{{ STATIC_URL }}%s"></script>' % filepath for filepath in self.included_files)
        content = content.replace(self.PLACEHOLDER, ('\n%s' % indent).join(js_libs_include))
        return content


    def find_static_file(self, filename):
        found_file = None
        for finder in finders.get_finders():
            found_file = finder.find(filename)
            if found_file is not None:
                break
        if found_file is None:
            raise Exception('RequireJS: Cannot find file %s to import' % filename)
        return found_file

    def find_js(self, content, before_filename=None):
        for line in content.split('\n'):
            match = self.JS_RE.match(line)
            if match:
                filename = match.group(1).strip()
                if filename not in self.included_files:
                    if before_filename is None:
                        self.included_files.append(filename)
                    else:
                        self.included_files.insert(self.included_files.index(before_filename), filename)

                    if filename not in self.processed_files:
                        static_filename = self.find_static_file(filename)
                        file = open(static_filename)
                        try:
                            self.find_js(file.read().decode(settings.FILE_CHARSET), filename)
                        finally:
                            file.close()
                        self.processed_files.append(filename)




    PLACEHOLDER_INDENT_RE = re.compile('^(\s*)<!-- INCLUDE_JS_HERE -->\s*', re.I|re.M)
    def get_include_placeholder_indent(self, content):
        match = self.PLACEHOLDER_INDENT_RE.match(content)
        if match:
            return match.group(1).strip('\n')
        return ''

_loader = Loader()
