# Copyright (c) 2015, The MITRE Corporation. All rights reserved.
# See LICENSE.txt for complete terms.

import stix
import stix.bindings.exploit_target as exploit_target_binding
from stix.common import StructuredTextList


class Weakness(stix.Entity):
    """Implementation of STIX ``Weakness``.

    Args:
        cwe_id(optional): Common Weakness Enumeration value as a string
        description (optional): A string description.

    """
    _binding = exploit_target_binding
    _binding_class = _binding.WeaknessType
    _namespace = "http://stix.mitre.org/ExploitTarget-1"

    def __init__(self, description=None, cwe_id=None):
        self.description = description
        self.cwe_id = cwe_id

    @property
    def cwe_id(self):
        """
        Common Weakness Enumeration value as a string
        """
        return self._cwe_id

    @cwe_id.setter
    def cwe_id(self, value):
        self._cwe_id = value

    @property
    def description(self):
        """A single description about the contents or purpose of this object.

        Default Value: ``None``

        Note:
            If this object has more than one description set, this will return
            the description with the lowest ordinality value.

        Returns:
            An instance of
            :class:`.StructuredText`

        """
        return next(iter(self.descriptions), None)

    @description.setter
    def description(self, value):
        self.descriptions = value

    @property
    def descriptions(self):
        """A :class:`.StructuredTextList` object, containing descriptions about
        the purpose or intent of this object.

        This is typically used for the purpose of providing multiple
        descriptions with different classificaton markings.

        Iterating over this object will yield its contents sorted by their
        ``ordinality`` value.

        Default Value: Empty :class:`.StructuredTextList` object.

        Note:
            IF this is set to a value that is not an instance of
            :class:`.StructuredText`, an effort will ne made to convert it.
            If this is set to an iterable, any values contained that are not
            an instance of :class:`.StructuredText` will be be converted.

        Returns:
            An instance of
            :class:`.StructuredTextList`

        """
        return self._description

    @descriptions.setter
    def descriptions(self, value):
        self._description = StructuredTextList(value)

    def add_description(self, description):
        """Adds a description to the ``descriptions`` collection.

        This is the same as calling "foo.descriptions.add(bar)".

        """
        self.descriptions.add(description)

    def to_obj(self, return_obj=None, ns_info=None):
        super(Weakness, self).to_obj(return_obj=return_obj, ns_info=ns_info)

        if not return_obj:
            return_obj = self._binding_class()

        if self.descriptions:
            return_obj.Description = self.descriptions.to_obj(ns_info=ns_info)
        return_obj.CWE_ID = self.cwe_id

        return return_obj

    @classmethod
    def from_obj(cls, obj, return_obj=None):
        if not obj:
            return None
        if not return_obj:
            return_obj = cls()

        return_obj.descriptions = StructuredTextList.from_obj(obj.Description)
        return_obj.cwe_id = obj.CWE_ID

        return return_obj

    def to_dict(self):
        return super(Weakness, self).to_dict()

    @classmethod
    def from_dict(cls, dict_repr, return_obj=None):
        if not dict_repr:
            return None
        if not return_obj:
            return_obj = cls()

        return_obj.descriptions = StructuredTextList.from_dict(dict_repr.get('description'))
        return_obj.cwe_id = dict_repr.get('cwe_id')

        return return_obj


class _Weaknesses(stix.TypedList):
    _contained_type = Weakness
