"""
Copyright 2018 Cognitive Scale, Inc. All Rights Reserved.

Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

   http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.
"""

import dill
import tempfile
from zipfile import ZipFile
from contextlib import closing
from cortex.action import Action
from cortex_client import ActionClient, ConnectionClient
from cortex.utils import md5sum, log_message
from cortex.logger import getLogger

log = getLogger(__name__)

_func_alias = 'f'

_main_shim = """# Auto-generated by the Cortex Python SDK
def main(params):
    return {func_name}(params)
""".format(func_name=_func_alias)

_requirements_txt = """# Auto-generated by the Cortex Python SDK
%s

"""

_sklearn_predict_shim = """# Auto-generated by the Cortex Python SDK
import numpy as np
import pandas as pd
import dill
from cortex import Cortex, Message
from cortex_client import ConnectionClient

_model_key = "{model_key}"


def main(params):
    msg = Message(params)
    client = Cortex.client(api_endpoint=msg.apiEndpoint, token=msg.token)
    content_client = ConnectionClient(msg.apiEndpoint, 2, msg.token)
    
    model_obj = dill.load(content_client.download(_model_key))
    model = model_obj['model']
    target = model_obj.get('target', 'y')
    x_pipe = model_obj.get('x_pipe')
    y_pipe = model_obj.get('y_pipe')
    
    x = msg.as_pandas(client)
    if x_pipe:
        x = x_pipe.run(x)
    
    y = model.predict(x)
    if y_pipe:
        y = pd.DataFrame(data=y, columns=[target])
        y = y_pipe.run(y)
        y = y[target].values
    
    
    return Message.with_payload({{"columns": [target], "values": y.tolist()}}).to_params()
"""

# Configure dill to 'recurse' dependencies when dumping objects
dill.settings['recurse'] = True


class ActionBuilder:

    def __init__(self, name: str, client: ActionClient, camel='1.0.0'):
        self._camel = camel
        self._name = name
        self._image = None
        self._client = client
        self._kind = None
        self._code_s = None
        self._code_f = None
        self._model = None
        self._requirements = []

    def docker(self, image: str):
        self._image = image
        self._kind = 'blackbox'
        return self

    def kind(self, kind: str):
        self._kind = kind
        return self

    def from_function(self, fn):
        self._code_s = dill.source.dumpsource(fn, alias=_func_alias, enclose=False)
        self._code_s += "\n%s" % _main_shim
        self._kind = 'python:3'
        return self

    def from_archive(self, archive, kind='python:3'):
        self._code_f = archive
        self._kind = kind
        return self

    def from_source(self, source_str: str):
        self._code_s = source_str
        self._kind = 'python:3'
        return self

    def from_model(self, model, model_type="sklearn", x_pipeline=None, y_pipeline=None, target='y'):
        self._model = {'model': model, 'type': model_type, 'x_pipe': x_pipeline, 'y_pipe': y_pipeline, 'target': target}
        self._kind = 'python:3'
        return self

    def with_pip_freeze(self):
        self._requirements = list(self._pip_freeze())
        return self

    def with_requirements(self, requirements: list):
        self._requirements = [str(r) for r in requirements]
        return self

    def build(self, **kwargs) -> Action:
        _simulate = kwargs.get('simulate', False)

        if self._code_s:
            # Deploy from code string
            with tempfile.NamedTemporaryFile(mode='w+b', suffix='.zip') as f:
                self._prepare_source_archive(f, self._code_s, _simulate)
                if not _simulate:
                    self._client.deploy_action(self._name, self._kind, self._image, f.name)
        elif self._code_f:
            # Deploy from code archive
            if not _simulate:
                self._client.deploy_action(self._name, self._kind, self._image, self._code_f)
        elif self._model:
            # Deploy from ML model object
            self._deploy_model(_simulate)
        else:
            # Deploy docker image
            if not _simulate:
                self._client.deploy_action(self._name, self._kind, self._image)

        return Action.get_action(self._name, self._client._serviceconnector)

    def _prepare_source_archive(self, f, source_code, simulate=False):
        with ZipFile(f, 'w') as z:
            reqs = _requirements_txt % ('\n'.join(self._requirements))
            if simulate:
                print('--- requirements.txt ---')
                print(reqs)
                print('-'*25)
            z.writestr('__main__.py', source_code)
            z.writestr('requirements.txt', reqs)

    def _deploy_model(self, simulate=False):
        content_client = ConnectionClient(
            self._client._serviceconnector.url,
            2,
            self._client._serviceconnector.token
        )

        # Upload model version to managed content
        with tempfile.NamedTemporaryFile(mode='w+b') as temp:
            with closing(open(temp.name, 'wb')) as f:
                dill.dump(self._model, f)

            md5 = md5sum(temp.name)
            model_key = '/cortex/models/%s/%s.pk' % (self._name, md5)

            if not content_client.exists(model_key):
                log_message('model version not found, pushing to remote storage: ' + model_key, log)
                with closing(open(temp.name, 'rb')) as f:
                    content_client.upload(key=model_key, stream_name='model', content_type='application/octet-stream', stream=f)

        # Deploy predict action
        with tempfile.NamedTemporaryFile(mode='w+b', suffix='.zip') as f:
            _source = _sklearn_predict_shim.format(model_key=model_key[1:])
            if simulate:
                print('--- __main__.py ---')
                print(_source)
                print('-' * 25)
            self._prepare_source_archive(f, _source, simulate)
            if not simulate:
                self._client.deploy_action(self._name, self._kind, self._image, f.name)


    @staticmethod
    def _pip_freeze():
        try:
            from pip._internal.operations import freeze
        except ImportError:  # pip < 10.0
            from pip.operations import freeze

        return freeze.freeze()
