# Author: Mr_Orange <mr_orange@hotmail.it>
# URL: http://github.com/SiCKRAGETV/SickRage/
#
# This file is part of SickRage.
#
# SickRage is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# SickRage is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with SickRage.  If not, see <http://www.gnu.org/licenses/>.

from __future__ import unicode_literals

import json
from base64 import b64encode

import sickrage
from sickrage.clients import GenericClient


class TransmissionAPI(GenericClient):
    def __init__(self, host=None, username=None, password=None):

        super(TransmissionAPI, self).__init__('Transmission', host, username, password)

        if not self.host.endswith('/'):
            self.host += '/'

        if self.rpcurl.startswith('/'):
            self.rpcurl = self.rpcurl[1:]

        if self.rpcurl.endswith('/'):
            self.rpcurl = self.rpcurl[:-1]

        self.url = self.host + self.rpcurl + '/rpc'

    def _get_auth(self):
        self.response = sickrage.srCore.srWebSession.post(self.url,
                                                          data=json.dumps({'method': 'session-get',}),
                                                          timeout=120,
                                                          auth=(self.username, self.password),
                                                          raise_exceptions=False,
                                                          verify=bool(sickrage.srCore.srConfig.TORRENT_VERIFY_CERT))

        # get auth session header
        self.auth = self.response.headers['x-transmission-session-id'] if self.response is not None else None
        if not self.auth:
            return

        # Validating Transmission authorization
        self._request(method='post',
                      data=json.dumps({'arguments': {}, 'method': 'session-get'}),
                      headers={'x-transmission-session-id': self.auth})

        return self.auth

    def _request(self, *args, **kwargs):
        kwargs.setdefault('headers', {}).update({'x-transmission-session-id': self.auth})
        return super(TransmissionAPI, self)._request(*args, **kwargs)

    def _add_torrent_uri(self, result):

        arguments = {'filename': result.url,
                     'paused': 1 if sickrage.srCore.srConfig.TORRENT_PAUSED else 0,
                     'download-dir': sickrage.srCore.srConfig.TORRENT_PATH}

        post_data = json.dumps({'arguments': arguments,
                                'method': 'torrent-add'})

        if self._request(method='post', data=post_data):
            return self.response.json()['result'] == "success"

    def _add_torrent_file(self, result):

        arguments = {'metainfo': b64encode(result.content),
                     'paused': 1 if sickrage.srCore.srConfig.TORRENT_PAUSED else 0,
                     'download-dir': sickrage.srCore.srConfig.TORRENT_PATH}

        post_data = json.dumps({'arguments': arguments,
                                'method': 'torrent-add'})

        if self._request(method='post', data=post_data):
            return self.response.json()['result'] == "success"

    def _set_torrent_ratio(self, result):

        ratio = None
        if isinstance(result.ratio, (int, long)):
            ratio = result.ratio

        mode = 0
        if ratio:
            if float(ratio) == -1:
                ratio = 0
                mode = 2
            elif float(ratio) >= 0:
                ratio = float(ratio)
                mode = 1  # Stop seeding at seedRatioLimit

        arguments = {'ids': [result.hash],
                     'seedRatioLimit': ratio,
                     'seedRatioMode': mode}

        post_data = json.dumps({'arguments': arguments,
                                'method': 'torrent-set'})

        if self._request(method='post', data=post_data):
            return self.response.json()['result'] == "success"

    def _set_torrent_seed_time(self, result):

        if sickrage.srCore.srConfig.TORRENT_SEED_TIME and sickrage.srCore.srConfig.TORRENT_SEED_TIME != -1:
            time = int(60 * float(sickrage.srCore.srConfig.TORRENT_SEED_TIME))
            arguments = {'ids': [result.hash],
                         'seedIdleLimit': time,
                         'seedIdleMode': 1}

            post_data = json.dumps({'arguments': arguments,
                                    'method': 'torrent-set'})

            if self._request(method='post', data=post_data):
                return self.response.json()['result'] == "success"
        else:
            return True

    def _set_torrent_priority(self, result):

        arguments = {'ids': [result.hash]}

        if result.priority == -1:
            arguments['priority-low'] = []
        elif result.priority == 1:
            # set high priority for all files in torrent
            arguments['priority-high'] = []
            # move torrent to the top if the queue
            arguments['queuePosition'] = 0
            if sickrage.srCore.srConfig.TORRENT_HIGH_BANDWIDTH:
                arguments['bandwidthPriority'] = 1
        else:
            arguments['priority-normal'] = []

        post_data = json.dumps({'arguments': arguments,
                                'method': 'torrent-set'})

        if self._request(method='post', data=post_data):
            return self.response.json()['result'] == "success"


api = TransmissionAPI()
