# Copyright (c) 2011-2020, Manfred Moitzi
# License: MIT License
from typing import Tuple
import math

RGB = Tuple[int, int, int]


def int2rgb(value: int) -> RGB:
    """ Split RGB integer `value` into ``(r, g, b)`` tuple. """
    return (
        (value >> 16) & 0xFF,  # red
        (value >> 8) & 0xFF,  # green
        value & 0xFF,  # blue
    )


def rgb2int(rgb: RGB) -> int:
    """ Combined integer value from ``(r, g, b)`` tuple."""
    r, g, b = rgb
    return ((int(r) & 0xff) << 16) | ((int(g) & 0xff) << 8) | (int(b) & 0xff)


def aci2rgb(index: int) -> RGB:
    """ Convert :ref:`ACI` into ``(r, g, b)`` tuple, based on default AutoCAD colors. """
    if index < 1:
        raise IndexError(index)
    return int2rgb(DXF_DEFAULT_COLORS[index])


def luminance(color: RGB) -> float:
    """ Returns perceived luminance for a RGB color in the range [0.0, 1.0]  from dark to light. """
    r, g, b = color
    r = float(r) / 255.0
    g = float(g) / 255.0
    b = float(b) / 255.0
    return round(math.sqrt(0.299 * r ** 2 + 0.587 * g ** 2 + 0.114 * b ** 2), 3)


# color codes are 1-indexed so an additional entry was put in the 0th position
# different plot styles may choose different colors for the same code
# from ftp://ftp.ecn.purdue.edu/jshan/86/help/html/import_export/dxf_colortable.htm
# alternative color tables can be found at:
#  - http://www.temblast.com/songview/color3.htm
#  - http://gohtx.com/acadcolors.php

# modelspace color palette for AutoCAD 2020
DXF_DEFAULT_COLORS = [
    0x000000, 0xff0000, 0xffff00, 0x00ff00, 0x00ffff, 0x0000ff, 0xff00ff,
    0xffffff, 0x808080, 0xc0c0c0, 0xff0000, 0xff7f7f, 0xa50000, 0xa55252,
    0x7f0000, 0x7f3f3f, 0x4c0000, 0x4c2626, 0x260000, 0x261313, 0xff3f00,
    0xff9f7f, 0xa52900, 0xa56752, 0x7f1f00, 0x7f4f3f, 0x4c1300, 0x4c2f26,
    0x260900, 0x261713, 0xff7f00, 0xffbf7f, 0xa55200, 0xa57c52, 0x7f3f00,
    0x7f5f3f, 0x4c2600, 0x4c3926, 0x261300, 0x261c13, 0xffbf00, 0xffdf7f,
    0xa57c00, 0xa59152, 0x7f5f00, 0x7f6f3f, 0x4c3900, 0x4c4226, 0x261c00,
    0x262113, 0xffff00, 0xffff7f, 0xa5a500, 0xa5a552, 0x7f7f00, 0x7f7f3f,
    0x4c4c00, 0x4c4c26, 0x262600, 0x262613, 0xbfff00, 0xdfff7f, 0x7ca500,
    0x91a552, 0x5f7f00, 0x6f7f3f, 0x394c00, 0x424c26, 0x1c2600, 0x212613,
    0x7fff00, 0xbfff7f, 0x52a500, 0x7ca552, 0x3f7f00, 0x5f7f3f, 0x264c00,
    0x394c26, 0x132600, 0x1c2613, 0x3fff00, 0x9fff7f, 0x29a500, 0x67a552,
    0x1f7f00, 0x4f7f3f, 0x134c00, 0x2f4c26, 0x092600, 0x172613, 0x00ff00,
    0x7fff7f, 0x00a500, 0x52a552, 0x007f00, 0x3f7f3f, 0x004c00, 0x264c26,
    0x002600, 0x132613, 0x00ff3f, 0x7fff9f, 0x00a529, 0x52a567, 0x007f1f,
    0x3f7f4f, 0x004c13, 0x264c2f, 0x002609, 0x135817, 0x00ff7f, 0x7fffbf,
    0x00a552, 0x52a57c, 0x007f3f, 0x3f7f5f, 0x004c26, 0x264c39, 0x002613,
    0x13581c, 0x00ffbf, 0x7fffdf, 0x00a57c, 0x52a591, 0x007f5f, 0x3f7f6f,
    0x004c39, 0x264c42, 0x00261c, 0x135858, 0x00ffff, 0x7fffff, 0x00a5a5,
    0x52a5a5, 0x007f7f, 0x3f7f7f, 0x004c4c, 0x264c4c, 0x002626, 0x135858,
    0x00bfff, 0x7fdfff, 0x007ca5, 0x5291a5, 0x005f7f, 0x3f6f7f, 0x00394c,
    0x26427e, 0x001c26, 0x135858, 0x007fff, 0x7fbfff, 0x0052a5, 0x527ca5,
    0x003f7f, 0x3f5f7f, 0x00264c, 0x26397e, 0x001326, 0x131c58, 0x003fff,
    0x7f9fff, 0x0029a5, 0x5267a5, 0x001f7f, 0x3f4f7f, 0x00134c, 0x262f7e,
    0x000926, 0x131758, 0x0000ff, 0x7f7fff, 0x0000a5, 0x5252a5, 0x00007f,
    0x3f3f7f, 0x00004c, 0x26267e, 0x000026, 0x131358, 0x3f00ff, 0x9f7fff,
    0x2900a5, 0x6752a5, 0x1f007f, 0x4f3f7f, 0x13004c, 0x2f267e, 0x090026,
    0x171358, 0x7f00ff, 0xbf7fff, 0x5200a5, 0x7c52a5, 0x3f007f, 0x5f3f7f,
    0x26004c, 0x39267e, 0x130026, 0x1c1358, 0xbf00ff, 0xdf7fff, 0x7c00a5,
    0x9152a5, 0x5f007f, 0x6f3f7f, 0x39004c, 0x42264c, 0x1c0026, 0x581358,
    0xff00ff, 0xff7fff, 0xa500a5, 0xa552a5, 0x7f007f, 0x7f3f7f, 0x4c004c,
    0x4c264c, 0x260026, 0x581358, 0xff00bf, 0xff7fdf, 0xa5007c, 0xa55291,
    0x7f005f, 0x7f3f6f, 0x4c0039, 0x4c2642, 0x26001c, 0x581358, 0xff007f,
    0xff7fbf, 0xa50052, 0xa5527c, 0x7f003f, 0x7f3f5f, 0x4c0026, 0x4c2639,
    0x260013, 0x58131c, 0xff003f, 0xff7f9f, 0xa50029, 0xa55267, 0x7f001f,
    0x7f3f4f, 0x4c0013, 0x4c262f, 0x260009, 0x581317, 0x000000, 0x656565,
    0x666666, 0x999999, 0xcccccc, 0xffffff,
]


# paperspace color palette for AutoCAD 2020
DXF_DEFAULT_PAPERSPACE_COLORS = [
    0x000000, 0xff0000, 0xffff00, 0x00ff00, 0x00ffff, 0x0000ff, 0xff00ff,
    0x000000, 0x808080, 0xc0c0c0, 0xff0000, 0xff7f7f, 0xa50000, 0xa55252,
    0x7f0000, 0x7f3f3f, 0x4c0000, 0x4c2626, 0x260000, 0x261313, 0xff3f00,
    0xff9f7f, 0xa52900, 0xa56752, 0x7f1f00, 0x7f4f3f, 0x4c1300, 0x4c2f26,
    0x260900, 0x261713, 0xff7f00, 0xffbf7f, 0xa55200, 0xa57c52, 0x7f3f00,
    0x7f5f3f, 0x4c2600, 0x4c3926, 0x261300, 0x261c13, 0xffbf00, 0xffdf7f,
    0xa57c00, 0xa59152, 0x7f5f00, 0x7f6f3f, 0x4c3900, 0x4c4226, 0x261c00,
    0x262113, 0xffff00, 0xffff7f, 0xa5a500, 0xa5a552, 0x7f7f00, 0x7f7f3f,
    0x4c4c00, 0x4c4c26, 0x262600, 0x262613, 0xbfff00, 0xdfff7f, 0x7ca500,
    0x91a552, 0x5f7f00, 0x6f7f3f, 0x394c00, 0x424c26, 0x1c2600, 0x212613,
    0x7fff00, 0xbfff7f, 0x52a500, 0x7ca552, 0x3f7f00, 0x5f7f3f, 0x264c00,
    0x394c26, 0x132600, 0x1c2613, 0x3fff00, 0x9fff7f, 0x29a500, 0x67a552,
    0x1f7f00, 0x4f7f3f, 0x134c00, 0x2f4c26, 0x092600, 0x172613, 0x00ff00,
    0x7fff7f, 0x00a500, 0x52a552, 0x007f00, 0x3f7f3f, 0x004c00, 0x264c26,
    0x002600, 0x132613, 0x00ff3f, 0x7fff9f, 0x00a529, 0x52a567, 0x007f1f,
    0x3f7f4f, 0x004c13, 0x264c2f, 0x002609, 0x132617, 0x00ff7f, 0x7fffbf,
    0x00a552, 0x52a57c, 0x007f3f, 0x3f7f5f, 0x004c26, 0x264c39, 0x002613,
    0x13261c, 0x00ffbf, 0x7fffdf, 0x00a57c, 0x52a591, 0x007f5f, 0x3f7f6f,
    0x004c39, 0x264c42, 0x00261c, 0x132621, 0x00ffff, 0x7fffff, 0x00a5a5,
    0x52a5a5, 0x007f7f, 0x3f7f7f, 0x004c4c, 0x264c4c, 0x002626, 0x132626,
    0x00bfff, 0x7fdfff, 0x007ca5, 0x5291a5, 0x005f7f, 0x3f6f7f, 0x00394c,
    0x26424c, 0x001c26, 0x132126, 0x007fff, 0x7fbfff, 0x0052a5, 0x527ca5,
    0x003f7f, 0x3f5f7f, 0x00264c, 0x26394c, 0x001326, 0x131c26, 0x003fff,
    0x7f9fff, 0x0029a5, 0x5267a5, 0x001f7f, 0x3f4f7f, 0x00134c, 0x262f4c,
    0x000926, 0x131726, 0x0000ff, 0x7f7fff, 0x0000a5, 0x5252a5, 0x00007f,
    0x3f3f7f, 0x00004c, 0x26264c, 0x000026, 0x131326, 0x3f00ff, 0x9f7fff,
    0x2900a5, 0x6752a5, 0x1f007f, 0x4f3f7f, 0x13004c, 0x2f264c, 0x090026,
    0x171326, 0x7f00ff, 0xbf7fff, 0x5200a5, 0x7c52a5, 0x3f007f, 0x5f3f7f,
    0x26004c, 0x39264c, 0x130026, 0x1c1326, 0xbf00ff, 0xdf7fff, 0x7c00a5,
    0x9152a5, 0x5f007f, 0x6f3f7f, 0x39004c, 0x42264c, 0x1c0026, 0x211326,
    0xff00ff, 0xff7fff, 0xa500a5, 0xa552a5, 0x7f007f, 0x7f3f7f, 0x4c004c,
    0x4c264c, 0x260026, 0x261326, 0xff00bf, 0xff7fdf, 0xa5007c, 0xa55291,
    0x7f005f, 0x7f3f6f, 0x4c0039, 0x4c2642, 0x26001c, 0x261321, 0xff007f,
    0xff7fbf, 0xa50052, 0xa5527c, 0x7f003f, 0x7f3f5f, 0x4c0026, 0x4c2639,
    0x260013, 0x26131c, 0xff003f, 0xff7f9f, 0xa50029, 0xa55267, 0x7f001f,
    0x7f3f4f, 0x4c0013, 0x4c262f, 0x260009, 0x261317, 0x000000, 0x2d2d2d,
    0x5b5b5b, 0x898989, 0xb7b7b7, 0xb3b3b3,
]
