use std::ffi::CStr;
use std::mem;
use std::os::raw::c_char;
use std::ptr;
use std::str::FromStr;

use symbolic::common::byteview::ByteView;
use symbolic::debuginfo::{DebugFeatures, DebugId, FatObject, Object};

use core::SymbolicStr;

/// A potential multi arch object.
pub struct SymbolicFatObject;

/// A single arch object.
pub struct SymbolicObject;

/// A list of object features.
#[repr(C)]
pub struct SymbolicObjectFeatures {
    data: *mut SymbolicStr,
    len: usize,
}

ffi_fn! {
    /// Loads a fat object from a given path.
    unsafe fn symbolic_fatobject_open(path: *const c_char) -> Result<*mut SymbolicFatObject> {
        let byteview = ByteView::from_path(CStr::from_ptr(path).to_str()?)?;
        let obj = FatObject::parse(byteview)?;
        Ok(Box::into_raw(Box::new(obj)) as *mut SymbolicFatObject)
    }
}

ffi_fn! {
    /// Frees the given fat object.
    unsafe fn symbolic_fatobject_free(sfo: *mut SymbolicFatObject) {
        if !sfo.is_null() {
            let fo = sfo as *mut FatObject<'static>;
            Box::from_raw(fo);
        }
    }
}

ffi_fn! {
    /// Returns the number of contained objects.
    unsafe fn symbolic_fatobject_object_count(sfo: *const SymbolicFatObject) -> Result<usize> {
        let fo = sfo as *const FatObject<'static>;
        Ok((*fo).object_count() as usize)
    }
}

ffi_fn! {
    /// Returns the n-th object.
    unsafe fn symbolic_fatobject_get_object(
        sfo: *const SymbolicFatObject,
        idx: usize,
    ) -> Result<*mut SymbolicObject> {
        let fo = sfo as *const FatObject<'static>;
        if let Some(obj) = (*fo).get_object(idx)? {
            Ok(Box::into_raw(Box::new(obj)) as *mut SymbolicObject)
        } else {
            Ok(ptr::null_mut())
        }
    }
}

ffi_fn! {
    /// Returns the architecture of the object.
    unsafe fn symbolic_object_get_arch(so: *const SymbolicObject) -> Result<SymbolicStr> {
        let o = so as *const Object<'static>;
        Ok(SymbolicStr::new((*o).arch()?.name()))
    }
}

ffi_fn! {
    /// Returns the debug identifier of the object.
    unsafe fn symbolic_object_get_id(so: *const SymbolicObject) -> Result<SymbolicStr> {
        let o = so as *const Object<'static>;
        Ok((*o).id().unwrap_or_default().to_string().into())
    }
}

ffi_fn! {
    /// Returns the object kind (e.g. MachO, ELF, ...).
    unsafe fn symbolic_object_get_kind(so: *const SymbolicObject) -> Result<SymbolicStr> {
        let o = so as *const Object<'static>;
        Ok(SymbolicStr::new((*o).kind().name()))
    }
}

ffi_fn! {
    /// Returns the desiganted use of the object file and hints at its contents (e.g. debug,
    /// executable, ...).
    unsafe fn symbolic_object_get_type(so: *const SymbolicObject) -> Result<SymbolicStr> {
        let o = so as *mut Object<'static>;
        Ok(SymbolicStr::new((*o).class().name()))
    }
}

ffi_fn! {
    /// Returns the kind of debug data contained in this object file, if any (e.g. DWARF).
    unsafe fn symbolic_object_get_debug_kind(so: *const SymbolicObject) -> Result<SymbolicStr> {
        let o = so as *const Object<'static>;
        Ok(if let Some(kind) = (*o).debug_kind() {
            SymbolicStr::new(kind.name())
        } else {
            SymbolicStr::default()
        })
    }
}

ffi_fn! {
    unsafe fn symbolic_object_get_features(
        so: *const SymbolicObject,
    ) -> Result<SymbolicObjectFeatures> {
        let o = so as *const Object<'static>;

        let mut features = Vec::new();
        for feature in (*o).features() {
            features.push(SymbolicStr::from(feature.to_string()));
        }
        features.shrink_to_fit();

        let result = SymbolicObjectFeatures {
            len: features.len(),
            data: features.as_mut_ptr(),
        };

        mem::forget(features);
        Ok(result)
    }
}

ffi_fn! {
    unsafe fn symbolic_object_features_free(f: *mut SymbolicObjectFeatures) {
        Vec::from_raw_parts((*f).data, (*f).len, (*f).len)
    }
}

ffi_fn! {
    /// Frees an object returned from a fat object.
    unsafe fn symbolic_object_free(so: *mut SymbolicObject) {
        if !so.is_null() {
            let o = so as *mut Object<'static>;
            Box::from_raw(o);
        }
    }
}

ffi_fn! {
    /// Converts a Breakpad CodeModuleId to DebugId.
    unsafe fn symbolic_id_from_breakpad(sid: *const SymbolicStr) -> Result<SymbolicStr> {
        Ok(DebugId::from_breakpad((*sid).as_str())?.to_string().into())
    }
}

ffi_fn! {
    /// Normalizes a debug identifier to default representation.
    unsafe fn symbolic_normalize_debug_id(sid: *const SymbolicStr) -> Result<SymbolicStr> {
        Ok(DebugId::from_str((*sid).as_str())?.to_string().into())
    }
}
