#!/usr/bin/env python
# vim: set fileencoding=utf-8 :
# Laurent El Shafey <Laurent.El-Shafey@idiap.ch>

"""This script creates the MOBIO database in a single pass.
"""

import os

from .models import *

def nodot(item):
  """Can be used to ignore hidden files, starting with the . character."""
  return item[0] != '.'

def add_files(session, imagedir, verbose):
  """Add files to the MOBIO database."""

  def add_file(session, basename, verbose):
    """Parse a single filename and add it to the list.
       Also add a client entry if not already in the database."""
    v = os.path.splitext(basename)[0].split('_')
    bname = os.path.splitext(basename)[0]
    gender = ''
    if v[0][0] == 'm': gender = 'male'
    if v[0][0] == 'f': gender = 'female'
    institute = int(v[0][1])
    if institute == 0:
      institute = 'idiap'
    elif institute == 1:
      institute = 'manchester'
    elif institute == 2:
      institute = 'surrey'
    elif institute == 3:
      institute = 'oulu'
    elif institute == 4:
      institute = 'brno'
    elif institute == 5:
      institute = 'avignon'
    client_id = v[0][1:4]

    if not (client_id in client_dict):
      if (institute == 'surrey' or institute == 'avignon'):
        group = 'world'
      elif (institute == 'manchester' or institute == 'oulu'):
        group = 'dev'
      elif (institute == 'idiap' or institute == 'brno'):
        group = 'eval'
      if verbose>1: print "  Adding client %d..." % int(client_id)
      session.add(Client(int(client_id), group, gender, institute))
      client_dict[client_id] = True

    session_id = int(v[1])
    speech_type = v[2][0]
    shot_id = v[2][1:3]
    environment = v[3][0]
    device = v[3][1]
    if( device == '0'):
      device = 'mobile'
    elif( device == '1'):
      device = 'laptop'
    channel = int(v[4][0])

    if verbose>1: print "    Adding file '%s'..." % bname
    session.add(File(int(client_id), bname, session_id, speech_type, shot_id, environment, device, channel))

  client_dict = {}
  if verbose: print "Adding clients and files ..."
  for client_id in filter(nodot, os.listdir(imagedir)):
    for filename in filter(nodot, os.listdir(os.path.join(imagedir, client_id))):
      if filename.endswith('.jpg'):
        add_file(session, os.path.join(client_id, os.path.basename(filename)), verbose)

def add_subworlds(session, verbose):
  """Adds subworlds"""

  twothirds_subsampled_filelist = [
    "f214/f214_01_p01_i0_0", "f214/f214_01_f12_i0_0", "f214/f214_01_l11_i0_0", "f214/f214_02_p01_i0_0", "f214/f214_02_f12_i0_0",
    "f214/f214_02_l11_i0_0", "f214/f214_03_p01_i0_0", "f214/f214_03_f12_i0_0", "f214/f214_03_l11_i0_0", "f214/f214_04_p01_i0_0",
    "f214/f214_04_f12_i0_0", "f214/f214_04_l11_i0_0", "f214/f214_05_p01_i0_0", "f214/f214_05_f12_i0_0", "f214/f214_05_l11_i0_0",
    "f214/f214_06_p01_i0_0", "f214/f214_06_f12_i0_0", "f214/f214_06_l11_i0_0", "f214/f214_07_p01_i0_0", "f214/f214_07_f07_i0_0",
    "f214/f214_07_l06_i0_0", "f214/f214_08_p01_i0_0", "f214/f214_08_f07_i0_0", "f214/f214_08_l06_i0_0", "f214/f214_09_p01_i0_0",
    "f214/f214_09_f07_i0_0", "f214/f214_09_l06_i0_0", "f214/f214_10_p01_i0_0", "f214/f214_10_f07_i0_0", "f214/f214_10_l06_i0_0",
    "f214/f214_11_p01_i0_0", "f214/f214_11_f07_i0_0", "f214/f214_11_l06_i0_0", "f214/f214_12_p01_i0_0", "f214/f214_12_f07_i0_0",
    "f214/f214_12_l06_i0_0", "f218/f218_01_p01_i0_0", "f218/f218_01_f12_i0_0", "f218/f218_01_l11_i0_0", "f218/f218_02_p01_i0_0",
    "f218/f218_02_f12_i0_0", "f218/f218_02_l11_i0_0", "f218/f218_03_p01_i0_0", "f218/f218_03_f12_i0_0", "f218/f218_03_l11_i0_0",
    "f218/f218_09_p01_i0_0", "f218/f218_09_f12_i0_0", "f218/f218_09_l11_i0_0", "f218/f218_10_p01_i0_0", "f218/f218_10_f12_i0_0",
    "f218/f218_10_l11_i0_0", "f218/f218_11_p01_i0_0", "f218/f218_11_f12_i0_0", "f218/f218_11_l11_i0_0", "f218/f218_12_p01_i0_0",
    "f218/f218_12_f07_i0_0", "f218/f218_12_l06_i0_0", "f218/f218_13_p01_i0_0", "f218/f218_13_f07_i0_0", "f218/f218_13_l06_i0_0",
    "f218/f218_14_p01_i0_0", "f218/f218_14_f07_i0_0", "f218/f218_14_l06_i0_0", "f218/f218_15_p01_i0_0", "f218/f218_15_f07_i0_0",
    "f218/f218_15_l06_i0_0", "f218/f218_16_p01_i0_0", "f218/f218_16_f07_i0_0", "f218/f218_16_l06_i0_0", "f218/f218_17_p01_i0_0",
    "f218/f218_17_f07_i0_0", "f218/f218_17_l06_i0_0", "f230/f230_01_p01_i0_0", "f230/f230_01_f12_i0_0", "f230/f230_01_l11_i0_0",
    "f230/f230_02_p01_i0_0", "f230/f230_02_f12_i0_0", "f230/f230_02_l11_i0_0", "f230/f230_03_p01_i0_0", "f230/f230_03_f12_i0_0",
    "f230/f230_03_l11_i0_0", "f230/f230_04_p01_i0_0", "f230/f230_04_f12_i0_0", "f230/f230_04_l11_i0_0", "f230/f230_05_p01_i0_0",
    "f230/f230_05_f12_i0_0", "f230/f230_05_l11_i0_0", "f230/f230_06_p01_i0_0", "f230/f230_06_f12_i0_0", "f230/f230_06_l11_i0_0",
    "f230/f230_07_p01_i0_0", "f230/f230_07_f07_i0_0", "f230/f230_07_l06_i0_0", "f230/f230_08_p01_i0_0", "f230/f230_08_f07_i0_0",
    "f230/f230_08_l06_i0_0", "f230/f230_09_p01_i0_0", "f230/f230_09_f07_i0_0", "f230/f230_09_l06_i0_0", "f230/f230_10_p01_i0_0",
    "f230/f230_10_f07_i0_0", "f230/f230_10_l06_i0_0", "f230/f230_11_p01_i0_0", "f230/f230_11_f07_i0_0", "f230/f230_11_l06_i0_0",
    "f230/f230_12_p01_i0_0", "f230/f230_12_f07_i0_0", "f230/f230_12_l06_i0_0", "f232/f232_01_p01_i0_0", "f232/f232_01_f12_i0_0",
    "f232/f232_01_l11_i0_0", "f232/f232_02_p01_i0_0", "f232/f232_02_f12_i0_0", "f232/f232_02_l11_i0_0", "f232/f232_03_p01_i0_0",
    "f232/f232_03_f12_i0_0", "f232/f232_03_l11_i0_0", "f232/f232_04_p01_i0_0", "f232/f232_04_f12_i0_0", "f232/f232_04_l11_i0_0",
    "f232/f232_05_p01_i0_0", "f232/f232_05_f12_i0_0", "f232/f232_05_l11_i0_0", "f232/f232_07_p01_i0_0", "f232/f232_07_f12_i0_0",
    "f232/f232_07_l11_i0_0", "f232/f232_08_p01_i0_0", "f232/f232_08_f07_i0_0", "f232/f232_08_l06_i0_0", "f232/f232_09_p01_i0_0",
    "f232/f232_09_f07_i0_0", "f232/f232_09_l06_i0_0", "f232/f232_10_p01_i0_0", "f232/f232_10_f07_i0_0", "f232/f232_10_l06_i0_0",
    "f232/f232_11_p01_i0_0", "f232/f232_11_f07_i0_0", "f232/f232_11_l06_i0_0", "f232/f232_12_p01_i0_0", "f232/f232_12_f07_i0_0",
    "f232/f232_12_l06_i0_0", "f232/f232_13_p01_i0_0", "f232/f232_13_f07_i0_0", "f232/f232_13_l06_i0_0", "f507/f507_01_p01_i0_0",
    "f507/f507_01_f12_i0_0", "f507/f507_01_l11_i0_0", "f507/f507_02_p01_i0_0", "f507/f507_02_f12_i0_0", "f507/f507_02_l11_i0_0",
    "f507/f507_03_p01_i0_0", "f507/f507_03_f12_i0_0", "f507/f507_03_l11_i0_0", "f507/f507_04_p01_i0_0", "f507/f507_04_f12_i0_0",
    "f507/f507_04_l11_i0_0", "f507/f507_05_p01_i0_0", "f507/f507_05_f12_i0_0", "f507/f507_05_l11_i0_0", "f507/f507_06_p01_i0_0",
    "f507/f507_06_f12_i0_0", "f507/f507_06_l11_i0_0", "f507/f507_07_p01_i0_0", "f507/f507_07_f07_i0_0", "f507/f507_07_l06_i0_0",
    "f507/f507_08_p01_i0_0", "f507/f507_08_f07_i0_0", "f507/f507_08_l06_i0_0", "f507/f507_09_p01_i0_0", "f507/f507_09_f07_i0_0",
    "f507/f507_09_l06_i0_0", "f507/f507_10_p01_i0_0", "f507/f507_10_f07_i0_0", "f507/f507_10_l06_i0_0", "f507/f507_11_p01_i0_0",
    "f507/f507_11_f07_i0_0", "f507/f507_11_l06_i0_0", "f507/f507_12_p01_i0_0", "f507/f507_12_f07_i0_0", "f507/f507_12_l06_i0_0",
    "f508/f508_01_p01_i0_0", "f508/f508_01_f12_i0_0", "f508/f508_01_l11_i0_0", "f508/f508_02_p01_i0_0", "f508/f508_02_f12_i0_0",
    "f508/f508_02_l11_i0_0", "f508/f508_03_p01_i0_0", "f508/f508_03_f12_i0_0", "f508/f508_03_l11_i0_0", "f508/f508_04_p01_i0_0",
    "f508/f508_04_f12_i0_0", "f508/f508_04_l11_i0_0", "f508/f508_05_p01_i0_0", "f508/f508_05_f12_i0_0", "f508/f508_05_l11_i0_0",
    "f508/f508_06_p01_i0_0", "f508/f508_06_f12_i0_0", "f508/f508_06_l11_i0_0", "f508/f508_07_p01_i0_0", "f508/f508_07_f07_i0_0",
    "f508/f508_07_l06_i0_0", "f508/f508_08_p01_i0_0", "f508/f508_08_f07_i0_0", "f508/f508_08_l06_i0_0", "f508/f508_09_p01_i0_0",
    "f508/f508_09_f07_i0_0", "f508/f508_09_l06_i0_0", "f508/f508_10_p01_i0_0", "f508/f508_10_f07_i0_0", "f508/f508_10_l06_i0_0",
    "f508/f508_11_p01_i0_0", "f508/f508_11_f07_i0_0", "f508/f508_11_l06_i0_0", "f508/f508_12_p01_i0_0", "f508/f508_12_f07_i0_0",
    "f508/f508_12_l06_i0_0", "f509/f509_01_p01_i0_0", "f509/f509_01_f12_i0_0", "f509/f509_01_l11_i0_0", "f509/f509_02_p01_i0_0",
    "f509/f509_02_f12_i0_0", "f509/f509_02_l11_i0_0", "f509/f509_03_p01_i0_0", "f509/f509_03_f12_i0_0", "f509/f509_03_l11_i0_0",
    "f509/f509_04_p01_i0_0", "f509/f509_04_f12_i0_0", "f509/f509_04_l11_i0_0", "f509/f509_05_p01_i0_0", "f509/f509_05_f12_i0_0",
    "f509/f509_05_l11_i0_0", "f509/f509_06_p01_i0_0", "f509/f509_06_f12_i0_0", "f509/f509_06_l11_i0_0", "f509/f509_07_p01_i0_0",
    "f509/f509_07_f07_i0_0", "f509/f509_07_l06_i0_0", "f509/f509_08_p01_i0_0", "f509/f509_08_f07_i0_0", "f509/f509_08_l06_i0_0",
    "f509/f509_09_p01_i0_0", "f509/f509_09_f07_i0_0", "f509/f509_09_l06_i0_0", "f509/f509_10_p01_i0_0", "f509/f509_10_f07_i0_0",
    "f509/f509_10_l06_i0_0", "f509/f509_11_p01_i0_0", "f509/f509_11_f07_i0_0", "f509/f509_11_l06_i0_0", "f509/f509_12_p01_i0_0",
    "f509/f509_12_f07_i0_0", "f509/f509_12_l06_i0_0", "f510/f510_01_p01_i0_0", "f510/f510_01_f12_i0_0", "f510/f510_01_l11_i0_0",
    "f510/f510_02_p01_i0_0", "f510/f510_02_f12_i0_0", "f510/f510_02_l11_i0_0", "f510/f510_03_p01_i0_0", "f510/f510_03_f12_i0_0",
    "f510/f510_03_l11_i0_0", "f510/f510_04_p01_i0_0", "f510/f510_04_f12_i0_0", "f510/f510_04_l11_i0_0", "f510/f510_05_p01_i0_0",
    "f510/f510_05_f12_i0_0", "f510/f510_05_l11_i0_0", "f510/f510_06_p01_i0_0", "f510/f510_06_f12_i0_0", "f510/f510_06_l11_i0_0",
    "f510/f510_07_p01_i0_0", "f510/f510_07_f07_i0_0", "f510/f510_07_l06_i0_0", "f510/f510_08_p01_i0_0", "f510/f510_08_f07_i0_0",
    "f510/f510_08_l06_i0_0", "f510/f510_09_p01_i0_0", "f510/f510_09_f07_i0_0", "f510/f510_09_l06_i0_0", "f510/f510_10_p01_i0_0",
    "f510/f510_10_f07_i0_0", "f510/f510_10_l06_i0_0", "f510/f510_11_p01_i0_0", "f510/f510_11_f07_i0_0", "f510/f510_11_l06_i0_0",
    "f510/f510_12_p01_i0_0", "f510/f510_12_f07_i0_0", "f510/f510_12_l06_i0_0", "f528/f528_01_p01_i0_0", "f528/f528_01_f12_i0_0",
    "f528/f528_01_l11_i0_0", "f528/f528_02_p01_i0_0", "f528/f528_02_f12_i0_0", "f528/f528_02_l11_i0_0", "f528/f528_03_p01_i0_0",
    "f528/f528_03_f12_i0_0", "f528/f528_03_l11_i0_0", "f528/f528_04_p01_i0_0", "f528/f528_04_f12_i0_0", "f528/f528_04_l11_i0_0",
    "f528/f528_05_p01_i0_0", "f528/f528_05_f12_i0_0", "f528/f528_05_l11_i0_0", "f528/f528_06_p01_i0_0", "f528/f528_06_f12_i0_0",
    "f528/f528_06_l11_i0_0", "f528/f528_07_p01_i0_0", "f528/f528_07_f07_i0_0", "f528/f528_07_l06_i0_0", "f528/f528_08_p01_i0_0",
    "f528/f528_08_f07_i0_0", "f528/f528_08_l06_i0_0", "f528/f528_09_p01_i0_0", "f528/f528_09_f07_i0_0", "f528/f528_09_l06_i0_0",
    "f528/f528_10_p01_i0_0", "f528/f528_10_f07_i0_0", "f528/f528_10_l06_i0_0", "f528/f528_11_p01_i0_0", "f528/f528_11_f07_i0_0",
    "f528/f528_11_l06_i0_0", "f528/f528_12_p01_i0_0", "f528/f528_12_f07_i0_0", "f528/f528_12_l06_i0_0", "m202/m202_01_p01_i0_0",
    "m202/m202_01_f12_i0_0", "m202/m202_01_l11_i0_0", "m202/m202_02_p01_i0_0", "m202/m202_02_f12_i0_0", "m202/m202_02_l11_i0_0",
    "m202/m202_03_p01_i0_0", "m202/m202_03_f12_i0_0", "m202/m202_03_l11_i0_0", "m202/m202_04_p01_i0_0", "m202/m202_04_f12_i0_0",
    "m202/m202_04_l11_i0_0", "m202/m202_05_p01_i0_0", "m202/m202_05_f12_i0_0", "m202/m202_05_l11_i0_0", "m202/m202_06_p01_i0_0",
    "m202/m202_06_f12_i0_0", "m202/m202_06_l11_i0_0", "m202/m202_07_p01_i0_0", "m202/m202_07_f07_i0_0", "m202/m202_07_l06_i0_0",
    "m202/m202_08_p01_i0_0", "m202/m202_08_f07_i0_0", "m202/m202_08_l06_i0_0", "m202/m202_09_p01_i0_0", "m202/m202_09_f07_i0_0",
    "m202/m202_09_l06_i0_0", "m202/m202_10_p01_i0_0", "m202/m202_10_f07_i0_0", "m202/m202_10_l06_i0_0", "m202/m202_11_p01_i0_0",
    "m202/m202_11_f07_i0_0", "m202/m202_11_l06_i0_0", "m202/m202_12_p01_i0_0", "m202/m202_12_f07_i0_0", "m202/m202_12_l06_i0_0",
    "m203/m203_01_p01_i0_0", "m203/m203_01_f12_i0_0", "m203/m203_01_l11_i0_0", "m203/m203_03_p01_i0_0", "m203/m203_03_f12_i0_0",
    "m203/m203_03_l11_i0_0", "m203/m203_04_p01_i0_0", "m203/m203_04_f12_i0_0", "m203/m203_04_l11_i0_0", "m203/m203_05_p01_i0_0",
    "m203/m203_05_f12_i0_0", "m203/m203_05_l11_i0_0", "m203/m203_06_p01_i0_0", "m203/m203_06_f12_i0_0", "m203/m203_06_l11_i0_0",
    "m203/m203_07_p01_i0_0", "m203/m203_07_f12_i0_0", "m203/m203_07_l11_i0_0", "m203/m203_08_p01_i0_0", "m203/m203_08_f07_i0_0",
    "m203/m203_08_l06_i0_0", "m203/m203_09_p01_i0_0", "m203/m203_09_f07_i0_0", "m203/m203_09_l06_i0_0", "m203/m203_10_p01_i0_0",
    "m203/m203_10_f07_i0_0", "m203/m203_10_l06_i0_0", "m203/m203_11_p01_i0_0", "m203/m203_11_f07_i0_0", "m203/m203_11_l06_i0_0",
    "m203/m203_12_p01_i0_0", "m203/m203_12_f07_i0_0", "m203/m203_12_l06_i0_0", "m203/m203_13_p01_i0_0", "m203/m203_13_f07_i0_0",
    "m203/m203_13_l06_i0_0", "m207/m207_01_p01_i0_0", "m207/m207_01_f12_i0_0", "m207/m207_01_l11_i0_0", "m207/m207_02_p01_i0_0",
    "m207/m207_02_f12_i0_0", "m207/m207_02_l11_i0_0", "m207/m207_03_p01_i0_0", "m207/m207_03_f12_i0_0", "m207/m207_03_l11_i0_0",
    "m207/m207_04_p01_i0_0", "m207/m207_04_f12_i0_0", "m207/m207_04_l11_i0_0", "m207/m207_05_p01_i0_0", "m207/m207_05_f12_i0_0",
    "m207/m207_05_l11_i0_0", "m207/m207_06_p01_i0_0", "m207/m207_06_f12_i0_0", "m207/m207_06_l11_i0_0", "m207/m207_07_p01_i0_0",
    "m207/m207_07_f07_i0_0", "m207/m207_07_l06_i0_0", "m207/m207_08_p01_i0_0", "m207/m207_08_f07_i0_0", "m207/m207_08_l06_i0_0",
    "m207/m207_09_p01_i0_0", "m207/m207_09_f07_i0_0", "m207/m207_09_l06_i0_0", "m207/m207_10_p01_i0_0", "m207/m207_10_f07_i0_0",
    "m207/m207_10_l06_i0_0", "m207/m207_11_p01_i0_0", "m207/m207_11_f07_i0_0", "m207/m207_11_l06_i0_0", "m207/m207_12_p01_i0_0",
    "m207/m207_12_f07_i0_0", "m207/m207_12_l06_i0_0", "m208/m208_01_p01_i0_0", "m208/m208_01_f12_i0_0", "m208/m208_01_l11_i0_0",
    "m208/m208_02_p01_i0_0", "m208/m208_02_f12_i0_0", "m208/m208_02_l11_i0_0", "m208/m208_03_p01_i0_0", "m208/m208_03_f12_i0_0",
    "m208/m208_03_l11_i0_0", "m208/m208_04_p01_i0_0", "m208/m208_04_f12_i0_0", "m208/m208_04_l11_i0_0", "m208/m208_05_p01_i0_0",
    "m208/m208_05_f12_i0_0", "m208/m208_05_l11_i0_0", "m208/m208_06_p01_i0_0", "m208/m208_06_f12_i0_0", "m208/m208_06_l11_i0_0",
    "m208/m208_07_p01_i0_0", "m208/m208_07_f07_i0_0", "m208/m208_07_l06_i0_0", "m208/m208_08_p01_i0_0", "m208/m208_08_f07_i0_0",
    "m208/m208_08_l06_i0_0", "m208/m208_09_p01_i0_0", "m208/m208_09_f07_i0_0", "m208/m208_09_l06_i0_0", "m208/m208_10_p01_i0_0",
    "m208/m208_10_f07_i0_0", "m208/m208_10_l06_i0_0", "m208/m208_11_p01_i0_0", "m208/m208_11_f07_i0_0", "m208/m208_11_l06_i0_0",
    "m208/m208_12_p01_i0_0", "m208/m208_12_f07_i0_0", "m208/m208_12_l06_i0_0", "m212/m212_01_p01_i0_0", "m212/m212_01_f12_i0_0",
    "m212/m212_01_l11_i0_0", "m212/m212_02_p01_i0_0", "m212/m212_02_f12_i0_0", "m212/m212_02_l11_i0_0", "m212/m212_03_p01_i0_0",
    "m212/m212_03_f12_i0_0", "m212/m212_03_l11_i0_0", "m212/m212_04_p01_i0_0", "m212/m212_04_f12_i0_0", "m212/m212_04_l11_i0_0",
    "m212/m212_05_p01_i0_0", "m212/m212_05_f12_i0_0", "m212/m212_05_l11_i0_0", "m212/m212_06_p01_i0_0", "m212/m212_06_f12_i0_0",
    "m212/m212_06_l11_i0_0", "m212/m212_07_p01_i0_0", "m212/m212_07_f07_i0_0", "m212/m212_07_l06_i0_0", "m212/m212_08_p01_i0_0",
    "m212/m212_08_f07_i0_0", "m212/m212_08_l06_i0_0", "m212/m212_09_p01_i0_0", "m212/m212_09_f07_i0_0", "m212/m212_09_l06_i0_0",
    "m212/m212_10_p01_i0_0", "m212/m212_10_f07_i0_0", "m212/m212_10_l06_i0_0", "m212/m212_11_p01_i0_0", "m212/m212_11_f07_i0_0",
    "m212/m212_11_l06_i0_0", "m212/m212_12_p01_i0_0", "m212/m212_12_f07_i0_0", "m212/m212_12_l06_i0_0", "m215/m215_02_p01_i0_0",
    "m215/m215_02_f12_i0_0", "m215/m215_02_l11_i0_0", "m215/m215_03_p01_i0_0", "m215/m215_03_f12_i0_0", "m215/m215_03_l11_i0_0",
    "m215/m215_04_p01_i0_0", "m215/m215_04_f12_i0_0", "m215/m215_04_l11_i0_0", "m215/m215_05_p01_i0_0", "m215/m215_05_f12_i0_0",
    "m215/m215_05_l11_i0_0", "m215/m215_06_p01_i0_0", "m215/m215_06_f12_i0_0", "m215/m215_06_l11_i0_0", "m215/m215_07_p01_i0_0",
    "m215/m215_07_f12_i0_0", "m215/m215_07_l11_i0_0", "m215/m215_08_p01_i0_0", "m215/m215_08_f07_i0_0", "m215/m215_08_l06_i0_0",
    "m215/m215_09_p01_i0_0", "m215/m215_09_f07_i0_0", "m215/m215_09_l06_i0_0", "m215/m215_10_p01_i0_0", "m215/m215_10_f07_i0_0",
    "m215/m215_10_l06_i0_0", "m215/m215_11_p01_i0_0", "m215/m215_11_f07_i0_0", "m215/m215_11_l06_i0_0", "m215/m215_12_p01_i0_0",
    "m215/m215_12_f07_i0_0", "m215/m215_12_l06_i0_0", "m215/m215_13_p01_i0_0", "m215/m215_13_f07_i0_0", "m215/m215_13_l06_i0_0",
    "m217/m217_01_p01_i0_0", "m217/m217_01_f12_i0_0", "m217/m217_01_l11_i0_0", "m217/m217_02_p01_i0_0", "m217/m217_02_f12_i0_0",
    "m217/m217_02_l11_i0_0", "m217/m217_03_p01_i0_0", "m217/m217_03_f12_i0_0", "m217/m217_03_l11_i0_0", "m217/m217_04_p01_i0_0",
    "m217/m217_04_f12_i0_0", "m217/m217_04_l11_i0_0", "m217/m217_05_p01_i0_0", "m217/m217_05_f12_i0_0", "m217/m217_05_l11_i0_0",
    "m217/m217_06_p01_i0_0", "m217/m217_06_f12_i0_0", "m217/m217_06_l11_i0_0", "m217/m217_07_p01_i0_0", "m217/m217_07_f07_i0_0",
    "m217/m217_07_l06_i0_0", "m217/m217_08_p01_i0_0", "m217/m217_08_f07_i0_0", "m217/m217_08_l06_i0_0", "m217/m217_09_p01_i0_0",
    "m217/m217_09_f07_i0_0", "m217/m217_09_l06_i0_0", "m217/m217_10_p01_i0_0", "m217/m217_10_f07_i0_0", "m217/m217_10_l06_i0_0",
    "m217/m217_11_p01_i0_0", "m217/m217_11_f07_i0_0", "m217/m217_11_l06_i0_0", "m217/m217_12_p01_i0_0", "m217/m217_12_f07_i0_0",
    "m217/m217_12_l06_i0_0", "m223/m223_01_p01_i0_0", "m223/m223_01_f12_i0_0", "m223/m223_01_l11_i0_0", "m223/m223_02_p01_i0_0",
    "m223/m223_02_f12_i0_0", "m223/m223_02_l11_i0_0", "m223/m223_03_p01_i0_0", "m223/m223_03_f12_i0_0", "m223/m223_03_l11_i0_0",
    "m223/m223_04_p01_i0_0", "m223/m223_04_f12_i0_0", "m223/m223_04_l11_i0_0", "m223/m223_05_p01_i0_0", "m223/m223_05_f12_i0_0",
    "m223/m223_05_l11_i0_0", "m223/m223_06_p01_i0_0", "m223/m223_06_f12_i0_0", "m223/m223_06_l11_i0_0", "m223/m223_07_p01_i0_0",
    "m223/m223_07_f07_i0_0", "m223/m223_07_l06_i0_0", "m223/m223_08_p01_i0_0", "m223/m223_08_f07_i0_0", "m223/m223_08_l06_i0_0",
    "m223/m223_09_p01_i0_0", "m223/m223_09_f07_i0_0", "m223/m223_09_l06_i0_0", "m223/m223_10_p01_i0_0", "m223/m223_10_f07_i0_0",
    "m223/m223_10_l06_i0_0", "m223/m223_11_p01_i0_0", "m223/m223_11_f07_i0_0", "m223/m223_11_l06_i0_0", "m223/m223_12_p01_i0_0",
    "m223/m223_12_f07_i0_0", "m223/m223_12_l06_i0_0", "m224/m224_01_p01_i0_0", "m224/m224_01_f12_i0_0", "m224/m224_01_l11_i0_0",
    "m224/m224_02_p01_i0_0", "m224/m224_02_f12_i0_0", "m224/m224_02_l11_i0_0", "m224/m224_03_p01_i0_0", "m224/m224_03_f12_i0_0",
    "m224/m224_03_l11_i0_0", "m224/m224_04_p01_i0_0", "m224/m224_04_f12_i0_0", "m224/m224_04_l11_i0_0", "m224/m224_05_p01_i0_0",
    "m224/m224_05_f12_i0_0", "m224/m224_05_l11_i0_0", "m224/m224_06_p01_i0_0", "m224/m224_06_f12_i0_0", "m224/m224_06_l11_i0_0",
    "m224/m224_07_p01_i0_0", "m224/m224_07_f07_i0_0", "m224/m224_07_l06_i0_0", "m224/m224_08_p01_i0_0", "m224/m224_08_f07_i0_0",
    "m224/m224_08_l06_i0_0", "m224/m224_09_p01_i0_0", "m224/m224_09_f07_i0_0", "m224/m224_09_l06_i0_0", "m224/m224_10_p01_i0_0",
    "m224/m224_10_f07_i0_0", "m224/m224_10_l06_i0_0", "m224/m224_11_p01_i0_0", "m224/m224_11_f07_i0_0", "m224/m224_11_l06_i0_0",
    "m224/m224_12_p01_i0_0", "m224/m224_12_f07_i0_0", "m224/m224_12_l06_i0_0", "m225/m225_01_p01_i0_0", "m225/m225_01_f12_i0_0",
    "m225/m225_01_l11_i0_0", "m225/m225_03_p01_i0_0", "m225/m225_03_f12_i0_0", "m225/m225_03_l11_i0_0", "m225/m225_04_p01_i0_0",
    "m225/m225_04_f12_i0_0", "m225/m225_04_l11_i0_0", "m225/m225_05_p01_i0_0", "m225/m225_05_f12_i0_0", "m225/m225_05_l11_i0_0",
    "m225/m225_06_p01_i0_0", "m225/m225_06_f12_i0_0", "m225/m225_06_l11_i0_0", "m225/m225_07_p01_i0_0", "m225/m225_07_f12_i0_0",
    "m225/m225_07_l11_i0_0", "m225/m225_08_p01_i0_0", "m225/m225_08_f07_i0_0", "m225/m225_08_l06_i0_0", "m225/m225_09_p01_i0_0",
    "m225/m225_09_f07_i0_0", "m225/m225_09_l06_i0_0", "m225/m225_10_p01_i0_0", "m225/m225_10_f07_i0_0", "m225/m225_10_l06_i0_0",
    "m225/m225_11_p01_i0_0", "m225/m225_11_f07_i0_0", "m225/m225_11_l06_i0_0", "m225/m225_12_p01_i0_0", "m225/m225_12_f07_i0_0",
    "m225/m225_12_l06_i0_0", "m225/m225_13_p01_i0_0", "m225/m225_13_f07_i0_0", "m225/m225_13_l06_i0_0", "m227/m227_01_p01_i0_0",
    "m227/m227_01_f12_i0_0", "m227/m227_01_l11_i0_0", "m227/m227_02_p01_i0_0", "m227/m227_02_f12_i0_0", "m227/m227_02_l11_i0_0",
    "m227/m227_03_p01_i0_0", "m227/m227_03_f12_i0_0", "m227/m227_03_l11_i0_0", "m227/m227_04_p01_i0_0", "m227/m227_04_f12_i0_0",
    "m227/m227_04_l11_i0_0", "m227/m227_05_p01_i0_0", "m227/m227_05_f12_i0_0", "m227/m227_05_l11_i0_0", "m227/m227_06_p01_i0_0",
    "m227/m227_06_f12_i0_0", "m227/m227_06_l11_i0_0", "m227/m227_07_p01_i0_0", "m227/m227_07_f07_i0_0", "m227/m227_07_l06_i0_0",
    "m227/m227_08_p01_i0_0", "m227/m227_08_f07_i0_0", "m227/m227_08_l06_i0_0", "m227/m227_09_p01_i0_0", "m227/m227_09_f07_i0_0",
    "m227/m227_09_l06_i0_0", "m227/m227_10_p01_i0_0", "m227/m227_10_f07_i0_0", "m227/m227_10_l06_i0_0", "m227/m227_11_p01_i0_0",
    "m227/m227_11_f07_i0_0", "m227/m227_11_l06_i0_0", "m227/m227_12_p01_i0_0", "m227/m227_12_f07_i0_0", "m227/m227_12_l06_i0_0",
    "m228/m228_01_p01_i0_0", "m228/m228_01_f12_i0_0", "m228/m228_01_l11_i0_0", "m228/m228_02_p01_i0_0", "m228/m228_02_f12_i0_0",
    "m228/m228_02_l11_i0_0", "m228/m228_03_p01_i0_0", "m228/m228_03_f12_i0_0", "m228/m228_03_l11_i0_0", "m228/m228_04_p01_i0_0",
    "m228/m228_04_f12_i0_0", "m228/m228_04_l11_i0_0", "m228/m228_05_p01_i0_0", "m228/m228_05_f12_i0_0", "m228/m228_05_l11_i0_0",
    "m228/m228_06_p01_i0_0", "m228/m228_06_f12_i0_0", "m228/m228_06_l11_i0_0", "m228/m228_07_p01_i0_0", "m228/m228_07_f07_i0_0",
    "m228/m228_07_l06_i0_0", "m228/m228_08_p01_i0_0", "m228/m228_08_f07_i0_0", "m228/m228_08_l06_i0_0", "m228/m228_09_p01_i0_0",
    "m228/m228_09_f07_i0_0", "m228/m228_09_l06_i0_0", "m228/m228_10_p01_i0_0", "m228/m228_10_f07_i0_0", "m228/m228_10_l06_i0_0",
    "m228/m228_11_p01_i0_0", "m228/m228_11_f07_i0_0", "m228/m228_11_l06_i0_0", "m228/m228_12_p01_i0_0", "m228/m228_12_f07_i0_0",
    "m228/m228_12_l06_i0_0", "m501/m501_01_p01_i0_0", "m501/m501_01_f12_i0_0", "m501/m501_01_l11_i0_0", "m501/m501_02_p01_i0_0",
    "m501/m501_02_f12_i0_0", "m501/m501_02_l11_i0_0", "m501/m501_03_p01_i0_0", "m501/m501_03_f12_i0_0", "m501/m501_03_l11_i0_0",
    "m501/m501_04_p01_i0_0", "m501/m501_04_f12_i0_0", "m501/m501_04_l11_i0_0", "m501/m501_05_p01_i0_0", "m501/m501_05_f12_i0_0",
    "m501/m501_05_l11_i0_0", "m501/m501_06_p01_i0_0", "m501/m501_06_f12_i0_0", "m501/m501_06_l11_i0_0", "m501/m501_07_p01_i0_0",
    "m501/m501_07_f07_i0_0", "m501/m501_07_l06_i0_0", "m501/m501_08_p01_i0_0", "m501/m501_08_f07_i0_0", "m501/m501_08_l06_i0_0",
    "m501/m501_09_p01_i0_0", "m501/m501_09_f07_i0_0", "m501/m501_09_l06_i0_0", "m501/m501_10_p01_i0_0", "m501/m501_10_f07_i0_0",
    "m501/m501_10_l06_i0_0", "m501/m501_11_p01_i0_0", "m501/m501_11_f07_i0_0", "m501/m501_11_l06_i0_0", "m501/m501_12_p01_i0_0",
    "m501/m501_12_f07_i0_0", "m501/m501_12_l06_i0_0", "m503/m503_01_p01_i0_0", "m503/m503_01_f12_i0_0", "m503/m503_01_l11_i0_0",
    "m503/m503_02_p01_i0_0", "m503/m503_02_f12_i0_0", "m503/m503_02_l11_i0_0", "m503/m503_03_p01_i0_0", "m503/m503_03_f12_i0_0",
    "m503/m503_03_l11_i0_0", "m503/m503_04_p01_i0_0", "m503/m503_04_f12_i0_0", "m503/m503_04_l11_i0_0", "m503/m503_05_p01_i0_0",
    "m503/m503_05_f12_i0_0", "m503/m503_05_l11_i0_0", "m503/m503_06_p01_i0_0", "m503/m503_06_f12_i0_0", "m503/m503_06_l11_i0_0",
    "m503/m503_07_p01_i0_0", "m503/m503_07_f07_i0_0", "m503/m503_07_l06_i0_0", "m503/m503_08_p01_i0_0", "m503/m503_08_f07_i0_0",
    "m503/m503_08_l06_i0_0", "m503/m503_09_p01_i0_0", "m503/m503_09_f07_i0_0", "m503/m503_09_l06_i0_0", "m503/m503_10_p01_i0_0",
    "m503/m503_10_f07_i0_0", "m503/m503_10_l06_i0_0", "m503/m503_11_p01_i0_0", "m503/m503_11_f07_i0_0", "m503/m503_11_l06_i0_0",
    "m503/m503_12_p01_i0_0", "m503/m503_12_f07_i0_0", "m503/m503_12_l06_i0_0", "m504/m504_01_p01_i0_0", "m504/m504_01_f12_i0_0",
    "m504/m504_01_l11_i0_0", "m504/m504_02_p01_i0_0", "m504/m504_02_f12_i0_0", "m504/m504_02_l11_i0_0", "m504/m504_03_p01_i0_0",
    "m504/m504_03_f12_i0_0", "m504/m504_03_l11_i0_0", "m504/m504_04_p01_i0_0", "m504/m504_04_f12_i0_0", "m504/m504_04_l11_i0_0",
    "m504/m504_05_p01_i0_0", "m504/m504_05_f12_i0_0", "m504/m504_05_l11_i0_0", "m504/m504_06_p01_i0_0", "m504/m504_06_f12_i0_0",
    "m504/m504_06_l11_i0_0", "m504/m504_07_p01_i0_0", "m504/m504_07_f07_i0_0", "m504/m504_07_l06_i0_0", "m504/m504_08_p01_i0_0",
    "m504/m504_08_f07_i0_0", "m504/m504_08_l06_i0_0", "m504/m504_09_p01_i0_0", "m504/m504_09_f07_i0_0", "m504/m504_09_l06_i0_0",
    "m504/m504_10_p01_i0_0", "m504/m504_10_f07_i0_0", "m504/m504_10_l06_i0_0", "m504/m504_11_p01_i0_0", "m504/m504_11_f07_i0_0",
    "m504/m504_11_l06_i0_0", "m504/m504_12_p01_i0_0", "m504/m504_12_f07_i0_0", "m504/m504_12_l06_i0_0", "m514/m514_01_p01_i0_0",
    "m514/m514_01_f12_i0_0", "m514/m514_01_l11_i0_0", "m514/m514_02_p01_i0_0", "m514/m514_02_f12_i0_0", "m514/m514_02_l11_i0_0",
    "m514/m514_03_p01_i0_0", "m514/m514_03_f12_i0_0", "m514/m514_03_l11_i0_0", "m514/m514_04_p01_i0_0", "m514/m514_04_f12_i0_0",
    "m514/m514_04_l11_i0_0", "m514/m514_05_p01_i0_0", "m514/m514_05_f12_i0_0", "m514/m514_05_l11_i0_0", "m514/m514_06_p01_i0_0",
    "m514/m514_06_f12_i0_0", "m514/m514_06_l11_i0_0", "m514/m514_07_p01_i0_0", "m514/m514_07_f07_i0_0", "m514/m514_07_l06_i0_0",
    "m514/m514_08_p01_i0_0", "m514/m514_08_f07_i0_0", "m514/m514_08_l06_i0_0", "m514/m514_09_p01_i0_0", "m514/m514_09_f07_i0_0",
    "m514/m514_09_l06_i0_0", "m514/m514_10_p01_i0_0", "m514/m514_10_f07_i0_0", "m514/m514_10_l06_i0_0", "m514/m514_11_p01_i0_0",
    "m514/m514_11_f07_i0_0", "m514/m514_11_l06_i0_0", "m514/m514_12_p01_i0_0", "m514/m514_12_f07_i0_0", "m514/m514_12_l06_i0_0",
    "m516/m516_01_p01_i0_0", "m516/m516_01_f12_i0_0", "m516/m516_01_l11_i0_0", "m516/m516_02_p01_i0_0", "m516/m516_02_f12_i0_0",
    "m516/m516_02_l11_i0_0", "m516/m516_03_p01_i0_0", "m516/m516_03_f12_i0_0", "m516/m516_03_l11_i0_0", "m516/m516_04_p01_i0_0",
    "m516/m516_04_f12_i0_0", "m516/m516_04_l11_i0_0", "m516/m516_05_p01_i0_0", "m516/m516_05_f12_i0_0", "m516/m516_05_l11_i0_0",
    "m516/m516_06_p01_i0_0", "m516/m516_06_f12_i0_0", "m516/m516_06_l11_i0_0", "m516/m516_07_p01_i0_0", "m516/m516_07_f07_i0_0",
    "m516/m516_07_l06_i0_0", "m516/m516_08_p01_i0_0", "m516/m516_08_f07_i0_0", "m516/m516_08_l06_i0_0", "m516/m516_09_p01_i0_0",
    "m516/m516_09_f07_i0_0", "m516/m516_09_l06_i0_0", "m516/m516_10_p01_i0_0", "m516/m516_10_f07_i0_0", "m516/m516_10_l06_i0_0",
    "m516/m516_11_p01_i0_0", "m516/m516_11_f07_i0_0", "m516/m516_11_l06_i0_0", "m516/m516_12_p01_i0_0", "m516/m516_12_f07_i0_0",
    "m516/m516_12_l06_i0_0", "m517/m517_01_p01_i0_0", "m517/m517_01_f12_i0_0", "m517/m517_01_l11_i0_0", "m517/m517_02_p01_i0_0",
    "m517/m517_02_f12_i0_0", "m517/m517_02_l11_i0_0", "m517/m517_03_p01_i0_0", "m517/m517_03_f12_i0_0", "m517/m517_03_l11_i0_0",
    "m517/m517_04_p01_i0_0", "m517/m517_04_f12_i0_0", "m517/m517_04_l11_i0_0", "m517/m517_05_p01_i0_0", "m517/m517_05_f12_i0_0",
    "m517/m517_05_l11_i0_0", "m517/m517_06_p01_i0_0", "m517/m517_06_f12_i0_0", "m517/m517_06_l11_i0_0", "m517/m517_07_p01_i0_0",
    "m517/m517_07_f07_i0_0", "m517/m517_07_l06_i0_0", "m517/m517_08_p01_i0_0", "m517/m517_08_f07_i0_0", "m517/m517_08_l06_i0_0",
    "m517/m517_09_p01_i0_0", "m517/m517_09_f07_i0_0", "m517/m517_09_l06_i0_0", "m517/m517_10_p01_i0_0", "m517/m517_10_f07_i0_0",
    "m517/m517_10_l06_i0_0", "m517/m517_11_p01_i0_0", "m517/m517_11_f07_i0_0", "m517/m517_11_l06_i0_0", "m517/m517_12_p01_i0_0",
    "m517/m517_12_f07_i0_0", "m517/m517_12_l06_i0_0", "m518/m518_01_p01_i0_0", "m518/m518_01_f12_i0_0", "m518/m518_01_l11_i0_0",
    "m518/m518_02_p01_i0_0", "m518/m518_02_f12_i0_0", "m518/m518_02_l11_i0_0", "m518/m518_03_p01_i0_0", "m518/m518_03_f12_i0_0",
    "m518/m518_03_l11_i0_0", "m518/m518_04_p01_i0_0", "m518/m518_04_f12_i0_0", "m518/m518_04_l11_i0_0", "m518/m518_05_p01_i0_0",
    "m518/m518_05_f12_i0_0", "m518/m518_05_l11_i0_0", "m518/m518_06_p01_i0_0", "m518/m518_06_f12_i0_0", "m518/m518_06_l11_i0_0",
    "m518/m518_07_p01_i0_0", "m518/m518_07_f07_i0_0", "m518/m518_07_l06_i0_0", "m518/m518_08_p01_i0_0", "m518/m518_08_f07_i0_0",
    "m518/m518_08_l06_i0_0", "m518/m518_09_p01_i0_0", "m518/m518_09_f07_i0_0", "m518/m518_09_l06_i0_0", "m518/m518_10_p01_i0_0",
    "m518/m518_10_f07_i0_0", "m518/m518_10_l06_i0_0", "m518/m518_11_p01_i0_0", "m518/m518_11_f07_i0_0", "m518/m518_11_l06_i0_0",
    "m518/m518_12_p01_i0_0", "m518/m518_12_f07_i0_0", "m518/m518_12_l06_i0_0", "m520/m520_01_p01_i0_0", "m520/m520_01_f12_i0_0",
    "m520/m520_01_l11_i0_0", "m520/m520_02_p01_i0_0", "m520/m520_02_f12_i0_0", "m520/m520_02_l11_i0_0", "m520/m520_03_p01_i0_0",
    "m520/m520_03_f12_i0_0", "m520/m520_03_l11_i0_0", "m520/m520_04_p01_i0_0", "m520/m520_04_f12_i0_0", "m520/m520_04_l11_i0_0",
    "m520/m520_05_p01_i0_0", "m520/m520_05_f12_i0_0", "m520/m520_05_l11_i0_0", "m520/m520_06_p01_i0_0", "m520/m520_06_f12_i0_0",
    "m520/m520_06_l11_i0_0", "m520/m520_07_p01_i0_0", "m520/m520_07_f07_i0_0", "m520/m520_07_l06_i0_0", "m520/m520_08_p01_i0_0",
    "m520/m520_08_f07_i0_0", "m520/m520_08_l06_i0_0", "m520/m520_09_p01_i0_0", "m520/m520_09_f07_i0_0", "m520/m520_09_l06_i0_0",
    "m520/m520_10_p01_i0_0", "m520/m520_10_f07_i0_0", "m520/m520_10_l06_i0_0", "m520/m520_11_p01_i0_0", "m520/m520_11_f07_i0_0",
    "m520/m520_11_l06_i0_0", "m520/m520_12_p01_i0_0", "m520/m520_12_f07_i0_0", "m520/m520_12_l06_i0_0", "m521/m521_01_p01_i0_0",
    "m521/m521_01_f12_i0_0", "m521/m521_01_l11_i0_0", "m521/m521_02_p01_i0_0", "m521/m521_02_f12_i0_0", "m521/m521_02_l11_i0_0",
    "m521/m521_03_p01_i0_0", "m521/m521_03_f12_i0_0", "m521/m521_03_l11_i0_0", "m521/m521_04_p01_i0_0", "m521/m521_04_f12_i0_0",
    "m521/m521_04_l11_i0_0", "m521/m521_05_p01_i0_0", "m521/m521_05_f12_i0_0", "m521/m521_05_l11_i0_0", "m521/m521_06_p01_i0_0",
    "m521/m521_06_f12_i0_0", "m521/m521_06_l11_i0_0", "m521/m521_07_p01_i0_0", "m521/m521_07_f07_i0_0", "m521/m521_07_l06_i0_0",
    "m521/m521_08_p01_i0_0", "m521/m521_08_f07_i0_0", "m521/m521_08_l06_i0_0", "m521/m521_09_p01_i0_0", "m521/m521_09_f07_i0_0",
    "m521/m521_09_l06_i0_0", "m521/m521_10_p01_i0_0", "m521/m521_10_f07_i0_0", "m521/m521_10_l06_i0_0", "m521/m521_11_p01_i0_0",
    "m521/m521_11_f07_i0_0", "m521/m521_11_l06_i0_0", "m521/m521_12_p01_i0_0", "m521/m521_12_f07_i0_0", "m521/m521_12_l06_i0_0",
    "m522/m522_01_p01_i0_0", "m522/m522_01_f12_i0_0", "m522/m522_01_l11_i0_0", "m522/m522_02_p01_i0_0", "m522/m522_02_f12_i0_0",
    "m522/m522_02_l11_i0_0", "m522/m522_03_p01_i0_0", "m522/m522_03_f12_i0_0", "m522/m522_03_l11_i0_0", "m522/m522_04_p01_i0_0",
    "m522/m522_04_f12_i0_0", "m522/m522_04_l11_i0_0", "m522/m522_05_p01_i0_0", "m522/m522_05_f12_i0_0", "m522/m522_05_l11_i0_0",
    "m522/m522_06_p01_i0_0", "m522/m522_06_f12_i0_0", "m522/m522_06_l11_i0_0", "m522/m522_07_p01_i0_0", "m522/m522_07_f07_i0_0",
    "m522/m522_07_l06_i0_0", "m522/m522_08_p01_i0_0", "m522/m522_08_f07_i0_0", "m522/m522_08_l06_i0_0", "m522/m522_09_p01_i0_0",
    "m522/m522_09_f07_i0_0", "m522/m522_09_l06_i0_0", "m522/m522_10_p01_i0_0", "m522/m522_10_f07_i0_0", "m522/m522_10_l06_i0_0",
    "m522/m522_11_p01_i0_0", "m522/m522_11_f07_i0_0", "m522/m522_11_l06_i0_0", "m522/m522_12_p01_i0_0", "m522/m522_12_f07_i0_0",
    "m522/m522_12_l06_i0_0", "m524/m524_01_p01_i0_0", "m524/m524_01_f12_i0_0", "m524/m524_01_l11_i0_0", "m524/m524_02_p01_i0_0",
    "m524/m524_02_f12_i0_0", "m524/m524_02_l11_i0_0", "m524/m524_03_p01_i0_0", "m524/m524_03_f12_i0_0", "m524/m524_03_l11_i0_0",
    "m524/m524_04_p01_i0_0", "m524/m524_04_f12_i0_0", "m524/m524_04_l11_i0_0", "m524/m524_05_p01_i0_0", "m524/m524_05_f12_i0_0",
    "m524/m524_05_l11_i0_0", "m524/m524_06_p01_i0_0", "m524/m524_06_f12_i0_0", "m524/m524_06_l11_i0_0", "m524/m524_07_p01_i0_0",
    "m524/m524_07_f07_i0_0", "m524/m524_07_l06_i0_0", "m524/m524_08_p01_i0_0", "m524/m524_08_f07_i0_0", "m524/m524_08_l06_i0_0",
    "m524/m524_09_p01_i0_0", "m524/m524_09_f07_i0_0", "m524/m524_09_l06_i0_0", "m524/m524_10_p01_i0_0", "m524/m524_10_f07_i0_0",
    "m524/m524_10_l06_i0_0", "m524/m524_11_p01_i0_0", "m524/m524_11_f07_i0_0", "m524/m524_11_l06_i0_0", "m524/m524_12_p01_i0_0",
    "m524/m524_12_f07_i0_0", "m524/m524_12_l06_i0_0", "m526/m526_01_p01_i0_0", "m526/m526_01_f12_i0_0", "m526/m526_01_l11_i0_0",
    "m526/m526_02_p01_i0_0", "m526/m526_02_f12_i0_0", "m526/m526_02_l11_i0_0", "m526/m526_03_p01_i0_0", "m526/m526_03_f12_i0_0",
    "m526/m526_03_l11_i0_0", "m526/m526_04_p01_i0_0", "m526/m526_04_f12_i0_0", "m526/m526_04_l11_i0_0", "m526/m526_05_p01_i0_0",
    "m526/m526_05_f12_i0_0", "m526/m526_05_l11_i0_0", "m526/m526_06_p01_i0_0", "m526/m526_06_f12_i0_0", "m526/m526_06_l11_i0_0",
    "m526/m526_07_p01_i0_0", "m526/m526_07_f07_i0_0", "m526/m526_07_l06_i0_0", "m526/m526_08_p01_i0_0", "m526/m526_08_f07_i0_0",
    "m526/m526_08_l06_i0_0", "m526/m526_09_p01_i0_0", "m526/m526_09_f07_i0_0", "m526/m526_09_l06_i0_0", "m526/m526_10_p01_i0_0",
    "m526/m526_10_f07_i0_0", "m526/m526_10_l06_i0_0", "m526/m526_11_p01_i0_0", "m526/m526_11_f07_i0_0", "m526/m526_11_l06_i0_0",
    "m526/m526_12_p01_i0_0", "m526/m526_12_f07_i0_0", "m526/m526_12_l06_i0_0", "m527/m527_01_p01_i0_0", "m527/m527_01_f12_i0_0",
    "m527/m527_01_l11_i0_0", "m527/m527_02_p01_i0_0", "m527/m527_02_f12_i0_0", "m527/m527_02_l11_i0_0", "m527/m527_03_p01_i0_0",
    "m527/m527_03_f12_i0_0", "m527/m527_03_l11_i0_0", "m527/m527_04_p01_i0_0", "m527/m527_04_f12_i0_0", "m527/m527_04_l11_i0_0",
    "m527/m527_05_p01_i0_0", "m527/m527_05_f12_i0_0", "m527/m527_05_l11_i0_0", "m527/m527_06_p01_i0_0", "m527/m527_06_f12_i0_0",
    "m527/m527_06_l11_i0_0", "m527/m527_07_p01_i0_0", "m527/m527_07_f07_i0_0", "m527/m527_07_l06_i0_0", "m527/m527_08_p01_i0_0",
    "m527/m527_08_f07_i0_0", "m527/m527_08_l06_i0_0", "m527/m527_09_p01_i0_0", "m527/m527_09_f07_i0_0", "m527/m527_09_l06_i0_0",
    "m527/m527_10_p01_i0_0", "m527/m527_10_f07_i0_0", "m527/m527_10_l06_i0_0", "m527/m527_11_p01_i0_0", "m527/m527_11_f07_i0_0",
    "m527/m527_11_l06_i0_0", "m527/m527_12_p01_i0_0", "m527/m527_12_f07_i0_0", "m527/m527_12_l06_i0_0"]

  snames = ['onethird', 'twothirds', 'twothirds-subsampled']
  onethird_list = [229, 502, 515, 529, 204, 205, 211, 220, 222, 226,
                   233, 505, 511, 512, 519, 523]
  twothirds_list = [214, 218, 230, 232, 507, 508, 509, 510, 528, 202,
                    203, 207, 208, 212, 215, 217, 223, 224, 225, 227,
                    228, 501, 503, 504, 514, 516, 517, 518, 520, 521,
                    522, 524, 526, 527]
  slists = [onethird_list, twothirds_list, twothirds_list]
  for k in range(len(snames)):
    if verbose: print "Adding subworld '%s'..." %(snames[k], )
    su = Subworld(snames[k])
    session.add(su)
    session.flush()
    session.refresh(su)
    l = slists[k]
    if k != 2: # Not twothirds-subsampled
      # Add clients
      for c_id in l:
        if verbose>1: print "  Adding client '%d' to subworld '%s'..." %(c_id, snames[k])
        su.clients.append(session.query(Client).filter(Client.id == c_id).first())
        # Add all files from this client
        q = session.query(File).join(Client).filter(Client.id == c_id)
        for c_file in q:
          if verbose>1: print "    Adding file '%s' to subworld '%s'..." %(c_file.path, snames[k])
          su.files.append(c_file)
    else: # twothirds-subsampled: Files were randomly selected from twothirds
      # Add clients
      for c_id in l:
        if verbose>1: print "  Adding client '%d' to subworld '%s'..." %(c_id, snames[k])
        su.clients.append(session.query(Client).filter(Client.id == c_id).first())
      # Add subsampled files only
      for path in twothirds_subsampled_filelist:
        q = session.query(File).filter(File.path == path)
        for c_file in q:
          if verbose>1: print "  Adding file '%s' to subworld '%s'..." %(c_file.path, snames[k])
          su.files.append(c_file)

def add_protocols(session, verbose):
  """Adds protocols"""

  # 1. DEFINITIONS
  # Numbers in the lists correspond to session identifiers
  protocol_definitions = {}

  # Protocol male: list of (client_id, first_session_id) # few exceptions with 2 as first session
  clients = [(  1,1), (  2,1), (  4,1), (  8,1), ( 11,1), ( 12,1), ( 15,1), ( 16,1), ( 17,1), ( 19,2),
             ( 21,1), ( 23,1), ( 24,1), ( 25,1), ( 26,1), ( 28,1), ( 29,1), ( 30,1), ( 31,1), ( 33,1),
             ( 34,1), (103,1), (104,1), (106,1), (107,1), (108,1), (109,1), (111,1), (112,1), (114,1),
             (115,1), (116,1), (117,1), (119,1), (120,1), (301,1), (304,1), (305,1), (308,1), (310,1),
             (313,1), (314,1), (315,1), (317,1), (319,1), (416,1), (417,1), (418,1), (419,1), (420,1),
             (421,1), (422,1), (423,1), (424,1), (425,1), (426,1), (427,1), (428,1), (429,1), (430,1),
             (431,1), (432,1)]
  enrol = ['p']
  probe = ['r', 'f']
  protocol_definitions['male'] = [clients, enrol, probe]

  # Protocol female
  clients = [(  7,2), (  9,1), ( 10,1), ( 22,1), ( 32,1), (118,1), (122,1), (123,1), (125,1), (126,1),
             (127,1), (128,1), (129,1), (130,1), (131,1), (133,1), (302,1), (303,1), (306,1), (307,1),
             (309,1), (311,1), (320,1), (401,1), (402,1), (403,1), (404,1), (405,2), (406,1), (407,1),
             (408,1), (409,1), (410,1), (411,1), (412,1), (413,1), (415,1), (433,1)]
  enrol = ['p']
  probe = ['r', 'f']
  protocol_definitions['female'] = [clients, enrol, probe]

  # 2. ADDITIONS TO THE SQL DATABASE
  protocolPurpose_list = [('world', 'train'), ('dev', 'enrol'), ('dev', 'probe'), ('eval', 'enrol'), ('eval', 'probe')]
  for proto in protocol_definitions:
    p = Protocol(proto)
    # Add protocol
    if verbose: print "Adding protocol '%s'..." % (proto)
    session.add(p)
    session.flush()
    session.refresh(p)

    # Add protocol purposes
    for key in range(len(protocolPurpose_list)):
      purpose = protocolPurpose_list[key]
      pu = ProtocolPurpose(p.id, purpose[0], purpose[1])
      if verbose>1: print "  Adding protocol purpose ('%s','%s')..." % (purpose[0], purpose[1])
      session.add(pu)
      session.flush()
      session.refresh(pu)

       # Add files attached with this protocol purpose
      client_group = ""
      speech_list = []
      if(key == 0): client_group = "world"
      elif(key == 1 or key == 2): client_group = "dev"
      elif(key == 3 or key == 4): client_group = "eval"
      if(key == 0):
        world_list = True
        session_list_in = False
      if(key == 1 or key == 3):
        world_list = False
        session_list_in = True
        speech_list = protocol_definitions[proto][1]
      elif(key == 2 or key == 4):
        world_list = False
        session_list_in = False
        speech_list = protocol_definitions[proto][2]

      # Adds 'protocol' files
      # World set
      if world_list:
        q = session.query(File).join(Client).filter(Client.sgroup == 'world').order_by(File.id)
        for k in q:
          if verbose>1: print "    Adding protocol file '%s'..." % (k.path)
          pu.files.append(k)
      # Dev/eval set
      else:
        for client in protocol_definitions[proto][0]:
          cid = client[0] # client id
          sid = client[1] # session id
          q = session.query(File).join(Client).\
                filter(Client.sgroup == client_group).filter(Client.id == cid)
          if session_list_in: q = q.filter(File.session_id == sid)
          else: q = q.filter(File.session_id != sid)
          if speech_list:
            q = q.filter(File.speech_type.in_(speech_list))
          q = q.order_by(File.id)
          for k in q:
            if verbose>1: print "    Adding protocol file '%s'..." % (k.path)
            pu.files.append(k)

def add_tmodels(session, verbose):
  """Adds T-Norm models"""

  # T-Models: client followed by list of session_ids (one session is used for one model,
  # leading to several T-Norm models per identity
  tmodels_list = [(214, ['01', '02', '03', '04', '05', '06', '07', '08', '09', '10', '11', '12']),
                  (218, ['01', '02', '03', '09', '10', '11', '12', '13', '14', '15', '16', '17']),
                  (229, ['01', '02', '03', '06', '07', '08', '09', '10', '11', '12', '13', '14']),
                  (230, ['01', '02', '03', '04', '05', '06', '07', '08', '09', '10', '11', '12']),
                  (232, ['01', '02', '03', '04', '05', '07', '08', '09', '10', '11', '12', '13']),
                  (502, ['01', '02', '03', '04', '05', '06', '07', '08', '09', '10', '11', '12']),
                  (507, ['01', '02', '03', '04', '05', '06', '07', '08', '09', '10', '11', '12']),
                  (508, ['01', '02', '03', '04', '05', '06', '07', '08', '09', '10', '11', '12']),
                  (509, ['01', '02', '03', '04', '05', '06', '07', '08', '09', '10', '11', '12']),
                  (510, ['01', '02', '03', '04', '05', '06', '07', '08', '09', '10', '11', '12']),
                  (515, ['01', '02', '03', '04', '05', '06', '07', '08', '09', '10', '11', '12']),
                  (528, ['01', '02', '03', '04', '05', '06', '07', '08', '09', '10', '11', '12']),
                  (529, ['01', '02', '03', '04', '05', '06', '07', '08', '09', '10', '11', '12']),
                  (202, ['01', '02', '03', '04', '05', '06', '07', '08', '09', '10', '11', '12']),
                  (203, ['01', '03', '04', '05', '06', '07', '08', '09', '10', '11', '12', '13']),
                  (204, ['01', '02', '04', '05', '06', '07', '08', '09', '10', '11', '12', '13']),
                  (205, ['01', '02', '03', '04', '05', '06', '07', '08', '09', '10', '11', '12']),
                  (207, ['01', '02', '03', '04', '05', '06', '07', '08', '09', '10', '11', '12']),
                  (208, ['01', '02', '03', '04', '05', '06', '07', '08', '09', '10', '11', '12']),
                  (211, ['01', '02', '03', '04', '05', '06', '07', '08', '09', '10', '11', '12']),
                  (212, ['01', '02', '03', '04', '05', '06', '07', '08', '09', '10', '11', '12']),
                  (215, ['02', '03', '04', '05', '06', '07', '08', '09', '10', '11', '12', '13']),
                  (217, ['01', '02', '03', '04', '05', '06', '07', '08', '09', '10', '11', '12']),
                  (220, ['01', '02', '03', '04', '05', '06', '07', '08', '09', '10', '11', '12']),
                  (222, ['01', '02', '03', '04', '06', '07', '08', '09', '10', '11', '12', '13']),
                  (223, ['01', '02', '03', '04', '05', '06', '07', '08', '09', '10', '11', '12']),
                  (224, ['01', '02', '03', '04', '05', '06', '07', '08', '09', '10', '11', '12']),
                  (225, ['01', '03', '04', '05', '06', '07', '08', '09', '10', '11', '12', '13']),
                  (226, ['01', '02', '03', '04', '06', '07', '08', '09', '10', '11', '12', '13']),
                  (227, ['01', '02', '03', '04', '05', '06', '07', '08', '09', '10', '11', '12']),
                  (228, ['01', '02', '03', '04', '05', '06', '07', '08', '09', '10', '11', '12']),
                  (233, ['01', '02', '03', '04', '05', '06', '07', '08', '09', '10', '11', '12']),
                  (501, ['01', '02', '03', '04', '05', '06', '07', '08', '09', '10', '11', '12']),
                  (503, ['01', '02', '03', '04', '05', '06', '07', '08', '09', '10', '11', '12']),
                  (504, ['01', '02', '03', '04', '05', '06', '07', '08', '09', '10', '11', '12']),
                  (505, ['01', '02', '03', '04', '05', '06', '07', '08', '09', '10', '11', '12']),
                  (511, ['01', '02', '03', '04', '05', '06', '07', '08', '09', '10', '11', '12']),
                  (512, ['01', '02', '03', '04', '05', '06', '07', '08', '09', '10', '11', '12']),
                  (514, ['01', '02', '03', '04', '05', '06', '07', '08', '09', '10', '11', '12']),
                  (516, ['01', '02', '03', '04', '05', '06', '07', '08', '09', '10', '11', '12']),
                  (517, ['01', '02', '03', '04', '05', '06', '07', '08', '09', '10', '11', '12']),
                  (518, ['01', '02', '03', '04', '05', '06', '07', '08', '09', '10', '11', '12']),
                  (519, ['01', '02', '03', '04', '05', '06', '07', '08', '09', '10', '11', '12']),
                  (520, ['01', '02', '03', '04', '05', '06', '07', '08', '09', '10', '11', '12']),
                  (521, ['01', '02', '03', '04', '05', '06', '07', '08', '09', '10', '11', '12']),
                  (522, ['01', '02', '03', '04', '05', '06', '07', '08', '09', '10', '11', '12']),
                  (523, ['01', '02', '03', '04', '05', '06', '07', '08', '09', '10', '11', '12']),
                  (524, ['01', '02', '03', '04', '05', '06', '07', '08', '09', '10', '11', '12']),
                  (526, ['01', '02', '03', '04', '05', '06', '07', '08', '09', '10', '11', '12']),
                  (527, ['01', '02', '03', '04', '05', '06', '07', '08', '09', '10', '11', '12'])]

  if verbose: print "Adding T-Norm models..."
  for model_list in tmodels_list:
    cid = model_list[0]
    slist = model_list[1]
    for sid in slist:
      tmodel_name = str(cid) + '_' + sid
      tmodel = TModel(tmodel_name, cid)
      if verbose>1: print "  Adding T-norm model ('%s')..." % tmodel_name
      session.add(tmodel)
      session.flush()
      session.refresh(tmodel)
      q = session.query(File).join(Client).\
            filter(and_(Client.id == cid, File.session_id == int(sid), File.speech_type == 'p')).\
            order_by(File.id)
      for k in q:
        if verbose>1: print "    Adding T-norm file '%s' to model '%s'..." % (k.path, tmodel_name)
        tmodel.files.append(k)

def create_tables(args):
  """Creates all necessary tables (only to be used at the first time)"""

  from bob.db.utils import create_engine_try_nolock

  engine = create_engine_try_nolock(args.type, args.files[0], echo=(args.verbose > 2))
  Base.metadata.create_all(engine)

# Driver API
# ==========

def create(args):
  """Creates or re-creates this database"""

  from bob.db.utils import session_try_nolock

  dbfile = args.files[0]

  if args.recreate:
    if args.verbose and os.path.exists(dbfile):
      print('unlinking %s...' % dbfile)
    if os.path.exists(dbfile): os.unlink(dbfile)

  if not os.path.exists(os.path.dirname(dbfile)):
    os.makedirs(os.path.dirname(dbfile))

  # the real work...
  create_tables(args)
  s = session_try_nolock(args.type, args.files[0], echo=(args.verbose > 2))
  add_files(s, args.imagedir, args.verbose)
  add_subworlds(s, args.verbose)
  add_protocols(s, args.verbose)
  add_tmodels(s, args.verbose)
  s.commit()
  s.close()

def add_command(subparsers):
  """Add specific subcommands that the action "create" can use"""

  parser = subparsers.add_parser('create', help=create.__doc__)

  parser.add_argument('-R', '--recreate', action='store_true', help="If set, I'll first erase the current database")
  parser.add_argument('-v', '--verbose', action='count', help="Do SQL operations in a verbose way?")
  parser.add_argument('-D', '--imagedir', metavar='DIR', default='/idiap/group/biometric/databases/mobio/still/images/selected-images/', help="Change the relative path to the directory containing the images of the MOBIO database.")

  parser.set_defaults(func=create) #action
