#MAEC Malware Subject Class

#Copyright (c) 2014, The MITRE Corporation
#All rights reserved

#Compatible with MAEC v4.1
#Last updated 02/18/2014

from cybox.common import VocabString, PlatformSpecification, ToolInformationList
from cybox.objects.file_object import File
from cybox.objects.uri_object import URI
from cybox.core import Object

import maec
import maec.bindings.maec_package as package_binding
from maec.bundle.bundle import Bundle
from maec.package.action_equivalence import ActionEquivalenceList
from maec.package.analysis import Analysis
from maec.package.malware_subject_reference import MalwareSubjectReference
from maec.package.object_equivalence import ObjectEquivalenceList


class MalwareSubject(maec.Entity):
    _namespace = maec.package._namespace

    def __init__(self, id, malware_instance_object_attributes = None):
        super(MalwareSubject, self).__init__()
        self.id = id
        #Set the Malware Instance Object Attributes (a CybOX object) if they are not none
        self.malware_instance_object_attributes = malware_instance_object_attributes
        self.label = []
        self.configuration_details = None
        self.minor_variants = MinorVariants()
        self.development_environment = None
        self.field_data = None
        #Instantiate the lists
        self.analyses = Analyses()
        self.findings_bundles = FindingsBundleList()
        self.relationships = MalwareSubjectRelationshipList()
        self.compatible_platform = []

    #Public methods
    #Set the Malware_Instance_Object_Attributes with a CybOX object
    def set_malware_instance_object_attributes(self, malware_instance_object_attributes):
        self.malware_instance_object_attributes = malware_instance_object_attributes

    #Add an Analysis to the Analyses
    def add_analysis(self, analysis):
        self.analyses.append(analysis)

    def get_analyses(self):
        return self.analyses

    #Get all Bundles in the Subject
    def get_all_bundles(self):
        return self.findings_bundles.bundles

    #Add a MAEC Bundle to the Findings Bundles
    def add_findings_bundle(self, bundle):
        self.findings_bundles.add_bundle(bundle)

    def to_obj(self):
        malware_subject_obj = package_binding.MalwareSubjectType(id = self.id)
        if self.malware_instance_object_attributes is not None: malware_subject_obj.set_Malware_Instance_Object_Attributes(self.malware_instance_object_attributes.to_obj())
        if self.minor_variants: malware_subject_obj.set_Minor_Variants(self.minor_variants.to_obj())
        if self.configuration_details: malware_subject_obj.set_Configuration_Details(self.configuration_details.to_obj())
        if self.development_environment: malware_subject_obj.set_Development_Environment(self.development_environment.to_obj())
        if self.field_data is not None: malware_subject_obj.set_Field_Data(self.field_data.to_obj())
        if self.analyses: malware_subject_obj.set_Analyses(self.analyses.to_obj())
        if self.findings_bundles is not None : malware_subject_obj.set_Findings_Bundles(self.findings_bundles.to_obj())
        if self.relationships: malware_subject_obj.set_Relationships(self.relationships.to_obj())
        if self.label:
            for labl in self.label:
                malware_subject_obj.add_Label(labl.to_obj())
        if self.compatible_platform:
            for platform in self.compatible_platform:
                malware_subject_obj.add_Compatible_Platform(platform.to_obj())
        return malware_subject_obj

    def to_dict(self):
        malware_subject_dict = {}
        if self.id is not None : malware_subject_dict['id'] = self.id
        if self.malware_instance_object_attributes is not None: malware_subject_dict['malware_instance_object_attributes'] = self.malware_instance_object_attributes.to_dict()
        if self.minor_variants : malware_subject_dict['minor_variants'] = self.minor_variants.to_list()
        if self.configuration_details : malware_subject_dict['configuration_details'] = self.configuration_details.to_dict()
        if self.development_environment : malware_subject_dict['development_environment'] = self.development_environment.to_dict()
        if self.field_data is not None: malware_subject_dict['field_data'] = self.field_data.to_dict()
        if self.analyses : malware_subject_dict['analyses'] = self.analyses.to_list()
        if self.findings_bundles is not None : malware_subject_dict['findings_bundles'] = self.findings_bundles.to_dict()
        if self.relationships : malware_subject_dict['relationships'] = self.relationships.to_list()
        if self.label:
            malware_subject_dict['label'] = [x.to_dict() for x in self.label]
        if self.compatible_platform:
            malware_subject_dict['compatible_platform'] = [x.to_dict() for x in self.compatible_platform]
        return malware_subject_dict

    #Build the Malware Subject from the input dictionary
    @staticmethod
    def from_dict(malware_subject_dict):
        if not malware_subject_dict:
            return None
        malware_subject_ = MalwareSubject(None)
        malware_subject_.id = malware_subject_dict.get('id')
        malware_subject_.malware_instance_object_attributes = Object.from_dict(malware_subject_dict.get('malware_instance_object_attributes'))
        malware_subject_.minor_variants = MinorVariants.from_list(malware_subject_dict.get('minor_variants'))
        malware_subject_.configuration_details = MalwareConfigurationDetails.from_dict(malware_subject_dict.get('configuration_details'))
        malware_subject_.development_environment = MalwareDevelopmentEnvironment.from_dict(malware_subject_dict.get('development_environment'))
        malware_subject_.field_data = None #TODO: add support
        malware_subject_.analyses = Analyses.from_list(malware_subject_dict.get('analyses'))
        malware_subject_.findings_bundles = FindingsBundleList.from_dict(malware_subject_dict.get('findings_bundles'))
        malware_subject_.relationships = MalwareSubjectRelationshipList.from_list(malware_subject_dict.get('id'))
        if malware_subject_dict.get('label'):
            malware_subject_.label = [VocabString.from_dict(x) for x in malware_subject_dict.get('label')]
        if malware_subject_dict.get('compatible_platform'):
            malware_subject_.compatible_platform = [PlatformSpecification.from_dict(x) for x in malware_subject_dict.get('compatible_platform')]
        return malware_subject_

    @staticmethod
    def from_obj(malware_subject_obj):
        if not malware_subject_obj:
            return None
        malware_subject_ = MalwareSubject(None)
        malware_subject_.id = malware_subject_obj.get_id()
        malware_subject_.malware_instance_object_attributes = Object.from_obj(malware_subject_obj.get_Malware_Instance_Object_Attributes())
        malware_subject_.minor_variants = MinorVariants.from_obj(malware_subject_obj.get_Minor_Variants())
        malware_subject_.configuration_details = MalwareConfigurationDetails.from_obj(malware_subject_obj.get_Configuration_Details())
        malware_subject_.development_environment = MalwareDevelopmentEnvironment.from_obj(malware_subject_obj.get_Development_Environment())
        malware_subject_.field_data = None #TODO: add support
        malware_subject_.analyses = Analyses.from_obj(malware_subject_obj.get_Analyses())
        malware_subject_.findings_bundles = FindingsBundleList.from_obj(malware_subject_obj.get_Findings_Bundles())
        malware_subject_.relationships = MalwareSubjectRelationshipList.from_obj(malware_subject_obj.get_Relationships())
        if malware_subject_obj.get_Label():
            malware_subject_.label = [VocabString.from_obj(x) for x in malware_subject_obj.get_Label()]
        if malware_subject_obj.get_Compatible_Platform():
            malware_subject_.compatible_platform = [PlatformSpecification.from_obj(x) for x in malware_subject_obj.get_Compatible_Platform()]
        return malware_subject_

    
class MinorVariants(maec.EntityList):
    _contained_type = Object
    _binding_class = package_binding.MinorVariantListType
    _binding_var = "Minor_Variant"
    _namespace = maec.package._namespace
    
class Analyses(maec.EntityList):
    _contained_type = Analysis
    _binding_class = package_binding.AnalysisListType
    _binding_var = "Analysis"
    _namespace = maec.package._namespace

class MalwareSubjectRelationship(maec.Entity):
    _namespace = maec.package._namespace

    def __init__(self):
        super(MalwareSubjectRelationship, self).__init__()
        self.type = None
        self.malware_subject_references = []

    def to_obj(self):
        malware_subject_relationship_obj = package_binding.MalwareSubjectRelationshipType()
        if self.type is not None : malware_subject_relationship_obj.set_Type(self.type.to_obj())
        if len(self.malware_subject_references) > 0:
            for malware_subject_reference in self.malware_subject_references:
                malware_subject_relationship_obj.add_Malware_Subject_Reference(malware_subject_reference.to_obj())
        return malware_subject_relationship_obj

    def to_dict(self):
        malware_subject_relationship_dict = {}
        if self.type is not None : malware_subject_relationship_dict['type'] = self.type.to_dict()
        if len(self.malware_subject_references) > 0:
            malware_subject_refs = []
            for malware_subject_reference in self.malware_subject_references:
                malware_subject_refs.append(malware_subject_reference.to_dict())
            malware_subject_relationship_dict['malware_subject_references'] = malware_subject_refs
        return malware_subject_relationship_dict

    @staticmethod
    def from_dict(malware_subject_relationship_dict):
        if not malware_subject_relationship_dict:
            return None
        malware_subject_relationship_ = MalwareSubjectRelationship()
        malware_subject_relationship_.type = VocabString.from_dict(malware_subject_relationship_dict.get('type'))
        malware_subject_relationship_.malware_subject_references = [MalwareSubjectReference.from_dict(x) for x in malware_subject_relationship_dict.get('malware_subject_references', [])]
        return malware_subject_relationship_

    @staticmethod
    def from_obj(malware_subject_relationship_obj):
        if not malware_subject_relationship_obj:
            return None
        malware_subject_relationship_ = MalwareSubjectRelationship()
        malware_subject_relationship_.type = VocabString.from_obj(malware_subject_relationship_obj.get_Type())
        malware_subject_relationship_.malware_subject_references = [MalwareSubjectReference.from_obj(x) for x in malware_subject_relationship_obj.get_Malware_Subject_Reference()]
        return malware_subject_relationship_

class MalwareSubjectRelationshipList(maec.EntityList):
    _contained_type = MalwareSubjectRelationship
    _binding_class = package_binding.MalwareSubjectRelationshipListType
    _binding_var = "Relationship"
    _namespace = maec.package._namespace

class FindingsBundleList(maec.Entity):
    _namespace = maec.package._namespace

    def __init__(self):
        super(FindingsBundleList, self).__init__()
        self.meta_analysis = None
        self.bundles = []
        self.bundle_external_references = []

    def add_bundle(self, bundle):
        self.bundles.append(bundle)

    def add_bundle_external_reference(self, bundle_external_reference):
        self.bundle_external_references.append(bundle_external_reference)

    def to_obj(self):
        findings_bundle_list_obj = package_binding.FindingsBundleListType()
        if self.meta_analysis is not None : findings_bundle_list_obj.set_Meta_Analysis(self.meta_analysis.to_obj())
        if len(self.bundles) > 0: 
            for bundle in self.bundles: findings_bundle_list_obj.add_Bundle(bundle.to_obj())
        if len(self.bundle_external_references) > 0: 
            for bundle_external_reference in self.bundle_external_references: findings_bundle_list_obj.add_Bundle_External_Reference(bundle_external_reference)
        return findings_bundle_list_obj

    def to_dict(self):
        findings_bundle_list_dict = {}
        if self.meta_analysis is not None : findings_bundle_list_dict['meta_analysis'] = self.meta_analysis.to_dict()
        if len(self.bundles) > 0: 
            bundle_list = []
            for bundle in self.bundles: bundle_list.append(bundle.to_dict())
            findings_bundle_list_dict['bundles'] = bundle_list
        if len(self.bundle_external_references) > 0: 
            bundle_external_refs_list = []
            for bundle_external_reference in self.bundle_external_references: bundle_external_refs_list.append(bundle_external_reference)
            findings_bundle_list_dict['bundle_external_references'] = bundle_external_refs_list
        return findings_bundle_list_dict

    @staticmethod
    def from_dict(findings_bundle_list_dict):
        if not findings_bundle_list_dict:
            return None
        findings_bundle_list_ = FindingsBundleList()
        findings_bundle_list_.meta_analysis = MetaAnalysis.from_dict(findings_bundle_list_dict.get('meta_analysis'))
        findings_bundle_list_.bundles = [Bundle.from_dict(x) for x in findings_bundle_list_dict.get('bundles', [])]
        findings_bundle_list_.bundle_external_references = [x for x in findings_bundle_list_dict.get('bundle_external_references', [])]
        return findings_bundle_list_

    @staticmethod
    def from_obj(findings_bundle_list_obj):
        if not findings_bundle_list_obj:
            return None
        findings_bundle_list_ = FindingsBundleList()
        findings_bundle_list_.meta_analysis = MetaAnalysis.from_obj(findings_bundle_list_obj.get_Meta_Analysis())
        findings_bundle_list_.bundles = [Bundle.from_obj(x) for x in findings_bundle_list_obj.get_Bundle()]
        findings_bundle_list_.bundle_external_references = [x for x in findings_bundle_list_obj.get_Bundle_External_Reference()]
        return findings_bundle_list_

class MetaAnalysis(maec.Entity):
    _namespace = maec.package._namespace

    def __init__(self):
        super(MetaAnalysis, self).__init__()
        self.action_equivalences = None
        self.object_equivalences = None

    def to_obj(self):
        meta_analysis_obj = package_binding.MetaAnalysisType()
        if self.action_equivalences is not None : meta_analysis_obj.set_Action_Equivalences(self.action_equivalences.to_obj())
        if self.object_equivalences is not None : meta_analysis_obj.set_Object_Equivalences(self.object_equivalences.to_obj())
        return meta_analysis_obj

    def to_dict(self):
        meta_analysis_dict = {}
        if self.action_equivalences is not None : meta_analysis_dict['action_equivalences'] = self.action_equivalences.to_list()
        if self.object_equivalences is not None : meta_analysis_dict['object_equivalences'] = self.object_equivalences.to_list()
        return meta_analysis_dict

    @staticmethod
    def from_dict(meta_analysis_dict):
        if not meta_analysis_dict:
            return None
        meta_analysis_ = MetaAnalysis()
        meta_analysis_.action_equivalences = ActionEquivalenceList.from_list(meta_analysis_dict.get('action_equivalences'))
        meta_analysis_.object_equivalences = ObjectEquivalenceList.from_list(meta_analysis_dict.get('object_equivalences'))
        return meta_analysis_

    @staticmethod
    def from_obj(meta_analysis_obj):
        if not meta_analysis_obj:
            return None
        meta_analysis_ = MetaAnalysis()
        meta_analysis_.action_equivalences = ActionEquivalenceList.from_obj(meta_analysis_obj.get_Action_Equivalences())
        meta_analysis_.object_equivalences = ObjectEquivalenceList.from_obj(meta_analysis_obj.get_Object_Equivalences())
        return meta_analysis_

class MalwareSubjectList(maec.EntityList):
    _contained_type = MalwareSubject
    _binding_class = package_binding.MalwareSubjectListType
    _binding_var = "Malware_Subject"
    _namespace = maec.package._namespace

class MalwareDevelopmentEnvironment(maec.Entity):
    _namespace = maec.package._namespace

    def __init__(self):
        super(MalwareDevelopmentEnvironment, self).__init__()
        self.tools = None
        self.debugging_file = None

    def to_obj(self):
        mal_dev_obj = package_binding.MalwareDevelopmentEnvironmentType()
        if self.tools is not None : mal_dev_obj.set_Tools(self.tools.to_obj())
        if self.debugging_file is not None: 
            mal_dev_obj.set_Debugging_File([x.to_obj() for x in self.debugging_file])
        return mal_dev_obj

    def to_dict(self):
        mal_dev_dict = {}
        if self.tools is not None : mal_dev_dict['tools'] = self.tools.to_list()
        if self.debugging_file is not None: 
            mal_dev_dict['debugging_file'] = [x.to_dict() for x in self.debugging_file]
        return mal_dev_dict

    @staticmethod
    def from_dict(mal_dev_dict):
        if not mal_dev_dict:
            return None
        mal_dev_env_ = MalwareDevelopmentEnvironment()
        mal_dev_env_.tools = ToolInformationList.from_list(mal_dev_dict['tools'])
        if mal_dev_dict['debugging_file']:
            mal_dev_env_.debugging_file = [File.from_dict(x) for x in mal_dev_dict['debugging_file']]
        return mal_dev_env_

    @staticmethod
    def from_obj(mal_dev_obj):
        if not mal_dev_obj:
            return None
        mal_dev_env_ = MalwareDevelopmentEnvironment()
        mal_dev_env_.tools = ToolInformationList.from_obj(mal_dev_obj.get_Tools())
        if mal_dev_obj.get_Debugging_File():
            mal_dev_env_.debugging_file = [File.from_obj(x) for x in mal_dev_obj.get_Debugging_File()]
        return mal_dev_env_

class MalwareConfigurationParameter(maec.Entity):
    _namespace = maec.package._namespace

    def __init__(self):
        super(MalwareConfigurationParameter, self).__init__()
        self.name = None
        self.value = None

    def to_obj(self):
        mal_conf_param_obj = package_binding.MalwareConfigurationParameterType()
        if self.name is not None : mal_dev_obj.set_Name(self.name.to_obj())
        if self.value is not None : mal_dev_obj.set_Value(self.value)
        return mal_conf_param_obj

    def to_dict(self):
        mal_conf_param_dict = {}
        if self.name is not None : mal_conf_param_dict['name'] = self.name.to_dict()
        if self.value is not None : mal_conf_param_dict['value'] = self.value
        return mal_conf_param_dict

    @staticmethod
    def from_dict(mal_conf_param_dict):
        if not mal_conf_param_dict:
            return None
        mal_conf_param_ = MalwareConfigurationParameter()
        mal_conf_param_.name = VocabString.from_dict(mal_conf_param_dict['name'])
        if mal_conf_param_dict['value']: mal_conf_param_.value = mal_conf_param_dict['value']
        return mal_conf_param_

    @staticmethod
    def from_obj(mal_conf_param_obj):
        if not mal_conf_param_obj:
            return None
        mal_conf_param_ = MalwareConfigurationParameter()
        mal_conf_param_.name = VocabString.from_obj(mal_conf_param_obj.get_Name())
        if mal_conf_param_obj.get_Value(): mal_conf_param_.value = mal_conf_param_obj.get_Value()
        return mal_conf_param_

class MalwareBinaryConfigurationStorageDetails(maec.Entity):
    _namespace = maec.package._namespace

    def __init__(self):
        super(MalwareBinaryConfigurationStorageDetails, self).__init__()
        self.file_offset = None
        self.section_name = None
        self.section_offset = None

    def to_obj(self):
        mal_binary_conf_storage_obj = package_binding.MalwareBinaryConfigurationStorageDetailsType()
        if self.file_offset is not None : mal_binary_conf_storage_obj.set_File_Offset(self.file_offset)
        if self.section_name is not None : mal_binary_conf_storage_obj.set_Section_Name(self.section_name)
        if self.section_offset is not None : mal_binary_conf_storage_obj.set_Section_Offset(self.section_offset)
        return mal_binary_conf_storage_obj

    def to_dict(self):
        mal_binary_conf_storage_dict = {}
        if self.file_offset is not None : mal_binary_conf_storage_dict['file_offset'] = self.file_offset
        if self.section_name is not None : mal_binary_conf_storage_dict['section_name'] = self.section_name
        if self.section_offset is not None : mal_binary_conf_storage_dict['section_offset'] = self.section_offset
        return mal_binary_conf_storage_dict

    @staticmethod
    def from_dict(mal_binary_conf_storage_dict):
        if not mal_binary_conf_storage_dict:
            return None
        mal_binary_conf_storage_ = MalwareBinaryConfigurationStorageDetails()
        if mal_conf_storage_dict['file_offset']: mal_binary_conf_storage_.file_offset = mal_conf_storage_dict['file_offset']
        if mal_conf_storage_dict['section_name']: mal_binary_conf_storage_.section_name = mal_conf_storage_dict['section_name']
        if mal_conf_storage_dict['section_offset']: mal_binary_conf_storage_.section_offset = mal_conf_storage_dict['section_offset']
        return mal_binary_conf_storage_

    @staticmethod
    def from_obj(mal_binary_conf_storage_obj):
        if not mal_binary_conf_storage_obj:
            return None
        mal_binary_conf_storage_ = MalwareBinaryConfigurationStorageDetails()
        mal_binary_conf_storage_.file_offset = mal_binary_conf_storage_obj.get_File_Offset()
        mal_binary_conf_storage_.section_name = mal_binary_conf_storage_obj.get_Section_Name()
        mal_binary_conf_storage_.section_offset = mal_binary_conf_storage_obj.get_Section_Offset()
        return mal_binary_conf_storage_

class MalwareConfigurationStorageDetails(maec.Entity):
    _namespace = maec.package._namespace

    def __init__(self):
        super(MalwareConfigurationStorageDetails, self).__init__()
        self.malware_binary = None
        self.file = None
        self.url = []

    def to_obj(self):
        mal_conf_storage_obj = package_binding.MalwareConfigurationStorageDetailsType()
        if self.malware_binary is not None : mal_conf_storage_obj.set_Malware_Binary(self.malware_binary.to_obj())
        if self.file is not None : mal_conf_storage_obj.set_File(self.file.to_obj())
        if self.url is not None: 
            mal_conf_storage_obj.set_URL([x.to_obj() for x in self.url])
        return mal_conf_storage_obj

    def to_dict(self):
        mal_conf_storage_dict = {}
        if self.malware_binary is not None : mal_conf_storage_dict['malware_binary'] = self.malware_binary.to_dict()
        if self.file is not None : mal_conf_storage_dict['file'] = self.file.to_dict()
        if self.url is not None: 
            mal_conf_storage_dict['url'] = [x.to_dict() for x in self.url]
        return mal_conf_storage_dict

    @staticmethod
    def from_dict(mal_conf_storage_dict):
        if not mal_conf_storage_dict:
            return None
        mal_conf_storage_ = MalwareConfigurationStorageDetails()
        mal_conf_storage_.malware_binary = MalwareBinaryConfigurationStorageDetails.from_dict(mal_conf_storage_dict['malware_binary'])
        mal_conf_storage_.file = File.from_dict(mal_conf_storage_dict['file'])
        if mal_conf_storage_dict['url']:
            mal_conf_storage_.url = [URI.from_dict(x) for x in mal_conf_storage_dict['configuration_parameter']]
        return mal_conf_storage_

    @staticmethod
    def from_obj(mal_conf_storage_obj):
        if not mal_conf_storage_obj:
            return None
        mal_conf_storage_ = MalwareConfigurationStorageDetails()
        mal_conf_storage_.malware_binary = MalwareBinaryConfigurationStorageDetails.from_obj(mal_conf_storage_obj.get_Malware_Binary())
        mal_conf_storage_.file = File.from_obj(mal_conf_storage_obj.get_File())
        if mal_conf_storage_obj.get_URL():
            mal_conf_storage_.url = [URI.from_obj(x) for x in mal_conf_obj.get_URL()]
        return mal_conf_storage_

class MalwareConfigurationObfuscationAlgorithm(maec.Entity):
    _namespace = maec.package._namespace

    def __init__(self):
        super(MalwareConfigurationObfuscationAlgorithm, self).__init__()
        self.ordinal_position = None
        self.key = None
        self.algorithm_name = None

    def to_obj(self):
        mal_binary_obfusc_alg_obj = package_binding.MalwareConfigurationObfuscationAlgorithmType()
        if self.ordinal_position is not None : mal_binary_obfusc_alg_obj.set_ordinal_position(self.ordinal_position)
        if self.key is not None : mal_binary_obfusc_alg_obj.set_Key(self.key)
        if self.algorithm_name is not None : mal_binary_obfusc_alg_obj.set_Algorithm_Name(self.algorithm_name.to_obj())
        return mal_binary_obfusc_alg_obj

    def to_dict(self):
        mal_binary_obfusc_alg_dict = {}
        if self.ordinal_position is not None : mal_binary_obfusc_alg_dict['ordinal_position'] = self.ordinal_position
        if self.key is not None : mal_binary_obfusc_alg_dict['key'] = self.key
        if self.algorithm_name is not None : mal_binary_obfusc_alg_dict['algorithm_name'] = self.algorithm_name.to_dict()
        return mal_binary_obfusc_alg_dict

    @staticmethod
    def from_dict(mal_binary_obfusc_alg_dict):
        if not mal_binary_obfusc_alg_dict:
            return None
        mal_binary_obfusc_alg_ = MalwareConfigurationObfuscationAlgorithm()
        if mal_binary_obfusc_alg_dict['ordinal_position']: mal_binary_obfusc_alg_.ordinal_position = mal_binary_obfusc_alg_dict['ordinal_position']
        if mal_binary_obfusc_alg_dict['key']: mal_binary_obfusc_alg_.key = mal_binary_obfusc_alg_dict['key']
        mal_binary_obfusc_alg_.algorithm_name = VocabString.from_dict(mal_binary_obfusc_alg_dict['algorithm_name'])
        return mal_binary_obfusc_alg_

    @staticmethod
    def from_obj(mal_binary_obfusc_alg_obj):
        if not mal_binary_obfusc_alg_obj:
            return None
        mal_binary_obfusc_alg_ = MalwareConfigurationObfuscationAlgorithm()
        mal_binary_obfusc_alg_.ordinal_position = mal_binary_obfusc_alg_obj.get_ordinal_position()
        mal_binary_obfusc_alg_.key = mal_binary_obfusc_alg_obj.get_Key()
        mal_binary_obfusc_alg_.algorithm_name = VocabString.from_obj(mal_binary_obfusc_alg_obj.get_Algorithm_name())
        return mal_binary_obfusc_alg_

class MalwareConfigurationObfuscationDetails(maec.Entity):
    _namespace = maec.package._namespace

    def __init__(self):
        super(MalwareConfigurationObfuscationDetails, self).__init__()
        self.is_encoded = None
        self.is_encrypted = None
        self.algorithm_details = []

    def to_obj(self):
        mal_conf_obfusc_obj = package_binding.MalwareConfigurationObfuscationDetailsType()
        if self.is_encoded is not None : mal_conf_obfusc_obj.set_is_encoded(self.is_encoded)
        if self.is_encrypted is not None : mal_conf_obfusc_obj.set_is_encrypted(self.is_encrypted)
        if self.algorithm_details is not None: 
            mal_conf_obfusc_obj.set_Algorithm_Details([x.to_obj() for x in self.algorithm_details])
        return mal_conf_obfusc_obj

    def to_dict(self):
        mal_conf_obfusc_dict = {}
        if self.is_encoded is not None : mal_conf_obfusc_dict['is_encoded'] = self.is_encoded
        if self.is_encrypted is not None : mal_conf_obfusc_dict['is_encrypted'] = self.is_encrypted
        if self.algorithm_details is not None: 
            mal_conf_obfusc_dict['algorithm_details'] = [x.to_dict() for x in self.algorithm_details]
        return mal_conf_obfusc_dict

    @staticmethod
    def from_dict(mal_conf_obfusc_dict):
        if not mal_conf_obfusc_dict:
            return None
        mal_conf_obfusc_ = MalwareConfigurationObfuscationDetails()
        if mal_conf_obfusc_dict['is_encoded']: mal_conf_obfusc_.is_encoded = mal_conf_obfusc_dict['is_encoded']
        if mal_conf_obfusc_dict['is_encrypted']: mal_conf_obfusc_.is_encrypted = mal_conf_obfusc_dict['is_encrypted']
        if mal_conf_obfusc_dict['algorithm_details']:
            mal_conf_obfusc_.algorithm_details = [MalwareConfigurationObfuscationAlgorithm.from_dict(x) for x in mal_conf_obfusc_dict['algorithm_details']]
        return mal_conf_obfusc_

    @staticmethod
    def from_obj(mal_conf_obfusc_obj):
        if not mal_conf_obfusc_obj:
            return None
        mal_conf_obfusc_ = MalwareConfigurationObfuscationDetails()
        mal_conf_obfusc_.is_encoded = mal_conf_obfusc_obj.get_is_encoded()
        mal_conf_obfusc_.is_encrypted = mal_conf_obfusc_obj.get_is_encrypted()
        if mal_conf_obfusc_obj.get_Algorithm_Details():
            mal_conf_obfusc_.algorithm_details = [MalwareConfigurationObfuscationAlgorithm.from_obj(x) for x in mal_conf_obfusc_obj.get_Algorithm_Details()]
        return mal_conf_obfusc_

class MalwareConfigurationDetails(maec.Entity):
    _namespace = maec.package._namespace

    def __init__(self):
        super(MalwareConfigurationDetails, self).__init__()
        self.storage = None
        self.obfuscation = None
        self.configuration_parameter = []

    def to_obj(self):
        mal_conf_obj = package_binding.MalwareConfigurationDetailsType()
        if self.storage is not None : mal_dev_obj.set_Storage(self.storage.to_obj())
        if self.obfuscation is not None : mal_dev_obj.set_Obfuscation(self.obfuscation.to_obj())
        if self.configuration_parameter is not None: 
            mal_dev_obj.set_Configuration_Parameter([x.to_obj() for x in self.configuration_parameter])
        return mal_conf_obj

    def to_dict(self):
        mal_conf_dict = {}
        if self.storage is not None : mal_conf_dict['storage'] = self.storage.to_dict()
        if self.obfuscation is not None : mal_conf_dict['obfuscation'] = self.obfuscation.to_dict()
        if self.configuration_parameter is not None: 
            mal_conf_dict['configuration_parameter'] = [x.to_dict() for x in self.configuration_parameter]
        return mal_conf_dict

    @staticmethod
    def from_dict(mal_conf_dict):
        if not mal_conf_dict:
            return None
        mal_conf_ = MalwareConfigurationDetails()
        mal_conf_.storage = MalwareConfigurationStorageDetails.from_dict(mal_conf_dict['storage'])
        mal_conf_.obfuscation = MalwareConfigurationStorageDetails.from_dict(mal_conf_dict['obfuscation'])
        if mal_dev_dict['configuration_parameter']:
            mal_conf_.configuration_parameter = [MalwareConfigurationParameter.from_dict(x) for x in mal_dev_dict['configuration_parameter']]
        return mal_conf_

    @staticmethod
    def from_obj(mal_conf_obj):
        if not mal_conf_obj:
            return None
        mal_conf_ = MalwareConfigurationDetails()
        mal_conf_.storage = MalwareConfigurationStorageDetails.from_obj(mal_conf_obj.get_Storage())
        mal_conf_.obfuscation = MalwareConfigurationStorageDetails.from_obj(mal_conf_obj.get_Obfuscation())
        if mal_conf_obj.get_Configuration_Parameter():
            mal_conf_.configuration_parameter = [MalwareConfigurationParameter.from_obj(x) for x in mal_conf_obj.get_Configuration_Parameter()]
        return mal_conf_