#-------------------------------------------------------------------------------
# Name:        output
# Purpose:     Encapsulation of parsing/handling of data from computation
#                textual output.
#
# Author:      Brian Skinn
#                bskinn@alum.mit.edu
#
# Created:     16 Feb 2015
# Copyright:   (c) Brian Skinn 2016
# License:     The MIT License; see "license.txt" for full license terms
#                   and contributor agreement.
#
#       This file is part of opan (Open Anharmonic), a system for automated
#       computation of anharmonic properties of molecular systems via wrapper
#       calls to computational/quantum chemical software packages.
#
#       http://www.github.com/bskinn/opan
#
#-------------------------------------------------------------------------------


""" Module implementing parsing of output data from external computations.

.. warning::

    This module **will** be refactored at some point, to introduce a
    superclass in the same vein as :class:`~opan.grad.SuperOpanGrad` and
    :class:`~opan.hess.SuperOpanHess`.  This is necessary because
    automated execution of external
    computation softwares will require some unified mechanism for
    indicating whether a particular computation completed successfully,
    independent of the
    identify of the software package that was executed.

.. warning::

    Further refactoring is also planned for :class:`OrcaOutput` generally.
    Beware relying heavily on the behavior of this class & module.

**Superclass**

*To be implemented*

|

**Implemented Subclasses**

.. note::

    Not yet actually a subclass of anything

:class:`OrcaOutput` -- Imports output files from |orca|

|

**Subclasses**

.. autoclass:: OrcaOutput


"""


# Debug constant
_DEBUG = False


class OrcaOutput(object):
    """ Container for parsed textual output generated by |orca|.

    All implemented results that are found in the indicated output are stored
    in the :class:`OrcaOutput` instance.  If a given quantity was
    not detectable, it
    is stored as |None| in the corresponding instance variable.

    .. note::

        In particular, thermochemistry from single atom/ion computations
        **should work**, with |None| or zero/negligible values returned
        for rotational and vibrational quantities.

    The verbose contents of the output file are not generally retained within
    the :class:`OrcaOutput` instance due to the potential for such to involve a
    tremendously large string. Exceptions include, if present:

        - THERMOCHEMISTRY section

    |

    **Contents**

    *   `Methods <OrcaOutput-Methods_>`_

        *   :meth:`~opan.output.OrcaOutput.__init__`

        *   :meth:`~opan.output.OrcaOutput.en_last`

    *   `Class Variables <OrcaOutput-ClassVars_>`_

        *   `Enumerations <OrcaOutput-ClassVars-Enums_>`_

            *   :class:`~opan.output.OrcaOutput.EN`

            *   :class:`~opan.output.OrcaOutput.SPINCONT`

            *   :class:`~opan.output.OrcaOutput.THERMO`

        *   `Regular Expression Patterns
            <OrcaOutput-ClassVars-Regex_>`_

            *   :class:`~opan.output.OrcaOutput.p_en`

            *   :class:`~opan.output.OrcaOutput.p_spincont`

            *   :class:`~opan.output.OrcaOutput.p_thermo`

    *   `Instance Variables <OrcaOutput-InstVars_>`_

    .. _OrcaOutput-Methods:

    **Methods**

    .. automethod:: __init__

    .. automethod:: en_last

    |

    .. _OrcaOutput-ClassVars:

    **Class Variables**

    .. _OrcaOutput-ClassVars-Enums:

    *Enumerations*

    .. class:: EN

        :class:`~opan.const.OpanEnum` for the energies reported
        at the end of SCF cycles.

        |

        .. attribute:: D3

            Grimme's D3BJ dispersion correction [Gri10]_. May or may not play
            nicely with D3ZERO. Likely non-functional with DFT-NL
            dispersion.

        .. attribute:: GCP

            Grimme's geometric counterpose (gCP) correction [Kru12]_

        .. attribute:: OCC

            COSMO outlying charge correction

            .. todo:: Need COSMO reference

        .. attribute:: SCFFINAL

            SCF energy including gCP and D3 corrections

        .. attribute:: SCFFINALOCC

            :attr:`SCFFINAL` energy, but also with COSMO outlying
            charge correction

        .. attribute:: SCFOCC

            SCF energy with only the COSMO outlying charge correction
            (no dispersion or gCP corrections)


    .. class:: OrcaOutput.SPINCONT

        :class:`~opan.const.OpanEnum` for the spin contamination
        values reported after each unrestricted SCF cycle.

        |

        .. attribute:: ACTUAL

            Calculated :math:`\\left<S^2\\right>` expectation value

        .. attribute:: DEV

            Deviation of :math:`\\left<S^2\\right>` (calculated
            minus ideal)

        .. attribute:: IDEAL

            Ideal :math:`\\left<S^2\\right>` expectation value


    .. class:: OrcaOutput.THERMO

        :class:`~opan.const.OpanEnum` for the quantities reported
        in the THERMOCHEMISTRY block.

        |

        .. attribute:: BLOCK

            Entire THERMOCHEMISTRY block (as |str|)

        .. attribute:: E_EL

            Electronic energy from the thermochemistry block, often
            slightly different than the last
            :attr:`EN.SCFFINAL <OrcaOutput.EN.SCFFINAL>` value
            :math:`\\left(\\mathrm{E_h}\\right)`

        .. attribute:: E_ROT

            Thermal rotational internal energy correction
            :math:`\\left(\\mathrm{E_h}\\right)`

        .. attribute:: E_TRANS

            Thermal translational internal energy correction
            :math:`\\left(\\mathrm{E_h}\\right)`

        .. attribute:: E_VIB

            Thermal vibrational internal energy correction
            :math:`\\left(\\mathrm{E_h}\\right)`

        .. attribute:: E_ZPE

            Zero-point energy correction
            :math:`\\left(\\mathrm{E_h}\\right)`

        .. attribute:: H_IG

            Ideal-gas
            :math:`\\left(k_\\mathrm{B} T\\right)`
            enthalpy contribution
            :math:`\\left(\\mathrm{E_h}\\right)`

        .. attribute:: PRESS

            Simulated pressure
            :math:`\\left(\\mathrm{atm}\\right)`

        .. attribute:: QROT

            Rotational partition function (unitless)

        .. attribute:: TEMP

            Simulated temperature
            :math:`\\left(\\mathrm K\\right)`

        .. attribute:: TS_EL

            Electronic :math:`TS` entropy contribution
            :math:`\\left(\\mathrm{E_h}\\right)`

        .. attribute:: TS_TRANS

            Translational :math:`TS` entropy contribution
            :math:`\\left(\\mathrm{E_h}\\right)`

        .. attribute:: TS_VIB

            Vibrational :math:`TS` entropy contribution
            :math:`\\left(\\mathrm{E_h}\\right)`

    .. _OrcaOutput-ClassVars-Regex:

    *Regular Expression Patterns*

    .. attribute:: OrcaOutput.p_en

        |dict| of |re.compile| for the energies reported
        at the end of SCF cycles. Keys are in :attr:`~OrcaOutput.EN`.

    .. attribute:: OrcaOutput.p_spincont

        |dict| of |re.compile| for the spin contamination block
        values. Keys are in :attr:`~OrcaOutput.SPINCONT`.

    .. attribute:: OrcaOutput.p_thermo

        |dict| of |re.compile| for the quantities extracted
        from the THERMOCHEMISTRY block. Keys are in
        :attr:`~OrcaOutput.THERMO`.

    |

    .. _OrcaOutput-InstVars:

    **Instance Variables**

    .. attribute:: OrcaOutput.completed

        |bool| --
        |True| if |orca| output reports normal termination, |False| otherwise.

    .. attribute:: OrcaOutput.converged

        |bool| --
        |True| if SCF converged ANYWHERE in run.

        .. todo:: Update oo.converged with any robustifications

    .. attribute:: OrcaOutput.en

        |dict| of |list| of |npfloat_|--
        Lists of the various energy values from the parsed output. Dict
        keys are those of :attr:`EN`, above.  Any energy type not found in the
        output is assigned as an empty list.

    .. attribute:: OrcaOutput.optimized

        |bool| --
        |True| if any OPT converged ANYWHERE in run. Fine for OPT,
        but ambiguous for scans.

        .. todo:: Update oo.optimized with any robustifications

    .. attribute:: OrcaOutput.spincont

        |dict| of |list| of |npfloat_|--
        Lists of the various values from the spin contamination calculations
        in the output, if present. Empty lists if absent. Dict keys are those
        of :attr:`SPINCONT`, above.

    .. attribute:: OrcaOutput.src_path

        |str| --
        Full path to the associated output file

    .. attribute:: OrcaOutput.thermo

        |dict| of |npfloat_|--
        Values from the thermochemistry block of the parsed output. Dict keys
        are those of :attr:`THERMO`, above.

    .. attribute:: OrcaOutput.thermo_block

        |str| --
        Full text of the thermochemistry block, if found.

    """

    # Imports
    import re as _re
    from .const import OpanEnum as _OpEnum


    # Various class-level RegEx patterns, collected into dictionaries to
    #  facilitate later iterable data retrieval.
    #
    # IN ALL PATTERNS the group name is the same -- this is to simplify the
    #  parsing process when these patterns are used -- no need to dynamically
    #  fiddle with substituting in custom group names each time! The .replace()
    #  call in each pattern definition saves work if P_GROUP ever needs to be
    #  changed.
    P_GROUP = "val"

    # Patterns for SCF energies, reported at the end of single-point and each
    #  step of geometry optimizations, etc., if not suppressed by %output
    #  settings.

    # String constants for retrieving energy quantities.
    # Prefix is the uppercase of the Regex dictionary name
    class EN(_OpEnum):
        SCFFINAL = "SCFFINAL"
        GCP = "GCP"
        D3  = "D3"
        SCFOCC = "SCFOCC"
        OCC = "OCC"
        SCFFINALOCC = "SCFFINALOCC"
    ## end class EN

    # Initialize dictionary
    p_en = dict()

    # Final SCF energy, with gCP, D3, corrections included... but NOT COSMO
    #  outlying charge correction.
    p_en.update({ EN.SCFFINAL :
        _re.compile("""
        -\\n                         # Hyphen on preceding line
        FINAL\\ SINGLE\\             # Key text 1
        POINT\\ ENERGY\\             # Key text 2
        [\\ ]+(?P<>[0-9.-]+)         # Energy on same line as key text
        .*\\n-                       # Hyphen starting following line
        """.replace("P<", "P<" + P_GROUP), _re.I | _re.X)
        })

    # gCP corrections entering into reported FINAL ENERGY values.
    p_en.update({ EN.GCP :
        _re.compile("""
        -\\n                        # Hyphen on preceding line
        gCP\\ correction            # Key text
        [\\ ]+(?P<>[0-9.-]+)        # Energy on same line as key text
        .*\\n-                      # Hyphen starting following line.
        """.replace("P<", "P<" + P_GROUP), _re.I | _re.X)
        })

    # D3 corrections entering into reported FINAL ENERGY values.
    p_en.update({ EN.D3 :
        _re.compile("""
        -\\n                        # Hyphen on preceding line
        Dispersion\\ correction     # Key text
        [\\ ]+(?P<>[0-9.-]+)        # Energy on same line as key text
        .*\\n-                      # Hyphen starting following line.
        """.replace("P<", "P<" + P_GROUP), _re.I | _re.X)
        })

    # COSMO SCF energies after the COSMO outlying charge correction BUT BEFORE
    #  any other augmentations to the energy (no D3, gCP, etc.)
    p_en.update({ EN.SCFOCC :
        _re.compile("""
        Total\\ Energy\\ after\\    # Key text 1
        outlying\\ charge\\         # Key text 2
        correction[\\ ]*=           # Key text 3
        [\\ ]+(?P<>[0-9.-]+)        # Energy following key text
        [\\ ]+Eh.*\\n               # 'Eh' units, then newline
        """.replace("P<", "P<" + P_GROUP), _re.I | _re.X)
        })

    # Patterns for the entire thermochemistry block, as well as the individual
    #  data elements therein.

    # String constants for retrieving energy quantities.
    # Prefix is the uppercase of the Regex dictionary name
    class THERMO(_OpEnum):
        BLOCK = "BLOCK"
        TEMP = "TEMP"
        PRESS = "PRESS"
        E_EL = "E_EL"
        E_ZPE = "E_ZPE"
        E_VIB = "E_VIB"
        E_ROT = "E_ROT"
        E_TRANS = "E_TRANS"
        H_IG = "H_IG"
        TS_EL = "TS_EL"
        TS_VIB = "TS_VIB"
        TS_TRANS = "TS_TRANS"
        QROT = "QROT"
    ## end class THERMO

    # Initialize dictionary
    p_thermo = dict()

    # Whole thermo block just in case; probably not needed for automated
    #  computation, but potentially handy for manual fiddling.
    p_thermo.update({ THERMO.BLOCK :
        _re.compile("""
        (?P<>-+\\n              # Hyphen line
        THERMOCHEMISTRY\\       # Header text
        AT\\ [0-9.]+\\ *K\\n    # Temperature
        -+\\n                   # Hyphen line
        (.|\\n)*)               # Everything until the end
        Timings\\ for\\         # Closing blip 1
        individual\\ modules    # Closing blip 2
        """.replace("P<", "P<" + P_GROUP), _re.I | _re.X)
        })

    # Individual quantities. Descriptions in pattern definition comments.
    p_thermo.update({ THERMO.TEMP :
        _re.compile("""
        temperature[\\ .]+      # Key text
        (?P<>[0-9.]+)           # Temperature value
        [\\ ]+K                 # in Kelvin
        """.replace("P<", "P<" + P_GROUP), _re.I | _re.X)
        })
    p_thermo.update({ THERMO.PRESS :
        _re.compile("""
        pressure[\\ .]+         # Key text
        (?P<>[0-9.]+)           # Pressure value
        [\\ ]+atm               # in atm
        """.replace("P<", "P<" + P_GROUP), _re.I | _re.X)
        })
    p_thermo.update({ THERMO.E_EL :
        _re.compile("""
        electronic\\ energy     # Key text 1
        [\\ .]+                 # Key text 2
        (?P<>[0-9.-]+)          # Electronic energy value
        [\\ ]+Eh                # in Hartrees
        """.replace("P<", "P<" + P_GROUP), _re.I | _re.X)
        })
    p_thermo.update({ THERMO.E_ZPE :
        _re.compile("""
        zero\\ point\\ energy   # Key text 1
        [\\ .]+                 # Key text 2
        (?P<>[0-9.-]+)          # ZPE energy value
        [\\ ]+Eh                # in Hartrees
        """.replace("P<", "P<" + P_GROUP), _re.I | _re.X)
        })
    p_thermo.update({ THERMO.E_VIB :
        _re.compile("""
        thermal\\ vibrational\\ # Key text 1
        correction[\\ .]+       # Key text 2
        (?P<>[0-9.-]+)          # Vibration energy correction value
        [\\ ]+Eh                # in Hartrees
        """.replace("P<", "P<" + P_GROUP), _re.I | _re.X)
        })
    p_thermo.update({ THERMO.E_ROT :
        _re.compile("""
        thermal\\ rotational\\  # Key text 1
        correction[\\ .]+       # Key text 2
        (?P<>[0-9.-]+)          # Rotation energy correction value
        [\\ ]+Eh                # in Hartrees
        """.replace("P<", "P<" + P_GROUP), _re.I | _re.X)
        })
    p_thermo.update({ THERMO.E_TRANS :
        _re.compile("""
        thermal\\               # Key text 1
        translational\\         # Key text 2
        correction[\\ .]+       # Key text 3
        (?P<>[0-9.-]+)          # Translation energy correction value
        [\\ ]+Eh                # in Hartrees
        """.replace("P<", "P<" + P_GROUP), _re.I | _re.X)
        })
    p_thermo.update({ THERMO.H_IG :
        _re.compile("""
        thermal\\ enthalpy\\    # Key text 1
        correction[\\ .]+       # Key text 2
        (?P<>[0-9.-]+)          # Ideal gas enthalpy correction
        [\\ ]+Eh                # in Hartrees
        """.replace("P<", "P<" + P_GROUP), _re.I | _re.X)
        })
    p_thermo.update({ THERMO.TS_EL :
        _re.compile("""
        electronic\\ entropy\\  # Key text
        [\\ .]+                 # Spacer
        (?P<>[0-9.-]+)          # Electronic entropy contribution
        [\\ ]+Eh                # in Hartrees
        """.replace("P<", "P<" + P_GROUP), _re.I | _re.X)
        })
    p_thermo.update({ THERMO.TS_VIB :
        _re.compile("""
        vibrational\\ entropy\\ # Key text
        [\\ .]+                 # Spacer
        (?P<>[0-9.-]+)          # Vibrational entropy contribution
        [\\ ]+Eh                # in Hartrees
        """.replace("P<", "P<" + P_GROUP), _re.I | _re.X)
        })
    p_thermo.update({ THERMO.TS_TRANS :
        _re.compile("""
        translational\\         # Key text 1
        entropy\\               # Key text 2
        [\\ .]+                 # Spacer
        (?P<>[0-9.-]+)          # Translational entropy contribution
        [\\ ]+Eh                # in Hartrees
        """.replace("P<", "P<" + P_GROUP), _re.I | _re.X)
        })
    p_thermo.update({ THERMO.QROT :
        _re.compile("""
        qrot\\ +=\\ +           # Key text
        (?P<>[0-9.]+)           # Rotational partition function
        """.replace("P<", "P<" + P_GROUP), _re.I | _re.X)
        })


    # Dipole moment pattern
    p_dipmom = _re.compile("""
        -+\\n                             # Hyphen line
        dipole\\ moment.*\\n              # Block label
        -+\\n                             # Hyphen line
        (.*\\n)+?                         # Lazy grab of any lines
        Magnitude\ \(debye\)\ +:\ +       # Line leader
        (?P<>[0-9.]+).*\\n                # Grab the dipole moment
        """.replace("P<", "P<" + P_GROUP), _re.I | _re.X)


    # Patterns for the spin contamination information
    # String constants for retrieving energy quantities.
    # Prefix is the uppercase of the Regex dictionary name
    class SPINCONT(_OpEnum):
        ACTUAL = "ACTUAL"
        IDEAL = "IDEAL"
        DEV = "DEV"
    ## end class SPINCONT

    # Initialize dictionary
    p_spincont = dict()

    # Patterns for the spin contamination information
    p_spincont.update({ SPINCONT.ACTUAL :
        _re.compile("""
        expectation[ ]value[ ]of[ ]<S\*\*2>         # Key text
        [ ]+:[ ]+                                   # Space and separator
        (?P<>[0-9.]+)                               # Grab the value
        """.replace("P<", "P<" + P_GROUP), _re.I | _re.X)
        })
    p_spincont.update({ SPINCONT.IDEAL :
        _re.compile("""
        ideal[ ]value[ ]s\\*\\(s\\+1\\)[ ]          # Key text 1
        for[ ]s=[0-9.]+                             # Key text 2
        [ ]+:[ ]+                                   # Space and separator
        (?P<>[0-9.]+)                               # Grab the value
        """.replace("P<", "P<" + P_GROUP), _re.I | _re.X)
        })
    p_spincont.update({ SPINCONT.DEV :
        _re.compile("""
        deviation                                   # Key text
        [ ]+:[ ]+                                   # Space and separator
        (?P<>[0-9.]+)                               # Grab the value
        """.replace("P<", "P<" + P_GROUP), _re.I | _re.X)
        })

    #RESUME: Make patterns and constants for the virial block; update docstrings

    ## end class variables

    def __init__(self, file_path):
        """ Initialize :class:`OrcaOutput` object.

        Imports the data found in the output file found at `file_path`.

        .. warning::

            THIS CLASS PRESENTLY ONLY WORKS ON A **VERY SMALL** SUBSET OF
            COMPUTATION TYPES, currently HF, LDA-DFT, GGA-DFT, and mGGA-DFT.
            *MAY* work on some double-hybrid or range-separated DFT.

        Available data includes:

         *  SCF energies (incl D3BJ, gCP, COSMO outlying charge corrections)

         *  Thermochemistry

         *  Spin expectation values (actual, ideal, and deviation)

        Success indicators include:

        *   `completed`

                Checks for the 'ORCA TERMINATED NORMALLY' report at the
                end of the file

        *   `converged`

                Checks for any occurrence of successful SCF convergence
                in the file (questionable for anything but single-point
                calculations)

        *   `optimized`

                Checks for any occurrence of "OPTIMIZATION HAS
                CONVERGED" in the file (questionable for anything but
                a standalone OPT -- i.e., not useful for a mode or
                internal coordinate scan)



        Parameters
        ----------
        file_path
            |str| --
            Full path to the output file to be parsed.

        Raises
        ------
        ~opan.error.OutputError
            (various typecodes) If indicated output is un-parseably
            malformed in some fashion

        """

        #TODO: (?) OrcaOutput: Add initialization parameter to indicate which
        # type of run should be expected?

        # Imports
        from .utils import pack_tups
        from .utils import safe_cast as scast
        from .error import OutputError
        import numpy as np

        # Get the output data
        with open(file_path) as in_f:
            datastr = in_f.read()
        ##end with

        # Check for normal termination (weird values in dicts, etc. would be
        #  diagnostic also, but might as well define this since it's easy).
        self.completed = datastr.find("ORCA TERMINATED NORMALLY") > -1

        # Simple check for single-point SCF convergence
        # TODO: Probably robustify convergence check to opt, and MDCI/MRCI/CAS
        self.converged = datastr.find("SCF CONVERGED AFTER") > -1

        # Simple check for optimization convergence
        # TODO: Probably robustify optimization convergence check for
        #  scans as well as single optimizations.  Multiple job runs promise
        #  to be thoroughly annoying.
        self.optimized = datastr.find("OPTIMIZATION HAS CONVERGED") > -1

        # Store the source information
        self.src_path = file_path

        # Initialize the energies dict as empty
        self.en = dict()

        # Populate with the Regex-retrieved values.
        #  If any are not found, this will store as an empty list.
        for (k,p) in self.p_en.items():
            self.en.update({ k :
                    [scast(m.group(self.P_GROUP), np.float_) for m in
                        p.finditer(datastr)] })
        ##next (k,p)

        # Calculate just the outlying charge correction, if COSMO enabled,
        #  and then calculate the SCFFINAL result including the OCC.
        if not self.en[self.EN.SCFOCC] == []:
            self.en.update({ self.EN.OCC :
                    [t[0] - (t[1] - t[2] - t[3]) for t in
                    pack_tups(
                        self.en[self.EN.SCFOCC],
                        self.en[self.EN.SCFFINAL],
                        self.en[self.EN.D3] if self.en[self.EN.D3] != []
                                                                else 0,
                        self.en[self.EN.GCP] if self.en[self.EN.GCP] != []
                                                                else 0
                              )
                    ]       })
            self.en.update({ self.EN.SCFFINALOCC :
                    [t[0] + t[1] for t in
                    pack_tups( # Could use zip() here, probably
                        self.en[self.EN.SCFFINAL],
                        self.en[self.EN.OCC]
                              )
                    ]       })
        ##end if

        # Now collect the thermo quantities
        # Just store the whole thermo block
        try:
            self.thermo_block = \
                    self.p_thermo[self.THERMO.BLOCK].search(datastr).group()
        except AttributeError:
            # Block not found; store as None
            self.thermo_block = None
        else:
            # Only store the block details if the block is actually found!

            # Initialize the empty dictionary for the numericals
            self.thermo = dict()

            # Iterate to pull the individual values
            for (k,p) in self.p_thermo.items():
                if k != self.THERMO.BLOCK:
                    try:
                        self.thermo.update({ k :
                                    scast(p.search(datastr)
                                            .group(self.P_GROUP), np.float_) })
                    except AttributeError:
                        # Value not found, probably due to monoatomic freq calc
                        #  to autogenerate, e.g., enthalpy calculation
                        # Add as just None'
                        self.thermo.update({ k: None })
                    ## end try
                ## end if
            ## next (k,p)
        ## end try

        #TODO: (?) OrcaOutput: Pull the final geometry and atom masses. Would
        #  be nice not to require a Hessian calculation in order to have this
        #  info available.
        #  Masses and/or geometries may not always be in the output file,
        #  depending on the %output settings.  Also have to address possible
        #  multiples of the coordinates in, e.g., scans.

        # Pull all dipole moments
        self.dipmoms = []
        for m in OrcaOutput.p_dipmom.finditer(datastr):
            self.dipmoms.append(scast(m.group(OrcaOutput.P_GROUP), np.float_))
        ## next m

        # Initialize the spin contamination dict as empty
        self.spincont = dict()

        # Populate with the Regex-retrieved values.
        #  If any are not found, this will store as an empty list.
        for (k,p) in self.p_spincont.items():
            self.spincont.update({ k :
                    [scast(m.group(self.P_GROUP), np.float_) for m in
                        p.finditer(datastr)] })
        ##next (k,p)

        #RESUME: Pull the virial block info (may be absent)

    ## end def __init__


    def en_last(self):
        """ Report the energies from the last SCF present in the output.

        Returns a |dict| providing the various energy values from the
        last SCF cycle performed in the output. Keys are those of
        :attr:`~opan.output.OrcaOutput.p_en`.
        Any energy value not relevant to the parsed
        output is assigned as |None|.

        Returns
        -------
        last_ens
            |dict| of |npfloat_|--
            Energies from the last SCF present in the output.

        """

        # Initialize the return dict
        last_ens = dict()

        # Iterate and store
        for (k,l) in self.en.items():
            last_ens.update({ k : l[-1] if l != [] else None })
        ##next (k,l)

        # Should be ready to return?
        return last_ens

    ## end def en_last


if __name__ == '__main__':
    print("Module not executable")
