"""
A set of functions required to post-process DAYSIM simulation output files
"""


import csv
import re

from pybps.util import dict_cleanconvert


def parse_da(file_abspathlist):
    """Parse results from DAYSIM DA results files.

    Parses all results from DA files and put them in a list of dicts, each dict
    corresponding to a particular sensor point.

    Args:
        file_abspathlist: list of absolute paths to DA result files.

    Returns:
        A list of dicts (one dict for each sensor point)

    Raises:
        IOError: problem reading out_file
    """

    da_results = []
    
    # Parse data from DA files generated by Daysim and return a list of
    # dicts (one dict per each sensor point in result file)
    for file_abspath in file_abspathlist:
        with open(file_abspath, 'rU') as out_f:
            line_1 = next(out_f) # Keep first line which identifies the data
            # Search for string that describes data contained in file
            match = re.search(r'# (.*) - Active User', line_1)
            da_data = match.group(1)
            # Extract illuminance level from found string
            match = re.search(r'(.*) \((\d{3}) lux\)', da_data)
            if match:
                da_data = match.group(1)
            next(out_f) # Skip second line which doesn't hold any useful info
            dr = csv.reader(out_f, delimiter='\t')
            dict_list = []
            # Build a list of dict (one row from file = one dict)
            for i, row in enumerate(dr):
                dict_list.append({})
                dict_list[i]['sens_x'] = row[0]
                dict_list[i]['sens_y'] = row[1]
                dict_list[i]['sens_z'] = row[2]
                if da_data == 'Daylight Autonomy':
                    dict_list[i]['DA'] = row[3]
                elif da_data == 'Continuous Daylight Autonomy':
                    dict_list[i]['DAcon'] = row[3]
                elif da_data == 'DA_max':
                    dict_list[i]['DAmax'] = row[3]
                elif da_data == 'Daylight Saturation Potential':
                    dict_list[i]['DSP'] = row[3]
                else:
                    dict_list[i][da_data] = row[3]    
            # Post-process generated list of dicts to remove unwanted 
            # whitespaced and convert strings with numbers to float type
            for dict in dict_list:
                dict = dict_cleanconvert(dict)
                
        # Group results in list of dicts
        if not da_results:
            da_results.extend(dict_list)
        else:
            for (res_dict, new_dict) in zip(da_results, dict_list):
                for k, v in new_dict.items():
                    res_dict[k] = new_dict[k]

    return da_results                    
    
    
def parse_el_lighting(file_abspath):
    """Parse results from DAYSIM electric lighting result file.

    Parses results from electric lighting result file and put them in a dict.

    Args:
        file_abspath: absolute path to electric lighting result file.

    Returns:
        A list of dicts containing the parsed results

    Raises:
        IOError: problem reading out_file
    """

    el_results = {}
    
    with open(file_abspath, 'rU') as out_f:
        temp = out_f.read()
        # Retrieve lighting power density value
        match = re.search(r'installed lighting power density of (\d+\.\d+)', temp)
        if match: 
            el_results['POWER_DENS'] = float(match.group(1))
        # Retrieve minimum illuminance level value
        match = re.search(r'minimum illuminance level of (\d+)', temp)
        if match: 
            el_results['MIN_ILL_LEV'] = float(match.group(1))
        # Retrieve ballast lost factor value
        match = re.search(r'ballast loss factor of (\d+)', temp)
        if match: 
            el_results['LOSS_FACTOR'] = float(match.group(1))
        # Retrieve standby power value
        match = re.search(r'standby power of (\d+\.\d+)', temp)
        if match: 
            el_results['STANDBY'] = float(match.group(1))
        # Retrieve Daylight Factor values
        match = re.search(r'\n(\d\.\d)</td>', temp)
        if match: 
            el_results['DF'] = float(match.group(1))
        # Retrieve Daylight Autonomy values
        match = re.search(r'The daylight autonomy for the core workplane sensor is (\d+)%', temp)
        if match: 
            el_results['DA'] = float(match.group(1))
        # Retrieve useful daylight autonomy for UDI < 100 lux
        match = re.search(r'UDI<sub><100</sub>=(\d+)%', temp)
        if match:
            el_results['UDI_<100'] = float(match.group(1))
        # Retrieve useful daylight autonomy for UDI between 100 and 2000 lux
        match = re.search(r'UDI<sub>100-2000</sub>=(\d+)%', temp)
        if match:
            el_results['UDI_100-2000'] = float(match.group(1))
        # Retrieve useful daylight autonomy for UDI > 2000 lux
        match = re.search(r'UDI<sub>>2000</sub>=(\d+)%', temp)
        if match:
            el_results['UDI_>2000'] = float(match.group(1))
        # Retrieve annual electric lighting energy use
        match = re.search(r'(\d{2}\.\d) kWh/unit area', temp)
        if match:
            el_results['elec_use'] = float(match.group(1))
        # Retrieve number of occupied hours
        match = re.search(r'occupancy at the work place are (\d+\.\d)', temp)
        if match:
            el_results['occ_hours'] = float(match.group(1))
        # Retrieve number of hours electrical lighting is activated        
        match = re.search(r'lighting is activated (\d+\.\d) hours', temp)
        if match:
            el_results['el_light_hours'] = float(match.group(1))

    return [el_results]