import typing
import warnings
from typing import Union

import fastobo

from ..metadata import Metadata, Subset
from ..definition import Definition
from ..ontology import Ontology
from ..term import Term
from ..pv import PropertyValue, LiteralPropertyValue, ResourcePropertyValue
from ..xref import Xref
from ..synonym import Synonym, SynonymData, SynonymType
from ..relationship import Relationship
from ..utils.warnings import NotImplementedWarning


class FastoboParser:

    ont: "Ontology"

    def _extract_definition(
        self, clause: Union[fastobo.term.DefClause, fastobo.typedef.DefClause]
    ) -> Definition:
        return Definition(clause.definition, map(self._extract_xref, clause.xrefs))

    def _extract_metadata(self, header: fastobo.header.HeaderFrame) -> Metadata:

        metadata = Metadata()

        def copy(src, dst=None, cb=None):
            cb = cb or (lambda x: x)
            dst = dst or src
            return lambda c: setattr(metadata, dst, cb(getattr(c, src)))

        def add(src, dst=None, cb=None):
            cb = cb or (lambda x: x)
            dst = dst or src
            return lambda c: getattr(metadata, dst).add(cb(getattr(c, src)))

        def todo():
            return lambda c: warnings.warn(
                f"cannot process `{c}`", NotImplementedWarning, stacklevel=3
            )

        _callbacks = {
            fastobo.header.AutoGeneratedByClause: copy("name", "auto_generated_by"),
            fastobo.header.DataVersionClause: copy("version", "data_version"),
            fastobo.header.DateClause: copy("date"),
            fastobo.header.DefaultNamespaceClause: copy(
                "namespace", "default_namespace", cb=str
            ),
            fastobo.header.FormatVersionClause: copy("version", "format_version"),
            fastobo.header.IdspaceClause: (
                lambda c: (metadata.idspace.__setitem__(
                    str(c.prefix),
                    (str(c.url), c.description)
                ))
            ),
            fastobo.header.ImportClause: add("reference", "imports"),
            fastobo.header.OntologyClause: copy("ontology"),
            fastobo.header.OwlAxiomsClause: lambda c: metadata.owl_axioms.append(
                c.axioms
            ),
            fastobo.header.PropertyValueClause: (
                lambda c: (
                    metadata.annotations.add(
                        self._extract_property_value(c.property_value)
                    )
                )
            ),
            fastobo.header.RemarkClause: add("remark", "remarks"),
            fastobo.header.SavedByClause: copy("name", "saved_by"),
            fastobo.header.SubsetdefClause: (
                lambda c: (
                    metadata.subsetdefs.add(Subset(str(c.subset), c.description))
                )
            ),
            fastobo.header.SynonymTypedefClause: (
                lambda c: (
                    metadata.synonymtypedefs.add(
                        SynonymType(str(c.typedef), c.description, c.scope)
                    )
                )
            ),
            fastobo.header.TreatXrefsAsEquivalentClause: todo(),
            fastobo.header.TreatXrefsAsGenusDifferentiaClause: todo(),
            fastobo.header.TreatXrefsAsHasSubclassClause: todo(),
            fastobo.header.TreatXrefsAsIsAClause: todo(),
            fastobo.header.TreatXrefsAsRelationshipClause: todo(),
            fastobo.header.TreatXrefsAsReverseGenusDifferentiaClause: todo(),
            fastobo.header.NamespaceIdRuleClause: copy("rule", "namespace_id_rule"),
            fastobo.header.UnreservedClause: (
                lambda c: (
                    metadata.unreserved.setdefault(c.raw_tag(), set()).add(
                        c.raw_value()
                    )
                )
            ),
        }

        for clause in header:
            try:
                _callbacks[type(clause)](clause)
            except KeyError:
                raise TypeError(f"unexpected type: {type(clause).__name__}")
        return metadata

    def _extract_property_value(
        self, pv: fastobo.pv.AbstractPropertyValue
    ) -> PropertyValue:
        if isinstance(pv, fastobo.pv.LiteralPropertyValue):
            return LiteralPropertyValue(str(pv.relation), pv.value, str(pv.datatype))
        elif isinstance(pv, fastobo.pv.ResourcePropertyValue):
            return ResourcePropertyValue(str(pv.relation), str(pv.value))
        else:
            msg = "'pv' must be AbstractPropertyValue, not {}"
            raise TypeError(msg.format(type(pv).__name__))

    def _extract_synonym_data(self, syn: fastobo.syn.Synonym) -> SynonymData:
        xrefs = map(self._extract_xref, syn.xrefs)
        type_ = str(syn.type) if syn.type is not None else None
        return SynonymData(syn.desc, syn.scope, type_, xrefs)

    def _extract_term(self, frame: fastobo.term.TermFrame) -> Term:

        id_ = str(frame.id)
        term = (self.ont.get_term if id_ in self.ont else self.ont.create_term)(id_)
        termdata = term._data()

        union_of = set()
        intersection_of = set()

        def copy(src, dst=None, cb=None):
            cb = cb or (lambda x: x)
            dst = dst or src
            return lambda c: setattr(termdata, dst, cb(getattr(c, src)))

        def add(src, dst=None, cb=None):
            cb = cb or (lambda x: x)
            dst = dst or src
            return lambda c: getattr(termdata, dst).add(cb(getattr(c, src)))

        def todo():
            return lambda c: warnings.warn(
                f"cannot process `{c}`", NotImplementedWarning, stacklevel=3
            )

        _callbacks = {
            fastobo.term.IsAnonymousClause: copy("anonymous"),
            fastobo.term.NameClause: copy("name"),
            fastobo.term.DefClause: lambda c: setattr(
                term, "definition", self._extract_definition(c)
            ),
            fastobo.term.AltIdClause: add("alt_id", "alternate_ids", cb=str),
            fastobo.term.IntersectionOfClause: lambda c: (
                intersection_of.add((str(c.typedef), str(c.term)))
                if c.typedef is not None
                else intersection_of.add(str(c.term))
            ),
            fastobo.term.UnionOfClause: lambda c: union_of.add(str(c.term)),
            fastobo.term.RelationshipClause: lambda c: termdata.relationships.setdefault(
                str(c.typedef), set()
            ).add(
                str(c.term)
            ),
            fastobo.term.BuiltinClause: copy("builtin"),
            fastobo.term.CommentClause: copy("comment"),
            fastobo.term.ConsiderClause: add("term", "consider", cb=str),
            fastobo.term.CreatedByClause: copy("creator", "created_by"),
            fastobo.term.CreationDateClause: copy("date", "creation_date"),
            fastobo.term.EquivalentToClause: add("term", "equivalent_to", cb=str),
            fastobo.term.IsAClause: lambda c: termdata.relationships.setdefault(
                "is_a", set()
            ).add(str(c.term)),
            fastobo.term.IsObsoleteClause: copy("obsolete"),
            fastobo.term.NamespaceClause: copy("namespace", cb=str),
            fastobo.term.SubsetClause: add("subset", "subsets", cb=str),
            fastobo.term.SynonymClause: lambda c: termdata.synonyms.add(
                self._extract_synonym_data(c.synonym)
            ),
            fastobo.term.DisjointFromClause: add("term", "disjoint_from", cb=str),
            fastobo.term.PropertyValueClause: (
                lambda c: (
                    termdata.annotations.add(
                        self._extract_property_value(c.property_value)
                    )
                )
            ),
            fastobo.term.ReplacedByClause: add("term", "replaced_by", cb=str),
            fastobo.term.XrefClause: add("xref", "xrefs", cb=self._extract_xref),
        }

        for clause in frame:
            try:
                _callbacks[type(clause)](clause)
            except KeyError:
                raise TypeError(f"unexpected type: {type(clause).__name__}")
        if len(union_of) == 1:
            raise ValueError("'union_of' cannot have a cardinality of 1")
        termdata.union_of = union_of
        if len(intersection_of) == 1:
            raise ValueError("'intersection_of' cannot have a cardinality of 1")
        termdata.intersection_of = intersection_of
        return term

    def _extract_relationship(
        self, frame: fastobo.typedef.TypedefFrame
    ) -> Relationship:

        id_ = str(frame.id)
        rship = (self.ont.get_relationship if id_ in self.ont else self.ont.create_relationship)(id_)
        rshipdata = rship._data()

        union_of = set()
        intersection_of = set()

        def copy(src, dst=None, cb=None):
            cb = cb or (lambda x: x)
            dst = dst or src
            return lambda c: setattr(rshipdata, dst, cb(getattr(c, src)))

        def add(src, dst=None, cb=None):
            cb = cb or (lambda x: x)
            dst = dst or src
            return lambda c: getattr(rshipdata, dst).add(cb(getattr(c, src)))

        def todo():
            return lambda c: warnings.warn(
                f"cannot process `{c}`",
                NotImplementedWarning,
                stacklevel=3
            )

        _callbacks = {
            fastobo.typedef.AltIdClause: add("alt_id", "alternate_ids", cb=str),
            fastobo.typedef.BuiltinClause: copy("builtin"),
            fastobo.typedef.CommentClause: copy("comment"),
            fastobo.typedef.ConsiderClause: add("typedef", "consider", cb=str),
            fastobo.typedef.CreatedByClause: copy("creator", "created_by"),
            fastobo.typedef.CreationDateClause: copy("date", "creation_date"),
            fastobo.typedef.DefClause: lambda c: setattr(
                rship, "definition", self._extract_definition(c)
            ),
            fastobo.typedef.DisjointFromClause: add("typedef", "disjoint_from", cb=str),
            fastobo.typedef.DisjointOverClause: add("typedef", "disjoint_over", cb=str),
            fastobo.typedef.DomainClause: lambda c: setattr(
                rshipdata, "domain", str(c.domain)
            ),
            fastobo.typedef.EquivalentToChainClause: todo(),
            fastobo.typedef.EquivalentToClause: add("typedef", "equivalent_to", cb=str),
            fastobo.typedef.ExpandAssertionToClause: (
                lambda c: rshipdata.expand_assertion_to.add(self._extract_definition(c))
            ),
            fastobo.typedef.ExpandExpressionToClause: (
                lambda c: rshipdata.expand_expression_to.add(
                    self._extract_definition(c)
                )
            ),
            fastobo.typedef.HoldsOverChainClause: (
                lambda c: rshipdata.holds_over_chain.add((str(c.first), str(c.last)))
            ),
            fastobo.typedef.IntersectionOfClause: lambda c: intersection_of.add(
                str(c.typedef)
            ),
            fastobo.typedef.InverseOfClause: lambda c: setattr(
                rshipdata, "inverse_of", str(c.typedef)
            ),
            fastobo.typedef.IsAClause: lambda c: rshipdata.relationships.setdefault(
                "is_a", set()
            ).add(str(c.typedef)),
            fastobo.typedef.IsAnonymousClause: copy("anonymous"),
            fastobo.typedef.IsAntiSymmetricClause: copy("antisymmetric"),
            fastobo.typedef.IsAsymmetricClause: copy("asymmetric"),
            fastobo.typedef.IsClassLevelClause: copy("class_level"),
            fastobo.typedef.IsCyclicClause: copy("cyclic"),
            fastobo.typedef.IsFunctionalClause: copy("functional"),
            fastobo.typedef.IsInverseFunctionalClause: copy("inverse_functional"),
            fastobo.typedef.IsMetadataTagClause: copy("metadata_tag"),
            fastobo.typedef.IsObsoleteClause: copy("obsolete"),
            fastobo.typedef.IsReflexiveClause: copy("reflexive"),
            fastobo.typedef.IsSymmetricClause: copy("symmetric"),
            fastobo.typedef.IsTransitiveClause: copy("transitive"),
            fastobo.typedef.NameClause: copy("name"),
            fastobo.typedef.NamespaceClause: copy("namespace", cb=str),
            fastobo.typedef.PropertyValueClause: (
                lambda c: (
                    rshipdata.annotations.add(
                        self._extract_property_value(c.property_value)
                    )
                )
            ),
            fastobo.typedef.RangeClause: lambda c: setattr(
                rshipdata, "range", str(c.range)
            ),
            fastobo.typedef.RelationshipClause: todo(),
            fastobo.typedef.ReplacedByClause: add("typedef", "replaced_by", cb=str),
            fastobo.typedef.SubsetClause: add("subset", "subsets", cb=str),
            fastobo.typedef.SynonymClause: lambda c: rshipdata.synonyms.add(
                self._extract_synonym_data(c.synonym)
            ),
            fastobo.typedef.TransitiveOverClause: lambda c: rshipdata.transitive_over.add(
                str(c.typedef)
            ),
            fastobo.typedef.UnionOfClause: lambda c: union_of.add(str(c.typedef)),
            fastobo.typedef.XrefClause: add("xref", "xrefs", cb=self._extract_xref),
        }

        for clause in frame:
            try:
                _callbacks[type(clause)](clause)
            except KeyError:
                raise TypeError(f"unexpected type: {type(clause).__name__}")
        if len(union_of) == 1:
            raise ValueError("'union_of' cannot have a cardinality of 1")
        rshipdata.union_of = union_of
        if len(intersection_of) == 1:
            raise ValueError("'intersection_of' cannot have a cardinality of 1")
        rshipdata.intersection_of = intersection_of
        return rship

    def _extract_xref(self, xref: fastobo.xref.Xref) -> "Xref":
        return Xref(str(xref.id), xref.desc)
