# Copyright (c) 2011-2020, Manfred Moitzi
# License: MIT License
from typing import Tuple
import math

RGB = Tuple[int, int, int]


def int2rgb(value: int) -> RGB:
    """ Split RGB integer `value` into ``(r, g, b)`` tuple. """
    return (
        (value >> 16) & 0xFF,  # red
        (value >> 8) & 0xFF,  # green
        value & 0xFF,  # blue
    )


def rgb2int(rgb: RGB) -> int:
    """ Combined integer value from ``(r, g, b)`` tuple."""
    r, g, b = rgb
    return ((int(r) & 0xff) << 16) | ((int(g) & 0xff) << 8) | (int(b) & 0xff)


def aci2rgb(index: int) -> RGB:
    """ Convert :ref:`ACI` into ``(r, g, b)`` tuple, based on default AutoCAD colors. """
    if index < 1:
        raise IndexError(index)
    return int2rgb(DXF_DEFAULT_COLORS[index])


def luminance(color: RGB) -> float:
    """ Returns perceived luminance for a RGB color in the range [0.0, 1.0]  from dark to light. """
    r, g, b = color
    r = float(r) / 255.0
    g = float(g) / 255.0
    b = float(b) / 255.0
    return round(math.sqrt(0.299 * r ** 2 + 0.587 * g ** 2 + 0.114 * b ** 2), 3)


DXF_DEFAULT_COLORS = [
    0x000000, 0xff0000, 0xffff00, 0x00ff00, 0x00ffff, 0x0000ff, 0xff00ff,
    0xffffff, 0x414141, 0x808080, 0xff0000,
    0xffaaaa, 0xbd0000, 0xbd7e7e, 0x810000, 0x815656, 0x680000, 0x684545,
    0x4f0000, 0x4f3535, 0xff3f00, 0xffbfaa,
    0xbd2e00, 0xbd8d7e, 0x811f00, 0x816056, 0x681900, 0x684e45, 0x4f1300,
    0x4f3b35, 0xff7f00, 0xffd4aa, 0xbd5e00,
    0xbd9d7e, 0x814000, 0x816b56, 0x683400, 0x685645, 0x4f2700, 0x4f4235,
    0xffbf00, 0xffeaaa, 0xbd8d00, 0xbdad7e,
    0x816000, 0x817656, 0x684e00, 0x685f45, 0x4f3b00, 0x4f4935, 0xffff00,
    0xffffaa, 0xbdbd00, 0xbdbd7e, 0x818100,
    0x818156, 0x686800, 0x686845, 0x4f4f00, 0x4f4f35, 0xbfff00, 0xeaffaa,
    0x8dbd00, 0xadbd7e, 0x608100, 0x768156,
    0x4e6800, 0x5f6845, 0x3b4f00, 0x494f35, 0x7fff00, 0xd4ffaa, 0x5ebd00,
    0x9dbd7e, 0x408100, 0x6b8156, 0x346800,
    0x566845, 0x274f00, 0x424f35, 0x3fff00, 0xbfffaa, 0x2ebd00, 0x8dbd7e,
    0x1f8100, 0x608156, 0x196800, 0x4e6845,
    0x134f00, 0x3b4f35, 0x00ff00, 0xaaffaa, 0x00bd00, 0x7ebd7e, 0x008100,
    0x568156, 0x006800, 0x456845, 0x004f00,
    0x354f35, 0x00ff3f, 0xaaffbf, 0x00bd2e, 0x7ebd8d, 0x00811f, 0x568160,
    0x006819, 0x45684e, 0x004f13, 0x354f3b,
    0x00ff7f, 0xaaffd4, 0x00bd5e, 0x7ebd9d, 0x008140, 0x56816b, 0x006834,
    0x456856, 0x004f27, 0x354f42, 0x00ffbf,
    0xaaffea, 0x00bd8d, 0x7ebdad, 0x008160, 0x568176, 0x00684e, 0x45685f,
    0x004f3b, 0x354f49, 0x00ffff, 0xaaffff,
    0x00bdbd, 0x7ebdbd, 0x008181, 0x568181, 0x006868, 0x456868, 0x004f4f,
    0x354f4f, 0x00bfff, 0xaaeaff, 0x008dbd,
    0x7eadbd, 0x006081, 0x567681, 0x004e68, 0x455f68, 0x003b4f, 0x35494f,
    0x007fff, 0xaad4ff, 0x005ebd, 0x7e9dbd,
    0x004081, 0x566b81, 0x003468, 0x455668, 0x00274f, 0x35424f, 0x003fff,
    0xaabfff, 0x002ebd, 0x7e8dbd, 0x001f81,
    0x566081, 0x001968, 0x454e68, 0x00134f, 0x353b4f, 0x0000ff, 0xaaaaff,
    0x0000bd, 0x7e7ebd, 0x000081, 0x565681,
    0x000068, 0x454568, 0x00004f, 0x35354f, 0x3f00ff, 0xbfaaff, 0x2e00bd,
    0x8d7ebd, 0x1f0081, 0x605681, 0x190068,
    0x4e4568, 0x13004f, 0x3b354f, 0x7f00ff, 0xd4aaff, 0x5e00bd, 0x9d7ebd,
    0x400081, 0x6b5681, 0x340068, 0x564568,
    0x27004f, 0x42354f, 0xbf00ff, 0xeaaaff, 0x8d00bd, 0xad7ebd, 0x600081,
    0x765681, 0x4e0068, 0x5f4568, 0x3b004f,
    0x49354f, 0xff00ff, 0xffaaff, 0xbd00bd, 0xbd7ebd, 0x810081, 0x815681,
    0x680068, 0x684568, 0x4f004f, 0x4f354f,
    0xff00bf, 0xffaaea, 0xbd008d, 0xbd7ead, 0x810060, 0x815676, 0x68004e,
    0x68455f, 0x4f003b, 0x4f3549, 0xff007f,
    0xffaad4, 0xbd005e, 0xbd7e9d, 0x810040, 0x81566b, 0x680034, 0x684556,
    0x4f0027, 0x4f3542, 0xff003f, 0xffaabf,
    0xbd002e, 0xbd7e8d, 0x81001f, 0x815660, 0x680019, 0x68454e, 0x4f0013,
    0x4f353b, 0x333333, 0x505050, 0x696969,
    0x828282, 0xbebebe, 0xffffff,
]

# color codes are 1-indexed so an additional entry was put in the 0th position
# different plot styles may choose different colors for the same code
# from ftp://ftp.ecn.purdue.edu/jshan/86/help/html/import_export/dxf_colortable.htm
# alternative color tables can be found at:
#  - http://www.temblast.com/songview/color3.htm
#  - http://gohtx.com/acadcolors.php

DXF_DEFAULT_COLORS_2 = [
    0x000000, 0xff0000, 0xffff00, 0x00ff00, 0x00ffff, 0x0000ff, 0xff00ff,
    0xffffff, 0x808080, 0xc0c0c0, 0xff0000,
    0xff7f7f, 0xa50000, 0xa55252, 0x7f0000, 0x7f3f3f, 0x4c0000, 0x4c2626,
    0x260000, 0x261313, 0xff3f00, 0xff9f7f,
    0xa52900, 0xa56752, 0x7f1f00, 0x7f4f3f, 0x4c1300, 0x4c2f26, 0x260900,
    0x261713, 0xff7f00, 0xffbf7f, 0xa55200,
    0xa57c52, 0x7f3f00, 0x7f5f3f, 0x4c2600, 0x4c3926, 0x261300, 0x261c13,
    0xffbf00, 0xffdf7f, 0xa57c00, 0xa59152,
    0x7f5f00, 0x7f6f3f, 0x4c3900, 0x4c4226, 0x261c00, 0x262113, 0xffff00,
    0xffff7f, 0xa5a500, 0xa5a552, 0x7f7f00,
    0x7f7f3f, 0x4c4c00, 0x4c4c26, 0x262600, 0x262613, 0xbfff00, 0xdfff7f,
    0x7ca500, 0x91a552, 0x5f7f00, 0x6f7f3f,
    0x394c00, 0x424c26, 0x1c2600, 0x212613, 0x7fff00, 0xbfff7f, 0x52a500,
    0x7ca552, 0x3f7f00, 0x5f7f3f, 0x264c00,
    0x394c26, 0x132600, 0x1c2613, 0x3fff00, 0x9fff7f, 0x29a500, 0x67a552,
    0x1f7f00, 0x4f7f3f, 0x134c00, 0x2f4c26,
    0x092600, 0x172613, 0x00ff00, 0x7fff7f, 0x00a500, 0x52a552, 0x007f00,
    0x3f7f3f, 0x004c00, 0x264c26, 0x002600,
    0x132613, 0x00ff3f, 0x7fff9f, 0x00a529, 0x52a567, 0x007f1f, 0x3f7f4f,
    0x004c13, 0x264c2f, 0x002609, 0x132617,
    0x00ff7f, 0x7fffbf, 0x00a552, 0x52a57c, 0x007f3f, 0x3f7f5f, 0x004c26,
    0x264c39, 0x002613, 0x13261c, 0x00ffbf,
    0x7fffdf, 0x00a57c, 0x52a591, 0x007f5f, 0x3f7f6f, 0x004c39, 0x264c42,
    0x00261c, 0x132621, 0x00ffff, 0x7fffff,
    0x00a5a5, 0x52a5a5, 0x007f7f, 0x3f7f7f, 0x004c4c, 0x264c4c, 0x002626,
    0x132626, 0x00bfff, 0x7fdfff, 0x007ca5,
    0x5291a5, 0x005f7f, 0x3f6f7f, 0x00394c, 0x26424c, 0x001c26, 0x132126,
    0x007fff, 0x7fbfff, 0x0052a5, 0x527ca5,
    0x003f7f, 0x3f5f7f, 0x00264c, 0x26394c, 0x001326, 0x131c26, 0x003fff,
    0x7f9fff, 0x0029a5, 0x5267a5, 0x001f7f,
    0x3f4f7f, 0x00134c, 0x262f4c, 0x000926, 0x131726, 0x0000ff, 0x7f7fff,
    0x0000a5, 0x5252a5, 0x00007f, 0x3f3f7f,
    0x00004c, 0x26264c, 0x000026, 0x131326, 0x3f00ff, 0x9f7fff, 0x2900a5,
    0x6752a5, 0x1f007f, 0x4f3f7f, 0x13004c,
    0x2f264c, 0x090026, 0x171326, 0x7f00ff, 0xbf7fff, 0x5200a5, 0x7c52a5,
    0x3f007f, 0x5f3f7f, 0x26004c, 0x39264c,
    0x130026, 0x1c1326, 0xbf00ff, 0xdf7fff, 0x7c00a5, 0x9152a5, 0x5f007f,
    0x6f3f7f, 0x39004c, 0x42264c, 0x1c0026,
    0x211326, 0xff00ff, 0xff7fff, 0xa500a5, 0xa552a5, 0x7f007f, 0x7f3f7f,
    0x4c004c, 0x4c264c, 0x260026, 0x261326,
    0xff00bf, 0xff7fdf, 0xa5007c, 0xa55291, 0x7f005f, 0x7f3f6f, 0x4c0039,
    0x4c2642, 0x26001c, 0x261321, 0xff007f,
    0xff7fbf, 0xa50052, 0xa5527c, 0x7f003f, 0x7f3f5f, 0x4c0026, 0x4c2639,
    0x260013, 0x26131c, 0xff003f, 0xff7f9f,
    0xa50029, 0xa55267, 0x7f001f, 0x7f3f4f, 0x4c0013, 0x4c262f, 0x260009,
    0x261317, 0x545454, 0x767676, 0xa0a0a0,
    0xc0c0c0, 0xe0e0e0, 0x000000,
]
