##############################################################################
#
# Copyright (c) 2007 Zope Corporation and Contributors.
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#
##############################################################################
import os
import re
import shutil
import sys
import tempfile
import urllib2
import urlparse
import setuptools.archive_util

EGG_INFO_CONTENT = """Metadata-Version: 1.0
Name: %s
Version: %s
"""


class FakeLibInfo(object):
    """
    a really simple to store informations about libraries to be faked
    as eggs.
    """
    version = ''
    name = ''

    def __init__(self, name, version='0.0'):
        self.version = version
        self.name = name


class Recipe:

    def __init__(self, buildout, name, options):
        self.buildout, self.name, self.options = buildout, name, options

        python = buildout['buildout']['python']
        options['executable'] = buildout[python]['executable']
        self.location = options.get('location', None)
        self.svn = options.get('svn', None)
        self.url = options.get('url', None)
        assert self.location or self.svn or self.url

        if self.location is not None:
            # We have an existing Zope installation; use it.
            assert os.path.exists(self.location), 'No such file or directory: %s' % (self.location,)
            options['location'] = self.location
            options['shared-zope'] = 'true'
        elif (self.svn is None and
            buildout['buildout'].get('zope-directory') is not None):
            # if we use a download, then we look for a directory with shared
            # Zope installations. TODO Sharing of SVN checkouts is not yet
            # supported
            _, _, urlpath, _, _, _ = urlparse.urlparse(self.url)
            fname = urlpath.split('/')[-1]
            # cleanup the name a bit
            for s in ('.tar', '.bz2', '.gz', '.tgz'):
                fname = fname.replace(s, '')
            # Include the Python version Zope is compiled with into the
            # download cache name, so you can have the same Zope version
            # compiled with for example Python 2.3 and 2.4 but still share it
            ver = sys.version_info[:2]
            pystring = 'py%s.%s' % (ver[0], ver[1])
            options['location'] = os.path.join(
                buildout['buildout']['zope-directory'],
                '%s-%s' % (fname, pystring))
            options['shared-zope'] = 'true'
        else:
            # put it into parts
            options['location'] = os.path.join(
                buildout['buildout']['parts-directory'],
                self.name)
        # We look for a download cache, where we put the downloaded tarball
        buildout['buildout'].setdefault(
            'download-cache',
            os.path.join(buildout['buildout']['directory'], 'downloads'))

        skip_fake_eggs = self.options.get('skip-fake-eggs', '')
        self.skip_fake_eggs = [e for e in skip_fake_eggs.split('\n') if e]

        additional = self.options.get('additional-fake-eggs', '')
        self.additional_fake_eggs = [e for e in additional.split('\n') if e]

        self.fake_zope_eggs = bool(options.get('fake-zope-eggs', False))
        self.fake_eggs_folder = self.options.get('fake-eggs-folder',
                                                 'fake-eggs')
        # Automatically activate fake eggs
        if self.skip_fake_eggs or self.additional_fake_eggs:
            self.fake_zope_eggs = True

    def _compiled(self, path):
        """returns True if the path is compiled"""
        for files, dirs, root in os.walk(path):
            for f in files:
                base, ext = os.path.splitext(f)
                if ext == '.c':
                    if sys.platform == 'win32':
                        compiled_ext = '.pyd'
                    else:
                        compiled_ext = '.so'
                    
                    compiled = os.path.join(root, '%s%s' % (base, compiled_ext))
                    if not os.path.exists(compiled):
                        return False
        return True

    def install(self):
        options = self.options
        location = options['location']
        download_dir = self.buildout['buildout']['download-cache']
        smart_recompile = options.get('smart-recompile') == 'true' 

        if os.path.exists(location):
            # if the zope installation exists and is shared, then we are done
            # and don't return a path, so the shared installation doesn't get
            # deleted on uninstall
            if options.get('shared-zope') == 'true':
                # We update the fake eggs in case we have special skips or
                # additions
                if self.fake_zope_eggs:
                    print 'Creating fake eggs'
                    self.fakeEggs()
                return []
        else:
            smart_recompile = True

        if smart_recompile and os.path.exists(location):
            # checking if the c source where compiled. 
            if self._compiled(location): 
                if self.fake_zope_eggs: 
                    print 'Creating fake eggs' 
                    self.fakeEggs() 
                return [] 

        # full installation 
        if os.path.exists(location): 
            shutil.rmtree(location) 
        
        if self.svn:
            assert os.system('svn co %s %s' % (options['svn'], location)) == 0
        else:
            if not os.path.isdir(download_dir):
                os.mkdir(download_dir)

            _, _, urlpath, _, _, _ = urlparse.urlparse(self.url)
            tmp = tempfile.mkdtemp('buildout-'+self.name)
            try:
                fname = os.path.join(download_dir, urlpath.split('/')[-1])
                # Have we already downloaded the file
                if not os.path.exists(fname):
                    f = open(fname, 'wb')
                    try:
                        f.write(urllib2.urlopen(self.url).read())
                    except:
                        os.remove(fname)
                        raise
                    f.close()

                setuptools.archive_util.unpack_archive(fname, tmp)
                # The Zope tarballs have a Zope-<version> folder at the root
                # level, so we need to move that one into the right place.
                files = os.listdir(tmp)
                if len(files) == 0:
                    raise ValueError('Broken Zope tarball named %s' % fname)
                shutil.move(os.path.join(tmp, files[0]), location)
            finally:
                shutil.rmtree(tmp)

        os.chdir(location)
        assert os.spawnl(
            os.P_WAIT, options['executable'], options['executable'],
            'setup.py',
            'build_ext', '-i',
            ) == 0

        # compile .py files to .pyc;
        # ignore return status since compilezpy.py will return
        # an exist status of 1 for even a single failed compile.
        os.spawnl(
            os.P_WAIT, options['executable'], options['executable'],
            os.path.join(location, 'utilities', 'compilezpy.py'),
            'build_ext', '-i',
            )

        if self.fake_zope_eggs:
            print 'Creating fake eggs'
            self.fakeEggs()
        if self.url and options.get('shared-zope') == 'true':
            # don't return path if the installation is shared, so it doesn't
            # get deleted on uninstall
            return []
        return location

    def _getInstalledLibs(self, location, prefix):
        installedLibs = []
        for lib in os.listdir(location):
            name = '%s.%s' % (prefix, lib)
            if (os.path.isdir(os.path.join(location, lib)) and
                name not in self.skip_fake_eggs and
                name not in [libInfo.name for libInfo in self.libsToFake]):
                # Only add the package if it's not yet in the list and it's
                # not in the skip list
                installedLibs.append(FakeLibInfo(name))
        return installedLibs

    def fakeEggs(self):
        zope2Location = self.options['location']
        zopeLibZopeLocation = os.path.join(zope2Location, 'lib', 'python',
                                           'zope')
        zopeLibZopeAppLocation = os.path.join(zope2Location, 'lib', 'python',
                                              'zope', 'app')
        fakeEggsFolderLocation = os.path.join(self.buildout['buildout']['directory'],
                                              self.fake_eggs_folder)
        if not os.path.isdir(fakeEggsFolderLocation):
            os.mkdir(fakeEggsFolderLocation)

        self.libsToFake = []
        for lib in self.additional_fake_eggs:
            # 2 forms available:
            #  * additional-fake-eggs = myEgg
            #  * additional-fake-eggs = myEgg=0.4
            if '=' in lib:
                lib = lib.strip().split('=')
                eggName = lib[0].strip()
                version = lib[1].strip()
                libInfo = FakeLibInfo(eggName, version)
            else:
                eggName = lib.strip()
                libInfo = FakeLibInfo(eggName)

            self.libsToFake.append(libInfo)

        self.libsToFake += self._getInstalledLibs(zopeLibZopeLocation, 'zope')
        self.libsToFake += self._getInstalledLibs(zopeLibZopeAppLocation,
                                             'zope.app')

        developEggDir = self.buildout['buildout']['develop-eggs-directory']
        for libInfo in self.libsToFake:
            fakeLibDirLocation = os.path.join(fakeEggsFolderLocation,
                                              libInfo.name)
            fake_egg_link = os.path.join(developEggDir, '%s.egg-link' %\
                                         libInfo.name)
            if not os.path.isdir(fakeLibDirLocation):
                os.mkdir(fakeLibDirLocation)
            fakeLibEggInfoFile = os.path.join(fakeLibDirLocation,
                                              '%s.egg-info' % libInfo.name)
            fd = open(fakeLibEggInfoFile, 'w')
            fd.write(EGG_INFO_CONTENT % (libInfo.name, libInfo.version))
            fd.close()
            fd = open(fake_egg_link, 'w')
            fd.write("%s\n." % fakeLibDirLocation)
            fd.close()

        # Delete fake eggs, when we don't want them anymore
        for name in self.skip_fake_eggs:
            fake_egg_link = os.path.join(developEggDir, '%s.egg-link' % name)
            fakeLibDir = os.path.join(fakeEggsFolderLocation, name)
            if os.path.isdir(fakeLibDir):
                shutil.rmtree(fakeLibDir)
            if os.path.exists(fake_egg_link):
                os.remove(fake_egg_link)

    def update(self):
        options = self.options
        location = options['location']
        shared = options.get('shared-zope') 
        if os.path.exists(location):
            # Don't do anything in offline mode
            if self.buildout['buildout'].get('offline') == 'true' or \
               self.buildout['buildout'].get('newest') == 'false':
                if self.fake_zope_eggs:
                    print 'Updating fake eggs'
                    self.fakeEggs()
                if options.get('shared-zope') == 'true':
                    return []
                return location

            # If we downloaded a tarball, we don't need to do anything while
            # updating, otherwise we have a svn checkout and should run
            # 'svn up' and see if there has been any changes so we recompile
            # the c extensions
            if self.location or self.url:
                if self.fake_zope_eggs:
                    print 'Updating fake eggs'
                    self.fakeEggs()
                if options.get('shared-zope') == 'true':
                    return []
                return location
            
            if (self._compiled(location) and
                options.get('smart-recompile') == 'true'): 
                return location 

            os.chdir(location)
            stdin, stdout, stderr = os.popen3('svn up')
            stdin.close()
            for line in stderr.readlines():
                sys.stderr.write(line)
            stderr.close()
            change = re.compile('[AUM] ').match
            for l in stdout:
                if change(l):
                    stdout.read()
                    stdout.close()
                    break
                else:
                    # No change, so all done
                    stdout.close()
                    return location

            assert os.spawnl(
                os.P_WAIT, options['executable'], options['executable'],
                'setup.py',
                'build_ext', '-i',
                ) == 0

            if self.fake_zope_eggs:
                print 'Updating fake eggs'
                self.fakeEggs()

        return location
