""""""
#------------------------------------------------------------------------------
# Description    : SCPI interface
# Git repository : https://gitlab.com/qblox/packages/software/qblox_instruments.git
# Copyright (C) Qblox BV (2020)
#------------------------------------------------------------------------------


#------------------------------------------------------------------------------
# THIS FILE IS AUTOMATICALLY GENERATED. DO NOT MODIFY THIS FILE MANUALLY!
#------------------------------------------------------------------------------


#-- include --------------------------------------------------------------------

import sys
import os
import struct

#Add IEEE488.2 support
from ieee488_2.ieee488_2 import ieee488_2
from ieee488_2.transport import transport

#-- class ----------------------------------------------------------------------

class pulsar_qrm_scpi_ifc(ieee488_2):
    """
    This interface provides an API for the mandatory and required SCPI calls and adds
    Pulsar related functionality (see `SCPI <https://www.ivifoundation.org/docs/scpi-99.pdf>`_).
    """

    #---------------------------------------------------------------------------
    def __init__(self, transport_inst, debug = 0):
        """
        Creates SCPI interface object.

        Parameters
        ----------
        transport_inst : :class:`~ieee488_2.transport`
            Transport class responsible for the lowest level of communication (e.g. ethernet).
        debug : int
            Debug level (0 = normal, 1 = no version check, >1 = no version or error checking).

        Returns
        ----------
        :class:`~pulsar_qrm.pulsar_qrm_scpi_ifc`
            SCPI interface object.

        Raises
        ----------
        Exception
            Debug level is 0 and there is a version mismatch.
        """

        #Store parameters for later use.
        self._debug = debug

        #Initialize parent class.
        super(pulsar_qrm_scpi_ifc, self).__init__(transport_inst)

        if self._debug == 0:
            try:
                #Check if build is compatible.
                build     = self._read("*IDN?").split(',')[-1]
                build_ref = "fwVersion=0.3.0 fwBuild=20/11/2020-12:50:26 fwHash=0xF4E352F5 fwDirty=0 kmodVersion=0.3.0 kmodBuild=20/11/2020-12:50:26 kmodHash=0xF4E352F5 kmodDirty=0 swVersion=0.3.0 swBuild=20/11/2020-12:50:26 swHash=0xF4E352F5 swDirty=0"
                if build != build_ref:
                    raise Exception("Error: pulsar_qrm_scpi_ifc version ({}) is not compatible with device version ({}).".format(build_ref, build))

                #Clear SCPI error queue.
                while int(self._read('SYSTem:ERRor:COUNt?')) != 0:
                    self._read('SYSTem:ERRor:NEXT?')
            except Exception as err:
                raise

    #---------------------------------------------------------------------------
    def _get_scpi_commands(self):
        """
        Get SCPI commands.

        Parameters
        ----------

        Returns
        ----------
        str
            Concatinated list of strings separated by the semicolon character. Each command consists of nine sections respectively order as:

            - SCPI command pattern
            - SCPI input type
            - SCPI output type
            - Python function
            - Python input types (comma separated)
            - Python input variable names (comma separated)
            - Python output types (comma separated)
            - User access level (0 = public, >=1 = private)
            - Comment

        Raises
        ----------
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        #SCPI call
        try:
            var0 = self._read('*CMDS?')

            return var0
        except Exception as err:
            self._check_error_queue(err)
        finally:
            self._check_error_queue()

    #---------------------------------------------------------------------------
    def _get_idn(self):
        """
        Get device identity and build information.

        Parameters
        ----------

        Returns
        ----------
        str
            Concatinated list of strings separated by the semicolon character. The IDN consists of four strings respectively ordered as:

            - Manufacturer
            - Model
            - Serial number
            - Build information

        Raises
        ----------
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        #SCPI call
        try:
            var0 = self._read('*IDN?')

            return var0
        except Exception as err:
            self._check_error_queue(err)
        finally:
            self._check_error_queue()

    #---------------------------------------------------------------------------
    def reset(self):
        """
        Reset device and clear all status and event registers (see `SCPI <https://www.ivifoundation.org/docs/scpi-99.pdf>`_).

        Parameters
        ----------

        Returns
        ----------

        Raises
        ----------
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        #SCPI call
        try:
            self._write('*RST')
        except Exception as err:
            self._check_error_queue(err)
        finally:
            self._check_error_queue()

    #---------------------------------------------------------------------------
    def clear(self):
        """
        Clear all status and event registers (see `SCPI <https://www.ivifoundation.org/docs/scpi-99.pdf>`_).

        Parameters
        ----------

        Returns
        ----------

        Raises
        ----------
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        #SCPI call
        try:
            self._write('*CLS')
        except Exception as err:
            self._check_error_queue(err)
        finally:
            self._check_error_queue()

    #---------------------------------------------------------------------------
    def get_status_byte(self):
        """
        Get status byte register. Register is only cleared when feeding registers are cleared (see `SCPI <https://www.ivifoundation.org/docs/scpi-99.pdf>`_).

        Parameters
        ----------

        Returns
        ----------
        int
            Status byte register.

        Raises
        ----------
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        #SCPI call
        try:
            var0 = self._read('*STB?')

            return int(var0)
        except Exception as err:
            self._check_error_queue(err)
        finally:
            self._check_error_queue()

    #---------------------------------------------------------------------------
    def set_service_request_enable(self, reg):
        """
        Set service request enable register (see `SCPI <https://www.ivifoundation.org/docs/scpi-99.pdf>`_).

        Parameters
        ----------
        reg : int
           Service request enable register.

        Returns
        ----------

        Raises
        ----------
        Exception
            Invalid input parameter type.
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        #Check input types.
        self._check_in_type(locals(), ['int'])

        #SCPI call
        try:
            self._write('*SRE {}'.format(reg))
        except Exception as err:
            self._check_error_queue(err)
        finally:
            self._check_error_queue()

    #---------------------------------------------------------------------------
    def get_service_request_enable(self):
        """
        Get service request enable register. The register is cleared after reading it (see `SCPI <https://www.ivifoundation.org/docs/scpi-99.pdf>`_).

        Parameters
        ----------

        Returns
        ----------
        int
           Service request enable register.

        Raises
        ----------
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        #SCPI call
        try:
            var0 = self._read('*SRE?')

            return int(var0)
        except Exception as err:
            self._check_error_queue(err)
        finally:
            self._check_error_queue()

    #---------------------------------------------------------------------------
    def set_standard_event_status_enable(self, reg):
        """
        Get standard event status enable register. The register is cleared after reading it (see `SCPI <https://www.ivifoundation.org/docs/scpi-99.pdf>`_).

        Parameters
        ----------

        Returns
        ----------
        int
           Standard event status enable register.

        Raises
        ----------
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        #Check input types.
        self._check_in_type(locals(), ['int'])

        #SCPI call
        try:
            self._write('*ESE {}'.format(reg))
        except Exception as err:
            self._check_error_queue(err)
        finally:
            self._check_error_queue()

    #---------------------------------------------------------------------------
    def get_standard_event_status_enable(self):
        """
        Set standard event status enable register (see `SCPI <https://www.ivifoundation.org/docs/scpi-99.pdf>`_).

        Parameters
        ----------
        reg : int
           Standard event status enable register.

        Returns
        ----------

        Raises
        ----------
        Exception
            Invalid input parameter type.
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        #SCPI call
        try:
            var0 = self._read('*ESE?')

            return int(var0)
        except Exception as err:
            self._check_error_queue(err)
        finally:
            self._check_error_queue()

    #---------------------------------------------------------------------------
    def get_standard_event_status(self):
        """
        Get standard event status register. The register is cleared after reading it (see `SCPI <https://www.ivifoundation.org/docs/scpi-99.pdf>`_).

        Parameters
        ----------

        Returns
        ----------
        int
           Standard event status register.

        Raises
        ----------
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        #SCPI call
        try:
            var0 = self._read('*ESR?')

            return int(var0)
        except Exception as err:
            self._check_error_queue(err)
        finally:
            self._check_error_queue()

    #---------------------------------------------------------------------------
    def set_operation_complete(self):
        """
        Set device in operation complete query active state (see `SCPI <https://www.ivifoundation.org/docs/scpi-99.pdf>`_).

        Parameters
        ----------

        Returns
        ----------

        Raises
        ----------
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        #SCPI call
        try:
            self._write('*OPC')
        except Exception as err:
            self._check_error_queue(err)
        finally:
            self._check_error_queue()

    #---------------------------------------------------------------------------
    def get_operation_complete(self):
        """
        Get operation complete state (see `SCPI <https://www.ivifoundation.org/docs/scpi-99.pdf>`_).

        Parameters
        ----------

        Returns
        ----------
        bool
           Operation complete state (False = running, True = completed).

        Raises
        ----------
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        #SCPI call
        try:
            var0 = self._read('*OPC?')

            return bool(int(var0))
        except Exception as err:
            self._check_error_queue(err)
        finally:
            self._check_error_queue()

    #---------------------------------------------------------------------------
    def test(self):
        """
        Run self-test. Currently not implemented (see `SCPI <https://www.ivifoundation.org/docs/scpi-99.pdf>`_).

        Parameters
        ----------

        Returns
        ----------
        bool
           Test result (False = failed, True = success).

        Raises
        ----------
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        #SCPI call
        try:
            var0 = self._read('*TST?')

            return bool(int(var0))
        except Exception as err:
            self._check_error_queue(err)
        finally:
            self._check_error_queue()

    #---------------------------------------------------------------------------
    def wait(self):
        """
        Wait until operations completed before continuing (see `SCPI <https://www.ivifoundation.org/docs/scpi-99.pdf>`_).

        Parameters
        ----------

        Returns
        ----------

        Raises
        ----------
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        #SCPI call
        try:
            self._write('*WAI')
        except Exception as err:
            self._check_error_queue(err)
        finally:
            self._check_error_queue()

    #---------------------------------------------------------------------------
    def get_system_error(self):
        """
        Get system error from queue (see `SCPI <https://www.ivifoundation.org/docs/scpi-99.pdf>`_).

        Parameters
        ----------

        Returns
        ----------
        str
           System error description string.
        """

        #SCPI call
        try:
            var0 = self._read('SYSTem:ERRor:NEXT?')

            return var0
        except:
            raise

    #---------------------------------------------------------------------------
    def get_num_system_error(self):
        """
        Get number of system errors (see `SCPI <https://www.ivifoundation.org/docs/scpi-99.pdf>`_).

        Parameters
        ----------

        Returns
        ----------
        int
           Current number of number of system errors.
        """

        #SCPI call
        try:
            var0 = self._read('SYSTem:ERRor:COUNt?')

            return int(var0)
        except:
            raise

    #---------------------------------------------------------------------------
    def get_system_version(self):
        """
        Get SCPI system version (see `SCPI <https://www.ivifoundation.org/docs/scpi-99.pdf>`_).

        Parameters
        ----------

        Returns
        ----------
        str
           SCPI system version.

        Raises
        ----------
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        #SCPI call
        try:
            var0 = self._read('SYSTem:VERSion?')

            return var0
        except Exception as err:
            self._check_error_queue(err)
        finally:
            self._check_error_queue()

    #---------------------------------------------------------------------------
    def preset_system_status(self):
        """
        Preset system status registers. Connects general system status flags for PLL unlock and temperature out-of-range indications
        to event status enable, status questionable temperature and status questionable frequency registers respectively
        (see `SCPI <https://www.ivifoundation.org/docs/scpi-99.pdf>`_).

        Parameters
        ----------

        Returns
        ----------

        Raises
        ----------
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        #SCPI call
        try:
            self._write('STATus:PRESet')
        except Exception as err:
            self._check_error_queue(err)
        finally:
            self._check_error_queue()

    #---------------------------------------------------------------------------
    def _get_system_status(self):
        """
        Get general system state.

        Parameters
        ----------

        Returns
        ----------
        str
            Concatinated list of strings separated by the semicolon character. Status is indicated by one status string and an optional number of flags respectively ordered as:

            :Status:

                - OKAY: System is okay.
                - CRITICAL: An error indicated by the flags occured, but has been resolved.
                - ERROR: An error indicated by the flags is occuring.

            :Flags:

                - CARRIER_PLL_UNLOCK: Carrier board PLL is unlocked.
                - FPGA_PLL_UNLOCK: FPGA PLL is unlocked.
                - FPGA_TEMP_OR: FPGA temperature is out-of-range.
                - CARRIER_TEMP_OR: Carrier board temperature is out-of-range.
                - AFE_TEMP_OR: Analog frontend board temperature is out-of-range.

        Raises
        ----------
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        #SCPI call
        try:
            var0 = self._read('STATus:GENeral:STATE?')

            return var0
        except Exception as err:
            self._check_error_queue(err)
        finally:
            self._check_error_queue()

    #---------------------------------------------------------------------------
    def get_questionable_condition(self):
        """
        Get status questionable condition register (see `SCPI <https://www.ivifoundation.org/docs/scpi-99.pdf>`_).

        Parameters
        ----------

        Returns
        ----------
        int
            Status questionable condition register.

        Raises
        ----------
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        #SCPI call
        try:
            var0 = self._read('STATus:QUEStionable:CONDition?')

            return int(var0)
        except Exception as err:
            self._check_error_queue(err)
        finally:
            self._check_error_queue()

    #---------------------------------------------------------------------------
    def get_questionable_event(self):
        """
        Get status questionable event register (see `SCPI <https://www.ivifoundation.org/docs/scpi-99.pdf>`_).

        Parameters
        ----------

        Returns
        ----------
        int
            Status questionable event register.

        Raises
        ----------
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        #SCPI call
        try:
            var0 = self._read('STATus:QUEStionable:EVENt?')

            return int(var0)
        except Exception as err:
            self._check_error_queue(err)
        finally:
            self._check_error_queue()

    #---------------------------------------------------------------------------
    def set_questionable_enable(self, reg):
        """
        Set status questionable enable register (see `SCPI <https://www.ivifoundation.org/docs/scpi-99.pdf>`_).

        Parameters
        ----------
        reg : int
            Status questionable enable register.

        Returns
        ----------

        Raises
        ----------
        Exception
            Invalid input parameter type.
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        #Check input types.
        self._check_in_type(locals(), ['int'])

        #SCPI call
        try:
            self._write('STATus:QUEStionable:ENABle {}'.format(reg))
        except Exception as err:
            self._check_error_queue(err)
        finally:
            self._check_error_queue()

    #---------------------------------------------------------------------------
    def get_questionable_enable(self):
        """
        Get status questionable enable register (see `SCPI <https://www.ivifoundation.org/docs/scpi-99.pdf>`_).

        Parameters
        ----------

        Returns
        ----------
        int
            Status questionable enable register.

        Raises
        ----------
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        #SCPI call
        try:
            var0 = self._read('STATus:QUEStionable:ENABle?')

            return int(var0)
        except Exception as err:
            self._check_error_queue(err)
        finally:
            self._check_error_queue()

    #---------------------------------------------------------------------------
    def get_current_fpga_temperature(self):
        """
        Get current FPGA junction temperature (inside device).

        Parameters
        ----------

        Returns
        ----------
        float
            Current FPGA junction temperature.

        Raises
        ----------
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        #SCPI call
        try:
            var0 = self._read('STATus:QUEStionable:TEMPerature:FPGA:CURRent?')

            return float(var0)
        except Exception as err:
            self._check_error_queue(err)
        finally:
            self._check_error_queue()

    #---------------------------------------------------------------------------
    def get_maximum_fpga_temperature(self):
        """
        Get maximum FPGA junction temperature since boot or clear (inside device).

        Parameters
        ----------

        Returns
        ----------
        float
            Maximum FPGA junction temperature.

        Raises
        ----------
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        #SCPI call
        try:
            var0 = self._read('STATus:QUEStionable:TEMPerature:FPGA:MAXimum?')

            return float(var0)
        except Exception as err:
            self._check_error_queue(err)
        finally:
            self._check_error_queue()

    #---------------------------------------------------------------------------
    def get_current_carrier_temperature(self):
        """
        Get current carrier board temperature (inside device).

        Parameters
        ----------

        Returns
        ----------
        float
            Current carrier board temperature.

        Raises
        ----------
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        #SCPI call
        try:
            var0 = self._read('STATus:QUEStionable:TEMPerature:CARRier:CURRent?')

            return float(var0)
        except Exception as err:
            self._check_error_queue(err)
        finally:
            self._check_error_queue()

    #---------------------------------------------------------------------------
    def get_maximum_carrier_temperature(self):
        """
        Get maximum carrier board temperature since boot or clear (inside device).

        Parameters
        ----------

        Returns
        ----------
        float
            Maximum carrier board temperature.

        Raises
        ----------
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        #SCPI call
        try:
            var0 = self._read('STATus:QUEStionable:TEMPerature:CARRier:MAXimum?')

            return float(var0)
        except Exception as err:
            self._check_error_queue(err)
        finally:
            self._check_error_queue()

    #---------------------------------------------------------------------------
    def get_current_afe_temperature(self):
        """
        Get current analog frontend board temperature (inside device).

        Parameters
        ----------

        Returns
        ----------
        float
            Current analog frontend board temperature.

        Raises
        ----------
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        #SCPI call
        try:
            var0 = self._read('STATus:QUEStionable:TEMPerature:AFE:CURRent?')

            return float(var0)
        except Exception as err:
            self._check_error_queue(err)
        finally:
            self._check_error_queue()

    #---------------------------------------------------------------------------
    def get_maximum_afe_temperature(self):
        """
        Get maximum analog frontend board temperature since boot or clear (inside device).

        Parameters
        ----------

        Returns
        ----------
        float
            Maximum analog frontend board temperature.

        Raises
        ----------
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        #SCPI call
        try:
            var0 = self._read('STATus:QUEStionable:TEMPerature:AFE:MAXimum?')

            return float(var0)
        except Exception as err:
            self._check_error_queue(err)
        finally:
            self._check_error_queue()

    #---------------------------------------------------------------------------
    def _set_reference_source(self, internal):
        """
        Set reference (10MHz) clock source.

        Parameters
        ----------
        internal : bool
            Reference clock source (False = External, True = Internal).

        Returns
        ----------

        Raises
        ----------
        Exception
            Invalid input parameter type.
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        #Check input types.
        self._check_in_type(locals(), ['bool'])

        #SCPI call
        try:
            self._write('STATus:QUEStionable:FREQuency:REFerence:SRC {}'.format(0 if internal == False else 1))
        except Exception as err:
            self._check_error_queue(err)
        finally:
            self._check_error_queue()

    #---------------------------------------------------------------------------
    def _get_reference_source(self):
        """
        Get reference (10MHz) clock source.

        Parameters
        ----------

        Returns
        ----------
        bool
            Reference clock source (False = External, True = Internal).

        Raises
        ----------
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        #SCPI call
        try:
            var0 = self._read('STATus:QUEStionable:FREQuency:REFerence:SRC?')

            return bool(int(var0))
        except Exception as err:
            self._check_error_queue(err)
        finally:
            self._check_error_queue()

    #---------------------------------------------------------------------------
    def get_operation_condition(self):
        """
        Get status operation condition register  (see `SCPI <https://www.ivifoundation.org/docs/scpi-99.pdf>`_).

        Parameters
        ----------

        Returns
        ----------
        int
            Status operation condition register.

        Raises
        ----------
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        #SCPI call
        try:
            var0 = self._read('STATus:OPERation:CONDition?')

            return int(var0)
        except Exception as err:
            self._check_error_queue(err)
        finally:
            self._check_error_queue()

    #---------------------------------------------------------------------------
    def get_operation_events(self):
        """
        Get status operation event register  (see `SCPI <https://www.ivifoundation.org/docs/scpi-99.pdf>`_).

        Parameters
        ----------

        Returns
        ----------
        int
            Status operation event register.

        Raises
        ----------
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        #SCPI call
        try:
            var0 = self._read('STATus:OPERation:EVENt?')

            return int(var0)
        except Exception as err:
            self._check_error_queue(err)
        finally:
            self._check_error_queue()

    #---------------------------------------------------------------------------
    def set_operation_enable(self, reg):
        """
        Set status operation enable register  (see `SCPI <https://www.ivifoundation.org/docs/scpi-99.pdf>`_).

        Parameters
        ----------
        reg : int
            Status operation enable register.

        Returns
        ----------

        Raises
        ----------
        Exception
            Invalid input parameter type.
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        #Check input types.
        self._check_in_type(locals(), ['int'])

        #SCPI call
        try:
            self._write('STATus:OPERation:ENABle {}'.format(reg))
        except Exception as err:
            self._check_error_queue(err)
        finally:
            self._check_error_queue()

    #---------------------------------------------------------------------------
    def get_operation_enable(self):
        """
        Get status operation enable register  (see `SCPI <https://www.ivifoundation.org/docs/scpi-99.pdf>`_).

        Parameters
        ----------

        Returns
        ----------
        int
            Status operation enable register.

        Raises
        ----------
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        #SCPI call
        try:
            var0 = self._read('STATus:OPERation:ENABle?')

            return int(var0)
        except Exception as err:
            self._check_error_queue(err)
        finally:
            self._check_error_queue()

    #---------------------------------------------------------------------------
    def get_assembler_status(self):
        """
        Get assembler status. Refer to the assembler log to get more information regarding the assembler result.

        Parameters
        ----------

        Returns
        ----------
        bool
            Assembler status (False = failed, True = success).

        Raises
        ----------
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        #SCPI call
        try:
            var0 = self._read('STATus:ASSEMbler:SUCCess?')

            return bool(int(var0))
        except Exception as err:
            self._check_error_queue(err)
        finally:
            self._check_error_queue()

    #---------------------------------------------------------------------------
    def get_assembler_log(self):
        """
        Get assembler log.

        Parameters
        ----------

        Returns
        ----------
        str
            Formatted assembler log string.

        Raises
        ----------
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        #SCPI call
        try:
            var0 = self._read_bin('STATus:ASSEMbler:LOG?')

            return var0.decode('utf-8', 'ignore')
        except Exception as err:
            self._check_error_queue(err)
        finally:
            self._check_error_queue()

    #---------------------------------------------------------------------------
    def _set_in_amp_gain_0(self, gain):
        """
        Set input amplifier gain for input 0.

        Parameters
        ----------
        gain : int
            Gain in dB in a range of -6 to 26.

        Returns
        ----------

        Raises
        ----------
        Exception
            Invalid input parameter type.
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        #Check input types.
        self._check_in_type(locals(), ['int'])

        #SCPI call
        try:
            self._write('AFE:GAIN:INAMP0 {}'.format(gain))
        except Exception as err:
            self._check_error_queue(err)
        finally:
            self._check_error_queue()

    #---------------------------------------------------------------------------
    def _get_in_amp_gain_0(self):
        """
        Get input amplifier gain for input 0.

        Parameters
        ----------

        Returns
        ----------
        int
            Gain in dB in a range of -6 to 26.

        Raises
        ----------
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        #SCPI call
        try:
            var0 = self._read('AFE:GAIN:INAMP0?')

            return var0
        except Exception as err:
            self._check_error_queue(err)
        finally:
            self._check_error_queue()

    #---------------------------------------------------------------------------
    def _set_in_amp_gain_1(self, gain):
        """
        Set input amplifier gain for input 1.

        Parameters
        ----------
        gain : int
            Gain in dB in a range of -6 to 26.

        Returns
        ----------

        Raises
        ----------
        Exception
            Invalid input parameter type.
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        #Check input types.
        self._check_in_type(locals(), ['int'])

        #SCPI call
        try:
            self._write('AFE:GAIN:INAMP1 {}'.format(gain))
        except Exception as err:
            self._check_error_queue(err)
        finally:
            self._check_error_queue()

    #---------------------------------------------------------------------------
    def _get_in_amp_gain_1(self):
        """
        Get input amplifier gain for input 1.

        Parameters
        ----------

        Returns
        ----------
        int
            Gain in dB in a range of -6 to 26.

        Raises
        ----------
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        #SCPI call
        try:
            var0 = self._read('AFE:GAIN:INAMP1?')

            return var0
        except Exception as err:
            self._check_error_queue(err)
        finally:
            self._check_error_queue()

    #---------------------------------------------------------------------------
    def _set_sequencer_program(self, sequencer, program):
        """
        Assemble and set the resulting Q1ASM program for the indexed sequencer. If an invalid sequencer index is given or assembling
        failes, an error is set in system error. The assembler status and log can be retrieved using separate commands. To set the Q1ASM
        program, the sequencer will be stopped.

        Parameters
        ----------
        sequencer : int
            Sequencer index.
        program : str
            Q1ASM program.

        Returns
        ----------

        Raises
        ----------
        Exception
            Invalid input parameter type.
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        #Check input types.
        self._check_in_type(locals(), ['int', 'str'])

        #SCPI call
        try:
            self._write_bin('SEQuencer{}:PROGram '.format(sequencer), program.encode('ascii'))
        except Exception as err:
            self._check_error_queue(err)
        finally:
            self._check_error_queue()

    #---------------------------------------------------------------------------
    def _set_sequencer_config(self, sequencer, config):
        """
        Set configuration of the indexed sequencer. The configuration consists of multiple parameters in a C struct format.
        If an invalid sequencer index is given or the configation struct does not have the correct format, an error is set in
        system error.

        Parameters
        ----------
        sequencer : int
            Sequencer index.
        config : bytes
            Configuration struct.

        Returns
        ----------

        Raises
        ----------
        Exception
            Invalid input parameter type.
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        #Check input types.
        self._check_in_type(locals(), ['int', 'bytes'])

        #SCPI call
        try:
            self._write_bin('SEQuencer{}:CONFiguration '.format(sequencer), config)
        except Exception as err:
            self._check_error_queue(err)
        finally:
            self._check_error_queue()

    #---------------------------------------------------------------------------
    def _get_sequencer_config(self, sequencer):
        """
        Get configuration of the indexed sequencer. The configuration consists of multiple parameters in a C struct format.
        If an invalid sequencer index is given, an error is set in system error.

        Parameters
        ----------
        sequencer : int
            Sequencer index.

        Returns
        ----------
        bytes
            Configuration struct.

        Raises
        ----------
        Exception
            Invalid input parameter type.
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        #Check input types.
        self._check_in_type(locals(), ['int'])

        #SCPI call
        try:
            var0 = self._read_bin('SEQuencer{}:CONFiguration?'.format(sequencer))

            return var0
        except Exception as err:
            self._check_error_queue(err)
        finally:
            self._check_error_queue()

    #---------------------------------------------------------------------------
    def _arm_sequencer(self, sequencer):
        """
        Prepare the indexed sequencer to start by putting it in the armed state. If no sequencer index is given, all sequencers are armed.
        Any sequencer that was already running is stopped and rearmed. If an invalid sequencer index is given, an error is set in system error.

        Parameters
        ----------
        sequencer : int
            Sequencer index.

        Returns
        ----------

        Raises
        ----------
        Exception
            Invalid input parameter type.
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        #Check input types.
        self._check_in_type(locals(), ['int'])

        #SCPI call
        try:
            self._write('SEQuencer{}:ARM'.format(sequencer))
        except Exception as err:
            self._check_error_queue(err)
        finally:
            self._check_error_queue()

    #---------------------------------------------------------------------------
    def _start_sequencer(self, sequencer):
        """
        Start the indexed sequencer, thereby putting it in the running state. If an invalid sequencer index is given or the indexed sequencer was
        not yet armed, an error is set in system error. If no sequencer index is given, all armed sequencers are started and any sequencer not in
        the armed state is ignored. However, if no sequencer index is given and no sequencers are armed, and error is set in system error.

        Parameters
        ----------
        sequencer : int
            Sequencer index.

        Returns
        ----------

        Raises
        ----------
        Exception
            Invalid input parameter type.
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        #Check input types.
        self._check_in_type(locals(), ['int'])

        #SCPI call
        try:
            self._write('SEQuencer{}:START'.format(sequencer))
        except Exception as err:
            self._check_error_queue(err)
        finally:
            self._check_error_queue()

    #---------------------------------------------------------------------------
    def _stop_sequencer(self, sequencer):
        """
        Stop the indexed sequencer, thereby putting it in the stopped state. If an invalid sequencer index is given, an error is set in system error.
        If no sequencer index is given, all sequencers are stopped.

        Parameters
        ----------
        sequencer : int
            Sequencer index.

        Returns
        ----------

        Raises
        ----------
        Exception
            Invalid input parameter type.
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        #Check input types.
        self._check_in_type(locals(), ['int'])

        #SCPI call
        try:
            self._write('SEQuencer{}:STOP'.format(sequencer))
        except Exception as err:
            self._check_error_queue(err)
        finally:
            self._check_error_queue()

    #---------------------------------------------------------------------------
    def _get_sequencer_state(self, sequencer):
        """
        Get the sequencer state. If an invalid sequencer index is given, an error is set in system error.

        Parameters
        ----------
        sequencer : int
            Sequencer index.

        Returns
        ----------
        str
            Concatinated list of strings separated by the semicolon character. Status is indicated by one status string and an optional number of flags respectively ordered as:

            :Status:

                - IDLE:
                - ARMED: System is okay.
                - Q1 STOPPED: An error indicated by the flags occured, but has been resolved.
                - STOPPED: An error indicated by the flags is occuring.

            :Flags:

                - DISARMED: Sequencer was disarmed.
                - FORCED STOP: Sequencer was stopped while still running.
                - SEQUENCE PROCESSOR Q1 ILLEGAL INSTRUCTION: Classical sequencer part executed an unknown instruction.
                - SEQUENCE PROCESSOR RT EXEC ILLEGAL INSTRUCTION: Real-time sequencer part executed an unknown instruction.
                - AWG WAVE PLAYBACK INVALID PATH 0: AWG path 0 tried to play an unknown waveform.
                - AWG WAVE PLAYBACK INVALID PATH 1: AWG path 1 tried to play an unknown waveform.
                - ACQ WAVE PLAYBACK INVALID PATH 0: Acquisition path 0 tried to play an unknown waveform.
                - ACQ WAVE PLAYBACK INVALID PATH 1: Acquisition path 1 tried to play an unknown waveform.
                - ACQ WAVE CAPTURE OUT-OF-RANGE PATH 0: Acquisition path 0 data was out-of-range.
                - ACQ WAVE CAPTURE DONE PATH 0: Acquisition path 0 has finished.
                - ACQ WAVE CAPTURE OVERWRITTEN PATH 0: Acquisition path 0 data was overwritten.
                - ACQ WAVE CAPTURE OUT-OF-RANGE PATH 1: Acquisition path 1 data was out-of-range.
                - ACQ WAVE CAPTURE DONE PATH 1: Acquisition path 1 has finished.
                - ACQ WAVE CAPTURE OVERWRITTEN PATH 1: Acquisition path 1 data was overwritten.
                - CLOCK INSTABILITY: Clock source instability occurred.

        Raises
        ----------
        Exception
            Invalid input parameter type.
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        #Check input types.
        self._check_in_type(locals(), ['int'])

        #SCPI call
        try:
            var0 = self._read('SEQuencer{}:STATE?'.format(sequencer))

            return var0
        except Exception as err:
            self._check_error_queue(err)
        finally:
            self._check_error_queue()

    #---------------------------------------------------------------------------
    def _add_awg_waveform(self, sequencer, name, size, is_integer):
        """
        Add new waveform to waveform list of indexed sequencer's AWG path. If an invalid sequencer index is given or if the waveform causes the waveform
        memory limit to be exceeded, an error is set in system error. The waveform names 'all' and 'ALL' are reserved and adding those will also result in an
        error being set in system error.

        Parameters
        ----------
        sequencer : int
            Sequencer index.
        name : str
            Waveform name.
        size : int
            Waveform length in number of samples.
        is_integer : bool
            Waveform is provided as integers (False = floats, True = integers).

        Returns
        ----------

        Raises
        ----------
        Exception
            Invalid input parameter type.
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        #Check input types.
        self._check_in_type(locals(), ['int', 'str', 'int', 'bool'])

        #SCPI call
        try:
            self._write('SEQuencer{}:AWG:WLISt:WAVeform:NEW "{}",{},{}'.format(sequencer, name, size, 0 if is_integer == False else 1))
        except Exception as err:
            self._check_error_queue(err)
        finally:
            self._check_error_queue()

    #---------------------------------------------------------------------------
    def _delete_awg_waveform(self, sequencer, name):
        """
        Delete waveform from waveform list of indexed sequencer's AWG path. If an invalid sequencer index is given or if a non-existing waveform name is given,
        an error is set in system error. The waveform names 'all' and 'ALL' are reserved and deleting those will cause a purge of the entire waveform list.

        Parameters
        ----------
        sequencer : int
            Sequencer index.
        name : str
            Waveform name.

        Returns
        ----------

        Raises
        ----------
        Exception
            Invalid input parameter type.
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        #Check input types.
        self._check_in_type(locals(), ['int', 'str'])

        #SCPI call
        try:
            self._write('SEQuencer{}:AWG:WLISt:WAVeform:DELete "{}"'.format(sequencer, name))
        except Exception as err:
            self._check_error_queue(err)
        finally:
            self._check_error_queue()

    #---------------------------------------------------------------------------
    def _set_awg_waveform_data(self, sequencer, name, waveform):
        """
        Set waveform data of waveform in waveform list of indexed sequencer's AWG path. If an invalid sequencer index is given or if a non-existing waveform
        name is given, an error is set in system error. If the wavefrom size does not match the size specified while adding the waveform or if the wavefrom
        samples are out-of-range, an error is set in the system error.

        Parameters
        ----------
        sequencer : int
            Sequencer index.
        name : str
            Waveform name.
        waveform : list
            List of floats in the range of 1.0 to -1.0 representing the waveform.

        Returns
        ----------

        Raises
        ----------
        Exception
            Invalid input parameter type.
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        #Check input types.
        self._check_in_type(locals(), ['int', 'str', 'float'])

        #SCPI call
        try:
            self._write_bin('SEQuencer{}:AWG:WLISt:WAVeform:DATA "{}",'.format(sequencer, name), struct.pack('f'*len(waveform), *waveform))
        except Exception as err:
            self._check_error_queue(err)
        finally:
            self._check_error_queue()

    #---------------------------------------------------------------------------
    def _get_awg_waveform_data(self, sequencer, name, start, size):
        """
        Get waveform data of waveform in waveform list of indexed sequencer's AWG path. If an invalid sequencer index is given or if a non-existing waveform
        name is given, an error is set in system error. The start and size arguments can be used to return a section of the waveform. If the start argument
        is not given, zero is used as the default start sample. If the size argument is not given, all samples starting from the start sample are returned.

        Parameters
        ----------
        sequencer : int
            Sequencer index.
        name : str
            Waveform name.
        start : int
            First sample within the waveform to be returned.
        size : int
            Number of samples starting from the start sample to be returned.

        Returns
        ----------
        list
            List of floats in the range of 1.0 to -1.0 representing the waveform.

        Raises
        ----------
        Exception
            Invalid input parameter type.
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        #Check input types.
        self._check_in_type(locals(), ['int', 'str', 'int', 'int'])

        #SCPI call
        try:
            var0 = self._read_bin('SEQuencer{}:AWG:WLISt:WAVeform:DATA? "{}",{},{}'.format(sequencer, name, start, size))

            return struct.unpack('f'*int(len(var0)/4), var0)
        except Exception as err:
            self._check_error_queue(err)
        finally:
            self._check_error_queue()

    #---------------------------------------------------------------------------
    def _set_awg_waveform_index(self, sequencer, name, index):
        """
        Set waveform index of waveform in waveform list of indexed sequencer's AWG path. The index is used by the sequencer Q1ASM program to refer to the waveform.
        If an invalid sequencer or waveform index is given or if a non-existing waveform name is given, an error is set in system error.

        Parameters
        ----------
        sequencer : int
            Sequencer index.
        name : str
            Waveform name.
        index : int
            Waveform index of the waveform in the waveform list.

        Returns
        ----------

        Raises
        ----------
        Exception
            Invalid input parameter type.
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        #Check input types.
        self._check_in_type(locals(), ['int', 'str', 'int'])

        #SCPI call
        try:
            self._write('SEQuencer{}:AWG:WLISt:WAVeform:INDex "{}",{}'.format(sequencer, name, index))
        except Exception as err:
            self._check_error_queue(err)
        finally:
            self._check_error_queue()

    #---------------------------------------------------------------------------
    def _get_awg_waveform_index(self, sequencer, name):
        """
        Get waveform index of waveform in waveform list of indexed sequencer's AWG path. The index is used by the sequencer Q1ASM program to refer to the waveform.
        If an invalid sequencer index is given or if a non-existing waveform name is given, an error is set in system error.

        Parameters
        ----------
        sequencer : int
            Sequencer index.
        name : str
            Waveform name.

        Returns
        ----------
        int
            Waveform index of the waveform in the waveform list.

        Raises
        ----------
        Exception
            Invalid input parameter type.
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        #Check input types.
        self._check_in_type(locals(), ['int', 'str'])

        #SCPI call
        try:
            var0 = self._read('SEQuencer{}:AWG:WLISt:WAVeform:INDex? "{}"'.format(sequencer, name))

            return int(var0)
        except Exception as err:
            self._check_error_queue(err)
        finally:
            self._check_error_queue()

    #---------------------------------------------------------------------------
    def _get_awg_waveform_length(self, sequencer, name):
        """
        Get length of waveform in waveform list of indexed sequencer's AWG path. If an invalid sequencer index is given or if a non-existing waveform name is given,
        an error is set in system error.

        Parameters
        ----------
        sequencer : int
            Sequencer index.
        name : str
            Waveform name.

        Returns
        ----------
        int
            Waveform length in number of samples.

        Raises
        ----------
        Exception
            Invalid input parameter type.
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        #Check input types.
        self._check_in_type(locals(), ['int', 'str'])

        #SCPI call
        try:
            var0 = self._read('SEQuencer{}:AWG:WLISt:WAVeform:LENGth? "{}"'.format(sequencer, name))

            return int(var0)
        except Exception as err:
            self._check_error_queue(err)
        finally:
            self._check_error_queue()

    #---------------------------------------------------------------------------
    def _get_awg_waveform_name(self, sequencer, index):
        """
        Get name of waveform in waveform list of indexed sequencer's AWG path based on its waveform index. If an invalid sequencer index is given or if a non-existing
        waveform index is given, an error is set in system error.

        Parameters
        ----------
        sequencer : int
            Sequencer index.
        index : str
            Waveform index.

        Returns
        ----------
        str
            Waveform name.

        Raises
        ----------
        Exception
            Invalid input parameter type.
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        #Check input types.
        self._check_in_type(locals(), ['int', 'int'])

        #SCPI call
        try:
            var0 = self._read('SEQuencer{}:AWG:WLISt:WAVeform:NAME? {}'.format(sequencer, index))

            return var0
        except Exception as err:
            self._check_error_queue(err)
        finally:
            self._check_error_queue()

    #---------------------------------------------------------------------------
    def _get_awg_num_waveforms(self, sequencer):
        """
        Get number of waveforms in waveform list of indexed sequencer's AWG path. If an invalid sequencer index is given, an error is set in system error.

        Parameters
        ----------
        sequencer : int
            Sequencer index.

        Returns
        ----------
        int
            Number of waveforms.

        Raises
        ----------
        Exception
            Invalid input parameter type.
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        #Check input types.
        self._check_in_type(locals(), ['int'])

        #SCPI call
        try:
            var0 = self._read('SEQuencer{}:AWG:WLISt:SIZE?'.format(sequencer))

            return int(var0)
        except Exception as err:
            self._check_error_queue(err)
        finally:
            self._check_error_queue()

    #---------------------------------------------------------------------------
    def _get_awg_waveform_names(self, sequencer):
        """
        Get all waveform names in waveform list of indexed sequencer's AWG path. If an invalid sequencer index is given, an error is set in system error.

        Parameters
        ----------
        sequencer : int
            Sequencer index.

        Returns
        ----------
        int
            Number of waveforms.

        Raises
        ----------
        Exception
            Invalid input parameter type.
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        #Check input types.
        self._check_in_type(locals(), ['int'])

        #SCPI call
        try:
            var0 = self._read('SEQuencer{}:AWG:WLISt?'.format(sequencer))

            return var0
        except Exception as err:
            self._check_error_queue(err)
        finally:
            self._check_error_queue()

    #---------------------------------------------------------------------------
    def _add_acq_waveform(self, sequencer, name, size, is_integer):
        """
        Add new waveform to waveform list of indexed sequencer's acquisition path. If an invalid sequencer index is given or if the waveform causes the waveform
        memory limit to be exceeded, an error is set in system error. The waveform names 'all' and 'ALL' are reserved and adding those will also result in an
        error being set in system error.

        Parameters
        ----------
        sequencer : int
            Sequencer index.
        name : str
            Waveform name.
        size : int
            Waveform length in number of samples.
        is_integer : bool
            Waveform is provided as integers (False = floats, True = integers).

        Returns
        ----------

        Raises
        ----------
        Exception
            Invalid input parameter type.
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        #Check input types.
        self._check_in_type(locals(), ['int', 'str', 'int', 'bool'])

        #SCPI call
        try:
            self._write('SEQuencer{}:ACQ:WLISt:WAVeform:NEW "{}",{},{}'.format(sequencer, name, size, 0 if is_integer == False else 1))
        except Exception as err:
            self._check_error_queue(err)
        finally:
            self._check_error_queue()

    #---------------------------------------------------------------------------
    def _delete_acq_waveform(self, sequencer, name):
        """
        Delete waveform from waveform list of indexed sequencer's acquisition path. If an invalid sequencer index is given or if a non-existing waveform name is given,
        an error is set in system error. The waveform names 'all' and 'ALL' are reserved and deleting those will cause a purge of the entire waveform list.

        Parameters
        ----------
        sequencer : int
            Sequencer index.
        name : str
            Waveform name.

        Returns
        ----------

        Raises
        ----------
        Exception
            Invalid input parameter type.
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        #Check input types.
        self._check_in_type(locals(), ['int', 'str'])

        #SCPI call
        try:
            self._write('SEQuencer{}:ACQ:WLISt:WAVeform:DELete "{}"'.format(sequencer, name))
        except Exception as err:
            self._check_error_queue(err)
        finally:
            self._check_error_queue()

    #---------------------------------------------------------------------------
    def _set_acq_waveform_data(self, sequencer, name, waveform):
        """
        Set waveform data of waveform in waveform list of indexed sequencer's acquisition path. If an invalid sequencer index is given or if a non-existing waveform
        name is given, an error is set in system error. If the wavefrom size does not match the size specified while adding the waveform or if the wavefrom
        samples are out-of-range, an error is set in the system error.

        Parameters
        ----------
        sequencer : int
            Sequencer index.
        name : str
            Waveform name.
        waveform : list
            List of floats in the range of 1.0 to -1.0 representing the waveform.

        Returns
        ----------

        Raises
        ----------
        Exception
            Invalid input parameter type.
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        #Check input types.
        self._check_in_type(locals(), ['int', 'str', 'float'])

        #SCPI call
        try:
            self._write_bin('SEQuencer{}:ACQ:WLISt:WAVeform:DATA "{}",'.format(sequencer, name), struct.pack('f'*len(waveform), *waveform))
        except Exception as err:
            self._check_error_queue(err)
        finally:
            self._check_error_queue()

    #---------------------------------------------------------------------------
    def _get_acq_waveform_data(self, sequencer, name, start, size):
        """
        Get waveform data of waveform in waveform list of indexed sequencer's acquisition path. If an invalid sequencer index is given or if a non-existing waveform
        name is given, an error is set in system error. The start and size arguments can be used to return a section of the waveform. If the start argument
        is not given, zero is used as the default start sample. If the size argument is not given, all samples starting from the start sample are returned.

        Parameters
        ----------
        sequencer : int
            Sequencer index.
        name : str
            Waveform name.
        start : int
            First sample within the waveform to be returned.
        size : int
            Number of samples starting from the start sample to be returned.

        Returns
        ----------
        list
            List of floats in the range of 1.0 to -1.0 representing the waveform.

        Raises
        ----------
        Exception
            Invalid input parameter type.
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        #Check input types.
        self._check_in_type(locals(), ['int', 'str', 'int', 'int'])

        #SCPI call
        try:
            var0 = self._read_bin('SEQuencer{}:ACQ:WLISt:WAVeform:DATA? "{}",{},{}'.format(sequencer, name, start, size))

            return struct.unpack('f'*int(len(var0)/4), var0)
        except Exception as err:
            self._check_error_queue(err)
        finally:
            self._check_error_queue()

    #---------------------------------------------------------------------------
    def _set_acq_waveform_index(self, sequencer, name, index):
        """
        Set waveform index of waveform in waveform list of indexed sequencer's acquisition path. The index is used by the sequencer Q1ASM program to refer to the waveform.
        If an invalid sequencer or waveform index is given or if a non-existing waveform name is given, an error is set in system error.

        Parameters
        ----------
        sequencer : int
            Sequencer index.
        name : str
            Waveform name.
        index : int
            Waveform index of the waveform in the waveform list.

        Returns
        ----------

        Raises
        ----------
        Exception
            Invalid input parameter type.
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        #Check input types.
        self._check_in_type(locals(), ['int', 'str', 'int'])

        #SCPI call
        try:
            self._write('SEQuencer{}:ACQ:WLISt:WAVeform:INDex "{}",{}'.format(sequencer, name, index))
        except Exception as err:
            self._check_error_queue(err)
        finally:
            self._check_error_queue()

    #---------------------------------------------------------------------------
    def _get_acq_waveform_index(self, sequencer, name):
        """
        Get waveform index of waveform in waveform list of indexed sequencer's acquisition path. The index is used by the sequencer Q1ASM program to refer to the waveform.
        If an invalid sequencer index is given or if a non-existing waveform name is given, an error is set in system error.

        Parameters
        ----------
        sequencer : int
            Sequencer index.
        name : str
            Waveform name.

        Returns
        ----------
        int
            Waveform index of the waveform in the waveform list.

        Raises
        ----------
        Exception
            Invalid input parameter type.
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        #Check input types.
        self._check_in_type(locals(), ['int', 'str'])

        #SCPI call
        try:
            var0 = self._read('SEQuencer{}:ACQ:WLISt:WAVeform:INDex? "{}"'.format(sequencer, name))

            return int(var0)
        except Exception as err:
            self._check_error_queue(err)
        finally:
            self._check_error_queue()

    #---------------------------------------------------------------------------
    def _get_acq_waveform_length(self, sequencer, name):
        """
        Get length of waveform in waveform list of indexed sequencer's acquisition path. If an invalid sequencer index is given or if a non-existing waveform name is given,
        an error is set in system error.

        Parameters
        ----------
        sequencer : int
            Sequencer index.
        name : str
            Waveform name.

        Returns
        ----------
        int
            Waveform length in number of samples.

        Raises
        ----------
        Exception
            Invalid input parameter type.
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        #Check input types.
        self._check_in_type(locals(), ['int', 'str'])

        #SCPI call
        try:
            var0 = self._read('SEQuencer{}:ACQ:WLISt:WAVeform:LENGth? "{}"'.format(sequencer, name))

            return int(var0)
        except Exception as err:
            self._check_error_queue(err)
        finally:
            self._check_error_queue()

    #---------------------------------------------------------------------------
    def _get_acq_waveform_name(self, sequencer, index):
        """
        Get name of waveform in waveform list of indexed sequencer's acquisition path based on its waveform index. If an invalid sequencer index is given or if a non-existing
        waveform index is given, an error is set in system error.

        Parameters
        ----------
        sequencer : int
            Sequencer index.
        index : str
            Waveform index.

        Returns
        ----------
        str
            Waveform name.

        Raises
        ----------
        Exception
            Invalid input parameter type.
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        #Check input types.
        self._check_in_type(locals(), ['int', 'int'])

        #SCPI call
        try:
            var0 = self._read('SEQuencer{}:ACQ:WLISt:WAVeform:NAME? {}'.format(sequencer, index))

            return var0
        except Exception as err:
            self._check_error_queue(err)
        finally:
            self._check_error_queue()

    #---------------------------------------------------------------------------
    def _get_acq_num_waveforms(self, sequencer):
        """
        Get number of waveforms in waveform list of indexed sequencer's acquisition path. If an invalid sequencer index is given, an error is set in system error.

        Parameters
        ----------
        sequencer : int
            Sequencer index.

        Returns
        ----------
        int
            Number of waveforms.

        Raises
        ----------
        Exception
            Invalid input parameter type.
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        #Check input types.
        self._check_in_type(locals(), ['int'])

        #SCPI call
        try:
            var0 = self._read('SEQuencer{}:ACQ:WLISt:SIZE?'.format(sequencer))

            return int(var0)
        except Exception as err:
            self._check_error_queue(err)
        finally:
            self._check_error_queue()

    #---------------------------------------------------------------------------
    def _get_acq_waveform_names(self, sequencer):
        """
        Get all waveform names in waveform list of indexed sequencer's acquisition path. If an invalid sequencer index is given, an error is set in system error.

        Parameters
        ----------
        sequencer : int
            Sequencer index.

        Returns
        ----------
        int
            Number of waveforms.

        Raises
        ----------
        Exception
            Invalid input parameter type.
        Exception
            An error is reported in system error and debug <= 1.
            All errors are read from system error and listed in the exception.
        """

        #Check input types.
        self._check_in_type(locals(), ['int'])

        #SCPI call
        try:
            var0 = self._read('SEQuencer{}:ACQ:WLISt?'.format(sequencer))

            return var0
        except Exception as err:
            self._check_error_queue(err)
        finally:
            self._check_error_queue()

    #---------------------------------------------------------------------------
    def _add_acq_acquisition(self, sequencer, path, name, size):
        """
        Add acquisition to sequencer acquisition list.
        """

        #Check input types.
        self._check_in_type(locals(), ['int', 'int', 'str', 'int'])

        #SCPI call
        try:
            self._write('SEQuencer{}:ACQ:PATH{}:ALISt:ACQuisition:NEW "{}",{}'.format(sequencer, path, name, size))
        except Exception as err:
            self._check_error_queue(err)
        finally:
            self._check_error_queue()

    #---------------------------------------------------------------------------
    def _delete_acq_acquisition(self, sequencer, path, name):
        """
        Delete acquisition from sequencer acquisition list.
        """

        #Check input types.
        self._check_in_type(locals(), ['int', 'int', 'str'])

        #SCPI call
        try:
            self._write('SEQuencer{}:ACQ:PATH{}:ALISt:ACQuisition:DELete "{}"'.format(sequencer, path, name))
        except Exception as err:
            self._check_error_queue(err)
        finally:
            self._check_error_queue()

    #---------------------------------------------------------------------------
    def _set_acq_acquisition_data(self, sequencer, path, name):
        """
        Set acquisition data in sequencer acquisition list from acquisition path.
        """

        #Check input types.
        self._check_in_type(locals(), ['int', 'int', 'str'])

        #SCPI call
        try:
            self._write('SEQuencer{}:ACQ:PATH{}:ALISt:ACQuisition:DATA "{}"'.format(sequencer, path, name))
        except Exception as err:
            self._check_error_queue(err)
        finally:
            self._check_error_queue()

    #---------------------------------------------------------------------------
    def _get_acq_acquisition_data(self, sequencer, path, name, start, size):
        """
        Get acquisition data from sequencer acquisition list.
        """

        #Check input types.
        self._check_in_type(locals(), ['int', 'int', 'str', 'int', 'int'])

        #SCPI call
        try:
            var0 = self._read_bin('SEQuencer{}:ACQ:PATH{}:ALISt:ACQuisition:DATA? "{}",{},{}'.format(sequencer, path, name, start, size))

            return struct.unpack('f'*int(len(var0)/4), var0)
        except Exception as err:
            self._check_error_queue(err)
        finally:
            self._check_error_queue()

    #---------------------------------------------------------------------------
    def _get_acq_acquisition_length(self, sequencer, path, name):
        """
        Get acquisition length from sequencer acquisition list.
        """

        #Check input types.
        self._check_in_type(locals(), ['int', 'int', 'str'])

        #SCPI call
        try:
            var0 = self._read('SEQuencer{}:ACQ:PATH{}:ALISt:ACQuisition:LENGth? "{}"'.format(sequencer, path, name))

            return int(var0)
        except Exception as err:
            self._check_error_queue(err)
        finally:
            self._check_error_queue()

    #---------------------------------------------------------------------------
    def _get_acq_num_acquisitions(self, sequencer, path):
        """
        Get number of acquisition in sequencer acquisition list.
        """

        #Check input types.
        self._check_in_type(locals(), ['int', 'int'])

        #SCPI call
        try:
            var0 = self._read('SEQuencer{}:ACQ:PATH{}:ALISt:SIZE?'.format(sequencer, path))

            return int(var0)
        except Exception as err:
            self._check_error_queue(err)
        finally:
            self._check_error_queue()

    #---------------------------------------------------------------------------
    def _get_acq_acquisition_names(self, sequencer, path):
        """
        Get all acquisition names from sequencer acquisition list.
        """

        #Check input types.
        self._check_in_type(locals(), ['int', 'int'])

        #SCPI call
        try:
            var0 = self._read('SEQuencer{}:ACQ:PATH{}:ALISt?'.format(sequencer, path))

            return var0
        except Exception as err:
            self._check_error_queue(err)
        finally:
            self._check_error_queue()

    #-----------------------------------------------------------------------
    def _check_in_type(self, in_arg_dict, in_type_list):
        """
        Checks input argument types against reference types.

        Parameters
        ----------
        in_arg_dict : dict
            Dictionary with input arguments created by locals().
        in_type_list : list
            List of reference input argument types.

        Returns
        ----------

        Raises
        ----------
        Exception
            Input argument type mismatch.
        """

        if self._debug <= 1:
            del in_arg_dict['self']
            in_val_list  = [in_arg_dict[name] for name in in_arg_dict]
            for i, (in_val, in_type) in enumerate(zip(in_val_list, in_type_list)):
                if str(type(in_val).__name__) == "list" or str(type(in_val).__name__) == "ndarray":
                    if len(in_val) > 0:
                        in_val = in_val[0]
                    else:
                        raise Exception("Error: unexpected type for input argument {}, expected {} but got empty {}.".format(i, in_type, str(type(in_val).__name__)))
                if str(type(in_val).__name__)[:len(in_type)] != in_type:
                    raise Exception("Error: unexpected type for input argument {}, expected {} but got {}.".format(i, in_type, str(type(in_val).__name__)))

    #---------------------------------------------------------------------------
    def _check_error_queue(self, err=None):
        """
        Check system error for errors. Empties and prints the complete error queue.

        Parameters
        ----------
        err : Exception
            Exception to reraise.

        Returns
        ----------

        Raises
        ----------
        Exception
            An error was found in system error or an exception was passed as input argument.
        """

        if self._debug <= 1:
            errors = [str(err)] if err is not None else []
            while int(self._read('SYSTem:ERRor:COUNt?')) != 0:
                errors.append(','.join(self._read('SYSTem:ERRor:NEXT?').split(',')[1:]))

            if len(errors):
                if err is not None:
                    err_type = type(err)
                else:
                    err_type = Exception
                raise err_type('\n'.join(errors)).with_traceback(sys.exc_info()[2]) from None
