import os
import sys
import logging
import pathlib
from threading import Thread
import socket
import time
import argparse
from typing import Tuple, List
from subprocess import Popen

log_level = logging.DEBUG

log_format = logging.Formatter('[%(asctime)s] [%(levelname)s] - %(message)s')
logger = logging.getLogger(__name__)
logger.setLevel(log_level)

# writing to stdout
handler = logging.StreamHandler(sys.stdout)
handler.setLevel(log_level)
handler.setFormatter(log_format)
logger.addHandler(handler)

CFG_NAMEFILE = r'LabRecorder.cfg'
CFG_AUTOGEN_NAMEFILE = r'LabRecorder_autogenerated.cfg'
DEFAULT_PATH_EXE = r'./LabRecorder/LabRecorder.exe'
ALT_PATH_EXE = r'~/LabRecorder/LabRecorder.exe'


class LabRecorderHandler:
    """
    Doc can be found here https://github.com/labstreaminglayer/App-LabRecorder
    """
    BRAINAMP_STREAMS = [('BrainAmpSeries-Dev_1', None),
                        ('BrainAmpSeries-Dev_1-Markers', None)]

    LIVEAMP_STREAMS = [('LiveAmpSN-100709-0788', None),
                       ('LiveAmpSN-100709-0788-DeviceTrigger', None)]


    def __init__(self, path_to_labrecorder_exe: str = None, labrecorder_cfg: str = None,
                 list_required_streams: List[Tuple[str]] = None):


        self.process = None  # keep the pointer to keep the process alive

        # Attempt alternative paths just in case
        labrecorder_exe = path_to_labrecorder_exe if path_to_labrecorder_exe is not None else DEFAULT_PATH_EXE
        if not pathlib.Path(labrecorder_exe).exists():
            labrecorder_exe = ALT_PATH_EXE if os.path.exists(ALT_PATH_EXE) else labrecorder_exe

        self.cmd = labrecorder_exe

        # configuration file is provided
        if labrecorder_cfg is not None:
            logger.info(f'Using configuration file provided {labrecorder_cfg}')
            if list_required_streams is not None:
                logger.warning('Ignoring the required streams due to provided config file')
            if pathlib.Path(labrecorder_cfg).exists():
                self.path_cfg = labrecorder_cfg
            else:
                raise FileNotFoundError(pathlib.Path(labrecorder_cfg).absolute())

        # generating configuration from template
        else:
            path_cfg_source = (pathlib.Path(labrecorder_exe).parent).joinpath(CFG_NAMEFILE)
            self.path_cfg = pathlib.Path(CFG_AUTOGEN_NAMEFILE)

            logger.info(f'Autogenerating config file from {path_cfg_source} to {self.path_cfg}')
            self.generate_cfg(template_cfg_path=path_cfg_source,
                              output_cfg_path=self.path_cfg,
                              list_params=list_required_streams,
                              current_host=socket.gethostname())



    def generate_cfg(self, template_cfg_path: str,
                     output_cfg_path: str,
                     list_params: List[Tuple[str]],
                     current_host: str):

        list_gen = []

        for stream_str, host_str in list_params:
            hostname = host_str if host_str is not None and len(host_str) > 0 else current_host

            list_gen.append(f'"{stream_str} ({hostname})"')

        required_streams_str = ' ,'.join(list_gen)

        try:
            with open(os.path.abspath(output_cfg_path), 'w') as fi_write:
                with open(os.path.abspath(template_cfg_path), 'r') as fi_read:
                    for line in fi_read.readlines():
                        if line.find('RequiredStreams=') >= 0:
                            fi_write.write(f'RequiredStreams={required_streams_str}')
                        else:
                            fi_write.write(line)
        except IOError:
            logger.error('Could not generate the optional configuration file')
            pass

    def start_lab_recorder(self):

        # define the path to the executable
        cmd_str = self.cmd
        if not os.path.exists(cmd_str):
            logger.critical(f'Could not find {cmd_str}')
            raise FileNotFoundError(f'Could not find {cmd_str}')

        if self.path_cfg:
            cmd_pipe = [cmd_str, '-c', str(self.path_cfg)]
        else:
            cmd_pipe = [cmd_str]
        logger.info(f'Pipe opening with popen command: {cmd_pipe}')

        self.process = Popen(cmd_pipe)  #, stdout=PIPE, stderr=PIPE)

    def start_recording(self, output_eeg_path: str, force_restart: bool = False):
        eeg_fullpath = os.path.abspath(output_eeg_path)
        self._set_recording(record=True, force_restart=force_restart, path_eeg=eeg_fullpath)
        if eeg_fullpath is None or not isinstance(eeg_fullpath, str):
            logger.warning('Please define an EEG output file')
            pass

    def stop_recording(self):
        self._set_recording(record=False)

    def _set_recording(self, record, force_restart: bool = False, path_eeg: str = None):

        # start recording
        if record:

            # restart lab recorder to force it to refresh the list of streams
            if force_restart:
                self.restart_lab_recorder()
            logger.info('recording to {}'.format(path_eeg))
            folder, fn = os.path.split(path_eeg)
            msg_to_send = ["update",
                           'start'
                           ]

            # only insert the target filename if the filename is valid
            if path_eeg is not None and isinstance(path_eeg, str):
                msg_to_send.insert(1, 'filename {root:%s} {template:%s}' % (folder, fn),
)
            success = self.send_msg_labRecorder(msg_to_send)
            self.is_recording = True
            return success
        # Stop recording
        else:
            logger.debug('Stop recording with current acquisition state is {}'.format(self.is_recording))
            if self.is_recording:
                # tell LabRecorder to stop recording
                success = self.send_msg_labRecorder(['stop'])
                # update the recording state
                self.is_recording = self.is_recording and (not success)
            logger.debug('Recording stopped')

            return not self.is_recording

    def send_msg_labRecorder(self, list_msg):
        s = socket.create_connection(("localhost", 22345))
        success = False
        for msg in list_msg:
            msg = msg + '\n'
            bytes_msg = msg.encode('ascii')
            s.sendall(bytes_msg)
            logger.debug('Sent {} through socket'.format(bytes_msg))
            data = s.recv(1024)
            reply = data.decode('ascii')

            logger.debug('Socket received' + reply)
            success = True  # todo: fix success

        return success

    def update_lsl(self):
        """awaiting for changes in labrecorder to activate"""
        lrh.send_msg_labRecorder(['update'])

    def restart_lab_recorder(self):
        logger.debug('Killing lab recorder process and starting it again')
        self.process.terminate()
        time.sleep(2)
        self.start_lab_recorder()


    def close(self):
        if self.process is not None:
            self.process.kill()


if __name__ == '__main__':
    parser = argparse.ArgumentParser(description='Start an interface to configure and control labrecorder')
    parser.add_argument('labrecorder', type=str, nargs='?',
                        help='path to LabRecorder folder')
    parser.add_argument('configfile', type=str, nargs='?',
                        help='optional configuration file for labrecorder')

    # Extract command line parameters
    args = parser.parse_args()
    path_to_labrecorder = args.labrecorder if args.labrecorder is not None else None
    config_labrecorder = args.configfile if args.configfile is not None else None


    # Define the stream names and hosts by hand, using none default to the computer's hostname
    lrh = LabRecorderHandler(path_to_labrecorder_exe=r'./LabRecorder/LabRecorder.exe',
                             list_required_streams=[('LiveAmpSN-100709-0788', None),
                                                    ('LiveAmpSN-100709-0788-DeviceTrigger', 'localhost')])

    # Use a custom configuration file
    lrh = LabRecorderHandler(path_to_labrecorder_exe=r'./LabRecorder/LabRecorder.exe',
                             labrecorder_cfg=r'./LabRecorder/LabRecorder.cfg')

    # Call labrecorder with default BrainAmpSeries parameters
    lrh = LabRecorderHandler(path_to_labrecorder_exe=DEFAULT_PATH_EXE,
                             list_required_streams=LabRecorderHandler.BRAINAMP_STREAMS)

    lrh.start_lab_recorder()

    # Start recording for 10 seconds
    lrh.start_recording(output_eeg_path=r'./test.eeg',  # change the output filename
                        force_restart=True)  # force to restart labrecorder to update streams
    time.sleep(10)

    lrh.update_lsl()

    time.sleep(10)

    # stop recording
    lrh.stop_recording()
    # close labrecorder (kills the process)
    lrh.close()
