import django
from django.conf import settings
from django.core.exceptions import FieldError, ValidationError
from django.forms.fields import CharField
from django.forms.formsets import formset_factory
from django.forms.models import (ModelForm, BaseModelForm, ModelFormMetaclass,
    fields_for_model, model_to_dict, construct_instance, BaseInlineFormSet, BaseModelFormSet,
    modelform_factory, inlineformset_factory)
if django.VERSION >= (1, 7):
    from django.forms.utils import ErrorList
else:
    from django.forms.util import ErrorList
from django.forms.widgets import Select
from django.utils.translation import get_language, ugettext as _
from hvad.compat.metaclasses import with_metaclass
from hvad.models import TranslatableModel, BaseTranslationModel
from hvad.utils import get_cached_translation, get_translation, combine
try:
    from collections import OrderedDict
except ImportError: #pragma: no cover (python < 2.7)
    from django.utils.datastructures import SortedDict as OrderedDict
import warnings

#=============================================================================

class TranslatableModelFormMetaclass(ModelFormMetaclass):
    ''' Metaclass used for forms with translatable fields.
        It wraps the regular ModelFormMetaclass to intercept translatable fields,
        otherwise it would choke. Translatable fields are then inserted back
        into the created class.
    '''
    def __new__(cls, name, bases, attrs):
        # Force presence of meta class, we need it
        meta = attrs.get('Meta')
        if meta is None:
            meta = attrs['Meta'] = type('Meta', (object,), {})

        model = getattr(meta, 'model', None)
        fields = getattr(meta, 'fields', None)

        # Force exclusion of language_code as we use cleaned_data['language_code']
        exclude = meta.exclude = list(getattr(meta, 'exclude', ()))
        if 'language_code' not in exclude:
            exclude.append('language_code')
        if fields is not None and 'language_code' in fields:
            raise FieldError('Field \'language_code\' is invalid.')

        # If a model is provided, handle translatable fields
        if model:
            if not issubclass(model, TranslatableModel):
                raise TypeError('TranslatableModelForm only works with TranslatableModel'
                                ' subclasses, which %s is not.' % model.__name__)

            # Additional exclusions
            exclude.append(model._meta.translations_accessor)
            if fields is not None and model._meta.translations_accessor in fields:
                raise FieldError('Field \'%s\' is invalid', model._meta.translations_accessor)

            # Get translatable fields
            tfields = fields_for_model(
                model._meta.translations_model,
                fields=fields,
                exclude=exclude + ['master'], # only exclude master here, it is valid shared field
                widgets=getattr(meta, 'widgets', None),
                formfield_callback=attrs.get('formfield_callback')
            )

            # Drop translatable fields from Meta.fields
            if fields is not None:
                meta.fields = [field for field in fields if tfields.get(field) is None]

        # Create the form class
        new_class = super(TranslatableModelFormMetaclass, cls).__new__(cls, name, bases, attrs)

        # Add translated fields into the form's base fields
        if model:
            if fields is None:
                # loop, as Django's variant of OrderedDict cannot consume generators
                for name, field in tfields.items():
                    if field is not None:
                        new_class.base_fields[name] = field
            else:
                # rebuild the fields to respect Meta.fields ordering
                new_class.base_fields = OrderedDict(
                    item for item in (
                        (name, new_class.base_fields.get(name, tfields.get(name)))
                        for name in fields
                    )
                    if item[1] is not None
                )
                # restore hijacked Meta.fields
                new_class._meta.fields = meta.fields = fields
        return new_class


#=============================================================================

class BaseTranslatableModelForm(BaseModelForm):
    ''' Base class for forms dealing with TranslatableModel
        It has two main responsibilities: loading translated fields into the form
        when __init__ialized and ensuring the right translation gets saved.
    '''
    def __init__(self, data=None, files=None, auto_id='id_%s', prefix=None,
                 initial=None, error_class=ErrorList, label_suffix=':',
                 empty_permitted=False, instance=None):

        # Insert values of instance's translated fields into 'initial' dict
        object_data = {}
        enforce = hasattr(self, 'language')
        language = getattr(self, 'language', None) or get_language()

        if instance is not None:
            translation = get_cached_translation(instance)
            if translation is None or (enforce and translation.language_code != language):
                try:
                    translation = get_translation(instance, language)
                except instance._meta.translations_model.DoesNotExist:
                    pass
            if translation is not None:
                object_data.update(
                    model_to_dict(translation, self._meta.fields, self._meta.exclude)
                )
        if initial is not None:
            object_data.update(initial)

        super(BaseTranslatableModelForm, self).__init__(
            data, files, auto_id, prefix, object_data,
            error_class, label_suffix, empty_permitted, instance
        )

    def clean(self):
        ''' If a language is set on the form, enforce it by overwriting it
            in the cleaned_data.
        '''
        data = super(BaseTranslatableModelForm, self).clean()
        if hasattr(self, 'language'):
            data['language_code'] = self.language
        return data

    def _post_clean(self):
        ''' Switch the translation on self.instance
            This cannot (and should not) be done in clean() because it could be
            overriden to change the language. Yet it should be done before save()
            to allow an overriden save to set some translated field values before
            invoking super().
        '''
        result = super(BaseTranslatableModelForm, self)._post_clean()

        enforce = 'language_code' in self.cleaned_data
        language = self.cleaned_data.get('language_code') or get_language()
        translation = self._get_translation(self.instance, language, enforce)
        translation.master = self.instance
        self.instance = combine(translation, self.instance.__class__)
        return result

    def save(self, commit=True):
        ''' Saves the model
            If will always use the language specified in self.cleaned_data, with
            the usual None meaning 'call get_language()'. If instance has
            another language loaded, it gets reloaded with the new language.

            If no language is specified in self.cleaned_data, assume the instance
            is preloaded with correct language.
        '''
        if not self.is_valid():
            warnings.warn('Calling save() on an invalid form is deprecated and '
                          'will fail in the future. Check the result of .is_valid() '
                          'before calling save().', DeprecationWarning, stacklevel=2)
            raise ValueError((
                _("The %s could not be created because the data didn't validate.")
                if self.instance.pk is None else
                _("The %s could not be changed because the data didn't validate.")
                ) % self.instance._meta.object_name
            )

        # Get the right translation for object and language
        # It should have been done in _post_clean, but instance may have been
        # changed since.
        enforce = 'language_code' in self.cleaned_data
        language = self.cleaned_data.get('language_code') or get_language()
        translation = self._get_translation(self.instance, language, enforce)

        # Fill the translated fields with values from the form
        excludes = list(self._meta.exclude) + ['master', 'language_code']
        translation = construct_instance(self, translation,
                                         self._meta.fields, excludes)
        translation.master = self.instance
        self.instance = combine(translation, self.instance.__class__)

        # Delegate shared fields to super()
        return super(BaseTranslatableModelForm, self).save(commit=commit)

    def _get_translation(self, instance, language, enforce):
        ''' Get a translation for the instance.
            Depending on enforce argument, the language will serve as a default
            or will be enforced by reloading a mismatching translation.

            If instance's active translation is in given language, this is
            a guaranteed no-op: it will be returned as is.
        '''
        trans_model = instance._meta.translations_model
        translation = get_cached_translation(instance)

        if translation is None or (enforce and translation.language_code != language):
            if instance.pk is None:
                translation = trans_model()
                translation.language_code = language
            else:
                try:
                    translation = get_translation(self.instance, language)
                except trans_model.DoesNotExist:
                    translation = trans_model()
                    translation.language_code = language
        return translation


if django.VERSION >= (1, 7):
    class TranslatableModelForm(with_metaclass(TranslatableModelFormMetaclass,
                                               BaseTranslatableModelForm)):
        pass
else:
    # Older django version have buggy metaclass
    class TranslatableModelForm(with_metaclass(TranslatableModelFormMetaclass,
                                               BaseTranslatableModelForm, ModelForm)):
        __metaclass__ = TranslatableModelFormMetaclass # Django 1.4 compatibility


#=============================================================================

def translatable_modelform_factory(language, model, form=TranslatableModelForm, *args, **kwargs):
    klass = modelform_factory(model, form, *args, **kwargs)
    klass.language = language
    return klass


def translatable_modelformset_factory(language, model, form=TranslatableModelForm,
                                      formfield_callback=None, formset=BaseModelFormSet,
                                      extra=1, can_delete=False, can_order=False,
                                      max_num=None, fields=None, exclude=None, **kwargs):

    # This Django API changes often, handle args we know and raise for others
    form_kwargs, formset_kwargs = {}, {}
    for key in ('widgets', 'localized_fields', 'labels', 'help_texts', 'error_messages'):
        if key in kwargs:
            form_kwargs[key] = kwargs.pop(key)
    for key in ('validate_max',):
        if key in kwargs:
            formset_kwargs[key] = kwargs.pop(key)
    if kwargs:
        raise TypeError('Unknown arguments %r for translatable_modelformset_factory. '
                        'If it is legit, it is probably new in Django. Please open '
                        'a ticket so we can add it.' % tuple(kwargs.keys()))

    form = translatable_modelform_factory(
        language, model, form=form, fields=fields, exclude=exclude,
        formfield_callback=formfield_callback, **form_kwargs
    )
    FormSet = formset_factory(form, formset, extra=extra, max_num=max_num,
                              can_order=can_order, can_delete=can_delete, **formset_kwargs)
    FormSet.model = model
    return FormSet


def translatable_inlineformset_factory(language, parent_model, model, form=TranslatableModelForm,
                                       formset=BaseInlineFormSet, fk_name=None,
                                       fields=None, exclude=None, extra=3,
                                       can_order=False, can_delete=True,
                                       max_num=None, formfield_callback=None, **kwargs):
    from django.forms.models import _get_foreign_key
    fk = _get_foreign_key(parent_model, model, fk_name=fk_name)
    if fk.unique:  #pragma: no cover (internal Django behavior)
        max_num = 1

    FormSet = translatable_modelformset_factory(language, model,
         form=form, formfield_callback=formfield_callback, formset=formset,
         extra=extra, can_delete=can_delete, can_order=can_order,
         fields=fields, exclude=exclude, max_num=max_num, **kwargs)
    FormSet.fk = fk
    return FormSet


#=============================================================================

class BaseTranslationFormSet(BaseInlineFormSet):
    """A kind of inline formset for working with an instance's translations.
    It keeps track of the real object and combine()s it to the translations
    for validation and saving purposes.
    It can delete translations, but will refuse to delete the last one.
    """
    def __init__(self, *args, **kwargs):
        super(BaseTranslationFormSet, self).__init__(*args, **kwargs)
        self.queryset = self.order_translations(self.queryset)

    def order_translations(self, qs):
        return qs.order_by('language_code')

    def clean(self):
        super(BaseTranslationFormSet, self).clean()

        # Trigger combined instance validation
        master = self.instance
        stashed = get_cached_translation(master)

        for form in self.forms:
            form.instance.master = master
            combined = combine(form.instance, master.__class__)
            exclusions = form._get_validation_exclusions()
            # fields from the shared model should not be validated
            exclusions.extend(f.name for f in combined._meta.fields)
            try:
                if django.VERSION >= (1, 6):
                    combined.full_clean(exclude=exclusions,
                                        validate_unique=form._validate_unique)
                else:
                    combined.full_clean(exclude=exclusions)
            except ValidationError as e:
                form._update_errors(e)

        if stashed is None:
            delattr(master, master._meta.translations_cache)
        else:
            setattr(master, master._meta.translations_cache, stashed)

        # Validate that at least one translation exists
        forms_to_delete = self.deleted_forms
        provided = [form for form in self.forms
                    if (getattr(form.instance, 'pk', None) is not None or
                        form.has_changed())
                       and not form in forms_to_delete]
        if len(provided) < 1:
            raise ValidationError(_('At least one translation must be provided'),
                                  code='notranslation')

    def _save_translation(self, form, commit=True):
        obj = form.save(commit=False)
        assert isinstance(obj, BaseTranslationModel)

        if commit:
            # We need to trigger custom save actions on the combined model
            master = self.instance
            stashed = get_cached_translation(master)
            obj.master = master
            combined = combine(obj, master.__class__)
            combined.save()
            if hasattr(combined, 'save_m2m'): # pragma: no cover
                # cannot happen, but feature could be added, be ready
                combined.save_m2m()
            if stashed is None:
                delattr(master, master._meta.translations_cache)
            else:
                setattr(master, master._meta.translations_cache, stashed)
        return obj

    def save_new(self, form, commit=True):
        return self._save_translation(form, commit)

    def save_existing(self, form, instance, commit=True):
        return self._save_translation(form, commit)

    def add_fields(self, form, index):
        super(BaseTranslationFormSet, self).add_fields(form, index)
        # Add the language code automagically
        if not 'language_code' in form.fields:
            form.fields['language_code'] = CharField(
                required=True, initial=form.instance.language_code,
                widget=Select(choices=(('', '--'),)+settings.LANGUAGES)
            )
            # Add language_code to self._meta.fields so it is included in validation stage
            try:
                form._meta.fields.append('language_code')
            except AttributeError: #pragma: no cover
                form._meta.fields += ('language_code',)

        # Remove the master foreignkey, we have this from self.instance already
        if 'master' in form.fields:
            del form.fields['master']

def translationformset_factory(model, **kwargs):
    """ Works as a regular inlineformset_factory except for:
    - it is set up to work on the given model's translations table
    - it uses a BaseTranslationFormSet to handle combine() and language_code

    Basic use: MyModelTranslationsFormSet = translationformset_factory(MyModel)
    """
    defaults = {
        'formset': BaseTranslationFormSet,
        'fk_name': 'master',
    }
    defaults.update(kwargs)
    return inlineformset_factory(model, model._meta.translations_model, **defaults)

