#!python
#-*- coding: utf-8 -*-
"""
Sample `xlwings` script
#######################
 
The functions included provide for running a batch of experiments in an excel-table 
with json-pointer paths as headers (see accompanying :file:`.xlsm`).

You can debug it by running it directly as a python script, as suggested by : 
  http://docs.xlwings.org/debugging.html

<< EDIT THIS SCRIPTTO PUT YOUR EXCEL/XLWINGS PYTHON-CODE, BELOW >>
"""



from __future__ import division, print_function, unicode_literals

import logging
import operator
import os
import re

from fuefit import model
from fuefit import processor
import pandas as pd
import xlwings as xw


def _init_logging(loglevel):
    logging.basicConfig(level=loglevel)
    logging.getLogger().setLevel(level=loglevel)

_init_logging(logging.INFO)
log = logging.getLogger(__name__)
 
## TODO: Convert Excel-ref RC-notation to A1
_excel_ref_specifier_regex = re.compile(r'''^\s*
            @
            (?:(?P<sheet>.+)!)?             # Sheet-name optional-group
            (?P<ref>                        # start Cell-ref 
                (?:R\d+C\d+ | [A-Z]+\d+)        # FROM-ref, RC or A1 notation
                (?:                             # start TO-Cell-ref optional-group
                    :
                    (?:R\d+C\d+ | [A-Z]{1,2}\d+)        # RC or A1 notation
                )?                              # end TO-Cell-ref optional-group
            )                               # end Cell-ref
            (?:                             # start Shape-specifier optional-group
                \.
                (?P<shape>table|row|column)     # One of xw.Range.[table|row|column] attributes
            )?                              # end Shape-specifier
            (?:\((?P<range_kws>                   # start RANGE-kws expression
                [^)]*
            )\))?                            # end RANGE-kws
            (?:{(?P<pandas_kws>                   # start PANDAS-kws expression
                [^)]*
            )})?                            # end PANDAS-kws
            \s*$''', re.X+re.IGNORECASE)
_undefined = object()
def _try_matching_excel_ref(ref_str, default=_undefined):
    """
    if `ref_str` it is it an excel-ref, it returns the refed-contents as DataFrame or a scalar.
    
    Excel-ref syntax is case-insensitive::
     
            @<sheet_name>!A1[:R10:c1][.table][(<python code for xlwings.Range kws>)][{<python code for Pandas-ctor>}]
            
    Note that the3 "RC-notation" is not converted, so Excel may not support it.
    Excel-ref examples::
    
            @a1
            @e5.column
            @some sheet name!R1C5.TABLE
            @some sheet name!A1:c5.table(header=False)
            @some sheet name!A1:c5.column(strict=True; atleast_2d=True)
            @some sheet name!A1.table(asarray=True){columns=['a','b']}
            @some sheet name!A1.table(header=True)              ## NOTE: Setting Range's `header` kw and 
                                                                #      DataFrame will parse 1st row as header
    
    """
    
    def _parse_kws(kws_str):
        kws = {}
        if kws_str:
            exec(kws_str, None, kws)
        return kws
    
    matcher = _excel_ref_specifier_regex.match(ref_str)
    if matcher:
        ref = matcher.groupdict()
        log.info("Parsed string(%s) as Excel-ref: %s", ref_str, ref)
        
        range_kws = _parse_kws(ref.get('range_kws'))
        ref_range = xw.Range(ref.get('sheet'), ref['ref'], **range_kws)
        range_shape = ref.get('shape')
        if range_shape:
            ref_range = operator.attrgetter(range_shape.lower())(ref_range)

        v = ref_range.value
        
        if ref_range.row1 != ref_range.row2 or ref_range.col1 != ref_range.col2:
            ## Parse as DataFrame when more than one cell.
            #
            pandas_kws = _parse_kws(ref.get('pandas_kws'))
            if 'header' in range_kws and not 'columns' in pandas_kws :
                ##Do not ignore explicit-columns.
                v = pd.DataFrame(v[1:], columns=v[0], **pandas_kws)
            else:
                v = pd.DataFrame(ref_range.value, **pandas_kws)
        else:
            v = ref_range.value
            
        log.debug("Excel-ref(%s) value: %s", ref, v)

        return v
    else:
        if default is _undefined:
            raise ValueError("Invalid excel-ref(%s)!" % ref_str)
        else:
            return default 
    
    
def read_input_as_df(table_ref_str):
    """
    Expects excel-table with jsonpointer paths as Header and 1st column named `id` as index, 
    like this::
    
        id     vehicle/test_mass  vehicle/p_rated           vehicle/gear_ratios 
        veh_1               1500              100  [120.75, 75, 50, 43, 37, 32] 
        veh_2               1600               80  [120.75, 75, 50, 43, 37, 32] 
        veh_3               1200               60  [120.75, 75, 50, 43, 37, 32] 
        
    """
    vehs = xw.Range('Input', table_ref_str).table.value
    vehs = pd.DataFrame(vehs[1:], columns=vehs[0]).set_index('id')
    
    return vehs

def add_results_as_sheet(veh_id, mdl_out):
    sheet = "results.%s"%veh_id
    try:
        xw.Sheet.add(sheet)
    except Exception as ex:
        log.info("Sheet(%s) already exists(%s).", sheet, ex)
        xw.Sheet(sheet).clear()
    xw.Range(sheet, 'A1', index=False).value = model.resolve_jsonpointer(mdl_out, '/engine/fc_map_params').to_frame().T
    xw.Range(sheet, 'A4', index=False).value = mdl_out['engine_map']
    
    

def build_models(vehs_df):
    """
    Builds all input-dataframes as Experiment classes and returns them in a list of (veh_id, exp) pairs.
    
    :param vehs_df:     A dataframe indexed by veh_id, and with columns *json-pointer* paths into the model
    :return: a list of (veh_id, model) tuples
    """
    experiment_pairs = []
    for veh_id, row in vehs_df.iterrows():
        try:
            mdl_in = model.base_model()
            for k, v in row.items():
                log.debug('veh_id(%s): Column(%s): %s', veh_id, k, v)
                if not v:
                    continue
                if isinstance(v, str):
                    
                    ## Is it an excel-ref like: 
                    #        @<sheet_name>!A1[:R10:c1].table
                    #
                    try:
                        v = _try_matching_excel_ref(v)
                    except ValueError:
                        ## Try to parse value as python-code.
                        #
                        try:
                            old_v = v
                            v = eval(v)
                        except Exception:
                            pass
                        else:
                            log.info("Parsed value(%s) as python code into: %s", old_v, v)
                model.set_jsonpointer(mdl_in, k, v)
            
            mdl_in = model.validate_model(mdl_in)
            experiment_pairs.append((veh_id, mdl_in))
        except Exception as ex:
            raise Exception('Invalid model for vehicle(%s): %s' % (veh_id, ex)) from ex 
        
    return experiment_pairs


def run_experiments(experiment_pairs):
    """
    Iterates `veh_df` and runs experiment_pairs.
    
    :param vehs_df:     A dataframe indexed by veh_id, and with columns *json-pointer* paths into the model
    """
    for veh_id, mdl_in in experiment_pairs:
        try:
            mdl_out = processor.run(mdl_in)
            
            add_results_as_sheet(veh_id, mdl_out)
        except Exception as ex:
            raise Exception('Experiment failed for vehicle(%s): %s' % (veh_id, ex)) from ex 
        
if __name__ == '__main__':
    ## Open and run experiments
    #
    log.info('CWD: %s', os.getcwd())
    excel_fname = os.path.join(os.path.dirname(__file__), '%s.xlsm' % os.path.splitext(os.path.basename(__file__))[0])
    wb_path = os.path.abspath(excel_fname)
    xw.Workbook(wb_path)
    
    vehs_df = read_input_as_df('D2')
    log.info('%s', vehs_df)
    exp_pairs = build_models(vehs_df)
    run_experiments(exp_pairs)
    
    