"""
Copyright 2018 Cognitive Scale, Inc. All Rights Reserved.

Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

   http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.
"""

from typing import Dict
from .camel import Document
from .env import CortexEnv


class Message(Document):
    """
    A message has the following params:
    - instanceId: the ID of the Agent instance that produced this message
    - typeName: the resource name of the type/schema contained in the payload of this message
    - token: the JWT token to be used for making authenticated requests to APIs needed to process this message
    - sessionId: The ID of the session associated with this Message
    - channelId: The ID of the channel (wire) this message was sent on
    - apiEndpoint: The URI of the API server to use when making requests to platform services in processing of this message
    - properties: an object who's keys represent meta-data properties set on this message
    - payload: the body of the message following one of the three supported payload formats (see the CAMEL spec for details)
    """

    def __init__(self, params:Dict = None):
        if params is None:
            params = {}

        super().__init__(params, False)
        self._params = params

    def to_params(self) -> Dict:
        return self._params

    def __setattr__(self, key, value):
        super().__setattr__(key, value)

        if not key.startswith('_'):
            self._params[key] = value

    def as_pandas(self, client=None):
        if not hasattr(self, 'payload') or self.payload is None:
            raise AttributeError('Message is missing payload attribute')

        columns = []
        values = []

        ds_ref = self.payload.get('$ref')
        if ds_ref:
            if client is None:
                from .client import Cortex
                client = Cortex.from_message(self)

            ds = client.dataset(ds_ref)
            df = ds.get_dataframe()
            columns = df.get('columns')
            values = df.get('values')

        elif 'records' in self.payload:
            records = self.payload.get('records')
            if len(records) > 0:
                columns = records[0].keys()
                for obj in records:
                    values.append([obj[key] for key in columns])

        else:
            values = self.payload.get('values')
            if not values:
                raise ValueError('Invalid DataFrame: values missing from payload')

            columns = self.payload.get('columns')
            if not columns:
                raise ValueError('Invalid DataFrame: columns missing from payload')

        try:
            import pandas as pd
            return pd.DataFrame(values, columns=columns)
        except ImportError:
            # TODO warn
            return {'columns': columns, 'values': values}

    def get_dataset(self):
        if not hasattr(self, 'payload') or self.payload is None:
            raise AttributeError('Message is missing payload attribute')

        ds_ref = self.payload.get('$ref')
        if ds_ref:
            from .client import Cortex
            client = Cortex.from_message(self)
            return client.dataset(ds_ref)

        raise AttributeError('Message payload does not contain a Dataset reference')

    @staticmethod
    def with_payload(payload):
        m = Message.from_env()
        m.payload = payload
        return m

    @staticmethod
    def from_env():
        env = CortexEnv()

        params = {}
        if env.api_endpoint:
            params['apiEndpoint'] = env.api_endpoint
        if env.token:
            params['token'] = env.token

        return Message(params)
