/*
 * Decompiled with CFR 0.152.
 */
package oracle.pgx.common.util;

import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.attribute.AclEntry;
import java.nio.file.attribute.AclEntryPermission;
import java.nio.file.attribute.AclEntryType;
import java.nio.file.attribute.AclFileAttributeView;
import java.nio.file.attribute.FileAttribute;
import java.nio.file.attribute.PosixFileAttributeView;
import java.nio.file.attribute.PosixFilePermission;
import java.nio.file.attribute.UserPrincipal;
import java.util.Arrays;
import java.util.List;
import java.util.Set;
import oracle.pgx.common.PgxCharset;
import oracle.pgx.common.util.ErrorMessages;
import oracle.pgx.common.util.SystemUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class FileUtils {
    private static final int DEFAULT_BUFFER_SIZE = 4096;
    public static final int EOF = -1;
    private static final int NOT_FOUND = -1;
    public static final char EXTENSION_SEPARATOR = '.';
    private static final char UNIX_SEPARATOR = '/';
    private static final char WINDOWS_SEPARATOR = '\\';
    private static final Logger LOG = LoggerFactory.getLogger(FileUtils.class);
    private static final byte[] UTF8_LINE_SEP = System.lineSeparator().getBytes(StandardCharsets.UTF_8);
    private static final int TEMP_DIR_ATTEMPTS = 1000;

    public static void write(File file, CharSequence string) throws IOException {
        FileUtils.write(file, string, StandardCharsets.UTF_8, false);
    }

    public static void write(File file, CharSequence string, Charset encoding) throws IOException {
        FileUtils.write(file, string, encoding, false);
    }

    public static void write(File file, CharSequence string, boolean append) throws IOException {
        FileUtils.write(file, string, StandardCharsets.UTF_8, append);
    }

    public static void write(File file, CharSequence string, Charset encoding, boolean append) throws IOException {
        try (FileOutputStream out = new FileOutputStream(file, append);){
            out.write(string.toString().getBytes(encoding));
            out.getFD().sync();
            out.getChannel().force(true);
        }
    }

    public static void writeln(File file, CharSequence string, boolean append) throws IOException {
        try (FileOutputStream out = new FileOutputStream(file, append);){
            out.write(string.toString().getBytes(StandardCharsets.UTF_8));
            out.write(UTF8_LINE_SEP);
        }
    }

    public static void delete(File file) throws IOException {
        if (file.isDirectory()) {
            FileUtils.deleteDirectory(file);
        }
        Files.deleteIfExists(file.toPath());
    }

    public static void deleteDirectory(File directory) throws IOException {
        File[] files = directory.listFiles();
        if (files == null) {
            throw new IOException("Cannot get content of " + directory.getName());
        }
        IOException exception = null;
        for (File file : files) {
            try {
                FileUtils.delete(file);
            }
            catch (IOException e) {
                exception = e;
            }
        }
        if (null != exception) {
            throw exception;
        }
    }

    public static void forceDeleteOnExit(File file) throws IOException {
        if (file.isDirectory()) {
            FileUtils.deleteDirectoryOnExit(file);
        } else {
            file.deleteOnExit();
        }
    }

    private static void deleteDirectoryOnExit(File directory) throws IOException {
        if (!directory.exists()) {
            return;
        }
        directory.deleteOnExit();
        if (Files.isSymbolicLink(directory.toPath())) {
            return;
        }
        File[] files = directory.listFiles();
        if (files == null) {
            throw new IOException("Cannot get content of " + directory.getName());
        }
        IOException exception = null;
        for (File file : files) {
            try {
                FileUtils.forceDeleteOnExit(file);
            }
            catch (IOException e) {
                exception = e;
            }
        }
        if (null != exception) {
            throw exception;
        }
    }

    private static int indexOfLastSeparator(String filePath) {
        if (filePath == null) {
            return -1;
        }
        int lastUnixPos = filePath.lastIndexOf(47);
        int lastWindowsPos = filePath.lastIndexOf(92);
        return Math.max(lastUnixPos, lastWindowsPos);
    }

    public static String getName(String filePath) {
        if (filePath == null) {
            return null;
        }
        FileUtils.checkNullByte(filePath);
        int index = FileUtils.indexOfLastSeparator(filePath);
        return filePath.substring(index + 1);
    }

    public static String getBaseName(String filePath) {
        String fileName = FileUtils.getName(filePath);
        if (fileName == null) {
            return null;
        }
        FileUtils.checkNullByte(fileName);
        int index = FileUtils.indexOfExtension(fileName);
        if (index == -1) {
            return fileName;
        }
        return fileName.substring(0, index);
    }

    private static void checkNullByte(String path) {
        if (path.indexOf(0) != -1) {
            throw new IllegalArgumentException("Null byte not allowed in file name or path");
        }
    }

    private static int indexOfExtension(String filePath) throws IllegalArgumentException {
        if (filePath == null) {
            return -1;
        }
        int extensionPos = filePath.lastIndexOf(46);
        int lastSeparator = FileUtils.indexOfLastSeparator(filePath);
        return lastSeparator > extensionPos ? -1 : extensionPos;
    }

    public static String toString(InputStream input) throws IOException {
        return FileUtils.toString(input, PgxCharset.getCharset());
    }

    public static String toString(InputStream input, Charset inputEncoding) throws IOException {
        int n;
        StringBuilder sw = new StringBuilder();
        if (inputEncoding == null) {
            throw new NullPointerException("charset should not be null");
        }
        InputStreamReader in = new InputStreamReader(input, inputEncoding);
        char[] buffer = new char[4096];
        while (-1 != (n = in.read(buffer))) {
            sw.append(buffer, 0, n);
        }
        return sw.toString();
    }

    public static File createTempDir() {
        String baseDirName = System.getProperty("java.io.tmpdir");
        String baseName = System.currentTimeMillis() + "-";
        for (int counter = 0; counter < 1000; ++counter) {
            try {
                Path path = Files.createTempDirectory(Paths.get(baseDirName, new String[0]), baseName + counter, new FileAttribute[0]);
                return path.toFile();
            }
            catch (IOException iOException) {
                continue;
            }
        }
        throw new IllegalStateException("Cannot create temporary directory inside " + baseDirName + ": attempts are " + baseName + "[0-" + 999 + "])");
    }

    public static FileOutputStream openOutputStream(File file) throws IOException {
        if (!file.exists()) {
            Path path = file.toPath();
            Path parent = path.getParent();
            if (parent != null && !Files.exists(parent, new LinkOption[0])) {
                Files.createDirectories(parent, new FileAttribute[0]);
            }
            Files.createFile(path, new FileAttribute[0]);
        }
        return new FileOutputStream(file);
    }

    public static void validateReadWriteOwnerOnlyPermission(String filePath) throws IllegalArgumentException {
        Path path = Paths.get(filePath, new String[0]);
        try {
            FileUtils.checkCurrentProcessCanOnlyReadAndWrite(path);
            if (!SystemUtils.IS_OS_WINDOWS) {
                FileUtils.checkPosixPermissions(path, true, PosixFilePermission.OWNER_READ, PosixFilePermission.OWNER_WRITE);
            } else {
                FileUtils.checkWindowsOwnerPermissions(path, AclEntryPermission.READ_DATA, AclEntryPermission.WRITE_DATA, AclEntryPermission.READ_ATTRIBUTES, AclEntryPermission.WRITE_ATTRIBUTES);
            }
        }
        catch (IllegalArgumentException e) {
            if (SystemUtils.IS_OS_WINDOWS) {
                LOG.warn("cannot check file permissions {} on Windows: allowing access despite potentially insecure", (Object)filePath);
                return;
            }
            throw new IllegalArgumentException(ErrorMessages.getMessage("FILE_PERMISSION_ERROR", filePath), e);
        }
        catch (IOException e) {
            throw new IllegalArgumentException(ErrorMessages.getMessage("FILE_IO_ERROR", filePath), e);
        }
    }

    public static void checkCurrentProcessCanOnlyReadAndWrite(Path path) {
        if (!Files.isReadable(path) || !Files.isWritable(path) || Files.isExecutable(path)) {
            throw new IllegalArgumentException(path.toString() + ": the owner cannot read and write");
        }
    }

    public static void checkPosixPermissions(Path path, boolean exclusive, PosixFilePermission ... permissions) throws IOException {
        if (permissions.length == 0) {
            throw new IllegalArgumentException("list of permissions to check cannot be empty");
        }
        Set<PosixFilePermission> filePermission = Files.getFileAttributeView(path, PosixFileAttributeView.class, new LinkOption[0]).readAttributes().permissions();
        for (PosixFilePermission p : permissions) {
            if (filePermission.contains((Object)p)) continue;
            throw new IllegalArgumentException(ErrorMessages.getMessage("FILE_LACKS_PERMISSION", path));
        }
        if (exclusive && filePermission.size() != permissions.length) {
            throw new IllegalArgumentException(ErrorMessages.getMessage("FILE_HAS_INSECURE_PERMISSIONS", path));
        }
    }

    static UserPrincipal getThisProcessUser(Path file) throws IOException {
        return file.getFileSystem().getUserPrincipalLookupService().lookupPrincipalByName(System.getProperty("user.name"));
    }

    public static void checkWindowsOwnerPermissions(Path path, AclEntryPermission ... permissions) throws IOException {
        AclFileAttributeView view = Files.getFileAttributeView(path, AclFileAttributeView.class, new LinkOption[0]);
        UserPrincipal owner = FileUtils.getThisProcessUser(path);
        List<AclEntryPermission> permissionsList = Arrays.asList(permissions);
        boolean userHasPermissions = false;
        for (AclEntry e : view.getAcl()) {
            String name;
            if (e.principal().equals(owner) && (e.type().equals((Object)AclEntryType.ALLOW) || e.type().equals((Object)AclEntryType.AUDIT)) && e.permissions().containsAll(permissionsList)) {
                userHasPermissions = true;
            }
            if ((name = e.principal().getName()).equals("SYSTEM") || name.equals("Administrators") || e.principal().equals(owner) || !e.type().equals((Object)AclEntryType.ALLOW) && !e.type().equals((Object)AclEntryType.AUDIT)) continue;
            throw new IllegalArgumentException(ErrorMessages.getMessage("FILE_HAS_INSECURE_PERMISSIONS", path));
        }
        if (!userHasPermissions) {
            throw new IllegalArgumentException(ErrorMessages.getMessage("FILE_LACKS_PERMISSION", path));
        }
    }
}

