# -*- coding: latin1 -*- vim: ts=8 sts=4 et si sw=4 
"""
Erzeugt URLs nach dem file://-Schema
"""
VERSION = '.'.join(map(str,
                       (0,
                        2, # importierbare Version
                        1, # Linux/Cygwin: splitunc, splitdrive n.v.; slashed_ap
                        'rev-%s' % '$Rev: 678 $'[6:-2],
                        )))
from os.path import isdir
from os.path import normpath, normcase, abspath, sep, altsep
from urllib import quote    # pathname2url: relativ, ohne Prfix...

__all__ = ['makeurl',
           'slashed_ap',    # abspath (implies normpath)+normcase with slashes
           ]

if sep == '/':
    def slashed_np(s):
        return normcase(normpath(s))
    def slashed_ap(s):
        return normcase(abspath(s))
else:
    def slashed_np(s):
        return normcase(normpath(s)).replace(sep, '/')
    def slashed_ap(s):
        return normcase(abspath(s)).replace(sep, '/')

_FUNCTIONS = list()

try:
    from os.path import splitunc
except ImportError:
    def handle_unc(fn, suffix=''):
        raise UnsupportedOnThisPlatform
else:
    def handle_unc(fn, suffix=''):
        assert not (altsep and '\\' in fn), fn
        hostandshare, dir = splitunc(fn)
        if not hostandshare:
            raise NoMatch
        # print 'unc: %s -> %s' % (fn, (hostandshare, dir))
        return ''.join(('file:',
                        quote(fn),
                        suffix,
                        ))

    _FUNCTIONS.append(handle_unc)

try:
    from os.path import splitdrive
except ImportError:
    def handle_driveletter(fn, suffix=''):
        raise UnsupportedOnThisPlatform
else:
    def handle_driveletter(fn, suffix=''):
        assert not (altsep and '\\' in fn), fn
        dl, dir = splitdrive(fn)
        if not dl:
            raise NoMatch
        # print 'driveletter: %s -> %s' % (fn, (dl, dir))
        return ''.join(('file:///', # sic
                        dl,
                        quote(dir),
                        suffix,
                        ))

    _FUNCTIONS.append(handle_driveletter)

class NoMatch(Exception):
    """
    The specialized function was called with "other" argument
    """
class UnsupportedOnThisPlatform(NoMatch):
    pass

def makeurl(fn, dirsuffix=0):
    """
    turn the given file spec. to a file:// url:

    fn -- the filename
    dirsuffix -- whether to append a trailing slash for existing directories

    If supported on the current platform, UNC names and drive letters are
    supported (splitunc, splitdrive).
    """
    # ne = normcase(realpath(normpath(fn)))
    ne = slashed_ap(fn)
    dirsuffix = dirsuffix and isdir(fn) and '/' or ''
    for f in _FUNCTIONS:
        try:
            return f(ne, dirsuffix)
        except NoMatch:
            pass
    res = ['file://']
    if ne.startswith('/'):
        res.append(quote(ne))
    else:
        res.extend(('/', quote(ne, '/:')))
    res.append(dirsuffix)
    return ''.join(res)

if __name__ == '__main__':
    from sys import argv, exit, stderr, exc_info
    from thebops.errors import err, warn, check_errors, fatal, progname

    try:
        from thebops.enhopa import OptionParser, OptionGroup
        MY_OPTPARSE = 1
    except:
        from optparse import OptionParser, OptionGroup
        MY_OPTPARSE = 0

    parser = OptionParser(usage="%prog [options] {Verzeichniseintrag} [...] | --help",
                          version=VERSION)
    parser.set_description(u'%(prog)s erzeugt fr die bergebenen Dateinamen'
                           ' URLs nach dem file://-Schema' % {'prog': progname()})
    group_ = OptionGroup(parser, _('Specific options'))
    group_.add_option('--suffix',
                      action='store_true',
                      help=_('append "/" for existing directories'))
    parser.add_option('--verbose', '-v',
                      action='count',
                      help=_('be verbose (-vv: even more verbose)'))
    group_.add_option('--self',
                      action='store_true',
                      help=u'bergibt sich ;-)')
    parser.add_option_group(group_)
    try:
        parser.set_collecting_group()
    except AttributeError:
        pass
    (option, args) = parser.parse_args()

    if option.self:
        args.insert(0, argv[0])

    if not args:
        err('Bitte mindestens eine Datei oder ein Verzeichnis angeben')
        check_errors()

    # import pdb; pdb.set_trace()
    for a in args:
        print makeurl(a, option.suffix)

    check_errors()
