from collections import namedtuple
from collections.abc import Mapping, MutableMapping
from copy import deepcopy
from typing import Any, Dict, Optional, Sequence, Set, Union

import attr
from fontTools.misc.transform import Transform
from fontTools.pens.boundsPen import BoundsPen, ControlBoundsPen

from ufoLib2.typing import Number

# sentinel value to signal a "lazy" object hasn't been loaded yet
_NOT_LOADED = object()


BoundingBox = namedtuple("BoundingBox", "xMin yMin xMax yMax")


def getBounds(drawable, layer):
    pen = BoundsPen(layer)
    # raise 'KeyError' when a referenced component is missing from glyph set
    pen.skipMissingComponents = False
    drawable.draw(pen)
    return pen.bounds if pen.bounds is None else BoundingBox(*pen.bounds)


def getControlBounds(drawable, layer):
    pen = ControlBoundsPen(layer)
    # raise 'KeyError' when a referenced component is missing from glyph set
    pen.skipMissingComponents = False
    drawable.draw(pen)
    return pen.bounds if pen.bounds is None else BoundingBox(*pen.bounds)


def _deepcopy_unlazify_attrs(self, memo):
    if getattr(self, "_lazy", True) and hasattr(self, "unlazify"):
        self.unlazify()
    return self.__class__(
        **{
            (a.name if a.name[0] != "_" else a.name[1:]): deepcopy(
                getattr(self, a.name), memo
            )
            for a in attr.fields(self.__class__)
            if a.init and a.metadata.get("copyable", True)
        },
    )


@attr.s(auto_attribs=True, slots=True, repr=False)
class DataStore(MutableMapping):
    listdir: Optional[Any] = None
    readf: Optional[Any] = None
    writef: Optional[Any] = None
    deletef: Optional[Any] = None

    _data: Dict[str, Any] = attr.ib(factory=dict)

    _reader: Optional[Any] = attr.ib(default=None, init=False, repr=False, eq=False)
    _scheduledForDeletion: Set[str] = attr.ib(factory=set, init=False, repr=False)

    @classmethod
    def read(cls, reader, lazy=True):
        self = cls()
        for fileName in cls.listdir(reader):
            if lazy:
                self._data[fileName] = _NOT_LOADED
            else:
                self._data[fileName] = cls.readf(reader, fileName)
        if lazy:
            self._reader = reader
        return self

    def unlazify(self):
        for _ in self.items():
            pass

    __deepcopy__ = _deepcopy_unlazify_attrs

    # MutableMapping methods

    def __len__(self):
        return len(self._data)

    def __iter__(self):
        return iter(self._data)

    def __getitem__(self, fileName):
        if self._data[fileName] is _NOT_LOADED:
            self._data[fileName] = self.__class__.readf(self._reader, fileName)
        return self._data[fileName]

    def __setitem__(self, fileName, data):
        # should we forbid overwrite?
        self._data[fileName] = data
        if fileName in self._scheduledForDeletion:
            self._scheduledForDeletion.remove(fileName)

    def __delitem__(self, fileName):
        del self._data[fileName]
        self._scheduledForDeletion.add(fileName)

    def __repr__(self):
        n = len(self._data)
        return "<{}.{} ({}) at {}>".format(
            self.__class__.__module__,
            self.__class__.__name__,
            "empty" if n == 0 else "{} file{}".format(n, "s" if n > 1 else ""),
            hex(id(self)),
        )

    def write(self, writer, saveAs=None):
        if saveAs is None:
            saveAs = self._reader is not writer
        # if in-place, remove deleted data
        if not saveAs:
            for fileName in self._scheduledForDeletion:
                self.__class__.deletef(writer, fileName)
        # Write data. Iterating over _data.items() prevents automatic loading.
        for fileName, data in self._data.items():
            # Two paths:
            # 1) We are saving in-place. Only write to disk what is loaded, it
            #    might be modified.
            # 2) We save elsewhere. Load all data files to write them back out.
            if data is _NOT_LOADED:
                if saveAs:
                    data = self.__class__.readf(self._reader, fileName)
                    self._data[fileName] = data
                else:
                    continue
            self.__class__.writef(writer, fileName, data)
        self._scheduledForDeletion = set()
        if saveAs:
            # all data was read by now, ref to reader no longer needed
            self._reader = None

    @property
    def fileNames(self):
        return list(self._data.keys())


class AttrDictMixin(Mapping):
    """Read attribute values using mapping interface.

    For use with Anchors and Guidelines classes, where client code
    expects them to behave as dict.
    """

    def __getitem__(self, key):
        try:
            return getattr(self, key)
        except AttributeError:
            raise KeyError(key)

    def __iter__(self):
        for key in attr.fields_dict(self.__class__):
            if getattr(self, key) is not None:
                yield key

    def __len__(self):
        return sum(1 for _ in self)


def _convert_transform(t: Union[Transform, Sequence[Number]]) -> Transform:
    """Return a passed-in Transform as is, otherwise convert a sequence of
    numbers to a Transform if need be."""
    return t if isinstance(t, Transform) else Transform(*t)
