from nose.tools import make_decorator
from matplotlib.testing.decorators import cleanup
import matplotlib.pyplot as plt


def _assert_same_figure_images(fig, name, test_file, tol=17):
    """Asserts that the figure object produces the right image"""
    import os
    import shutil
    from matplotlib import cbook
    from matplotlib.testing.compare import compare_images

    if not ".png" in name:
        name = name+".png"

    basedir = os.path.abspath(os.path.dirname(test_file))
    basename = os.path.basename(test_file)
    subdir = os.path.splitext(basename)[0]

    baseline_dir = os.path.join(basedir, 'baseline_images', subdir)
    result_dir = os.path.abspath(os.path.join('result_images', subdir))

    if not os.path.exists(result_dir):
        cbook.mkdirs(result_dir)

    orig_expected_fname = os.path.join(baseline_dir, name)
    actual_fname = os.path.join(result_dir, name)

    def make_test_fn(fname, purpose):
        base, ext = os.path.splitext(fname)
        return '%s-%s%s' % (base, purpose, ext)
    expected_fname = make_test_fn(actual_fname, 'expected')
    # Save the figure before testing whether the original image
    # actually exists. This make creating new tests much easier,
    # as the result image can afterwards just be copied.
    fig.savefig(actual_fname)
    if os.path.exists(orig_expected_fname):
        shutil.copyfile(orig_expected_fname, expected_fname)
    else:
        raise Exception("Baseline image %s is missing" % orig_expected_fname)
    err = compare_images(expected_fname, actual_fname,
                         tol, in_decorator=True)
    if err:
        msg = 'images not close: {actual:s} vs. {expected:s} (RMS {rms:.2f})'.format(**err)
        raise ImagesComparisonFailure(msg)
    return err


def image_comparison(baseline_images=None, tol=17, extensions=None):
    """
    call signature::
      image_comparison(baseline_images=['my_figure'], tol=17)
    Compare images generated by the test with those specified in
    *baseline_images*, which must correspond else an
    ImagesComparisonFailure exception will be raised.
    Keyword arguments:
      *baseline_images*: list
        A list of strings specifying the names of the images generated
        by calls to :meth:`matplotlib.figure.savefig`.
      *tol*: (default 13)
        The RMS threshold above which the test is considered failed.
    """

    if baseline_images is None:
        raise ValueError('baseline_images must be specified')

    if extensions:
        # ignored, only for compatibility with matplotlibs decorator!
        pass

    def compare_images_decorator(func):
        import inspect
        _file = inspect.getfile(func)
        def decorated():
            # make sure we don't carry over bad images from former tests.
            assert len(plt.get_fignums()) == 0, "no of open figs: %s -> find the last test with ' " \
                                        "python tests.py -v' and add a '@cleanup' decorator." % \
                                        str(plt.get_fignums())
            func()
            assert len(plt.get_fignums()) == len(baseline_images), "different number of " \
                                                                   "baseline_images and actuall " \
                                                                   "plots."
            for fignum, baseline in zip(plt.get_fignums(), baseline_images):
                figure = plt.figure(fignum)
                _assert_same_figure_images(figure, baseline, _file, tol=tol)
        # also use the cleanup decorator to close any open figures!
        return make_decorator(cleanup(func))(decorated)
    return compare_images_decorator
