#!/usr/bin/env python
# Lint as: python3
"""Functions to run individual GRR components during self-contained testing."""

import atexit
import os
import shutil
import signal
import subprocess
import sys
import tempfile
import threading
import time

from typing import Dict, Iterable, List, Optional, Tuple, Union

import portpicker

from grr_response_core.lib import package
from grr_response_test.lib import api_helpers

ComponentOptions = Dict[str, Union[int, str]]


class Error(Exception):
  """Module-specific base error class."""


class ConfigInitializationError(Error):
  """Raised when a self-contained config can't be written."""


def _ComponentOptionsToArgs(options):
  if options is None:
    return []

  args = []
  for k, v in options.items():
    args.extend(["-p", "%s=%s" % (k, v)])
  return args


def _GetServerComponentArgs(config_path):
  """Returns a set of command line arguments for server components.

  Args:
    config_path: Path to a config path generated by
      self_contained_config_writer.

  Returns:
    An iterable with command line arguments to use.
  """

  primary_config_path = package.ResourcePath(
      "grr-response-core", "install_data/etc/grr-server.yaml")
  secondary_config_path = package.ResourcePath(
      "grr-response-test", "grr_response_test/test_data/grr_test.yaml")
  return [
      "--config",
      primary_config_path,
      "--secondary_configs",
      ",".join([secondary_config_path, config_path]),
      "-p",
      "Monitoring.http_port=%d" % portpicker.pick_unused_port(),
      "-p",
      "AdminUI.webauth_manager=NullWebAuthManager",
  ]


def _GetRunEndToEndTestsArgs(
    client_id,
    server_config_path,
    tests = None,
    manual_tests = None):
  """Returns arguments needed to configure run_end_to_end_tests process.

  Args:
    client_id: String with a client id pointing to an already running client.
    server_config_path: Path to the server configuration file.
    tests: (Optional) List of tests to run.
    manual_tests: (Optional) List of manual tests to not skip.

  Returns:
    An iterable with command line arguments.
  """
  port = api_helpers.GetAdminUIPortFromConfig(server_config_path)

  api_endpoint = "http://localhost:%d" % port
  args = [
      "--api_endpoint",
      api_endpoint,
      "--api_user",
      "admin",
      "--api_password",
      "admin",
      "--client_id",
      client_id,
      "--ignore_test_context",
      "True",
  ]
  if tests is not None:
    args += ["--whitelisted_tests", ",".join(tests)]
  if manual_tests is not None:
    args += ["--manual_tests", ",".join(manual_tests)]

  return args


def _StartComponent(main_package, args):
  """Starts a new process with a given component.

  This starts a Python interpreter with a "-m" argument followed by
  the main package name, thus effectively executing the main()
  function of a given package.

  Args:
    main_package: Main package path.
    args: An iterable with program arguments (not containing the program
      executable).

  Returns:
    Popen object corresponding to a started process.
  """
  popen_args = [sys.executable, "-m", main_package] + args
  print("Starting %s component: %s" % (main_package, " ".join(popen_args)))
  process = subprocess.Popen(popen_args)
  print("Component %s pid: %d" % (main_package, process.pid))

  def KillOnExit():
    if process.poll() is None:
      print("Killing %s." % main_package)
      process.kill()
      process.wait()

  atexit.register(KillOnExit)

  return process


def InitConfigs(mysql_database,
                mysql_username = None,
                mysql_password = None,
                logging_path = None,
                osquery_path = None):
  """Initializes server and client config files."""

  # Create 2 temporary files to contain server and client configuration files
  # that we're about to generate.
  #
  # TODO(user): migrate to TempFilePath as soon grr.test_lib is moved to
  # grr_response_test.
  fd, built_server_config_path = tempfile.mkstemp(".yaml")
  os.close(fd)
  print("Using temp server config path: %s" % built_server_config_path)
  fd, built_client_config_path = tempfile.mkstemp(".yaml")
  os.close(fd)
  print("Using temp client config path: %s" % built_client_config_path)

  def CleanUpConfigs():
    os.remove(built_server_config_path)
    os.remove(built_client_config_path)

  atexit.register(CleanUpConfigs)

  # Generate server and client configs.
  config_writer_flags = [
      "--dest_server_config_path",
      built_server_config_path,
      "--dest_client_config_path",
      built_client_config_path,
      "--config_mysql_database",
      mysql_database,
  ]

  if mysql_username is not None:
    config_writer_flags.extend(["--config_mysql_username", mysql_username])

  if mysql_password is not None:
    config_writer_flags.extend(["--config_mysql_password", mysql_password])

  if logging_path is not None:
    config_writer_flags.extend(["--config_logging_path", logging_path])

  if osquery_path is not None:
    config_writer_flags.extend(["--config_osquery_path", osquery_path])

  p = _StartComponent(
      "grr_response_test.lib.self_contained_config_writer",
      config_writer_flags)
  if p.wait() != 0:
    raise ConfigInitializationError("ConfigWriter execution failed: {}".format(
        p.returncode))

  return (built_server_config_path, built_client_config_path)


def StartServerProcesses(
    server_config_path,
    component_options = None
):

  def Args():
    return _GetServerComponentArgs(
        server_config_path) + _ComponentOptionsToArgs(component_options)

  return [
      _StartComponent(
          "grr_response_server.gui.admin_ui",
          Args()),
      _StartComponent(
          "grr_response_server.bin.frontend",
          Args()),
      _StartComponent(
          "grr_response_server.bin.worker",
          Args()),
  ]


def StartClientProcess(client_config_path,
                       component_options = None,
                       verbose = False):
  return _StartComponent(
      "grr_response_client.client",
      ["--config", client_config_path] + (["--verbose"] if verbose else []) +
      _ComponentOptionsToArgs(component_options))


def RunEndToEndTests(client_id,
                     server_config_path,
                     tests = None,
                     manual_tests = None):
  """Runs end to end tests on a given client."""
  p = _StartComponent(
      "grr_response_test.run_end_to_end_tests",
      _GetServerComponentArgs(server_config_path) + _GetRunEndToEndTestsArgs(
          client_id, server_config_path, tests=tests,
          manual_tests=manual_tests))
  if p.wait() != 0:
    raise RuntimeError("RunEndToEndTests execution failed.")


def RunBuildTemplate(server_config_path,
                     component_options = None,
                     version_ini = None):
  """Runs end to end tests on a given client."""
  output_dir = tempfile.mkdtemp()

  def CleanUpTemplate():
    shutil.rmtree(output_dir)

  atexit.register(CleanUpTemplate)

  options = dict(component_options or {})
  if version_ini:
    fd, version_ini_path = tempfile.mkstemp(".ini")
    try:
      os.write(fd, version_ini.encode("ascii"))
    finally:
      os.close(fd)

    options["ClientBuilder.version_ini_path"] = version_ini_path

  p = _StartComponent(
      "grr_response_client_builder.client_build",
      _GetServerComponentArgs(server_config_path) +
      _ComponentOptionsToArgs(options) + ["build", "--output", output_dir])
  if p.wait() != 0:
    raise RuntimeError("RunBuildTemplate execution failed.")

  return os.path.join(output_dir, os.listdir(output_dir)[0])


def RunRepackTemplate(
    server_config_path,
    template_path,
    component_options = None):
  """Runs 'grr_client_builder repack' to repack a template."""
  output_dir = tempfile.mkdtemp()

  def CleanUpInstaller():
    shutil.rmtree(output_dir)

  atexit.register(CleanUpInstaller)

  p = _StartComponent(
      "grr_response_client_builder.client_build",
      _GetServerComponentArgs(server_config_path) +
      _ComponentOptionsToArgs(component_options) +
      ["repack", "--template", template_path, "--output_dir", output_dir])
  if p.wait() != 0:
    raise RuntimeError("RunRepackTemplate execution failed.")

  # Repacking may apparently generate more than one file. Just select the
  # biggest one: it's guaranteed to be the template.
  paths = [os.path.join(output_dir, fname) for fname in os.listdir(output_dir)]
  sizes = [os.path.getsize(p) for p in paths]
  _, biggest_path = max(zip(sizes, paths))

  return biggest_path


def RunUploadExe(server_config_path,
                 exe_path,
                 platform,
                 component_options = None):
  """Runs 'grr_config_upater upload_exe' to upload a binary to GRR."""
  p = _StartComponent(
      "grr_response_server.bin.config_updater",
      _GetServerComponentArgs(server_config_path) +
      _ComponentOptionsToArgs(component_options) + [
          "upload_exe", "--file", exe_path, "--platform", platform,
          "--upload_subdirectory", "test"
      ])
  if p.wait() != 0:
    raise RuntimeError("RunUploadExe execution failed.")

  return "%s/test/%s" % (platform, os.path.basename(exe_path))


_PROCESS_CHECK_INTERVAL = 0.1


def _DieIfSubProcessDies(processes,
                         already_dead_event):
  """Synchronously waits for processes and dies if one dies."""
  while True:
    for p in processes:
      if p.poll() not in [None, 0]:
        # Prevent a double kill. When the main process exits, it kills the
        # children. We don't want a child's death to cause a SIGTERM being
        # sent to a process that's already exiting.
        if already_dead_event.is_set():
          return

        # DieIfSubProcessDies runs in a background thread, raising an exception
        # will just kill the thread while what we want is to fail the whole
        # process.
        print("Subprocess %s died unexpectedly. Killing main process..." %
              p.pid)
        for kp in processes:
          try:
            os.kill(kp.pid, signal.SIGTERM)
          except OSError:
            pass
        # sys.exit only exits a thread when called from a thread.
        # Killing self with SIGTERM to ensure the process runs necessary
        # cleanups before exiting.
        os.kill(os.getpid(), signal.SIGTERM)
    time.sleep(_PROCESS_CHECK_INTERVAL)


def DieIfSubProcessDies(
    processes):
  """Kills the process if any of given processes dies.

  This function is supposed to run in a background thread and monitor provided
  processes to ensure they don't die silently.

  Args:
    processes: An iterable with multiprocessing.Process instances.

  Returns:
    Background thread started to monitor the processes.
  """
  already_dead_event = threading.Event()
  t = threading.Thread(
      target=_DieIfSubProcessDies, args=(processes, already_dead_event))
  t.daemon = True
  t.start()

  def PreventDoubleDeath():
    already_dead_event.set()

  atexit.register(PreventDoubleDeath)

  return t
