#!/usr/bin/env python
# Lint as: python3
"""API errors definitions."""
from __future__ import absolute_import
from __future__ import division
from __future__ import unicode_literals

from typing import Tuple

VersionTuple = Tuple[int, int, int, int]


class Error(RuntimeError):
  """Errors generated by API client library."""


class ResourceNotFoundError(Error):
  """Raised when resource is not found."""


class AccessForbiddenError(Error):
  """Raised when resource access is forbidden."""


class ApiNotImplementedError(Error):
  """Raised when API method is not implemented."""


class UnknownError(Error):
  """Unknown server error."""


class PollTimeoutError(Error):
  """Raised when poll operations times out."""


class FlowFailedError(Error):
  """Raised when waiting on a flow that eventually fails."""


class InvalidArgumentError(Error):
  """Raised when invalid argument(s) is/are passed to the API call."""


class VersionMismatchError(Error):
  """Raised when API client version is incompatible with the server."""

  def __init__(
      self,
      server_version: VersionTuple,
      api_client_version: VersionTuple,
  ):
    message = "GRR server is {server}, but API client is only {client}"
    message = message.format(server=server_version, client=api_client_version)

    super().__init__(message)

    self.server_version = server_version
    self.api_client_version = api_client_version
